<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.io
 */
/**
 * @file Dir.php
 * @brief Dir utility.
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: Dir.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * Dir utility.
 */
require_once('ficus/lang/String.php');
require_once('ficus/lang/Assert.php');
require_once('ficus/io/PathPattern.php');
require_once('ficus/exception/DirException.php');
/**
 * @class Ficus_Dir
 */
class Ficus_Dir {

    /**
     * @var $pattern Ficus_PathPattern
     */
    private $pattern;

    /**
     * @var $path Ficus_PathPattern
     */
    private $path;

    /**
     * Constructor.
     *
     * @param $path string path
     * @param $pattern Ficus_PathPattern of search pattern.
     */
    public function __construct($path, $pattern = null) {
        $this->setPath($path);
        $this->setPattern($pattern);
    }

    /**
     * Get pattern.
     *
     * @return Ficus_PathPattern of search pattern.
     */
    public function getPattern() { return $this->pattern; }

    /**
     * Set pattern.
     *
     * @param $pattern Ficus_PathPattern of search pattern.
     */
    public function setPattern($pattern) {
        Ficus_Assert::isTypeAllowNull($pattern, "Ficus_PathPattern");
        if(is_null($pattern)){
            $pattern = new Ficus_PathPattern();
        }
        $this->pattern = $pattern;
    }

    /**
     * Get path.
     *
     * @return string of search pattern.
     */
    public function getPath() { return $this->path; }

    /**
     * Set path.
     *
     * @param $path string path
     */
    public function setPath($path) {
        if(is_dir($path)){
            $path = $path . "/";
        }else if(is_file($path)){
            $path = dirname($path) . "/" . basename($path);
        }else{
            throw new Ficus_IllegalArgumentException("$path is not dir nor file. check is readable");
        }
        $this->path = self::normalize($path);
    }

    /**
     * add Path
     * @param $right String path
     * @param $check boolean check is dir exists.
     * @return Ficus_Dir this object
     */
    public function add($right){
        Ficus_Assert::isPrimitiveType($right, "string");
        $path = $this->path . "/$right";
        if(is_dir($path)){
            $path .= "/";
        }
        $path = Ficus_Dir::normalize($path);

        $dir = new Ficus_Dir($path, $this->pattern);
        return $dir;
    }
    /**
     * Each files.
     *
     * @param $callback callback of each files.
     * @param $path string start path.
     */
    public function each($callback, $path = null) {
        if(is_null($path)){
            $path = $this->path;
        }
        foreach (scandir($path, 1) as $entry) {
            $fullpath = $path  . '/' . $entry;
            if (is_dir($fullpath) && $entry{0} != '.') {
                $this->each($callback, $fullpath);
            } else if (is_null($this->pattern) ==  false ||
                       $this->pattern->match($fullpath)) {
                call_user_func($callback, $fullpath);
            }
        }
    }

    /**
     * extract dir from passed dir
     * @param $path string path
     * @param $mark string mark
     * @param $additional string additional path
     * @param String
     */
    public static function createInstance($path,
                                          $mark = null,
                                          $additional = null){
        if(is_null($mark) == false){
            $index = strrpos($path, $mark);
            if($index === false){
                throw new Ficus_IllegalArgumentException("$mark is not found in $path");
            }
            $path = substr($path, 0, $index);
        }
        if(is_dir($path)){
            $dir = $path;
        }else if(is_file($path)){
            $dir = dirname($path);
        }else{
            throw new Ficus_IllegalArgumentException("$path is not exist");
        }

        if(is_null($additional) == false){
            $dir = $dir . "/" . $additional;
        }
        $dir .= "/";
        return new Ficus_Dir($dir);
    }

    /**
     * equals
     * @param $right Ficus_Dir dir
     * @return boolean true if same
     */
    public function equals($right){
        if($right instanceof Ficus_Dir){
            return ($right->path === $this->path);
        }else if(is_string($right)){
            $dir = new Ficus_Dir($right);
            return $this->equals($dir);
        }else{
            throw new Ficus_IllegalArgumentException("{$this->path} is not equals with {$right}");
        }            
    }

    /**
     * list files
     * @param $path String base dir
     * @return Array of found file and dirs.
     */
    public function getList($path = null){
        if(is_null($path)){
            $path = $this->path;
        }

        $result = array();
        foreach(scandir($path, 1) as $entry){
            if($entry == "." || $entry == ".."){
                continue;
            }
            $fullpath = $path  . '/' . $entry;
            if(is_dir($fullpath) &&
               $this->pattern->match($fullpath)){
                $array = $this->getList($fullpath);
                $result = array_merge($result, $array);
            }else if($this->pattern->match($fullpath)){
                $result[$entry] = $fullpath;
            }
        }
        return $result;
    }

    /**
     * Normailze path separator.
     *
     * @param $path string path.
     * @param $separator string directory separator.
     * @return string normalized path.
     */
    public static function normalizePathSeparator($path, $separator = null) {
        if ($separator === null) {
            $separator = DIRECTORY_SEPARATOR;
        } else {
            assert($separator != '/' || $separator != '\\');
        }
        $separatorPattern = array('\\' => '/\//', '/' => '/\\\/');
        return preg_replace($separatorPattern[$separator], $separator, $path);
    }

    /**
     * Normalize path.
     *
     * @param $path string path.
     * @param $separator string directory separator.
     * @return string resolved path.
     */
    public static function normalize($path, $separator = null) {
        Ficus_Assert::isPrimitiveType($path, "string");
        $separator = ($separator === null) ? DIRECTORY_SEPARATOR : $separator;
        assert($separator != '/' || $separator != '\\');

        $path = self::normalizePathSeparator($path, $separator);

        // //
        $schemePattern = array('/' => '/^[^:]+:\/\//',
                               '\\' => '/^[^:]+:\\\\/');
        if (preg_match($schemePattern[$separator], $path) > 0) {
            $str = new Ficus_String($path);
            $currentPattern = array('/' => '/\/\//',
                                    '\\' => '/\\\\/');
            $ary = preg_split($currentPattern[$separator], $path, -1, PREG_SPLIT_NO_EMPTY);
            $path = $ary[0] . $separator . $separator . join($separator, array_slice($ary, 1));
            if($str->endsWith("\\") || $str->endsWith("/")){
                $path .= $separator;
            }
        } else {
            $currentPattern = array('/' => '/\/(?=\/)/',
                                    '\\' => '/\\\\(?=\\\\)/');
            while (preg_match($currentPattern[$separator], $path) >= 1) {
                $path = preg_replace($currentPattern[$separator], '', $path);
            }
        }

        // ./
        $currentPattern = array('/' => '/(?:\/\.\/|\/\.$)/',
                                '\\' => '/(?:\\\\.\\\|\\\\.$)/');
        while (preg_match($currentPattern[$separator], $path) >= 1) {
            $path = preg_replace($currentPattern[$separator], $separator, $path);
        }

        // ../
        $parentPattern = array('/' => '/\/[^\.\/]+(?:\/\.\.\/|\/\.\.$)/',
                               '\\' => '/\\\[^\.\\\]+(?:\\\\.\.\\\|\\\\.\.$)/');
        while (preg_match($parentPattern[$separator], $path) >= 1) {
            $path = preg_replace($parentPattern[$separator], $separator, $path);
        }

        return $path;
    }

    /**
     * Is base dir.
     *
     * @param $path string path.
     * @return boolean true if $path is base dir.
     */
    public static function isBaseDir($path) {
        if (strlen($path) === 0) {
            return false;
        }
        return ($path{0} == '/' || strpos($path, ':') !== false);
    }

    /**
     * Resolve path.
     *
     * @param $base string base path.
     * @param $relative string relative path.
     * @param $separator string directory separator.
     * @return string resolved path.
     * @throw Ficus_DirException no base dir or no relative dir.
     */
    public static function resolve($base, $relative, $separator = null) {
        $separator = ($separator === null) ? DIRECTORY_SEPARATOR : $separator;
        if (!self::isBaseDir($base)) {
            throw new Ficus_DirException("No base dir: '$base'");
        }
        if (self::isBaseDir($relative)) {
            throw new Ficus_DirException("No relative dir: '$relative'");
        }
        $base = self::normalizePathSeparator($base, $separator);
        $relative = self::normalizePathSeparator($relative, $separator);
        return self::normalize($base . $separator . $relative, $separator);
    }

    /**
     * create dir
     *
     * @param $dir
     * @throw Ficus_DirException cannot create dir
     */
    public static function mkdir($dir, $mode=0755){
        $dir = self::normalize($dir);
        if(is_readable($dir)){
            return true;
        }
        if(mkdir($dir, $mode, true) === false){
            throw new Ficus_DirException("cannot create $dir");
        }
        if(is_dir($dir) === false){
            throw new Ficus_DirException("dir created, but $dir is not valid");
        }
        return true;
    }

    /**
     * get string form
     *
     * @return string string
     */
    public function __toString(){
        return $this->getPath();
    }
}
?>
