// $Id: exerb.cpp,v 1.75 2003/01/10 04:42:50 yuya Exp $

////////////////////////////////////////////////////////////////////////////////

#include "exerb.h"

////////////////////////////////////////////////////////////////////////////////

static void  ExInitRuby(int argc, char **argv);
static void  ExInitModule();
static void  ExReplaceRequire();
static bool  ExMapping();
static void  ExUnMapping();
static bool  ExSetup();
static void  ExSetupTable(PEXERB_ARCHIVE_HEADER pArchive);
static void  ExSetupKcode(PEXERB_ARCHIVE_HEADER pArchive);
static int   ExExecute();
#ifdef RUBY18
static VALUE ExExecuteFirstScript(VALUE data);
#else
static VALUE ExExecuteFirstScript();
#endif
static void  ExCleanup();
static void  ExFinalRuby();
static int   ExFail();
static void  ExOnErrorID(int iResourceID);
static void  ExRaiseLoadError(int iResourceID);
static void  ExCreateTemporaryFile(char *pszFilePathBuffer);
static void  ExWriteFile(char *pszFilePath, void *pvBuffer, DWORD dwSize);
static HMODULE ExLoadDll0(char *pszFilePath, VALUE vFileName);
static VALUE ExGetInitFunctionName(VALUE vFileName);
static bool  ExRequireDll(VALUE vFileName, DWORD dwID);
static bool  ExRequireScript(VALUE vFileName, DWORD dwID);
static void  ExGetSelfFilePath(LPSTR pszBuffer, UINT uiBufferSize);
static VALUE ExEvalString(VALUE vString, VALUE vFileName);
static bool  ExFindInsideFile(VALUE vFilename, bool *pbShared, VALUE *pvRealname, DWORD *pdwID);
static VALUE ExRequire(VALUE vObject, VALUE vFileName);

static VALUE ExRequireOutside(VALUE fname);
static bool  ExFindOutsideFile(VALUE fname, VALUE *realname, VALUE *filename);
static void  ExLoadOutsideScript(VALUE fname, VALUE name);
static void  ExLoadOutsideLibrary(VALUE fname, VALUE name);

static VALUE rb_exerb_s_runtime_p(VALUE self);

VALUE WINAPI ex_require(const char *fname);
VALUE WINAPI ex_f_require(VALUE obj, VALUE fname);

////////////////////////////////////////////////////////////////////////////////

extern "C" VALUE ruby_errinfo;
extern "C" VALUE rb_load_path;

////////////////////////////////////////////////////////////////////////////////

typedef void (*INITPROC)(void);
typedef VALUE (__cdecl *RUBYPROC)(...);

////////////////////////////////////////////////////////////////////////////////

static void* g_pArchive = NULL;

static DWORD g_dwFirstScriptNameID = 0;

static vector<HMODULE> g_DllTable;
static map<string, DWORD> g_Name2IdTable;
static map<DWORD, string> g_Id2NameTable;
static map<DWORD, DWORD> g_ScriptPtrTable;
static map<DWORD, DWORD> g_ScriptSizeTable;
static map<DWORD, DWORD> g_DllPtrTable;
static map<DWORD, DWORD> g_DllSizeTable;

static ON_ERROR_FNC ExOnError = NULL;
static ON_FAIL_FNC  ExOnFail  = NULL;

char g_szPhiSoFileName[MAX_PATH] = "";

////////////////////////////////////////////////////////////////////////////////

int
ExMain(int argc, char **argv, ON_ERROR_FNC onerror, ON_FAIL_FNC onfail)
{
	int iResultCode = 1;

	ExOnError = onerror;
	ExOnFail  = onfail;

	::ExInitRuby(argc, argv);
	::ExInitModule();
	::ExReplaceRequire();
	
	if ( ::ExMapping() ) {
		if ( ::ExSetup() ) {
			iResultCode = ::ExExecute();
			::ExFinalRuby();
			::ExCleanup();
		}
		::ExUnMapping();
	}

	return iResultCode;
}

static void
ExInitRuby(int argc, char **argv)
{
	::NtInitialize(&argc, &argv);
	::ruby_init();
	::ruby_set_argv(argc - 1, argv + 1);

	::rb_ary_push(rb_load_path, ::rb_str_new2("."));
}

static void
ExInitModule()
{
	VALUE mExerb = ::rb_define_module("Exerb");
	::rb_define_singleton_method(mExerb, "runtime?", (RUBYPROC)rb_exerb_s_runtime_p, 0);
}

static void
ExReplaceRequire()
{
	::rb_define_global_function("require", (RUBYPROC)ExRequire, 1);
}

static bool
ExMapping()
{
	PEXERB_DOS_HEADER pDosHeader = (PEXERB_DOS_HEADER)::GetModuleHandle(NULL);
	if ( !pDosHeader->OffsetToArchive || !pDosHeader->SizeOfArchive ) {
	    ::ExOnErrorID(IDS_HEADER_NOT_FOUND);
	    return false;
	}

	const int nArchiveSize = (pDosHeader->SizeOfCompressedArchive > 0 ? pDosHeader->SizeOfCompressedArchive : pDosHeader->SizeOfArchive);
	g_pArchive = new char[nArchiveSize];

	char szFilePath[MAX_PATH] = "";
	::ExGetSelfFilePath(szFilePath, sizeof(szFilePath));

	HANDLE hFile = ::CreateFile(szFilePath, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
	if ( hFile == INVALID_HANDLE_VALUE ) {
		::ExOnErrorID(IDS_FAIL_CREATEFILE);
		return false;
	}

	DWORD dwRead = 0;
	::SetFilePointer(hFile, pDosHeader->OffsetToArchive, 0, FILE_BEGIN);
	::ReadFile(hFile, g_pArchive, nArchiveSize, &dwRead, NULL);

	::CloseHandle(hFile);

	if ( pDosHeader->SizeOfCompressedArchive > 0 ) {
#ifdef USE_ZLIB
		DWORD dwOriginalSize   = pDosHeader->SizeOfArchive;
		DWORD dwCompressedSize = pDosHeader->SizeOfCompressedArchive;
		void* pDecompressedArchive = new char[dwOriginalSize];
		::uncompress((BYTE*)pDecompressedArchive, &dwOriginalSize, (BYTE*)g_pArchive, dwCompressedSize);
		delete[] g_pArchive;
		g_pArchive = pDecompressedArchive;
#else
	    ::ExOnError("The compressed archive isn't supported by the core.");
	    return false;
#endif
	}

	return true;
}

static void
ExUnMapping()
{
	delete[] g_pArchive;
}

static bool
ExSetup()
{
	PEXERB_ARCHIVE_HEADER pArchive = (PEXERB_ARCHIVE_HEADER)g_pArchive;

	if ( pArchive->NumberOfScript == 0 ) {
		::ExOnErrorID(IDS_SCRIPT_NOT_FOUND);
		return false;
	}

	::ExSetupTable(pArchive);
	::ExSetupKcode(pArchive);

	return true;
}

static void
ExSetupTable(PEXERB_ARCHIVE_HEADER pArchive)
{
	DWORD i = 0;

	const DWORD dwBaseOfNameTable   = (DWORD)pArchive + pArchive->OffsetToName;
	const DWORD dwBaseOfScriptTable = (DWORD)pArchive + pArchive->OffsetToScript;
	const DWORD dwBaseOfDllTable    = (DWORD)pArchive + pArchive->OffsetToDLL;

	PEXERB_NAME_HEADER pNameHeader = (PEXERB_NAME_HEADER)dwBaseOfNameTable;
	for ( i = 0; i < pArchive->NumberOfName; i++, pNameHeader++ ) {
		const char* name = (char*)(pNameHeader->Offset + dwBaseOfNameTable);
		const DWORD id   = pNameHeader->ID;
		g_Name2IdTable[name] = id;
		g_Id2NameTable[id]   = name;	
	}

	PEXERB_SCRIPT_HEADER pScriptHeader = (PEXERB_SCRIPT_HEADER)dwBaseOfScriptTable;
	g_dwFirstScriptNameID = pScriptHeader->NameID;
	for ( i = 0; i < pArchive->NumberOfScript; i++, pScriptHeader++ ) {
		g_ScriptPtrTable[pScriptHeader->NameID]  = pScriptHeader->Offset + dwBaseOfScriptTable;
		g_ScriptSizeTable[pScriptHeader->NameID] = pScriptHeader->Size;
	}

	PEXERB_DLL_HEADER pDllHeader = (PEXERB_DLL_HEADER)dwBaseOfDllTable;
	for ( i = 0; i < pArchive->NumberOfDLL; i++, pDllHeader++ ) {
		g_DllPtrTable[pDllHeader->NameID]  = pDllHeader->Offset + dwBaseOfDllTable;
		g_DllSizeTable[pDllHeader->NameID] = pDllHeader->Size;
	}
}

static void
ExSetupKcode(PEXERB_ARCHIVE_HEADER pArchive)
{
	switch ( pArchive->Options.Kcode ) {
	case EXERB_OPTIONS_KCODE_NONE: ::rb_set_kcode("n"); break; // None
	case EXERB_OPTIONS_KCODE_EUC:  ::rb_set_kcode("e"); break; // EUC
	case EXERB_OPTIONS_KCODE_SJIS: ::rb_set_kcode("s"); break; // Shift-JIS
	case EXERB_OPTIONS_KCODE_UTF8: ::rb_set_kcode("u"); break; // UTF-8
	}
}

static int
ExExecute()
{
	int state = 0;

	::rb_protect(ExExecuteFirstScript, 0, &state);

	if ( state ) {
		return ::ExFail();
	} else {
		return 0;
	}
}

static VALUE
#ifdef RUBY18
ExExecuteFirstScript(VALUE data)
#else
ExExecuteFirstScript()
#endif
{
	const VALUE vScirptSource = ::rb_str_new((char*)g_ScriptPtrTable[g_dwFirstScriptNameID], g_ScriptSizeTable[g_dwFirstScriptNameID]);
	const VALUE vScritpName   = ::rb_str_new2(g_Id2NameTable[g_dwFirstScriptNameID].c_str());
	::ruby_script(RSTRING(vScritpName)->ptr);
	::ExEvalString(vScirptSource, vScritpName);

	return Qnil;
}

static void
ExCleanup()
{
	for ( int i = g_DllTable.size() - 1; i >= 0 ; i-- ) {
		HMODULE hModule = g_DllTable[i];
		char filepath[MAX_PATH] = "";
		::GetModuleFileName(hModule, filepath, sizeof(filepath));
		::FreeLibrary(hModule);
		::DeleteFile(filepath);
	}

	g_DllTable.clear();
	g_Name2IdTable.clear();
	g_Id2NameTable.clear();
	g_ScriptPtrTable.clear();
	g_ScriptSizeTable.clear();
	g_DllPtrTable.clear();
	g_DllSizeTable.clear();
}

static void
ExFinalRuby()
{
	::ruby_finalize();
}

static int
ExFail()
{
	if ( ruby_errinfo ) {
		if ( ::rb_obj_is_kind_of(ruby_errinfo, rb_eSystemExit) == Qtrue ) {
			return 0;
		} else {
			ExOnFail(ruby_errinfo);
			return 1;
		}
	} else {
		::ExOnErrorID(IDS_CANNOT_OBTAIN_EXCEPTION);
		return 1;
	}
}

static void
ExOnErrorID(int iResourceID)
{
	char message[1024] = "";
	::LoadString(::GetModuleHandle(NULL), iResourceID, message, sizeof(message));
	::ExOnError(message);
}

static void
ExRaiseLoadError(int iResourceID)
{
	char message[1024] = "";
	::LoadString(::GetModuleHandle(NULL), iResourceID, message, sizeof(message));
	::rb_raise(rb_eLoadError, message);
}

static void
ExCreateTemporaryFile(char *pszFilePathBuffer)
{
	char szTemporaryDirectoryPath[MAX_PATH] = "";
	::GetTempPath(sizeof(szTemporaryDirectoryPath), szTemporaryDirectoryPath);
	::GetTempFileName(szTemporaryDirectoryPath, "ruby", 0, pszFilePathBuffer);
}

static void
ExWriteFile(char *pszFilePath, void *pvBuffer, DWORD dwSize)
{
	FILE *file = ::fopen(pszFilePath, "wb");
	::fwrite(pvBuffer, 1, dwSize, file);
	::fclose(file);
}

static bool
ExLoadDll(char *filepath, VALUE filename)
{
	const HMODULE hModule = ::ExLoadDll0(filepath, filename);
	if ( !hModule ) return false;

	g_DllTable.push_back(hModule);

	VALUE basename = ::rb_funcall(rb_cFile, ::rb_intern("basename"), 1, filename);
	if ( ::stricmp(RSTRING(basename)->ptr, "phi.so") == 0 ) {
		char fullpath[MAX_PATH] = "";
		char *phiname = NULL;
		::GetFullPathName(filepath, sizeof(fullpath), fullpath, &phiname);
		::strcpy(g_szPhiSoFileName, phiname);
	}

	return true;
}

static HMODULE
ExLoadDll0(char *pszFilePath, VALUE vFileName)
{
	DEBUGMSG2("ExLoadDll0('%s', '%s')\n", pszFilePath, RSTRING(vFileName)->ptr);

	const HMODULE hModule = ::LoadLibraryEx(pszFilePath, NULL, LOAD_WITH_ALTERED_SEARCH_PATH);
	const DWORD   dwError = ::GetLastError();

	if ( !hModule ) {
		::DeleteFile(pszFilePath);
		::rb_raise(rb_eLoadError, "LoadLibraryEx() failed. [Error:%i].", dwError);
		return NULL;
	}

	const VALUE vInitFunctionName = ::ExGetInitFunctionName(vFileName);

	INITPROC fpInitProc = (INITPROC)::GetProcAddress(hModule, RSTRING(vInitFunctionName)->ptr);

	if ( !fpInitProc ) {
		::FreeLibrary(hModule);
		::DeleteFile(pszFilePath);
		::ExRaiseLoadError(IDS_FAIL_GETPROCADDRESS);
		return NULL;
	}

	DEBUGMSG1("  hModule           = 0x%08X\n", hModule);
	DEBUGMSG1("  vInitFunctionName = '%s'\n", RSTRING(vInitFunctionName)->ptr);
	DEBUGMSG1("  fpInitProc        = 0x%08X\n", fpInitProc);
	(*fpInitProc)();
	DEBUGMSG0("  succeeded\n");

	return hModule;
}

static VALUE
ExGetInitFunctionName(VALUE vFileName)
{
	static const ID id_basename = ::rb_intern("basename");
	static const ID id_sub      = ::rb_intern("sub");

	VALUE vBaseName     = ::rb_funcall(rb_cFile, id_basename, 1, vFileName);
	VALUE vRegexpString = ::rb_str_new2("(\\.so|\\.dll)$");
	VALUE vRegexp       = ::rb_reg_new(RSTRING(vRegexpString)->ptr, RSTRING(vRegexpString)->len, 0);
	VALUE vFeatureName  = ::rb_funcall(vBaseName, id_sub, 2, vRegexp, ::rb_str_new2(""));
	VALUE vFunctionName = ::rb_str_concat(::rb_str_new2("Init_"), vFeatureName);

	return vFunctionName;
}

static bool
ExRequireDll(VALUE vFileName, DWORD dwID)
{
	const DWORD dwOffset = g_DllPtrTable[dwID];
	const DWORD dwSize   = g_DllSizeTable[dwID];

	if ( dwOffset && dwSize ) {
		if ( !::ExReplaceImportTable((void*)dwOffset) ) {
			::ExRaiseLoadError(IDS_FAIL_MODIFY_IMPORT_TABLE);
		}

		char szTemporaryFilePath[MAX_PATH] = "";
		::ExCreateTemporaryFile(szTemporaryFilePath);

		::ExWriteFile(szTemporaryFilePath, (void*)dwOffset, dwSize);

		return ::ExLoadDll(szTemporaryFilePath, vFileName);
	} else {
		return false;
	}
}

static bool
ExRequireScript(VALUE vFileName, DWORD dwID)
{
	const DWORD dwOffset = g_ScriptPtrTable[dwID];
	const DWORD dwSize   = g_ScriptSizeTable[dwID];

	if ( dwOffset && dwSize ) {
		VALUE vScript = ::rb_str_new((char*)dwOffset, dwSize);
		::ExEvalString(vScript, vFileName);
		return true;
	} else {
		::ExRaiseLoadError(IDS_CANNOT_EXECUTE_SCRIPT);
		return false;
	}
}

static void
ExGetSelfFilePath(LPSTR pszBuffer, UINT uiBufferSize)
{
	::GetModuleFileName(NULL, pszBuffer, uiBufferSize);
}

void
ExGetSelfFileName(LPSTR pszBuffer, UINT uiBufferSize)
{
	char szSelfFilePath[MAX_PATH]     = "";
	char szSelfFilePathTemp[MAX_PATH] = "";
	char *pszSelfFileName             = NULL;

	::ExGetSelfFilePath(szSelfFilePath, sizeof(szSelfFilePath));
	::GetFullPathName(szSelfFilePath, sizeof(szSelfFilePathTemp), szSelfFilePathTemp, &pszSelfFileName);
	::strncpy(pszBuffer, pszSelfFileName, uiBufferSize);
}

static VALUE
ExEvalString(VALUE vString, VALUE vFileName)
{
	static const ID    id_eval  = ::rb_intern("eval");
	static const VALUE vBinding = ::rb_const_get(rb_mKernel, ::rb_intern("TOPLEVEL_BINDING"));
	static const VALUE vLine    = INT2FIX(1);
	return ::rb_funcall(rb_mKernel, id_eval, 4, vString, vBinding, vFileName, vLine);
}

static bool
ExFindInsideFile(VALUE vFilename, bool *pbShared, VALUE *pvRealname, DWORD *pdwID)
{
	const string filename     = RSTRING(vFilename)->ptr;
	const string filename_rb  = string(filename).append(".rb");
	const string filename_so  = string(filename).append(".so");
	const string filename_dll = string(filename).append(".dll");

	*pvRealname = Qnil;

	if ( (*pdwID = g_Name2IdTable[filename]) ) {
		if ( g_ScriptPtrTable[*pdwID] ) {
			*pbShared   = false;
			*pvRealname = ::rb_str_new2(filename.c_str());
		} else if ( g_DllPtrTable[*pdwID] ) {
			*pbShared   = true;
			*pvRealname = ::rb_str_new2(filename.c_str());
		}
	} else if ( (*pdwID = g_Name2IdTable[filename_rb]) ) {
		*pbShared   = false;
		*pvRealname = ::rb_str_new2(filename_rb.c_str());
	} else if ( (*pdwID = g_Name2IdTable[filename_so]) ) {
		*pbShared   = true;
		*pvRealname = ::rb_str_new2(filename_so.c_str());
	} else if ( (*pdwID = g_Name2IdTable[filename_dll]) ) {
		*pbShared   = true;
		*pvRealname = ::rb_str_new2(filename_dll.c_str());
	}

	return (*pvRealname != Qnil);
}

static VALUE
ExRequire(VALUE vObject, VALUE vFileName)
{
	bool  bShared   = false;
	VALUE vRealname = Qnil;
	DWORD dwID      = 0;

	::Check_SafeStr(vFileName);

	if ( !::ExFindInsideFile(vFileName, &bShared, &vRealname, &dwID) ) {
		DEBUGMSG1("'%s' was not found in the archive.\n", RSTRING(vFileName)->ptr);
		return ::ExRequireOutside(vFileName);
	}

	DEBUGMSG1("'%s' was found in the archive.\n", RSTRING(vFileName)->ptr);
	DEBUGMSG2("the realname of '%s' is '%s'.\n", RSTRING(vFileName)->ptr, RSTRING(vRealname)->ptr);

	const VALUE features = ::rb_gv_get("$\"");
	if ( ::rb_ary_includes(features, vRealname) ) {
		return Qfalse;
	}
	::rb_ary_push(features, vRealname);

	if ( bShared ) {
		DEBUGMSG1("'%s' is a shared library.\n", RSTRING(vFileName)->ptr);
		::ExRequireDll(vRealname, dwID);
	} else {
		DEBUGMSG1("'%s' is a script.\n", RSTRING(vFileName)->ptr);
		::ExRequireScript(vRealname, dwID);
	}

	return Qtrue;
}


static VALUE
ExRequireOutside(VALUE fname)
{
	VALUE realname, filename;

	if ( !::ExFindOutsideFile(fname, &realname, &filename) ) {
		::rb_raise(rb_eLoadError, "No such file to load -- %s", RSTRING(fname)->ptr);
	}

	DEBUGMSG0("ExRequireOutside(...)\n");
	DEBUGMSG1("  fname    => %s\n", RSTRING(::rb_inspect(fname))->ptr);
	DEBUGMSG1("  realname => %s\n", RSTRING(::rb_inspect(realname))->ptr);
	DEBUGMSG1("  filename => %s\n", RSTRING(::rb_inspect(filename))->ptr);

	const VALUE features = ::rb_gv_get("$\"");
	if ( ::rb_ary_includes(features, filename) ) {
		return Qfalse;
	}
	::rb_ary_push(features, filename);

	const char* ext = ::strrchr(RSTRING(realname)->ptr, '.');

	if ( ext ) {
		if ( ::stricmp(ext, ".rb") == 0 ) {
			::ExLoadOutsideScript(realname, filename);
		} else if ( ::stricmp(ext, ".so") == 0 || ::stricmp(ext, ".dll") == 0 ) {
			::ExLoadOutsideLibrary(realname, filename);
		} else {
			::rb_bug("unknown file type");
		}
	} else {
		::rb_bug("has not extension");
	}

	return Qtrue;
}

static bool
ExFindOutsideFile(VALUE fname, VALUE *realname, VALUE *filename)
{
	if ( *realname = ::rb_find_file(fname) ) {
		*filename = fname;
		return true;
	} else {
		const VALUE name_rb  = ::rb_str_concat(::rb_str_dup(fname), ::rb_str_new2(".rb"));
		const VALUE name_so  = ::rb_str_concat(::rb_str_dup(fname), ::rb_str_new2(".so"));
		const VALUE name_dll = ::rb_str_concat(::rb_str_dup(fname), ::rb_str_new2(".dll"));

		if ( *realname = ::rb_find_file(name_rb) ) {
			*filename = name_rb;
			return true;
		} else if ( *realname = ::rb_find_file(name_so) ) {
			*filename = name_so;
			return true;
		} else if ( *realname = ::rb_find_file(name_dll) ) {
			*filename = name_dll;
			return true;
		} else {
			*realname = Qnil;
			*filename = Qnil;
			return false;
		}
	}
}

static void
ExLoadOutsideScript(VALUE fname, VALUE name)
{
	DEBUGMSG2("ExLoadOutsideScript(%s, %s)\n", RSTRING(fname)->ptr, RSTRING(name)->ptr);

	::rb_load(fname, 0);
}

static void
ExLoadOutsideLibrary(VALUE fname, VALUE name)
{
	DEBUGMSG2("ExLoadOutsideLibrary(%s, %s)\n", RSTRING(fname)->ptr, RSTRING(name)->ptr);

	struct _stat fs = {0};
	if ( ::_stat(RSTRING(fname)->ptr, &fs) != 0 ) {
		::rb_load_fail(RSTRING(fname)->ptr);
	}

	char* buffer = (char*)::xmalloc(fs.st_size);

	FILE* file = ::fopen(RSTRING(fname)->ptr, "rb");
	if ( !file ) {
		::rb_load_fail(RSTRING(fname)->ptr);
	}

	if ( ::fread(buffer, 1, fs.st_size, file) != fs.st_size ) {
		::fclose(file);
		::rb_load_fail(RSTRING(fname)->ptr);
	}

	::fclose(file);

	if ( !::ExReplaceImportTable(buffer) ) {
		::ExRaiseLoadError(IDS_FAIL_MODIFY_IMPORT_TABLE);
	}

	char tmpfile[MAX_PATH] = "";
	::ExCreateTemporaryFile(tmpfile);

	::ExWriteFile(tmpfile, buffer, fs.st_size);

	::free(buffer);

	::ExLoadDll(tmpfile, name);
}

////////////////////////////////////////////////////////////////////////////////

static VALUE
rb_exerb_s_runtime_p(VALUE self)
{
	return Qtrue;
}

////////////////////////////////////////////////////////////////////////////////

VALUE WINAPI
ex_require(const char *fname)
{
	DEBUGMSG1("ex_require('%s')\n", fname);
	return ::ExRequire(Qnil, ::rb_str_new2(fname));
}

VALUE WINAPI
ex_f_require(VALUE obj, VALUE fname)
{
	DEBUGMSG1("ex_f_require('%s')\n", RSTRING(fname)->ptr);
	return ::ExRequire(Qnil, fname);
}

////////////////////////////////////////////////////////////////////////////////
