#! /usr/bin/ruby

#==============================================================================#
# exerb.rb
# $Id: exerb.rb,v 1.20 2002/12/27 06:34:45 yuya Exp $
#==============================================================================#

require 'getoptlong'
require 'exerb/exerb'
require 'exerb/config'

#==============================================================================#
# ExerbCommand Module
module ExerbCommand

  def self.main
    options = parse_argv

    if options[:version]
      print_version
      return
    end

    if options[:help] || ARGV.size < 1
      print_usage
      return
    end

    rbcfile = ARGV[0]
    rbafile = make_filename(rbcfile, 'rba')

    rbc = RecipeFile.new(rbcfile)
    arc = rbc.archive

    corefile = options[:corefile]
    corefile ||= lookup_corefile(options[:corename]) if options[:corename]
    corefile ||= rbc.corefile                        if rbc.corefile
    corefile ||= lookup_corefile(rbc.corename)       if rbc.corename
    corefile ||= lookup_corefile('cui')

    exefile = options[:output]
    exefile ||= rbc.output                           if rbc.output
    exefile ||= make_filename(rbcfile, 'exe')

    if options[:debug]
      File.open(rbafile, 'wb') { |file|
        arc.output(file)
      }
    end

    compress = options[:compress]
    compress ||= rbc.compress

    arcbin  = Exerb::Binary::Archive.new(arc.pack, compress)
    corebin = Exerb::Binary::Core.new_from_file(corefile)
    exebin  = Exerb::Binary::Executable.new(arcbin, corebin)
    exebin.selfcheck

    File.open(exefile, 'wb') { |file|
      exebin.output(file)
    }
  end

  def self.lookup_corefile(name)
    if Exerb::CORE.has_key?(name)
      return Exerb::CORE[name]
    else
      raise "core name not found in exerb/config.rb : #{name}"
    end
  end

  def self.make_filename(filename, extension)
    if /\.rbc$/i =~ filename
      return filename.sub(/\.rbc$/i, '.' + extension)
    else
      return filename + '.' + extension
    end
  end

  def self.parse_argv
    parser = GetoptLong.new
    parser.set_options(
      ['--corename', '-c', GetoptLong::REQUIRED_ARGUMENT],
      ['--corefile', '-C', GetoptLong::REQUIRED_ARGUMENT],
      ['--output',   '-o', GetoptLong::REQUIRED_ARGUMENT],
      ['--compress', '-z', GetoptLong::NO_ARGUMENT],
      ['--debug',    '-g', GetoptLong::NO_ARGUMENT],
      ['--version',  '-V', GetoptLong::NO_ARGUMENT],
      ['--help',     '-h', GetoptLong::NO_ARGUMENT]
    )

    options = {}
    parser.each_option { |name, argument|
      options[name.sub(/^--/, '').intern] = argument
    }

    options[:compress] = !!options[:compress]
    options[:debug]    = !!options[:debug]
    options[:version]  = !!options[:version]
    options[:help]     = !!options[:help]

    return options
  end

  def self.print_usage
    puts "Exerb #{Exerb::VERSION}"
    puts ""

    puts "Usage: exerb [options] rbc-file"
    puts ""

    puts "Options:"
    puts "  -c  --corename  specifies exerb-core name."
    puts "  -C  --corefile  specifies exerb-core file."
    puts "  -o  --output    specifies output file."
    puts "  -z  --compress  compress the archive. (using Ruby/zlib)"
    puts "  -g  --debug     enable debug mode."
    puts "  -V  --version   display version."
    puts "  -h  --help      display this information."
    puts ""

    puts "Registered Core:"
    Exerb::CORE.keys.sort.each { |name|
      puts "  #{name}\t#{Exerb::CORE[name] || '<not found>'}"
    }
    puts ""
  end

  def self.print_version
    puts "Exerb #{Exerb::VERSION}"
  end

  #============================================================================#
  # RecipeFile Class
  class RecipeFile

    def initialize(filepath)
      @basedir = File.dirname(File.expand_path(filepath))
      @archive = Exerb::Archive.new

      @corename = nil
      @corefile = nil
      @output   = nil
      @compress = false

      File.open(filepath) { |file|
        file.to_a
      }.collect { |line|
        line.strip
      }.reject { |line|
        line.empty? || /^#/ =~ line
      }.collect { |line|
        line.split(/\t+/)
      }.each { |type, p1, p2|
        case type.downcase
        when 'kcode'    then @archive.set_kcode(p1)
        when 'corename' then @corename = p1
        when 'corefile' then @corefile = File.expand_path(p1, @basedir)
        when 'output'   then @output   = File.expand_path(p1, @basedir)
        when 'compress' then @compress = yesno(p1)
        when 'script'   then @archive.add_script(p1, File.expand_path(p2 || p1, @basedir))
        when 'library'  then @archive.add_dll(p1, File.expand_path(p2 || p1, @basedir))
        else raise "syntax error : #{type}"
        end
      }
    end

    attr_reader :archive, :corename, :corefile, :output, :compress

    def yesno(str)
      case str
      when 'yes' then true
      when 'no'  then false
      else raise 'yes or no'
      end
    end
    private :yesno

  end # RecipeFile

end # ExerbCommand

#==============================================================================#

ExerbCommand.main

#==============================================================================#
#==============================================================================#
