/**
 * @file osdepend.h
 * @brief OS依存処理
 * @author BananaJinn
 * @version $Id: osdepend.c,v 1.3 2010/11/05 17:24:03 bananajinn Exp $
 * 円盤複写屋
 * Copyright (C) 2004-2010 BananaJinn<banana@mxh.mesh.ne.jp>
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/select.h>
#include <sys/wait.h>
#include <fcntl.h>
#include "osdepend.h"
#include "ebstring.h"

/**
 * @brief プロセス作成
 * @param[in] cmdline コマンドライン
 * @param[out] pid_ret プロセスID
 * @param[out] stdin_ret 標準入力ディスクリプタ
 * @param[out] stdout_ret 標準出力ディスクリプタ
 * @param[out] stderr_ret 標準エラー出力ディスクリプタ
 * @return 成功時は TRUE を返す。
 */
BOOL OSDCreateProcess(const char *cmdline, OSD_PID *pid_ret,
					  OSD_FD *stdin_ret, OSD_FD *stdout_ret,
					  OSD_FD *stderr_ret)
{
	int fd1[2], fd2[2], fd3[2];
	pid_t pid;

	if(stdin_ret != NULL){
		if(pipe(fd1) < 0){
			return FALSE;
		}
	}
	if(stdout_ret != NULL){
		if(pipe(fd2) < 0){
			if(stdin_ret != NULL){
				close(fd1[0]); close(fd1[1]);
			}
			return FALSE;
		}
	}
	if(stderr_ret != NULL){
		if(pipe(fd3) < 0){
			if(stdin_ret != NULL){
				close(fd1[0]); close(fd1[1]);
			}
			if(stdout_ret != NULL){
				close(fd2[0]); close(fd2[1]);
			}
			return FALSE;
		}
	}

	pid = fork();
	if(pid < 0){
		if(stdin_ret != NULL){
			close(fd1[0]); close(fd1[1]);
		}
		if(stdout_ret != NULL){
			close(fd2[0]); close(fd2[1]);
		}
		if(stderr_ret != NULL){
			close(fd3[0]); close(fd3[1]);
		}
		return FALSE;
	}
	if(pid == 0){
		/* 子 */
		if(stdin_ret != NULL){
			close(fd1[1]); /* 標準入力の書き込み側を閉じる */
			dup2(fd1[0], STDIN_FILENO); /* 標準入力の読み込み側を複製 */
		}
		if(stdout_ret != NULL){
			close(fd2[0]); /* 標準出力の読み込み側を閉じる */
			dup2(fd2[1], STDOUT_FILENO); /* 標準出力の書き込み側を複製 */
		}
		if(stderr_ret != NULL){
			close(fd3[0]); /* 標準エラー出力の読み込み側を閉じる */
			dup2(fd3[1], STDERR_FILENO); /* 標準エラー出力の書き込み側を複製 */
		}
		if(execlp("sh", "sh", "-c", cmdline, NULL) < 0){
			exit(1);
		}
	}
	if(stdin_ret != NULL){
		close(fd1[0]); /* 標準入力の読み込み側を閉じる */
		*stdin_ret = fd1[1];
	}
	if(stdout_ret != NULL){
		close(fd2[1]); /* 標準出力の書き込み側を閉じる */
		*stdout_ret = fd2[0];
	}
	if(stderr_ret != NULL){
		close(fd3[1]); /* 標準エラー出力の書き込み側を閉じる */
		*stderr_ret = fd3[0];
	}
	*pid_ret = pid;
	return TRUE;
}

/**
 * @brief プロセスを強制終了する
 * @param[in] pid プロセスID
 * return 成功時は TRUE を返す。
 */
BOOL OSDTerminateProcess(OSD_PID pid)
{
	return kill(pid, SIGTERM) == 0;
}

/**
 * @brief プロセスの終了を待つ
 * @param[in] pid プロセスID
 * @param[out] exitcode_ret 終了コード
 * @return 成功時は TRUE を返す。
 */
BOOL OSDWaitProcess(OSD_PID pid, int *exitcode_ret)
{
	int status=0;
	
	if(pid > 0){
		waitpid(pid, &status, 0);
	}
	else{
		wait(&status);
	}

	if(WIFSIGNALED(status)){
		printf("signaled %d\n", WTERMSIG(status));
	}
	if(exitcode_ret != NULL){
		*exitcode_ret = WEXITSTATUS(status);
	}

	return TRUE;
}

/**
 * @brief ディスクリプタから指定サイズの読み込みを行う。
 * @param[in] fd_stdout 標準出力ディスクリプタ
 * @param[in] fd_stderr 標準エラー出力ディスクリプタ
 * @param[out] buff_stdout 標準出力バッファ
 * @param[in] size_stdout 標準出力バッファサイズ
 * @param[out] stderr_ret 標準エラー出力返却用文字列ポインタのアドレス
 * @return 読み込んだサイズ(バイト数)を返す。
 */
int OSDRead(OSD_FD fd_stdout, OSD_FD fd_stderr,
			unsigned char *buff_stdout, int size_stdout,
			char **stderr_ret)
{
	struct timeval select_timeout;
	fd_set fdset;
	int fd_max = 0;
	int ret;
	int read_len = 0;
	int offset_stdout = 0;
	char buff_stderr[256+1];
	BOOL done = FALSE;
	
	if(fd_stdout >= 0){
		fcntl(fd_stdout, F_SETFL, O_NONBLOCK);
	}
	if(fd_stderr >= 0){
		fcntl(fd_stderr, F_SETFL, O_NONBLOCK);
		if(stderr_ret != NULL){
			*stderr_ret = NULL;
		}
	}
	
	while(!done && (fd_stdout >= 0 || fd_stderr >= 0)){
		select_timeout.tv_sec = 0;
		select_timeout.tv_usec = 5000000;
		FD_ZERO(&fdset);
		fd_max = 0;
		if(fd_stdout >= 0){
			FD_SET(fd_stdout, &fdset);
			if(fd_max < fd_stdout) fd_max = fd_stdout;
		}
		if(fd_stderr >= 0){
			FD_SET(fd_stderr, &fdset);
			if(fd_max < fd_stderr) fd_max = fd_stderr;
		}
		ret = select(fd_max+1, &fdset, NULL, NULL, &select_timeout);
		if(ret < 0){
			break;
		}
		if(fd_stdout >= 0){
			if(FD_ISSET(fd_stdout, &fdset)){
				read_len = read(fd_stdout, buff_stdout+offset_stdout,
								size_stdout-offset_stdout);
				if(read_len > 0){
					offset_stdout += read_len;
					if(offset_stdout >= size_stdout){
						done = TRUE;
					}
				}
				else{
					fd_stdout = -1;
				}
			}
		}
		if(fd_stderr >= 0){
			if(FD_ISSET(fd_stderr, &fdset) && stderr_ret != NULL){
				read_len = read(fd_stderr, buff_stderr, sizeof(buff_stderr)-1);
				if(read_len > 0){
					buff_stderr[read_len] = '\0';
					*stderr_ret = EbStringAppend(*stderr_ret, buff_stderr);
				}
				else{
					fd_stderr = -1;
				}
			}
		}
	}
	return offset_stdout;
}

/**
 * @brief ディスクリプタを閉じる。
 * @param[in] fd ディスクリプタ
 */
void OSDFDClose(OSD_FD fd)
{
	close(fd);
}
