/**
 * @file filedlg.c
 * @author BananaJinn
 * @version $Id: filedlg.c,v 1.5 2010/11/01 14:34:11 bananajinn Exp $
 * 円盤複写屋
 * Copyright (C) 2004-2010 BananaJinn<banana@mxh.mesh.ne.jp>
 */
#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "filedlg.h"
#include "text.h"

/**
 * @brief ファイル/フォルダ選択ダイアログを表示する。
 * @param[in] action GtkFileChooserDialogに対するアクション
 * @param[out] filename 選択されたファイル名
 * @param[in] size ファイル名格納領域サイズ(バイト数)
 * @param[in] suffix 拡張子
 * @retval FDR_OK ファイルが選択された
 * @retval FDR_CANCEL ファイル選択がキャンセルされた
 */
static int FileChooserDialog(
	GtkFileChooserAction action, char *filename, int size, const char *suffix)
{
	GtkWidget *file_selector;
	const char *title = "";
	GtkFileFilter *filter;
	char *mask = NULL;
	int retCode = FDR_CANCEL;
	const char *button = GTK_STOCK_OPEN;

	if(action == GTK_FILE_CHOOSER_ACTION_OPEN){
		title = GT("open file");
		button = GTK_STOCK_OPEN;
	}
	else if(action == GTK_FILE_CHOOSER_ACTION_SAVE){
		title = GT("save file");
		button = GTK_STOCK_SAVE;
	}
	else if(action == GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER){
		title = GT("select folder");
		button = GTK_STOCK_OPEN;
	}
	
	if(suffix!=NULL){
		mask=(char *)malloc(strlen(suffix)+1+2);
	}
	if(mask!=NULL){
		strcpy(mask, "*.");
		strcat(mask, suffix);
	}
	file_selector = gtk_file_chooser_dialog_new(
		title, NULL, action,
		GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
		button, GTK_RESPONSE_ACCEPT,
		NULL);
	if(mask != NULL){
		filter = gtk_file_filter_new();
		gtk_file_filter_set_name(filter, mask);
		gtk_file_filter_add_pattern(filter, mask);
		gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(file_selector), filter);
	}
	filter = gtk_file_filter_new();
	gtk_file_filter_set_name(filter, "All Files");
	gtk_file_filter_add_pattern(filter, "*.*");
	gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(file_selector), filter);

	if(gtk_dialog_run(GTK_DIALOG(file_selector)) == GTK_RESPONSE_ACCEPT){
		char *selected = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(file_selector));
		strncpy(filename, selected, size);
		filename[size-1]='\0';
		g_free(selected);
		retCode = FDR_OK;
	}
	gtk_widget_destroy(file_selector);
	if(mask != NULL){
		free(mask);
	}
	return retCode;
}

/**
 * @brief ファイル選択ダイアログを表示する。
 * @param[in] bOpen ファイルを開く場合は TRUE、ファイル保存の場合は FALSE
 * @param[out] filename 選択されたファイル名
 * @param[in] size ファイル名格納領域サイズ(バイト数)
 * @param[in] suffix 拡張子
 * @retval FDR_OK ファイルが選択された
 * @retval FDR_CANCEL ファイル選択がキャンセルされた
 */
int FileDialog(int bOpen, char *filename, int size, const char *suffix)
{
	return FileChooserDialog(
		bOpen ? GTK_FILE_CHOOSER_ACTION_OPEN : GTK_FILE_CHOOSER_ACTION_SAVE,
		filename, size, suffix);
}

/**
 * @brief フォルダ選択ダイアログを表示する。
 * @param[out] filename 選択されたファイル名
 * @param[in] size ファイル名格納領域サイズ(バイト数)
 * @retval FDR_OK ファイルが選択された
 * @retval FDR_CANCEL ファイル選択がキャンセルされた
 */
int FolderDialog(char *filename, int size)
{
	return FileChooserDialog(
		GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER, filename, size, NULL);
}


