/**
 * @file offsetiso.c
 * @brief ISOイメージを外側にずらす
 * @author BananaJinn
 * @version $Id: offsetiso.c,v 1.6 2010/11/01 14:34:11 bananajinn Exp $
 * 円盤複写屋
 * Copyright (C) 2004-2006 BananaJinn<banana@mxh.mesh.ne.jp>.
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "mem.h"
#include "cmd.h"
#include "ui.h"
#include "text.h"
#include "log.h"

#define BLOCKKIND_VOLUME 0
#define BLOCKKIND_PATHTABLE_L_PVD 1
#define BLOCKKIND_PATHTABLE_M_PVD 2
#define BLOCKKIND_PATHTABLE_L_SVD 3
#define BLOCKKIND_PATHTABLE_M_SVD 4
#define BLOCKKIND_DIRECTORY 5

#define GET4BYTES(p,l) (l) ? Get4bytesLE(p) : Get4bytes(p)
#define GET2BYTES(p,l) (l) ? Get4bytesLE(p) : Get4bytes(p)
#define SET4BYTES(p,v,l) (l) ? Set4bytesLE(p,v) : Set4bytes(p,v)
#define SET2BYTES(p,v,l) (l) ? Set2bytesLE(p,v) : Set2bytes(p,v)


/**
 * @brief 置き換えアドレス保持構造体
 */
typedef struct {
  int kind;	/**< データの種類 */
  DWORD lba;	/**< アドレス */
  DWORD size;	/**< サイズ */
} REPLBA;


static DWORD gOffset = 0;	/**< 外側にずらすブロック数 */
static REPLBA *gRepLBA = NULL;	/**< 置き換えアドレス配列 */
static int gNumRepLBA = 0;	/**< 置き換えアドレス配列サイズ */
static int gNext = -1;		/**< 次に参照する置き換えアドレス */
static BYTE gVolumeDesc = 0;	/**< ボリューム記述子の種別 */


/**
 * @brief 置き換えアドレス配列に追加
 * @param[in] kind データの種類
 * @param[in] lba アドレス
 * @param[in] size サイズ
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 * @retval	RET_MEMERR メモリエラー
 */
static int AddRepLBA(int kind, DWORD lba, DWORD size)
{
  int i;

  lba += gOffset;

  gRepLBA = MemResize(gRepLBA, sizeof(REPLBA)*(gNumRepLBA+1));
  if(gRepLBA == NULL){
    return RET_MEMERR;
  }
  for(i=0; i<gNumRepLBA; i++){
    if(lba < gRepLBA[i].lba){
      break;
    }
    else if(lba == gRepLBA[i].lba){
      return RET_OK;
    }
  }

  if((gNumRepLBA-i) > 0){
    memmove(gRepLBA+i+1, gRepLBA+i, (gNumRepLBA-i)*sizeof(REPLBA));
  }
  gRepLBA[i].kind = kind;
  gRepLBA[i].lba = lba;
  gRepLBA[i].size = size;
  gNumRepLBA++;

  if(gNext < 0){
    gNext = 0;
  }
  return RET_OK;
}


/**
 * @brief 置き換えアドレス配列を解放
 */
static void FreeRepLBA()
{
  MemFree(gRepLBA);
  gRepLBA = NULL;
  gNumRepLBA = 0;
}

/**
 * @brief ディレクトリレコードをOFFSET
 * @param[in] top ブロック先頭ポインタ
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 */
static int OffsetLbaAtDirectoryRecord(BYTE *top)
{
  DWORD lba;
  DWORD size;
  int ret;
  int i;

  lba = Get4bytesLE(top+0x02);
  size = Get4bytesLE(top+0x0a);
  if(top[0x19] & 0x02){
    /* ファイルフラグがディレクトリの場合 */
    i = 0;
    while(size > 0){
      ret = AddRepLBA(BLOCKKIND_DIRECTORY, lba+i,
		      size>0x800 ? 0x800:size);
      if(ret != RET_OK)
	return ret;
      size -= size>0x800 ? 0x800:size;
      i++;
    }
  }
  Set4bytesLE(top+0x02, lba+gOffset);
  Set4bytes(top+0x06, lba+gOffset);

  return RET_OK;
}


/**
 * @brief パステーブルレコードをOFFSET
 * @param[in] top ブロック先頭ポインタ
 * @param[in] kind ブロック種別(BLOCKKIND_PATHTABLE_[LM]_[PS]VD)
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 */
static int OffsetLbaAtPathTableRecord(BYTE *top, int kind)
{
  DWORD lba;
  //int ret;
  BOOL le = FALSE;

  le = (kind==BLOCKKIND_PATHTABLE_L_PVD) || (kind==BLOCKKIND_PATHTABLE_L_SVD);
  lba = GET4BYTES(top+0x02, le);
#if 0
  ret = AddRepLBA(BLOCKKIND_DIRECTORY, lba);
  if(ret != RET_OK)
    return ret;
#endif
  SET4BYTES(top+0x02, lba+gOffset, le);

  return RET_OK;
}


/**
 * @brief ボリュームブロックをOFFSET
 * @param[in] top ブロック先頭ポインタ
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 */
static int OffsetLbaAtVolume(BYTE *top)
{
  DWORD size;
  DWORD blocks, i;
  DWORD lba;
  int ret;

  if(top[0] == 0x03){
    /* ?? */
    lba = Get4bytesLE(top+0x48);
    Set4bytesLE(top+0x48, lba+gOffset);
    Set4bytes(top+0x4c, lba+gOffset);
  }
  else if((top[0] == 0x01) || (top[0] == 0x02)){
    /* PVD/SVD */

    /* パステーブルサイズ(bytes) */
    size = Get4bytes(top+0x88);
    blocks = (size+2047)/2048;
    
    /* L形パステーブルLBA */
    lba = Get4bytesLE(top+0x8c);
    for(i=0; i<blocks; i++){
      ret = AddRepLBA(top[0] == 1 ?
		      BLOCKKIND_PATHTABLE_L_PVD : BLOCKKIND_PATHTABLE_L_SVD,
		      lba+i, size);
      if(ret != RET_OK)
	return ret;
    }
    Set4bytesLE(top+0x8c, lba+gOffset);
    DebugLog("L type PathTable : 0x%08lX -> 0x%08lX\n",
	     lba, lba+gOffset);
    
    /* 任意L型パステーブルLBA */
    lba = Get4bytesLE(top+0x90);
    if(lba != 0x00000000){
      for(i=0; i<blocks; i++){
	ret = AddRepLBA(top[0] == 1 ?
			BLOCKKIND_PATHTABLE_L_PVD : BLOCKKIND_PATHTABLE_L_SVD,
			lba+i, size);
	if(ret != RET_OK)
	  return ret;
      }
      Set4bytesLE(top+0x90, lba+gOffset);
      DebugLog("L type PathTable2 : 0x%08lX -> 0x%08lX\n",
	       lba, lba+gOffset);
    }
    
    /* M形パステーブルLBA */
    lba = Get4bytes(top+0x94);
    for(i=0; i<blocks; i++){
      ret = AddRepLBA(top[0] == 1 ?
		      BLOCKKIND_PATHTABLE_M_PVD : BLOCKKIND_PATHTABLE_M_SVD,
		      lba+i, size);
      if(ret != RET_OK)
	return ret;
    }
    Set4bytes(top+0x94, lba+gOffset);
    DebugLog("M type PathTable : 0x%08lX -> 0x%08lX\n",
	     lba, lba+gOffset);
    
    /* 任意M型パステーブルLBA */
    lba = Get4bytes(top+0x98);
    if(lba != 0x00000000){
      for(i=0; i<blocks; i++){
	ret = AddRepLBA(top[0] == 1 ?
			BLOCKKIND_PATHTABLE_M_PVD : BLOCKKIND_PATHTABLE_M_SVD,
			lba+i, size);
	if(ret != RET_OK)
	  return ret;
      }
      Set4bytes(top+0x98, lba+gOffset);
      DebugLog("M type PathTable2 : 0x%08lX -> 0x%08lX\n",
	       lba, lba+gOffset);
    }
    
    ret = OffsetLbaAtDirectoryRecord(top+0x9c);
    if(ret != RET_OK)
      return ret;
  }

  return RET_OK;
}

/**
 * @brief パステーブルブロックをOFFSET
 * @param[in] top ブロック先頭ポインタ
 * @param[in] kind ブロック種別(BLOCKKIND_PATHTABLE_[LM]_[PS]VD)
 * @param[in] size パステーブルサイズ(バイト数)
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 */
static int OffsetLbaAtPathTable(BYTE *top, int kind, DWORD size)
{
  DWORD reclen = 0;
  DWORD i = 0;
  int ret;

  while(i < size){
    ret = OffsetLbaAtPathTableRecord(top+i, kind);
    if(ret != RET_OK){
      return ret;
    }
    reclen = top[i]+8;
    if(reclen & 1) reclen++;
    i += reclen;
  }

  return RET_OK;
}

/**
 * @brief ディレクトリブロックをOFFSET
 * @param[in] top ブロック先頭ポインタ
 * @param[in] size ディレクトリサイズ(バイト数)
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 */
static int OffsetLbaAtDirectory(BYTE *top, DWORD size)
{
  DWORD reclen = 0;
  DWORD i = 0;
  int ret;

  while(i < size){
    reclen  = top[i];
    if(reclen == 0){
      break;
    }
    ret = OffsetLbaAtDirectoryRecord(top+i);
    if(ret != RET_OK){
      return ret;
    }
    i += reclen;
  }

  return RET_OK;
}

/**
 * @brief 種類に応じて分岐
 * @param[in] top 1ブロックの先頭ポインタ
 * @param[in] 種類
 * @param[in] size サイズ(バイト数)
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 */
static int OffsetLBA(BYTE *top, int kind, DWORD size)
{
  int ret;

  switch(kind){
  case BLOCKKIND_VOLUME:
    ret = OffsetLbaAtVolume(top);
    break;
  case BLOCKKIND_PATHTABLE_L_PVD:
  case BLOCKKIND_PATHTABLE_L_SVD:
  case BLOCKKIND_PATHTABLE_M_PVD:
  case BLOCKKIND_PATHTABLE_M_SVD:
    ret = OffsetLbaAtPathTable(top, kind, size);
    break;
  case BLOCKKIND_DIRECTORY: 
    ret = OffsetLbaAtDirectory(top, size);
    break;
  default:
    return RET_NG;
  }
  return ret;
}
  

/**
 * @brief ISO9660イメージを外側にずらすブロック数を設定する。
 * @param[in] offset ブロック数
 */
void SetOffset(DWORD offset)
{
  gOffset = offset;

  /* 初期化 */
  FreeRepLBA();
  gNext = -1;
  gVolumeDesc = 0;
}


/**
 * @brief ISO9660イメージを外側にずらすブロック数を取得する。
 * @return ブロック数
 */
DWORD GetOffset()
{
  return gOffset;
}


/**
 * @brief ISO9660イメージを外側にずらす
 * @param[in] data データ
 * @param[in] lba データのアドレス
 * @param[in] blocksize 1ブロックのバイト数(2048)
 * @param[in] len データの長さ(ブロック数)
 */
int OffsetISO(BYTE *data, DWORD lba, DWORD blocksize, DWORD len)
{
  DWORD i;
  DWORD curr_lba;
  BYTE *top;
  int ret;

  if(gOffset == 0){
    return RET_OK;
  }

  for(i=0; i<len; i++){
    top = data+i*blocksize;
    if(gVolumeDesc < 0xff){
      if(!memcmp(top+1, "CD001", 5)){
	gVolumeDesc = top[0];
	ret = OffsetLBA(top, BLOCKKIND_VOLUME, 0);
	if(ret != RET_OK){
	  FreeRepLBA();
	  return ret;
	}
      }
    }
    else if(gNext >= 0){
      curr_lba = lba+i;
      while(curr_lba > gRepLBA[gNext].lba){
	if(gNext >= gNumRepLBA-1){
	  /* 終了(これ以降のデータにはアドレス置き換えがない) */
	  gNext = -1;
	  return RET_OK;
	}
	gNext++;
      }
      if(curr_lba == gRepLBA[gNext].lba){
	ret = OffsetLBA(top, gRepLBA[gNext].kind, gRepLBA[gNext].size);
	if(ret != RET_OK){
	  FreeRepLBA();
	  return ret;
	}
	if(gNext >= gNumRepLBA-1){
	  /* 終了(これ以降のデータにはアドレス置き換えがない) */
	  gNext = -1;
	  return RET_OK;
	}
      }
    }
  }
  
  return RET_OK;
}
