/**
 * @file mkisofs.c
 * @brief mkisofsによるISOイメージ作成
 * @auther BananaJinn
 * @version $Id: mkisofs.c,v 1.3 2010/11/05 17:24:03 bananajinn Exp $
 * 円盤複写屋
 * Copyright (C) 2004-2010 BananaJinn<banana@mxh.mesh.ne.jp>.
 */
#if defined(WIN32)
# include <io.h>
# define R_OK 0
#endif
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "mkisofs.h"
#include "drive.h"
#include "mem.h"
#include "ebstring.h"
#include "ui.h"
#include "text.h"
#include "log.h"

#if defined(WIN32)
# define MKISOFS_CMD "genisoimage.exe"
#else
# define MKISOFS_CMD "mkisofs"
#endif

#define VIDEO_OPTION "-dvd-video"

/**
 * @brief エラーメッセージ設定
 * @param[in] mif MKISOFS構造体
 * @param[in] error_message エラーメッセージ
 */
static void SetErrorMessage(MKISOFS *mif, const char *error_message)
{
	if(mif->error_message != NULL){
		EbStringFree(mif->error_message);
	}
	mif->error_message = EbStringNew(error_message);
}

/**
 * @brief エラーメッセージ取得
 * @param[in] mif MKISOFS構造体
 * @return エラーメッセージを返す。設定されていない場合は NULL を返す。
 */
const char *MIFGetErrorMessage(MKISOFS *mif)
{
	return mif->error_message;
}


/**
 * @brief 構造体初期化
 * @param[out] mif 初期化する構造体のポインタ
 * @retval RET_OK 正常終了
 */
int MIFInitialize(MKISOFS *mif)
{
	memset(mif, 0, sizeof(MKISOFS));
	return RET_OK;
}

/**
 * @brief 構造体解放
 * @param[in] mif 解放する構造体のポインタ
 * @retval RET_OK 正常終了
 * @retval RET_NG エラー
 */
int MIFFree(MKISOFS *mif)
{
	int ret = RET_OK;
	BOOL success;
	int exitcode = 0;

	if(mif != NULL){
		if(mif->pid > 0){
			OSDTerminateProcess(mif->pid);
			success = OSDWaitProcess(mif->pid, &exitcode);
			if(!success || exitcode != 0){
				ret = RET_NG;
			}
			mif->pid = 0;
		}
		if(mif->fd_stdout >= 0){
			OSDFDClose(mif->fd_stdout);
			mif->fd_stdout = -1;
		}
		if(mif->fd_stderr >= 0){
			OSDFDClose(mif->fd_stderr);
			mif->fd_stderr = -1;
		}
		EbStringFree(mif->src_path);
		EbStringFree(mif->volume);
		EbStringFree(mif->error_message);
	}
	return ret;
}

/**
 * @brief ISOイメージを作成する元のディレクトリパスを設定する
 * @param[in] mif MKISOFS構造体
 * @param[in] src_path 設定するディレクトリパス
 * @retval RET_OK 正常終了
 * @retval RET_NG エラー
 */
int MIFSetSrcPath(MKISOFS *mif, const char *src_path)
{
	char *check_path;
	char *wp;
	int ret;

	if(mif != NULL){
		if(mif->src_path != NULL){
			EbStringFree(mif->src_path);
		}
		mif->src_path = EbStringNew(src_path);
		/* VIDEO_TSフォルダがあるか確認 */
		check_path = EbStringNew(src_path);
		check_path = EbStringAppendPath(check_path, "VIDEO_TS");
		mif->video = FALSE;
		if(access(check_path,
#if defined(WIN32)
				  R_OK
#else
				  R_OK|X_OK
#endif
				  ) == 0){
			ret = UIDispMessage(MSG_VIDEO_MODE, UIDMT_QUESTION);
			if(ret == UIDMRET_OK){
				mif->video = TRUE;
			}
		}
		if(mif->video){
			/* フォルダ名をボリュームラベルに */
			wp = strrchr(check_path, PATH_SEPARATOR);
			if(wp != NULL){
				*wp = '\0';
				wp = strrchr(check_path, PATH_SEPARATOR);
				if(wp != NULL){
					wp++;
				}
				else{
					wp = check_path;
				}
				mif->volume = EbStringNew(wp);
			}
		}
		EbStringFree(check_path);
	}
	return RET_OK;
}

/**
 * @brief mkisofsオプションを追加する
 * @param[in] mif MKISOFS構造体
 * @param[in] cmdline コマンドライン
 * @return 追加したコマンドライン文字列を返す。
 */
static char *AppendMkisofsOptions(MKISOFS *mif, char *cmdline)
{
	cmdline = EbStringAppend(cmdline, " -J -r");
	cmdline = EbStringAppend(cmdline, " -input-charset UTF-8");
	if(mif->video){
		cmdline = EbStringAppendWithFormat(cmdline, " %s", VIDEO_OPTION);
		if(mif->volume != NULL){
			cmdline = EbStringAppendWithFormat(cmdline, " -V \"%s\"",
											   mif->volume);
		}
	}
	return cmdline;
}
	

/**
 * @brief ISOイメージのブロックサイズを取得
 * @param[in] mif MKISOFS構造体
 * @return ブロックサイズを返す。取得できない場合は 0 を返す。
 */
DWORD MIFGetBlockCount(MKISOFS *mif)
{
	char *cmdline = NULL;
	BOOL success = TRUE;
	OSD_PID pid;
	OSD_FD sout;
	OSD_FD serr;
	BYTE buff[256];
	int read_length = 0;
	int exitcode = 0;
	char *stderr_string = NULL;

	cmdline = EbStringNewWithFormat("%s -print-size", MKISOFS_CMD);
	cmdline = AppendMkisofsOptions(mif, cmdline);
	cmdline = EbStringAppendWithFormat(cmdline, " %s", mif->src_path);
	DebugLog("run [%s]\n", cmdline);
	success = OSDCreateProcess(cmdline, &pid, NULL, &sout, &serr);
	if(!success){
		cmdline = EbStringFree(cmdline);
		return 0;
	}
	cmdline = EbStringFree(cmdline);

	memset(buff, 0, sizeof(buff));
	read_length = OSDRead(sout, serr, buff, sizeof(buff)-1, &stderr_string);
	success = OSDWaitProcess(pid, &exitcode);
	DebugLog("exitcode=%d, read_length=%d\n", exitcode, read_length);
	if(!success || read_length <= 0){
		SetErrorMessage(mif, stderr_string);
		EbStringFree(stderr_string);
		return 0;
	}
	DebugLog("block count=%ld\n", atol((char *)buff));
	EbStringFree(stderr_string);
	return (DWORD)atol((char *)buff);
}

/**
 * @brief mkisofs開始
 * @param[in] mif MKISOFS構造体
 * @retval RET_OK 成功
 * @retval RET_NG 失敗
 */
static int StartMkisofs(MKISOFS *mif)
{
	char *cmdline = NULL;
	BOOL success = TRUE;

	cmdline = EbStringNewWithFormat("%s -quiet", MKISOFS_CMD);
	cmdline = AppendMkisofsOptions(mif, cmdline);
	cmdline = EbStringAppendWithFormat(cmdline, " %s", mif->src_path);
	DebugLog("run [%s]\n", cmdline);
	success = OSDCreateProcess(cmdline, &mif->pid, NULL,
							   &mif->fd_stdout, &mif->fd_stderr);
	if(!success){
		cmdline = EbStringFree(cmdline);
		return RET_NG;
	}
	cmdline = EbStringFree(cmdline);

	mif->started = TRUE;
	return RET_OK;
}

/**
 * @brief mkisofsの標準出力を読み込む
 * @param[in] mif MKISOFS構造体
 * @param[out] buff バッファ
 * @param[in] len 読み込むバイト数
 * @retval RET_OK 正常終了
 * @retval RET_NG エラー
 */
int MIFRead(MKISOFS *mif, BYTE *buff, DWORD len)
{
	int ret;
	int read_length = 0;
	char *err = NULL;

	if(!mif->started){
		ret = StartMkisofs(mif);
		if(ret != RET_OK){
			return ret;
		}
		DebugLog("mkisofs started\n");
	}

	read_length = OSDRead(mif->fd_stdout, mif->fd_stderr,
						  buff, len, &err);
	if(read_length < len){
		char *message = EbStringNewWithFormat(MSG_MKISOFS_ERROR_, err);
		DebugLog("read_length=%d len=%ld\n", read_length, len);
		DebugLog("stderr=%s\n", err);
		UIDispMessage(message, UIDMT_ERROR);
		EbStringFree(message);
		EbStringFree(err);
		return RET_NG;
	}
	EbStringFree(err);
	return RET_OK;
}
