/**
 * @file cuesheet.h
 * @brief 円盤を複写
 * @author BananaJinn
 * @version $Id: cuesheet.c,v 1.4 2010/11/01 14:34:11 bananajinn Exp $
 * 円盤複写屋
 * Copyright (C) 2004-2010 BananaJinn<banana@mxh.mesh.ne.jp>.
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "mem.h"
#include "aspi.h"
#include "struct.h"
#include "cmd.h"
#include "log.h"
#include "discinfo.h"

#define SEND_GAPDATA 1

/**
 * Next Writable Address を取得
 * @param[in] drive 書込装置構造体
 * @param[in] track_num トラック番号
 * @param[out] nwa_ret NWA
 * @retval RET_OK 正常終了
 * @retval RET_NG エラー
 */
static int GetNWA(CMDDRIVE *drive, WORD track_num, DWORD *nwa_ret)
{
	DWORD nwa = -150;
	int ret;
	struct _TRACKINFO *ti;
	
	if(REALDRIVE(drive)){
		ret = SendReadTrackInfo(drive, track_num);
		if(ret!=RET_OK){
			DispCommandError(drive);
			return ret;
		}
		ti = (struct _TRACKINFO *)drive->data_buf;
		nwa = Get4bytes(ti->next_writable_addr);
		if(nwa == (DWORD)0){
			nwa -= 150;
		}
	}
	*nwa_ret = nwa;
	return RET_OK;
}

/**
 * 新しいキューシート用のメモリを確保
 * @param[in,out] cs キューシート
 * @param[in,out] num_cs キューシートのエントリ数
 * @param[in] num 確保するエントリ数
 * @return 確保した新しい領域のポインタを返す。エラーの場合は NULL を返す。
 */
static struct _CUESHEET *NewSheet(struct _CUESHEET **cs, int *num_cs, int num)
{
	struct _CUESHEET *result = NULL;
	*cs = (struct _CUESHEET *)MemResize(
		*cs, (*num_cs+num)*sizeof(struct _CUESHEET));
	if(*cs != NULL){
		result = &(*cs)[*num_cs];
		*num_cs += num;
	}
	return result;
}

/**
 * メディアカタログ番号設定
 * @param[in,out] cs キューシート
 * @param[in,out] num_cs キューシートのエントリ数
 * @param[in] discinfo ディスク情報
 * @retval RET_OK	正常終了
 * @retval RET_MEMERR メモリ不足
 */
static int SetMCN(struct _CUESHEET **cs, int *num_cs, CPDISCINFO *discinfo)
{
	struct _CUESHEET *csp = NewSheet(cs, num_cs, 2);
	DebugLog("CreateCueSheet: Media catalog number.\n");
	if(csp == NULL){
		return RET_MEMERR;
	}

	csp->ctladr = CSADR_MCN;
	memcpy(&csp->tno, discinfo->media_catalog_number, 7);
	csp++;
	csp->ctladr = CSADR_MCN;
	memcpy(&csp->tno, discinfo->media_catalog_number+7, 7);
	return RET_OK;
}


/**
 * ISRC設定
 * @param[in,out] cs キューシート
 * @param[in,out] num_cs キューシートのエントリ数
 * @param[in] track_num トラック番号
 * @param[in] cpti トラック情報
 * @retval RET_OK	正常終了
 * @retval RET_MEMERR メモリ不足
 */
static int SetISRC(struct _CUESHEET **cs, int *num_cs, WORD track_num,
				   CPTRACKINFO *cpti)
{
	struct _CUESHEET *csp = NewSheet(cs, num_cs, 2);
	DebugLog("CreateCueSheet: ISRC.\n");
	if(csp == NULL){
		return RET_MEMERR;
	}

	csp->ctladr = CSADR_ISRC;
	csp->tno = (BYTE)track_num;
	memcpy(&csp->index, cpti->isrc, 6);
	csp++;
	csp->ctladr = CSADR_ISRC;
	csp->tno = (BYTE)track_num;
	memcpy(&csp->index, cpti->isrc+6, 6);
	return RET_OK;
}


/**
 * Lead-in設定
 * @param[in,out] cs キューシート
 * @param[in,out] num_cs キューシートのエントリ数
 * @param[in] cpti トラック情報
 * @param[in] cdtext CD-TEXT有効フラグ
 * @param[in] nwa NWA
 * @retval RET_OK	正常終了
 * @retval RET_MEMERR メモリ不足
 */
static int SetLeadIn(struct _CUESHEET **cs, int *num_cs, CPTRACKINFO *cpti,
					 BOOL cdtext, DWORD nwa)
{
	struct _CUESHEET *csp = NewSheet(cs, num_cs, 1);
	DebugLog("CreateCueSheet: Lead-in.\n");
	if(csp == NULL){
		return RET_MEMERR;
	}
	csp->ctladr = IS_TRACKMODE_DATA(cpti->trackinfo.track_mode)
		? CSCTL_DATA : CSCTL_AUDIO;
	csp->ctladr |= CSADR_NORMAL;
	csp->tno = 0;
	csp->index = 0;
	csp->dataform = cdtext ? 0x41 : 0x01;
	csp->scms = 0;
	/*LBA2MSF(nwa, &csp->min, &csp->sec, &csp->frame);*/
	/** Lead-in は常に0 */
	csp->min = 0;
	csp->sec = 0;
	csp->frame = 0;
	return RET_OK;
}

/**
 * Lead-out設定
 * @param[in,out] cs キューシート
 * @param[in,out] num_cs キューシートのエントリ数
 * @param[in] cpti トラック情報
 * @param[in] nwa NWA
 * @retval RET_OK	正常終了
 * @retval RET_MEMERR メモリ不足
 */
static int SetLeadOut(struct _CUESHEET **cs, int *num_cs,
					  CPTRACKINFO *cpti, DWORD nwa)
{
	struct _CUESHEET *csp = NewSheet(cs, num_cs, 1);
	DebugLog("CreateCueSheet: Lead-out.\n");
	if(csp == NULL){
		return RET_MEMERR;
	}
	csp->ctladr = IS_TRACKMODE_DATA(cpti->trackinfo.track_mode)
		? CSCTL_DATA : CSCTL_AUDIO;
	csp->ctladr |= CSADR_NORMAL;
	csp->tno = 0xAA;
	csp->index = 1;
	if(!IS_TRACKMODE_DATA(cpti->trackinfo.track_mode))
		csp->dataform = 0x01;
	else
		csp->dataform = cpti->mode2 ? 0x24 : 0x14;
	csp->scms = cpti->trackinfo.copy ? 0x80 : 0;
	LBA2MSF(nwa, &csp->min, &csp->sec, &csp->frame);
	return RET_OK;
}


/**
 * プリギャップ設定
 * @param[in,out] cs キューシート
 * @param[in,out] num_cs キューシートのエントリ数
 * @param[in] track_num トラック番号
 * @param[in] lcpti 直前トラック情報
 * @param[in] cpti 現在トラック情報
 * @param[in] gap_len プリギャップ長
 * @param[in] twopart 2パートプリギャップかどうか
 * @param[in,out] nwa NWA
 * @retval RET_OK	正常終了
 * @retval RET_MEMERR メモリ不足
 */
static int SetPreGap(struct _CUESHEET **cs, int *num_cs, WORD track_num,
					 CPTRACKINFO *lcpti, CPTRACKINFO *cpti,
					 DWORD gap_len, BOOL twopart, DWORD *nwa)
{
	struct _CUESHEET *csp = NULL;
	DebugLog("CreateCueSheet: PreGap. track=%d gap=%ld nwa=0x%08lX\n",
			 track_num, (long)gap_len, *nwa);
	if(gap_len > 0){
		if(twopart && gap_len > 150){
			csp = NewSheet(cs, num_cs, 1);
			if(csp == NULL){
				return RET_MEMERR;
			}
			csp->ctladr =
				IS_TRACKMODE_DATA(lcpti->trackinfo.track_mode)
				? CSCTL_DATA : CSCTL_AUDIO;
			csp->ctladr |= CSADR_NORMAL;
			csp->tno = (BYTE)track_num;
			csp->index = 0;
#if SEND_GAPDATA
			if(!IS_TRACKMODE_DATA(lcpti->trackinfo.track_mode)){
				csp->dataform = 0x00;
			}
			else{
				csp->dataform = lcpti->mode2 ? 0x20 : 0x10;
			}
#else
			if(!IS_TRACKMODE_DATA(lcpti->trackinfo.track_mode)){
				csp->dataform = 0x01;
			}
			else{
				csp->dataform = lcpti->mode2 ? 0x24 : 0x14;
			}
#endif
			csp->scms = lcpti->trackinfo.copy ? 0x80 : 0;
			LBA2MSF(*nwa, &csp->min, &csp->sec, &csp->frame);
			*nwa += (gap_len-150);
			gap_len = 150;
		}
		csp = NewSheet(cs, num_cs, 1);
		if(csp == NULL){
			return RET_MEMERR;
		}
		csp->ctladr = IS_TRACKMODE_DATA(cpti->trackinfo.track_mode)
			? CSCTL_DATA : CSCTL_AUDIO;
		csp->ctladr |= CSADR_NORMAL;
		csp->tno = (BYTE)track_num;
		csp->index = 0;
#if SEND_GAPDATA
		if(!IS_TRACKMODE_DATA(cpti->trackinfo.track_mode)){
			csp->dataform = 0x00;
		}
		else{
			csp->dataform = cpti->mode2 ? 0x20 : 0x10;
		}
#else
		if(!IS_TRACKMODE_DATA(cpti->trackinfo.track_mode)){
			csp->dataform = 0x01;
		}
		else{
			csp->dataform = cpti->mode2 ? 0x24 : 0x14;
		}
#endif
		csp->scms = cpti->trackinfo.copy ? 0x80 : 0;
		LBA2MSF(*nwa, &csp->min, &csp->sec, &csp->frame);
		*nwa += gap_len;
	}
	return RET_OK;
}


/**
 * Audioまたはデータトラック設定
 * @param[in,out] cs キューシート
 * @param[in,out] num_cs キューシートのエントリ数
 * @param[in] track_num トラック番号
 * @param[in] cpti トラック情報
 * @param[in,out] nwa NWA
 * @retval RET_OK	正常終了
 * @retval RET_MEMERR メモリ不足
 */
static int SetTrack(struct _CUESHEET **cs, int *num_cs,
					WORD track_num, CPTRACKINFO *cpti, DWORD *nwa)
{
	struct _CUESHEET *csp = NewSheet(cs, num_cs, 1);
	DebugLog("CreateCueSheet: Track%d.\n", track_num);
	if(csp == NULL){
		return RET_MEMERR;
	}
	csp->ctladr = IS_TRACKMODE_DATA(cpti->trackinfo.track_mode)
		? CSCTL_DATA : CSCTL_AUDIO;
	csp->ctladr |= CSADR_NORMAL;
	csp->tno = (BYTE)track_num;
	csp->index = 1;
	if(!IS_TRACKMODE_DATA(cpti->trackinfo.track_mode))
		csp->dataform = 0x00;
	else
		csp->dataform = cpti->mode2 ? 0x20 : 0x10;
	csp->scms = cpti->trackinfo.copy ? 0x80 : 0;
	LBA2MSF(*nwa, &csp->min, &csp->sec, &csp->frame);
	*nwa += Get4bytes(cpti->trackinfo.track_size);

	return RET_OK;
}

/**
 * SAO記録用CueSheetの作成
 * @param[in]	drive	書込装置構造体
 * @param[in]	discinfo  ディスク情報
 * @param[in]	track_start 開始トラック番号
 * @param[out]	cs_ret	作成したCueSheet
 * @param[out]  numcs_ret 作成したCueSheet配列の要素数
 * @param[in]   enable_cdtext もしあればCD-TEXTを有効にするかどうか
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 */
int CreateCueSheet(CMDDRIVE *drive, CPDISCINFO *discinfo,
				   WORD track_start,
				   struct _CUESHEET **cs_ret, int *numcs_ret,
				   BOOL enable_cdtext)
{
	struct _CUESHEET *cs = NULL;
	int cur_cs = 0;
	int num_cs = 0;
	DWORD gap_len;
	WORD sess_num = 0;
	DWORD nwa;
	CPTRACKINFO *cpti=NULL, *lcpti=NULL;
	int ret;
	BOOL gap_2part;
	BOOL cdtext=FALSE;
	WORD track_num;

	if(track_start==1 && discinfo->cdtext_size>0 && enable_cdtext){
		cdtext = TRUE;
	}

	track_num = track_start;
	cur_cs = 0;
	num_cs = 0;
	sess_num = 0;
	
	ret = GetNWA(drive, track_num, &nwa);
	if(ret != RET_OK){
		return ret;
	}

	if(strlen((char *)discinfo->media_catalog_number)!=0){
		/* Media Catalog Number */
		ret = SetMCN(&cs, &num_cs, discinfo);
		if(ret != RET_OK){
			return ret;
		}
	}
	
	cpti = NULL;
	lcpti = NULL;
	ret = RET_OK;
	for( ; track_num <= discinfo->tracks; track_num++){
		lcpti = cpti;
		cpti = &discinfo->trackinfo[track_num-1];
		gap_len = 0;
		if(sess_num==0){
			sess_num = SESSION_NUMBER(&cpti->trackinfo);
			ret = SetLeadIn(&cs, &num_cs, cpti, cdtext, nwa);
			if(ret != RET_OK){
				return ret;
			}
			if(sess_num >= 2){
				nwa -= 150;	/* pregap分 */
			}
			gap_len = Get4bytes(cpti->trackinfo.track_start) - nwa;
		}
		else if(sess_num != SESSION_NUMBER(&cpti->trackinfo)){
			cpti = lcpti;
			break;
		}

		if(strlen((char *)cpti->isrc)!=0){
			ret = SetISRC(&cs, &num_cs, track_num, cpti);
			if(ret != RET_OK){
				return ret;
			}
		}
		gap_2part = FALSE;
		if(gap_len==0){
#if 1	/* こっちの方が良いかな? */
			if(lcpti!=NULL){
				gap_len = Get4bytes(cpti->trackinfo.track_start) -
					(Get4bytes(lcpti->trackinfo.track_start) +
					 Get4bytes(lcpti->trackinfo.track_size));
				if(IS_TRACKMODE_DATA(cpti->trackinfo.track_mode)){
					/* DATA */
					if(!IS_TRACKMODE_DATA(lcpti->trackinfo.track_mode) ||
					   lcpti->mode2 != cpti->mode2){
						/* 2part */
						DebugLog("CreateCueSheet: Two part pregap.(cdda -> data)\n");
						gap_2part = TRUE;
					}
				}
				else{
					/* AUDIO */
					if(IS_TRACKMODE_DATA(lcpti->trackinfo.track_mode)){
						/* 前回DATAの場合もかな? */
						DebugLog("CreateCueSheet: Two part pregap.(data -> cdda)\n");
						gap_2part = TRUE;
					}
				}
			}
#else
			if(!IS_TRACKMODE_DATA(cpti->trackinfo.track_mode)){
				/* AUDIO */
				gap_len = cpti->pause_len;
			}
			else if(lcpti!=NULL){
				/* DATA で 2nd 以降 */
				gap_len = 150;
				if(!IS_TRACKMODE_DATA(lcpti->trackinfo.track_mode) ||
				   lcpti->mode2 != cpti->mode2){
					/* 2part */
					gap_len = 75+150;
					gap_2part = TRUE;
				}
			}
#endif
		}
		nwa = Get4bytes(cpti->trackinfo.track_start) - gap_len;
		ret = SetPreGap(&cs, &num_cs, track_num, lcpti, cpti,
						gap_len, gap_2part, &nwa);
		if(ret != RET_OK){
			return ret;
		}
		/* Audio/Data */
		ret = SetTrack(&cs, &num_cs, track_num, cpti, &nwa);
		if(ret != RET_OK){
			return ret;
		}
	}
	/* Lead-out */
	ret = SetLeadOut(&cs, &num_cs, cpti, nwa);
	if(ret != RET_OK){
		return ret;
	}
	
	*cs_ret = cs;
	*numcs_ret = num_cs;
	DebugLog("CreateCueSheet: Done.\n");
	return RET_OK;
}

