package org.j69.eewiki.wiki.transformer;


/**
 * キーワード「>>>」「>>」「>」を処理する
 *
 * >>> : <blockquote><blockquote><blockquote>[通常文字列]
 *          </blockquote></blockquote></blockquote>
 * >> : <blockquote><blockquote>[通常文字列]
 *          </blockquote></blockquote>
 * > : <blockquote>[通常文字列]</blockquote>
 *
 * @see DataLineCreator
 */
public class BlockquoteDataLine implements DataLine {
    private String keyword_,                        //キーワード（置き換え対象）
                    lineData_,                      //データ
                    dataHoldTag_ = "",              //データ自体をはさむタグ
                    outerHoldTag_ = "blockquote",   //データの外側をはさむタグ
                    baseSpaces_;

    /**
     * 初期化
     *
     * @param lineData 行データ（キーワード分解前）
     **/
    public void init(String lineData) {
        int length = 0;
        if (lineData.indexOf("&gt;&gt;&gt;") == 0) {
            keyword_ = ">>>";
            baseSpaces_ = "  ";
            length = 12;
        } else if(lineData.indexOf("&gt;&gt;") == 0) {
            keyword_ = ">>";
            baseSpaces_ = " ";
            length = 8;
        } else {
            keyword_ = ">";
            baseSpaces_ = "";
            length = 4;
        }

        lineData_ = lineData.substring(length);
    }

    /**
     * 行頭キーワード取得
     *
     * @return 行頭キーワード
     **/
    public String getKeyword() {
        return keyword_;
    }

    /**
     * データの外側を挟む開始タグ（<blockquote>）取得
     *
     * @param oldLine 前の行データ
     * @return <blockquote>
     **/
    public String openTag(DataLine oldLine) {
        if(oldLine != null && oldLine instanceof BlockquoteDataLine) {
            //前の行も BLOCKQUOTE の場合
            BlockquoteDataLine oldBlockquoteLine = (BlockquoteDataLine)oldLine;
            int depth = getKeyword().length() - oldBlockquoteLine.getKeyword().length();
            switch(depth) {
                case 0 :
                    //前のBLOCKQUOTE と同じ深さの場合はなし
                    return "";
                case 1 :
                    return baseSpaces_ + "<" + outerHoldTag_ + ">\n";
                case 2 :
                    return " <" + outerHoldTag_ + ">\n" +
                            baseSpaces_ + "<" + outerHoldTag_ + ">\n";
                default :
                    return "";
            }
        } else {
            int depth = getKeyword().length();
            switch(depth) {
                case 1 :
                    return baseSpaces_ + "<" + outerHoldTag_ + ">\n";
                case 2 :
                    return "<" + outerHoldTag_ + ">\n" +
                            baseSpaces_ + "<" + outerHoldTag_ + ">\n";
                case 3 :
                    return "<" + outerHoldTag_ + ">\n" +
                            " <" + outerHoldTag_ + ">\n" +
                            baseSpaces_ + "<" + outerHoldTag_ + ">\n";
                default :
                    return "";
            }
        }
    }

    /**
     * 行頭に付加する半角スペース（" "）取得
     *
     * @param spaceNum スペース数
     * @return 行頭に付加するスペース
     **/
    private String getBaseSpaces(int spaceNum){
        StringBuffer buf = new StringBuffer();
        for(int i = 0 ; i < spaceNum ; i++) {
            buf.append(" ");
        }

        return buf.toString();
    }

    /**
     * データの外側を挟む終了タグ（<blockquote>）取得
     *
     * @param oldLine 前の行データ
     * @return </blockquote>
     **/
    public String closeTag(DataLine newLine) {
        if(newLine != null && newLine instanceof BlockquoteDataLine) {
            //前の行も BLOCKQUOTE の場合
            BlockquoteDataLine oldULLine = (BlockquoteDataLine)newLine;
            int depth = getKeyword().length() - newLine.getKeyword().length();
            switch(depth) {
                case 0 :
                    //前のBLOCKQUOTE と同じ深さの場合はなし
                    return "";
                case 1 :
                    return baseSpaces_ + "</" + outerHoldTag_ + ">\n";
                case 2 :
                    return baseSpaces_ + "</" + outerHoldTag_ + ">\n" +
                            " </" + outerHoldTag_ + ">\n";
                default :
                    return "";
            }
        } else {
            int depth = getKeyword().length();
            switch(depth) {
                case 1 :
                    return baseSpaces_ + "</" + outerHoldTag_ + ">\n";
                case 2 :
                    return baseSpaces_ + "</" + outerHoldTag_ + ">\n" +
                            "</" + outerHoldTag_ + ">\n";
                case 3 :
                    return baseSpaces_ + "</" + outerHoldTag_ + ">\n" +
                            " </" + outerHoldTag_ + ">\n" +
                            "</" + outerHoldTag_ + ">\n";
                default :
                    return "";
            }
        }
    }

    /**
     * 行データ取得
     *
     * @return 行データ
     **/
    public String get(int row) {
        return baseSpaces_ + lineData_ + "<br>\n";
    }
}

