#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/proc_fs.h>
#include <asm/uaccess.h>

#include "drbl.h"

int server=0;
MODULE_PARM(server, "i");

struct Drbl_Dev *drbl_devices=NULL;

int drbl_trim(struct Drbl_Dev *dev) {
  struct Drbl_Dev *next, *dptr;
  int qset = dev->qset;
  int i;

  for(dptr = dev; dptr; dptr = next) {
    if (dptr->data) {
      for(i = 0; i < qset; i++)
        if (dptr->data[i]) 
          kfree(dptr->data[i]);
      kfree(dptr->data);
      dptr->data=NULL;
    }
    next = dptr->next;
    if(dptr!=dev) kfree(dptr); /* all of them but the first */
  }
  dev->size = 0;
  dev->quantum = DRBL_QUANTUM;
  dev->qset = DRBL_QSET;
  dev->next = NULL;
  return 0;
}

struct Drbl_Dev *drbl_follow(struct Drbl_Dev *dev, int n) {
  while(n--) {
    if(!dev->next) {
      dev->next = kmalloc(sizeof(struct Drbl_Dev), GFP_USER);
      memset(dev->next, 0, sizeof(struct Drbl_Dev));
    }
    dev = dev->next;
    continue;
  }
  return dev;
}

int drbl_open(struct inode *inode, struct file *flip) {
  struct Drbl_Dev *dev = (struct Drbl_Dev *)flip->private_data;
  int num = MINOR(inode->i_rdev);
  if(!dev) {
    if(num >= DRBL_DEVICES) return -ENODEV;
    dev = &drbl_devices[num];
    flip->private_data = dev;
  }

  MOD_INC_USE_COUNT;
  if( (flip->f_flags & O_ACCMODE) == O_WRONLY ) {
    if(down_interruptible(&dev->sem)) {
      MOD_DEC_USE_COUNT;
      return -ERESTARTSYS;
    }
    drbl_trim(dev);
    up(&dev->sem);
  }
  return 0; /* success */
}

int drbl_release(struct inode *inode, struct file *flip) {
  MOD_DEC_USE_COUNT;
  return 0; /* success */
}

ssize_t drbl_read(struct file *filp, char *buf, size_t count, loff_t *f_pos) {
  struct Drbl_Dev *dev = filp->private_data;
  struct Drbl_Dev *dptr;
  int quantum = dev->quantum;
  int qset = dev->qset;
  int itemsize = quantum * qset;
  int item, s_pos, q_pos, rest;
  size_t ret = 0;

  if ( down_interruptible(&dev->sem) ) return -ERESTARTSYS;

  if(*f_pos >= dev->size) goto out;
  if(*f_pos + count > dev->size)
    count = dev->size - *f_pos;
  /* find the list item, qset index, and offset in the quantum */
  item = (long)*f_pos / itemsize;
  rest = (long)*f_pos % itemsize;
  s_pos = rest / quantum;
  q_pos = rest % quantum;

  /* follow the list up to the right position */
  dptr = drbl_follow(dev, item);
  if(!dptr->data) goto out;
  if(!dptr->data[s_pos]) goto out;

  /* read only up to the end of this quantum */
  if( count > quantum - q_pos )
    count = quantum - q_pos;

  if(copy_to_user(buf, dptr->data[s_pos]+q_pos, count)) {
    ret = -EFAULT;
    goto out;
  }
  *f_pos += count;
  ret = count;

out:
  up(&dev->sem);
  return ret;
 
}

ssize_t drbl_write(struct file *filp, const char *buf, size_t count, loff_t *f_pos) {
  struct Drbl_Dev *dev = filp->private_data;
  struct Drbl_Dev *dptr;
  int quantum = dev->quantum;
  int qset = dev->qset;
  int itemsize = quantum * qset;
  int item, s_pos, q_pos, rest;
  size_t ret = -ENOMEM;

  if ( down_interruptible(&dev->sem) ) return -ERESTARTSYS;
  /* find the first item, qset index and offset in the quantum */
  item = (long)*f_pos / itemsize;
  rest = (long)*f_pos % itemsize;
  s_pos = rest / quantum; 
  q_pos = rest % quantum;

  /* follow the list up to the right position */ 
  dptr = drbl_follow(dev, item);
  if(!dptr->data) {
    dptr->data = kmalloc(quantum, GFP_USER);
    if(!dptr->data) goto out;
    memset(dptr->data, 0, qset * sizeof(char *));
  }
  if(!dptr->data[s_pos]) {
    dptr->data[s_pos] = kmalloc(quantum, GFP_USER);
    if(!dptr->data[s_pos]) goto out;
  }

  /* write only yo to the end of this quantum */
  if( count > quantum - q_pos )
    count = quantum - q_pos;

  if(copy_from_user(dptr->data[s_pos]+q_pos, buf, count)) {
    ret = -EFAULT;
    goto out;
  }
  *f_pos += count;
  ret = count;

  /* update the size */
  if(dev->size < *f_pos) dev->size = *f_pos;

out:
  up(&dev->sem);
  return ret;
}

struct file_operations drbl_fops = {
  read:    drbl_read,
  write:   drbl_write,
  open:    drbl_open,
  release: drbl_release,
};

static int __init init_drbl(void)
{
  int result, i;

  result = register_chrdev(DRBL_MAJOR, DRBL_DEVNAME, &drbl_fops);
  if (result<0) return result;

  drbl_devices = kmalloc(DRBL_DEVICES * sizeof(struct Drbl_Dev), GFP_USER);
  if(!drbl_devices) {
    result = -ENOMEM;
    goto fail_malloc;
  }
  memset(drbl_devices, 0, DRBL_DEVICES * sizeof(struct Drbl_Dev));
  for(i=0; i<DRBL_DEVICES; i++) {
    drbl_devices[i].quantum = 4000;
    drbl_devices[i].qset = 1000;
    sema_init(&drbl_devices[i].sem, 1);
  }

  proc_init();

  printk("<1>drbl_module: init (server=%d)\n",server);
  return 0;

fail_malloc:
  unregister_chrdev(DRBL_MAJOR, DRBL_DEVNAME);
  return result;
}

static void __exit cleanup_drbl(void)
{
  int i;
  unregister_chrdev(DRBL_MAJOR, DRBL_DEVNAME);

  proc_exit();

  printk("<1>drbl_module: exit (server=%d)\n",server);
}

module_init(init_drbl);
module_exit(cleanup_drbl);

MODULE_AUTHOR("Blake, Kuo-Lien Huang");
MODULE_DESCRIPTION("DRBL module");

EXPORT_NO_SYMBOLS;
