/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2136 $
|                     $Date: 2009-03-02 17:51:16 +0100 (lun, 02 mar 2009) $
\*============================================================================*/

/******************************************************************************\
 *                         Generic message processing                         *
\******************************************************************************/
#include "dc/dc_GenericInvocation.h"
#include "dc/dc_Runtime.h"
#include "dc/dc_Constants.h"
#include "dc/dc_Dpws.h"
#include "dcGSOAP_Runtime.h"
#include "dcCOMN_Tools.h"

/*----------------------------------- Data -----------------------------------*/

struct Namespace generic_snd_namespaces[] =
    {
        {SOAP_ENV_PREFIX, SOAP_ENV_URI, SOAP_ENV_WILDCARD, NULL},
        {WSA_PREFIX, WSA_200408_URI, WSA_WILDCARD, NULL},
#ifdef WITH_WSMAN
        {WSMAN_PREFIX, WSMAN_URI, WSMAN_WILDCARD, NULL},
#endif
        {NULL, NULL, NULL, NULL}
    };

/*------------------------- Static Functions prototypes ----------------------*/

static int send_msg(
    struct dpws *dpws,
    struct wsa_endpoint_ref *to,
    char * action,
    struct prefix_def * root_pfxs,
    int nb_pfxs,
    serialize_cbk request_cbk,
    void * user_data,
	DC_BOOL oneway
);

/*----------------------------------------------------------------------------*\
 *                          Server message processing                         *
\*----------------------------------------------------------------------------*/
int dpws_process_request(
    struct dpws *dpws,
    char * resp_action,
    char * fault_action,
	struct prefix_def * resp_root_pfxs,
	int nb_pfxs,
    serve_cbk serve_func
	)
{
    struct soap *soap = dpws_dpws2soap(dpws);
    serialize_cbk cbk = NULL;
    void * user_data = NULL;
    void * parser_ctx = epx_new_parser(NULL, dpws);
    void * serializer_ctx = NULL;
    epx_event p_event;

    soap->encodingStyle = NULL;
    epx_start_fragment_parsing(parser_ctx, EPX_TRUE);
    if (serve_func(dpws, parser_ctx, &cbk, &user_data)) {
    	if (soap->error == SOAP_FAULT) {
    		soap->error = WSA_ERR_SOAP_FAULT;
    		dpws->action = fault_action;
    	}
    	epx_delete_parser(parser_ctx);
        return soap->error;
    }
    // consume events that were no read
    for (
    	p_event = epx_get_event(parser_ctx);
    	p_event != EPX_EVT_END_FRAGMENT;
    	p_event = epx_next(parser_ctx)
    );
    epx_delete_parser(parser_ctx);
    if (soap_body_end_in(soap)
            || soap_envelope_end_in(soap)
            || soap_end_recv(soap))
        return soap->error;
    if (resp_action) {
    	int i;
	    if (dpws_check_headers(dpws))
	        return soap->error;
		dpws_set_namespaces(dpws, generic_snd_namespaces);
	    if (soap->error)
	        return soap->error;
	    dpws_set_reply_headers(dpws, resp_action);
	    if (soap_begin_count(soap))
	        return soap->error;
	    serializer_ctx = epx_new_serializer(NULL, dpws);
		for (i = 0; i < nb_pfxs; i++)
			soap_check_prefix_definition(soap, resp_root_pfxs[i].ns_uri, resp_root_pfxs[i].ns_prefix);
	    if (soap->mode & SOAP_IO_LENGTH)
	    {
	        if (soap_envelope_begin_out(soap)
	                || soap_putheader(soap)
	                || soap_body_begin_out(soap)
	                || (cbk && cbk(serializer_ctx, user_data))
	                || soap_body_end_out(soap)
	                || soap_envelope_end_out(soap))
 					goto resp_error;
	    }
		if (nb_pfxs > 0)
			soap_pop_namespace(soap);
	    if (soap_end_count(soap)
	            || dpws_response(dpws, SOAP_OK))
	        goto resp_error;
		for (i = 0; i < nb_pfxs; i++)
			soap_check_prefix_definition(soap, resp_root_pfxs[i].ns_uri, resp_root_pfxs[i].ns_prefix);
	    if (soap_envelope_begin_out(soap)
	            || soap_putheader(soap)
	            || soap_body_begin_out(soap)
	            || (cbk && cbk(serializer_ctx, user_data))
	            || soap_body_end_out(soap)
	            || soap_envelope_end_out(soap)
	            || soap_end_send(soap))
	        goto resp_error;
		epx_delete_serializer(serializer_ctx);
		if (nb_pfxs > 0)
		    soap_pop_namespace(soap);
    }
    return soap_closesock(soap);

resp_error:
	epx_delete_serializer(serializer_ctx);
    return soap->error;
}

/*----------------------------------------------------------------------------*\
 *                         Message sending functions                          *
\*----------------------------------------------------------------------------*/

static int send_msg(
    struct dpws *dpws,
    struct wsa_endpoint_ref *to,
    char * action,
    struct prefix_def * root_pfxs,
    int nb_pfxs,
    serialize_cbk request_cbk,
    void * user_data,
	DC_BOOL oneway
)
{
    struct soap *soap = &dpws->soap;
    void * serializer_ctx = (request_cbk || root_pfxs) ? epx_new_serializer(NULL, soap) : NULL;
    int i;

	if (dpws_check_endpoint(dpws, to))
		goto error;
    dpws_set_namespaces(dpws, generic_snd_namespaces);
	if (oneway)
		dpws_set_oneway_headers(dpws, to, action);
	else
		dpws_set_call_headers(dpws, to, NULL, action);
    soap->encodingStyle = NULL;
    soap_begin(soap);
    if (soap_begin_count(soap))
        goto error;
	for (i = 0; i < nb_pfxs; i++)
		soap_check_prefix_definition(soap, root_pfxs[i].ns_uri, root_pfxs[i].ns_prefix);
    if (soap->mode & SOAP_IO_LENGTH)
    {
        if (soap_envelope_begin_out(soap)
                || soap_putheader(soap)
                || soap_body_begin_out(soap)
                || (request_cbk && request_cbk(serializer_ctx, user_data))
                || soap_body_end_out(soap)
                || soap_envelope_end_out(soap))
            goto error;
    }
	if (nb_pfxs > 0)
		soap_pop_namespace(soap);
    if (soap_end_count(soap))
        goto error;
    if (dpws_connect(dpws, to->address, action))
    {
        if (soap->error == SOAP_TCP_ERROR)
            dpws_invalidate(dpws, to);
        soap_closesock(soap);
        goto error;
    }
	for (i = 0; i < nb_pfxs; i++)
		soap_check_prefix_definition(soap, root_pfxs[i].ns_uri, root_pfxs[i].ns_prefix);
    if (soap_envelope_begin_out(soap)
            || soap_putheader(soap)
            || soap_body_begin_out(soap)
            || (request_cbk && request_cbk(serializer_ctx, user_data))
            || soap_body_end_out(soap)
            || soap_envelope_end_out(soap)
            || soap_end_send(soap))
    {
        if (soap->error == SOAP_TCP_ERROR)
            dpws_invalidate(dpws, to);
        soap_closesock(soap);
        goto error;
    }
	if (nb_pfxs > 0)
		soap_pop_namespace(soap);
error:
	if (serializer_ctx)
		epx_delete_serializer(serializer_ctx);
	return soap->error;
}

int dpws_send(
    struct dpws *dpws,
    struct wsa_endpoint_ref *to,
    char * action,
    struct prefix_def * root_pfxs,
    int nb_pfxs,
    serialize_cbk request_cbk,
    void * user_data
)
{
	DC_CHECK_PARAM(dpws && to && action && (root_pfxs || nb_pfxs <= 0));

	return send_msg(dpws, to, action, root_pfxs, nb_pfxs, request_cbk, user_data, DC_TRUE) ?
			dpws->soap.error : dpws_receive_empty_response(dpws);
}

int dpws_call(
    struct dpws *dpws, struct wsa_endpoint_ref *to, char * action, struct prefix_def * root_pfxs,
    int nb_pfxs, serialize_cbk request_cbk, parser_cbk response_cbk, void * user_data
)
{
    struct soap *soap = &dpws->soap;

	DC_CHECK_PARAM(dpws && to && action && (root_pfxs || nb_pfxs <= 0));

	if (send_msg(dpws, to, action, root_pfxs, nb_pfxs, request_cbk, user_data, DC_FALSE))
		return soap->error;

    if (dpws_is_asynchronous(dpws))
        return SOAP_OK;
    else
    {
	    dpws_set_namespaces(dpws, dpws->protocols->default_namespaces);
        dpws_init_headers(dpws);
        if (soap_begin_recv(soap)
                || soap_envelope_begin_in(soap)
                || soap_recv_header(soap)
                || soap_body_begin_in(soap))
            return soap_closesock(soap);
		if (!strcmp(dpws->action, dpws->wsa_version->wsa_fault_uri))
        	soap->error = SOAP_FAULT;
        else if (response_cbk)
        {
    		epx_event p_event;
    	    void * parser_ctx = response_cbk ? epx_new_parser(NULL, soap) : NULL;
            epx_start_fragment_parsing(parser_ctx, EPX_TRUE);
            soap->error = response_cbk(parser_ctx, user_data);
		    // consume events that were no read
            if (!soap->error) {
			    for (
			    	p_event = epx_get_event(parser_ctx);
			    	p_event != EPX_EVT_END_FRAGMENT;
			    	p_event = epx_next(parser_ctx)
			    );
            }
            epx_delete_parser(parser_ctx);
        }
        if (soap->error)
        {
            if (soap->error == SOAP_FAULT)
                return soap_recv_fault(soap);
            return soap_closesock(soap);
        }
        if (soap_body_end_in(soap)
                || soap_envelope_end_in(soap)
                || soap_end_recv(soap))
            return soap_closesock(soap);
        return soap_closesock(soap);
    }
}
