/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2154 $
|                     $Date: 2009-03-05 17:38:15 +0100 (jeu, 05 mar 2009) $
\*============================================================================*/

/***************************************************************************//**
* \file
* Streaming XML processing API using the pull-parsing approach.
* NOTES:<ol>
* <li>This XML parsing API is currently focused and optimised for the XML subset
*    used by SOAP 1.2. This is why for instance, processing instructions are
*    not considered. Entities are also supposed to be resolved by the parser.
*    As a low-level API with no validation, whitespaces are processed like any
*    other character data.</li>
* <li>The typed API is still experimental and may change in subsequent versions.</li></ol>
*******************************************************************************/

#ifndef EPX_H_
#define EPX_H_

#include <stdlib.h>

#include "dc/dc_XMLTypes.h"


#ifndef DC_RT_FMAC1	/* runtime functions declaration prefix */
# define DC_RT_FMAC1
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*----------------------------------------------------------------------------*\
 *                                Error codes                                 *
\*----------------------------------------------------------------------------*/

/* Error codes */
#define EPX_OK							0	/**< No error */
#define EPX_ERR_UNKNOWN_OPTION			-1	/**< The option is not recognized by the implementation */
#define EPX_ERR_NOT_SUPPORTED			-2	/**< Call not supported by the parser or for the current event. */
#define EPX_ERR_INVALID_REPRESENTATION	-3	/**< Typed API could not convert XML to native data */
#define EPX_ERR_OUT_OF_RANGE			-4	/**< Typed API could convert XML to native data but the value is out of the target type range */
#define EPX_ERR_IMPLEMENTATION			-5	/**< Implementation-specific error to retrieve using a dedicated API */
#define EPX_ERR_INVALID_PARAMETER		-6	/**< A parameter passed to the API is incorrect */


/*----------------------------------------------------------------------------*\
 *                              Type definitions                              *
\*----------------------------------------------------------------------------*/

/** XML events definition which is fundamental for API usage.
 * Note that among available events, some of them have an order that is not
 * obvious and are specific to the EPX API. These are the events related to
 * element tag parsing which order must be:<ol>
 * <li>EPX_EVT_START_ELEMENT</li>
 * <li>EPX_EVT_PREFIX_DEFINITION</li>
 * <li>EPX_EVT_ATTRIBUTE (xsi:type, then xsi:nil should be first)</li></ol>
 */
typedef enum {
	EPX_EVT_ERROR = -1,	/**< The processor stopped because of an error */
	EPX_EVT_IDLE = 0, /**< No parsing running */
	EPX_EVT_START_DOCUMENT = 1,	/**< XML document prolog event */
	EPX_EVT_END_DOCUMENT = 2,	/**< The XML file processing is finished */
	EPX_EVT_START_ELEMENT = 3,	/**< Element start-tag event */
	EPX_EVT_PREFIX_DEFINITION = 4,	/**< Prefix definition event (<em>xmlns</em> attributes) */
	EPX_EVT_ATTRIBUTE = 5,	/**< Attribute event */
	EPX_EVT_END_ELEMENT = 6,	/**< Element end-tag event */
	EPX_EVT_CHARACTERS = 7,	/**< Characters events that may occur several time consecutively if the parser decides it. Coalescing is not required. */
	EPX_EVT_COMMENT = 8,	/**< Comment event */
	EPX_EVT_START_CDATA = 9,	/**< Some implementations like the EPX/gSOAP may not be able to produce CDATA events. */
	EPX_EVT_END_CDATA = 10,	/**< Some implementations like the EPX/gSOAP may not be able to produce CDATA events. */
	EPX_EVT_END_FRAGMENT = 11	/**< Generated when a fragment end has been reached. A fragment is started using epx_start_fragment_parsing. */
} epx_event;

/*----------------------------------------------------------------------------*\
 *                                 Constants                                  *
\*----------------------------------------------------------------------------*/

/* Parser boolean capabilities */

#define EPX_OPT_COALESCING 0x01	/**< Returns all consecutive char data ignoring comments in one event (PI are excluded for SOAP 1.2). */
#define EPX_OPT_IGNORE_COMMENTS 0x02	/**< Parsing will not produce COMMENT events. */
#define EPX_OPT_UTF8_OUTPUT 0x04	/**< output are in multi-byte UTF-8, else UCS-4 cast to char (i.e. ISO-LATIN-1). */
#define EPX_OPT_PARSER_STREAMING 0x08	/**< Streaming on data buffers is supported. */
#define EPX_OPT_GENERATE_PREFIX_DEFINITIONS 0x10	/**< Prefix definition will be generated instead of ordinary attribute events. */
#define EPX_OPT_FRAGMENT_PARSING 0x20	/**< Fragment feature supported. */

/***************************************************************************//**
* \name Parsing API
* @{
*******************************************************************************/

/** Gets the parser default options settings.
 * @param implementation A pointer on the EPX implementation. Reserved for later
 * use (set NULL).
 * @return An integer describing the default behavior of the parser during
 * parsing. The default options values maybe the only one supported. This can
 * be established by calling the is_processor_option_settable API for a
 * particular option.
 */
DC_RT_FMAC1 int epx_get_parser_options(void * implementation);

/** Test if a parser option can be enabled of disabled.
 * @param implementation A pointer on the EPX implementation. Reserved for later
 * use (set NULL).
 * @param option The option to test.
 * @return 1 if settable, 0 if not and EPX_ERR_UNKNOWN_OPTION if the option is
 * not recognized.
 */
DC_RT_FMAC1 int epx_is_parser_option_settable(void * implementation, int option);

/** Creates a new parser from a given EPX implementation.
 * @param implementation A pointer on the EPX implementation. Reserved for later
 * use (setting NULL will select the implementation over gSOAP).
 * @param impl_data Custom data that may be required by the implementation for
 * instance to avoid dynamic allocation for the parser context. If NULL is
 * provided as \a implementation, this should point to a \a soap structure.
 * @return An implementation dependent parsing context. If an error occurred the
 * implementation is responsible for supplying an error code through the parsing
 * context.
 */
DC_RT_FMAC1 void * epx_new_parser(void * implementation, void * impl_data);

/**	Start the XML parsing for the implementation opaque id.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @param source Data identifying in an implementation-dependent way the XML
 * file or stream. If the implementation is gSOAP, may be NULL (current message
 * processing).
 * @param options Which capabilities should be used for XML processing.
 * @return An implementation-dependent error code.
 */
DC_RT_FMAC1 int epx_start_parsing(void * context, void * source, int options);

/**	Set parsing options while parsing is running.
 * May not be supported by the implementation.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @param options Which capabilities should be used for XML processing.
 * @return An implementation-dependent error code, especially EPX_ERR_NOT_SUPPORTED
 */
DC_RT_FMAC1 int epx_set_parsing_options(void * context, int options);

/**	Get the current parsing options.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @return Which capabilities should be used for XML processing.
 */
DC_RT_FMAC1 int epx_get_parsing_options(void * context);

/** Start the parsing of an XML fragment when an XML parsing is running,
 * meaning that the parsing will stop after the child elements have been
 * completely parsed. Note that all applicable prefix definitions (potentially
 * defined on any element from the root of the document to the root of the
 * fragment) can be sent in a row when starting to parse the fragment in order
 * to provide all the information available to the XML stream handler.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @param gen_all_pfx_events If EPX_TRUE, all event (minus doubles) should be sent
 * at the root of the fragment. Relevant only if prefix generation is enabled.
 * @return An implementation-dependent error code.
 */
DC_RT_FMAC1 int epx_start_fragment_parsing(void * context, epx_boolean_t gen_all_pfx_events);

/** Resume a document parsing when the parsing of an XML fragment has been
 * initiated or achieved.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @return An implementation-dependent error code.
 */
DC_RT_FMAC1 int epx_reset_fragment_parsing(void * context);

/** Returns the last the implementation dependent error code.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @return An implementation-dependent error code.
 */
DC_RT_FMAC1 int epx_get_parser_error(void * context);

/** Allows the implementation to free parsing resources.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 */
DC_RT_FMAC1 void epx_delete_parser(void * context);

/** Makes the parser to process the next XML event.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @return The event that was processed. IDLE is returned if no more events are
 * expected. ERROR if an error occured. The implementation is responsible for
 * supplying an error code through the parsing context.
 */
DC_RT_FMAC1 epx_event epx_next(void * context);

/** Get the current XML event.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @return The current parsing XML event.
 */
DC_RT_FMAC1 epx_event epx_get_event(void * context);

/** Reads character data for ATTRIBUTE, CHARACTERS an COMMENT events in a row.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @return A character buffer containing UTF-8 or ISO-LATIN-1 (according to the
 * option set for parsing). The data is supposed to be valid and accessible
 * from the moment the API is called until next() is called. NULL if an error
 * occured. The implementation is responsible for supplying an error code
 * through the parsing context.
 */
DC_RT_FMAC1 char * epx_get_characters(void * context);

/** Streaming access to the CHARACTERS event data.
 * With this API, data can be accessed block by block by the application.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @param dest The destination address for the copied character data. The data
 * is either UTF-8 or ISO-LATIN-1 (according to the option set for parsing).
 * @param length The size of the destination buffer and then the length of
 * the data that must be copied.
 * @return The size of the data actually read or an EPX error code.
 */
DC_RT_FMAC1 size_t epx_read_characters(void * context, char * dest, size_t length);

/** Get the namespace prefix for an element.
 * Callable on the START_ELEMENT, PREFIX_DEFINITION, ATTRIBUTE and END_ELEMENT
 * events.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @return A character buffer containing UTF-8 or ISO-LATIN-1 (according to the
 * option set for parsing). The namespace prefix is supposed to be valid and
 * accessible from the moment the API is called until next() is called. NULL if
 * an error occured. The implementation is responsible for supplying an error
 * code through the parsing context.
 */
DC_RT_FMAC1 char * epx_get_ns_prefix(void * context);

/** Get the namespace URI for an element.
 * Callable on the START_ELEMENT, PREFIX_DEFINITION, ATTRIBUTE and END_ELEMENT
 * events.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @return A character buffer containing UTF-8 or ISO-LATIN-1 (according to the
 * option set for parsing). The namespace URI is supposed to be valid and
 * accessible from the moment the API is called until next() is called. NULL if
 * an error occured. The implementation is responsible for supplying an error
 * code through the parsing context.
 */
DC_RT_FMAC1 char * epx_get_ns_uri(void * context);

/** Get the local name for an element.
 * Callable on the START_ELEMENT, ATTRIBUTE and END_ELEMENT events.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @return A character buffer containing UTF-8 or ISO-LATIN-1 (according to the
 * option set for parsing). The local name is supposed to be valid and
 * accessible from the moment the API is called until next() is called. NULL if
 * an error occured. The implementation is responsible for supplying an error
 * code through the parsing context.
 */
DC_RT_FMAC1 char * epx_get_lname(void * context);

/** Get the number of attributes for an element.
 * Callable on the START_ELEMENT event.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @deprecated May be removed in next version.
 * @return The number of attributes. -1 if an error occured. The implementation
 * is responsible for supplying an error code through the parsing context.
 */
DC_RT_FMAC1 int epx_get_att_nb(void * context);

/** Get an attribute namespace prefix.
 * Callable on the START_ELEMENT event.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @param index The index of the attribute.
 * @deprecated May be removed in next version.
 * @return A character buffer containing UTF-8 or ISO-LATIN-1 (according to the
 * option set for parsing). The namespace prefix is supposed to be valid and
 * accessible from the moment the API is called until next() is called. NULL if
 * an error occured. The implementation is responsible for supplying an error
 * code through the parsing context.
 */
DC_RT_FMAC1 char * epx_get_att_ns_prefix(void * context, int index);

/** Get an attribute namespace URI.
 * Callable on the START_ELEMENT event.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @param index The index of the attribute.
 * @deprecated May be removed in next version.
 * @return A character buffer containing UTF-8 or ISO-LATIN-1 (according to the
 * option set for parsing). The namespace URI is supposed to be valid and
 * accessible from the moment the API is called until next() is called. NULL if
 * an error occured. The implementation is responsible for supplying an error
 * code through the parsing context.
 */
DC_RT_FMAC1 char * epx_get_att_ns_uri(void * context, int index);

/** Get an attribute local name.
 * Callable on the START_ELEMENT event.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @param index The index of the attribute.
 * @deprecated May be removed in next version.
 * @return A character buffer containing UTF-8 or ISO-LATIN-1 (according to the
 * option set for parsing). The local name is supposed to be valid and
 * accessible from the moment the API is called until next() is called. NULL if
 * an error occured. The implementation is responsible for supplying an error
 * code through the parsing context.
 */
DC_RT_FMAC1 char * epx_get_att_lname(void * context, int index);

/** Get an attribute value.
 * Callable on the START_ELEMENT event.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @param index The index of the attribute.
 * @deprecated May be removed in next version.
 * @return A character buffer containing UTF-8 or ISO-LATIN-1 (according to the
 * option set for parsing). The value is supposed to be valid and
 * accessible from the moment the API is called until next() is called. NULL if
 * an error occured. The implementation is responsible for supplying an error
 * code through the parsing context.
 */
DC_RT_FMAC1 char * epx_get_att_value(void * context, int index);

/** Get a namespace URI for a particular prefix.
 * Should be callable on any "parsing" event.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @param prefix The prefix which namespace must be retrieved.
 * @return A character buffer containing UTF-8 or ISO-LATIN-1 (according to the
 * option set for parsing). The value is supposed to be valid and
 * accessible from the moment the API is called until the next xxxx_ELEMENT
 * occurs. NULL if an error occured. The implementation is responsible for
 * supplying an error code through the parsing context.
 */
DC_RT_FMAC1 char * epx_get_prefix_uri(void * context, char * prefix);

/** Get a namespace prefix for a particular URI.
 * Should be callable on any "parsing" event. Note that if multiple prefixes
 * are available the latest defined should be returned.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @param uri The namespace URI for which a prefix must be retrieved.
 * @return A character buffer containing UTF-8 or ISO-LATIN-1 (according to the
 * option set for parsing). The value is supposed to be valid and
 * accessible from the moment the API is called until the next xxxx_ELEMENT
 * occurs. NULL if an error occured. The implementation is responsible for
 * supplying an error code through the parsing context.
 */
DC_RT_FMAC1 char * epx_get_uri_prefix(void * context, char * uri);

/** \name Typed parsing API
 * @{
 */

/** Reads base 64-encoded binary data for CHARACTERS or ATTRIBUTE events in a
 * row.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @param[out] length The size of the returned buffer.
 * @return A character buffer containing binary data. If NULL, check for an
 * implementation error code.
 */
DC_RT_FMAC1 unsigned char * epx_get_base64_binary(void * context, size_t * length);

/** Streaming access to the CHARACTERS event base 64-encoded binary data.
 * With this API, data can be accessed block by block by the application.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @param dest The destination address for the copied binary data.
 * @param length The size of the destination buffer and then the length of
 * the data that must be copied.
 * @return The size of the data actually copied or an EPX error code.
 */
DC_RT_FMAC1 size_t epx_read_base64_binary(void * context, unsigned char * dest, size_t length);

/** Reads hexadecimal-encoded binary data for CHARACTERS or ATTRIBUTE events in
 * a row.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @param[out] length The size of the returned buffer.
 * @return A character buffer containing binary data. If NULL, check for an
 * implementation error code.
 */
DC_RT_FMAC1 unsigned char * epx_get_hex_binary(void * context, size_t * length);

/** Streaming access to the CHARACTERS event hexadecimal-encoded binary data.
 * With this API, data can be accessed block by block by the application.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @param dest The destination address for the copied binary data.
 * @param length The size of the destination buffer and then the length of
 * the data that must be copied.
 * @return The size of the data actually copied or an EPX error code.
 */
DC_RT_FMAC1 size_t epx_read_hex_binary(void * context, unsigned char * dest, size_t length);

/** Reads the data for CHARACTERS or ATTRIBUTE events as a boolean.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @param[out] b A pointer to a boolean supplied by the caller.
 * @return An EPX error code, especially EPX_ERR_INVALID_REPRESENTATION if the
 * XML value is not a valid XML schema boolean.
 */
DC_RT_FMAC1 int epx_get_boolean(void * context, epx_boolean_t * b);

/** Reads the data for CHARACTERS or ATTRIBUTE events as a long int.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @param[out] l A pointer to a long integer supplied by the caller.
 * @return An EPX error code, especially EPX_ERR_INVALID_REPRESENTATION if the
 * XML value is not a valid XML schema integer or EPX_ERR_OUT_OF_RANGE if it
 * exceeds the output capacity.
 */
DC_RT_FMAC1 int epx_get_long(void * context, long * l);

/** Reads the data for CHARACTERS or ATTRIBUTE events as a unsigned long int.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @param[out] ul A pointer to a unsigned long integer supplied by the caller.
 * @return An EPX error code, especially EPX_ERR_INVALID_REPRESENTATION if the
 * XML value is not a valid XML schema integer or EPX_ERR_OUT_OF_RANGE if it
 * exceeds the output capacity.
 */
DC_RT_FMAC1 int epx_get_ulong(void * context, unsigned long * ul);

/** Reads the data for CHARACTERS or ATTRIBUTE events as a float.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @param[out] f A pointer to a float supplied by the caller.
 * @return An EPX error code, especially EPX_ERR_INVALID_REPRESENTATION if the
 * XML value is not a valid XML schema float or EPX_ERR_OUT_OF_RANGE if it
 * exceeds the output capacity.
 */
DC_RT_FMAC1 int epx_get_float(void * context, float * f);

/** Reads the data for CHARACTERS or ATTRIBUTE events as a double.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @param[out] d A pointer to a double supplied by the caller.
 * @return An EPX error code, especially EPX_ERR_INVALID_REPRESENTATION if the
 * XML value is not a valid XML schema double or EPX_ERR_OUT_OF_RANGE if it
 * exceeds the output capacity.
 */
DC_RT_FMAC1 int epx_get_double(void * context, double * d);

/** Reads the data for CHARACTERS or ATTRIBUTE events as a qualified name.
 * @param context The implementation dependent parsing context returned by
 * epx_new_parser.
 * @param[out] qn A pointer to a qualified name structure supplied by the caller.
 * @return An EPX error code, especially EPX_ERR_INVALID_REPRESENTATION if the
 * XML value is not a valid XML schema QName.
 */
DC_RT_FMAC1 int epx_get_qname(void * context, epx_qname_t * qn);

// @}
// @}

/***************************************************************************//**
* \name Serialization API
* @{
*******************************************************************************/

/*----------------------------------------------------------------------------*\
 *                                 Constants                                  *
\*----------------------------------------------------------------------------*/

/* Serializer boolean capabilities */

#define EPX_OPT_UTF8_INPUT 0x01	/**< Inputs are supposed in multi-byte UTF-8, else UCS-4 cast to char (i.e. ISO-LATIN-1). */
#define EPX_OPT_GENERATE_PREFIXES 0x02	/**< Generate automatically namespace prefixes when not specified using the serialization API.  */
#define EPX_OPT_GENERATE_UTF16 0x04	/**< The XML generated is encoded using UTF-16. The default that must be supported is UTF-8. */
#define EPX_OPT_SERIALIZER_STREAMING 0x08	/**< Streaming on data buffers is supported. */
#define EPX_OPT_SUPPORTS_COMMENTS 0x10	/**< The serializer implementation supports comment events. */
#define EPX_OPT_INDENT 0x20	/**< If enabled, output will be indented automatically. */

/** Gets the serializer default options settings.
 * @param implementation A pointer on the EPX implementation. Reserved for later
 * use (set NULL).
 * @return An integer describing the default behavior of the serializer during
 * processing. The default options values maybe the only one supported. This can
 * be established by calling the is_serializer_option_settable API for a
 * particular option.
 */
DC_RT_FMAC1 int epx_get_serializer_options(void * implementation);

/** Test if a parser option can be enabled of disabled.
 * @param implementation A pointer on the EPX implementation. Reserved for later
 * use (set NULL).
 * @param option The option to test.
 * @return 1 if settable, 0 if not and EPX_ERR_UNKNOWN_OPTION if the option is
 * not recognized.
 */
DC_RT_FMAC1 int epx_is_serializer_option_settable(void * implementation, int option);

/** Creates a new serializer from a given EPX implementation.
 * @param implementation A pointer on the EPX implementation. Reserved for later
 * use (setting NULL will select the implementation over gSOAP).
 * @param impl_data Custom data that may be required by the implementation for
 * instance to avoid dynamic allocation for the serializer context. If NULL is
 * provided as \a implementation, this should point to a \a soap structure.
 * @return An implementation dependent serialization context. If an error
 * occurred the implementation is responsible for supplying an error code
 * through the parsing context.
 */
DC_RT_FMAC1 void * epx_new_serializer(void * implementation, void * impl_data);

/** Returns the last the implementation dependent error code.
 * @param context The implementation dependent serialization context returned by
 * epx_new_serializer.
 * @return An implementation-dependent error code.
 */
DC_RT_FMAC1 int epx_get_serializer_error(void * context);

/** Allows the implementation to free serialization resources.
 * @param context The implementation dependent serialization context returned by
 * epx_new_serializer.
 */
DC_RT_FMAC1 void epx_delete_serializer(void * context);

/**	Start the XML serialization for the implementation opaque id.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param dest Data identifying somehow the XML file or stream. If the
 * implementation is gSOAP, may be NULL (current message processing).
 * @param options Which capabilities should be used for XML processing.
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_start_document(void * context, void * dest, int options);

/** Ends the XML document serialization.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_end_document(void * context);

/** Generates a START_ELEMENT event.
 * Implementation should start to serialize the start tag but should wait for
 * the attributes and prefix events to be sent and to be able to close the
 * tag or generate an empty element tag.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param ns_uri The namespace URI of the element. If NULL, the element is
 * considered to have no namespace. If no prefix was defined for the
 * namespace the implementation will either generate a prefix or raise an error
 * according to the GENERATE_PREFIXES option.
 * @param local_name The namespace URI of the element.
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_start_element(void * context, char * ns_uri, char * local_name);

/** Generates a PREFIX_DEFINITION event.
 * The prefix definition will be attached to the previous start element event.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param prefix The prefix name. If NULL, the default prefix is being defined.
 * @param ns_uri The namespace URI for which the prefix is defined.
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_define_prefix(void * context, char * prefix, char * ns_uri);

/** Generates an ATTRIBUTE event.
 * Attributes will be attached to the previous start element event.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param ns_uri The namespace URI of the attribute. If NULL, the attribute is
 * considered element local (no namespace). If no prefix was defined for the
 * namespace the implementation will either generate a prefix or raise an error
 * according to the GENERATE_PREFIXES option.
 * @param local_name The namespace URI of the attribute.
 * @param value The value of the attribute.
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_add_attribute(void * context, char * ns_uri, char * local_name, char * value);

/** Generates an END_ELEMENT event.
 * Implementation should generate an end tag or close the pending empty element
 * tag.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param ns_uri The namespace URI of the element. If NULL, the element is
 * considered to have no namespace. If no prefix was defined for the
 * namespace the implementation will either generate a prefix or raise an error
 * according to the GENERATE_PREFIXES option.
 * @param local_name The namespace URI of the element.
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_end_element(void * context, char * ns_uri, char * local_name);

/** Writes character data using a string.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param value A string containing UTF-8 or ISO-LATIN-1 (according
 * to the option set for serialization).
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_put_characters(void * context, char* value);

/** Writes character data using buffer copy (direct access).
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param buffer A character buffer containing UTF-8 or ISO-LATIN-1 (according
 * to the option set for serialization).
 * @param length The size of character data contained in the buffer.
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_write_characters(void * context, char* buffer, size_t length);

/** Generates a START_CDATA event.
 * A CDATA start tag is generated.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_start_cdata(void * context);

/** Generates a END_CDATA event.
 * A CDATA end tag is generated.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_end_cdata(void * context);

/** Generates a COMMENT event.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param value The comment string.
 * @return EPX_OK if no error occurred. An EPX error code otherwise especially
 * EPX_ERR_NOT_SUPPORTED if option SUPPORTS_COMMENTS is not supported or
 * activated.
 */
DC_RT_FMAC1 int epx_comment(void * context, char * value);

/** \name Typed serialization API
 * @{
 */

/** Writes a CHARACTER event with base64 binary data into the XML stream.
 * Note that no more that one CHARACTER event should be sent consecutively
 * because of base64 padding.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param buffer A character buffer containing binary data.
 * @param length The size of binary data contained in the buffer.
 * @return EPX_OK if no error occurred. An EPX error code otherwise especially
 * EPX_ERR_NOT_SUPPORTED if option PARSER_STREAMING is not supported or
 * activated.
 */
DC_RT_FMAC1 int epx_write_base64_binary(void * context, unsigned char * buffer, size_t length);

/** Writes hexadecimal-encoded binary data into the XML stream.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param buffer A character buffer containing binary data.
 * @param length The size of binary data contained in the buffer.
 * @return EPX_OK if no error occurred. An EPX error code otherwise especially
 * EPX_ERR_NOT_SUPPORTED if option PARSER_STREAMING is not supported or
 * activated.
 */
DC_RT_FMAC1 int epx_write_hex_binary(void * context, unsigned char * buffer, size_t length);

/** Produces a CHARACTER event from a boolean.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param b A boolean value.
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_put_boolean(void * context, epx_boolean_t b);

/** Produces a CHARACTER event from a long integer.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param l A long integer value.
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_put_long(void * context, long l);

/** Produces a CHARACTER event from a unsigned long integer.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param ul A unsigned long integer value.
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_put_ulong(void * context, unsigned long ul);

/** Produces a CHARACTER event from a float.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param f A float value.
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_put_float(void * context, float f);

/** Produces a CHARACTER event from a double.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param d A double value.
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_put_double(void * context, double d);

/** Produces a CHARACTER event from a qualified name.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param qn A qualified name value.
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_put_qname(void * context, epx_qname_t * qn);

/** Writes a base64 binary ATTRIBUTE event into the XML stream.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param ns_uri The namespace URI of the attribute. If NULL, the attribute is
 * considered element local (no namespace). If no prefix was defined for the
 * namespace the implementation will either generate a prefix or raise an error
 * according to the GENERATE_PREFIXES option.
 * @param local_name The namespace URI of the attribute.
 * @param buffer A character buffer containing binary data.
 * @param length The size of binary data contained in the buffer.
 * @return EPX_OK if no error occurred. An EPX error code otherwise especially
 * EPX_ERR_NOT_SUPPORTED if option PARSER_STREAMING is not supported or
 * activated.
 */
DC_RT_FMAC1 int epx_add_base64_attribute(void * context, char * ns_uri, char * local_name, unsigned char * buffer, size_t length);

/** Writes hexadecimal-encoded binary ATTRIBUTE event into the XML stream.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param ns_uri The namespace URI of the attribute. If NULL, the attribute is
 * considered element local (no namespace). If no prefix was defined for the
 * namespace the implementation will either generate a prefix or raise an error
 * according to the GENERATE_PREFIXES option.
 * @param local_name The namespace URI of the attribute.
 * @param buffer A character buffer containing binary data.
 * @param length The size of binary data contained in the buffer.
 * @return EPX_OK if no error occurred. An EPX error code otherwise especially
 * EPX_ERR_NOT_SUPPORTED if option PARSER_STREAMING is not supported or
 * activated.
 */
DC_RT_FMAC1 int epx_add_hex_attribute(void * context, char * ns_uri, char * local_name, unsigned char * buffer, size_t length);

/** Produces an ATTRIBUTE event from a boolean.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param ns_uri The namespace URI of the attribute. If NULL, the attribute is
 * considered element local (no namespace). If no prefix was defined for the
 * namespace the implementation will either generate a prefix or raise an error
 * according to the GENERATE_PREFIXES option.
 * @param local_name The namespace URI of the attribute.
 * @param b A boolean value.
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_add_boolean_attribute(void * context, char * ns_uri, char * local_name, epx_boolean_t b);

/** Produces an ATTRIBUTE event from a long integer.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param ns_uri The namespace URI of the attribute. If NULL, the attribute is
 * considered element local (no namespace). If no prefix was defined for the
 * namespace the implementation will either generate a prefix or raise an error
 * according to the GENERATE_PREFIXES option.
 * @param local_name The namespace URI of the attribute.
 * @param l A long integer value.
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_add_long_attribute(void * context, char * ns_uri, char * local_name, long l);

/** Produces an ATTRIBUTE event from a unsigned long integer.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param ns_uri The namespace URI of the attribute. If NULL, the attribute is
 * considered element local (no namespace). If no prefix was defined for the
 * namespace the implementation will either generate a prefix or raise an error
 * according to the GENERATE_PREFIXES option.
 * @param local_name The namespace URI of the attribute.
 * @param ul A unsigned long integer value.
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_add_ulong_attribute(void * context, char * ns_uri, char * local_name, unsigned long ul);

/** Produces an ATTRIBUTE event from a float.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param ns_uri The namespace URI of the attribute. If NULL, the attribute is
 * considered element local (no namespace). If no prefix was defined for the
 * namespace the implementation will either generate a prefix or raise an error
 * according to the GENERATE_PREFIXES option.
 * @param local_name The namespace URI of the attribute.
 * @param f A float value.
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_add_float_attribute(void * context, char * ns_uri, char * local_name, float f);

/** Produces an ATTRIBUTE event from a double.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param ns_uri The namespace URI of the attribute. If NULL, the attribute is
 * considered element local (no namespace). If no prefix was defined for the
 * namespace the implementation will either generate a prefix or raise an error
 * according to the GENERATE_PREFIXES option.
 * @param local_name The namespace URI of the attribute.
 * @param d A double value.
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_add_double_attribute(void * context, char * ns_uri, char * local_name, double d);

/** Produces an ATTRIBUTE event from a qualified name.
 * @param context The implementation dependent serialization context returned
 * by epx_new_serializer.
 * @param ns_uri The namespace URI of the attribute. If NULL, the attribute is
 * considered element local (no namespace). If no prefix was defined for the
 * namespace the implementation will either generate a prefix or raise an error
 * according to the GENERATE_PREFIXES option.
 * @param local_name The namespace URI of the attribute.
 * @param qn A qualified name value.
 * @return EPX_OK if no error occurred. An EPX error code otherwise.
 */
DC_RT_FMAC1 int epx_add_qname_attribute(void * context, char * ns_uri, char * local_name, epx_qname_t * qn);

// @}
// @}

#ifdef __cplusplus
}
#endif

#endif /*EPX_H_*/
