/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2111 $
|                     $Date: 2009-02-24 13:53:36 +0100 (mar, 24 fév 2009) $
\*============================================================================*/

/******************************************************************************\
 *                          XML configuration Serializer                      *
\******************************************************************************/
#include "dc/dc_Constants.h"
#include "dc/dc_Epx.h"
#include "dcCOMN_Tools.h"
#include "dcXCONF_Serializer.h"
#include "dcXCONF_Manager.h"
#include "dcXCONF_Definitions.h"

/*----------------------------------- Types ----------------------------------*/

typedef void * (*get_cbk) (int index, void * cbk_data);

/*------------------------- Static Functions prototypes ----------------------*/

static int serialize_list_element(void * szr_ctx, int nb_values, get_cbk cbk, void * cbk_data);
static int serialize_qname_list(void * szr_ctx, int nb_values, get_cbk cbk, void * cbk_data);
static struct prefixed_qname * get_dev_type_cbk(int index, short * href_device);
static int serialize_types(void * szr_ctx, short href_device, DC_BOOL dyndepl_pfx);
static void * get_scope_cbk (int index, short * href_device);
static int serialize_scopes(void * szr_ctx, short href_device, DC_BOOL dyndepl_pfx);
static int serialize_this_model(void * szr_ctx, short href_device, DC_BOOL wdp_pfx);
static int serialize_this_device(void * szr_ctx, short href_device, DC_BOOL wdp_pfx);
static struct prefixed_qname * get_type_hint_cbk(int index, struct discovery_hint * hint);
static char * get_scope_hint_cbk(int index, struct discovery_hint * hint);
static int serialize_service(void * szr_ctx, short href_service, DC_BOOL pfxs);
static int serialize_device(void * szr_ctx, short href_device, DC_BOOL config);
static int add_pqn_att(void * szr_ctx, char * ns_uri, char * lname, struct prefixed_qname *pqn);
static int serialize_service_class(void * szr_ctx, short href_sclass, DC_BOOL dyndepl_pfx);

/*----------------------------------------------------------------------------*\
 *                           Serialization utilities                          *
\*----------------------------------------------------------------------------*/

static int serialize_list_element(void * szr_ctx, int nb_values, get_cbk cbk, void * cbk_data)
{
	int i = 0, ret = DPWS_OK;
	if ((ret = epx_put_characters(szr_ctx, cbk(i++, cbk_data))))
		goto exit;
	for (; i < nb_values; i++) {
		if ((ret = epx_put_characters(szr_ctx, " ")) || (ret = epx_put_characters(szr_ctx, cbk(i, cbk_data))))
			goto exit;
	}
exit:
	return ret;
}

static int serialize_qname_list(void * szr_ctx, int nb_values, get_cbk cbk, void * cbk_data)
{
	int i = 0, ret = DPWS_OK;
	struct prefixed_qname *pqn;

	// prefix definition
	for (i = 0; i < nb_values; i++) {
		pqn = cbk(i, cbk_data);
		if ((ret = epx_define_prefix(szr_ctx, pqn->prefix, pqn->qname.ns)))
			goto exit;
	}

	for (i = 0; i < nb_values; i++) {
		pqn = cbk(i, cbk_data);
		if (i > 0 && (ret = epx_put_characters(szr_ctx, " ")))
			goto exit;
		if ((ret = epx_put_qname(szr_ctx, (epx_qname_t *)&pqn->qname)))
			goto exit;
	}
exit:
	return ret;
}

/*----------------------------------------------------------------------------*\
 *                            Device serialization                            *
\*----------------------------------------------------------------------------*/

// NOTE: functions exist to retrieve a device by its uuid. Check the handle is checkout
// to ensure data returned won't be unallocated

/* Registry serialization */

// Types
static struct prefixed_qname * get_dev_type_cbk(int index, short * href_device) {
	return dpws_get_ptr_att_item(*href_device, DPWS_PTR_PREFIXED_TYPE, index);
}

static int serialize_types(void * szr_ctx, short href_device, DC_BOOL dyndepl_pfx)
{
	int n = dpws_get_att_count(href_device, DPWS_PTR_PREFIXED_TYPE), ret = DPWS_OK;
	if (n > 0)
	{
		if ((ret = epx_start_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_TYPES)))
			goto exit;
		if (dyndepl_pfx && (ret = epx_define_prefix(szr_ctx, DYNDEPL_NS_PREFIX, DYNDEPL_NS_URI)))
			goto exit;
		if ((ret = serialize_qname_list(szr_ctx, n, (get_cbk)get_dev_type_cbk, &href_device))
				|| (ret = epx_end_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_TYPES)));
			goto exit;
	}
exit:
	return ret;
}

int dpws_serialize_types(void * szr_ctx, short href_device)
{
	return serialize_types(szr_ctx, href_device, DC_TRUE);
}

// Scopes
static void * get_scope_cbk (int index, short * href_device)
{
	return dpws_get_ptr_att_item(*href_device, DPWS_STR_SCOPE, index);
}

static int serialize_scopes(void * szr_ctx, short href_device, DC_BOOL dyndepl_pfx)
{
	int n = dpws_get_att_count(href_device, DPWS_STR_SCOPE), ret = DPWS_OK;
	if (n > 0)
	{
		if ((ret = epx_start_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_SCOPES)))
			goto exit;
		if (dyndepl_pfx && (ret = epx_define_prefix(szr_ctx, DYNDEPL_NS_PREFIX, DYNDEPL_NS_URI)))
			goto exit;
		if ((ret = serialize_list_element(szr_ctx, n, (get_cbk)get_scope_cbk, &href_device))
				|| (ret = epx_end_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_SCOPES)))
			goto exit;
	}
exit:
	return ret;
}

int dpws_serialize_scopes(void * szr_ctx, short href_device)
{
	return serialize_scopes(szr_ctx, href_device, DC_TRUE);
}

// ThisModel
static int serialize_this_model(void * szr_ctx, short href_device, DC_BOOL wdp_pfx)
{
	struct localized_string * pls;
	char * s;
	int att_nb, i, ret = DPWS_OK;

	if ((ret = epx_start_element(szr_ctx, WDP_URI, WDP_ELT_THIS_MODEL)))
		goto exit;
	if (wdp_pfx && (ret = epx_define_prefix(szr_ctx, WDP_PREFIX, WDP_URI)))
		goto exit;

	att_nb = dpws_get_att_count(href_device, DPWS_PTR_MANUFACTURER);
	for (i = 0; i < att_nb; i ++) {
		pls = dpws_get_ptr_att(href_device, DPWS_PTR_MANUFACTURER);
		if ((ret = epx_start_element(szr_ctx, WDP_URI, WDP_ELT_MANUFACTURER)))
			goto exit;
		if (pls->lang && (ret = epx_add_attribute(szr_ctx, EPX_XML_NS_URI, XML_ATT_LANG, pls->lang)))
			goto exit;
		if ((ret = epx_put_characters(szr_ctx, pls->s))
				|| (ret = epx_end_element(szr_ctx, WDP_URI, WDP_ELT_MANUFACTURER)))
			goto exit;
	}

	s = dpws_get_ptr_att(href_device, DPWS_STR_MANUFACTURER_URL);
	if (s) {
		if ((ret = epx_start_element(szr_ctx, WDP_URI, WDP_ELT_MANUFACTURER_URL))
				|| (ret = epx_put_characters(szr_ctx, s))
				|| (ret = epx_end_element(szr_ctx, WDP_URI, WDP_ELT_MANUFACTURER_URL)))
			goto exit;
	}

	att_nb = dpws_get_att_count(href_device, DPWS_PTR_MODEL_NAME);
	for (i = 0; i < att_nb; i ++) {
		pls = dpws_get_ptr_att(href_device, DPWS_PTR_MODEL_NAME);
		if ((ret = epx_start_element(szr_ctx, WDP_URI, WDP_ELT_MODEL_NAME)))
			goto exit;
		if (pls->lang && (ret = epx_add_attribute(szr_ctx, EPX_XML_NS_URI, XML_ATT_LANG, pls->lang)))
			goto exit;
		if ((ret = epx_put_characters(szr_ctx, pls->s))
				|| (ret = epx_end_element(szr_ctx, WDP_URI, WDP_ELT_MODEL_NAME)))
			goto exit;
	}

	s = dpws_get_ptr_att(href_device, DPWS_STR_MODEL_NUMBER);
	if (s) {
		if ((ret = epx_start_element(szr_ctx, WDP_URI, WDP_ELT_MODEL_NUMBER))
				|| (ret = epx_put_characters(szr_ctx, s))
				|| (ret = epx_end_element(szr_ctx, WDP_URI, WDP_ELT_MODEL_NUMBER)))
			goto exit;
	}

	s = dpws_get_ptr_att(href_device, DPWS_STR_MODEL_URL);
	if (s) {
		if ((ret = epx_start_element(szr_ctx, WDP_URI, WDP_ELT_MODEL_URL))
				|| (ret = epx_put_characters(szr_ctx, s))
				|| (ret = epx_end_element(szr_ctx, WDP_URI, WDP_ELT_MODEL_URL)))
			goto exit;
	}

	s = dpws_get_ptr_att(href_device, DPWS_STR_PRESENTATION_URL);
	if (s) {
		if ((ret = epx_start_element(szr_ctx, WDP_URI, WDP_ELT_PRESENTATION_URL))
		|| (ret = epx_put_characters(szr_ctx, s))
		|| (ret = epx_end_element(szr_ctx, WDP_URI, WDP_ELT_PRESENTATION_URL)))
			goto exit;
	}

	ret = epx_end_element(szr_ctx, WDP_URI, WDP_ELT_THIS_MODEL);

exit:
	return ret;
}

int dpws_serialize_this_model(void * szr_ctx, short href_device)
{
	return serialize_this_model(szr_ctx, href_device, DC_TRUE);
}

// ThisDevice
static int serialize_this_device(void * szr_ctx, short href_device, DC_BOOL wdp_pfx)
{
	int att_nb, i, ret = DPWS_OK;
	struct localized_string * pls;
	char * s;

	if ((ret = epx_start_element(szr_ctx, WDP_URI, WDP_ELT_THIS_DEVICE)))
		goto exit;

	if (wdp_pfx && (ret = epx_define_prefix(szr_ctx, WDP_PREFIX, WDP_URI)))
		goto exit;

	att_nb = dpws_get_att_count(href_device, DPWS_PTR_FRIENDLY_NAME);
	for (i = 0; i < att_nb; i ++) {
		pls = dpws_get_ptr_att(href_device, DPWS_PTR_FRIENDLY_NAME);
		if ((ret = epx_start_element(szr_ctx, WDP_URI, WDP_ELT_FRIENDLY_NAME)))
			goto exit;
		if (pls->lang && (ret = epx_add_attribute(szr_ctx, EPX_XML_NS_URI, XML_ATT_LANG, pls->lang)))
			goto exit;
		if ((ret = epx_put_characters(szr_ctx, pls->s))
				|| (ret = epx_end_element(szr_ctx, WDP_URI, WDP_ELT_FRIENDLY_NAME)))
			goto exit;
	}

	s = dpws_get_ptr_att(href_device, DPWS_STR_FIRMWARE_VERSION);
	if (s) {
		if ((ret = epx_start_element(szr_ctx, WDP_URI, WDP_ELT_FIRMWARE_VERSION))
				|| (ret = epx_put_characters(szr_ctx, s))
				|| (ret = epx_end_element(szr_ctx, WDP_URI, WDP_ELT_FIRMWARE_VERSION)))
			goto exit;
	}

	s = dpws_get_ptr_att(href_device, DPWS_STR_SERIAL_NUMBER);
	if (s) {
		if ((ret = epx_start_element(szr_ctx, WDP_URI, WDP_ELT_SERIAL_NUMBER))
				|| (ret = epx_put_characters(szr_ctx, s))
				|| (ret = epx_end_element(szr_ctx, WDP_URI, WDP_ELT_SERIAL_NUMBER)))
			goto exit;
	}

	ret = epx_end_element(szr_ctx, WDP_URI, WDP_ELT_THIS_DEVICE);

exit:
	return ret;
}

int dpws_serialize_this_device(void * szr_ctx, short href_device)
{
	return serialize_this_device(szr_ctx, href_device, DC_TRUE);
}

// Service
static struct prefixed_qname * get_type_hint_cbk(int index, struct discovery_hint * hint) {
	return (struct prefixed_qname *)GET_ENTRY(&hint->types, index);
}

static char * get_scope_hint_cbk(int index, struct discovery_hint * hint) {
	return *(char **)GET_ENTRY(&hint->scopes, index);
}

static int serialize_service(void * szr_ctx, short href_service, DC_BOOL pfxs)
{
	int sz = 5, status = DPWS_OK, i, j;
	short * hrefs = NULL;
	unsigned long h_sclass;
	struct ws_binding_ref * b_ref;
	struct discovery_hint * d_hint;
	struct property_value * p_value;
	struct service_config *s_config = xconf_get_service(href_service);
	DC_BOOL free_serv = DC_FALSE;

	if (!s_config) {	// the dyndepl service object may not be created if no ref or property
		if (!(s_config = xconf_init_service()))
			return SOAP_EOM;
		s_config->href = href_service;
		free_serv = DC_TRUE;
	}

	if ((status = epx_start_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_SERVICE)))
		goto exit;
	if (pfxs &&
			((status = epx_define_prefix(szr_ctx, DYNDEPL_NS_PREFIX, DYNDEPL_NS_URI))
			|| (status = epx_define_prefix(szr_ctx, WSA_PREFIX, WSA_URI))))
			goto exit;
	if ((status = epx_add_attribute(szr_ctx, NULL, DYNDEPL_ATT_SERVICE_ID, dpws_get_ptr_att(s_config->href, DPWS_STR_SERVICE_ID))))
		goto exit;

	// service class
	h_sclass = dpws_get_int_att(s_config->href, DPWS_INT_CLASS_HREF);
	if (h_sclass < 0)
		goto exit;
	if ((status = epx_start_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_SERVICE_CLASS))
			|| (status = epx_add_attribute(szr_ctx, NULL, DYNDEPL_ATT_SERVICE_CLASS_ID, dpws_get_ptr_att((short)h_sclass, DPWS_STR_ID)))
			|| (status = epx_end_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_SERVICE_CLASS)))
		goto exit;

	// service ports
	do {
		sz*=2;
		if (hrefs)
			DC_FREE(DC_MEM_TRANSIENT, hrefs);
		if (!(hrefs = DC_MALLOC(DC_MEM_TRANSIENT, sz * sizeof(short)))) {
			status = SOAP_EOM;
			goto exit;
		}
		status = dpws_get_service_port_handles(s_config->href, hrefs, &sz);
	}
	while (status == DPWS_ERR_MORE_RESULTS);

	if (status)
		goto exit;

	for (i = 0; i < sz; i++) {
		if ((status = epx_start_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_SERVICE_PORT))
				|| (status = epx_start_element(szr_ctx, WSA_URI, WSA_ELT_ADDRESS))
				|| (status = epx_put_characters(szr_ctx, dpws_get_ptr_att(hrefs[i], DPWS_STR_ADDRESS)))
				|| (status = epx_end_element(szr_ctx, WSA_URI, WSA_ELT_ADDRESS))
				|| (status = epx_end_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_SERVICE_PORT)))
			goto exit;
	}

	// references
	for (i = 0; i < s_config->refs.nb; i++) {
		b_ref = (struct ws_binding_ref *)GET_ENTRY(&s_config->refs, i);
		if ((status = epx_start_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_REFERENCE))
				|| (status = epx_add_attribute(szr_ctx, NULL, DYNDEPL_ATT_NAME, b_ref->name)))
			goto exit;
		if (b_ref->epr && !b_ref->wsd_hints) {	// Note: the EPR field can be used as a cache
			if ((status = epx_start_element(szr_ctx, WSA_URI, WSA_ELT_ENDPOINT_REFERENCE))
					|| (status = epx_start_element(szr_ctx, WSA_URI, WSA_ELT_ADDRESS))
					|| (status = epx_put_characters(szr_ctx, b_ref->epr->address))
					|| (status = epx_end_element(szr_ctx, WSA_URI, WSA_ELT_ADDRESS))
					|| (status = epx_end_element(szr_ctx, WSA_URI, WSA_ELT_ENDPOINT_REFERENCE)))
				goto exit;
		}
		else {	// discovery hint
			if ((status = epx_start_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_DISCOVERY_HINTS)))
				goto exit;
			if (b_ref->wsd_hints->binding_time == WSD_HINT_BT_RUNTIME &&
					(status = epx_add_attribute(szr_ctx, NULL, DYNDEPL_ATT_BINDING_TIME, DYNDEPL_ENUM_WSD_HINT_BT_RUNTIME)))	// deployement is default
				goto exit;
			if (b_ref->wsd_hints->on_multiple_matches == WSD_HINT_MM_ACTION_FAIL &&
					(status = epx_add_attribute(szr_ctx, NULL, DYNDEPL_ATT_ON_MULTIPLE_MATCHES, DYNDEPL_ENUM_WSD_HINT_MM_ACTION_FAIL)))	// pick one is the default
				goto exit;

			switch (b_ref->wsd_hints->on_reference_lost)
			{
				case WSD_HINT_RL_ACTION_FAIL:
					if ((status = epx_add_attribute(szr_ctx, NULL, DYNDEPL_ATT_ON_REFERENCE_LOST, DYNDEPL_ENUM_WSD_HINT_RL_ACTION_FAIL)))
						goto exit;
					break;
				case WSD_HINT_RL_ACTION_IGNORE:
					if ((status = epx_add_attribute(szr_ctx, NULL, DYNDEPL_ATT_ON_REFERENCE_LOST, DYNDEPL_ENUM_WSD_HINT_RL_ACTION_IGNORE)))
						goto exit;
					break;
				default:
				// case WSD_HINT_RL_ACTION_RETRY:	default att value
					break;
			}
			if (b_ref->wsd_hints->service_id &&
					(status = epx_add_attribute(szr_ctx, NULL, DYNDEPL_ATT_SERVICE_ID, b_ref->wsd_hints->service_id)))
				goto exit;

			for (j = 0; j < b_ref->wsd_hints->hints.nb; j++)
			{
				d_hint = (struct discovery_hint *) GET_ENTRY(&b_ref->wsd_hints->hints, j);
				if ((status = epx_start_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_HINT)))
					goto exit;

				if (d_hint->types.nb > 0)
				{
					if ((status = epx_start_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_TYPES))
							|| (status = serialize_qname_list(szr_ctx, d_hint->types.nb, (get_cbk)get_type_hint_cbk, d_hint))
							|| (status = epx_end_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_TYPES)))
						goto exit;
				}

				if (d_hint->scopes.nb > 0)
				{
					if ((status = epx_start_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_SCOPES))
							|| (status = serialize_list_element(szr_ctx, d_hint->scopes.nb, (get_cbk)get_scope_hint_cbk, d_hint))
							|| (status = epx_end_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_SCOPES)))
						goto exit;
				}
				if ((status = epx_end_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_HINT)))
					goto exit;
			}

			if ((status = epx_end_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_DISCOVERY_HINTS)))
				goto exit;
		}
		if ((status = epx_end_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_REFERENCE)))
			goto exit;
	}

	// properties
	for (i = 0; i < s_config->property_values.nb; i++) {
		p_value = (struct property_value *)GET_ENTRY(&s_config->property_values, i);
		if ((status = epx_start_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_PROPERTY_VALUE))
				|| (status = epx_add_attribute(szr_ctx, NULL, DYNDEPL_ATT_NAME, p_value->name))
				|| (status = epx_put_characters(szr_ctx, p_value->value))
				|| (status = epx_end_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_PROPERTY_VALUE)))
			goto exit;
	}

	status = epx_end_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_SERVICE);

exit:
	if (free_serv)
		xconf_free_service(s_config);
	if (hrefs) {
		status = dpws_release_handles(hrefs, sz);
		DC_FREE(DC_MEM_TRANSIENT, hrefs);
	}
	return status;
}

int dpws_serialize_service(void * szr_ctx, short href_service)
{
	return serialize_service(szr_ctx, href_service, DC_TRUE);
}

// whole device
static int serialize_device(void * szr_ctx, short href_device, DC_BOOL config)
{
	int sz = 5, status = DPWS_OK, i;
	short * hrefs = NULL;
	char * s;

	if (config) {
		if ((status = epx_start_element(szr_ctx, DC_CONFIG_NS_URI, DYNDEPL_ELT_DEVICE))
				|| (status = epx_add_ulong_attribute(szr_ctx, NULL, DC_CONFIG_ATT_METADATA_VERSION, dpws_get_int_att(href_device, DPWS_INT_METADATA_VERSION))))
			goto exit;
	}
	else {
		if ((status = epx_start_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_DEVICE))
				|| (status = epx_define_prefix(szr_ctx, DYNDEPL_NS_PREFIX, DYNDEPL_NS_URI))
				|| (status = epx_define_prefix(szr_ctx, WDP_PREFIX, WDP_URI))
				|| (status = epx_define_prefix(szr_ctx, WSA_PREFIX, WSA_URI)))
			goto exit;
	}

	/* address */
	s = dpws_get_ptr_att(href_device, DPWS_STR_DEVICE_ID);
	if (s) {
		if ((status = epx_start_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_ADDRESS))
				|| (status = epx_put_characters(szr_ctx, s))
				|| (status = epx_end_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_ADDRESS)))
			goto exit;
	}

	if ((status = serialize_types(szr_ctx, href_device, DC_FALSE))
			|| (status = serialize_scopes(szr_ctx, href_device, DC_FALSE))
			|| (status = serialize_this_model(szr_ctx, href_device, DC_FALSE))
			|| (status = serialize_this_device(szr_ctx, href_device, DC_FALSE)))
		goto exit;

	// services
	do {
		sz*=2;
		if (hrefs)
			DC_FREE(DC_MEM_TRANSIENT, hrefs);
		if (!(hrefs = DC_MALLOC(DC_MEM_TRANSIENT, sz * sizeof(short))))
			return SOAP_EOM;
		status = dpws_get_service_handles(href_device, hrefs, &sz);
	}
	while (status == DPWS_ERR_MORE_RESULTS);

	if (status)
		goto exit;

	for (i = 0; i < sz; i++) {
		if ((status = serialize_service(szr_ctx, hrefs[i], DC_FALSE)))
			goto exit;
	}

	status = epx_end_element(szr_ctx, config ? DC_CONFIG_NS_URI : DYNDEPL_NS_URI, DYNDEPL_ELT_DEVICE);

exit:
	if (hrefs) {
		status = dpws_release_handles(hrefs, sz);
		DC_FREE(DC_MEM_TRANSIENT, hrefs);
	}
	return status;
}

int dpws_serialize_device(void * szr_ctx, short href_device)
{
	return serialize_device(szr_ctx, href_device, DC_FALSE);
}

/*----------------------------------------------------------------------------*\
 *                         Service class serialization                        *
\*----------------------------------------------------------------------------*/

// NOTE: functions exist to retrieve a service class by its id. Check the handle is checkout
// to ensure data returned won't be unallocated
static int add_pqn_att(void * szr_ctx, char * ns_uri, char * lname, struct prefixed_qname *pqn)
{
	int ret = DPWS_OK;
	if (pqn->qname.ns && (ret =	epx_define_prefix(szr_ctx, pqn->prefix, pqn->qname.ns)))
		goto exit;
	ret = epx_add_qname_attribute(szr_ctx, ns_uri, lname, (epx_qname_t *)&pqn->qname);
exit:
	return ret;
}

static int serialize_service_class(void * szr_ctx, short href_sclass, DC_BOOL dyndepl_pfx)
{
	int status = DPWS_OK, i, att_nb;
	struct reference * ref;
	struct property * prop;
	struct wsdl_info * wsdl;
	struct service_class_config * sc_config = xconf_get_service_class(href_sclass);

	if ((status = epx_start_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_SERVICE_CLASS)))
		goto exit;
	if (dyndepl_pfx && (status = epx_define_prefix(szr_ctx, DYNDEPL_NS_PREFIX, DYNDEPL_NS_URI)))
		goto exit;
	if ((status = epx_add_attribute(szr_ctx, NULL, DYNDEPL_ATT_SERVICE_CLASS_ID, dpws_get_ptr_att(sc_config->href, DPWS_STR_ID))))
		goto exit;

	// interfaces
	att_nb = dpws_get_att_count(sc_config->href, DPWS_PTR_PREFIXED_TYPE);
	for (i = 0; i < att_nb;	i++)
	{
		if ((status = epx_start_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_INTERFACE))
				|| (status = add_pqn_att(szr_ctx, NULL, DYNDEPL_ATT_TYPE,
						dpws_get_ptr_att_item(sc_config->href, DPWS_PTR_PREFIXED_TYPE, i)))
				|| (status = epx_end_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_INTERFACE)))	// name @ not used yet
			goto exit;
	}

	// references
	for (i = 0; i < sc_config->refs.nb; i++) {
		ref = (struct reference *)GET_ENTRY(&sc_config->refs, i);
		if ((status = epx_start_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_REFERENCE))
				|| (status = add_pqn_att(szr_ctx, NULL, DYNDEPL_ATT_TYPE, &ref->type)))
			goto exit;
		if (ref->name && (status = epx_add_attribute(szr_ctx, NULL, DYNDEPL_ATT_NAME, ref->name)))
			goto exit;
		if ((status = epx_add_boolean_attribute(szr_ctx, NULL, DYNDEPL_ATT_MUST_SUPPLY, ref->must_supply))
				|| (status = epx_end_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_REFERENCE)))
			goto exit;
	}

	// properties
	for (i = 0; i < sc_config->properties.nb; i++) {
		prop = (struct property *)GET_ENTRY(&sc_config->properties, i);
		if ((status = epx_start_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_PROPERTY))
				|| (status = epx_add_attribute(szr_ctx, NULL, DYNDEPL_ATT_NAME, prop->p.name))
				|| (status = add_pqn_att(szr_ctx, NULL, DYNDEPL_ATT_TYPE, &prop->p.type)))
			goto exit;
		if (!prop->p.must_supply &&
				(status = epx_add_boolean_attribute(szr_ctx, NULL, DYNDEPL_ATT_MUST_SUPPLY, prop->p.must_supply)))
			goto exit;
		if (prop->multiple && epx_add_boolean_attribute(szr_ctx, NULL, DYNDEPL_ATT_MULTIPLE, prop->multiple))
			goto exit;
		if ((status = epx_put_characters(szr_ctx, prop->default_value))
				|| (status = epx_end_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_PROPERTY)))
			goto exit;
	}

	// WSDLs
	att_nb = dpws_get_att_count(sc_config->href, DPWS_PTR_WSDL);
	for (i = 0; i < att_nb;	i++) {
		wsdl = dpws_get_ptr_att_item(sc_config->href, DPWS_PTR_WSDL, i);
		if ((status = epx_start_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_WSDL_INFO))
				|| (status = epx_add_attribute(szr_ctx, NULL, DYNDEPL_ATT_TARGET_NS, wsdl->target_ns))
				|| (status = epx_add_attribute(szr_ctx, NULL, DYNDEPL_ATT_LOCATION, wsdl->location))
				|| (status = epx_end_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_WSDL_INFO)))
			goto exit;
	}

	// Implementation
	if ((status = sc_config->cbks.serialize_impl(sc_config->href, szr_ctx)))
		goto exit;

	status = epx_end_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_SERVICE_CLASS);

exit:
	return status;
}

int dpws_serialize_service_class(void * szr_ctx, short href_sclass)
{
	return serialize_service_class(szr_ctx, href_sclass, DC_TRUE);
}

/*----------------------------------------------------------------------------*\
 *                          DC Registry serialization                         *
\*----------------------------------------------------------------------------*/
int dpws_save_config(void * ctx, DC_BOOL inc_boot_seq, DC_BOOL as_fragment)
{
	int isz = 5, sz = 0, status = DPWS_OK, i;
	short * hrefs;
	char * str;
	long ul, ul2;

	DC_CHECK_PARAM(ctx);

	hrefs = DC_MALLOC(DC_MEM_TRANSIENT, isz * sizeof(short));
	if (!hrefs)
		return SOAP_EOM;

    if (inc_boot_seq && (status = dpws_set_int_att(DC_REGISTRY_HANDLE, DPWS_INT_BOOT_SEQ, dpws_get_int_att(DC_REGISTRY_HANDLE, DPWS_INT_BOOT_SEQ) + 1)))
   		goto exit;

	/* Start serialization */
    if (!as_fragment &&	(status = epx_start_document(ctx, NULL, EPX_OPT_SERIALIZER_STREAMING|EPX_OPT_INDENT)))
   		goto exit;

	if ((status = epx_start_element(ctx, DC_CONFIG_NS_URI, DC_CONFIG_ELT_CONFIG))
			|| (status = epx_define_prefix(ctx, DC_CONFIG_NS_PREFIX, DC_CONFIG_NS_URI))
			|| (status = epx_define_prefix(ctx, DYNDEPL_NS_PREFIX, DYNDEPL_NS_URI))
			|| (status = epx_define_prefix(ctx, WDP_PREFIX, WDP_URI))
			|| (status = epx_define_prefix(ctx, WSA_PREFIX, WSA_URI)))
  		goto exit;

	// boot sequence
	if ((status = epx_start_element(ctx, DC_CONFIG_NS_URI, DC_CONFIG_ELT_BOOT_SEQ))
			|| (status = epx_put_ulong(ctx, dpws_get_int_att(DC_REGISTRY_HANDLE, DPWS_INT_BOOT_SEQ)))
			|| (status = epx_end_element(ctx, DC_CONFIG_NS_URI, DC_CONFIG_ELT_BOOT_SEQ)))
  		goto exit;

	// Preferred lang
	str = dpws_get_ptr_att(DC_REGISTRY_HANDLE, DPWS_STR_PREFERRED_LANG);
	if (str) {
		if ((status = epx_start_element(ctx, DC_CONFIG_NS_URI, DC_CONFIG_ELT_PREFERRED_LANG))
				|| (status = epx_put_characters(ctx, str))
				|| (status = epx_end_element(ctx, DC_CONFIG_NS_URI, DC_CONFIG_ELT_PREFERRED_LANG)))
	  		goto exit;
	}

	/* HTTP listener */
	if ((status = epx_start_element(ctx, DC_CONFIG_NS_URI, DC_CONFIG_ELT_HTTP_LISTENER))
			|| (status = epx_add_long_attribute(ctx, NULL, DC_CONFIG_ATT_PORT, dpws_get_int_att(DC_REGISTRY_HANDLE, DPWS_INT_HTTP_PORT)))	// HTTP port
			|| (status = epx_end_element(ctx, DC_CONFIG_NS_URI, DC_CONFIG_ELT_HTTP_LISTENER)))
  		goto exit;

	/* Registry */
	if ((status = epx_start_element(ctx, DC_CONFIG_NS_URI, DC_CONFIG_ELT_REGISTRY)))
  		goto exit;

	// services classes
	sz = isz;
	do {
		if (status == DPWS_ERR_MORE_RESULTS) {
			if ((status = dpws_release_handles(hrefs, sz)))
				goto exit;
			sz = isz*=2;
			DC_FREE(DC_MEM_TRANSIENT, hrefs);
			if (!(hrefs = DC_MALLOC(DC_MEM_TRANSIENT, sz * sizeof(short)))) {
				status = SOAP_EOM;
				goto exit;
			}
		}
		status = dpws_get_service_class_handles(hrefs, &sz);
	}
	while (status == DPWS_ERR_MORE_RESULTS);

	if (status)	// Should not occur
		goto exit;

	for (i = 0; i < sz; i++) {
		if ((status = serialize_service_class(ctx, hrefs[i], DC_FALSE)))
			goto exit;
	}

	if ((status = dpws_release_handles(hrefs, sz)))
		goto exit;

	// devices
	status = DPWS_OK;
	sz = isz;
	do {
		if (status == DPWS_ERR_MORE_RESULTS) {
			if ((status = dpws_release_handles(hrefs, sz)))
				goto exit;
			sz = isz*=2;
			DC_FREE(DC_MEM_TRANSIENT, hrefs);
			if (!(hrefs = DC_MALLOC(DC_MEM_TRANSIENT, sz * sizeof(short)))) {
				status = SOAP_EOM;
				goto exit;
			}
		}
		status = dpws_get_device_handles(hrefs, &sz);
	}
	while (status == DPWS_ERR_MORE_RESULTS);

	if (status)	// Should not occur
		goto exit;

	for (i = 0; i < sz; i++) {
		if ((status = serialize_device(ctx, hrefs[i], DC_TRUE)))
			goto exit;
	}

	if ((status = epx_end_element(ctx, DC_CONFIG_NS_URI, DC_CONFIG_ELT_REGISTRY)))
		goto exit;

	/* Cache */
	ul = dpws_get_int_att(DC_CACHE_HANDLE, DPWS_INT_MAX_DEVICE_PROXIES);
	if (ul < 0xFFFF) {
		if ((status = epx_start_element(ctx, DC_CONFIG_NS_URI, DC_CONFIG_ELT_CACHE))
				|| (status = epx_add_long_attribute(ctx, NULL, DC_CONFIG_ATT_SIZE, (long)ul))
				|| (status = epx_end_element(ctx, DC_CONFIG_NS_URI, DC_CONFIG_ELT_CACHE)))
			goto exit;
	}

	/* Subscription Manager */
	ul = dpws_get_int_att(DC_SUBSC_MANAGER_HANDLE, DPWS_INT_MAX_SUBSC_NB);
	ul2 = dpws_get_int_att(DC_SUBSC_MANAGER_HANDLE, DPWS_INT_MAX_SUBSC_DURATION);

	if (ul < 0xFFFF || ul2 != 86400)
	{
		if ((status = epx_start_element(ctx, DC_CONFIG_NS_URI, DC_CONFIG_ELT_SUBSC_MANAGER)))
			goto exit;
		if (ul < 0xFFFF && (status = epx_add_long_attribute(ctx, NULL, DC_CONFIG_ATT_SIZE, (long)ul)))
			goto exit;
		if (ul2 != 86400) {
			char subsc_eventing[16];
			duration2xmlduration(subsc_eventing, ul2);
			if ((status = epx_add_attribute(ctx, NULL, DC_CONFIG_ATT_MAX_DURATION, subsc_eventing)))
				goto exit;
		}

		if ((status = epx_end_element(ctx, DC_CONFIG_NS_URI, DC_CONFIG_ELT_SUBSC_MANAGER)))
			goto exit;
	}
	/* Ends serialization */
	if ((status = epx_end_element(ctx, DC_CONFIG_NS_URI, DC_CONFIG_ELT_CONFIG)))
		goto exit;
    if (!as_fragment && (status = epx_end_document(ctx)))
    	goto exit;

exit:
	if (sz > 0) {
		status = dpws_release_handles(hrefs, sz);
		DC_FREE(DC_MEM_TRANSIENT, hrefs);
	}

    return (status == EPX_ERR_IMPLEMENTATION) ? epx_get_serializer_error(ctx) : status;
}

