/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2210 $
|                     $Date: 2009-03-19 11:37:32 +0100 (jeu, 19 mar 2009) $
\*============================================================================*/

/******************************************************************************\
 *                          XML configuration Parser                          *
\******************************************************************************/
#include "dc/dc_Constants.h"
#include "dcCOMN_Tools.h"
#include "dcXCONF_Parser.h"
#include "dcXCONF_Manager.h"
#include "dcXCONF_Definitions.h"
#include "dcXTOOL_SchemaParsing.h"

/*----------------------------------- Types ----------------------------------*/

struct conf_parsing_info {
	short href;
	int att;
};

struct multi_conf_parsing_info {
	struct conf_parsing_info cpi;
	DC_BOOL first;
};

struct lang_parsing_info {
	struct multi_conf_parsing_info * p_info;
	char * lang;
};

struct list_elt_parsing_info {
	struct multi_conf_parsing_info mepi;
	sgxp_token_parser_cbk list_cbk;
};

struct service_parsing_info {
	short href_device;
	DC_BOOL overwrite;	// replace if the service exists
	char * service_id;
	struct service_config *s_config;	// to fill
};

struct device_parsing_info {
	short href;
	DC_BOOL config;
	DC_BOOL enable;
};

/*------------------------- Static Functions prototypes ----------------------*/

static int parse_ptr_conf_elt_cbk(void * psr_ctx, char * ns_uri, char * lname, struct conf_parsing_info * p_info);
static int parse_int_conf_cbk(void * psr_ctx, char * ns_uri, char * lname, struct conf_parsing_info * p_info);
//static int parse_bool_conf_cbk(void * psr_ctx, char * ns_uri, char * lname, struct conf_parsing_info * p_info);
static int parse_lang_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct lang_parsing_info * l_info);
static int parse_localized_elt_value_cbk(void * psr_ctx, char * ns_uri, char * lname, struct lang_parsing_info * l_info);
static int parse_localized_elt_cbk(void * psr_ctx, char * ns_uri, char * lname, struct multi_conf_parsing_info * p_info);
static int sgxp_parse_list_conf_elt_cbk(void * psr_ctx, char * ns_uri, char * lname, struct list_elt_parsing_info * p_info);
static void parse_pqname(void * psr_ctx, struct prefixed_qname * pqn, char * qname);
static int clone_pqname(struct prefixed_qname * pqn);
static int parse_multi_conf_qname_cbk(void * psr_ctx, struct prefixed_qname * pqn, struct multi_conf_parsing_info * p_info);
static int parse_type_cbk(void * psr_ctx, char * token, struct multi_conf_parsing_info * p_info);
static int parse_types_cbk(void * psr_ctx, char * ns_uri, char * lname, short * href_device);
static int parse_scope_cbk(void * psr_ctx, char * token, struct multi_conf_parsing_info * p_info);
static int parse_scopes_cbk(void * psr_ctx, char * ns_uri, char * lname, short * href_device);
static int parse_this_model_cbk(void * psr_ctx, char * ns_uri, char * lname, short * href_device);
static int parse_this_device_cbk(void * psr_ctx, char * ns_uri, char * lname, short * href_device);
static int parse_service_id_cbk(void * psr_ctx, char * ns_uri, char * lname, struct service_parsing_info * s_info);
static int parse_service_class_id_cbk(void * psr_ctx, char * ns_uri, char * lname, struct service_parsing_info * s_info);
static int parse_sclass_cbk(void * psr_ctx, char * ns_uri, char * lname, struct service_parsing_info * s_info);
static int parse_serv_port_address_cbk(void * psr_ctx, char * ns_uri, char * lname, struct service_parsing_info * s_info);
static int parse_service_port_cbk(void * psr_ctx, char * ns_uri, char * lname, struct service_parsing_info * s_info);
static int parse_ref_epr_address_cbk(void * psr_ctx, char * ns_uri, char * lname, struct wsa_endpoint_ref * epr);
static int parse_epr_ref_cbk(void * psr_ctx, char * ns_uri, char * lname, struct ws_binding_ref * b_ref);
static int parse_mm_disc_hint_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct discovery_hints * d_hints);
static int parse_rl_disc_hint_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct discovery_hints * d_hints);
static int parse_bt_disc_hint_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct discovery_hints * d_hints);
static int parse_service_id_disc_hint_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct discovery_hints * d_hints);
static int parse_hint_type_cbk(void * psr_ctx, char * token, DA_TYPED(pqn) * types);
static int parse_hint_types_cbk(void * psr_ctx, char * ns_uri, char * lname, struct discovery_hint * d_hint);
static int parse_hint_scope_cbk(void * psr_ctx, char * token, DA_TYPED(str) * scopes);
static int parse_hint_scopes_cbk(void * psr_ctx, char * ns_uri, char * lname, struct discovery_hint * d_hint);
static int parse_ref_name_cbk(void * psr_ctx, char * ns_uri, char * lname, struct ws_binding_ref * b_ref);
static int parse_hint_cbk(void * psr_ctx, char * ns_uri, char * lname, struct discovery_hints * d_hints);
static int parse_discovery_hints(void * psr_ctx, char * ns_uri, char * lname, struct ws_binding_ref * b_ref);
static int parse_reference_cbk(void * psr_ctx, char * ns_uri, char * lname, struct service_parsing_info * s_info);
static int parse_prop_value_name_cbk(void * psr_ctx, char * ns_uri, char * lname, struct property_value * p_value);
static int parse_prop_value_cbk(void * psr_ctx, char * ns_uri, char * lname, struct property_value * p_value);
static int parse_property_value_cbk(void * psr_ctx, char * ns_uri, char * lname, struct service_parsing_info * s_info);
static int parse_service_cbk(void * psr_ctx, char * ns_uri, char * lname, struct service_parsing_info * s_info);
static int parse_device_cbk(void * psr_ctx, char * ns_uri, char * lname, struct device_parsing_info * d_info);
static int parse_serv_class_id_cbk(void * psr_ctx, char * ns_uri, char * lname, short * href_sclass);
static int parse_type_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct multi_conf_parsing_info * mcp_info);
static int parse_interface_cbk(void * psr_ctx, char * ns_uri, char * lname, struct multi_conf_parsing_info * mcp_info);
static int parse_ref_name_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct reference * ref);
static int parse_ref_type_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct prefixed_qname * pqn);
static int parse_ref_ms_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct reference * ref);
static int parse_ref_cbk(void * psr_ctx, char * ns_uri, char * lname, DA_TYPED(ref) * refs);
static int parse_prop_mult_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct property * prop);
static int parse_prop_cbk(void * psr_ctx, char * ns_uri, char * lname, struct property * prop);
static int parse_property_cbk(void * psr_ctx, char * ns_uri, char * lname, DA_TYPED(prop) * props);
static int parse_wsdl_tns_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct wsdl_info * wsdl);
static int parse_wsdl_location_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct wsdl_info * wsdl);
static int parse_wsdl_info_cbk(void * psr_ctx, char * ns_uri, char * lname, struct multi_conf_parsing_info * mcp_info);
static int parse_impl_cbk(void * psr_ctx, char * ns_uri, char * lname, struct service_class_config * sc_config);
static int parse_service_class_cbk(void * psr_ctx, char * ns_uri, char * lname, short * href_service_class);
static int parse_http_listener_cbk(void * psr_ctx, char * ns_uri, char * lname, void * user_data);
static int parse_registry_cbk(void * psr_ctx, char * ns_uri, char * lname, void * user_data);
static int parse_cache_cbk(void * psr_ctx, char * ns_uri, char * lname, void * user_data);
static int parse_subsc_manager_cbk(void * psr_ctx, char * ns_uri, char * lname, void * user_data);

/*----------------------------------------------------------------------------*\
 *                             Parsing utilities                              *
\*----------------------------------------------------------------------------*/
static int parse_ptr_conf_elt_cbk(void * psr_ctx, char * ns_uri, char * lname, struct conf_parsing_info * p_info)
{
	return dpws_set_ptr_att(p_info->href, p_info->att, epx_get_characters(psr_ctx));
}

static int parse_int_conf_cbk(void * psr_ctx, char * ns_uri, char * lname, struct conf_parsing_info * p_info)
{
	int ret = DPWS_OK;
	unsigned long ul;
	if (!epx_get_ulong(psr_ctx, &ul))
		ret = dpws_set_int_att(p_info->href, p_info->att, ul);
	return ret;
}

/*
static int parse_bool_conf_cbk(void * psr_ctx, char * ns_uri, char * lname, struct conf_parsing_info * p_info)
{
	int ret = DPWS_OK;
	epx_boolean_t b;
	if (!epx_get_boolean(psr_ctx, &b))
		ret = dpws_set_int_att(p_info->href, p_info->att, b);
	return ret;
}
*/

static int parse_lang_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct lang_parsing_info * l_info)
{
	l_info->lang = epx_get_characters(psr_ctx);
	return DPWS_OK;
}

static int parse_localized_elt_value_cbk(void * psr_ctx, char * ns_uri, char * lname, struct lang_parsing_info * l_info)
{
	int ret = DPWS_OK;
	struct localized_string ls = {NULL, NULL};
	ls.s = epx_get_characters(psr_ctx);	// COALESCING expected
	ls.lang = l_info->lang;
	if (l_info->p_info->first) {
		ret = dpws_set_ptr_att(l_info->p_info->cpi.href, l_info->p_info->cpi.att, &ls);
		l_info->p_info->first = DC_FALSE;
	}
	else
		ret = dpws_add_ptr_att(l_info->p_info->cpi.href, l_info->p_info->cpi.att, &ls);
	return ret;
}

static int parse_localized_elt_cbk(void * psr_ctx, char * ns_uri, char * lname, struct multi_conf_parsing_info * p_info)
{
	struct lang_parsing_info l_info;
	struct att_step atts[] = {{EPX_XML_NS_PREFIX, XML_ATT_LANG, XS_ATT_OPTIONAL, (value_cbk)parse_lang_att_cbk, NULL}};

	l_info.p_info = p_info;
	l_info.lang = NULL;
	atts[0].user_data = &l_info;

	return sgxp_parse_simple_content(psr_ctx, atts, SGXP_NB_ATT_STEPS(atts), (value_cbk)parse_localized_elt_value_cbk, &l_info);
}

static int sgxp_parse_list_conf_elt_cbk(void * psr_ctx, char * ns_uri, char * lname, struct list_elt_parsing_info * p_info)
{
	return sgxp_parse_list(psr_ctx, epx_get_characters(psr_ctx), p_info->list_cbk, &p_info->mepi);
}

static void parse_pqname(void * psr_ctx, struct prefixed_qname * pqn, char * qname)
{
	char * colon = strchr(qname, ':');
	if (colon) {
		*colon = '\0';	// ! string is modified (optimization)
		pqn->prefix = qname;
		pqn->qname.lname = colon + 1;
		pqn->qname.ns = epx_get_prefix_uri(psr_ctx, qname);
	}
	else
		pqn->qname.lname = qname;
}

static int clone_pqname(struct prefixed_qname * pqn)
{
	if ((!(pqn->prefix = DC_STRDUP(DC_MEM_CONF_MANAGER, pqn->prefix)) && pqn->prefix)
		|| (!(pqn->qname.ns = DC_STRDUP(DC_MEM_CONF_MANAGER, pqn->qname.ns)) && pqn->qname.ns)
		|| (!(pqn->qname.lname = DC_STRDUP(DC_MEM_CONF_MANAGER, pqn->qname.lname)) && pqn->qname.lname))
		return DPWS_ERR_EOM;
	return DPWS_OK;
}

/*----------------------------------------------------------------------------*\
 *                               Device parsing                               *
\*----------------------------------------------------------------------------*/

/* Device creation/update */

// Types
static int parse_multi_conf_qname_cbk(void * psr_ctx, struct prefixed_qname * pqn, struct multi_conf_parsing_info * p_info)
{
	int ret = DPWS_OK;
	if (p_info->first) {
		ret = dpws_set_ptr_att(p_info->cpi.href, DPWS_PTR_PREFIXED_TYPE, pqn);
		p_info->first = DC_FALSE;
	}
	else
		ret = dpws_add_ptr_att(p_info->cpi.href, DPWS_PTR_PREFIXED_TYPE, pqn);

	return ret;
}

static int parse_type_cbk(void * psr_ctx, char * token, struct multi_conf_parsing_info * p_info)
{
	struct prefixed_qname pqn = {{NULL, NULL}, NULL};
	parse_pqname(psr_ctx, &pqn, token);
	return parse_multi_conf_qname_cbk(psr_ctx, &pqn, p_info);
}

static int parse_types_cbk(void * psr_ctx, char * ns_uri, char * lname, short * href_device)
{
	struct list_elt_parsing_info pi = {{{-1, DPWS_PTR_PREFIXED_TYPE}, DC_TRUE}, (sgxp_token_parser_cbk) parse_type_cbk};

	pi.mepi.cpi.href = *href_device;
	return sgxp_parse_simple_content(psr_ctx, NULL, 0, (value_cbk)sgxp_parse_list_conf_elt_cbk, &pi);
}

int dpws_load_types(void * psr_ctx, short href_device)
{
	int ret = DPWS_OK;
	epx_set_parsing_options(psr_ctx, EPX_OPT_COALESCING|EPX_OPT_IGNORE_COMMENTS|EPX_OPT_PARSER_STREAMING);
	ret = sgxp_check_start_tag(psr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_TYPES, DC_TRUE);
	if (!ret)
		ret = parse_types_cbk(psr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_TYPES, &href_device);
	return ret;
}

// Scopes
static int parse_scope_cbk(void * psr_ctx, char * token, struct multi_conf_parsing_info * p_info)
{
	int ret = DPWS_OK;

	if (p_info->first) {
		ret = dpws_set_ptr_att(p_info->cpi.href, DPWS_STR_SCOPE, token);
		p_info->first = DC_FALSE;
	}
	else
		ret = dpws_add_ptr_att(p_info->cpi.href, DPWS_STR_SCOPE, token);

	return ret;
}

static int parse_scopes_cbk(void * psr_ctx, char * ns_uri, char * lname, short * href_device)
{
	struct list_elt_parsing_info pi = {{{-1, DPWS_STR_SCOPE}, DC_TRUE}, (sgxp_token_parser_cbk) parse_scope_cbk};

	pi.mepi.cpi.href = *href_device;

	return sgxp_parse_simple_content(psr_ctx, NULL, 0, (value_cbk)sgxp_parse_list_conf_elt_cbk, &pi);
}

int dpws_load_scopes(void * psr_ctx, short href_device)
{
	int ret = DPWS_OK;
	epx_set_parsing_options(psr_ctx, EPX_OPT_COALESCING|EPX_OPT_IGNORE_COMMENTS|EPX_OPT_PARSER_STREAMING);
	ret = sgxp_check_start_tag(psr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_SCOPES, DC_TRUE);
	if (!ret)
		ret = parse_scopes_cbk(psr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_SCOPES, &href_device);
	return ret;
}

// ThisModel
static int parse_this_model_cbk(void * psr_ctx, char * ns_uri, char * lname, short * href_device)
{
	struct multi_conf_parsing_info
		p_info1 = {{-1, DPWS_PTR_MANUFACTURER}, DC_TRUE},
		p_info3 = {{-1, DPWS_PTR_MODEL_NAME}, DC_TRUE};
	struct conf_parsing_info
		p_info2 = {-1, DPWS_STR_MANUFACTURER_URL},
		p_info4 = {-1, DPWS_STR_MODEL_NUMBER},
		p_info5 = {-1, DPWS_STR_MODEL_URL},
		p_info6 = {-1, DPWS_STR_PRESENTATION_URL};
	struct elt_step complex_type[] = {
		{WDP_URI, WDP_ELT_MANUFACTURER, 1, XS_OCC_UNBOUNDED, XS_MG_SEQUENCE, 1, (value_cbk)parse_localized_elt_cbk, NULL, DC_FALSE},
		{WDP_URI, WDP_ELT_MANUFACTURER_URL, 0, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_ptr_conf_elt_cbk, NULL, DC_TRUE},
		{WDP_URI, WDP_ELT_MODEL_NAME, 1, XS_OCC_UNBOUNDED, XS_MG_SEQUENCE, 1, (value_cbk)parse_localized_elt_cbk, NULL, DC_FALSE},
		{WDP_URI, WDP_ELT_MODEL_NUMBER, 0, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_ptr_conf_elt_cbk, NULL, DC_TRUE},
		{WDP_URI, WDP_ELT_MODEL_URL, 0, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_ptr_conf_elt_cbk, NULL, DC_TRUE},
		{WDP_URI, WDP_ELT_PRESENTATION_URL, 0, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_ptr_conf_elt_cbk, NULL, DC_TRUE},
		{WILDCARD_NS_URI, WILDCARD_LNAME, 0, XS_OCC_UNBOUNDED, XS_MG_SEQUENCE, 1, NULL, NULL, DC_TRUE}
	};

	p_info1.cpi.href = p_info3.cpi.href = *href_device;
	p_info2.href = p_info4.href = p_info5.href = p_info6.href = *href_device;

	complex_type[0].user_data = &p_info1;
	complex_type[1].user_data = &p_info2;
	complex_type[2].user_data = &p_info3;
	complex_type[3].user_data = &p_info4;
	complex_type[4].user_data = &p_info5;
	complex_type[5].user_data = &p_info6;

	return sgxp_parse_complex_content(psr_ctx, NULL, 0, complex_type, SGXP_NB_ELT_STEPS(complex_type));
}

int dpws_load_this_model(void * psr_ctx, short href_device)
{
	int ret = DPWS_OK;
	epx_set_parsing_options(psr_ctx, EPX_OPT_COALESCING|EPX_OPT_IGNORE_COMMENTS|EPX_OPT_PARSER_STREAMING);
	ret = sgxp_check_start_tag(psr_ctx, WDP_URI, WDP_ELT_THIS_MODEL, DC_TRUE);
	if (!ret)
		ret = parse_this_model_cbk(psr_ctx, WDP_URI, WDP_ELT_THIS_MODEL, &href_device);
	return ret;
}

// ThisDevice
static int parse_this_device_cbk(void * psr_ctx, char * ns_uri, char * lname, short * href_device)
{
	struct multi_conf_parsing_info
		p_info1 = {{-1, DPWS_PTR_FRIENDLY_NAME}, DC_TRUE};
	struct conf_parsing_info
		p_info2 = {-1, DPWS_STR_FIRMWARE_VERSION},
		p_info3 = {-1, DPWS_STR_SERIAL_NUMBER};

	struct elt_step complex_type[] = {
		{WDP_URI, WDP_ELT_FRIENDLY_NAME, 1, XS_OCC_UNBOUNDED, XS_MG_SEQUENCE, 1, (value_cbk)parse_localized_elt_cbk, NULL, DC_FALSE},
		{WDP_URI, WDP_ELT_FIRMWARE_VERSION, 0, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_ptr_conf_elt_cbk, NULL, DC_TRUE},
		{WDP_URI, WDP_ELT_SERIAL_NUMBER, 0, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_ptr_conf_elt_cbk, NULL, DC_TRUE},
		{WILDCARD_NS_URI, WILDCARD_LNAME, 0, XS_OCC_UNBOUNDED, XS_MG_SEQUENCE, 1, NULL, NULL, DC_TRUE}
	};


	p_info1.cpi.href = p_info2.href = p_info3.href = *href_device;

	complex_type[0].user_data = &p_info1;
	complex_type[1].user_data = &p_info2;
	complex_type[2].user_data = &p_info3;

	return sgxp_parse_complex_content(psr_ctx, NULL, 0, complex_type, SGXP_NB_ELT_STEPS(complex_type));
}

int dpws_load_this_device(void * psr_ctx, short href_device)
{
	int ret = DPWS_OK;
	epx_set_parsing_options(psr_ctx, EPX_OPT_COALESCING|EPX_OPT_IGNORE_COMMENTS|EPX_OPT_PARSER_STREAMING);
	ret = sgxp_check_start_tag(psr_ctx, WDP_URI, WDP_ELT_THIS_DEVICE, DC_TRUE);
	if (!ret)
		ret = parse_this_device_cbk(psr_ctx, WDP_URI, WDP_ELT_THIS_DEVICE, &href_device);
	return ret;
}

// Service
static int parse_service_id_cbk(void * psr_ctx, char * ns_uri, char * lname, struct service_parsing_info * s_info)
{
	char * service_id = epx_get_characters(psr_ctx);

	// look for the service and raise an error if no overwrite
	if ((s_info->s_config->href = dpws_get_service_handle(s_info->href_device, service_id)) >= 0)
	{
	 	if (s_info->overwrite) {
	 		dpws_delete_hosted_service(s_info->s_config->href);
	 		s_info->s_config->href = -1;
	 	}
	 	else
			return DPWS_ERR_RESOURCE_ALREADY_EXIST;
	}
	else if (s_info->overwrite)
		return DPWS_ERR_SERVICE_NOT_FOUND;

	if (!(s_info->service_id = DC_STRDUP(DC_MEM_CONF_MANAGER, service_id)) && service_id)
		return DPWS_ERR_EOM;

	return DPWS_OK;
}

static int parse_service_class_id_cbk(void * psr_ctx, char * ns_uri, char * lname, struct service_parsing_info * s_info)
{
	int ret = DPWS_OK;
	short href_class;
	// look for the service and raise an error if no overwrite
	if ((href_class = dpws_get_service_class_handle(epx_get_characters(psr_ctx))) < 0)
		return DPWS_ERR_SERVICE_CLASS_NOT_FOUND;

	if ((ret = dpws_release_handle(href_class)))
		goto exit;

	// create the service
	if ((s_info->s_config->href = dpws_create_hosted_service(s_info->href_device, href_class)) < 0)
		return s_info->s_config->href;

	ret = dpws_set_ptr_att(s_info->s_config->href, DPWS_STR_SERVICE_ID, s_info->service_id);
	DC_FREE(DC_MEM_CONF_MANAGER, s_info->service_id);

exit:
	return ret;
}

static int parse_sclass_cbk(void * psr_ctx, char * ns_uri, char * lname, struct service_parsing_info * s_info)
{
	struct att_step atts[] = {
		{NULL, DYNDEPL_ATT_SERVICE_CLASS_ID, XS_ATT_REQUIRED, (value_cbk)parse_service_class_id_cbk, NULL}
	};

	atts[0].user_data = s_info;

	return sgxp_parse_simple_content(psr_ctx, atts, SGXP_NB_ATT_STEPS(atts), NULL, NULL);
}

static int parse_serv_port_address_cbk(void * psr_ctx, char * ns_uri, char * lname, struct service_parsing_info * s_info)
{
	short href_serv_port = dpws_create_service_port();
	int ret = DPWS_OK;

	if (href_serv_port < 0)
		return href_serv_port;

	if ((ret = DPWS_SET_STR_ATT(href_serv_port, DPWS_STR_ADDRESS, epx_get_characters(psr_ctx))))
		goto exit;

	ret = dpws_bind_service(s_info->s_config->href, href_serv_port);

exit:
	ret = dpws_release_handle(href_serv_port);
	return ret;
}

static int parse_service_port_cbk(void * psr_ctx, char * ns_uri, char * lname, struct service_parsing_info * s_info)
{
	struct elt_step complex_type[] = {
		{WSA_URI, WSA_ELT_ADDRESS, 0, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_serv_port_address_cbk, NULL, DC_TRUE},
		{WILDCARD_NS_URI, WILDCARD_LNAME, 0, XS_OCC_UNBOUNDED, XS_MG_SEQUENCE, 1, NULL, NULL, DC_TRUE}
	};

	complex_type[0].user_data = s_info;

	return sgxp_parse_complex_content(psr_ctx, NULL, 0, complex_type, SGXP_NB_ELT_STEPS(complex_type));
}

static int parse_ref_epr_address_cbk(void * psr_ctx, char * ns_uri, char * lname, struct wsa_endpoint_ref * epr)
{
	char * s = epx_get_characters(psr_ctx);
	if (!(epr->address = DC_STRDUP(DC_MEM_CONF_MANAGER, s)) && s)
		return DPWS_ERR_EOM;
	return DPWS_OK;
}

static int parse_epr_ref_cbk(void * psr_ctx, char * ns_uri, char * lname, struct ws_binding_ref * b_ref)
{
	struct elt_step complex_type[] = {
		{WSA_URI, WSA_ELT_ADDRESS, 0, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_ref_epr_address_cbk, NULL, DC_TRUE},
		{WILDCARD_NS_URI, WILDCARD_LNAME, 0, XS_OCC_UNBOUNDED, XS_MG_SEQUENCE, 1, NULL, NULL, DC_TRUE}
	};
	complex_type[0].user_data = b_ref->epr = DC_MALLOC(DC_MEM_CONF_MANAGER, sizeof(struct wsa_endpoint_ref));
	if (!b_ref->epr)
		return DPWS_ERR_EOM;
	b_ref->wsd_hints = NULL;
	return sgxp_parse_complex_content(psr_ctx, NULL, 0, complex_type, SGXP_NB_ELT_STEPS(complex_type));
}

static int parse_mm_disc_hint_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct discovery_hints * d_hints)
{
	char * value = epx_get_characters(psr_ctx);
	if (!strcmp(value, DYNDEPL_ENUM_WSD_HINT_MM_ACTION_PICK_ONE))
		d_hints->on_multiple_matches = WSD_HINT_MM_ACTION_PICK_ONE;
	else if (!strcmp(value, DYNDEPL_ENUM_WSD_HINT_MM_ACTION_FAIL))
		d_hints->on_multiple_matches = WSD_HINT_MM_ACTION_FAIL;
	else
		return SGXP_ERR_INVALID_ENUM;

	return DPWS_OK;
}

static int parse_rl_disc_hint_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct discovery_hints * d_hints)
{
	char * value = epx_get_characters(psr_ctx);
	if (!strcmp(value, DYNDEPL_ENUM_WSD_HINT_RL_ACTION_FAIL))
		d_hints->on_reference_lost = WSD_HINT_RL_ACTION_FAIL;
	else if (!strcmp(value, DYNDEPL_ENUM_WSD_HINT_RL_ACTION_IGNORE))
		d_hints->on_reference_lost = WSD_HINT_RL_ACTION_IGNORE;
	else if (!strcmp(value, DYNDEPL_ENUM_WSD_HINT_RL_ACTION_RETRY))
		d_hints->on_reference_lost = WSD_HINT_RL_ACTION_RETRY;
	else
		return SGXP_ERR_INVALID_ENUM;

	return DPWS_OK;
}

static int parse_bt_disc_hint_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct discovery_hints * d_hints)
{
	char * value = epx_get_characters(psr_ctx);
	if (!strcmp(value, DYNDEPL_ENUM_WSD_HINT_BT_DEPLOYMENT))
		d_hints->binding_time = WSD_HINT_BT_DEPLOYMENT;
	else if (!strcmp(value, DYNDEPL_ENUM_WSD_HINT_BT_RUNTIME))
		d_hints->binding_time = WSD_HINT_BT_RUNTIME;
	else
		return SGXP_ERR_INVALID_ENUM;

	return DPWS_OK;
}

static int parse_service_id_disc_hint_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct discovery_hints * d_hints)
{
	char * s = epx_get_characters(psr_ctx);
	if (!(d_hints->service_id = DC_STRDUP(DC_MEM_CONF_MANAGER, s)) && s)
		return DPWS_ERR_EOM;
	return DPWS_OK;
}

static int parse_hint_type_cbk(void * psr_ctx, char * token, DA_TYPED(pqn) * types)
{
	struct prefixed_qname * pqn = DA_ADD(types);
	if (!pqn)
		return DPWS_ERR_EOM;
	parse_pqname(psr_ctx, pqn, token);
	return clone_pqname(pqn);
}

static int parse_hint_types_cbk(void * psr_ctx, char * ns_uri, char * lname, struct discovery_hint * d_hint)
{
	return sgxp_parse_list(psr_ctx, epx_get_characters(psr_ctx), (sgxp_token_parser_cbk)parse_hint_type_cbk, &d_hint->types);
}

static int parse_hint_scope_cbk(void * psr_ctx, char * token, DA_TYPED(str) * scopes)
{
	char ** scope = DA_ADD(scopes);
	if (!scope)
		return DPWS_ERR_EOM;
	*scope = DC_STRDUP(DC_MEM_CONF_MANAGER, token);
	return (*scope || !token) ? DPWS_OK : DPWS_ERR_EOM;
}

static int parse_hint_scopes_cbk(void * psr_ctx, char * ns_uri, char * lname, struct discovery_hint * d_hint)
{
	return sgxp_parse_list(psr_ctx, epx_get_characters(psr_ctx), (sgxp_token_parser_cbk)parse_hint_scope_cbk, &d_hint->scopes);
}

static int parse_ref_name_cbk(void * psr_ctx, char * ns_uri, char * lname, struct ws_binding_ref * b_ref)
{
	char * s = epx_get_characters(psr_ctx);
	b_ref->name = DC_STRDUP(DC_MEM_CONF_MANAGER, s);
	return (b_ref->name || !s) ? DPWS_OK : DPWS_ERR_EOM;
}

static int parse_hint_cbk(void * psr_ctx, char * ns_uri, char * lname, struct discovery_hints * d_hints)
{
	struct discovery_hint * d_hint = DA_ADD(&d_hints->hints);
	struct elt_step complex_type[] = {
		{DYNDEPL_NS_URI, DYNDEPL_ELT_TYPES, 0, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_hint_types_cbk, NULL, DC_TRUE},
		{DYNDEPL_NS_URI, DYNDEPL_ELT_SCOPES, 0, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_hint_scopes_cbk, NULL, DC_TRUE}
	};

	if (!d_hint)
		return DPWS_ERR_EOM;

	complex_type[0].user_data = complex_type[1].user_data = d_hint;

	DA_INIT(struct prefixed_qname, &d_hint->types, DC_MEM_CONF_MANAGER, p_default_allocator, 1);
	DA_INIT(char *, &d_hint->scopes, DC_MEM_CONF_MANAGER, p_default_allocator, 1);
	return sgxp_parse_complex_content(psr_ctx, NULL, 0, complex_type, SGXP_NB_ELT_STEPS(complex_type));
}

static int parse_discovery_hints(void * psr_ctx, char * ns_uri, char * lname, struct ws_binding_ref * b_ref)
{
	int i;
	struct att_step atts[] = {
		{NULL, DYNDEPL_ATT_ON_MULTIPLE_MATCHES, XS_ATT_OPTIONAL, (value_cbk)parse_mm_disc_hint_att_cbk, NULL},
		{NULL, DYNDEPL_ATT_BINDING_TIME, XS_ATT_OPTIONAL, (value_cbk)parse_bt_disc_hint_att_cbk, NULL},
		{NULL, DYNDEPL_ATT_ON_REFERENCE_LOST, XS_ATT_OPTIONAL, (value_cbk)parse_rl_disc_hint_att_cbk, NULL},
		{NULL, DYNDEPL_ATT_SERVICE_ID, XS_ATT_OPTIONAL, (value_cbk)parse_service_id_disc_hint_att_cbk, NULL}
	};
	struct elt_step complex_type[] = {
		{DYNDEPL_NS_URI, DYNDEPL_ELT_HINT, 1, XS_OCC_UNBOUNDED, XS_MG_SEQUENCE, 1, (value_cbk)parse_hint_cbk, NULL, DC_FALSE}
	};
	b_ref->epr = NULL;
	complex_type[0].user_data = b_ref->wsd_hints = (struct discovery_hints *)DC_MALLOC(DC_MEM_CONF_MANAGER, sizeof(struct discovery_hints));
	if (!b_ref->wsd_hints)
		return DPWS_ERR_EOM;
	for (i = 0; i < SGXP_NB_ATT_STEPS(atts); i++)
		atts[i].user_data = b_ref->wsd_hints;
	DA_INIT(struct discovery_hint, &b_ref->wsd_hints->hints, DC_MEM_CONF_MANAGER, p_default_allocator, 1);
	b_ref->wsd_hints->on_multiple_matches = WSD_HINT_MM_ACTION_PICK_ONE;
	b_ref->wsd_hints->binding_time = WSD_HINT_BT_DEPLOYMENT;
	b_ref->wsd_hints->on_reference_lost = WSD_HINT_RL_ACTION_RETRY;
	b_ref->wsd_hints->service_id = NULL;
	return sgxp_parse_complex_content(psr_ctx, atts, SGXP_NB_ATT_STEPS(atts), complex_type, SGXP_NB_ELT_STEPS(complex_type));
}

static int parse_reference_cbk(void * psr_ctx, char * ns_uri, char * lname, struct service_parsing_info * s_info)
{
	struct ws_binding_ref * b_ref = DA_ADD(&s_info->s_config->refs);
	struct att_step atts[] = {{NULL, DYNDEPL_ATT_NAME, XS_ATT_REQUIRED, (value_cbk)parse_ref_name_cbk, NULL}};
	struct elt_step complex_type[] = {
		{WSA_URI, WSA_ELT_ENDPOINT_REFERENCE, 0, 1, XS_MG_CHOICE, 1, (value_cbk)parse_epr_ref_cbk, NULL, DC_FALSE},
		{DYNDEPL_NS_URI, DYNDEPL_ELT_DISCOVERY_HINTS, 0, 1, XS_MG_CHOICE, 1, (value_cbk)parse_discovery_hints, NULL, DC_FALSE}
	};

	if (!b_ref)
		return DPWS_ERR_EOM;

	atts[0].user_data = complex_type[0].user_data = complex_type[1].user_data = b_ref;

	// ref will be created at the lower level
	b_ref->proxy = -1;	// Other fields should be set to NULL
	return sgxp_parse_complex_content(psr_ctx, atts, SGXP_NB_ATT_STEPS(atts), complex_type, SGXP_NB_ELT_STEPS(complex_type));
}

static int parse_prop_value_name_cbk(void * psr_ctx, char * ns_uri, char * lname, struct property_value * p_value)
{
	char * s = epx_get_characters(psr_ctx);
	p_value->name = DC_STRDUP(DC_MEM_CONF_MANAGER, s);
	return (p_value->name || !s) ? DPWS_OK : DPWS_ERR_EOM;
}

static int parse_prop_value_cbk(void * psr_ctx, char * ns_uri, char * lname, struct property_value * p_value)
{
	char * s = epx_get_characters(psr_ctx);
	p_value->value = DC_STRDUP(DC_MEM_CONF_MANAGER, s);
	return (p_value->value || !s) ? DPWS_OK : DPWS_ERR_EOM;
}

static int parse_property_value_cbk(void * psr_ctx, char * ns_uri, char * lname, struct service_parsing_info * s_info)
{
	struct property_value * p_value = DA_ADD(&s_info->s_config->property_values);
	struct att_step atts[] = {{NULL, DYNDEPL_ATT_NAME, XS_ATT_REQUIRED, (value_cbk)parse_prop_value_name_cbk, NULL}};

	if (!p_value)
		return DPWS_ERR_EOM;

	atts[0].user_data = p_value;

	return sgxp_parse_simple_content(psr_ctx, atts, SGXP_NB_ATT_STEPS(atts), (value_cbk)parse_prop_value_cbk, p_value);
}

static int parse_service_cbk(void * psr_ctx, char * ns_uri, char * lname, struct service_parsing_info * s_info)
{
	int ret = DPWS_OK;
	struct att_step atts[] = {{NULL, DYNDEPL_ATT_SERVICE_ID, XS_ATT_OPTIONAL, (value_cbk)parse_service_id_cbk, NULL}};
	struct elt_step complex_type[] = {
		{DYNDEPL_NS_URI, DYNDEPL_ELT_SERVICE_CLASS, 1, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_sclass_cbk, NULL, DC_FALSE},
		{DYNDEPL_NS_URI, DYNDEPL_ELT_SERVICE_PORT, 0, XS_OCC_UNBOUNDED, XS_MG_SEQUENCE, 1, (value_cbk)parse_service_port_cbk, NULL, DC_FALSE},
		{DYNDEPL_NS_URI, DYNDEPL_ELT_REFERENCE, 0, XS_OCC_UNBOUNDED, XS_MG_SEQUENCE, 1, (value_cbk)parse_reference_cbk, NULL, DC_FALSE},
		{DYNDEPL_NS_URI, DYNDEPL_ELT_PROPERTY_VALUE, 0, XS_OCC_UNBOUNDED, XS_MG_SEQUENCE, 1, (value_cbk)parse_property_value_cbk, NULL, DC_FALSE}
	};

	atts[0].user_data = complex_type[0].user_data = complex_type[1].user_data = complex_type[2].user_data = complex_type[3].user_data = s_info;

	if (!(s_info->s_config = xconf_init_service()))
		return DPWS_ERR_EOM;

	ret = sgxp_parse_complex_content(psr_ctx, atts, SGXP_NB_ATT_STEPS(atts), complex_type, SGXP_NB_ELT_STEPS(complex_type));

	// Store service_config in the serv_manager
	if (ret)
		xconf_free_service(s_info->s_config);
	else {
		short href;
		int len = 1;
		if ((ret = dpws_get_service_port_handles(s_info->s_config->href, &href, &len)))
			goto exit;
		if (len == 0) {
			short href_serv_port = dpws_create_service_port();

			if (href_serv_port >= 0) {
				ret = dpws_bind_service(s_info->s_config->href, href_serv_port);
				dpws_release_handle(href_serv_port);
			}
		}
		if ((ret = xconf_new_service(s_info->s_config)))
			xconf_free_service(s_info->s_config);
	}
exit:
	return ret;
}

int dpws_load_service(void * psr_ctx, short href_device, short * href_service, DC_BOOL overwrite)
{
	int ret = DPWS_OK;
	struct service_parsing_info s_info = {-1, DC_FALSE, NULL};

	s_info.href_device = href_device;
	s_info.overwrite = overwrite;

	epx_set_parsing_options(psr_ctx, EPX_OPT_COALESCING|EPX_OPT_IGNORE_COMMENTS|EPX_OPT_PARSER_STREAMING);
	ret = sgxp_check_start_tag(psr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_SERVICE, DC_TRUE);
	if (!ret)
		ret = parse_service_cbk(psr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_SERVICE, &s_info);
	if (!ret)
		*href_service = s_info.s_config->href;
	return ret;
}

// whole device
static int parse_device_cbk(void * psr_ctx, char * ns_uri, char * lname, struct device_parsing_info * d_info)
{
	int ret = DPWS_OK;
	struct conf_parsing_info p_info1 = {-1, DPWS_STR_DEVICE_ID}, p_info2 = {-1, DPWS_INT_METADATA_VERSION};
	struct service_parsing_info s_info = {-1, DC_FALSE, NULL};
	struct elt_step complex_type[] = {
		{DYNDEPL_NS_URI, DYNDEPL_ELT_ADDRESS, 0, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_ptr_conf_elt_cbk, NULL, DC_TRUE},
		{DYNDEPL_NS_URI, DYNDEPL_ELT_TYPES, 0, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_types_cbk, NULL, DC_FALSE},
		{DYNDEPL_NS_URI, DYNDEPL_ELT_SCOPES, 0, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_scopes_cbk, NULL, DC_FALSE},
		{WDP_URI, WDP_ELT_THIS_MODEL, 0, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_this_model_cbk, NULL, DC_FALSE},
		{WDP_URI, WDP_ELT_THIS_DEVICE, 0, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_this_device_cbk, NULL, DC_FALSE},
		{DYNDEPL_NS_URI, DYNDEPL_ELT_SERVICE, 0, XS_OCC_UNBOUNDED, XS_MG_SEQUENCE, 1, (value_cbk)parse_service_cbk, NULL, DC_FALSE}
	};
	struct att_step atts[] = {{NULL, DC_CONFIG_ATT_METADATA_VERSION, XS_ATT_REQUIRED, (value_cbk)parse_int_conf_cbk, NULL}};

	complex_type[0].user_data = &p_info1;
	complex_type[1].user_data = complex_type[2].user_data = complex_type[3].user_data = complex_type[4].user_data = &d_info->href;
	complex_type[5].user_data = &s_info;
	atts[0].user_data = &p_info2;

	d_info->href = p_info1.href= p_info2.href = s_info.href_device = dpws_create_custom_device(RANDOM_DEVICE_ID, -1);
	if (d_info->href >= 0) {
		ret = sgxp_parse_complex_content(psr_ctx, atts, d_info->config ? SGXP_NB_ATT_STEPS(atts) : 0, complex_type, SGXP_NB_ELT_STEPS(complex_type));
		if (ret)
			ret = dpws_release_handle(d_info->href);
		else if (d_info->enable)
		{
			if (!(ret = dpws_enable_device(d_info->href)))
				ret = dpws_release_handle(d_info->href);
		}
	}
	return ret;
}

int dpws_load_device(void * psr_ctx, short * href_device, DC_BOOL enable)
{
	struct device_parsing_info d_info = {-1, DC_FALSE, DC_FALSE};
	int ret = DPWS_OK;

	d_info.enable = enable;

	epx_set_parsing_options(psr_ctx, EPX_OPT_COALESCING|EPX_OPT_IGNORE_COMMENTS|EPX_OPT_PARSER_STREAMING);
	ret = sgxp_check_start_tag(psr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_DEVICE, DC_TRUE);
	if (!ret)
		ret = parse_device_cbk(psr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_DEVICE, &d_info);
	*href_device = d_info.href;
	return ret;
}

/*----------------------------------------------------------------------------*\
 *                            Service class parsing                           *
\*----------------------------------------------------------------------------*/

static int parse_serv_class_id_cbk(void * psr_ctx, char * ns_uri, char * lname, short * href_sclass)
{
	return dpws_set_ptr_att(*href_sclass, DPWS_STR_ID, epx_get_characters(psr_ctx));
}

static int parse_type_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct multi_conf_parsing_info * mcp_info)
{
	struct prefixed_qname pqn;
	epx_get_qname(psr_ctx, (epx_qname_t *)&pqn.qname);
	pqn.prefix = epx_get_uri_prefix(psr_ctx, pqn.qname.ns);
	return parse_multi_conf_qname_cbk(psr_ctx, &pqn, mcp_info);
}

static int parse_interface_cbk(void * psr_ctx, char * ns_uri, char * lname, struct multi_conf_parsing_info * mcp_info)
{
	struct att_step atts[] = {{NULL, DYNDEPL_ATT_TYPE, XS_ATT_REQUIRED, (value_cbk)parse_type_att_cbk, NULL}};

	atts[0].user_data = mcp_info;

	return sgxp_parse_simple_content(psr_ctx, atts, SGXP_NB_ATT_STEPS(atts), NULL, NULL);
}

static int parse_ref_name_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct reference * ref)
{
	char * s = epx_get_characters(psr_ctx);
	ref->name = DC_STRDUP(DC_MEM_CONF_MANAGER, s);
	return (ref->name || !s) ? DPWS_OK : DPWS_ERR_EOM;
}

static int parse_ref_type_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct prefixed_qname * pqn)
{
	epx_get_qname(psr_ctx, (epx_qname_t *)&pqn->qname);
	pqn->prefix = epx_get_uri_prefix(psr_ctx, pqn->qname.ns);
	return clone_pqname(pqn);
}

static int parse_ref_ms_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct reference * ref)
{
	return epx_get_boolean(psr_ctx, &ref->must_supply);
}

static int parse_ref_cbk(void * psr_ctx, char * ns_uri, char * lname, DA_TYPED(ref) * refs)
{
	struct att_step atts[] = {
		{NULL, DYNDEPL_ATT_NAME, XS_ATT_REQUIRED, (value_cbk)parse_ref_name_att_cbk, NULL},
		{NULL, DYNDEPL_ATT_TYPE, XS_ATT_REQUIRED, (value_cbk)parse_ref_type_att_cbk, NULL},
		{NULL, DYNDEPL_ATT_MUST_SUPPLY, XS_ATT_OPTIONAL, (value_cbk)parse_ref_ms_att_cbk, NULL}
	};
	struct reference * ref = DA_ADD(refs);

	if (!ref)
		return DPWS_ERR_EOM;

	atts[0].user_data = ref;
	atts[1].user_data = &ref->type;
	atts[2].user_data = ref;

	// Init default
	ref->must_supply = DC_TRUE;
	return sgxp_parse_simple_content(psr_ctx, atts, SGXP_NB_ATT_STEPS(atts), NULL, NULL);
}

static int parse_prop_mult_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct property * prop)
{
	return epx_get_boolean(psr_ctx, &prop->multiple);
}

static int parse_prop_cbk(void * psr_ctx, char * ns_uri, char * lname, struct property * prop)
{
	char * s = epx_get_characters(psr_ctx);
	prop->default_value = DC_STRDUP(DC_MEM_CONF_MANAGER, s);
	return (prop->default_value || !s) ? DPWS_OK : DPWS_ERR_EOM;
}

static int parse_property_cbk(void * psr_ctx, char * ns_uri, char * lname, DA_TYPED(prop) * props)
{
	struct att_step atts[] = {
		{NULL, DYNDEPL_ATT_NAME, XS_ATT_REQUIRED, (value_cbk)parse_ref_name_att_cbk, NULL},
		{NULL, DYNDEPL_ATT_TYPE, XS_ATT_REQUIRED, (value_cbk)parse_ref_type_att_cbk, NULL},
		{NULL, DYNDEPL_ATT_MUST_SUPPLY, XS_ATT_OPTIONAL, (value_cbk)parse_ref_ms_att_cbk, NULL},
		{NULL, DYNDEPL_ATT_MULTIPLE, XS_ATT_OPTIONAL, (value_cbk)parse_prop_mult_att_cbk, NULL}
	};
	struct property * prop = DA_ADD(props);

	if (!prop)
		return DPWS_ERR_EOM;

	atts[0].user_data = &prop->p;
	atts[1].user_data = &prop->p.type;
	atts[2].user_data = &prop->p;
	atts[3].user_data = prop;

	// Init default
	prop->p.must_supply = DC_TRUE;
	prop->multiple = DC_FALSE;
	return sgxp_parse_simple_content(psr_ctx, atts, SGXP_NB_ATT_STEPS(atts), (value_cbk)parse_prop_cbk, prop);
}

static int parse_wsdl_tns_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct wsdl_info * wsdl)
{
	wsdl->target_ns = epx_get_characters(psr_ctx);	// att values are valid during start element event
	return DPWS_OK;
}

static int parse_wsdl_location_att_cbk(void * psr_ctx, char * ns_uri, char * lname, struct wsdl_info * wsdl)
{
	wsdl->location = epx_get_characters(psr_ctx);	// att values are valid during start element event
	return DPWS_OK;
}

static int parse_wsdl_info_cbk(void * psr_ctx, char * ns_uri, char * lname, struct multi_conf_parsing_info * mcp_info)
{
	int ret;
	struct wsdl_info wsdl = {NULL, NULL};
	struct att_step atts[] = {
		{NULL, DYNDEPL_ATT_TARGET_NS, XS_ATT_REQUIRED, (value_cbk)parse_wsdl_tns_att_cbk, NULL},
		{NULL, DYNDEPL_ATT_LOCATION, XS_ATT_REQUIRED, (value_cbk)parse_wsdl_location_att_cbk, NULL}
	};

	atts[0].user_data = atts[1].user_data = &wsdl;

	ret = sgxp_parse_simple_content(psr_ctx, atts, SGXP_NB_ATT_STEPS(atts), NULL, NULL);
	if (mcp_info->first) {
		ret = dpws_set_ptr_att(mcp_info->cpi.href, mcp_info->cpi.att, &wsdl);
		mcp_info->first = DC_FALSE;
	}
	else
		ret = dpws_add_ptr_att(mcp_info->cpi.href, mcp_info->cpi.att, &wsdl);

	return ret;
}

static int parse_impl_cbk(void * psr_ctx, char * ns_uri, char * lname, struct service_class_config * sc_config)
{
	struct qname qn;

	qn.ns = ns_uri;
	qn.lname = lname;

	return xconf_new_service_class(sc_config, &qn, psr_ctx);
}

/* Service class creation/update */
static int parse_service_class_cbk(void * psr_ctx, char * ns_uri, char * lname, short * href_service_class)
{
	int ret = DPWS_OK;
	struct multi_conf_parsing_info	mcp_info1 = {{-1, DPWS_PTR_PREFIXED_TYPE}, DC_TRUE},
									mcp_info2 = {{-1, DPWS_PTR_WSDL}, DC_TRUE};
	struct service_class_config * sc_config = xconf_init_service_class();
	struct att_step atts[] = {
		{NULL, DYNDEPL_ATT_SERVICE_CLASS_ID, XS_ATT_OPTIONAL, (value_cbk)parse_serv_class_id_cbk, NULL}
	};
	struct elt_step complex_type[] = {
		{DYNDEPL_NS_URI, DYNDEPL_ELT_INTERFACE, 0, XS_OCC_UNBOUNDED, XS_MG_SEQUENCE, 1, (value_cbk)parse_interface_cbk, NULL, DC_FALSE},
		{DYNDEPL_NS_URI, DYNDEPL_ELT_REFERENCE, 0, XS_OCC_UNBOUNDED, XS_MG_SEQUENCE, 1, (value_cbk)parse_ref_cbk, NULL, DC_FALSE},
		{DYNDEPL_NS_URI, DYNDEPL_ELT_PROPERTY, 0, XS_OCC_UNBOUNDED, XS_MG_SEQUENCE, 1, (value_cbk)parse_property_cbk, NULL, DC_FALSE},
		{DYNDEPL_NS_URI, DYNDEPL_ELT_WSDL_INFO, 0, XS_OCC_UNBOUNDED, XS_MG_SEQUENCE, 1, (value_cbk)parse_wsdl_info_cbk, NULL, DC_FALSE},
		{WILDCARD_NS_URI, WILDCARD_LNAME, 1, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_impl_cbk, NULL, DC_FALSE}
	};

	if (!sc_config)
		return DPWS_ERR_EOM;

	atts[0].user_data = href_service_class;
	complex_type[0].user_data = &mcp_info1;
	complex_type[1].user_data = &sc_config->refs;
	complex_type[2].user_data = &sc_config->properties;
	complex_type[3].user_data = &mcp_info2;
	complex_type[4].user_data = sc_config;


	*href_service_class = mcp_info1.cpi.href = mcp_info2.cpi.href = sc_config->href = dpws_create_service_class();
	if (sc_config->href >= 0) {
		ret = sgxp_parse_complex_content(psr_ctx, atts, SGXP_NB_ATT_STEPS(atts), complex_type, SGXP_NB_ELT_STEPS(complex_type));
		if (ret)
			xconf_delete_service_class(sc_config->href);
	}
	else
		xconf_free_service_class(sc_config);

    return ret;
}

int dpws_load_service_class(void * psr_ctx, short * href_service_class)
{
	int ret = DPWS_OK;
	epx_set_parsing_options(psr_ctx, EPX_OPT_COALESCING|EPX_OPT_IGNORE_COMMENTS|EPX_OPT_PARSER_STREAMING);
	ret = sgxp_check_start_tag(psr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_SERVICE_CLASS, DC_TRUE);
	if (!ret)
		ret = parse_service_class_cbk(psr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_SERVICE_CLASS, href_service_class);
	return ret;
}

/*----------------------------------------------------------------------------*\
 *                             DC Registry parsing                            *
\*----------------------------------------------------------------------------*/

static int parse_http_listener_cbk(void * psr_ctx, char * ns_uri, char * lname, void * user_data)
{
	struct conf_parsing_info p_info = {DC_REGISTRY_HANDLE, DPWS_INT_HTTP_PORT};

	struct att_step atts[] = {
		{NULL, DC_CONFIG_ATT_PORT, XS_ATT_OPTIONAL, (value_cbk)parse_int_conf_cbk, NULL}
	};

	atts[0].user_data = &p_info;

	return sgxp_parse_complex_content(psr_ctx, atts, SGXP_NB_ATT_STEPS(atts), NULL, 0);
}

static int parse_registry_cbk(void * psr_ctx, char * ns_uri, char * lname, void * user_data)
{
	struct device_parsing_info d_info = {-1, DC_TRUE, DC_TRUE};
	short href_class;
 	struct elt_step complex_type[] = {
		{DYNDEPL_NS_URI, DYNDEPL_ELT_SERVICE_CLASS, 0, XS_OCC_UNBOUNDED, XS_MG_SEQUENCE, 1, (value_cbk)parse_service_class_cbk, NULL, DC_FALSE},
		{DC_CONFIG_NS_URI, DYNDEPL_ELT_DEVICE, 0, XS_OCC_UNBOUNDED, XS_MG_SEQUENCE, 1, (value_cbk)parse_device_cbk, NULL, DC_FALSE}
	};

	complex_type[0].user_data = &href_class;
	complex_type[1].user_data = &d_info;

	return sgxp_parse_complex_content(psr_ctx, NULL, 0, complex_type, SGXP_NB_ELT_STEPS(complex_type));
}

static int parse_cache_cbk(void * psr_ctx, char * ns_uri, char * lname, void * user_data)
{
	struct conf_parsing_info p_info1 = {DC_CACHE_HANDLE, DPWS_INT_MAX_DEVICE_PROXIES};

	struct att_step atts[] = {
		{NULL, DC_CONFIG_ATT_SIZE, XS_ATT_OPTIONAL, (value_cbk)parse_int_conf_cbk, NULL}
	};

	atts[0].user_data = &p_info1;

	return sgxp_parse_complex_content(psr_ctx, atts, SGXP_NB_ATT_STEPS(atts), NULL, 0);
}

static int parse_subsc_manager_cbk(void * psr_ctx, char * ns_uri, char * lname, void * user_data)
{
	struct conf_parsing_info	p_info1 = {DC_SUBSC_MANAGER_HANDLE, DPWS_STR_MAX_SUBSC_DURATION},
								p_info2 = {DC_SUBSC_MANAGER_HANDLE, DPWS_INT_MAX_SUBSC_NB};

	struct att_step atts[] = {
			{NULL, DC_CONFIG_ATT_MAX_DURATION, XS_ATT_OPTIONAL, (value_cbk)parse_ptr_conf_elt_cbk, NULL},
			{NULL, DC_CONFIG_ATT_SIZE, XS_ATT_OPTIONAL, (value_cbk)parse_int_conf_cbk, NULL}
	};

	atts[0].user_data = &p_info1;
	atts[1].user_data = &p_info2;

	return sgxp_parse_complex_content(psr_ctx, atts, SGXP_NB_ATT_STEPS(atts), NULL, 0);
}

int dpws_load_config(void * ctx)
{
	int ret = DPWS_OK;
    epx_event evt;
	struct conf_parsing_info
						p_info1 = {DC_REGISTRY_HANDLE, DPWS_INT_BOOT_SEQ},
						p_info2 = {DC_REGISTRY_HANDLE, DPWS_STR_PREFERRED_LANG},
						p_info3 = {DC_REGISTRY_HANDLE, DPWS_INT_HTTP_PORT};
 	struct elt_step complex_type[] = {
		{DC_CONFIG_NS_URI, DC_CONFIG_ELT_BOOT_SEQ, 1, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_int_conf_cbk, NULL, DC_TRUE},
		{DC_CONFIG_NS_URI, DC_CONFIG_ELT_PREFERRED_LANG, 0, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_ptr_conf_elt_cbk, NULL, DC_TRUE},
		{DC_CONFIG_NS_URI, DC_CONFIG_ELT_HTTP_LISTENER, 0, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_http_listener_cbk, NULL, DC_FALSE},
		{DC_CONFIG_NS_URI, DC_CONFIG_ELT_REGISTRY, 1, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_registry_cbk, NULL, DC_FALSE},
		{DC_CONFIG_NS_URI, DC_CONFIG_ELT_CACHE, 0, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_cache_cbk, NULL, DC_FALSE},
		{DC_CONFIG_NS_URI, DC_CONFIG_ELT_SUBSC_MANAGER, 0, 1, XS_MG_SEQUENCE, 1, (value_cbk)parse_subsc_manager_cbk, NULL, DC_FALSE}
	};

	DC_CHECK_PARAM(ctx);

	complex_type[0].user_data = &p_info1;
	complex_type[1].user_data = &p_info2;
	complex_type[2].user_data = &p_info3;

	if ((ret = epx_start_parsing(ctx, NULL, EPX_OPT_COALESCING|EPX_OPT_IGNORE_COMMENTS|EPX_OPT_PARSER_STREAMING)))
		goto exit;

	ret = sgxp_check_start_tag(ctx, DC_CONFIG_NS_URI, DC_CONFIG_ELT_CONFIG, DC_TRUE);
	if (!ret)
		ret = sgxp_parse_complex_content(ctx, NULL, 0, complex_type, SGXP_NB_ELT_STEPS(complex_type));
	else if (epx_get_event(ctx) != EPX_EVT_START_DOCUMENT)
	{
		ret = SGXP_ERR_EVENT_MISMATCH;
		goto exit;
	}
	else if	((ret = sgxp_parse_complex_element(ctx, DC_CONFIG_NS_URI, DC_CONFIG_ELT_CONFIG, DC_TRUE, NULL, 0, complex_type, SGXP_NB_ELT_STEPS(complex_type))))
		goto exit;

	// consume events until the end
	do {
		evt = epx_next(ctx);
	} while (evt != EPX_EVT_END_DOCUMENT && evt != EPX_EVT_IDLE && evt != EPX_EVT_ERROR && evt != EPX_EVT_END_FRAGMENT);

exit:
    return ret;
}
