/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 827 $
|                     $Date: 2008-03-13 16:34:05 +0100 (jeu., 13 mars 2008) $
\*============================================================================*/

/******************************************************************************\
 *                         WS-Management generic server                       *
\******************************************************************************/
#include "dc/dc_WsMan.h"	// 1st to define WITH_WSMAN
#include "dc/dc_WsManConstants.h"
#include "dc/dc_Constants.h"
#include "dcDPWS_Memory.h"
#include "dcWSMAN_Server.h"
#include "dcCOMN_DynArray.h"
#include "dcCOMN_Tools.h"
#include "dcGSOAP_Runtime.h"


/*----------------------------------- Types ----------------------------------*/

struct wsman_callback_data {
	serve_cbk func;
	void* user_data;
};

#define XPATH_ROOT_TOKEN "/"
#define IMPLEMENTATION_ERROR_MSG "Implementation error: %d"

/*----------------------------------- Data -----------------------------------*/

static struct qname wst_resource_port_type = { WST_URI, WST_RESOURCE_PORT_TYPE };
static struct qname wst_resource_factory_port_type = { WST_URI, WST_RESOURCE_FACTORY_PORT_TYPE };

/*------------------------- Static Functions prototypes ----------------------*/

static struct wsman_resource_handler * get_resource_handler(char * resource_uri, struct wsman_handlers_list * handlers_list);
static int wsman_process_request(struct dpws * dpws, void * parser_context, serialize_cbk *response_cbk, void **p_user_data);
static int parse_token(struct dpws * dpws, char * token, DA_TYPED(qn) * tokens, int nb_pfx_def, struct prefix_def * ns_ctx);
static int wsman_parse_xpath_level_1(struct dpws * dpws, struct xpath_expression * xpath_expr, DA_TYPED(qn) * tokens);

/*----------------------------------------------------------------------------*\
 *                           Server implementation                            *
\*----------------------------------------------------------------------------*/

static struct wsman_resource_handler * get_resource_handler(char * resource_uri, struct wsman_handlers_list * handlers_list)
{
	struct wsman_resource_handler * result = NULL;
	int i = 0;
	if (handlers_list) {
		result = handlers_list->default_handler;
		for (i = 0; i < handlers_list->handlers_count; i++) {
			struct wsman_resource_handler * handler = handlers_list->handlers[i];
			if (handler->resource_uri && !strcmp(resource_uri, handler->resource_uri)) {
				result = handler;
				break;
			}
		}
	}
	return result;
}

static int wsman_process_request(struct dpws * dpws, void * parser_context, serialize_cbk *response_cbk, void **p_user_data)
{
	int ret = DPWS_OK;
	char* resource_uri = wsman_get_resource_uri(dpws);
    char* action = dpws->action ? dpws->action : "";

	if (resource_uri) {
		struct wsman_resource_handler * handler = get_resource_handler(resource_uri,
			(struct wsman_handlers_list *)dpws_get_service_class_user_data(dpws));
		if (handler) {
			if (!strcmp(action, WSMAN_ACTION_GET))
				ret = handler->get_func(dpws, parser_context, response_cbk, p_user_data);
			else if (!strcmp(action, WSMAN_ACTION_PUT))
				ret = handler->put_func(dpws, parser_context, response_cbk, p_user_data);
			else if (!strcmp(action, WSMAN_ACTION_CREATE))
				ret = handler->create_func(dpws, parser_context, response_cbk, p_user_data);
			else if (!strcmp(action, WSMAN_ACTION_DELETE))
				ret = handler->delete_func(dpws, parser_context, response_cbk, p_user_data);
		}
		else
			ret = WSMAN_ERR_INVALID_RESOURCE;

		if (ret && !*soap_faultstring(dpws_dpws2soap(dpws))) // check if the function has already be called in user code
			ret = wsman_set_fault(dpws, ret, 0);
	}
	else {
		ret = dpws_dpws2soap(dpws)->error = SOAP_NO_METHOD;
	}
	return ret;
}

int wsman_serve_request(struct dpws *dpws)
{
    char* action = dpws->action ? dpws->action : "";
    if (!strcmp(action, WSMAN_ACTION_GET))
        return dpws_process_request(dpws, WSMAN_ACTION_GET_RESPONSE, WSMAN_ACTION_FAULT, NULL, 0, wsman_process_request);
    else if (!strcmp(action, WSMAN_ACTION_PUT))
        return dpws_process_request(dpws, WSMAN_ACTION_PUT_RESPONSE, WSMAN_ACTION_FAULT, NULL, 0, wsman_process_request);
    else if (!strcmp(action, WSMAN_ACTION_CREATE))
        return dpws_process_request(dpws, WSMAN_ACTION_CREATE_RESPONSE, WSMAN_ACTION_FAULT, NULL, 0, wsman_process_request);
    else if (!strcmp(action, WSMAN_ACTION_DELETE))
        return dpws_process_request(dpws, WSMAN_ACTION_DELETE_RESPONSE, WSMAN_ACTION_FAULT, NULL, 0, wsman_process_request);
    return dpws_dpws2soap(dpws)->error = SOAP_NO_METHOD;
}

/*----------------------------------------------------------------------------*\
 *                        Service Class and Service Configuration             *
\*----------------------------------------------------------------------------*/

void wsman_set_resource_handlers(short hman_class, struct wsman_resource_handler ** handlers, int handlers_count, struct wsman_resource_handler * default_handler)
{
	DC_CHECK_PARAM_RETURN(hman_class >= 0 && handlers && handlers_count > 0,);
	if (hman_class >= 0) {
		struct wsman_handlers_list * handlers_list = (struct wsman_handlers_list *)SOAP_MALLOC(NULL, sizeof(struct wsman_handlers_list));
		int handlers_list_size = sizeof(struct wsman_resource_handler *) * handlers_count;

		handlers_list->handlers = (struct wsman_resource_handler **)SOAP_MALLOC(NULL, handlers_list_size);
		memcpy(handlers_list->handlers, handlers, handlers_list_size);
		handlers_list->handlers_count = handlers_count;
		handlers_list->default_handler = default_handler;
		DPWS_SET_PTR_ATT(hman_class, DPWS_PTR_USER_DATA, handlers_list);
	}
}

short wsman_create_management_class(struct wsman_resource_handler ** handlers, int handlers_count, struct wsman_resource_handler * default_handler, char * class_id)
{
	short hman_class;

	DC_CHECK_PARAM(handlers && handlers_count > 0);

	hman_class = dpws_create_service_class();
	if (hman_class >= 0) {
		if (!class_id)
			class_id = WSMAN_SERVICE_ID;
		DPWS_ADD_PTR_ATT(hman_class, DPWS_PTR_TYPE, &wst_resource_port_type);
		DPWS_ADD_PTR_ATT(hman_class, DPWS_PTR_TYPE, &wst_resource_factory_port_type);
		DPWS_ADD_PTR_ATT(hman_class, DPWS_STR_ID, class_id);
		DPWS_SET_PTR_ATT(hman_class, DPWS_PTR_HANDLING_FUNCTION, &wsman_serve_request);
		wsman_set_resource_handlers(hman_class, handlers, handlers_count, default_handler);
	}
	return hman_class;
}

short wsman_create_management_service(short href_device, short href_class, char* context_path)
{
	short hman_service;

	DC_CHECK_PARAM(href_device >= 0 && href_class >= 0 && context_path);

	hman_service = dpws_create_hosted_service(href_device, href_class);
	if (hman_service >= 0) {
		short hport = dpws_create_service_port();
		if (context_path)
			DPWS_SET_STR_ATT(hport, DPWS_STR_ADDRESS, context_path);
		dpws_bind_service(hman_service, hport);
	}
	return hman_service;
}

int wsman_configure_management_service(short href_service, char* resource_uri, void* configuration_data)
{
	struct wsman_service_data * service_data;
	int i;

	DC_CHECK_PARAM(href_service >= 0 && resource_uri);

	service_data = (struct wsman_service_data *)dpws_get_ptr_att(href_service,DPWS_PTR_USER_DATA);
	if (!service_data) {
		short href_class = (short)dpws_get_int_att(href_service, DPWS_INT_CLASS_HREF);
		service_data = (struct wsman_service_data *)SOAP_MALLOC(NULL, sizeof(struct wsman_service_data));
		service_data->href_service = href_service;
		service_data->handlers = (struct wsman_handlers_list *)dpws_get_ptr_att(href_class,DPWS_PTR_USER_DATA);
		service_data->configuration =
			(struct wsman_configuration_data *)SOAP_MALLOC(NULL, service_data->handlers->handlers_count * sizeof(struct wsman_configuration_data));
		memset(service_data->configuration, 0, service_data->handlers->handlers_count * sizeof(struct wsman_configuration_data));
		DPWS_SET_PTR_ATT(href_service, DPWS_PTR_USER_DATA, service_data);
	}
	for (i = 0; i < service_data->handlers->handlers_count; i++) {
		if ((!service_data->configuration[i].resource_uri && !service_data->configuration[i].configuration_data)
			|| (!resource_uri && !service_data->configuration[i].resource_uri)
			|| (resource_uri && service_data->configuration[i].resource_uri && !strcmp(resource_uri, service_data->configuration[i].resource_uri))) {
			service_data->configuration[i].resource_uri = resource_uri; // parameters are supposed to be long-lived
			service_data->configuration[i].configuration_data = configuration_data;  // parameters are supposed to be long-lived
			break;
		}
	}
	return DPWS_OK;
}

/*----------------------------------------------------------------------------*\
 *                        Run-time Accessors and Functions                    *
\*----------------------------------------------------------------------------*/

void * wsman_get_configuration_data(struct dpws * dpws, char* resource_uri)
{
	void* result = NULL;
	struct wsman_service_data * service_data = (struct wsman_service_data *)dpws_get_endpoint_user_data(dpws);
	if (service_data) {
		int i;
		for (i = 0; service_data->configuration[i].configuration_data && i < service_data->handlers->handlers_count; i++) {
			if ((!resource_uri && !service_data->configuration[i].resource_uri)
				|| (resource_uri && service_data->configuration[i].resource_uri && !strcmp(resource_uri, service_data->configuration[i].resource_uri))) {
					result = service_data->configuration[i].configuration_data;
					break;
			}
		}
	}
	return result;
}

char * wsman_get_resource_uri(struct dpws * dpws)
{
	return dpws->wsman_headers.ref_params.resource_uri;
}

int wsman_get_selectors(struct dpws * dpws, struct wsman_selector ** selectors, int * selectors_count)
{
	*selectors = dpws->wsman_headers.ref_params.selectors;
	*selectors_count = dpws->wsman_headers.ref_params.nb_selectors;
	return DPWS_OK;
}

int wsman_use_fragment_transfer(struct dpws * dpws)
{
	return dpws->wsman_headers.fragment_transfer != NULL;
}

int wsman_get_xpath_level_1(struct dpws * dpws, struct qname ** xpath, int * segment_count)
{
	DA_TYPED(qn) tokens;
	struct xpath_expression * xpath_expr = dpws->wsman_headers.fragment_transfer;
    int ret = DPWS_OK;

	DA_INIT(struct qname, &tokens, DC_MEM_WSMAN, p_default_allocator, 3);
    *xpath = NULL;
	*segment_count = 0;
	if (xpath_expr) {
		ret = wsman_parse_xpath_level_1(dpws, xpath_expr, &tokens);
		if (!ret) {
			*xpath = tokens.tab;
			*segment_count = tokens.nb;
		}
	}
	return ret;
}

int wsman_serialize_created_epr_callback(void * szr_context, void* user_data)
{
	int ret = DPWS_OK;
	struct wsa_endpoint_ref * body = (struct wsa_endpoint_ref *)user_data;
	epx_start_element(szr_context, WST_URI, WST_ELT_RESOURCE_CREATED);
	epx_define_prefix(szr_context, WST_PREFIX, WST_URI);
	epx_start_element(szr_context, WSA_URI, WSA_ELT_ADDRESS);
	epx_put_characters(szr_context, body->address);
	epx_end_element(szr_context, WSA_URI, WSA_ELT_ADDRESS);

	if (szr_context, body->wsman_params.resource_uri || body->wsman_params.nb_selectors > 0) {
		epx_start_element(szr_context, WSA_URI, WSA_ELT_REFERENCE_PARAMS);

		if (body->wsman_params.resource_uri) {
			epx_start_element(szr_context, WSMAN_URI, WSMAN_ELT_RESOURCE_URI);
			epx_put_characters(szr_context, body->wsman_params.resource_uri);
			epx_end_element(szr_context, WSMAN_URI, WSMAN_ELT_RESOURCE_URI);
		}

		if (body->wsman_params.nb_selectors > 0) {
			int i;
			epx_start_element(szr_context, WSMAN_URI, WSMAN_ELT_SELECTOR_SET);
			for (i = 0; i < body->wsman_params.nb_selectors; i++) {
				epx_start_element(szr_context, WSMAN_URI, WSMAN_ELT_SELECTOR);
				epx_add_attribute(szr_context, NULL, WSMAN_ATT_SELECTOR_NAME, body->wsman_params.selectors[i].name);
				epx_put_characters(szr_context, body->wsman_params.selectors[i].value);
				epx_end_element(szr_context, WSMAN_URI, WSMAN_ELT_SELECTOR);
			}
			epx_end_element(szr_context, WSMAN_URI, WSMAN_ELT_SELECTOR_SET);
		}
		epx_end_element(szr_context, WSA_URI, WSA_ELT_REFERENCE_PARAMS);
	}
	epx_end_element(szr_context, WST_URI, WST_ELT_RESOURCE_CREATED);

	return ret ? WSMAN_ERR_INTERNAL_ERROR: DPWS_OK;
}

/*----------------------------------------------------------------------------*\
 *                                 Utilities                                  *
\*----------------------------------------------------------------------------*/

static int parse_token(struct dpws * dpws, char * token, DA_TYPED(qn) * tokens, int nb_pfx_def, struct prefix_def * ns_ctx)
{
	size_t len;
	int i;
	struct qname * qn;
	char * colon;

	if (!(qn = DA_ADD(tokens)))
		return DPWS_ERR_EOM;
	if ((colon = strchr(token, ':'))) {
		qn->lname = colon + 1;
		len = colon - token;
		for (i = 0; i < nb_pfx_def; i++) {
			if (!strncmp(ns_ctx[i].ns_prefix, token, len)) {
				qn->ns = ns_ctx[i].ns_uri;
				break;
			}
		}
		if (i == nb_pfx_def)
			return DPWS_ERR_XPATH_FRAGMENT_PARSING;
	}
	else
		qn->lname = token;

	return DPWS_OK;
}

// Only qname list currently
static int wsman_parse_xpath_level_1(struct dpws * dpws, struct xpath_expression * xpath_expr, DA_TYPED(qn) * tokens)
{
    char * p = xpath_expr->exp, * token = NULL;
    int ret = DPWS_OK;
    for (;;p++)
    {
        switch (*p)
        {
        case '\0':
            if (token)
                ret = parse_token(dpws, token, tokens, xpath_expr->nb_pfx_def, xpath_expr->ns_ctx);
            return ret;
        case '/':
            if (token)
                *p = '\0';	// ! string is modified (optimization)
            else
            	token = XPATH_ROOT_TOKEN;
            ret = parse_token(dpws, token, tokens, xpath_expr->nb_pfx_def, xpath_expr->ns_ctx);
            if (ret)
                return ret;
            break;
        default:
            if (!token)
                token = p;
        }
    }
}

int wsman_set_fault(struct dpws * dpws, int err, int underlying)
{
	int ret = DPWS_OK;

	switch (err)
	{
	case WSMAN_ERR_INVALID_RESOURCE:
		ret = WSA_ERR_DESTINATION_UNREACHABLE;
		*soap_faultdetail(dpws_dpws2soap(dpws)) = WSMAN_FAULT_DETAIL_INVALID_RESOURCE;
		break;
	case WSMAN_ERR_INSUFFICIENT_SELECTORS:
		ret = soap_sender_fault_subcode(
				dpws_dpws2soap(dpws), WSMAN_FAULT_SC_INVALID_SELECTORS,
				NULL, WSMAN_FAULT_DETAIL_INSUFFICIENT_SELECTORS
				);
		break;
	case WSMAN_ERR_SELECTOR_TYPE_MISMATCH:
		ret = soap_sender_fault_subcode(
				dpws_dpws2soap(dpws), WSMAN_FAULT_SC_INVALID_SELECTORS,
				NULL, WSMAN_FAULT_DETAIL_TYPE_MISMATCH
				);
		break;
	case WSMAN_ERR_INVALID_SELECTOR_VALUE:
		ret = soap_sender_fault_subcode(
				dpws_dpws2soap(dpws), WSMAN_FAULT_SC_INVALID_SELECTORS,
				NULL, WSMAN_FAULT_DETAIL_INVALID_VALUE
				);
		break;
	case WSMAN_ERR_UNEXPECTED_SELECTORS:
		ret = soap_sender_fault_subcode(
				dpws_dpws2soap(dpws), WSMAN_FAULT_SC_INVALID_SELECTORS,
				NULL, WSMAN_FAULT_DETAIL_UNEXPECTED_SELECTORS
				);
		break;
	case WSMAN_ERR_DUPLICATE_SELECTORS:
		ret = soap_sender_fault_subcode(
				dpws_dpws2soap(dpws), WSMAN_FAULT_SC_INVALID_SELECTORS,
				NULL, WSMAN_FAULT_DETAIL_DUPLICATE_SELECTORS
				);
		break;
	case WSMAN_ERR_INTERNAL_ERROR:	// subcode : wsman:InternalError
		ret = soap_sender_fault_subcode(
				dpws_dpws2soap(dpws), WSMAN_FAULT_SC_INTERNAL_ERROR, NULL, NULL
				);
		break;
	case WSMAN_ERR_INVALID_VALUES:
		ret = soap_sender_fault_subcode(
				dpws_dpws2soap(dpws), WSMAN_FAULT_SC_INVALID_REPRESENTATION,
				NULL, WSMAN_FAULT_DETAIL_INVALID_VALUES
				);
		break;
	case WSMAN_ERR_MISSING_VALUES:
		ret = soap_sender_fault_subcode(
				dpws_dpws2soap(dpws), WSMAN_FAULT_SC_INVALID_REPRESENTATION,
				NULL, WSMAN_FAULT_DETAIL_MISSING_VALUES
				);
		break;
	case WSMAN_ERR_INVALID_NAMESPACES:
		ret = soap_sender_fault_subcode(
				dpws_dpws2soap(dpws), WSMAN_FAULT_SC_INVALID_REPRESENTATION,
				NULL, WSMAN_FAULT_DETAIL_INVALID_NAMESPACES
				);
		break;
	case WSMAN_ERR_ALREADY_EXISTS:
		ret = soap_sender_fault_subcode(
				dpws_dpws2soap(dpws), WSMAN_FAULT_SC_ALREADY_EXISTS, NULL, NULL
				);
		break;
	case WSMAN_ERR_UNSUPPORTED_FRAGMENT_ACCESS:	// subcode : wsman:UnsupportedFeature
		ret = soap_sender_fault_subcode(
				dpws_dpws2soap(dpws), WSMAN_FAULT_SC_UNSUPPORTED_FEATURE,
				NULL, WSMAN_FAULT_DETAIL_FRAGMENT_LEVEL_ACCESS
				);
		break;
	case WSMAN_ERR_INVALID_FRAGMENT:
		ret = soap_sender_fault_subcode(
				dpws_dpws2soap(dpws), WSMAN_FAULT_SC_INVALID_REPRESENTATION,
				NULL, WSMAN_FAULT_DETAIL_INVALID_FRAGMENT
				);
		break;
	case WSMAN_ERR_CONCURRENCY:
		ret = soap_sender_fault_subcode(
				dpws_dpws2soap(dpws), WSMAN_FAULT_SC_CONCURRENCY, NULL, NULL
				);
		break;
	case WSMAN_ERR_SCHEMA_VALIDATION_ERROR:
		ret = soap_sender_fault_subcode(
				dpws_dpws2soap(dpws), WSMAN_FAULT_SC_SCHEMA_VALIDATION_ERROR, NULL, NULL
				);
		break;
	}
	if (underlying) {
		char * buf;
		DC_ERROR_ASSERT_ALLOC(buf = DC_MSG_MALLOC(DC_MEM_WSMAN, dpws, sizeof(IMPLEMENTATION_ERROR_MSG) + 9));
		sprintf(buf, IMPLEMENTATION_ERROR_MSG, underlying);
		*soap_faultstring(dpws_dpws2soap(dpws)) = buf;
	}
	else
		*soap_faultstring(dpws_dpws2soap(dpws)) = NULL;

DC_FUNC_ERROR
	return ret;
}
