/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|                      ->>  Copyright 2008 Odonata <<-                         |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 1772 $
|                     $Date: 2008-10-09 11:57:03 +0200 (jeu., 09 oct. 2008) $
\*============================================================================*/

#include "dcGSOAP_Runtime.h"
#include "dc/dc_Types.h"
#include "dcCOMN_Tools.h"

/* Includes the gSOAP runtime, to gain access to static functions
 * that need to be made public
 */
#ifdef WIN32
#ifdef __GNUC__
#undef WIN32
#endif
#include "stdsoap2.inc"
#ifdef __GNUC__
#define WIN32
#endif
#else
#include "stdsoap2.inc"
#endif

/*----------------------------------- Types ----------------------------------*/

DA_TYPED_DECL(pdef, struct prefix_def);

/*------------------------- Static Functions prototypes ----------------------*/

static void * gsoap_alloc(int mod, void * param, size_t len);
static void gsoap_free(int mod, void * param, void * buf);
static DC_BOOL pfx_compare(struct prefix_def * pfx_def, const char * pfx);

/*----------------------------------------------------------------------------*\
 *                    GSOAP ALLOCATOR FOR DC DYNAMIC ARRAYS                    *
\*----------------------------------------------------------------------------*/

static void * gsoap_alloc(int mod, void * param, size_t len) { return soap_malloc((struct soap *)param, len);}
static void gsoap_free(int mod, void * param, void * buf) {}

da_allocator_t * soap_get_da_allocator(struct soap * soap, da_allocator_t * allocator)
{
	allocator->alloc_cbk = gsoap_alloc;
	allocator->free_cbk = gsoap_free;
	allocator->param = soap;
	return allocator;
}

/*----------------------------------------------------------------------------*\
 *                       CUSTOM GSOAP TYPES MARSHALLING                       *
\*----------------------------------------------------------------------------*/

int soap_simple_element_out(struct soap *soap, const char *tag, const char *value)
{
    if (soap_element_begin_out(soap, tag, -1, "")
		|| soap_string_out(soap, value, 0)
		|| soap_element_end_out(soap, tag))
		return soap->error;
	return SOAP_OK;
}

int soap_simple_element_in(struct soap *soap, const char *tag, char** value)
{
	int status;
    if (!(status = soap_element_begin_in(soap, tag, 0)) && soap->body) {
	    *value = soap_string_in(soap, 0, -1, -1);
	    status = soap_element_end_in(soap, tag);
    }
    return status;
}

int soap_empty_element_out(struct soap *soap, const char *tag)
{
    if (soap_element(soap, tag, -1, ""))
        return soap->error;
    return soap_element_start_end_out(soap, tag);
}


/*----------------------------------------------------------------------------*\
 *                              RUNTIME ADJUNCTS                              *
\*----------------------------------------------------------------------------*/
char * soap_get_namespace_uri(struct soap * soap, const char * prefix, size_t p_len)
{
    char * ns_uri = NULL;

	if (p_len > 0 ? !strncmp(prefix, XML_NS_PREFIX, p_len) : !strcmp(prefix, XML_NS_PREFIX))
        ns_uri = XML_NS_URI;
	else if (p_len > 0 ? !strncmp(prefix, XMLNS_NS_PREFIX, p_len) : !strcmp(prefix, XMLNS_NS_PREFIX))
        ns_uri = XMLNS_NS_URI;
    else
    {
        struct soap_nlist *np = soap->nlist;

        while (np && (p_len > 0 ? (strncmp(np->id, prefix, p_len) || np->id[p_len]): strcmp(np->id, prefix)))
            np = np->next;
        if (np)
        {	// found
            if (np->index >= 0)
            {
                struct Namespace ns = soap->local_namespaces[np->index];
                ns_uri = (char *) (ns.out ? ns.out : ns.ns);
            }
            else
                ns_uri = np->ns;
        }
    }
    return ns_uri;
}

char * soap_get_ns_uri(struct soap * soap, const char * qname)
{
    char * colon = (char*) strchr(qname, ':');
    return colon ? soap_get_namespace_uri(soap, qname, colon - qname) : soap_get_namespace_uri(soap, "", 0);
}

char * soap_get_lname(struct soap * soap, const char * qname)
{
    char * colon = (char*)strchr(qname, ':');
    return colon ? colon + 1 : (char *)qname;
}

const char * soap_check_prefix_definition(struct soap * soap, const char *ns_uri, const char * pfx)
{
    struct Namespace *p;
    char *t = NULL;
    struct soap_nlist *np = soap->nlist;
	struct Namespace * ns = soap->local_namespaces;

	if (ns) {
		for (; ns->id && strcmp(ns_uri, ns->ns); ns++);
		t = (char *)ns->id;
	}
	if (!t) {
	    for (; np; np = np->next)
	    {
	        if (np->index >= 0)
	        {
	            p = soap->local_namespaces + np->index;
	            if (p->ns)
	                if (!strcmp(ns_uri, p->ns))
	                    break;
	            if (p->in)
	                if (!soap_tag_cmp(ns_uri, p->in))
	                    break;
	        }
	        else if (!strcmp(ns_uri, np->ns))
	            break;
	    }
		if (np)
			t = np->id;
	}
    if (!t)
    {
	   	if (!strcmp(ns_uri, XML_NS_URI))
	   		t = XML_NS_PREFIX;
	   	else {
	        t = soap->tmpbuf + 6;	// xmlns offset
	   		strcpy(soap->tmpbuf, "xmlns:");
	    	if (pfx)
	    		strcpy(t, pfx);
	    	else
	        	sprintf(t, "_%d", soap->idnum);	// buffer overflow possible (see gSOAP too, code copied)
	        soap_set_attr(soap, soap->tmpbuf, ns_uri);
			if (!soap_push_namespace(soap, soap->tmpbuf + 6, ns_uri)) {
				t = soap->nlist->id;
			}
	        soap->idnum++;
	   	}
    }
    return t;
}

void soap_update_namespace_uri(struct soap * soap, const char * prefix, char * uri)
{
    struct Namespace * ns = soap->local_namespaces;

	while (ns->id) {
		if (!strcmp(ns->id, prefix)) {
			ns->ns = uri;
			break;
		}
		ns++;
	}
}

/*----------------------------------------------------------------------------*\
 *                                 UTILITIES                                  *
\*----------------------------------------------------------------------------*/

static DC_BOOL pfx_compare(struct prefix_def * pfx_def, const char * pfx)
{
	return strcmp(pfx_def->ns_prefix, pfx) ? DC_FALSE : DC_TRUE;
}

struct prefix_def * soap_build_ns_context(struct soap *soap, int * nb_def)
{
    struct soap_nlist *np = soap->nlist;
	da_allocator_t daa;
    DA_TYPED(pdef) ns_ctx;

    DA_INIT(struct prefix_def, &ns_ctx, DC_MEM_TRANSIENT, soap_get_da_allocator(soap, &daa), 5);

    // Last defined are first in the list
    for (; np; np = np->next)
    {
    	// look for the prefix in the built list to see if it is not already defined
    	int i = DA_BROWSE(&ns_ctx, pfx_compare, np->id);
    	if (i == ns_ctx.nb) {	// not found
			struct prefix_def * pfx_def;
			DA_ADD(&ns_ctx);
			pfx_def = DA_GET(&ns_ctx,ns_ctx.nb - 1);
    		pfx_def->ns_prefix = soap_strdup(soap, np->id);
    		pfx_def->ns_uri = soap_strdup(soap, np->index >= 0 ? soap->local_namespaces[np->index].ns : np->ns);
    	}
    }
    *nb_def = ns_ctx.nb;
    return ns_ctx.tab;
}

/*----------------------------------------------------------------------------*\
 *                                 STATIC GSOAP FUNCTIONS                     *
\*----------------------------------------------------------------------------*/

size_t dc_soap_count_attachments(struct soap *soap)
{
	return soap_count_attachments(soap);
}
