/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2128 $
|                     $Date: 2009-02-27 15:07:28 +0100 (ven, 27 fév 2009) $
\*============================================================================*/

/******************************************************************************\
 *                         Dynamic deployment server                          *
\******************************************************************************/
#include "dc/dc_Dyndepl.h"
#include "dc/dc_DyndeplConstants.h"
#include "dc/dc_Constants.h"
#include "dc/dc_Dpws.h"
#include "dc/dc_GenericInvocation.h"
#include "dc/dc_XMLConfiguration.h"
#include "dc/dc_WsMan.h"
#include "dcCOMN_Tools.h"
#include "dcXCONF_Definitions.h"
#include "dcXCONF_Serializer.h"
#include "dcXCONF_Parser.h"
#include "dcXCONF_Manager.h"
#include "dcGSOAP_Runtime.h"
#include "dcXTOOL_SchemaParsing.h"

/*----------------------------------- Types ----------------------------------*/

typedef enum {TYPES_FRAGMENT, SCOPES_FRAGMENT, THIS_MODEL_FRAGMENT, THIS_DEVICE_FRAGMENT, UNSUPPORTED_FRAGMENT, NO_FRAGMENT} fragment_t;

struct resource_info {
	short href_resource;
	int count;
};

/*------------------------- Static Functions prototypes ----------------------*/

static int dyndepl_serialize_impl(short href_sclass, void * szr_ctx);
static int dyndepl_load_cbk(
	short href_sclass, void * psr_ctx,
	dispatch_cbk * p_dispatch_cbk, struct scl_callbacks * p_cbks
	);
static int parse_fragment(struct dpws * dpws, fragment_t * fragment);
static int get_resource_href(struct dpws * dpws, char * selector, short (*retrieval_api)(char *), short * href_resource);
static int get_service_href(struct dpws * dpws, short * href_resource);
static int device_response_cbk(void * szr_context, struct resource_info * resource);
static int device_types_response_cbk(void * szr_context, struct resource_info * resource);
static int device_scopes_response_cbk(void * szr_context, struct resource_info * resource);
static int this_model_response_cbk(void * szr_context, struct resource_info * resource);
static int this_device_response_cbk(void * szr_context, struct resource_info * resource);
static int service_class_response_cbk(void * szr_context, struct resource_info * resource);
static int service_response_cbk(void * szr_context, struct resource_info * resource);
static int set_parsing_error(struct dpws * dpws, int err);
static int device_resource_get(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data);
static int device_resource_put(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data);
static int device_resource_create(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data);
static int device_resource_delete(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data);
static int service_class_resource_get(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data);
static int service_class_resource_put(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data);
static int service_class_resource_create(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data);
static int service_class_resource_delete(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data);
static int service_resource_get(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data);
static int service_resource_put(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data);
static int service_resource_create(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data);
static int service_resource_delete(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data);

/*----------------------------------- Data -----------------------------------*/

static config_cbk config_callback;
static struct wsman_resource_handler device_resource_handler =
	{ DYNDEPL_RESOURCE_DEVICE, device_resource_get, device_resource_put, device_resource_create, device_resource_delete };
static struct wsman_resource_handler service_class_resource_handler =
	{ DYNDEPL_RESOURCE_SERVICE_CLASS, service_class_resource_get, service_class_resource_put, service_class_resource_create, service_class_resource_delete };
static struct wsman_resource_handler service_resource_handler =
	{ DYNDEPL_RESOURCE_SERVICE, service_resource_get, service_resource_put, service_resource_create, service_resource_delete };
static struct wsman_resource_handler * resource_handlers[] = { &device_resource_handler, &service_class_resource_handler, &service_resource_handler };

/*----------------------------------------------------------------------------*\
 *                        Static Service Class Loader                         *
\*----------------------------------------------------------------------------*/

static int dyndepl_serialize_impl(short href_sclass, void * szr_ctx)
{
	epx_start_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_SSCL_IMPL);
//	epx_define_prefix(szr_ctx, DYNDEPL_NS_PREFIX, DYNDEPL_NS_URI);	// no need: defined at the root
	epx_end_element(szr_ctx, DYNDEPL_NS_URI, DYNDEPL_ELT_SSCL_IMPL);
	return DPWS_OK;
}

static int dyndepl_load_cbk(
	short href_sclass, void * psr_ctx,
	dispatch_cbk * p_dispatch_cbk, struct scl_callbacks * p_cbks
	)
{
	wsman_set_resource_handlers(href_sclass, resource_handlers, 3, NULL);
	*p_dispatch_cbk = wsman_serve_request;
	p_cbks->serialize_impl = dyndepl_serialize_impl;
	if (epx_next(psr_ctx) != EPX_EVT_END_ELEMENT || QNAME_NOT_EQUALS_WILDCARD(
							epx_get_ns_uri(psr_ctx), epx_get_lname(psr_ctx),
							DYNDEPL_NS_URI, DYNDEPL_ELT_SSCL_IMPL))
		return DPWS_ERR_INCORRECT_IMPL_TAG;
	return DPWS_OK;
}

DC_RT_FMAC1 int dpws_register_dyndepl_loader()
{
	struct qname dyndepl_iqn = {DYNDEPL_NS_URI, DYNDEPL_ELT_SSCL_IMPL};
	return dpws_register_loader(&dyndepl_iqn, dyndepl_load_cbk);
}

DC_RT_FMAC1 int dpws_register_config_cbk(config_cbk cbk)
{
	DC_CHECK_PARAM(cbk);

	config_callback = cbk;
	return DPWS_OK;
}

/*----------------------------------------------------------------------------*\
 *                                 Utilities                                  *
\*----------------------------------------------------------------------------*/

static int parse_fragment(struct dpws * dpws, fragment_t * fragment)
{
	struct qname * tokens;
	int ntokens;
    int ret = DPWS_OK;

	ret = wsman_get_xpath_level_1(dpws, &tokens, &ntokens);
	if (ret || ntokens == 0)
		*fragment = NO_FRAGMENT;
	else {
		if (ntokens > 1)
			*fragment = UNSUPPORTED_FRAGMENT;
		else {
			if (strcmp(DYNDEPL_NS_URI, tokens[0].ns))
				*fragment = UNSUPPORTED_FRAGMENT;
			else {
				if (!strcmp(DYNDEPL_ELT_TYPES, tokens[0].lname))
					*fragment = TYPES_FRAGMENT;
				else if (!strcmp(DYNDEPL_ELT_SCOPES, tokens[0].lname))
					*fragment = SCOPES_FRAGMENT;
				else if (!strcmp(WDP_ELT_THIS_MODEL, tokens[0].lname))
					*fragment = THIS_MODEL_FRAGMENT;
				else if (!strcmp(WDP_ELT_THIS_DEVICE, tokens[0].lname))
					*fragment = THIS_DEVICE_FRAGMENT;
			}
		}
	}

	return ret;
}

static int get_resource_href(struct dpws * dpws, char * selector, short (*retrieval_api)(char *), short * href_resource)
{
	int ret = DPWS_OK, i;
	struct wsman_selector * selectors;
	int nb_selectors;

	*href_resource = -1;
	ret = wsman_get_selectors(dpws, &selectors, &nb_selectors);

	if (ret || nb_selectors == 0)
		return WSMAN_ERR_INSUFFICIENT_SELECTORS;
	for(i = 0; i < nb_selectors; i++) {
		if (!strcasecmp(selectors[i].name, selector)) {
			if (*href_resource >= 0)
				return WSMAN_ERR_DUPLICATE_SELECTORS;
			if (selectors[i].value) {
				*href_resource = retrieval_api(selectors[i].value);
				if (*href_resource < 0)
					return WSMAN_ERR_INVALID_SELECTOR_VALUE;
			}
			else
				return WSMAN_ERR_INVALID_SELECTOR_VALUE;
		}
	}
	if (*href_resource < 0)
		return WSMAN_ERR_INSUFFICIENT_SELECTORS;
	if (i > 1)
		return WSMAN_ERR_UNEXPECTED_SELECTORS;

	return ret;
}

static int get_service_href(struct dpws * dpws, short * href_resource)
{
	int i, ret = DPWS_OK;
	short href_device = -1;
	char * service_id = NULL;
	struct wsman_selector * selectors;
	int nb_selectors;

	*href_resource = -1;
	ret = wsman_get_selectors(dpws, &selectors, &nb_selectors);

	if (ret || nb_selectors == 0)
		return WSMAN_ERR_INSUFFICIENT_SELECTORS;
	for(i = 0; i < nb_selectors; i++) {
		if (!strcasecmp(selectors[i].name, DYNDEPL_SELECTOR_DEVICE)) {
			if (href_device >= 0)
				return WSMAN_ERR_DUPLICATE_SELECTORS;
			if (selectors[i].value) {
				href_device = dpws_get_device_handle(selectors[i].value);
				if (href_device < 0)
					return WSMAN_ERR_INVALID_SELECTOR_VALUE;
			}
			else
				return WSMAN_ERR_INVALID_SELECTOR_VALUE;
		}
		else if (!strcasecmp(selectors[i].name, DYNDEPL_SELECTOR_SERVICE)) {
			if (service_id)
				return WSMAN_ERR_DUPLICATE_SELECTORS;
			if (selectors[i].value)
				service_id = selectors[i].value;
			else
				return WSMAN_ERR_INVALID_SELECTOR_VALUE;
		}
	}
	if (href_device < 0 || !service_id)
		return WSMAN_ERR_INVALID_SELECTOR_VALUE;
	if (i > 2)
		return WSMAN_ERR_UNEXPECTED_SELECTORS;

	*href_resource = dpws_get_service_handle(href_device, service_id);
	if (*href_resource < 0)
		return WSMAN_ERR_INVALID_SELECTOR_VALUE;

	dpws_release_handle(href_device);	// For the retrieval API

	return ret;
}

struct wsa_endpoint_ref * create_new_resource_epr(struct dpws * dpws, char * name1, char * value1, char * name2, char * value2)
{
	struct wsa_endpoint_ref * epr;
	struct wsman_selector * selectors;
	int nb_selectors;

	DC_ERROR_ASSERT_NO_RC(epr = DC_MSG_MALLOC(DC_MEM_DYNDEPL, dpws, sizeof(struct wsa_endpoint_ref)));

	memset(epr, 0, sizeof(struct wsa_endpoint_ref));
	epr->address = dpws->to;
	epr->wsman_params.resource_uri = wsman_get_resource_uri(dpws);
	if (name2) {
		nb_selectors = 2;
		DC_ERROR_ASSERT_NO_RC(selectors = DC_MSG_MALLOC(DC_MEM_DYNDEPL, dpws, 2 * sizeof(struct wsman_selector)));
		selectors[1].name = name2;
		selectors[1].value = value2;

	}
	else {
		nb_selectors = 1;
		DC_ERROR_ASSERT_NO_RC(selectors = DC_MSG_MALLOC(DC_MEM_DYNDEPL, dpws, sizeof(struct wsman_selector)));
	}
	selectors[0].name = name1;
	selectors[0].value = value1;
	epr->wsman_params.selectors = selectors;
	epr->wsman_params.nb_selectors = nb_selectors;

DC_FUNC_ERROR
	return epr;
}
/*----------------------------------------------------------------------------*\
 *                     Management Service implementation                      *
\*----------------------------------------------------------------------------*/
static int device_response_cbk(void * szr_context, struct resource_info * resource)
{
	int ret = dpws_serialize_device(szr_context, resource->href_resource);
	if (--resource->count == 0)
		dpws_release_handle(resource->href_resource);	// For the retrieval API
	return ret;
}

static int device_types_response_cbk(void * szr_context, struct resource_info * resource)
{
	int ret = dpws_serialize_types(szr_context, resource->href_resource);
	if (--resource->count == 0)
		dpws_release_handle(resource->href_resource);	// For the retrieval API
	return ret;
}

static int device_scopes_response_cbk(void * szr_context, struct resource_info * resource)
{
	int ret = dpws_serialize_scopes(szr_context, resource->href_resource);
	if (--resource->count == 0)
		dpws_release_handle(resource->href_resource);	// For the retrieval API
	return ret;
}

static int this_model_response_cbk(void * szr_context, struct resource_info * resource)
{
	int ret = dpws_serialize_this_model(szr_context, resource->href_resource);
	if (--resource->count == 0)
		dpws_release_handle(resource->href_resource);	// For the retrieval API
	return ret;
}

static int this_device_response_cbk(void * szr_context, struct resource_info * resource)
{
	int ret = dpws_serialize_this_device(szr_context, resource->href_resource);
	if (--resource->count == 0)
		dpws_release_handle(resource->href_resource);	// For the retrieval API
	return ret;
}

static int service_class_response_cbk(void * szr_context, struct resource_info * resource)
{
	int ret = dpws_serialize_service_class(szr_context, resource->href_resource);
	if (--resource->count == 0)
		dpws_release_handle(resource->href_resource);	// For the retrieval API
	return ret;
}

static int service_response_cbk(void * szr_context, struct resource_info * resource)
{
	int ret = dpws_serialize_service(szr_context, resource->href_resource);
	if (--resource->count == 0)
		dpws_release_handle(resource->href_resource);	// For the retrieval API
	return ret;
}

static int set_parsing_error(struct dpws * dpws, int err)
{
	switch(err)
	{
	case DPWS_OK:
		return DPWS_OK;
	case SGXP_ERR_MISSING_EXPECTED_ELT:
	case SGXP_ERR_MISSING_REQUIRED_ATT:
		return WSMAN_ERR_MISSING_VALUES;
	default:
		return WSMAN_ERR_INVALID_VALUES;
	}
}

/***********************************************************************************/
/*                              Device resource                                    */
/***********************************************************************************/

int device_resource_get(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data)
{
	fragment_t fragment;
	struct resource_info * r_info;
	int ret = DPWS_OK;

	DC_RETURN_ASSERT_ALLOC(r_info = DC_MSG_MALLOC(DC_MEM_DYNDEPL, dpws, sizeof(struct resource_info)));
	if (dpws->soap.mode & SOAP_IO_LENGTH)
		r_info->count = 2;
	else
		r_info->count = 1;

	DC_RETURN_ASSERT_CALL(get_resource_href(dpws, DYNDEPL_SELECTOR_DEVICE, dpws_get_device_handle, &r_info->href_resource));
	DC_ERROR_ASSERT_CALL(parse_fragment(dpws, &fragment));

	switch (fragment)
	{
	case TYPES_FRAGMENT:
		*response_cbk = (serialize_cbk)device_types_response_cbk;
		break;
	case SCOPES_FRAGMENT:
		*response_cbk = (serialize_cbk)device_scopes_response_cbk;
		break;
	case THIS_MODEL_FRAGMENT:
		*response_cbk = (serialize_cbk)this_model_response_cbk;
		break;
	case THIS_DEVICE_FRAGMENT:
		*response_cbk = (serialize_cbk)this_device_response_cbk;
		break;
	case NO_FRAGMENT:
		*response_cbk = (serialize_cbk)device_response_cbk;
		break;
	case UNSUPPORTED_FRAGMENT:
		return WSMAN_ERR_UNSUPPORTED_FRAGMENT_ACCESS;
	}
	*p_user_data = r_info;

DC_FUNC_ERROR
	if (ret)
		dpws_release_handle(r_info->href_resource);		// For the retrieval API
	return ret;
}

int device_resource_put(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data)
{
	fragment_t fragment;
	short href_device, href_new;
	struct resource_info * r_info;
	int ret = DPWS_OK;

	DC_RETURN_ASSERT_ALLOC(r_info = DC_MSG_MALLOC(DC_MEM_DYNDEPL, dpws, sizeof(struct resource_info)));
	if (dpws->soap.mode & SOAP_IO_LENGTH)
		r_info->count = 2;
	else
		r_info->count = 1;

	DC_RETURN_ASSERT_CALL(get_resource_href(dpws, DYNDEPL_SELECTOR_DEVICE, dpws_get_device_handle, &href_device));
	DC_ERROR_ASSERT_CALL(parse_fragment(dpws, &fragment));
	DC_ERROR_ASSERT(fragment != UNSUPPORTED_FRAGMENT, WSMAN_ERR_UNSUPPORTED_FRAGMENT_ACCESS);
	if (fragment == NO_FRAGMENT)
	{
		ret = set_parsing_error(dpws, dpws_load_device(parsing_ctx, &href_new, DC_FALSE));
		if (!ret) {
			ret = dpws_replace_device(href_device, href_new);
			if (!ret)
				*response_cbk = (serialize_cbk)device_response_cbk;
		}
	}
	else
	{
		href_new = dpws_clone_device(href_device);

		if (href_new >= 0)
		{
			switch (fragment)
			{
			case TYPES_FRAGMENT:
				ret = set_parsing_error(dpws, dpws_load_types(parsing_ctx, href_new));
				*response_cbk = (serialize_cbk)device_types_response_cbk;
				break;
			case SCOPES_FRAGMENT:
				ret = set_parsing_error(dpws, dpws_load_scopes(parsing_ctx, href_new));
				*response_cbk = (serialize_cbk)device_scopes_response_cbk;
				break;
			case THIS_MODEL_FRAGMENT:
				ret = set_parsing_error(dpws, dpws_load_this_model(parsing_ctx, href_new));
				*response_cbk = (serialize_cbk)this_model_response_cbk;
				break;
			case THIS_DEVICE_FRAGMENT:
				ret = set_parsing_error(dpws, dpws_load_this_device(parsing_ctx, href_new));
				*response_cbk = (serialize_cbk)this_device_response_cbk;
				break;
			default:
				break;
			}
		}
		else
			ret = dpws->soap.error = href_new;
		if (ret)
			*response_cbk = NULL;
		else
			DC_ERROR_ASSERT_CALL(dpws_replace_device(href_device, href_new));
	}
	if (!ret) {
		r_info->href_resource = href_new;	// new handle will be released by the callback
		if (config_callback)
			config_callback();
	}
	*p_user_data = r_info;

DC_FUNC_ERROR
	dpws_release_handle(href_device);		// For the retrieval API
	return ret;
}

int device_resource_create(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data)
{
	int ret = DPWS_OK;
	short href_new_resource;

	if (wsman_use_fragment_transfer(dpws))
		return WSMAN_ERR_UNSUPPORTED_FRAGMENT_ACCESS;

	ret = set_parsing_error(dpws, dpws_load_device(parsing_ctx, &href_new_resource, DC_TRUE));
	if (!ret)
	{
		if (config_callback)
			config_callback();
		*response_cbk = (serialize_cbk)wsman_serialize_created_epr_callback;
		*p_user_data = create_new_resource_epr(dpws, DYNDEPL_SELECTOR_DEVICE, (char*)dpws_get_ptr_att(href_new_resource, DPWS_STR_DEVICE_ID), NULL, NULL);
	}
	return ret;
}

int device_resource_delete(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data)
{
	int ret = DPWS_OK;
	short href_resource;

	if (wsman_use_fragment_transfer(dpws))
		return WSMAN_ERR_UNSUPPORTED_FRAGMENT_ACCESS;

	if ((ret = get_resource_href(dpws, DYNDEPL_SELECTOR_DEVICE, dpws_get_device_handle, &href_resource)))
		return ret;
	ret = dpws_disable_device(href_resource);	// Only hold normally by the invocation list (disable == release)
	dpws_release_handle(href_resource);	// For the retrieval API
	if (config_callback)
		config_callback();
	return ret;
}

/***********************************************************************************/
/*                             Service class resource                              */
/***********************************************************************************/

int service_class_resource_get(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data)
{
	int ret = DPWS_OK;
	struct resource_info * r_info = DC_MSG_MALLOC(DC_MEM_DYNDEPL, dpws, sizeof(struct resource_info));

	if (dpws->soap.mode & SOAP_IO_LENGTH)
		r_info->count = 2;
	else
		r_info->count = 1;

	if (wsman_use_fragment_transfer(dpws))
		return WSMAN_ERR_UNSUPPORTED_FRAGMENT_ACCESS;

	if ((ret = get_resource_href(dpws, DYNDEPL_SELECTOR_SERVICE_CLASS, dpws_get_service_class_handle, &r_info->href_resource)))
		return ret;

	*response_cbk = (serialize_cbk)service_class_response_cbk;
	*p_user_data = r_info;

	return ret;
}

int service_class_resource_put(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data)
{
	int ret = DPWS_OK;
	if (parsing_ctx)
		ret = WSA_ERR_ACTION_NOT_SUPPORTED;
	return ret;
}

int service_class_resource_create(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data)
{
	int ret = DPWS_OK;
	short href_new_resource;

	if (wsman_use_fragment_transfer(dpws))
		return WSMAN_ERR_UNSUPPORTED_FRAGMENT_ACCESS;

	ret = set_parsing_error(dpws, dpws_load_service_class(parsing_ctx, &href_new_resource));
	if (!ret)
	{
		if (config_callback)
			config_callback();
		*response_cbk = (serialize_cbk)wsman_serialize_created_epr_callback;
		*p_user_data = create_new_resource_epr(dpws, DYNDEPL_SELECTOR_SERVICE_CLASS, (char*)dpws_get_ptr_att(href_new_resource, DPWS_STR_ID), NULL, NULL);
	}
	return ret;
}

int service_class_resource_delete(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data)
{
	int ret = DPWS_OK;
	short href_resource;

	if (wsman_use_fragment_transfer(dpws))
		return WSMAN_ERR_UNSUPPORTED_FRAGMENT_ACCESS;

	if ((ret = get_resource_href(dpws, DYNDEPL_SELECTOR_SERVICE_CLASS, dpws_get_service_class_handle, &href_resource)))
		return ret;
	dpws_release_handle(href_resource);	// For the retrieval API
	if (dpws_get_handle_use_count(href_resource) == 1)
		xconf_delete_service_class(href_resource);	// releases the creation handle
	else
		return WSMAN_ERR_CONCURRENCY;
	if (config_callback)
		config_callback();
	return ret;
}

/***********************************************************************************/
/*                              Service resource                                   */
/***********************************************************************************/


int service_resource_get(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data)
{
	int ret = DPWS_OK;
	struct resource_info * r_info = DC_MSG_MALLOC(DC_MEM_DYNDEPL, dpws, sizeof(struct resource_info));

	if (dpws->soap.mode & SOAP_IO_LENGTH)
		r_info->count = 2;
	else
		r_info->count = 1;

	if (wsman_use_fragment_transfer(dpws))
		return WSMAN_ERR_UNSUPPORTED_FRAGMENT_ACCESS;

	if ((ret = get_service_href(dpws, &r_info->href_resource)))
		return ret;

	*response_cbk = (serialize_cbk)service_response_cbk;
	*p_user_data = r_info;
	return ret;
}

int service_resource_put(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data)
{
	int ret = DPWS_OK;
	if (parsing_ctx)
		ret = WSA_ERR_ACTION_NOT_SUPPORTED;
	return ret;
}

int service_resource_create(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data)
{
	int ret = DPWS_OK;
	short href_new_resource, href_device, href_clone;

	if (wsman_use_fragment_transfer(dpws))
		return WSMAN_ERR_UNSUPPORTED_FRAGMENT_ACCESS;

	if ((ret = get_resource_href(dpws, DYNDEPL_SELECTOR_DEVICE, dpws_get_device_handle, &href_device)))
		return ret;
	href_clone = dpws_clone_device(href_device);

	if (!(ret = set_parsing_error(dpws, dpws_load_service(parsing_ctx, href_clone, &href_new_resource, DC_FALSE))))
	{
		ret = dpws_replace_device(href_device, href_clone);
	}
	dpws_release_handle(href_clone);	// Now enabled
	dpws_release_handle(href_device);	// For the retrieval API
	if (!ret)
	{
		if (config_callback)
			config_callback();
		*response_cbk = (serialize_cbk)wsman_serialize_created_epr_callback;
		*p_user_data =
			create_new_resource_epr(dpws, DYNDEPL_SELECTOR_DEVICE, (char*)dpws_get_ptr_att(href_clone, DPWS_STR_DEVICE_ID),
									DYNDEPL_SELECTOR_SERVICE, (char*)dpws_get_ptr_att(href_new_resource, DPWS_STR_SERVICE_ID));
	}
	return ret;
}

int service_resource_delete(struct dpws * dpws, void * parsing_ctx, serialize_cbk * response_cbk, void ** p_user_data)
{
	int ret = DPWS_OK;
	short href_resource, original, clone;
	char * service_id;

	if (wsman_use_fragment_transfer(dpws))
		return WSMAN_ERR_UNSUPPORTED_FRAGMENT_ACCESS;

	if ((ret = get_service_href(dpws, &href_resource)))
		return ret;

	original = dpws_get_service_device(href_resource);
	service_id = dpws_get_ptr_att(href_resource, DPWS_STR_SERVICE_ID);
	xconf_delete_service(href_resource);
	clone = dpws_clone_device(original);
	dpws_release_handle(original);	// For the retrieval API
	dpws_release_handle(href_resource);	// For the retrieval API
	href_resource = dpws_get_service_handle(clone, service_id);
	dpws_release_handle(href_resource);	// For the retrieval API
	if (!(ret = dpws_delete_hosted_service(href_resource)) && !(ret = dpws_replace_device(original, clone)))
		dpws_release_handle(clone);	// only held by the invokation list

	if (config_callback)
		config_callback();

	return ret;
}
