/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|                      ->>  Copyright 2008 Odonata <<-                         |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 1772 $
|                     $Date: 2008-10-09 11:57:03 +0200 (jeu., 09 oct. 2008) $
\*============================================================================*/

#include "dcDPWS_Udp.h"
#include "dcDPWS_Reactor.h"
#include "dcDCPL_Socket.h"
#include "dcDPWS_Dpws.h"
#include "dcDPWS_Discovery.h"
#include "dcCOMN_Tools.h"
#include "dc/dc_Error.h"
#include "dc/dc_Types.h"
#include "dc/dc_DpwsRequest.h"

/*------------------------- Static Functions prototypes ----------------------*/

static int convert_dcpl_udp_error(struct dcpl_error * dcpl_error);
//static int convert_dcpl_server_error(struct dpws * dpws, struct dcpl_error * dcpl_error);
static int convert_dcpl_client_error(struct dpws * dpws, struct dcpl_error * dcpl_error);
static int wsd_mcast_recv_callback(struct dpws * dpws, struct reactor_item * item, void* cbk_data);
static udp_transport_data_t * init_udp_transport_data(udp_transport_data_t * tdata);
static int udp_setup_channel(struct dpws * dpws, struct transport_data * tdata);
static int udp_teardown_channel(struct dpws * dpws, struct transport_data * tdata);
static int udp_send(struct dpws * dpws, void * transport_data, const char * buf, size_t buflen);
static int udp_recv(struct dpws * dpws, void * transport_data, char * buf, size_t buflen);
static int udp_close(struct dpws * dpws);
static int udp_response(struct dpws * dpws, void * transport_data, int status, media_type_t * ctype, size_t len);
static int udp_recvemptyresponse(struct dpws * dpws);

/*----------------------------------- Data -----------------------------------*/

static struct transport_class udp_transport_class = {
	udp_send,
	udp_recv,
	udp_response,
	dc_udp_connect,
	udp_recvemptyresponse,
	udp_close,
	NULL,
	udp_setup_channel,
	udp_teardown_channel,
	NULL
};

static int convert_dcpl_udp_error(struct dcpl_error * dcpl_error)
{
	int dpwserror;

	switch (dcpl_error->error) {
		case DCPL_SOCKET_CREATE_ERROR: dpwserror = DPWS_ERR_COULD_NOT_CREATE_SOCKET; break;
		case DCPL_SOCKET_BIND_ERROR: dpwserror = DPWS_ERR_COULD_NOT_BIND_SOCKET; break;
		case DCPL_SOCKET_REUSEADDR_ERROR: dpwserror = DPWS_ERR_COULD_NOT_SET_REUSE_ADDR_OPTION; break;
		case DCPL_SOCKET_JOIN_GROUP_ERROR: dpwserror = DPWS_ERR_COULD_NOT_JOIN_MULTICAST_GROUP; break;
		default: dpwserror = DPWS_ERR_SOCKET_ERROR; break;
	}
	return dpwserror;
}

/*
static int convert_dcpl_server_error(struct dpws * dpws, struct dcpl_error * dcpl_error)
{
	int dpwserror = convert_dcpl_udp_error(dcpl_error);
	if (dpws) {
		struct soap * soap = dpws_dpws2soap(dpws);
		soap->errnum = dcpl_error->syserr;
		soap_set_receiver_error(soap, dcpl_error_string(dcpl_error, soap->msgbuf, sizeof(soap->msgbuf)), dcpl_error->detail, SOAP_TCP_ERROR);
	}
	return dpwserror;
}
*/

static int convert_dcpl_client_error(struct dpws * dpws, struct dcpl_error * dcpl_error)
{
	int dpwserror = convert_dcpl_udp_error(dcpl_error);
	if (dpws) {
		struct soap * soap = dpws_dpws2soap(dpws);
		soap->errnum = dcpl_error->syserr;
		soap_set_sender_error(soap, dcpl_error_string(dcpl_error, soap->msgbuf, sizeof(soap->msgbuf)), dcpl_error->detail, SOAP_TCP_ERROR);
	}
	return dpwserror;
}

static int wsd_mcast_recv_callback(struct dpws * dpws, struct reactor_item * item, void* cbk_data)
{
	DPWS_SET_MULTICAST(dpws->status);	// fork using another way ?
	return dc_udp_serve_request(dpws, item, wsd_mcast_recv, cbk_data);
}

static udp_transport_data_t * init_udp_transport_data(udp_transport_data_t * tdata)
{
	return (udp_transport_data_t *)dc_transport_data_init(
			(transport_data_t *)tdata, sizeof(udp_transport_data_t), &udp_transport_class);
}

int dc_udp_add_multicast_listener(struct reactor * reactor, int family, dcpl_ip_addr_t * bind_addr, unsigned short port, const char* multicast_addr)
{
	struct dcpl_error error;
	DCPL_SOCKET socket;
	int ret = DPWS_OK;

	if ((socket = dcpl_multicast_bind(
			family,
			bind_addr ? bind_addr->ip_address : NULL,
			bind_addr ? bind_addr->itf_nb : 0,
			port,
			multicast_addr,
			&error
			)) == DCPL_INVALID_SOCKET) {
		reactor->error = error.error;
		reactor->detail = error.detail;
		reactor->syserr = error.syserr;
		return convert_dcpl_udp_error(&error);
	}
	if ((ret = dc_reactor_register_socket(reactor, socket, DC_RI_READ, wsd_mcast_recv_callback, bind_addr, NULL, NULL))) {
		dcpl_closesocket(socket, NULL);
		return ret;
	}
	return ret;
}

int dc_udp_serve_request(struct dpws * dpws, struct reactor_item * item, int (*cbk)(struct dpws *), void* callback_data)
{
	int ret = DPWS_OK;
	udp_transport_data_t udp_data;
	DCPL_SOCKET socket;
	dc_reactor_item_get_socket(item, &socket);
	init_udp_transport_data(&udp_data);
	dc_transport_setup_channel(dpws, (transport_data_t *)&udp_data, socket, DC_TRANSPORT_INPUT);
	DPWS_SET_UDP(dpws->status);
#ifndef DCPL_HAVE_PKTINFO
	if (callback_data)	// only passed when multicast. Else should make 2 fonctions ?
		((udp_transport_data_t *)dpws->transport_data)->base.itf = ((dcpl_ip_addr_t *)callback_data)->itf_nb;
#endif
	ret = cbk(dpws);
	dc_transport_teardown_channel(dpws, (transport_data_t *)&udp_data);
	dc_transport_data_clear((transport_data_t *)&udp_data);
	return ret;
}

int dc_udp_connect(struct dpws * dpws, const char * endpoint, const char * action, struct media_type * mtype, size_t len)
{
	udp_transport_data_t * tdata = (udp_transport_data_t *)dpws->transport_data;
	struct soap * soap = dpws_dpws2soap(dpws);
	int ret = DPWS_OK;

	if (endpoint)
		soap_set_endpoint(soap, endpoint);
	if (!tdata) {
		int family = is_ipv6_addr(soap->host) ? DCPL_AF_INET6 : DCPL_AF_INET;
		if ((ret = dc_udp_open_output_channel(dpws, NULL, family)))
			return ret;
		tdata = (udp_transport_data_t *)dpws->transport_data;
		tdata->base.status |= DC_TRANSPORT_INTERNAL;
	}
	if (endpoint && soap->host) {
		strcpy(tdata->base.peer_addr, soap->host);
		tdata->base.peer_port = soap->port;
	}
	return ret;
}

int dc_udp_open_output_channel(struct dpws * dpws, udp_transport_data_t * tdata, int family)
{
	struct dcpl_error error;
	DCPL_SOCKET socket = dcpl_udp_open(family, &error);
	int ret = DPWS_OK;

	if (socket == DCPL_INVALID_SOCKET)
		return convert_dcpl_client_error(dpws, &error);
	ret = dc_udp_setup_output_channel(dpws, tdata, socket);
	if (ret) {
		dcpl_closesocket(socket, NULL);
	}
	return ret;
}

int dc_udp_open_response_channel(struct dpws * dpws, udp_transport_data_t * tdata, udp_transport_data_t * request_data)
{
	int family;
	int ret = DPWS_OK;
	int user_mode = dpws->user_mode;
	if (!request_data)
		return DPWS_ERR_INVALID_PARAMETER;
	family = is_ipv6_addr(request_data->base.peer_addr) ? DCPL_AF_INET6 : DCPL_AF_INET;
	if ((ret = dc_udp_open_output_channel(dpws, tdata, family)))
		return ret;
	tdata = (udp_transport_data_t *)dpws->transport_data;
	strcpy(tdata->base.peer_addr, request_data->base.peer_addr);
	tdata->base.peer_port = request_data->base.peer_port;
	tdata->base.itf = request_data->base.itf;
	dpws->user_mode = user_mode;
	return ret;
}

int dc_udp_setup_output_channel(struct dpws * dpws, udp_transport_data_t * tdata, DCPL_SOCKET socket)
{
	tdata = init_udp_transport_data(tdata);
	if (!tdata)
		return DPWS_ERR_EOM;
	return dc_transport_setup_channel(dpws, (transport_data_t *)tdata, socket, DC_TRANSPORT_OUTPUT);
}

int dc_udp_open_multicast_output_channel(struct dpws * dpws, udp_transport_data_t * tdata, const char * mcast_addr, uint16_t port, dcpl_ip_addr_t * out_netif_addr)
{
	struct dcpl_error error;
	DCPL_SOCKET socket = dcpl_udp_open(DCPL_GET_ADDR_FAMILY(out_netif_addr), &error);
	int ret = DPWS_OK;

	if (socket == DCPL_INVALID_SOCKET)
		return convert_dcpl_client_error(dpws, &error);
	ret = dc_udp_setup_multicast_output_channel(dpws, tdata, socket, mcast_addr, port, out_netif_addr);
	if (ret)
		dcpl_closesocket(socket, NULL);

	return ret;
}

int dc_udp_setup_multicast_output_channel(struct dpws * dpws, udp_transport_data_t * tdata, DCPL_SOCKET socket, const char * mcast_addr, uint16_t port, dcpl_ip_addr_t * out_netif_addr)
{
	tdata = init_udp_transport_data(tdata);
	if (!tdata)
		return DPWS_ERR_EOM;
	strcpy(tdata->base.peer_addr, mcast_addr);
	tdata->base.peer_port = port;
	tdata->base.itf = out_netif_addr->itf_nb;
	if (dcpl_set_multicast_netif(
			socket,
			DCPL_GET_ADDR_FAMILY(out_netif_addr),
			DCPL_GET_MULTICAST_NETIF_INDEX(out_netif_addr),
			NULL
			)
		)
		return DPWS_ERR_COULD_NOT_SELECT_MULTICAST_NETIF;

	return dc_transport_setup_channel(dpws, (transport_data_t *)tdata, socket, DC_TRANSPORT_OUTPUT);
}

static int udp_setup_channel(struct dpws * dpws, transport_data_t * tdata)
{
	struct soap * soap = dpws_dpws2soap(dpws);
	soap->omode &= ~SOAP_IO;
	soap->omode &= ~SOAP_IO_KEEPALIVE;
	soap->omode |= SOAP_ENC_XML;
	soap->omode |= SOAP_IO_BUFFER;
	return DPWS_OK;
}

static int udp_teardown_channel(struct dpws * dpws, transport_data_t * tdata)
{
	return DPWS_OK;
}

static int udp_send(struct dpws * dpws, void* transport_data, const char * buf, size_t buflen)
{
	udp_transport_data_t * tdata = (udp_transport_data_t *)transport_data;
	struct soap * soap = dpws_dpws2soap(dpws);

	DPWSMSG(SENT, buf, buflen);
	return dcpl_udp_send(tdata->base.socket, buf, buflen, soap->socket_flags,
						 tdata->base.peer_addr, tdata->base.itf, tdata->base.peer_port,
						 &tdata->base.error);
}

static int udp_recv(struct dpws * dpws, void* transport_data, char * buf, size_t buflen)
{
	udp_transport_data_t * tdata = (udp_transport_data_t *)transport_data;
	int nread = dcpl_udp_recv(tdata->base.socket, buf, buflen, dpws_dpws2soap(dpws)->socket_flags,
		                      tdata->base.peer_addr, sizeof(tdata->base.peer_addr),
				    	      &tdata->base.peer_port, DPWS_IS_MULTICAST(dpws->status) ? &tdata->base.itf : NULL,
						      &tdata->base.error);
	DPWSMSG(RECV, buf, nread);
	return nread;
}

static int udp_close(struct dpws * dpws)
{
	udp_transport_data_t * tdata = (udp_transport_data_t *)dpws->transport_data;
	if (tdata->base.status & DC_TRANSPORT_INTERNAL)
		dc_transport_close_channel(dpws);
	return SOAP_OK;
}

static int udp_response(struct dpws * dpws, void* transport_data, int status, struct media_type * mtype, size_t len)
{
	struct soap * soap = dpws_dpws2soap(dpws);
	soap->omode |= SOAP_ENC_XML;
	return DPWS_OK;
}

static int udp_recvemptyresponse(struct dpws * dpws)
{
	return DPWS_OK;
}

