/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2234 $
|                     $Date: 2009-03-31 14:56:48 +0200 (mar, 31 mar 2009) $
\*============================================================================*/

#include "dcDPWS_SubscManager.h"
#include "dcDPWS_Event.h"
#include "dcDPWS_Registry.h"
#include "dcDPWS_Memory.h"
#include "dcGSOAP_Runtime.h"
#include "dcDPWS_Utils.h"
#include "dc/dc_Plugin.h"
#include "dcCOMN_Tools.h"

/*----------------------------------- Types ----------------------------------*/

typedef DC_BOOL (*subsc_match_cbk)(struct event_subscription * w_sub, void * cbk_param);
struct subsc_browse_info {
	struct dpws* dpws;
	DA_TYPED(pepr) sinks;
	char * action;
	int ret;
};

DA_TYPED_DECL(sm,struct subscription_manager);

/*----------------------------------- Data -----------------------------------*/

dcpl_mutex_t * subsc_manager_lock = NULL;
struct dc_subsc_config sm_config = {0xFFFF, MAX_SUBSC_DURATION_DEFAULT};
DA_TYPED(sm) subscription_managers = DA_INITIALIZER(struct subscription_manager, DC_MEM_SUBSCRIPTION, 5);
uint16_t	subsc_nb = 0;	// Current number of subscriptions

// wse: built-in compiler prefix (ns table not generated)
struct Namespace subscmanager_snd_namespaces[] =
{
	{"SOAP-ENV", "http://www.w3.org/2003/05/soap-envelope", "http://www.w3.org/2003/05/soap-envelope", NULL},
	{"wsa", "http://schemas.xmlsoap.org/ws/2004/08/addressing", "http://schemas.xmlsoap.org/ws/*/addressing", NULL},
	{"wse", "http://schemas.xmlsoap.org/ws/2004/08/eventing", "http://schemas.xmlsoap.org/ws/*/eventing", NULL},
	{NULL, NULL, NULL, NULL}
};

/*------------------------- Static Functions prototypes ----------------------*/

static void action_free_cbk(int mod, const da_allocator_t * p_alloc, char **action);
static void free_subscription(struct event_subscription * wes);
static void remove_subscription(struct subscription_manager * sm, struct event_subscription * prev, struct event_subscription * current);
static DC_BOOL subscription_manager_match(struct subscription_manager *sm, short *hrefService);
static struct subscription_manager * find_subscription_manager(short hrefService);
static struct event_subscription * browse_subscriptions(struct subscription_manager * sm, subsc_match_cbk match_cbk, void * cbk_param, DC_BOOL remove);
static struct event_subscription * find_subscription(short hrefService, char * id);
static void free_subsc_manager(int mod, const da_allocator_t * p_alloc, struct subscription_manager * sm);
static DC_BOOL clean_subsc_manager(struct subscription_manager *sm, void * param);
static int clean_subscriptions();
static DC_BOOL is_rfc2396_included(DA_TYPED(str) * uri_set1, DA_TYPED(str) * uri_set2);

/*******************************************************************************
*                                 MARSHALLING                                  *
*******************************************************************************/
#define WITH_NOGLOBAL
#ifdef SOAP_FMAC3
#undef SOAP_FMAC3
#endif
#define SOAP_FMAC3 static
#include "subscmanagerC.c"
#undef WITH_NOGLOBAL

int subscManager_putelement(struct dpws *dpws, const void *ptr, const char *tag, int type)
{
	return soap_putelement(dpws_dpws2soap(dpws), ptr, tag, -1, type);
}

/*******************************************************************************
*                                    STUB                                      *
*******************************************************************************/
#define subscmanager_snd_namespaces (dpws->protocols->eventing_snd_namespaces)
#define subscmanager_namespaces (dpws->protocols->default_namespaces)
#include "subscmanagerClient.c"
#undef subscmanager_namespaces
#undef subscmanager_snd_namespaces

/*******************************************************************************
*                                  SKELETON                                    *
*******************************************************************************/
#define WITH_NOSERVEREQUEST
#define subscmanager_snd_namespaces (dpws_soap2dpws(soap)->protocols->eventing_snd_namespaces)
#define subscmanager_namespaces (dpws->protocols->default_namespaces)
#include "subscmanagerServer.c"
#undef subscmanager_namespaces
#undef subscmanager_snd_namespaces
#undef WITH_NOSERVEREQUEST

int subscmanager_serve_request(struct dpws *dpws)
{
	struct soap *soap = &dpws->soap;
	const char* action = dpws->action ? dpws->action : "";
	struct dpws_protocols * protocols[N_DPWS_VERSIONS];
	int i, size;

	size = get_protocols_versions(dpws, protocols, N_DPWS_VERSIONS);
	for (i = 0; i < size; i++) {
		dpws->protocols = protocols[i];
		dpws_set_namespaces(dpws, dpws->protocols->default_namespaces);
		soap_peek_element(soap);
		if (!strcmp(action, dpws->protocols->wse_renew_action) || (!strlen(action) && !soap_match_tag(soap, soap->tag, "wse:Renew")))
			return soap_serve___wse__RenewOp(soap);
		if (!strcmp(action, dpws->protocols->wse_getstatus_action) || (!strlen(action) && !soap_match_tag(soap, soap->tag, "wse:GetStatus")))
			return soap_serve___wse__GetStatusOp(soap);
		if (!strcmp(action, dpws->protocols->wse_unsubscribe_action) || (!strlen(action) && !soap_match_tag(soap, soap->tag, "wse:Unsubscribe")))
			return soap_serve___wse__UnsubscribeOp(soap);
	}
	return soap->error = SOAP_NO_METHOD;
}

/******************************************************************************\
 *                                                                            *
 * Service Operations (Server-side)                                           *
 *                                                                            *
\******************************************************************************/

static void action_free_cbk(int mod, const da_allocator_t * p_alloc, char **action)
{
	p_alloc->free_cbk(mod, p_alloc->param, *action);
}

static void free_subscription(struct event_subscription * wes)
{
	DC_FREE(DC_MEM_SUBSCRIPTION, wes->address);
	dpws_endpoint_ref_free(wes->notify_to);
	dpws_endpoint_ref_free(wes->end_to);
	DA_FREE(&wes->actions, action_free_cbk);
	DC_FREE(DC_MEM_SUBSCRIPTION, wes);
}

static void remove_subscription(struct subscription_manager * sm, struct event_subscription * prev, struct event_subscription * current)
{
	if (prev)
		prev->next = current->next;
	else
		sm->subscriptions = current->next;
	free_subscription(current);
	subsc_nb--;
}

static DC_BOOL subscription_manager_match(struct subscription_manager *sm, short *hrefService)
{
	return sm->hrefService == *hrefService;
}

static struct subscription_manager * find_subscription_manager(short hrefService)
{
	int i = DA_BROWSE(&subscription_managers, subscription_manager_match, &hrefService);
	return 	i == subscription_managers.nb ? NULL : DA_GET(&subscription_managers, i);
}

static struct event_subscription * browse_subscriptions(struct subscription_manager * sm, subsc_match_cbk match_cbk, void * cbk_param, DC_BOOL remove)
{
	struct event_subscription * w_sub = NULL, *prev = NULL, *next = NULL;
	uint32_t currentTime;	// to avoid a system call for every subscription

	dcpl_get_time(&currentTime, NULL);
	for (w_sub = sm->subscriptions; w_sub; w_sub = next)
	{
		next = w_sub->next;

		// check that subscription must not be discarded (expired, lazy update)
		if (w_sub->expiration < currentTime)
		{
			remove_subscription(sm, prev, w_sub);
			continue; // Do not change prev when w_sub has been deleted
		}
		if (match_cbk && match_cbk(w_sub, cbk_param))
		{
			if (remove)
				remove_subscription(sm, prev, w_sub);
			return w_sub;	// The pointer is invalid but used as return value to relay that removal occured
		}
		prev = w_sub;
	}
	return NULL;
}

DC_BOOL subsc_id_match_cbk(struct event_subscription * w_sub, char * id)
{
	return strcmp(w_sub->id, id) ? DC_FALSE : DC_TRUE;
}

static struct event_subscription * find_subscription(short hrefService, char * id)
{
	struct subscription_manager * sm = find_subscription_manager(hrefService);
	return sm ? browse_subscriptions(sm, (subsc_match_cbk)subsc_id_match_cbk, id, DC_FALSE) : NULL;
}

int wse_delete_subscription(struct dpws* dpws, char * id)
{
	struct subscription_manager * sm = find_subscription_manager(dpws->href_endpoint);
	if (sm)
		return browse_subscriptions(sm, (subsc_match_cbk)subsc_id_match_cbk, id, DC_TRUE) ? DPWS_OK : WSE_ERR_INVALID_SUBSCRIPTION_ID;
	else
		return WSE_ERR_INVALID_SUBSCRIPTION;
}

int __wse__RenewOp(struct dpws * dpws, struct _wse__Renew * renew, struct _wse__RenewResponse * response)
{
	uint32_t duration;
	int ret = DPWS_OK;

	if (renew->wse__Expires)
	{
		duration = xmlduration2duration(renew->wse__Expires);
		DC_ERROR_ASSERT(duration != 0, WSE_ERR_UNSUPPORTED_EXPIRATION_TYPE);
		DC_ERROR_ASSERT(duration > 0, WSE_ERR_INVALID_EXPIRATION_TIME);
		if (duration > sm_config.max_subsc_duration)
			duration = sm_config.max_subsc_duration;
	}
	else
		duration = sm_config.max_subsc_duration;

#ifdef WITH_WSMAN
	ret = dpws->eventing_sm->renew_subsc(dpws, dpws->subscription_id, duration);
#else
	ret = wse_renew_subsc(dpws, dpws->subscription_id, duration);
#endif

	if (ret) {
		dpws->err_detail_type = SOAP_TYPE__wse__Renew;
		dpws->err_detail_data = NULL;
		dpws->err_detail_data_elt = NULL;
	}
	else {
		DC_ERROR_ASSERT_ALLOC(response->wse__Expires = (char*)DC_MSG_MALLOC(DC_MEM_SUBSCRIPTION, dpws, DURATION_MAX_SIZE));
		duration2xmlduration(response->wse__Expires, duration); // DPWS authorizes to reply with a duration
	}
DC_FUNC_ERROR
	return ret;
}

int __wse__GetStatusOp(struct dpws * dpws, struct _wse__GetStatus * status, struct _wse__GetStatusResponse * response)
{
	uint32_t duration;
	int ret = DPWS_OK;

#ifdef WITH_WSMAN
	ret = dpws->eventing_sm->subsc_status(dpws, dpws->subscription_id, &duration);
#else
	ret = wse_subsc_status(dpws, dpws->subscription_id, &duration);
#endif

	if (ret) {
		dpws->err_detail_type = SOAP_TYPE__wse__GetStatus;
		dpws->err_detail_data = NULL;
		dpws->err_detail_data_elt = NULL;
	}
	else {
		DC_ERROR_ASSERT_ALLOC(response->wse__Expires = (char*)DC_MSG_MALLOC(DC_MEM_SUBSCRIPTION, dpws, DURATION_MAX_SIZE));
		duration2xmlduration(response->wse__Expires, duration); // DPWS authorizes to reply with a duration
	}
DC_FUNC_ERROR
	return ret;
}

int __wse__UnsubscribeOp(struct dpws* dpws, struct _wse__Unsubscribe * in, struct __wse__UnsubscribeOpResponse *out)
{
	int ret = DPWS_OK;

	if (dpws->subscription_id) {
		out->dummy = NULL;
#ifdef WITH_WSMAN
		DC_ERROR_ASSERT_CALL(dpws->eventing_sm->delete_subsc(dpws, dpws->subscription_id));
#else
		DC_ERROR_ASSERT_CALL(wse_delete_subscription(dpws, dpws->subscription_id));
#endif
	}
	else
		DC_ERROR_THROW(WSE_ERR_INVALID_SUBSCRIPTION_ID);

	return DPWS_OK;

DC_FUNC_ERROR
	dpws->err_detail_type = SOAP_TYPE__wse__Unsubscribe;
	dpws->err_detail_data = in;
	dpws->err_detail_data_elt = "wse:UnsubscribeOp";
	return ret;
}

/*----------------------------------------------------------------------------*\
 *                              CONFIGURATION API                             *
\*----------------------------------------------------------------------------*/
int set_subsc_manager_config_ptr_att(int att, const void * value, DC_BOOL reset)
{
	int ret = DPWS_OK;
	uint32_t t;
	switch (att)
	{
		case DPWS_INT_MAX_SUBSC_NB:
			sm_config.max_subsc_nb = *(uint16_t *)value;
			break;
		case DPWS_STR_MAX_SUBSC_DURATION:
			if (*(char *)value != 'P')
				return DPWS_ERR_INVALID_PARAMETER;
			t = xmlduration2duration((char *)value);
			if (t == -1)
				return DPWS_ERR_INVALID_PARAMETER;
			else
				sm_config.max_subsc_duration = t;
			break;
		case DPWS_INT_MAX_SUBSC_DURATION:
			sm_config.max_subsc_duration = *(uint32_t *)value;
			break;
		default:
			return DPWS_ERR_NO_SUCH_ATT_ON_OBJECT;
	}
	return ret;
}

unsigned long get_subsc_manager_config_int_att(int att, int index)
{
	switch (att)
	{
		case DPWS_INT_MAX_SUBSC_NB:
			return sm_config.max_subsc_nb;
		case DPWS_INT_MAX_SUBSC_DURATION:
			return sm_config.max_subsc_duration;
		default:
			return -1;
	}
}

int get_subsc_manager_config_att_count(int att)
{
	return (att == DPWS_INT_MAX_SUBSC_NB || att == DPWS_STR_MAX_SUBSC_DURATION) ? 1 : 0;
}

/*******************************************************************************
*                           SUBSCRIPTION MANAGEMENT                            *
*******************************************************************************/
int init_subsc_managers()
{
	subscription_managers.allocator = p_default_allocator;	// could not be initialized as a global variable
	return ((subsc_manager_lock = (dcpl_mutex_t*)dcpl_mutex_init()) == NULL)
			? DPWS_ERR_CREATING_MUTEX : DPWS_OK;
}

static void free_subsc_manager(int mod, const da_allocator_t * p_alloc, struct subscription_manager * sm)
{
	struct event_subscription * pwes;
	while (sm->subscriptions) {
		pwes = sm->subscriptions;
		sm->subscriptions = sm->subscriptions->next;
		free_subscription(pwes);
	}
}

void uninit_subsc_managers()
{
	// free remaining structures
	DA_FREE(&subscription_managers, free_subsc_manager);
	if (subsc_manager_lock != NULL)
	{
		dcpl_mutex_delete(subsc_manager_lock);
	  	subsc_manager_lock = NULL;
	}
}

static DC_BOOL clean_subsc_manager(struct subscription_manager *sm, void * param)
{
	browse_subscriptions(sm, NULL, NULL, DC_FALSE);
	return DC_FALSE;
}

static int clean_subscriptions()
{
	DA_BROWSE(&subscription_managers, clean_subsc_manager, NULL);
	return DC_FALSE;
}

static DC_BOOL is_rfc2396_included(DA_TYPED(str) * uri_set1, DA_TYPED(str) * uri_set2)
{
    return DA_IS_INCLUDED(uri_set1, uri_set2, (da_cmp_cbk)rfc2396_match);
}

int wse_create_subsc_manager(struct dpws * dpws, uint32_t duration, char ** filter, int filter_size, struct wsa_endpoint_ref * notify_to, struct wsa_endpoint_ref * end_to, char ** uuid)
{
	struct subscription_manager * sm = NULL;
	struct event_subscription * subscription, * previous = NULL;
	DA_TYPED(str) filter_da;	// original array was a dyn_array (passed thru a public API)
	uint32_t t;
	int ret = DPWS_OK;

	// Some compiler don't like initializers with function parameters
	filter_da.tab = filter;
	filter_da.nb = filter_size;
	filter_da.size = filter_size + 1;
	filter_da.f_size = sizeof(char *);
	filter_da.inc = 1;

	dcpl_mutex_lock(subsc_manager_lock);

	// find the subscription manager for the service
	sm = find_subscription_manager(dpws->href_endpoint);
	if (!sm) {
		DC_ERROR_ASSERT_ALLOC(sm = DA_ADD(&subscription_managers));
		sm->hrefService = dpws->href_endpoint;
		sm->subscriptions = NULL;
	}

	// create or overwrite the subscription
	for(subscription = sm->subscriptions; subscription; subscription = subscription->next) {
		if (!strcmp(subscription->notify_to->address, notify_to->address)) {
			// verify filters are equals
			if ((filter_size <= 0 && subscription->actions.nb == 0) || (filter_size > 0
				&& is_rfc2396_included(&filter_da, &subscription->actions)
				&& is_rfc2396_included(&subscription->actions, &filter_da)))
				break;
			previous = subscription;
		}
	}

	// Subscription creation
	if (!subscription) {
		if ((subsc_nb + 1) > sm_config.max_subsc_nb)	// maximum allowed reached
			clean_subscriptions();
		DC_ERROR_ASSERT(subsc_nb < sm_config.max_subsc_nb, DPWS_ERR_EOM);
		DC_ERROR_ASSERT_ALLOC(subscription = (struct event_subscription*)DC_MALLOC(DC_MEM_SUBSCRIPTION, sizeof(struct event_subscription)));
		subsc_nb ++;
		subscription->next = sm->subscriptions;
		sm->subscriptions = subscription;
	}

	dpws_schemed_uuid_create(subscription->id);
	DC_ERROR_ASSERT_ALLOC(subscription->address = DC_STRDUP(DC_MEM_SUBSCRIPTION, dpws->to));
	subscription->notify_to = dpws_endpoint_ref_dup(notify_to);
	DA_INIT(char *, &subscription->actions, DC_MEM_SUBSCRIPTION, p_default_allocator, 1);
	if (filter_size > 0)
		DC_ERROR_ASSERT_ALLOC(DA_STRDUP(&subscription->actions, &filter_da));

	DC_ERROR_ASSERT_ALLOC((subscription->end_to = dpws_endpoint_ref_dup(end_to)) || !end_to);
	dcpl_get_time(&t, NULL);
	subscription->expiration = t + duration;

	*uuid = subscription->id;	// NOTE: memory risk if the registration is removed by a concurrent thread. Should not occur since registration is event sink private (except odd design)

DC_FUNC_ERROR
	if (ret && sm && subscription)
		remove_subscription(sm, previous, subscription);
	dcpl_mutex_unlock(subsc_manager_lock);
	return ret;
}

void delete_subsc_manager(short hrefService)
{
	int i;

	dcpl_mutex_lock(subsc_manager_lock);
	i = DA_BROWSE(&subscription_managers, subscription_manager_match, &hrefService);
	if (i < subscription_managers.nb)
		DA_REMOVE(&subscription_managers, i, free_subsc_manager);
	dcpl_mutex_unlock(subsc_manager_lock);
}

int clone_subsc_manager(short hrefService, short hrefClone)
{
	struct subscription_manager * sm = NULL, *clone = NULL;
	struct event_subscription * subscription, *cloneSubsc;
	int ret = DPWS_OK;

	dcpl_mutex_lock(subsc_manager_lock);
	sm = find_subscription_manager(hrefService);
	DC_ERROR_ASSERT(sm, DPWS_ERR_NO_HANDLE_FOUND);

	DC_ERROR_ASSERT_ALLOC(clone = DA_ADD(&subscription_managers));
	clone->hrefService = hrefClone;

	for(subscription = sm->subscriptions; subscription; subscription = subscription->next)
	{
		DC_ERROR_ASSERT_ALLOC(cloneSubsc = (struct event_subscription*)DC_MALLOC(DC_MEM_SUBSCRIPTION, sizeof(struct event_subscription)));
		memcpy(cloneSubsc->id, subscription->id, SCHEME_UUID_STRING_SIZE);	// NOTE: prefix stored to make serialization more efficient
		cloneSubsc->notify_to = dpws_endpoint_ref_dup(subscription->notify_to);
		cloneSubsc->end_to = dpws_endpoint_ref_dup(subscription->end_to);
		DA_INIT(char *, &cloneSubsc->actions, DC_MEM_SUBSCRIPTION, p_default_allocator, 2);
		DC_ERROR_ASSERT_ALLOC(DA_STRDUP(&cloneSubsc->actions, &subscription->actions));
		cloneSubsc->expiration = subscription->expiration;	// seconds since 00:00:00 GMT, January 1, 1970
		// insert
		cloneSubsc->next = clone->subscriptions;
		clone->subscriptions = cloneSubsc;
	}

DC_FUNC_ERROR
	if (ret && clone)
		free_subsc_manager(DC_MEM_SUBSCRIPTION, NULL, clone);
	dcpl_mutex_unlock(subsc_manager_lock);
	return ret;
}

int wse_renew_subsc(struct dpws * dpws, char * subsc_id, uint32_t duration)
{
	int ret = DPWS_OK;
	struct event_subscription * subscription = NULL;
	uint32_t t;

	dcpl_mutex_lock(subsc_manager_lock);
	if (subsc_id)
		subscription = find_subscription(dpws->href_endpoint, subsc_id);

	if (!subscription) {
		ret = WSE_ERR_UNABLE_TO_RENEW;
		goto exit;
	}
	dcpl_get_time(&t, NULL);
	subscription->expiration = t + duration;

exit:
	dcpl_mutex_unlock(subsc_manager_lock);
	return ret;
}

int wse_subsc_status(struct dpws * dpws, char * subsc_id, uint32_t * duration)
{
	int ret = DPWS_OK;
	struct event_subscription * subscription = NULL;
	uint32_t t;

	dcpl_mutex_lock(subsc_manager_lock);
	if (subsc_id)
		subscription = find_subscription(dpws->href_endpoint, subsc_id);

	if (!subscription) {
		ret = WSE_ERR_INVALID_SUBSCRIPTION_ID;
		goto exit;
	}
	dcpl_get_time(&t, NULL);
	*duration = subscription->expiration - t;

exit:
	dcpl_mutex_unlock(subsc_manager_lock);
	return ret;
}

DC_BOOL send_eventing_end_msg(short * hrefService, struct end_info *eventEndInfo)
{
	struct event_subscription * subscription = NULL;
	struct subscription_manager * sm = find_subscription_manager(*hrefService);

	if (sm) {
		while (sm->subscriptions) {
			subscription = sm->subscriptions;
			if (subscription->end_to) {
				eventEndInfo->pBody->wse__SubscriptionManager->address = subscription->address;
				eventEndInfo->pBody->wse__SubscriptionManager->subscription_id = subscription->id;
				dpws_send___wse__SubscriptionEnd(eventEndInfo->dpws, subscription->end_to, eventEndInfo->pBody);	// NOTE: continue even if an error encountered
				soap_closesock(dpws_dpws2soap(eventEndInfo->dpws));	// one-way does not close the socket
				dpws_end(eventEndInfo->dpws);
			}
			sm->subscriptions = subscription->next;
			free_subscription(subscription);
		}
	}
	return DC_FALSE;
}

// Access by operation
DC_BOOL subsc_action_cbk(struct event_subscription * subscription, struct subsc_browse_info * b_info)
{
	char ** p_action;
	struct wsa_endpoint_ref** ppEpr;

	if (subscription->actions.tab) {	// check action against filter
		for (p_action = (char**)subscription->actions.tab; *p_action; p_action++) {
			// collect endpoint
			if (rfc2396_match(*p_action, b_info->action)) {
				if (!(ppEpr = DA_ADD(&b_info->sinks))
						||!(*ppEpr = endpoint_ref_dup(b_info->dpws, subscription->notify_to)))
				{
					b_info->ret = DPWS_ERR_EOM;
					return DC_TRUE;
				}
				break;
			}
		}
	}
	else {	// no filter
		if (!(ppEpr = DA_ADD(&b_info->sinks))
				||!(*ppEpr = endpoint_ref_dup(b_info->dpws, subscription->notify_to)))
		{
			b_info->ret = DPWS_ERR_EOM;
			return DC_TRUE;
		}
	}
	return DC_FALSE;
}

struct wsa_endpoint_ref** dpws_get_subscriptors(struct dpws* dpws, short event_source, char* action)
{
	struct subscription_manager * sm;
	da_allocator_t daa;
	struct subsc_browse_info b_info = {
		NULL,
		DA_INITIALIZER(struct wsa_endpoint_ref *, DC_MEM_SUBSCRIPTION, EVENT_ALLOC_INCREMENT),
		NULL,
		DPWS_OK
	};

	sm = find_subscription_manager(event_source);
	if (sm) {
		b_info.dpws = dpws;	// for old compilers...
		b_info.action = action;	// for old compilers...
		b_info.sinks.allocator = soap_get_da_allocator(dpws_dpws2soap(dpws), &daa);	// for old compilers...
		browse_subscriptions(sm, (subsc_match_cbk)subsc_action_cbk, &b_info, DC_FALSE);
	}

	dpws->err = b_info.ret;
	return dpws->err ? NULL : b_info.sinks.tab;
}


int dpws_remove_subscriber(short event_source, struct wsa_endpoint_ref * notify_to)
{
	struct subscription_manager * sm;
	int ret = 0;

	DC_CHECK_PARAM(event_source >= 0 && notify_to);

	sm = find_subscription_manager(event_source);
	if (sm) {
		struct event_subscription * subscription, *previous = NULL;
		for (subscription = sm->subscriptions; subscription; subscription = subscription->next) {
			if (!strcmp(subscription->notify_to->address, notify_to->address)) {
				if (previous) {
					previous->next = subscription->next;
				} else {
					sm->subscriptions = subscription->next;
				}
				free_subscription(subscription);
				ret = 1;
			}
			previous = subscription;
		}
	}
	return ret;
}

