/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2258 $
|                     $Date: 2009-04-09 14:01:57 +0200 (jeu, 09 avr 2009) $
\*============================================================================*/

/******************************************************************************\
 *                                  SERVER DATA                               *
\******************************************************************************/
#include "dcDPWS_Dpws.h"
#include "dcDPWS_Registry.h"
#include "dcDPWS_Reactor.h"
#include "dcDPWS_Network.h"
#include "dcDPWS_Discovery.h"
#include "dcDPWS_Configuration.h"
#include "dcDPWS_Event.h"
#include "dcDPWS_SubscManager.h"
#include "dcCOMN_DynArray.h"
#include "dcDCPL_Os.h"
#include "dcCOMN_Uuid.h"
#include "dcGSOAP_Runtime.h"
#include "dcDPWS_Memory.h"
#include "dcDPWS_Utils.h"
#include "dc/dc_Plugin.h"

// NOTE: PERF could use soap_unlink to avoid free + malloc when adding dynamic
// data to registry. But it would prevent to implement a more efficient memory
// allocation for gSOAP (global free)

/*----------------------------------- Data -----------------------------------*/

struct dpws_reg_config registry_cfg = {0xFFFFFFFF, NULL, DC_FALSE, NULL, DC_FALSE, DC_TRUE, HTTP_BACKLOG_DEFAULT};

short hidden_device_href = -1;

struct transport HTTP_TRANSPORT = {"http", "http:"};

struct network_endpoint http_endpoint = {&HTTP_TRANSPORT, HTTP_PORT_DEFAULT, NULL};	// NOTE: only one network endpoint in the 2.0 version of the tk

handlePool_t registryPool;

dcpl_mutex_t * invocation_lock = NULL, * reg_conf_lock = NULL, * handle_lock = NULL;
static int prefix_counter = 0;

/*-------------------------------------- Types -------------------------------*/

DA_TYPED_DECL(ns,struct Namespace);

struct ns_info {
	struct dpws * dpws;
	DA_TYPED(ns) *table;
};

struct clone_info {
	DA_TYPED(sp) * ports;
	struct device * device;
};

typedef DC_BOOL (*filter_cbk_t)(void *object, void * param);

struct gather_handles_info {
	short* hrefs;
	int max_len;
	int nb;
	filter_cbk_t filter;
	void * filter_param;
};

/*------------------------- Static Functions prototypes ----------------------*/

static void wsdl_info_free_cbk(int mod, const da_allocator_t * p_alloc, struct wsdl_info *pwsdl);
static void prefixed_qname_free_cbk(int mod, const da_allocator_t * p_alloc, struct prefixed_qname *pqname);
static void localizedStringFree(int mod, const da_allocator_t * p_alloc, struct localized_string * pLs);
static void scope_free_cbk(int mod, const da_allocator_t * p_alloc, char **scope);
static DC_BOOL clone_localized_string(int mod, const da_allocator_t * allocator, struct localized_string * dest, struct localized_string * original, void * dummy);
static void freeServiceStruct(struct service_endpoint * service);
static struct service_endpoint * create_hosted_service(struct device * device, short hrefClass);
static short create_service_port(char * context_path);
static DC_BOOL hrefRelease(short * href, void * dummy);
static void freeDeviceStruct(struct device * device);
static int freeDevice(handle_s *handleInfo);
static int freeService(handle_s *handleInfo);
static void freeDeviceModelStruct(struct device_model * deviceModel);
static int freeDeviceModel(handle_s *handleInfo);
static void freeServiceClassStruct(struct service_class * servClass);
static int freeServiceClass(handle_s *handleInfo);
static void freeServicePortStruct(struct service_port * servPort);
static int freeServicePort(handle_s *handleInfo);
static short create_service_class();
static short create_custom_device(short localID, short hrefModel);
static int releaseHandleHook(handle_s *handleInfo);
static DC_BOOL device_model_equals(short type, struct device_model * devModel, char * id);
static DC_BOOL service_class_equals(short type, struct service_class * servClass, char * id);
static DC_BOOL device_equals(short type, struct device * device, char * uuid);
static DA_TYPED(href) * find_service_endpoint_ports(short href);
static int bind_service(short hrefService, short hrefServicePort);
static int check_device(struct device *device);
static invocation_list_t * checkout_invocation_list(invocation_list_t * list);
static DC_BOOL add_ns_entry(struct prefixed_qname *pPQName, struct ns_info *ns_info);
static DC_BOOL add_service_ns(short * hrefService, struct ns_info *ns_info);
static int add_implicit_ns(struct dpws * dpws, DA_TYPED(ns) * ns_tab);
static void release_invocation_list(invocation_list_t * list);
static int send_hello_cbk(dcpl_ip_addr_t * netif_addr, struct device * device);
static DC_BOOL advertize_enabled_device(short type, struct device * device, int * ret);
static int send_bye_cbk(dcpl_ip_addr_t * netif_addr, struct device * device);
static int shutdown_device_callback(struct dpws * dpws, struct reactor_item * item, void *callback_data);
static DC_BOOL shutdown_enabled_device(short type, struct device * device, void * dummy);
static DC_BOOL find_service_hook(short * hrefService, char * service_id);
static DC_BOOL find_service_hook_by_class(short * hrefService, short * hrefServiceClass);
static int port_context_path_cmp(struct service_port **serv_port, const char * ref);
static DC_BOOL gather_service_ports(short * hrefPort, invocation_list_t * invoc_list);
static DC_BOOL gather_device_ports(short * hrefService, invocation_list_t * invoc_list);
static DC_BOOL build_clone_list(struct service_port ** ppServicePort, struct clone_info * cloneInfo);;
static invocation_list_t * new_invocation_list();
static int substitute_invocation_list(struct device * to_disable, struct device * to_enable);
static int enable_device(short deviceHandleRef);
static DC_BOOL clone_service_port(int mod, struct da_allocator * allocator, short * dest, short * original, struct service_endpoint * service);
static struct service_endpoint * _clone_hosted_service(short * original, struct device * device);
static DC_BOOL clone_hosted_service(int mod, struct da_allocator * allocator, short * dest, short * original, struct device * device);
static short clone_device(short href);
static int replace_device(short href_original, short href_clone);
static DC_BOOL find_href_hook(short * href1, short * href2);
static DC_BOOL gather_handles(handle_s * h, struct gather_handles_info * gh_info);
static short create_random_service_port();
static short create_device_model();
static short find_handle(short typeFilter, handle_cbk callback, char * id);
static short find_service(short deviceHandleRef, da_browse_cbk cbk, void * cbk_param);
static DC_BOOL checkout_cbk(short * p_href, void * param);
static int get_handles(short* hrefs, int* len, short h_type, filter_cbk_t filter_cbk, void * filter_param);
static DC_BOOL gather_visible_devices_cbk(struct device * device, void * dummy);
static DC_BOOL gather_public_services_class_cbk(struct service_class * s_class, void * dummy);
static DC_BOOL gather_services_by_class_filter_cbk(struct service_endpoint * service, short * href_class);

/*----------------------------------------------------------------------------*\
 *                               PRIVATE TOOLS                                *
\*----------------------------------------------------------------------------*/
static void wsdl_info_free_cbk(int mod, const da_allocator_t * p_alloc, struct wsdl_info *pwsdl)
{
	p_alloc->free_cbk(mod, p_alloc->param, pwsdl->target_ns);
	p_alloc->free_cbk(mod, p_alloc->param, pwsdl->location);
}

static void prefixed_qname_free_cbk(int mod, const da_allocator_t * p_alloc, struct prefixed_qname *pqname)
{
	p_alloc->free_cbk(mod, p_alloc->param, pqname->prefix);
	p_alloc->free_cbk(mod, p_alloc->param, pqname->qname.ns);
	p_alloc->free_cbk(mod, p_alloc->param, pqname->qname.lname);
}

static void localizedStringFree(int mod, const da_allocator_t * p_alloc, struct localized_string * pLs)
{
	p_alloc->free_cbk(mod, p_alloc->param, pLs->lang);
	p_alloc->free_cbk(mod, p_alloc->param, pLs->s);
}

static void scope_free_cbk(int mod, const da_allocator_t * p_alloc, char **scope)
{
	p_alloc->free_cbk(mod, p_alloc->param, *scope);
}

static DC_BOOL clone_localized_string(int mod, const da_allocator_t * allocator, struct localized_string * dest, struct localized_string * original, void * dummy)
{
	if ((!(dest->lang = DC_STRDUP(mod, original->lang)) && original->lang)
			|| (!(dest->s = DC_STRDUP(mod, original->s)) && original->s))
		return DC_FALSE;
	return DC_TRUE;
}

/*----------------------------------------------------------------------------*\
 *                              CONFIGURATION API                             *
\*----------------------------------------------------------------------------*/

int set_service_endpoint_ptr_att(struct service_endpoint * pService, int att, const void * value, DC_BOOL reset)
{
	switch (att)
	{
		case DPWS_STR_SERVICE_ID:
			DC_FREE(DC_MEM_REGISTRY, pService->service_id);
			if (!(pService->service_id = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)) && value)
				return DPWS_ERR_EOM;
			break;
		case DPWS_STR_HARDWARE_ID:
			DC_FREE(DC_MEM_REGISTRY, pService->hardware_id);
			if (!(pService->hardware_id = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)) && value)
				return DPWS_ERR_EOM;
			break;
		case DPWS_STR_COMPATIBLE_ID:
			DC_FREE(DC_MEM_REGISTRY, pService->compatible_id);
			if (!(pService->compatible_id = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)) && value)
				return DPWS_ERR_EOM;
			break;
		case DPWS_PTR_USER_DATA:
			pService->user_data = value;
			break;
		case DPWS_PTR_HANDLING_FUNCTION: case DPWS_PTR_FAULT_FUNCTION: case DPWS_PTR_CALLBACK_EVENT_END:	// NOTE: Not ideal, a service class may be shared if it is a Hosted Service endpoint...
			set_service_class_ptr_att((struct service_class *)getObject(&registryPool, pService->hrefServiceClass), att, value, reset);
			break;
		case DPWS_INT_CLASS_HREF:
			return DPWS_ERR_IMMUTABLE_USED_OBJECT;
		default:
			return DPWS_ERR_NO_SUCH_ATT_ON_OBJECT;
	}
	return DPWS_OK;
}

int set_service_port_ptr_att(struct service_port * pServicePort, int att, const void * value, DC_BOOL reset)
{
	switch (att)
	{
		case DPWS_STR_ADDRESS:
			DC_FREE(DC_MEM_REGISTRY, pServicePort->context_path);
			if (!(pServicePort->context_path = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)) && value)
				return DPWS_ERR_EOM;
			break;
		default:
			return DPWS_ERR_NO_SUCH_ATT_ON_OBJECT;
	}
	return DPWS_OK;
}

int set_service_class_ptr_att(struct service_class * pServiceClass, int att, const void * value, DC_BOOL reset)
{
	struct prefixed_qname * pqName, * pqnParam, pqn;
	struct wsdl_info *wsdlInfo;
	char buffer[QNAME_PREFIX_GEN_LEN];
	DA_TYPED(disp) *handlers;
	dispatch_cbk *pHandler = NULL;
	int i;

	switch (att)
	{
		case DPWS_STR_ID:
			DC_FREE(DC_MEM_REGISTRY, pServiceClass->id);
			if (!(pServiceClass->id = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)) && value)
				return DPWS_ERR_EOM;
			break;
		case DPWS_PTR_TYPE:
		case DPWS_PTR_PREFIXED_TYPE:
			if (!((struct qname *)value)->ns)
				return DPWS_ERR_INVALID_NS;
			for (i = 0; i < N_DPWS_VERSIONS; i++) {
				if (QNAME_EQUALS(dpws_protocols[i]->wdp_uri, WDP_DEVICE_TYPE, ((struct qname *)value)->ns, ((struct qname *)value)->lname))
					return DPWS_OK;
			}
			if (reset)
				DA_EMPTY(&pServiceClass->types, prefixed_qname_free_cbk);
			if (!(pqName = DA_ADD(&pServiceClass->types)))
				return DPWS_ERR_EOM;
			if (att == DPWS_PTR_PREFIXED_TYPE)
				pqnParam = (struct prefixed_qname *)value;
			else {
				pqnParam = (prefixed_qname_t*)memcpy(&pqn, value, sizeof(struct qname));
				sprintf(buffer, "%s%d", QNAME_PREFIX_GEN_PREFIX, prefix_counter++);	// NOTE: counter is global since a prefix must be unique for a device but a service class can be shared
				pqnParam->prefix = buffer;
			}
			if ((!(pqName->prefix = DC_STRDUP(DC_MEM_REGISTRY, pqnParam->prefix)) && pqnParam->prefix)
				|| (!(pqName->qname.ns = DC_STRDUP(DC_MEM_REGISTRY, pqnParam->qname.ns)) && pqnParam->qname.ns)
				|| (!(pqName->qname.lname = DC_STRDUP(DC_MEM_REGISTRY, pqnParam->qname.lname)) && pqnParam->qname.lname))
			{
				prefixed_qname_free_cbk(DC_MEM_REGISTRY, p_default_allocator, pqName);
				pServiceClass->types.nb--;
				return DPWS_ERR_EOM;
			}
			break;
		case DPWS_PTR_WSDL:
			if (reset)
				DA_EMPTY(&pServiceClass->wsdl_files, wsdl_info_free_cbk);
			if (!(wsdlInfo = DA_ADD(&pServiceClass->wsdl_files)))
				return DPWS_ERR_EOM;
			if ((!(wsdlInfo->target_ns = DC_STRDUP(DC_MEM_REGISTRY, ((struct wsdl_info *)value)->target_ns)) && (((struct wsdl_info *)value)->target_ns))
					|| (!(wsdlInfo->location = DC_STRDUP(DC_MEM_REGISTRY, ((struct wsdl_info *)value)->location)) && (((struct wsdl_info *)value)->location)))
			{
				wsdl_info_free_cbk(DC_MEM_REGISTRY, p_default_allocator, wsdlInfo);
				pServiceClass->wsdl_files.nb--;
				return DPWS_ERR_EOM;
			}
			break;
		case DPWS_PTR_HANDLING_FUNCTION:
			handlers = &pServiceClass->funcs;
			if (reset)
				DA_EMPTY(handlers, NULL);
			/* Check for handler doubles */
			if (handlers->tab)
				for(pHandler = (int (**)(struct dpws *))handlers->tab; *pHandler && *pHandler != value; pHandler++);
			if (pHandler && *pHandler)
				return DPWS_ERR_CONFLICTING_OPERATION;
			else
			{
				if (!(pHandler = DA_ADD(handlers)))
					return DPWS_ERR_EOM;
				*pHandler = (dispatch_cbk)value;
			}
			break;
		case DPWS_PTR_FAULT_FUNCTION:
			pServiceClass->fault_func = (dispatch_cbk)value;
			break;
		case DPWS_PTR_CALLBACK_EVENT_END:
			pServiceClass->event_end_hook = (event_end_cbk)value;
			set_service_class_ptr_att(pServiceClass, DPWS_PTR_HANDLING_FUNCTION, (const void *)&eventing_handle_event, DC_FALSE);
			break;
		case DPWS_PTR_USER_DATA:
			pServiceClass->user_data = value;
			break;
		default:
			return DPWS_ERR_NO_SUCH_ATT_ON_OBJECT;
	}
	return DPWS_OK;
}

int set_device_model_ptr_att(struct device_model * pDeviceModel, int att, const void * value, DC_BOOL reset)
{
	struct localized_string * pLs;
	int ret = DPWS_OK;
	switch (att)
	{
		case DPWS_STR_ID:
			DC_FREE(DC_MEM_REGISTRY, pDeviceModel->id);
			if (!(pDeviceModel->id = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)) && value)
				return DPWS_ERR_EOM;
			break;
		case DPWS_PTR_MODEL_NAME:
			if (reset)
				DA_EMPTY(&pDeviceModel->model_names, localizedStringFree);
			if (!(pLs = DA_ADD(&pDeviceModel->model_names)))
				return DPWS_ERR_EOM;
			if (!clone_localized_string(DC_MEM_REGISTRY, p_default_allocator, pLs, (struct localized_string *)value, NULL)) {
				pDeviceModel->model_names.nb--;
				ret = DPWS_ERR_EOM;
			}
			break;
		case DPWS_STR_MODEL_NUMBER:
			DC_FREE(DC_MEM_REGISTRY, pDeviceModel->model_number);
			if (!(pDeviceModel->model_number = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)) && value)
				return DPWS_ERR_EOM;
			break;
		case DPWS_STR_MODEL_URL:
			DC_FREE(DC_MEM_REGISTRY, pDeviceModel->model_url);
			if (!(pDeviceModel->model_url = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)) && value)
				return DPWS_ERR_EOM;
			break;
		case DPWS_STR_PRESENTATION_URL:
			DC_FREE(DC_MEM_REGISTRY, pDeviceModel->presentation_url);
			if (!(pDeviceModel->presentation_url = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)) && value)
				return DPWS_ERR_EOM;
			break;
		case DPWS_PTR_MANUFACTURER:
			if (reset)
				DA_EMPTY(&pDeviceModel->manufacturers, localizedStringFree);
			if (!(pLs = DA_ADD(&pDeviceModel->manufacturers)))
				return DPWS_ERR_EOM;
			if (!clone_localized_string(DC_MEM_REGISTRY, p_default_allocator, pLs, (struct localized_string *)value, NULL)) {
				pDeviceModel->manufacturers.nb--;
				return DPWS_ERR_EOM;
			}
			break;
		case DPWS_STR_MANUFACTURER_URL:
			DC_FREE(DC_MEM_REGISTRY, pDeviceModel->manufacturer_url);
			if (!(pDeviceModel->manufacturer_url = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)) && value)
				return DPWS_ERR_EOM;
			break;
		default:
			return DPWS_ERR_NO_SUCH_ATT_ON_OBJECT;
	}
	return DPWS_OK;
}

int set_device_ptr_att(struct device * pDevice, int att, const void * value, DC_BOOL reset)
{
	struct localized_string * pLs;
	switch (att)
	{
		case DPWS_STR_DEVICE_ID:
			DC_FREE(DC_MEM_REGISTRY, pDevice->uuid);
			if (value && is_physical_address((char *)value)) {
				if (!(pDevice->uuid = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)))
					return DPWS_ERR_EOM;
			} else {
				if (!(pDevice->uuid = DC_MALLOC(DC_MEM_REGISTRY, SCHEME_UUID_STRING_SIZE)))
					return DPWS_ERR_EOM;
				if (!format_scheme_uuid(pDevice->uuid, (char *)value))
					return DPWS_ERR_INVALID_PARAMETER;
			}
			break;
		case DPWS_STR_ADDRESS:
			DC_FREE(DC_MEM_REGISTRY, pDevice->context_path);
			if (!(pDevice->context_path = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)) && value)
				return DPWS_ERR_EOM;
			break;
		case DPWS_STR_SCOPE:
			if (value && strlen((char *)value) > MAX_URI_SIZE)
				return DPWS_ERR_INVALID_PARAMETER;
			if (reset)
				DA_EMPTY(&pDevice->scopes, scope_free_cbk);
			if (value) {
				char ** pstr;
				if ((pstr = DA_ADD(&pDevice->scopes)))
				{
					if (!(*pstr = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)) && value)
					{
						pDevice->scopes.nb--;
						return DPWS_ERR_EOM;
					}
				}
				else
					return DPWS_ERR_EOM;
			}
			break;
		case DPWS_PTR_MODEL_NAME:
			if (reset)
				DA_EMPTY(&pDevice->model_names, localizedStringFree);
			if ((pLs = DA_ADD(&pDevice->model_names))) {
				if (!clone_localized_string(DC_MEM_REGISTRY, p_default_allocator, pLs, (struct localized_string *)value, NULL)) {
					pDevice->model_names.nb--;
					return DPWS_ERR_EOM;
				}
			}
			else
				return DPWS_ERR_EOM;
			break;
		case DPWS_STR_MODEL_NUMBER:
			DC_FREE(DC_MEM_REGISTRY, pDevice->model_number);
			if (!(pDevice->model_number = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)) && value)
				return DPWS_ERR_EOM;
			break;
		case DPWS_STR_MODEL_URL:
			DC_FREE(DC_MEM_REGISTRY, pDevice->model_url);
			if (!(pDevice->model_url = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)) && value)
				return DPWS_ERR_EOM;
			break;
		case DPWS_STR_PRESENTATION_URL:
			DC_FREE(DC_MEM_REGISTRY, pDevice->presentation_url);
			if (!(pDevice->presentation_url = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)) && value)
				return DPWS_ERR_EOM;
			break;
		case DPWS_PTR_MANUFACTURER:
			if (reset)
				DA_EMPTY(&pDevice->manufacturers, localizedStringFree);
			if ((pLs = DA_ADD(&pDevice->manufacturers))) {
				if (!clone_localized_string(DC_MEM_REGISTRY, p_default_allocator, pLs, (struct localized_string *)value, NULL)) {
					pDevice->manufacturers.nb--;
					return DPWS_ERR_EOM;
				}
			}
			else
				return DPWS_ERR_EOM;
			break;
		case DPWS_STR_MANUFACTURER_URL:
			DC_FREE(DC_MEM_REGISTRY, pDevice->manufacturer_url);
			if (!(pDevice->manufacturer_url = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)) && value)
				return DPWS_ERR_EOM;
			break;
		case DPWS_PTR_FRIENDLY_NAME:
			if (reset)
				DA_EMPTY(&pDevice->friendly_names, localizedStringFree);
			if ((pLs = DA_ADD(&pDevice->friendly_names))) {
				if (!clone_localized_string(DC_MEM_REGISTRY, p_default_allocator, pLs, (struct localized_string *)value, NULL)) {
					pDevice->friendly_names.nb--;
					return DPWS_ERR_EOM;
				}
			}
			else
				return DPWS_ERR_EOM;
			break;
		case DPWS_STR_FIRMWARE_VERSION:
			if (!(pDevice->firmware_version = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)) && value)
				return DPWS_ERR_EOM;
			break;
		case DPWS_STR_SERIAL_NUMBER:
			if (!(pDevice->serial_number = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)) && value)
				return DPWS_ERR_EOM;
			break;
		case DPWS_STR_DEVICE_CATEGORY:
			if (!(pDevice->device_category = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)) && value)
				return DPWS_ERR_EOM;
			break;
		case DPWS_STR_MODEL_ID:
			if (!(pDevice->model_id = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)) && value)
				return DPWS_ERR_EOM;
			break;
		case DPWS_INT_METADATA_VERSION:
			pDevice->metadata_version = *(unsigned short *)value;
			break;
		case DPWS_PTR_USER_DATA:
			pDevice->user_data = value;
			break;
		case DPWS_INT_CLASS_HREF:
			return DPWS_ERR_IMMUTABLE_USED_OBJECT;
		case DPWS_PTR_TYPE:
		case DPWS_PTR_PREFIXED_TYPE:
			return set_service_class_ptr_att((struct service_class *)getObject(&registryPool, pDevice->super->hrefServiceClass), att, value, reset);
		default:
			set_service_endpoint_ptr_att(pDevice->super, att, value, reset);
	}
	return DPWS_OK;
}

int set_registry_config_ptr_att(int att, const void * value, DC_BOOL reset)
{
	switch (att)
	{
		case DPWS_INT_BOOT_SEQ:
			registry_cfg.boot_seq = *(unsigned long *)value;
			break;
		case DPWS_PTR_CALLBACK_DEVICE_METADATA:
			if (reset)
				registry_cfg.dev_metadata_hook = (metadata_cbk)value;
			else
				return DPWS_ERR_MONOVALUED_ATT_TYPE;
			break;
		case DPWS_BOOL_TCP_LISTEN:
			registry_cfg.tcp_listen = *(DC_BOOL *)value;
			break;
		case DPWS_INT_HTTP_BACKLOG:
			registry_cfg.tcp_backlog = *(unsigned long *)value;
			break;
		case DPWS_INT_HTTP_PORT:
			http_endpoint.port = (uint16_t)*(unsigned long *)value;
			break;
		case DPWS_STR_PREFERRED_LANG:
			DC_FREE(DC_MEM_REGISTRY, registry_cfg.preferred_lang);
			if (!(registry_cfg.preferred_lang = DC_STRDUP(DC_MEM_REGISTRY, (char *)value)) && value)
				return DPWS_ERR_EOM;
			break;
		case DPWS_BOOL_BP1_1_COMPATIBILITY:
			registry_cfg.bp1_1_compatibility = *(DC_BOOL *)value;
			break;
		default:
			return DPWS_ERR_NO_SUCH_ATT_ON_OBJECT;
	}
	return DPWS_OK;
}

unsigned long get_registry_config_int_att(int att, int index)
{
	switch (att)
	{
		case DPWS_INT_BOOT_SEQ:
			return registry_cfg.boot_seq;
		case DPWS_BOOL_TCP_LISTEN:
			return registry_cfg.tcp_listen;
		case DPWS_INT_HTTP_BACKLOG:
			return registry_cfg.tcp_backlog;
		case DPWS_INT_HTTP_PORT:
			return http_endpoint.port;
		case DPWS_BOOL_BP1_1_COMPATIBILITY:
			return registry_cfg.bp1_1_compatibility;
		default:
			return 0xFFFFFFFF;
	}
}

void * get_registry_config_ptr_att(int att, int index)
{
	switch (att)
	{
		case DPWS_PTR_CALLBACK_DEVICE_METADATA:
			return (void *)registry_cfg.dev_metadata_hook;
		case DPWS_STR_PREFERRED_LANG:
			return registry_cfg.preferred_lang;
		default:
			return NULL;
	}
}

void * get_service_class_ptr_att(struct service_class * pServiceClass, int att, int index)
{
	switch (att)
	{
		case DPWS_STR_ID:
			return pServiceClass->id;
		case DPWS_PTR_TYPE:
		case DPWS_PTR_PREFIXED_TYPE:
			return GET_ENTRY(&pServiceClass->types, index);
		case DPWS_PTR_WSDL:
			return GET_ENTRY(&pServiceClass->wsdl_files, index);
		case DPWS_PTR_HANDLING_FUNCTION:
			return (void*)*(int (**)(struct dpws*))GET_ENTRY(&pServiceClass->funcs, index);
		case DPWS_PTR_FAULT_FUNCTION:
			return (void*)pServiceClass->fault_func;
		case DPWS_PTR_CALLBACK_EVENT_END:
			return (void*)pServiceClass->event_end_hook;
		case DPWS_PTR_USER_DATA:
			return (void *)pServiceClass->user_data;
		default:
			return NULL;
	}
}

int get_service_class_att_count(struct service_class * pServiceClass, int att)
{
	switch (att)
	{
		case DPWS_PTR_TYPE:
		case DPWS_PTR_PREFIXED_TYPE:
			return pServiceClass->types.nb;
		case DPWS_PTR_WSDL:
			return pServiceClass->wsdl_files.nb;
		case DPWS_PTR_HANDLING_FUNCTION:
			return pServiceClass->funcs.nb;
		default:
			return 1;
	}
}

void * get_device_model_ptr_att(struct device_model * pDeviceModel, int att, int index)
{
	switch (att)
	{
		case DPWS_STR_ID:
			return pDeviceModel->id;
		case DPWS_PTR_MODEL_NAME:
			return GET_ENTRY(&pDeviceModel->model_names, index);
		case DPWS_STR_MODEL_NUMBER:
			return pDeviceModel->model_number;
		case DPWS_STR_MODEL_URL:
			return pDeviceModel->model_url;
		case DPWS_STR_PRESENTATION_URL:
			return pDeviceModel->presentation_url;
		case DPWS_PTR_MANUFACTURER:
			return GET_ENTRY(&pDeviceModel->manufacturers, index);
		case DPWS_STR_MANUFACTURER_URL:
			return pDeviceModel->manufacturer_url;
		default:
			return NULL;
	}
}

unsigned long get_service_endpoint_int_att(struct service_endpoint * pService, int att, int index)
{
	switch (att)
	{
		case DPWS_INT_CLASS_HREF:
			return pService->hrefServiceClass;
		default:
			return 0xFFFFFFFF;
	}
}

void * get_service_endpoint_ptr_att(struct service_endpoint * pService, int att, int index)
{
	switch (att)
	{
		case DPWS_STR_SERVICE_ID:
			return pService->service_id;
		case DPWS_STR_HARDWARE_ID:
			return pService->hardware_id;
		case DPWS_STR_COMPATIBLE_ID:
			return pService->compatible_id;
		case DPWS_PTR_USER_DATA:
			return (void *)pService->user_data;
		case DPWS_INT_CLASS_HREF:
			return &pService->hrefServiceClass;
		case DPWS_PTR_HANDLING_FUNCTION: case DPWS_PTR_FAULT_FUNCTION: case DPWS_PTR_CALLBACK_EVENT_END:
			return get_service_class_ptr_att((struct service_class *)getObject(&registryPool, pService->hrefServiceClass), att, index);
		default:
			return NULL;
	}
}

int get_service_endpoint_att_count(struct service_endpoint * pService, int att)
{
	switch (att)
	{
		case DPWS_PTR_HANDLING_FUNCTION: case DPWS_PTR_FAULT_FUNCTION: case DPWS_PTR_CALLBACK_EVENT_END:
			return get_service_class_att_count((struct service_class *)getObject(&registryPool, pService->hrefServiceClass), att);
		default:
			return 1;
	}
}

void * get_service_port_ptr_att(struct service_port * pServicePort, int att, int index)
{
	switch (att)
	{
		case DPWS_STR_ADDRESS:
			return pServicePort->context_path;
		default:
			return NULL;
	}
}

unsigned long get_device_int_att(struct device * pDevice, int att, int index)
{
	switch (att)
	{
		case DPWS_INT_METADATA_VERSION:
			return pDevice->metadata_version;
		default:
			return get_service_endpoint_int_att(pDevice->super, att, index);
	}
}

void * get_device_ptr_att(struct device * pDevice, int att, int index)
{
	switch (att)
	{
		case DPWS_STR_DEVICE_ID:
			return pDevice->uuid;
		case DPWS_STR_ADDRESS:
			return pDevice->context_path;
		case DPWS_STR_SCOPE:
			return *(char **)GET_ENTRY(&pDevice->scopes, index);
		case DPWS_PTR_MODEL_NAME:
			return GET_ENTRY(&pDevice->model_names, index);
		case DPWS_STR_MODEL_NUMBER:
			return pDevice->model_number;
		case DPWS_STR_MODEL_URL:
			return pDevice->model_url;
		case DPWS_STR_PRESENTATION_URL:
			return pDevice->presentation_url;
		case DPWS_PTR_MANUFACTURER:
			return GET_ENTRY(&pDevice->manufacturers, index);
		case DPWS_STR_MANUFACTURER_URL:
			return pDevice->manufacturer_url;
		case DPWS_PTR_FRIENDLY_NAME:
			return GET_ENTRY(&pDevice->friendly_names, index);
		case DPWS_STR_FIRMWARE_VERSION:
			return pDevice->firmware_version;
		case DPWS_STR_SERIAL_NUMBER:
			return pDevice->serial_number;
		case DPWS_STR_DEVICE_CATEGORY:
			return pDevice->device_category;
		case DPWS_STR_MODEL_ID:
			return pDevice->model_id;
		case DPWS_PTR_USER_DATA:
			return (void *)pDevice->user_data;
		case DPWS_PTR_TYPE:
		case DPWS_PTR_PREFIXED_TYPE:
			return get_service_class_ptr_att((struct service_class *)getObject(&registryPool, pDevice->super->hrefServiceClass), att, index);
		default:
			return get_service_endpoint_ptr_att(pDevice->super, att, index);
	}
}

int get_device_model_att_count(struct device_model * pDeviceModel, int att)
{
	switch (att)
	{
		case DPWS_PTR_MANUFACTURER:
			return pDeviceModel->manufacturers.nb;
		case DPWS_PTR_MODEL_NAME:
			return pDeviceModel->model_names.nb;
		default:
			return 1;
	}
}

int get_device_att_count(struct device * pDevice, int att)
{
	switch (att)
	{
		case DPWS_STR_SCOPE:
			return pDevice->scopes.nb;
		case DPWS_PTR_MANUFACTURER:
			return pDevice->manufacturers.nb;
		case DPWS_PTR_MODEL_NAME:
			return pDevice->model_names.nb;
		case DPWS_PTR_FRIENDLY_NAME:
			return pDevice->friendly_names.nb;
		case DPWS_PTR_TYPE:
		case DPWS_PTR_PREFIXED_TYPE:
			return get_service_class_att_count((struct service_class *)getObject(&registryPool, pDevice->super->hrefServiceClass), att);
		default:
			return 1;
	}
}

static void freeServiceStruct(struct service_endpoint * service)
{
	DC_FREE(DC_MEM_REGISTRY, service->service_id);
	DC_FREE(DC_MEM_REGISTRY, service->hardware_id);
	DC_FREE(DC_MEM_REGISTRY, service->compatible_id);
	DA_FREE(&service->service_ports, NULL);
	DC_FREE(DC_MEM_REGISTRY, service);
}

static struct service_endpoint * create_hosted_service(struct device * device, short hrefClass)
{
	struct service_endpoint * service;
	struct service_class * servClass = (struct service_class *)checkoutObject(&registryPool, hrefClass);
	DPWSLOG1(DC_REGISTRY, "++ Handle for service class %d checked out by service\n", hrefClass);

	service = (struct service_endpoint*)DC_MALLOC(DC_MEM_REGISTRY, sizeof(struct service_endpoint));
	if (service) {
		if (!(service->service_id = DC_STRDUP(DC_MEM_REGISTRY, servClass->id)) && servClass->id)
			goto error;
		service->hardware_id = NULL;
		service->compatible_id = NULL;
		service->hrefServiceClass = hrefClass;
		DA_INIT(short, &service->service_ports, DC_MEM_REGISTRY, p_default_allocator, 5);
		service->device = device;
		service->user_data = NULL;
		service->href = createHandle(&registryPool, SERVICE_ENDPOINT_TYPE, service);
		service->subsc_manager = DC_FALSE;
		DPWSLOG1(DC_REGISTRY, "+ Handle for hosted service %d created\n", service->href);
		if (service->href < 0)
			goto error;
		else {
			short * phref;
			if (!(phref = DA_ADD(&device->hosted_services)))
				goto error;
			*phref = service->href;
		}
	}
	return service;

error:
	freeServiceStruct(service);
	return NULL;
}

static short create_service_port(char * context_path)
{
	short href = -1;
	struct service_port *serv_port;

	// clone service port list
	if (!(serv_port = (struct service_port*)DC_MALLOC(DC_MEM_REGISTRY, sizeof(struct service_port))))
		return DPWS_ERR_EOM;

	// init service port
	serv_port->context_path = context_path;
	serv_port->network_endpoint = &http_endpoint;
	serv_port->invoc_bind_list = NULL;
	serv_port->service = NULL;

	href = createHandle(&registryPool, SERVICE_PORT_TYPE, serv_port);
	DPWSLOG1(DC_REGISTRY, "+ Handle for service port %d created\n", href);
	if (href < 0) {
		DC_FREE(DC_MEM_REGISTRY, context_path);
		href = DPWS_ERR_EOM;
	}
	return href;
}

static DC_BOOL hrefRelease(short * href, void * dummy)
{
	releaseHandleUnsafe(&registryPool, *href);
	DPWSLOG1(DC_REGISTRY, "- Handle %d released\n", *href);
	return DC_FALSE;
}

static void freeDeviceStruct(struct device * device)
{
	DA_FREE(&device->hosted_services, NULL);
	DA_FREE(&device->model_names, localizedStringFree);
	DA_FREE(&device->manufacturers, localizedStringFree);
	DA_FREE(&device->friendly_names, localizedStringFree);
	DA_FREE(&device->scopes, scope_free_cbk);

	DC_FREE(DC_MEM_REGISTRY, device->uuid);
	DC_FREE(DC_MEM_REGISTRY, device->context_path);
	DC_FREE(DC_MEM_REGISTRY, device->model_number);
	DC_FREE(DC_MEM_REGISTRY, device->model_url);
	DC_FREE(DC_MEM_REGISTRY, device->presentation_url);
	DC_FREE(DC_MEM_REGISTRY, device->manufacturer_url);
	DC_FREE(DC_MEM_REGISTRY, device->firmware_version);
	DC_FREE(DC_MEM_REGISTRY, device->serial_number);
	DC_FREE(DC_MEM_REGISTRY, device->device_category);
	DC_FREE(DC_MEM_REGISTRY, device->model_id);
	DC_FREE(DC_MEM_REGISTRY, device);
}

static int freeDevice(handle_s *handleInfo)
{
	struct device * device = (struct device *)handleInfo->pObject;

	DPWSLOG1(DC_REGISTRY, "- Device service endpoint handle %d released by the device\n", device->super->href);
	DA_BROWSE(&device->hosted_services, hrefRelease, NULL);
	freeDeviceStruct(device);

	return DPWS_OK;
}

static int freeService(handle_s *handleInfo)
{
	struct service_endpoint * service = (struct service_endpoint *)handleInfo->pObject;

	releaseHandleUnsafe(&registryPool, service->hrefServiceClass);
	DPWSLOG1(DC_REGISTRY, "- Service class handle %d released by the service\n", service->hrefServiceClass);
	DA_BROWSE(&service->service_ports, hrefRelease, NULL);
	delete_subsc_manager(handleInfo->handleRef);
	freeServiceStruct(service);

	return DPWS_OK;
}

static void freeDeviceModelStruct(struct device_model * deviceModel)
{
	DA_FREE(&deviceModel->service_classes, NULL);
	DA_FREE(&deviceModel->model_names, localizedStringFree);
	DA_FREE(&deviceModel->manufacturers, localizedStringFree);

	DC_FREE(DC_MEM_REGISTRY, deviceModel->id);
	DC_FREE(DC_MEM_REGISTRY, deviceModel->model_number);
	DC_FREE(DC_MEM_REGISTRY, deviceModel->model_url);
	DC_FREE(DC_MEM_REGISTRY, deviceModel->presentation_url);
	DC_FREE(DC_MEM_REGISTRY, deviceModel->manufacturer_url);
	DC_FREE(DC_MEM_REGISTRY, deviceModel);
}

static int freeDeviceModel(handle_s *handleInfo)
{
	struct device_model * deviceModel = (struct device_model *)handleInfo->pObject;

	DA_BROWSE(&deviceModel->service_classes, hrefRelease, NULL);
	freeDeviceModelStruct(deviceModel);
	return DPWS_OK;
}

static void freeServiceClassStruct(struct service_class * servClass)
{
	DA_FREE(&servClass->types, prefixed_qname_free_cbk);
	DA_FREE(&servClass->wsdl_files, wsdl_info_free_cbk);
	DA_FREE(&servClass->funcs, NULL);

	DC_FREE(DC_MEM_REGISTRY, servClass->id);
	DC_FREE(DC_MEM_REGISTRY, servClass);
}

static int freeServiceClass(handle_s *handleInfo)
{
	freeServiceClassStruct((struct service_class *)handleInfo->pObject);
	return DPWS_OK;
}

static void freeServicePortStruct(struct service_port * servPort)
{
	DC_FREE(DC_MEM_REGISTRY, servPort->context_path);
	DC_FREE(DC_MEM_REGISTRY, servPort);
}

static int freeServicePort(handle_s *handleInfo)
{
	freeServicePortStruct((struct service_port *)handleInfo->pObject);
	return DPWS_OK;
}

static short create_service_class()
{
	short href = -1;
	struct service_class *sClass;

	if (!(sClass = (struct service_class*)DC_MALLOC(DC_MEM_REGISTRY, sizeof(struct service_class))))
		return DPWS_ERR_EOM;

	sClass->fault_func = NULL;
	sClass->event_end_hook = NULL;
	sClass->user_data = NULL;
	DA_INIT(struct prefixed_qname, &sClass->types, DC_MEM_REGISTRY, p_default_allocator, DISCOVERY_ALLOC_INCREMENT);
	DA_INIT(struct wsdl_info, &sClass->wsdl_files, DC_MEM_REGISTRY, p_default_allocator, DISCOVERY_ALLOC_INCREMENT);
	DA_INIT(dispatch_cbk, &sClass->funcs, DC_MEM_REGISTRY, p_default_allocator, DISCOVERY_ALLOC_INCREMENT);
	if (!(sClass->id = (char*)DC_MALLOC(DC_MEM_REGISTRY, 6)))	// max string size for short
	{
		DC_FREE(DC_MEM_REGISTRY, sClass);
		return DPWS_ERR_EOM;
	}

	href = createHandle(&registryPool, SERVICE_CLASS_TYPE, sClass);
	DPWSLOG1(DC_REGISTRY, "+ Handle for service class %d created\n", href);
	if (href < 0)
		DC_FREE(DC_MEM_REGISTRY, sClass);
	else
		sprintf(sClass->id, "%d", href);

	return href;
}

static short create_custom_device(short localID, short hrefModel)
{
	struct device * device;
	struct device_model * devModel;
	short hrefBuiltinServiceClass = -1;
	struct service_class * sClass = NULL;

	devModel = (struct device_model *)getObject(&registryPool, hrefModel);

	if (!(device = (struct device *)DC_MALLOC(DC_MEM_REGISTRY, sizeof(struct device))))
		return DPWS_ERR_EOM;

	device->uuid = NULL;
	device->local_id = localID;
	device->context_path = NULL;
	device->metadata_version = 1;
	device->firmware_version = NULL;
	device->serial_number = NULL;
	device->user_data = NULL;
	device->status = INIT;
	device->device_category = NULL;
	device->model_id = NULL;

	DA_INIT(char *, &device->scopes, DC_MEM_REGISTRY, p_default_allocator, DISCOVERY_ALLOC_INCREMENT);
	DA_INIT(short, &device->hosted_services, DC_MEM_REGISTRY, p_default_allocator, 2);
	DA_INIT(localized_string_t, &device->friendly_names, DC_MEM_REGISTRY, p_default_allocator, 1);
	DA_INIT(localized_string_t, &device->manufacturers, DC_MEM_REGISTRY, p_default_allocator, 1);
	DA_INIT(localized_string_t, &device->model_names, DC_MEM_REGISTRY, p_default_allocator, 1);
	hrefBuiltinServiceClass = create_service_class();
	sClass = (struct service_class *)getObject(&registryPool, hrefBuiltinServiceClass);
	set_service_class_ptr_att(sClass, DPWS_STR_ID, NULL, DC_TRUE); // Added to remove default id which breaks everything and to be able to tell public SC apart!
	// Add a place in types for wsdp:device (seen on the network but not thru the API
	if (!DA_ADD(&sClass->types))
		goto error;
	sClass->types.nb--;

	if (devModel) {
		if (DA_COPY(&device->manufacturers, &devModel->manufacturers, clone_localized_string, NULL)
			|| (!(device->manufacturer_url = DC_STRDUP(DC_MEM_REGISTRY, devModel->manufacturer_url)) && devModel->manufacturer_url)
			|| DA_COPY(&device->model_names, &devModel->model_names, clone_localized_string, NULL)
			|| (!(device->model_number = DC_STRDUP(DC_MEM_REGISTRY, devModel->model_number)) && devModel->model_number)
			|| (!(device->model_url = DC_STRDUP(DC_MEM_REGISTRY, devModel->model_url)) && devModel->model_url)
			|| (!(device->presentation_url = DC_STRDUP(DC_MEM_REGISTRY, devModel->presentation_url)) && devModel->presentation_url))
			goto error;
	}
	else {
		device->manufacturer_url = NULL;
		device->model_number = NULL;
		device->model_url = NULL;
		device->presentation_url = NULL;
	}

	device->super = create_hosted_service(device, hrefBuiltinServiceClass);

	if (!device->super || (device->href = createHandle(&registryPool, DEVICE_INSTANCE_TYPE, device)) < 0)
		goto error;
	DPWSLOG1(DC_REGISTRY,"+ Handle for device %d created\n", device->href);

	return device->href;

error:
	freeDeviceStruct(device);
	return DPWS_ERR_EOM;
}

static int releaseHandleHook(handle_s *handleInfo)
{
	int ret = 0;
	switch (handleInfo->type)
	{
		case DEVICE_INSTANCE_TYPE:
			DPWSLOG1(DC_REGISTRY, "# Free device instance with handle %d.\n", handleInfo->handleRef);
			ret = freeDevice(handleInfo);
			break;
		case SERVICE_ENDPOINT_TYPE:
			DPWSLOG1(DC_REGISTRY, "# Free service endpoint with handle %d.\n", handleInfo->handleRef);
			ret = freeService(handleInfo);
			break;
		case DEVICE_MODEL_TYPE:
			DPWSLOG1(DC_REGISTRY, "# Free device model with handle %d.\n", handleInfo->handleRef);
			ret = freeDeviceModel(handleInfo);
			break;
		case SERVICE_CLASS_TYPE:
			DPWSLOG1(DC_REGISTRY, "# Free service class with handle %d.\n", handleInfo->handleRef);
			ret = freeServiceClass(handleInfo);
			break;
		case SERVICE_PORT_TYPE:
			DPWSLOG1(DC_REGISTRY, "# Free service port with handle %d.\n", handleInfo->handleRef);
			ret = freeServicePort(handleInfo);
			break;
		default:
			break;
	}
	return ret;
}

static DC_BOOL device_model_equals(short type, struct device_model * devModel, char * id)
{
	return !strcmp(devModel->id, id);
}

static DC_BOOL service_class_equals(short type, struct service_class * servClass, char * id)
{
	return servClass->id && !strcmp(servClass->id, id);	// ignore built-in device SC
}

static DC_BOOL device_equals(short type, struct device * device, char * uuid)
{
	return API_VISIBLE_DEVICE(device) && !strcasecmp(device->uuid, uuid);
}

static DA_TYPED(href) * find_service_endpoint_ports(short href)
{
	struct service_endpoint * endpoint;
	handle_s *handle = getHandle(&registryPool, href);
	if (!handle)
		return NULL;

	switch (handle->type)
	{
		case DEVICE_INSTANCE_TYPE:
			endpoint = (struct service_endpoint *)getObject(&registryPool, ((struct device *)handle->pObject)->super->href);
			break;
		case SERVICE_ENDPOINT_TYPE:
			endpoint = (struct service_endpoint *)handle->pObject;
			break;
		default:
			return NULL;
	}

	return &endpoint->service_ports;
}

static int bind_service(short hrefService, short hrefServicePort)
{
	struct service_endpoint * serviceEndpoint;
	struct service_port * servicePort;
	short * phref;

	servicePort = (struct service_port *)checkoutObject(&registryPool, hrefServicePort);	// the service endpoint will hold a use
	DPWSLOG1(DC_REGISTRY, "++ Handle for service port %d checked out by hosted service\n", hrefServicePort);
	if (!servicePort)
		return DPWS_ERR_NO_HANDLE_FOUND;

	serviceEndpoint = (struct service_endpoint *)getObject(&registryPool, hrefService);
	if (!serviceEndpoint)
		return DPWS_ERR_NO_HANDLE_FOUND;

	if (STARTED_DEVICE(serviceEndpoint->device) || (servicePort->service && STARTED_DEVICE(servicePort->service->device)))
		return DPWS_ERR_IMMUTABLE_USED_OBJECT;

	if (!(phref = DA_ADD(&serviceEndpoint->service_ports)))
		return DPWS_ERR_EOM;
	*phref = hrefServicePort;
	servicePort->service = serviceEndpoint;

	return DPWS_OK;
}

static int check_device(struct device *device)
{
	char * path;
	int ret = DPWS_OK;

	switch (get_device_mode(device))
	{
		case DPWS_DEVICE_USE:

			if (device->manufacturers.nb == 0)
				return DPWS_ERR_MISSING_MANUFACTURER;
			if (device->model_names.nb == 0)
				return DPWS_ERR_MISSING_MODEL_NAME;
			if (device->friendly_names.nb == 0)
				return DPWS_ERR_MISSING_FRIENDLY_NAME;

			// no break

		case DPWS_EVENTING_USE:

			if (!device->uuid)
			{
				if (!(device->uuid = DC_MALLOC(DC_MEM_REGISTRY, SCHEME_UUID_STRING_SIZE)))
					return DPWS_ERR_EOM;
				if (device->local_id == RANDOM_DEVICE_ID)
					dpws_schemed_uuid_create(device->uuid);
				else {
					uint32_t uuid;
					MAKE_32b(device->local_id, (uint16_t)http_endpoint.port, uuid)
					stable_uuid_create(device->uuid, uuid, dcpl_mac_address);
				}
			}
			// create device physical address
			if (is_physical_address(device->uuid)) {
				path = strchr(strchr(device->uuid, ':')+3, '/')+1;
			} else if (device->context_path) {
				path = device->context_path;
			} else
				path = device->uuid + UUID_SCHEME_SIZE;
			if (!(path = DC_STRDUP(DC_MEM_REGISTRY, path)))
				return DPWS_ERR_EOM;
			ret = bind_service(device->super->href, create_service_port(path));
			break;

		default:
			break;
	}
	return ret;
}

static invocation_list_t * checkout_invocation_list(invocation_list_t * list)
{
	invocation_list_t * ret;
	dcpl_mutex_lock(invocation_lock);
	list->use_count++;
	DPWSLOG1(DC_REGISTRY, "-> Checking out invocation list %X\n", list);
	ret = list;
	dcpl_mutex_unlock(invocation_lock);
	return ret;
}

/* Namespace table management (gSOAP) format */
static DC_BOOL add_ns_entry(struct prefixed_qname *pPQName, struct ns_info *ns_info)
{
	struct Namespace *nns = NULL, *wns = NULL;
	int i;

	// check that if prefix is used twice, the namespace is the same...
	for(i = 0; i < ns_info->table->nb; i++) {
		wns = DA_GET(ns_info->table, i);
		if (!strcmp(pPQName->prefix, wns->id)) {
			if (strcmp(pPQName->qname.ns, wns->ns))
				return DC_TRUE;
			else
				return DC_FALSE;	// ns is the same
		}
	}
	// add entry
	DC_ERROR_ASSERT_NO_RC(nns = DA_ADD(ns_info->table));
	nns->id = pPQName->prefix;
	nns->ns = pPQName->qname.ns;
	nns->in = NULL;
	nns->out = NULL;
	return DC_FALSE;

DC_FUNC_ERROR
	return DC_TRUE;
}

static DC_BOOL add_service_ns(short * hrefService, struct ns_info *ns_info)
{
	struct service_endpoint * service = (struct service_endpoint *)getObject(&registryPool, *hrefService);
	DA_TYPED(pqn) * types = &((struct service_class *)getObject(&registryPool, service->hrefServiceClass))->types;
	return DA_BROWSE(types,	add_ns_entry, ns_info) == types->nb ? DC_FALSE : DC_TRUE;
}

static int add_implicit_ns(struct dpws * dpws, DA_TYPED(ns) * ns_tab)
{
	struct Namespace *wdp_ns = NULL;
	int ret = DPWS_OK;

	DC_ERROR_ASSERT_ALLOC(wdp_ns = DA_ADD(ns_tab));
	wdp_ns->id = WDP_PREFIX;
	wdp_ns->ns = dpws->protocols->wdp_uri;
	wdp_ns->in = NULL;
	wdp_ns->out = NULL;

DC_FUNC_ERROR
	return ret;
}

struct Namespace * get_device_discovery_ns_table(struct dpws * dpws, struct device * device)
{
	da_allocator_t daa;
	DA_TYPED(ns) ns_tab;
	struct ns_info ns_info;
	DA_TYPED(pqn) * types = &((struct service_class *)getObject(&registryPool, device->super->hrefServiceClass))->types;
	int i;

	DA_INIT(struct Namespace, &ns_tab, DC_MEM_DISCOVERY, soap_get_da_allocator(dpws_dpws2soap(dpws), &daa), DISCOVERY_ALLOC_INCREMENT);
	ns_info.dpws  = dpws;
 	ns_info.table = &ns_tab;

	// init default ns
	for (i = 0; dpws->protocols->discovery_snd_namespaces[i].id; i++) {
		struct Namespace * nns;
		DC_ERROR_ASSERT_NO_RC(nns = DA_ADD(&ns_tab));
		nns->id = dpws->protocols->discovery_snd_namespaces[i].id;
		nns->ns = dpws->protocols->discovery_snd_namespaces[i].ns;
		nns->in = dpws->protocols->discovery_snd_namespaces[i].in;
		nns->out = dpws->protocols->discovery_snd_namespaces[i].out;
	}
	// set discovery ns table
 	DC_ERROR_ASSERT_NO_RC(DA_BROWSE(types, add_ns_entry, &ns_info) == types->nb);

	DC_ERROR_ASSERT_NO_RC(!add_implicit_ns(dpws, &ns_tab));

	return ns_tab.tab;

DC_FUNC_ERROR
	dpws->err = DPWS_ERR_EOM;
	return NULL;
}

struct Namespace * get_device_metadata_ns_table(struct dpws * dpws, struct device * device)
{
	da_allocator_t daa;
	DA_TYPED(ns) ns_tab;
	struct ns_info ns_info;
	int i;

	DA_INIT(struct Namespace, &ns_tab, DC_MEM_DISCOVERY, soap_get_da_allocator(dpws_dpws2soap(dpws), &daa), DISCOVERY_ALLOC_INCREMENT);
 	ns_info.dpws  = dpws;
 	ns_info.table = &ns_tab;

	// init default ns & set Metadata Exchange ns table
	for (i = 0; dpws->protocols->transfer_snd_namespaces[i].id; i++) {
		struct Namespace * nns;
		DC_ERROR_ASSERT_NO_RC(nns = DA_ADD(&ns_tab));
		nns->id = dpws->protocols->transfer_snd_namespaces[i].id;
		nns->ns = dpws->protocols->transfer_snd_namespaces[i].ns;
		nns->in = dpws->protocols->transfer_snd_namespaces[i].in;
		nns->out = dpws->protocols->transfer_snd_namespaces[i].out;
	}

 	DC_ERROR_ASSERT_NO_RC(DA_BROWSE(&device->hosted_services, add_service_ns, &ns_info) == device->hosted_services.nb);

	return ns_tab.tab;

DC_FUNC_ERROR
	dpws->err = DPWS_ERR_EOM;
	return NULL;
}

static void release_invocation_list(invocation_list_t * list)
{
	DPWSLOG2(DC_REGISTRY, "<- Releasing invocation list %X use_count %d\n", list, list->use_count);
	if (--list->use_count == 0)
	{
		if (list->off_device) {
			DPWSLOG1(DC_REGISTRY, "- Device handle %d released by invocation list\n", list->off_device->href);
			releaseHandle(&registryPool, list->off_device->href);
		}
		if (!list->is_online) {
			DPWSLOG1(DC_REGISTRY, "Releasing offline invocation list %X\n", list);
			DA_FREE(&list->ports, NULL);
			DC_FREE(DC_MEM_REGISTRY, list);
		}
	}
}

static int send_hello_cbk(dcpl_ip_addr_t * netif_addr, struct device * device)
{
	struct dpws_protocols * protocols[N_DPWS_VERSIONS];
	int i, size, ret = DPWS_OK;
	size = get_protocols_versions(NULL, protocols, N_DPWS_VERSIONS);
	for (i = 0; i < size; i++) {
		if ((ret = schedule_discovery_msg(netif_addr, device, DPWS_MSG_HELLO, protocols[i])))
			return ret;
	}
	return ret;
}

static DC_BOOL advertize_enabled_device(short type, struct device * device, int * ret)
{
	if (API_VISIBLE_DEVICE(device) && get_device_mode(device) == DPWS_DEVICE_USE)
		*ret = network_browse_interfaces(DCPL_AF_INET|DCPL_AF_INET6, (network_addr_cbk)send_hello_cbk, device);
	return *ret ? DC_TRUE : DC_FALSE;
}

static int send_bye_cbk(dcpl_ip_addr_t * netif_addr, struct device * device)
{
	struct dpws_protocols * protocols[N_DPWS_VERSIONS];
	int i, size, ret = DPWS_OK;
	size = get_protocols_versions(NULL, protocols, N_DPWS_VERSIONS);
	for (i = 0; i < size; i++) {
		if ((ret = schedule_discovery_msg(netif_addr, device, DPWS_MSG_BYE, protocols[i])))
			return ret;
	}
	return ret;
}

static int shutdown_device_callback(struct dpws * dpws, struct reactor_item * item, void *callback_data)
{
	int ret = DPWS_OK;
	struct device * device = (struct device *)callback_data;
	send_eventing_end(dpws, device);
	// now that event end  are sent, BYE can be scheduled
	ret = network_browse_interfaces(DCPL_AF_INET|DCPL_AF_INET6, (network_addr_cbk)send_bye_cbk, device);
	if (!ret)
		ret = dc_reactor_interrupt(&reactor);

	return ret;
}

static DC_BOOL shutdown_enabled_device(short type, struct device * device, void * dummy)
{
	if (STARTED_DEVICE(device) && get_device_mode(device) == DPWS_DEVICE_USE) {
		device->status = OFFLINE;
		dc_reactor_register_timer(&reactor, DC_FALSE, 0, 0, shutdown_device_callback, device, NULL, NULL);
	}
	return DC_FALSE;
}

static DC_BOOL find_service_hook(short * hrefService, char * service_id)
{
	struct service_endpoint * service = (struct service_endpoint *)getObject(&registryPool, *hrefService);
	return service->service_id && !strcmp(service->service_id, service_id);
}

static DC_BOOL find_service_hook_by_class(short * hrefService, short * hrefServiceClass)
{
	struct service_endpoint * service = (struct service_endpoint *)getObject(&registryPool, *hrefService);
	return service->hrefServiceClass == *hrefServiceClass;
}

static int port_context_path_cmp(struct service_port **serv_port, const char * ref)
{
	return strcmp((*serv_port)->context_path, ref);
}

static DC_BOOL gather_service_ports(short * hrefPort, invocation_list_t * invoc_list)
{
	struct service_port * port = (struct service_port*)getObject(&registryPool, *hrefPort);
	int i = DA_ADD_SORTED(&invoc_list->ports, port_context_path_cmp, port->context_path);
	if (i < 0)
		return DC_TRUE;
	*DA_GET(&invoc_list->ports, i) = port;
	return DC_FALSE;
}

static DC_BOOL gather_device_ports(short * hrefService, invocation_list_t * invoc_list)
{
	struct service_endpoint * service = (struct service_endpoint*)getObject(&registryPool, *hrefService);
	return DA_BROWSE(&service->service_ports, gather_service_ports, invoc_list) == service->service_ports.nb
		? DC_FALSE : DC_TRUE;
}

static DC_BOOL build_clone_list(struct service_port ** ppServicePort, struct clone_info * cloneInfo)
{
	if ((*ppServicePort)->service->device != cloneInfo->device) {	// if device == NULL, all added. It works.
		struct service_port ** ppsp;
		if (!(ppsp = DA_ADD(cloneInfo->ports)))
			return DC_TRUE;
		*ppsp = *ppServicePort;
	}
	return DC_FALSE;
}

static invocation_list_t * new_invocation_list()
{
	invocation_list_t * invoc_list = (invocation_list_t*)DC_MALLOC(DC_MEM_REGISTRY, sizeof(invocation_list_t));
	if (invoc_list) {
		DA_INIT(struct service_port *, &invoc_list->ports, DC_MEM_REGISTRY, p_default_allocator, 5);
		invoc_list->use_count = 1;	// hold by the network_endpoint for which it is created
		invoc_list->is_online = DC_TRUE;
		invoc_list->off_device = NULL;
		invoc_list->network_endpoint = &http_endpoint;	// affect to the current sole network endpoint.
	}
	return invoc_list;
}

static int substitute_invocation_list(struct device * to_disable, struct device * to_enable)
{
	struct clone_info cloneInfo;
	invocation_list_t	*new_list, *old_list;

	if (!(new_list = new_invocation_list()))	// use count = 1
		return DPWS_ERR_EOM;

	old_list = http_endpoint.invoc_list;

	// build a list copy potentially without original device service ports
	cloneInfo.device = to_disable;	// can be null
	cloneInfo.ports = &new_list->ports;
	if (DA_BROWSE(&old_list->ports, build_clone_list, &cloneInfo) < old_list->ports.nb)
		return DPWS_ERR_EOM;

	// add new device service ports to the new invocation list
	if (to_enable) {
		if (DA_BROWSE(&to_enable->hosted_services, gather_device_ports, new_list) < to_enable->hosted_services.nb)
			return DPWS_ERR_EOM;
	}

	// if device substitution
	if (to_enable) {
		if (to_disable) {
			to_enable->metadata_version++;
			if (registry_cfg.dev_metadata_hook)
				registry_cfg.dev_metadata_hook(to_enable->href);
		}
	}

	// perform substitution
	dcpl_mutex_lock(invocation_lock);
	http_endpoint.invoc_list = new_list;
	old_list->is_online = DC_FALSE;	// only the configuration thread can set it
	old_list->off_device = to_disable;	// the handle will be released when the invocation list is released
	DPWSLOG2(DC_REGISTRY, "<-> Substituting invocation list %X with %X\n", old_list, new_list);
	release_invocation_list(old_list);
	dcpl_mutex_unlock(invocation_lock);

	return DPWS_OK;
}

static int enable_device(short deviceHandleRef)
{
	int ret = DPWS_OK;
	struct device *device;

	device = (struct device *)checkoutObject(&registryPool, deviceHandleRef);	// hold a counter when used by invocation
	DPWSLOG1(DC_REGISTRY, "++ Handle for device %d checked out when set online\n", deviceHandleRef);
	if (!device)
		return DPWS_ERR_NO_HANDLE_FOUND;

	if (STARTED_DEVICE(device) || device->status == REPLACED) {
		releaseHandle(&registryPool, deviceHandleRef);
		return DPWS_ERR_ILLEGAL_STATE;
	}

	if (device->status == INIT && !(ret = check_device(device)))
		device->status = OFFLINE;

	if (!ret)
	{
		ret = substitute_invocation_list(NULL, device);

		if (!ret && registry_cfg.server_up && get_device_mode(device) == DPWS_DEVICE_USE)
		{
			ret = network_browse_interfaces(DCPL_AF_INET|DCPL_AF_INET6, (network_addr_cbk)send_hello_cbk, device);

			/* Send a dummy connection on the TCP socket to wake up accept */
			if (!ret)
				ret = dc_reactor_interrupt(&reactor);
		}
		// else scheduling will be done by dpws_server_init()
	}
	return ret;
}

static DC_BOOL clone_service_port(int mod, struct da_allocator * allocator, short * dest, short * original, struct service_endpoint * service)
{
	struct service_port * clone, *port;

	if (!(clone = (struct service_port*)DC_MALLOC(mod, sizeof(struct service_port))))
		return DC_FALSE;

	port = (struct service_port *)getObject(&registryPool, *original);
	if (!(clone->context_path = DC_STRDUP(mod, port->context_path)) && port->context_path)
		goto error;
	clone->service = service;
	clone->network_endpoint = port->network_endpoint;

	*dest = createHandle(&registryPool, SERVICE_PORT_TYPE, clone);
	DPWSLOG1(DC_REGISTRY, "+ Handle for cloned service port %d created\n", *dest);
	if (*dest < 0)
		goto error;

	return DC_TRUE;

error:
	freeServicePortStruct(clone);
	return DC_FALSE;
}

static struct service_endpoint * _clone_hosted_service(short * original, struct device * device)
{
	struct service_endpoint * clone, *service;
	int status;

	if (!(clone = (struct service_endpoint*)DC_MALLOC(DC_MEM_REGISTRY, sizeof(struct service_endpoint))))
		return NULL;

	service = (struct service_endpoint *)getObject(&registryPool, *original);
	if (!(clone->service_id = DC_STRDUP(DC_MEM_REGISTRY, service->service_id)) && service->service_id)
		goto error;
	if (!(clone->hardware_id = DC_STRDUP(DC_MEM_REGISTRY, service->hardware_id)) && service->hardware_id)
		goto error;
	if (!(clone->compatible_id = DC_STRDUP(DC_MEM_REGISTRY, service->compatible_id)) && service->compatible_id)
		goto error;
	clone->hrefServiceClass = service->hrefServiceClass;
	checkoutHandle(&registryPool, clone->hrefServiceClass);
	DPWSLOG1(DC_REGISTRY, "++ Handle for service class %d checked out by cloned hosted service\n", clone->hrefServiceClass);

	DA_INIT(short, &clone->service_ports, DC_MEM_REGISTRY, p_default_allocator, 2);
	if (DA_COPY(&clone->service_ports, &service->service_ports, clone_service_port, clone))
		goto error;

	clone->user_data = service->user_data;
	clone->device = device;

	clone->href = createHandle(&registryPool, SERVICE_ENDPOINT_TYPE, clone);
	DPWSLOG1(DC_REGISTRY, "+ Handle for cloned service port %d created\n", clone->href);
	if (clone->href < 0)
		goto error;

	if ((status = clone_subsc_manager(*original, clone->href)) && (status != DPWS_ERR_NO_HANDLE_FOUND))
		goto error;

	return clone;

error:
	freeServiceStruct(clone);
	return NULL;
}

static DC_BOOL clone_hosted_service(int mod, struct da_allocator * allocator, short * dest, short * original, struct device * device)
{
	struct service_endpoint * ept = _clone_hosted_service(original, device);
	return ept && (*dest = ept->href) ? DC_TRUE : DC_FALSE;
}

static short clone_device(short href)
{
	int ret = DPWS_OK;
	struct device * device, * clone;

	device = (struct device *)getObject(&registryPool, href);
	if (!device)
		return DPWS_ERR_NO_HANDLE_FOUND;

	if (device->status != ONLINE)
		return DPWS_ERR_ILLEGAL_STATE;

	clone = (struct device*)DC_MALLOC(DC_MEM_REGISTRY, sizeof(struct device));
	if (!clone)
		return DPWS_ERR_EOM;

	if ((!(clone->uuid = DC_STRDUP(DC_MEM_REGISTRY, device->uuid)) && device->uuid)
		|| (!(clone->context_path = DC_STRDUP(DC_MEM_REGISTRY, device->context_path)) && device->context_path))
		goto error;
	clone->local_id = device->local_id;
	clone->metadata_version = device->metadata_version;
	clone->user_data = device->user_data;
	clone->status = CLONED;
	clone->super = NULL;
	DA_INIT(char *, &clone->scopes, DC_MEM_REGISTRY, p_default_allocator, 2);
	DA_INIT(short, &clone->hosted_services, DC_MEM_REGISTRY, p_default_allocator, 3);
	DA_INIT(localized_string_t, &clone->manufacturers, DC_MEM_REGISTRY, p_default_allocator, 1);
	DA_INIT(localized_string_t, &clone->model_names, DC_MEM_REGISTRY, p_default_allocator, 1);
	DA_INIT(localized_string_t, &clone->friendly_names, DC_MEM_REGISTRY, p_default_allocator, 1);

	if (!DA_STRDUP(&clone->scopes, &device->scopes)
		||DA_COPY(&clone->hosted_services, &device->hosted_services, clone_hosted_service, clone)
			|| !(clone->super = _clone_hosted_service(&device->super->href, clone)))
		goto error;

	if (DA_COPY(&clone->manufacturers, &device->manufacturers, clone_localized_string, NULL)
		|| (!(clone->manufacturer_url = DC_STRDUP(DC_MEM_REGISTRY, device->manufacturer_url)) && device->manufacturer_url)
		|| DA_COPY(&clone->model_names, &device->model_names, clone_localized_string, NULL)
		|| (!(clone->model_number = DC_STRDUP(DC_MEM_REGISTRY, device->model_number)) && device->model_number)
		|| (!(clone->model_url = DC_STRDUP(DC_MEM_REGISTRY, device->model_url)) && device->model_url)
		|| (!(clone->presentation_url = DC_STRDUP(DC_MEM_REGISTRY, device->presentation_url)) && device->presentation_url)
		|| DA_COPY(&clone->friendly_names, &device->friendly_names, clone_localized_string, NULL)
		|| (!(clone->firmware_version = DC_STRDUP(DC_MEM_REGISTRY, device->firmware_version)) && device->firmware_version)
		|| (!(clone->serial_number = DC_STRDUP(DC_MEM_REGISTRY, device->serial_number)) && device->serial_number)
		|| (!(clone->device_category = DC_STRDUP(DC_MEM_REGISTRY, device->device_category)) && device->device_category)
		|| (!(clone->model_id = DC_STRDUP(DC_MEM_REGISTRY, device->model_id)) && device->model_id))
		goto error;

	ret = clone->href = createHandle(&registryPool, DEVICE_INSTANCE_TYPE, clone);
	DPWSLOG1(DC_REGISTRY, "+ Handle for cloned device %d created\n", clone->href);
	if (ret < 0)
		goto error;

	return ret;

error:
	freeDeviceStruct(clone);
	return DPWS_ERR_EOM;
}

static int replace_device(short href_original, short href_clone)
{
	int ret = DPWS_OK;
	struct device * device, * new_device;

	device = (struct device *)getObject(&registryPool, href_original);
	if (!device)
		return DPWS_ERR_NO_HANDLE_FOUND;

	new_device = (struct device *)checkoutObject(&registryPool, href_clone);	// a hold is taken for the online device
	DPWSLOG1(DC_REGISTRY, "++ Handle for device %d checked out when set online\n", href_clone);
	if (new_device == NULL)
		return DPWS_ERR_NO_HANDLE_FOUND;
	else {
		if (new_device->status == INIT) {
			if ((ret = check_device(new_device))) {
				releaseHandle(&registryPool, href_clone);
				return ret;
			}
			else
				new_device->status = OFFLINE;
		}
		switch(new_device->status)
		{
		case CLONED:
			if (strcasecmp(new_device->uuid, device->uuid))
				return DPWS_ERR_DEVICE_NOT_CLONED;
			break;
		case OFFLINE:
			if (!(new_device->uuid = DC_STRDUP(DC_MEM_REGISTRY, device->uuid)) && device->uuid)
				return DPWS_ERR_EOM;
			new_device->metadata_version = device->metadata_version;
			break;
		default:
			return DPWS_ERR_ILLEGAL_STATE;
		}
	}

	if (!(ret = substitute_invocation_list(device, new_device)))
	{
		device->status = REPLACED;	// won't be set by the BYE msg processing since no bye...

		if (registry_cfg.server_up && get_device_mode(new_device) == DPWS_DEVICE_USE)
		{
			ret = network_browse_interfaces(DCPL_AF_INET|DCPL_AF_INET6, (network_addr_cbk)send_hello_cbk, new_device);
			/* Send a dummy connection on the TCP socket to wake up accept */
			if (!ret)
				ret = dc_reactor_interrupt(&reactor);
		}
		// else scheduling will be done by dpws_server_init()
	}
	return ret;
}

static DC_BOOL find_href_hook(short * href1, short * href2)
{
	return *href1 == *href2;
}

static DC_BOOL gather_handles(handle_s * h, struct gather_handles_info * gh_info) {
	if (gh_info->nb == gh_info->max_len)
		return DC_TRUE;
	if (!gh_info->filter || gh_info->filter(h->pObject, gh_info->filter_param)) {
		gh_info->hrefs[gh_info->nb++] = h->handleRef;
		h->useCount++;
	}
	return gh_info->nb >= gh_info->max_len;
}

/*----------------------------------------------------------------------------*\
 *               INTERNAL API: CONFIGURATION & INITIALIZATION                 *
\*----------------------------------------------------------------------------*/

int get_device_mode(struct device* pDevice)
{
	/* Check if it is the hidden device */
	return pDevice->href == hidden_device_href ? DPWS_EVENTING_USE : DPWS_DEVICE_USE;
}

int check_configuration()
{
	int ret = DPWS_OK;

	if (registry_cfg.boot_seq == 0xFFFFFFFF && get_toolkit_mode() == DPWS_DEVICE_USE)
		return DPWS_ERR_MISSING_BOOT_SEQUENCE;

	return ret;
}

int init_registry()
{
	if ((reg_conf_lock = (dcpl_mutex_t*)dcpl_mutex_init()) == NULL)
		return DPWS_ERR_CREATING_MUTEX;
	if ((invocation_lock = (dcpl_mutex_t*)dcpl_mutex_init()) == NULL)
		return DPWS_ERR_CREATING_MUTEX;
	if ((handle_lock = (dcpl_mutex_t*)dcpl_mutex_init()) == NULL)
		return DPWS_ERR_CREATING_MUTEX;
	if (createHandlePool(
					DC_MEM_REGISTRY,
					&registryPool,
					REGISTRY_POOL_NB_BUCKETS,
					handle_lock,
					&releaseHandleHook
					))
		return DPWS_ERR_EOM;
	http_endpoint.invoc_list = new_invocation_list();
	init_subsc_managers();
	return DPWS_OK;
}

void uninit_registry()
{
	if (reg_conf_lock)
	{
		deleteHandlePool(&registryPool);
		dcpl_mutex_delete(reg_conf_lock);
		dcpl_mutex_delete(invocation_lock);
		dcpl_mutex_delete(handle_lock);
		DA_FREE(&http_endpoint.invoc_list->ports, NULL);
	  	DC_FREE(DC_MEM_REGISTRY, http_endpoint.invoc_list);
	  	reg_conf_lock = NULL;	// sentinel
	  	uninit_subsc_managers();
	}
}

/*----------------------------------------------------------------------------*\
 *                          INTERNAL API: RUNTIME                             *
\*----------------------------------------------------------------------------*/

struct service_port * find_endpoint_port(const char * address)
{
	DA_TYPED(sp)* servPorts = NULL;
	const char * service_context_path = NULL;
	int i;

	if ((service_context_path = is_physical_address(address)))	// Test if the address is a physical http address or a logical UUID
	{
		if ((address = strchr(service_context_path, '/')))
		{
			service_context_path = address + 1;
			if (*service_context_path == '/')
			{
				service_context_path++;
				if ((address = strchr(service_context_path, '/')))	// jump host
					service_context_path = address + 1;
				else
					return NULL;
			}
		}
		else
			return NULL;
	}
	else if (!strncmp(address, UUID_SCHEME, UUID_SCHEME_SIZE))
		service_context_path = address + UUID_SCHEME_SIZE;
	else
		service_context_path = (address[0] == '/' ? address + 1 : address);

	if (service_context_path[0] == 0)
		return NULL;

	// find the corresponding service port using dichotomy
	// look first the upper bound because the integral division may prevent convergence on it
	servPorts = &checkout_invocation_list(http_endpoint.invoc_list)->ports;

	i = DA_SORTED_FIND(servPorts, port_context_path_cmp, service_context_path);

	if (i >= 0)
	{
		struct service_port * p_sport = *DA_GET(servPorts,	i);
		// Assert(p_sport != NULL)
		p_sport->invoc_bind_list = http_endpoint.invoc_list; // set the list where the port was found (one today)
		return p_sport;
	}
	// NOTE: not protected since only device can conflict with the device_up field (other fields are protected by this field)

	release_invocation_list(http_endpoint.invoc_list);
	return NULL;
}

void release_endpoint_port(struct service_port * endpoint)
{
	dcpl_mutex_lock(invocation_lock);
	release_invocation_list(endpoint->invoc_bind_list);
	dcpl_mutex_unlock(invocation_lock);
}

struct service_endpoint * get_endpoint(short href)
{
	struct service_endpoint * endpoint = NULL;
	if (href >= 0) {
		handle_s * h = getHandle(&registryPool, href);
		switch (h->type)
		{
			case SERVICE_ENDPOINT_TYPE:
				endpoint = (struct service_endpoint *)h->pObject;
				break;
			case SERVICE_PORT_TYPE:	// NOTE: added for compatibility reasons in notify. Could remove in v3
				endpoint = ((struct service_port *)h->pObject)->service;
				break;
		}
	}
	return endpoint;
}

DA_TYPED(pqn) * get_service_types(struct service_endpoint * service)
{
	return &((struct service_class *)getObject(&registryPool, service->hrefServiceClass))->types;
}

DA_TYPED(pqn) * get_device_types(struct dpws * dpws, struct device * device, DA_TYPED(pqn) * all_types, DC_BOOL safe)
{
	struct prefixed_qname * pqn;
	DA_TYPED(pqn) * types;

	if (safe)
		types = &((struct service_class *)getObject(&registryPool, device->super->hrefServiceClass))->types;
	else
		types = &((struct service_class *)getObjectUnsafe(&registryPool, device->super->hrefServiceClass))->types;
	memcpy(all_types, types, sizeof(DA_TYPED(pqn)));
	pqn = DA_GET(all_types, all_types->nb);
	pqn->prefix = WDP_PREFIX;
	pqn->qname.lname = WDP_DEVICE_TYPE;
	if (dpws->protocols)
		pqn->qname.ns = dpws->protocols->wdp_uri;
	else
		pqn->qname.ns = DPWS10_WDP_URI;
	all_types->nb++;	// hack: uses the null boundary to store the device type

	return all_types;
}

int count_tns_wsdl(struct service_endpoint *service)
{
	struct service_class * servClass = (struct service_class *)getObject(&registryPool, service->hrefServiceClass);
	return servClass->wsdl_files.nb;
}

void clean_registry()
{
	if (reg_conf_lock != NULL)
	{
		dcpl_mutex_lock(reg_conf_lock);

		registry_cfg.boot_seq = 0xFFFFFFFF;
		registry_cfg.dev_metadata_hook = NULL;
		registry_cfg.server_up = DC_FALSE;
		registry_cfg.tcp_listen = DC_TRUE;
		registry_cfg.tcp_backlog = HTTP_BACKLOG_DEFAULT;
		http_endpoint.port = HTTP_PORT_DEFAULT;
		hidden_device_href = -1;
		DC_FREE(DC_MEM_REGISTRY, registry_cfg.preferred_lang);
		registry_cfg.preferred_lang = NULL;

		// delete registry objects
		DPWSLOG(DC_REGISTRY, "# Release device handles\n");
		deleteHandles(&registryPool, DEVICE_INSTANCE_TYPE);	// should destroy in the right order.
		DPWSLOG(DC_REGISTRY, "# Release all handles\n");
		deleteHandles(&registryPool, ANY_TYPE);	// bleed other objects
		DA_EMPTY(&http_endpoint.invoc_list->ports, NULL);

		dcpl_mutex_unlock(reg_conf_lock);
	}
}

char* get_transport_address(struct dpws* dpws, const char * host, short hrefServicePort)
{
	struct service_port * port;
	char* result = NULL;

	port = (struct service_port *)getTypedObject(&registryPool, hrefServicePort, SERVICE_PORT_TYPE);
	if (!port) {
		dpws->err = DPWS_ERR_NO_HANDLE_FOUND;
		return NULL;
	}

	if (port->context_path && is_physical_address(port->context_path))
		result = dpws ? port->context_path: DC_STRDUP(DC_MEM_API, port->context_path);
	else
	{
		size_t len_addr;
		char * format;

		if (strchr(host, '.'))	// IPv4
		{
			format = "%s//%s:%d/%s";
			len_addr = 10; // port is 5 + "//:/\0"
		}
		else
		{
			format = "%s//[%s]:%d/%s";
			len_addr = 12; // port is 5 + "//[]:/\0"
		}

		len_addr += strlen(port->network_endpoint->transport->scheme) + strlen(host) + strlen(port->context_path);

		if (dpws)
			result = (char*)DC_MSG_MALLOC(DC_MEM_REGISTRY, dpws, len_addr);
		else
			result = (char*)DC_MALLOC(DC_MEM_API, len_addr);

		if (result)
			sprintf(result, format, port->network_endpoint->transport->scheme, host, port->network_endpoint->port, port->context_path);
		else if (dpws)
			dpws->err = DPWS_ERR_EOM;
	}

	return result;
}

int browse_service_ports(short href, service_port_cbk hook, void * param)
{
	DA_TYPED(href) * service_port_hrefs = find_service_endpoint_ports(href);
	if (!service_port_hrefs)
		return -1;
	return DA_BROWSE(service_port_hrefs, hook, param) == service_port_hrefs->nb ? 0 : -1;
}

int advertize_enabled_devices()
{
	int ret = DPWS_OK;
	/* schedule hellos for enabled devices */
	dcpl_mutex_lock(reg_conf_lock);
	handlePoolIterate(&registryPool, DEVICE_INSTANCE_TYPE, (handle_cbk)advertize_enabled_device, &ret);
	registry_cfg.server_up = DC_TRUE;
	dcpl_mutex_unlock(reg_conf_lock);
	return ret;
}

void shutdown_enabled_devices()
{
	/* schedule byes for enabled devices */
	dcpl_mutex_lock(reg_conf_lock);
	handlePoolIterate(&registryPool, DEVICE_INSTANCE_TYPE, (handle_cbk)shutdown_enabled_device, NULL);
	dcpl_mutex_unlock(reg_conf_lock);
}

struct wsa_endpoint_ref * build_local_endpoint_ref(struct dpws* dpws, struct wsa_endpoint_ref * dest, const char * host, short service_port_href)
{
	char * Xaddr = NULL;

	dcpl_mutex_lock(reg_conf_lock);
	if ((Xaddr = get_transport_address(dpws, host, service_port_href)))
	{
		dpws_default_wsa_endpoint_ref(dest);
		dest->address = Xaddr;
	}
	else
		dest = NULL;
	dcpl_mutex_unlock(reg_conf_lock);

	return dest;
}

/*----------------------------------------------------------------------------*\
 *                     USER API: REGISTRY CONFIGURATION                       *
\*----------------------------------------------------------------------------*/
static short create_random_service_port()
{
	char * uuid_buf = (char*)DC_MALLOC(DC_MEM_REGISTRY, UUID_STRING_SIZE);

	if (!uuid_buf)
		return DPWS_ERR_EOM;

	return create_service_port(uuid_create(uuid_buf, dcpl_mac_address));
}

short dpws_create_service_port()
{
	short href = -1;
	dcpl_mutex_lock(reg_conf_lock);
	href = create_random_service_port();
	dcpl_mutex_unlock(reg_conf_lock);
	return href;
}

int dpws_bind_service(short hrefService, short hrefServicePort)
{
	int ret = DPWS_OK;

	DC_CHECK_PARAM(hrefService >= 0 && hrefServicePort >= 0);

	dcpl_mutex_lock(reg_conf_lock);
	ret = bind_service(hrefService, hrefServicePort);
	dcpl_mutex_unlock(reg_conf_lock);

	return ret;
}

int dpws_release_handle(short handleRef)
{
	int ret = DPWS_OK;

	DC_CHECK_PARAM(handleRef >= 0);

	dcpl_mutex_lock(reg_conf_lock);
	ret = releaseHandle(&registryPool, handleRef);
	DPWSLOG1(DC_REGISTRY, "- Handle %d released by API\n", handleRef);
	dcpl_mutex_unlock(reg_conf_lock);
	return ret;
}

short dpws_get_handle_use_count(short href)
{
	DC_CHECK_PARAM(href >= 0);
	return getHandleUseCount(&registryPool, href);
}

int dpws_release_handles(short* hrefs, int len)
{
	int ret = DPWS_OK, i;

	DC_CHECK_PARAM(hrefs && len > 0);

	dcpl_mutex_lock(reg_conf_lock);
	for (i = 0; ret == DPWS_OK && i < len; i++) {
		ret = releaseHandle(&registryPool, hrefs[i]);
		DPWSLOG1(DC_REGISTRY, "- Handle %d released by API\n", hrefs[i]);
	}
	dcpl_mutex_unlock(reg_conf_lock);
	return ret;
}

static short create_device_model()
{
	short href = -1;
	struct device_model *model = (struct device_model*)DC_MALLOC(DC_MEM_REGISTRY, sizeof(struct device_model));
	if (!model)
		return DPWS_ERR_EOM;

	// init device model
	memset(model, 0, sizeof(struct device_model));
	DA_INIT(short, &model->service_classes, DC_MEM_REGISTRY, p_default_allocator, DISCOVERY_ALLOC_INCREMENT);
	DA_INIT(struct localized_string, &model->manufacturers, DC_MEM_REGISTRY, p_default_allocator, 1);
	DA_INIT(struct localized_string, &model->model_names, DC_MEM_REGISTRY, p_default_allocator, 1);
	if (!(model->id = (char*)DC_MALLOC(DC_MEM_REGISTRY, 6))) {	// max string size for short
		href = DPWS_ERR_EOM;
		goto error;
	}

	href = createHandle(&registryPool, DEVICE_MODEL_TYPE, model);
	DPWSLOG1(DC_REGISTRY, "+ Handle for device model %d created\n", href);
	if (href < 0)
		DC_FREE(DC_MEM_REGISTRY, model);
	else
		sprintf(model->id, "%d", href);

	return href;

error:
	freeDeviceModelStruct(model);
	return href;
}

short dpws_create_device_model()
{
	short href = -1;
	dcpl_mutex_lock(reg_conf_lock);
	href = create_device_model();
	dcpl_mutex_unlock(reg_conf_lock);
	return href;
}

short dpws_create_service_class()
{
	short href = -1;

	dcpl_mutex_lock(reg_conf_lock);
	href = create_service_class();
	dcpl_mutex_unlock(reg_conf_lock);

	return href;
}

int dpws_register_service_class(short deviceModelHandleRef, short hrefServiceClass)
{
	struct service_class * servClass;
	struct device_model * devModel;
	short * phref;
	int ret = DPWS_OK;

	DC_CHECK_PARAM(deviceModelHandleRef >= 0 && hrefServiceClass >= 0);

	dcpl_mutex_lock(reg_conf_lock);
	servClass = (struct service_class *)checkoutObject(&registryPool, hrefServiceClass);
	DPWSLOG1(DC_REGISTRY, "++ Handle for service class %d checked out by hosted service\n", hrefServiceClass);

	if (!servClass) {
		ret = DPWS_ERR_NO_HANDLE_FOUND;
		goto error;
	}

	// check that service class has types and handlers
	if (servClass->types.nb == 0 || servClass->funcs.nb == 0) {
		ret = DPWS_ERR_UNCOMPLETE_SERVICE;
		goto error;
	}

	devModel = (struct device_model *)getObject(&registryPool, deviceModelHandleRef);
	if (!devModel) {
		ret = DPWS_ERR_NO_HANDLE_FOUND;
		goto error;
	}

	// create device class/service class association
	if (!(phref = DA_ADD(&devModel->service_classes))) {
		ret = DPWS_ERR_EOM;
		goto error;
	}

	*phref = hrefServiceClass;
	goto exit;

error:
	releaseHandle(&registryPool, hrefServiceClass);
	DPWSLOG1(DC_REGISTRY, "- Service class handle %d released on error\n", hrefServiceClass);

exit:
	dcpl_mutex_unlock(reg_conf_lock);
	return ret;
}

short dpws_create_custom_device(unsigned short localID, short hrefModel)
{
	short href = -1;

	DC_CHECK_PARAM(hrefModel >= -1);

	dcpl_mutex_lock(reg_conf_lock);
	href = create_custom_device(localID, hrefModel);
	dcpl_mutex_unlock(reg_conf_lock);
	return href;
}

short dpws_create_hosted_service(short deviceHandleRef, short hrefClass)
{
	short href = -1;
	handle_s *handle;
	struct device * device;

	DC_CHECK_PARAM(deviceHandleRef >= 0 && hrefClass >= 0);

	dcpl_mutex_lock(reg_conf_lock);
	handle = getHandle(&registryPool, hrefClass);
	if (!handle)
		goto exit;

	device = (struct device*)getObject(&registryPool, deviceHandleRef);
	if (device){
	 	if (STARTED_DEVICE(device))
			href = DPWS_ERR_IMMUTABLE_USED_OBJECT;
		else {
			struct service_endpoint * se =  create_hosted_service(device, hrefClass);
			href = (se ? se->href : DPWS_ERR_EOM);
		}
	}
	else
		href = DPWS_ERR_NO_HANDLE_FOUND;

exit:
	dcpl_mutex_unlock(reg_conf_lock);
	return href;
}

short dpws_create_endpoint()
{
	short ret = -1, href = -1, hrefDevice = -1, hrefEndpoint = -1;
	struct device* hiddenDev = NULL;
	struct service_endpoint * se;

	dcpl_mutex_lock(reg_conf_lock);

	// create the hidden device if necessary
	if (hidden_device_href < 0)
	{
		href = create_device_model();
		hidden_device_href = create_custom_device(RANDOM_DEVICE_ID, href);
		if (hidden_device_href < 0) {
			ret = hidden_device_href; // ! lock
			goto exit;
		}
	}
	hiddenDev = (struct device *)getObject(&registryPool, hidden_device_href);
	hrefDevice = (STARTED_DEVICE(hiddenDev) ? clone_device(hidden_device_href) : hidden_device_href);

	href = create_service_class();		// NOTE: a little heavy but needed to carry handlers
	if (href < 0)
		goto exit;

	se = create_hosted_service((struct device *)getObject(&registryPool, hrefDevice), href);
	hrefEndpoint = se ? se->href : -1;
	releaseHandle(&registryPool, href);	// now service class is used or useless
	if (hrefEndpoint < 0)
		goto exit;

	href = create_random_service_port();
	if (href < 0)
		goto exit;

	bind_service(hrefEndpoint, href);	// no error should occur in this context
	releaseHandle(&registryPool, href);	// now service port is used

	if (hrefDevice == hidden_device_href)
		ret = enable_device(hidden_device_href);
	else {
		ret = replace_device(hidden_device_href, hrefDevice);
		hidden_device_href = hrefDevice;
	}

exit:
	dcpl_mutex_unlock(reg_conf_lock);
	return ret ? ret : hrefEndpoint;
}

static short find_handle(short typeFilter, handle_cbk callback, char * id)
{
	handle_s * h;
	short ret = -1;
	dcpl_mutex_lock(reg_conf_lock);
	h = handlePoolIterate(&registryPool, typeFilter, callback, id);
	if (h) {
		ret = h->handleRef;
		checkoutHandle(&registryPool, ret);
	}
	dcpl_mutex_unlock(reg_conf_lock);
	return ret;
}
short dpws_get_device_model_handle(char * id)
{
	DC_CHECK_PARAM(id);
	return find_handle(DEVICE_MODEL_TYPE, (handle_cbk)device_model_equals, id);
}

short dpws_get_service_class_handle(char * id)
{
	DC_CHECK_PARAM(id);
	return find_handle(SERVICE_CLASS_TYPE, (handle_cbk)service_class_equals, id);
}

short dpws_get_device_handle(char * uuid)
{
	DC_CHECK_PARAM(uuid);
	return find_handle(DEVICE_INSTANCE_TYPE, (handle_cbk)device_equals, uuid);
}

static short find_service(short deviceHandleRef, da_browse_cbk cbk, void * cbk_param)
{
	short href = -1;
	int i;
	struct device *device;

	dcpl_mutex_lock(reg_conf_lock);

	device = (struct device*)getObject(&registryPool, deviceHandleRef);
	if (!device)
		return -1;

	i = DA_BROWSE(&device->hosted_services, cbk, cbk_param);
	if (i < device->hosted_services.nb) {
		href = * (short *)GET_ENTRY(&device->hosted_services, i);
		checkoutHandle(&registryPool, href);
	}
	dcpl_mutex_unlock(reg_conf_lock);

	return href;
}

short dpws_get_service_handle(short deviceHandleRef, char * service_id)
{
	DC_CHECK_PARAM(deviceHandleRef >= 0 && service_id);
	return find_service(deviceHandleRef, (da_browse_cbk)find_service_hook, service_id);
}

short dpws_get_service_by_class(short deviceHandleRef, short hrefServiceClass)
{
	DC_CHECK_PARAM(deviceHandleRef >= 0 && hrefServiceClass >= 0);
	return find_service(deviceHandleRef, (da_browse_cbk)find_service_hook_by_class, &hrefServiceClass);
}

short dpws_create_device(unsigned short localID, short hrefModel)
{
	short hRef;

	DC_CHECK_PARAM(hrefModel >= -1);

	dcpl_mutex_lock(reg_conf_lock);
	hRef = create_custom_device(localID, hrefModel);
	if (hRef >= 0 && hrefModel >= 0)
	{
		struct device * device;
		struct device_model * deviceModel;
		struct service_endpoint * se;
		short hrefServiceClass, hrefServiceEndpoint, hrefServicePort;
		int i;

		device = (struct device *)getObject(&registryPool, hRef);
		deviceModel = (struct device_model *)getObject(&registryPool, hrefModel);

		// loop over service classes
		for(i = 0; i < deviceModel->service_classes.nb; i++) {
			hrefServiceClass = *(short *)GET_ENTRY(&deviceModel->service_classes, i);
			se = create_hosted_service(device, hrefServiceClass);
			hrefServiceEndpoint = (se ? se->href : -1);
			if (hrefServiceEndpoint < 0 || (hrefServicePort = create_random_service_port()) < 0)
			{
				releaseHandle(&registryPool, device->href);
				break;
			}
			bind_service(hrefServiceEndpoint, hrefServicePort);
			releaseHandle(&registryPool, hrefServicePort);	// hold by the endpoint
		}
	}
	dcpl_mutex_unlock(reg_conf_lock);
	return hRef;
}

struct wsa_endpoint_ref * dpws_create_device_endpoint_ref(char * uuid)
{
	struct wsa_endpoint_ref *dev_endpoint = NULL;

	DC_CHECK_PARAM_RETURN(uuid, NULL);

	if (uuid) {
		dev_endpoint = (struct wsa_endpoint_ref*)DC_MALLOC(DC_MEM_API, sizeof(struct wsa_endpoint_ref));
		if (dev_endpoint) {
			dpws_default_wsa_endpoint_ref(dev_endpoint);
			if (!(dev_endpoint->address = DC_STRDUP(DC_MEM_API, uuid)))
			{
				DC_FREE(DC_MEM_API, dev_endpoint);
				dev_endpoint = NULL;
			}
		}
	}
	return dev_endpoint;
}

short dpws_get_default_service_port(short href)
{
	short ret = -1;
	DA_TYPED(href) * result;

	DC_CHECK_PARAM(href >= 0);

	dcpl_mutex_lock(reg_conf_lock);
	result = find_service_endpoint_ports(href);
	if (result && result->nb > 0) {
		ret = *DA_GET(result, 0);
		checkoutHandle(&registryPool, ret);
	}
	dcpl_mutex_unlock(reg_conf_lock);
	return ret;
}

struct wsa_endpoint_ref * dpws_get_local_endpoint_ref(short service_port_href)
{
	struct wsa_endpoint_ref * epr =
		(struct wsa_endpoint_ref *)DC_MALLOC(DC_MEM_API, sizeof(struct wsa_endpoint_ref));
	if (epr)
	{
		// The configuration API is used by the metadata server
		dpws_default_wsa_endpoint_ref(epr);
		dcpl_mutex_lock(reg_conf_lock);
		epr->address = get_transport_address(NULL, dc_netif_addrs[0].ip_address, service_port_href);
		dcpl_mutex_unlock(reg_conf_lock);
		if (!epr->address) {
			DC_FREE(DC_MEM_API, epr);
			epr = NULL;
		}
	}
	return epr;
}

char* dpws_get_transport_address(short hrefServPort)
{
	char * XAddr = NULL;
	dcpl_mutex_lock(reg_conf_lock);
	if (dc_netif_nb_addrs > 0)
		XAddr = get_transport_address(NULL, dc_netif_addrs[0].ip_address, hrefServPort);
	dcpl_mutex_unlock(reg_conf_lock);
	return XAddr;
}

char ** dpws_get_transport_addresses(short href_service_port, int * res_size)
{
	char ** XAddrs = NULL;

	DC_CHECK_PARAM_RETURN(href_service_port >= 0 && res_size, NULL);

	XAddrs = DC_MALLOC(DC_MEM_API, dc_netif_nb_addrs * sizeof(char *));

	if (XAddrs)
	{
		dcpl_mutex_lock(reg_conf_lock);

		for (*res_size = 0; *res_size < dc_netif_nb_addrs; (*res_size)++)
		{
			if (!(XAddrs[*res_size] = get_transport_address(NULL, dc_netif_addrs[*res_size].ip_address, href_service_port)))
				break;
		}
		dcpl_mutex_unlock(reg_conf_lock);
	}

	return XAddrs;
}

int dpws_get_service_handles(short deviceHandleRef, short* hrefs, int* len)
{
	struct device *device;
	struct service_endpoint *wEndpoint;
	int count = 0, res = DPWS_OK, i;

	DC_CHECK_PARAM(deviceHandleRef >= 0 && hrefs && len && *len > 0);

	dcpl_mutex_lock(reg_conf_lock);
	device = (struct device*)getObject(&registryPool, deviceHandleRef);
	if (!device) {
		res = -1;
		goto exit;
	}

	for (i = 0; i < device->hosted_services.nb; i++) {
		wEndpoint = (struct service_endpoint *) getObject(&registryPool, * (short *)GET_ENTRY(&device->hosted_services, i));
		if (wEndpoint->service_id) {
			if (count < *len) {
				hrefs[count] = wEndpoint->href;
				checkoutHandle(&registryPool, wEndpoint->href);
			}
			else {
				res = DPWS_ERR_MORE_RESULTS;
				goto exit;
			}
			count++;
		}
	}
	*len = count;
exit:
	dcpl_mutex_unlock(reg_conf_lock);
	return res;
}

static DC_BOOL checkout_cbk(short * p_href, void * param)
{
	checkoutHandle(&registryPool, *p_href);
	return DC_FALSE;
}

int dpws_get_service_port_handles(short endpoint_href, short* hrefs, int* len)
{
	int res = DPWS_OK;
	DA_TYPED(href) * result;

	DC_CHECK_PARAM(endpoint_href >= 0 && hrefs && len && *len > 0);

	dcpl_mutex_lock(reg_conf_lock);
	result = find_service_endpoint_ports(endpoint_href);
	if (!result) {
		res = -1;
		goto exit;
	}
	if (result->nb > *len)
		res = DPWS_ERR_MORE_RESULTS;
	else
		*len = result->nb;

	// checkout handles
	DA_BROWSE(result, checkout_cbk, NULL);
	memcpy(hrefs, result->tab, *len * sizeof(short));

exit:
	dcpl_mutex_unlock(reg_conf_lock);
	return res;
}

static int get_handles(short* hrefs, int* len, short h_type, filter_cbk_t filter_cbk, void * filter_param)
{
	int res = DPWS_OK;
	struct gather_handles_info gh_info;

	gh_info.hrefs = hrefs;
	gh_info.max_len = *len;
	gh_info.nb = 0;
	gh_info.filter = filter_cbk;
	gh_info.filter_param = filter_param;

	dcpl_mutex_lock(reg_conf_lock);
	if (handleIterate(&registryPool, h_type, (handle_it_cbk)gather_handles, &gh_info))
		res = DPWS_ERR_MORE_RESULTS;
	*len = gh_info.nb;
	dcpl_mutex_unlock(reg_conf_lock);

	return res;
}

static DC_BOOL gather_visible_devices_cbk(struct device * device, void * dummy)
{
	return API_VISIBLE_DEVICE(device) ? DC_TRUE : DC_FALSE;
}

int dpws_get_device_handles(short* hrefs, int* len)
{
	DC_CHECK_PARAM(hrefs && len && *len > 0);
	return get_handles(hrefs, len, DEVICE_INSTANCE_TYPE, (filter_cbk_t)gather_visible_devices_cbk, NULL);
}

static DC_BOOL gather_public_services_class_cbk(struct service_class * s_class, void * dummy)
{
	return s_class->id ? DC_TRUE : DC_FALSE;
}

int dpws_get_service_class_handles(short* hrefs, int* len)
{
	DC_CHECK_PARAM(hrefs && len && *len > 0);
	return get_handles(hrefs, len, SERVICE_CLASS_TYPE, (filter_cbk_t)gather_public_services_class_cbk, NULL);
}

// user data protection is user business...
void * dpws_get_device_user_data(struct dpws* dpws)
{
	void * uData = NULL;
	struct service_endpoint * endpoint;

	DC_CHECK_PARAM_NO_RC(dpws->err, dpws, NULL);

	endpoint = get_endpoint(dpws->href_endpoint);
	if (endpoint)
		uData = (void *) endpoint->device->user_data;
	return uData;
}

void * dpws_get_endpoint_user_data(struct dpws* dpws)
{
	void * uData = NULL;
	struct service_endpoint * endpoint;

	DC_CHECK_PARAM_NO_RC(dpws->err, dpws, NULL);

	endpoint = get_endpoint(dpws->href_endpoint);
	if (endpoint)
		uData = (void *) endpoint->user_data;
	return uData;
}

void * dpws_get_service_class_user_data(struct dpws* dpws)
{
	void * uData = NULL;
	struct service_endpoint * endpoint;

	DC_CHECK_PARAM_NO_RC(dpws->err, dpws, NULL);

	endpoint = get_endpoint(dpws->href_endpoint);
	if (endpoint) {
		struct service_class * serv_class = (struct service_class *)getObject(&registryPool, endpoint->hrefServiceClass);
		if (serv_class)
			uData = (void *) serv_class->user_data;
	}
	return uData;
}

int dpws_enable_device(short deviceHandleRef)
{
	int ret = DPWS_OK;

	DC_CHECK_PARAM(deviceHandleRef >= 0);

	dcpl_mutex_lock(reg_conf_lock);
	ret = enable_device(deviceHandleRef);
	dcpl_mutex_unlock(reg_conf_lock);
	return ret;
}

int dpws_disable_device(short deviceHandleRef)
{
	int ret = DPWS_OK;
	struct device *device;

	DC_CHECK_PARAM(deviceHandleRef >= 0);

	dcpl_mutex_lock(reg_conf_lock);

	device = (struct device *)checkoutObject(&registryPool, deviceHandleRef);	// checked out to keep the object until byes are sent
	if (!device) {
		ret = DPWS_ERR_NO_HANDLE_FOUND;
		goto exit;
	}
	if (device->status != ONLINE) {
		ret = DPWS_ERR_ILLEGAL_STATE;
		goto exit;
	}

	if (registry_cfg.server_up && get_device_mode(device) == DPWS_DEVICE_USE) {	// else scheduling will be done by dpws_stop_server()
		// NOTE send_eventing_end will pipe the BYE
		device->status = OFFLINE;
		dc_reactor_register_timer(&reactor, DC_FALSE, 0, 0, shutdown_device_callback, device, NULL, NULL);
		/* Send a dummy connection on the TCP socket to wake up accept */
		ret = dc_reactor_interrupt(&reactor);
	}

	// substitute service port list
	ret = substitute_invocation_list(device, NULL);
	DPWSLOG1(DC_REGISTRY, "- Device handle %d released for device when set offline.\n", deviceHandleRef);

exit:
	dcpl_mutex_unlock(reg_conf_lock);
	return ret;
}

// NOTE : no dpws_remove_device(short href). Release handle is enough.

short dpws_clone_device(short href)
{
	int ret = DPWS_OK;

	DC_CHECK_PARAM(href >= 0);

	dcpl_mutex_lock(reg_conf_lock);
	ret = clone_device(href);
	dcpl_mutex_unlock(reg_conf_lock);
	return ret;
}

int dpws_replace_device(short href_online, short href_offline)
{
	int ret = DPWS_OK;

	DC_CHECK_PARAM(href_online >= 0 && href_offline >= 0);

	dcpl_mutex_lock(reg_conf_lock);
	ret = replace_device(href_online, href_offline);
	dcpl_mutex_unlock(reg_conf_lock);
	return ret;
}

int dpws_delete_hosted_service(short serviceHandleRef)
{
	DA_TYPED(href) * hs;
	struct service_endpoint * service = NULL;
	int ret = DPWS_OK;

	DC_CHECK_PARAM(serviceHandleRef >= 0);

	dcpl_mutex_lock(reg_conf_lock);
	service = (struct service_endpoint *)getObject(&registryPool, serviceHandleRef);
	if (!service) {
		ret = DPWS_ERR_NO_HANDLE_FOUND;
		goto exit;
	}
	if (STARTED_DEVICE(service->device)) {
		ret = DPWS_ERR_ILLEGAL_STATE;
		goto exit;
	}
	hs = &service->device->hosted_services;
	DA_REMOVE(hs, DA_BROWSE(hs, find_href_hook, &serviceHandleRef), NULL);
	ret = releaseHandle(&registryPool, serviceHandleRef);
	DPWSLOG1(DC_REGISTRY, "- Service handle %d released on hosted service deletion.\n", ret);

exit:
	dcpl_mutex_unlock(reg_conf_lock);
	return ret;
}


static DC_BOOL gather_services_by_class_filter_cbk(struct service_endpoint * service, short * href_class)
{
	return service->hrefServiceClass == *href_class ? DC_TRUE : DC_FALSE;
}

int dpws_get_services_by_class(short href, short* hrefs, int* len)
{
	int res = DPWS_OK;
	struct gather_handles_info sInfo;

	DC_CHECK_PARAM(href >= 0 && hrefs && len && *len > 0);

	sInfo.hrefs = hrefs;
	sInfo.max_len = *len;
	sInfo.nb = 0;
	sInfo.filter = (filter_cbk_t)gather_services_by_class_filter_cbk;
	sInfo.filter_param = &href;

	dcpl_mutex_lock(reg_conf_lock);
	if (handleIterate(&registryPool, SERVICE_ENDPOINT_TYPE, (handle_it_cbk)gather_handles, &sInfo))
		res = DPWS_ERR_MORE_RESULTS;
	*len = sInfo.nb;
	dcpl_mutex_unlock(reg_conf_lock);

	return res;
}

short dpws_get_service_device(short href)
{
	struct service_endpoint * service;
	short ret = -1;

	DC_CHECK_PARAM(href >= 0);

	dcpl_mutex_lock(reg_conf_lock);
	service = (struct service_endpoint *)getObject(&registryPool, href);
	if (!service) {
		ret = DPWS_ERR_NO_HANDLE_FOUND;
		goto exit;
	}
	ret = service->device->href;
	checkoutHandle(&registryPool, ret);

exit:
	dcpl_mutex_unlock(reg_conf_lock);
	return ret;
}
