/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2091 $
|                     $Date: 2009-02-17 15:38:44 +0100 (mar, 17 fév 2009) $
\*============================================================================*/

/*******************************************************************************
*                    WS-Discovery Protocol Implementation                      *
*******************************************************************************/
#ifndef DISCOVERY_H
# define DISCOVERY_H

# include "dcCOMN_Tools.h"
# include "dcCOMN_Handle.h"
# include "dcDPWS_Reactor.h"
# include "dcDPWS_Registry.h"
# include "dcDPWS_Transport.h"
# include "dcDCPL_Net.h"

/*----------------------------------------------------------------------------*\
 *                                SOAP over UDP                               *
\*----------------------------------------------------------------------------*/
/** Retrieves the delay for a SOAP over UDP message.
 * @param[in,out] send_nb The last message number. incremented by the function.
 * @param nb_msg The number of total repetitions.
 * @param has_initial_delay A boolean that indicates if the message uses an
 * initial delay before the first message.
 * @param last_delay The last returned delay used to calculate the next value.
 * @return The delay for the next message. If the maximum number was reached,
 * 0xFFFFFFFF is returned.
 */
uint32_t get_soap_over_udp_schedule(short * send_nb, int nb_msg, int has_initial_delay, uint32_t last_delay);

/*******************************************************************************
*                                   MODULE API                                 *
*******************************************************************************/

/** Initialize structures for asynchronous WS-Discovery message sending.
 * @return A DPWS error code.
 */
int init_discovery();

/** Cleans WS-Discovery module structures.
 * Created device are destroyed and configuration parameters reset.
 */
void uninit_discovery();

/** Programs a WS-Discovery message for sending.
 * @netif_addr DCPL structure identifying the network interface on which the
 * message will be sent.
 * @param device The pointer to the registry device structure.
 * @param msgType The WS-Discovery message type of the form DPWS_MSG_XXXXX.
 * @return A DPWS error code.
 */
int schedule_discovery_msg(dcpl_ip_addr_t * netif_addr, struct device *device, short msgType, struct dpws_protocols * protocols);

/** Changes the WS-Addressing "AppSequence InstanceID".
 * Protected with the configuration lock.
 * @param boot_seq Contains the requested new boot number.
 * @return DPWS_OK or DPWS_ERR_INVALID_PARAMETER if for instance if the supplied
 * boot number is out of sequence
 */
int new_instance_id(uint32_t boot_seq);

/*******************************************************************************
*                       WS-DISCOVERY STUB & SKELETON                           *
*******************************************************************************/
#ifdef SOAP_FMAC3
# undef SOAP_FMAC3
#endif
#define SOAP_FMAC3 static
# include "discoveryStub.h"

extern struct Namespace dpws10_discovery_snd_namespaces[];	/**< WS-Discovery sGOAP "send" namespace table for DPWS 1.0 */
extern struct Namespace dpws11_discovery_snd_namespaces[];	/**< WS-Discovery sGOAP "send" namespace table for DPWS 1.1 */

extern struct wsa_endpoint_ref * dp_EPR;	/**< Discovery proxy EPR if detected. */

/** Callback used for multicast WS-Discovery packets reception.
 * @param dpws The message context.
 * @return A DPWS error code.
 */
int wsd_mcast_recv(struct dpws *dpws);

/** Callback used for WS-Discovery probe match packets reception.
 * @param dpws The message context.
 * @return A DPWS error code.
 */
int wsd_ucast_recv_probematch(struct dpws *dpws);

/** Callback used for WS-Discovery resolve match packets reception.
 * @param dpws The message context.
 * @return A DPWS error code.
 */
int wsd_ucast_recv_resolvematch(struct dpws *dpws);

/** Callback that processes probe messages over HTTP (directed probes).
 * @param dpws The message context.
 * @return A DPWS error code.
 */
int wsd_serve_directed_probe(struct dpws *dpws);

/** Client function that performs Probe messages sending.
 * @param dpws The message context.
 * @param filter Scope and type information for server-side filtering.
 * @param message_id The message id (UUID) used for all repetitions of the
 * message.
 * @return A DPWS error code.
 */
int send_probe(struct dpws * dpws, discovery_filter_t * filter, const char * message_id);

/** Client function that performs Resolve messages sending.
 * @param dpws The message context.
 * @param discovery_endpoint The EPR containing the device UUID.
 * @param message_id The message id (UUID) used for all repetitions of the
 * message.
 * @return A DPWS error code.
 */
int send_resolve(struct dpws * dpws, struct wsa_endpoint_ref *discovery_endpoint, const char * message_id);

/** Client function that performs Directed Probe messages sending.
 * @param dpws The message context.
 * @param address The HTTP url of the device.
 * @param filter Scope and type information for server-side filtering.
 * @return A DPWS error code.
 */
char* call_directed_probe(struct dpws * dpws, char * address, discovery_filter_t * filter);

/** Client function that sends Probe messages to a Discovery Proxy.
 * @param dpws The message context.
 * @param proxy The discovery proxy EPR.
 * @param filter A structure containing scope and type information for
 * device filtering.
 * @param timeout Connection timeout in ms.
 * @return A DPWS error code.
 */
int call_discovery_proxy_probe(struct dpws* dpws, struct wsa_endpoint_ref * proxy, discovery_filter_t * filter, int timeout);

/** Client function that sends a resolve message to a Discovery Proxy.
 * @param dpws The message context.
 * @param proxy The discovery proxy EPR.
 * @param id The device ID to test.
 * @param timeout Connection timeout in ms.
 * @return A cached device proxy structure id found. NULL else.
 */
struct device_proxy * call_discovery_proxy_resolve(struct dpws* dpws, struct wsa_endpoint_ref * proxy, char * id, int timeout);

#endif
