/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 820 $
|                     $Date: 2008-03-13 14:17:02 +0100 (jeu., 13 mars 2008) $
\*============================================================================*/
#include "dc/dc_Dpws.h"
#include "dcCOMN_Tools.h"
#include "dcDPWS_Memory.h"
#include "dc/dc_ConnPool.h"
#include "dcDPWS_ConnPool.h"
#include "dcDCPL_Os.h"
#include "dcDPWS_Dpws.h"
#include "dcDCPL_Socket.h"
#include "dcDPWS_Transport.h"
#include "dcDPWS_Network.h"
#include "dcDPWS_Cache.h"


#define DEFAULT_POOL_SIZE 10
#define MAX_HOSTNAME_SIZE 256

struct pooled_connection
{
	char host[MAX_HOSTNAME_SIZE];
	int port;
	DCPL_SOCKET socket;
	uint64_t last_used_time;
	struct pooled_connection* next;
};

static struct pooled_connection connection_array[DEFAULT_POOL_SIZE];

struct connection_pool
{
	struct pooled_connection * connections;
	struct pooled_connection * free_connections;
	struct pooled_connection * pooled_connections;
	struct pooled_connection * connections_in_use;
	int n_used;
	int max_size;
	uint64_t max_idle_time;
	void * lock;
};

static struct connection_pool pool = { connection_array, NULL, NULL, NULL, 0, DEFAULT_POOL_SIZE, 10000, NULL };

/*------------------------- Static Functions prototypes ----------------------*/

static struct pooled_connection * get_pooled_connection(const char* host, int port);
static struct pooled_connection * get_free_connection(DCPL_SOCKET socket, const char* host, int port);
static struct pooled_connection * get_connection_in_use(DCPL_SOCKET socket);
static void store_connection_in_use(struct pooled_connection * conn);
static void store_pooled_connection(struct pooled_connection * conn);
static int close_pooled_connection(struct pooled_connection * conn);
static int unsafe_close_pooled_connection(struct pooled_connection * conn);

/*----------------------------------------------------------------------------*/

int dpws_init_connection_pool(int max_size, int max_idle_time)
{
	int status = DPWS_OK;

	DC_CHECK_PARAM(max_size > 0 && max_idle_time > 0);

	if (pool.lock == NULL) pool.lock = cache_lock;
	dcpl_mutex_lock(pool.lock);
	if (pool.pooled_connections != NULL || pool.connections_in_use != NULL) {
		dcpl_mutex_unlock(pool.lock);
		dpws_shutdown_connection_pool();
		dcpl_mutex_lock(pool.lock);
	}
	if (max_size > 0) {
		if (max_size > pool.max_size) {
			if (pool.connections != connection_array) {
				DC_FREE(DC_MEM_CONN_POOL, pool.connections);
			}
			pool.connections = (struct pooled_connection *)DC_MALLOC(DC_MEM_CONN_POOL, max_size * sizeof(struct pooled_connection));
			if (pool.connections == NULL)
				status = DPWS_ERR_EOM;
		}
		pool.max_size = max_size;
	}
	if (max_idle_time > 0) {
		pool.max_idle_time = max_idle_time * 1000;
	}
	dcpl_mutex_unlock(pool.lock);
	return status;
}

int dpws_use_connection_pool(struct dpws* dpws, DC_BOOL on)
{
	struct soap* soap = dpws_dpws2soap(dpws);

	DC_CHECK_PARAM(dpws);

	if (pool.lock == NULL) pool.lock = cache_lock;
	if (on) soap->omode |= SOAP_IO_KEEPALIVE;
	else soap->omode &= ~SOAP_IO_KEEPALIVE;
	return DPWS_OK;
}

int dpws_shutdown_connection_pool()
{
	struct pooled_connection* c;
	dcpl_mutex_lock(pool.lock);
	while ((c = pool.pooled_connections) != NULL) {
		pool.pooled_connections = c->next;
		unsafe_close_pooled_connection(c);
	}
	while ((c = pool.connections_in_use) != NULL) {
		pool.connections_in_use = c->next;
		unsafe_close_pooled_connection(c);
	}
	dcpl_mutex_unlock(pool.lock);
	return SOAP_OK;
}

DCPL_SOCKET dc_connpool_connect(const char *host, unsigned short port, int timeout, struct socket_data * data, struct dcpl_error * error)
{
	DCPL_SOCKET socket;
	struct pooled_connection * c;

	if (pool.lock == NULL) pool.lock = cache_lock;
	c = get_pooled_connection(host, port);
	if (c == NULL) {
		DPWSLOG2(DC_TRANSPORT, "Opening new pooled connection for %s:%d", host, port);
		// Connection not found in pool
		socket = network_tcp_connect(host, port, timeout, data, error);
		if (socket != DCPL_INVALID_SOCKET)
			c = get_free_connection(socket, host, port);
	} else {
		DPWSLOG2(DC_TRANSPORT, "Reusing connection for %s:%d", host, port);
		socket = c->socket;
	}
	if (c) {
		DPWSLOG2(DC_TRANSPORT, "Using pooled connection for %s:%d", c->host, c->port);
		store_connection_in_use(c);
	} else {
		DPWSLOG(DC_TRANSPORT, "Cannot pool connection: too many connections in use");
	}
	return socket;
}

int dc_connpool_closesocket(DCPL_SOCKET socket, int reuse, struct dcpl_error * error)
{
	struct pooled_connection * c;
	if (pool.lock == NULL) pool.lock = cache_lock;
	c = get_connection_in_use(socket);
	if (c != NULL) {
		if (reuse) {
			DPWSLOG2(DC_TRANSPORT, "Pooling connection for %s:%d", c->host, c->port);
			store_pooled_connection(c);
			return DPWS_OK;
		} else {
			DPWSLOG2(DC_TRANSPORT, "Closing pooled connection for %s:%d", c->host, c->port);
			return close_pooled_connection(c);
		}
	} else {
		DPWSLOG(DC_TRANSPORT, "Closing unmanaged connection");
		return dcpl_closesocket(socket, error);
	}
}

struct pooled_connection * get_pooled_connection(const char* host, int port)
{
	struct pooled_connection * c = pool.pooled_connections;
	struct pooled_connection * prev = NULL;
	struct pooled_connection * result = NULL;
	uint64_t curtime;
	curtime = dcpl_get_clock();

	dcpl_mutex_lock(pool.lock);
	while (c != NULL) {
		int remove = 0;
		struct pooled_connection * next = c->next;
		if (port == c->port && !strcmp(host, c->host)) {
			// pooled connection matches requested address
			remove = 1;
			result = c;
		} else if ((curtime - c->last_used_time) > pool.max_idle_time) {
			// pooled connection is too old;
			remove = 1;
		}
		if (remove) {
			if (prev != NULL) {
				prev->next = next;
			} else {
				pool.pooled_connections = next;
			}
			if (result && dcpl_tcp_socket_valid(result->socket)) {
				break;
			} else {
				// old or invalid connection
				result = NULL;
				DPWSLOG2(DC_TRANSPORT, "Closing connection in pool for %s:%d", c->host, c->port);
				unsafe_close_pooled_connection(c);
			}
		} else {
			prev = c;
		}
		c = next;
	}
	dcpl_mutex_unlock(pool.lock);
	return result;
}

struct pooled_connection * get_free_connection(DCPL_SOCKET socket, const char* host, int port)
{
	struct pooled_connection * result = NULL;
	dcpl_mutex_lock(pool.lock);
	if (pool.free_connections != NULL) {
		result = pool.free_connections;
		pool.free_connections = result->next;
	} else if (pool.n_used + 1 < pool.max_size) {
		result = &pool.connections[pool.n_used++];
	}
	dcpl_mutex_unlock(pool.lock);
	if (result != NULL) {
		result->socket = socket;
		strncpy(result->host, host, sizeof(result->host));
		result->host[strlen(host)] = '\0';
		result->port = port;
	}
	return result;
}

struct pooled_connection * get_connection_in_use(DCPL_SOCKET socket)
{
	struct pooled_connection* c = pool.connections_in_use;
	struct pooled_connection* prev = NULL;

	dcpl_mutex_lock(pool.lock);
	while (c != NULL) {
		if (c->socket == socket) {
			if (prev != NULL) {
				prev->next = c->next;
			} else {
				pool.connections_in_use = c->next;
			}
			break;
		}
		prev = c;
		c = c->next;
	}
	dcpl_mutex_unlock(pool.lock);
	return c;
}

void store_connection_in_use(struct pooled_connection * conn)
{
	if (conn) {
		dcpl_mutex_lock(pool.lock);
		conn->next = pool.connections_in_use;
		pool.connections_in_use = conn;
		dcpl_mutex_unlock(pool.lock);
	}
}

void store_pooled_connection(struct pooled_connection * conn)
{
	if (conn) {
		conn->last_used_time = dcpl_get_clock();
		dcpl_mutex_lock(pool.lock);
		conn->next = pool.pooled_connections;
		pool.pooled_connections = conn;
		dcpl_mutex_unlock(pool.lock);
	}
}


int close_pooled_connection(struct pooled_connection * conn)
{
	int status;
	dcpl_mutex_lock(pool.lock);
	status = unsafe_close_pooled_connection(conn);
	dcpl_mutex_unlock(pool.lock);
	return status;
}

int unsafe_close_pooled_connection(struct pooled_connection * conn)
{
	int status = SOAP_OK;
	if (conn->socket != SOAP_INVALID_SOCKET)
		status = dcpl_closesocket(conn->socket, NULL);
	conn->socket = SOAP_INVALID_SOCKET;
	conn->next = pool.free_connections;
	pool.free_connections = conn;
	return status;
}
