/*============================================================================*\
|                                                                              |
|                       SOA4D DPWSCore (C DPWS toolkit)                        |
|                                                                              |
|            ->>  Copyright 2004-2009 Schneider Electric SA <<-                |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 820 $
|                     $Date: 2008-03-13 14:17:02 +0100 (jeu., 13 mars 2008) $
\*============================================================================*/

/*******************************************************************************
*                           Common Configuration API                           *
*******************************************************************************/
#include "dcDPWS_Registry.h"
#include "dcDPWS_Cache.h"
#include "dcDPWS_SubscManager.h"


/*------------------------- Static Functions prototypes ----------------------*/

static DC_BOOL browse_device_use(short href, struct device * device, int * use);

/*----------------------------------------------------------------------------*/

static DC_BOOL browse_device_use(short href, struct device * device, int * use)
{
	*use = get_device_mode(device);
	return (*use == DPWS_DEVICE_USE) ? DC_TRUE : DC_FALSE;
}

int get_toolkit_mode()
{
	int use = DPWS_CLIENT_USE;
	handlePoolIterate(&registryPool, DEVICE_INSTANCE_TYPE, (handle_cbk)browse_device_use, &use);
	if (use == DPWS_CLIENT_USE)		// was not changed: client or discovery monitor
		use = (cache_cfg.bye_hook || cache_cfg.hello_hook) ? DPWS_DISCOVERY_USE : DPWS_CLIENT_USE;
	return use;
}

int dpws_set_ptr_att(short handleRef, int att, const void * value)
{
	int ret = DPWS_OK;
	struct device *device;
	struct service_endpoint *service;
	struct service_port *servicePort;
	handle_s *handle;

	DC_CHECK_PARAM(handleRef >= MIN_CONF_HANDLE && att > 0 && value);

	switch (handleRef)
	{
	case DC_REGISTRY_HANDLE:
		dcpl_mutex_lock(reg_conf_lock);
		if (registry_cfg.server_up) // used by more than the user
			ret = DPWS_ERR_IMMUTABLE_USED_OBJECT;
		else
			ret = set_registry_config_ptr_att(att, value, DC_TRUE);
		dcpl_mutex_unlock(reg_conf_lock);
		break;

	case DC_CACHE_HANDLE:
		dcpl_mutex_lock(cache_lock);
		ret = set_cache_config_ptr_att(att, value, DC_TRUE);
		dcpl_mutex_unlock(cache_lock);
		break;

	case DC_SUBSC_MANAGER_HANDLE:
		dcpl_mutex_lock(subsc_manager_lock);
		ret = set_subsc_manager_config_ptr_att(att, value, DC_TRUE);
		dcpl_mutex_unlock(subsc_manager_lock);
		break;

	default:
		dcpl_mutex_lock(reg_conf_lock);
		if (handleRef < 0 || !(handle = getHandle(&registryPool, handleRef)))
			ret = DPWS_ERR_NO_HANDLE_FOUND;
		else
			switch (handle->type)
			{
				case DEVICE_INSTANCE_TYPE:
					device = (struct device *)handle->pObject;
					if (registry_cfg.server_up && STARTED_DEVICE(device))
						ret = DPWS_ERR_IMMUTABLE_USED_OBJECT;
					else
						ret = set_device_ptr_att(device, att, value, DC_TRUE);
					break;
				case SERVICE_ENDPOINT_TYPE:
					service = (struct service_endpoint *)handle->pObject;
					if (registry_cfg.server_up && STARTED_DEVICE(service->device))
						ret = DPWS_ERR_IMMUTABLE_USED_OBJECT;
					else
						ret = set_service_endpoint_ptr_att(service, att, value, DC_TRUE);
					break;
				case SERVICE_PORT_TYPE:
					servicePort = (struct service_port *)handle->pObject;
					if (registry_cfg.server_up && servicePort->service && STARTED_DEVICE(servicePort->service->device))	// service port may not be bound
						ret = DPWS_ERR_IMMUTABLE_USED_OBJECT;
					else
						ret = set_service_port_ptr_att(servicePort, att, value, DC_TRUE);
					break;
				case DEVICE_MODEL_TYPE:
					// access is protected when a device is created (the only use)
					ret = set_device_model_ptr_att((struct device_model *)handle->pObject, att, value, DC_TRUE);
					break;
				case SERVICE_CLASS_TYPE:
					if (registry_cfg.server_up && handle->useCount > 1)	// used by more than the creators
						ret = DPWS_ERR_IMMUTABLE_USED_OBJECT;
					else
						ret = set_service_class_ptr_att((struct service_class *)handle->pObject, att, value, DC_TRUE);
					break;
				default:
					ret = DPWS_ERR_NO_SUCH_ATT_ON_OBJECT;
			}
		dcpl_mutex_unlock(reg_conf_lock);
	}

	return ret;
}

int dpws_set_int_att(short handleRef, int att, unsigned long value)
{
	int ret = DPWS_OK;
	struct device *device;
	struct service_endpoint *service;
	handle_s *handle;

	DC_CHECK_PARAM(handleRef >= MIN_CONF_HANDLE && att > 0);

	switch (handleRef)
	{
	case DC_REGISTRY_HANDLE:
		dcpl_mutex_lock(reg_conf_lock);
		if (registry_cfg.server_up)	// used by more than the user
			ret = DPWS_ERR_IMMUTABLE_USED_OBJECT;
		else
			ret = set_registry_config_ptr_att(att, &value, DC_TRUE);
		dcpl_mutex_unlock(reg_conf_lock);
		break;

	case DC_CACHE_HANDLE:
		dcpl_mutex_lock(cache_lock);
		ret = set_cache_config_ptr_att(att, &value, DC_TRUE);
		dcpl_mutex_unlock(cache_lock);
		break;

	case DC_SUBSC_MANAGER_HANDLE:
		dcpl_mutex_lock(subsc_manager_lock);
		ret = set_subsc_manager_config_ptr_att(att, &value, DC_TRUE);
		dcpl_mutex_unlock(subsc_manager_lock);
		break;

	default:
		dcpl_mutex_lock(reg_conf_lock);
		if (handleRef < 0 || !(handle = getHandle(&registryPool, handleRef)))
			ret = DPWS_ERR_NO_HANDLE_FOUND;
		else {
			switch (handle->type)
			{
				case DEVICE_INSTANCE_TYPE:
					device = (struct device *)handle->pObject;
					if (registry_cfg.server_up && STARTED_DEVICE(device))
						ret = DPWS_ERR_IMMUTABLE_USED_OBJECT;
					else
						ret = set_device_ptr_att(device, att, &value, DC_TRUE);
					break;
				case SERVICE_ENDPOINT_TYPE:
					service = (struct service_endpoint *)handle->pObject;
					if (registry_cfg.server_up && STARTED_DEVICE(service->device))
						ret = DPWS_ERR_IMMUTABLE_USED_OBJECT;
					else
						ret = set_service_endpoint_ptr_att(service, att, &value, DC_TRUE);
					break;
				case DEVICE_MODEL_TYPE:
					ret = set_device_model_ptr_att((struct device_model *)handle->pObject, att, &value, DC_TRUE);
					break;
				case SERVICE_CLASS_TYPE:
					if (registry_cfg.server_up && handle->useCount > 1)	// used by more than the creator
						ret = DPWS_ERR_IMMUTABLE_USED_OBJECT;
					else
						ret = set_service_class_ptr_att((struct service_class *)handle->pObject, att, &value, DC_TRUE);
					break;
				default:
					ret = DPWS_ERR_NO_SUCH_ATT_ON_OBJECT;
			}
		}
		dcpl_mutex_unlock(reg_conf_lock);
	}
	return ret;
}

int dpws_add_ptr_att(short handleRef, int att, const void * value)
{
	int ret = DPWS_OK;
	struct device *device;
	struct service_endpoint *service;
	handle_s *handle;

	DC_CHECK_PARAM(handleRef >= MIN_CONF_HANDLE && att > 0 && value);

	switch (handleRef)
	{
	case DC_REGISTRY_HANDLE:
		dcpl_mutex_lock(reg_conf_lock);
		if (registry_cfg.server_up)	// used by more than the user
			ret = DPWS_ERR_IMMUTABLE_USED_OBJECT;
		else
			ret = set_registry_config_ptr_att(att, value, DC_FALSE);
		dcpl_mutex_unlock(reg_conf_lock);
		break;

	case DC_CACHE_HANDLE:
		dcpl_mutex_lock(cache_lock);
		ret = set_cache_config_ptr_att(att, value, DC_FALSE);
		dcpl_mutex_unlock(cache_lock);
		break;

	case DC_SUBSC_MANAGER_HANDLE:
		ret = DPWS_ERR_NO_SUCH_ATT_ON_OBJECT;
		break;

	default:
		dcpl_mutex_lock(reg_conf_lock);
		if (handleRef < 0 || !(handle = getHandle(&registryPool, handleRef)))
			ret = DPWS_ERR_NO_HANDLE_FOUND;
		else
			switch (handle->type)
			{
				case DEVICE_INSTANCE_TYPE:
					device = (struct device *)handle->pObject;
					if (registry_cfg.server_up && STARTED_DEVICE(device))
						ret = DPWS_ERR_IMMUTABLE_USED_OBJECT;
					else
						ret = set_device_ptr_att(device, att, value, DC_FALSE);
					break;
				case SERVICE_ENDPOINT_TYPE:
					service = (struct service_endpoint *)handle->pObject;
					if (registry_cfg.server_up && STARTED_DEVICE(service->device))
						ret = DPWS_ERR_IMMUTABLE_USED_OBJECT;
					else
						ret = set_service_endpoint_ptr_att(service, att, value, DC_FALSE);
					break;
				case DEVICE_MODEL_TYPE:
					ret = set_device_model_ptr_att((struct device_model *)handle->pObject, att, value, DC_FALSE);
					break;
				case SERVICE_CLASS_TYPE:
					if (registry_cfg.server_up && handle->useCount > 1)	// used by more than the creator
						ret = DPWS_ERR_IMMUTABLE_USED_OBJECT;
					else
						ret = set_service_class_ptr_att((struct service_class *)handle->pObject, att, value, DC_FALSE);
					break;
				default:
					ret = DPWS_ERR_NO_SUCH_ATT_ON_OBJECT;
			}
		dcpl_mutex_unlock(reg_conf_lock);
	}
	return ret;
}

int dpws_add_int_att(short handleRef, int att, unsigned long value)
{
	int ret = DPWS_OK;
	struct device *device;
	struct service_endpoint *service;
	handle_s *handle;

	DC_CHECK_PARAM(handleRef >= MIN_CONF_HANDLE && att > 0);

	dcpl_mutex_lock(reg_conf_lock);
	switch (handleRef)
	{
	case DC_REGISTRY_HANDLE:
		if (registry_cfg.server_up)	// used by more than the user
			ret = DPWS_ERR_IMMUTABLE_USED_OBJECT;
		else
			ret = set_registry_config_ptr_att(att, &value, DC_FALSE);
		break;

	case DC_CACHE_HANDLE:
		ret = DPWS_ERR_MONOVALUED_ATT_TYPE;
		break;

	case DC_SUBSC_MANAGER_HANDLE:
		ret = DPWS_ERR_MONOVALUED_ATT_TYPE;
		break;

	default:
		if (handleRef < 0 || !(handle = getHandle(&registryPool, handleRef)))
			ret = DPWS_ERR_NO_HANDLE_FOUND;
		else
			switch (handle->type)
			{
				case DEVICE_INSTANCE_TYPE:
					device = (struct device *)handle->pObject;
					if (registry_cfg.server_up && STARTED_DEVICE(device))
						ret = DPWS_ERR_IMMUTABLE_USED_OBJECT;
					else
						ret = set_device_ptr_att(device, att, &value, DC_FALSE);
					break;
				case SERVICE_ENDPOINT_TYPE:
					service = (struct service_endpoint *)handle->pObject;
					if (registry_cfg.server_up && STARTED_DEVICE(service->device))
						ret = DPWS_ERR_IMMUTABLE_USED_OBJECT;
					else
						ret = set_service_endpoint_ptr_att(service, att, &value, DC_FALSE);
					break;
				case DEVICE_MODEL_TYPE:
					ret = set_device_model_ptr_att((struct device_model *)handle->pObject, att, &value, DC_FALSE);
					break;
				case SERVICE_CLASS_TYPE:
					if (registry_cfg.server_up && handle->useCount > 1)	// used by more than the user
						ret = DPWS_ERR_IMMUTABLE_USED_OBJECT;
					else
						ret = set_service_class_ptr_att((struct service_class *)handle->pObject, att, &value, DC_FALSE);
					break;
				default:
					ret = DPWS_ERR_NO_SUCH_ATT_ON_OBJECT;
			}
	}
	dcpl_mutex_unlock(reg_conf_lock);
	return ret;
}

void * dpws_get_ptr_att_item(short handleRef, int att, int index)
{
	void * ret = NULL;
	handle_s *handle;

	DC_CHECK_PARAM_RETURN(handleRef >= MIN_CONF_HANDLE && att > 0 && index >= 0, NULL);

	switch (handleRef)
	{
	case DC_REGISTRY_HANDLE:
		dcpl_mutex_lock(reg_conf_lock);
		ret = get_registry_config_ptr_att(att, index);
		dcpl_mutex_unlock(reg_conf_lock);
		break;

	case DC_CACHE_HANDLE:
		dcpl_mutex_lock(cache_lock);
		ret = get_cache_config_ptr_att(att, index);
		dcpl_mutex_unlock(cache_lock);
		break;

	case DC_SUBSC_MANAGER_HANDLE:
		break;

	default:
		dcpl_mutex_lock(reg_conf_lock);
		if (handleRef >= 0 && (handle = getHandle(&registryPool, handleRef)))
		{
			switch (handle->type)
			{
				case DEVICE_INSTANCE_TYPE:
					ret = get_device_ptr_att((struct device *)handle->pObject, att, index);
					break;
				case SERVICE_ENDPOINT_TYPE:
					ret = get_service_endpoint_ptr_att((struct service_endpoint *)handle->pObject, att, index);
					break;
				case SERVICE_PORT_TYPE:
					ret = get_service_port_ptr_att((struct service_port *)handle->pObject, att, index);
					break;
				case DEVICE_MODEL_TYPE:
					ret = get_device_model_ptr_att((struct device_model *)handle->pObject, att, index);
					break;
				case SERVICE_CLASS_TYPE:
					ret = get_service_class_ptr_att((struct service_class *)handle->pObject, att, index);
					break;
				default:
					break;
			}
		}
		dcpl_mutex_unlock(reg_conf_lock);
	}
	return ret;
}

int dpws_get_att_count(short handleRef, int att)
{
	int ret = -1;
	handle_s *handle;

	DC_CHECK_PARAM(handleRef >= MIN_CONF_HANDLE && att > 0);

	switch (handleRef)
	{
	case DC_REGISTRY_HANDLE:
		ret = 1;
		break;

	case DC_CACHE_HANDLE:
		dcpl_mutex_lock(cache_lock);
		ret = get_cache_config_att_count(att);
		dcpl_mutex_unlock(cache_lock);
		break;

	case DC_SUBSC_MANAGER_HANDLE:
		dcpl_mutex_lock(subsc_manager_lock);
		ret = get_subsc_manager_config_att_count(att);
		dcpl_mutex_unlock(subsc_manager_lock);
		break;

	default:
		dcpl_mutex_lock(reg_conf_lock);
		if (handleRef < 0 || !(handle = getHandle(&registryPool, handleRef)))
			ret = DPWS_ERR_NO_HANDLE_FOUND;
		else
			switch (handle->type)
			{
				case DEVICE_MODEL_TYPE:
					ret = get_device_model_att_count((struct device_model *)handle->pObject, att);
					break;
				case DEVICE_INSTANCE_TYPE:
					ret = get_device_att_count((struct device *)handle->pObject, att);
					break;
				case SERVICE_ENDPOINT_TYPE:
					ret = get_service_endpoint_att_count((struct service_endpoint *)handle->pObject, att);
					break;
				case SERVICE_CLASS_TYPE:
					ret = get_service_class_att_count((struct service_class *)handle->pObject, att);
					break;
				default:
					ret = 1;
			}
		dcpl_mutex_unlock(reg_conf_lock);
	}
	return ret;
}

void * dpws_get_ptr_att(short handleRef, int att)
{
	return dpws_get_ptr_att_item(handleRef, att, 0);
}

unsigned long dpws_get_int_att(short handleRef, int att)
{
	unsigned long ret = 0xFFFFFFFF;
	handle_s *handle;

	DC_CHECK_PARAM_RETURN(handleRef >= MIN_CONF_HANDLE && att > 0, 0xFFFFFFFF);

	switch (handleRef)
	{
	case DC_REGISTRY_HANDLE:
		dcpl_mutex_lock(reg_conf_lock);
		ret = get_registry_config_int_att(att, 0);
		dcpl_mutex_unlock(reg_conf_lock);
		break;

	case DC_CACHE_HANDLE:
		dcpl_mutex_lock(cache_lock);
		ret = get_cache_config_int_att(att, 0);
		dcpl_mutex_unlock(cache_lock);
		break;

	case DC_SUBSC_MANAGER_HANDLE:
		dcpl_mutex_lock(subsc_manager_lock);
		ret = get_subsc_manager_config_int_att(att, 0);
		dcpl_mutex_unlock(subsc_manager_lock);
		break;

	default:
		dcpl_mutex_lock(reg_conf_lock);
		if (handleRef < 0 || !(handle = getHandle(&registryPool, handleRef)))
			ret = DPWS_ERR_NO_HANDLE_FOUND;
		else
			switch (handle->type)
			{
				case DEVICE_INSTANCE_TYPE:
					ret = get_device_int_att((struct device *)handle->pObject, att, -1);
					break;
				case SERVICE_ENDPOINT_TYPE:
					ret = get_service_endpoint_int_att((struct service_endpoint *)handle->pObject, att, -1);
					break;
				default:
					ret = 0xFFFFFFFF;
			}
		dcpl_mutex_unlock(reg_conf_lock);
	}
	return ret;
}
