/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|                      ->>  Copyright 2008 Odonata <<-                         |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 1772 $
|                     $Date: 2008-10-09 11:57:03 +0200 (jeu., 09 oct. 2008) $
\*============================================================================*/

/*******************************************************************************
*                                    Reactor                                   *
*******************************************************************************/

#ifndef DCDPWS_REACTOR_H_
#define DCDPWS_REACTOR_H_

#include "dc/dc_Types.h"
#include "dcDCPL_Os.h"
#include "dcDCPL_Socket.h"

#define DC_RI_DONE 		0

// First 4 bits used for execution control
#define DC_RI_INTERNAL 	(1 << 0)
#define DC_RI_FACTORY 	(1 << 1)
#define DC_RI_ONCE		(1 << 2)

// Next 4 bits used for type
#define DC_RI_TIMER		(1 << 4)
#define DC_RI_READ		(2 << 4)
#define DC_RI_WRITE		(3 << 4)
#define DC_RI_EXCEPTION	(4 << 4)

#define DC_RI_TYPE_MASK 0xF0

// Use to qualify the type of the object pointed by the handle pool
#define DC_RI_HANDLE_TYPE	1

// State is used for controlling interruptions
#define DC_REACTOR_RUNNING_STATE 0x01
#define DC_REACTOR_WAITING_STATE 0x02
#define DC_REACTOR_INTERRUPTED_STATE 0x04
#define DC_REACTOR_DONE_STATE 0x08
#define DC_REACTOR_SHUTDOWN_STATE 0x10

// Status is used to represent the phases in the reactor life cycle
#define DC_REACTOR_INIT_STATUS 1
#define DC_REACTOR_ACTIVE_STATUS 2
#define DC_REACTOR_STOPPED_STATUS 3
#define DC_REACTOR_TIMED_OUT_STATUS 4
#define DC_REACTOR_DONE_STATUS 5
#define DC_REACTOR_ERROR_STATUS 6

struct reactor;
struct reactor_item;

/**
 * The type of callback function to be used in reactor items.
 */
typedef int (*reactor_cbk)(
		struct dpws * dpws,
		struct reactor_item * item,
		void * callback_data
);

typedef int (*reactor_del_cbk)(
		struct reactor_item * item,
		DCPL_SOCKET socket,
		void * callback_data
);

/**
 * The implementation structure of the reactor object
 */
typedef struct reactor {
	dcpl_mutex_t * lock;
	/* shared fields */
	uint64_t next_wakeup_date;
	struct reactor_item * free_list;
	struct reactor_item * wait_list;
	int outstanding_items;
	uint16_t state;
	/* private fields */
	uint16_t status;
	struct reactor_item * timer_list;
	struct reactor_item * read_list;
	struct reactor_item * write_list;
	struct reactor_item * exception_list;
	struct reactor_item * pending_list;
	struct reactor_item * results; /**< Only used by dc_reactor_wait_single to hold the current result list */
	void* interrupt_data; /**< Opaque data structure holding platform-specific info */
	int error; /**< Internal error code */
	int syserr; /**< System error code */
	char* detail; /**< Optional detail field */
} reactor_t;

/**
 * The common type of all items managed by a reactor
 */
typedef struct reactor_item {
	struct reactor_item * next;
	struct reactor * reactor;
	uint8_t status;
	uint8_t priority;
	int16_t href;
	reactor_cbk callback;
	void * callback_data;
	reactor_del_cbk del_callback;
	DCPL_SOCKET socket;
	uint64_t exp_date;
	uint32_t period;
} reactor_item_t;

extern reactor_t reactor;	/**< The stack global reactor. */

/******************** API functions ***********************/

/**
 * Initializes the reactor structure.
 * @param reactor pointer to the reactor structure
 * @param shared a boolean indicating whether the reactor will be shared by several threads
 * @return DPWS_OK or DPWS_ERR_INTERNAL_ERROR in case of error
 */
int dc_reactor_init(struct reactor * reactor, DC_BOOL shared);

/**
 * Blocks until events occur or timeout is reached,
 * and returns the list of active reactor items.
 * @param reactor pointer to the reactor structure
 * @param active_list an out parameter pointing to the start of the active item list
 * @param dpws a message context structure used for callback execution of internal active items.
 * May be NULL if internal active items do not need it.
 * @param timeout a timeout in milliseconds, or -1 for no timeout.
 * @return DPWS_OK, a positive status code in case of interruption or a negative error code.
 */
int dc_reactor_wait(struct reactor * reactor, struct reactor_item ** active_list, struct dpws * dpws, int32_t timeout);

/**
 * Blocks until events occur or timeout is reached,
 * and returns the first available active reactor item.
 * @param reactor pointer to the reactor structure
 * @param active_item an out parameter pointing to the active item
 * @param dpws a message context structure used for callback execution of internal active items.
 * May be NULL if internal active items do not need it.
 * @param timeout a timeout in milliseconds, or -1 for no timeout.
 * @return DPWS_OK, a positive status code in case of interruption or a negative error code.
 */
int dc_reactor_wait_single(struct reactor * reactor, struct reactor_item ** active_item, struct dpws * dpws, int32_t timeout);

/**
 * Executes the callback of the selected active item.
 * @param reactor pointer to the reactor structure
 * @param item the selected item
 * @param dpws a message context structure used for callback execution
 * @return DPWS_OK or the error code of the callback.
 */
int dc_reactor_execute_item(struct reactor * reactor, struct reactor_item * item, struct dpws * dpws);

/**
 * Executes the callback if the selected active item is a factory item. Factory items are
 * used to create a new item and associate it as active item in the dpws structure
 * @param reactor pointer to the reactor structure
 * @param item the selected item
 * @param dpws a message context structure used for callback execution
 * @return DPWS_OK or the error code of the callback.
 */
int dc_reactor_execute_factory_item(struct reactor * reactor, struct reactor_item * item, struct dpws * dpws);

/**
 * Runs the reactor by repeatedly waiting for events and executing active items.
 * @param reactor pointer to the reactor structure
 * @param dpws a message context structure used to execute active reactor items
 * @param timeout a duration after which the reactor will stop.
 * @return DPWS_OK, a positive status code in case of interruption or a negative error code.
 */
int dc_reactor_start(struct reactor * reactor, struct dpws * dpws, uint32_t timeout);

/**
 * Asynchronously stops the reactor operation. All reactor items are closed and destroyed when the timeout is reached.
 * This causes the dc_reactor_wait function to return with a DC_REACTOR_STOPPED status.
 * @param reactor pointer to the reactor structure
 * @param timeout a duration after which the reactor will stop.
 * @return DPWS_OK or an error code
 */
int dc_reactor_stop(struct reactor * reactor, uint32_t timeout);

/**
 * Asynchronously shutdowns the reactor operation. All read socket items are closed and destroyed,
 * but other items are executed until the timeout is reached.
 * This causes the dc_reactor_wait function to return with a DC_REACTOR_STOPPED status.
 * @param reactor pointer to the reactor structure
 * @param timeout a duration after which the reactor will stop.
 * @return DPWS_OK or an error code
 */
int dc_reactor_shutdown(struct reactor * reactor, uint32_t timeout);

/**
 * Creates a socket item associated to the reactor. The item will need to be subsequently registered or
 * returned to the reactor. Socket items can be of four types:
 * 1. Listen socket items are used to create connected sockets when they receive a connection request.
 * The created socket should be used to add a new socket item to the reactor.
 * 2. Read socket items are used to wait until the socket is ready for reading.
 * 3. Write socket items are used to wait until the socket is ready for writing.
 * 4. Exception socket items are used to wait until the socket is ready for exception (whatever it means!).
 * @param reactor pointer to the reactor structure
 * @param socket a platform-dependent socket descriptor. In case of listen socket items, this descriptor
 * must have been properly initialized and bound to the listen address before registration
 * @param status the type of the socket item (must be one of DC_RI_READ, DC_RI_WRITE, DC_RI_EXCEPTION), or-ed
 * with appropriate execution flags.
 * @param callback the callback function to be called when a connection request is received.
 * In case of listen socket items, it is responsible for accepting the connection,
 * setting up and registering the new connected socket with the reactor.
 * @param callback_data data passed to the callback function
 * @param del_callback the callback function to be called when the reactor item is destroyed
 * @param href optional out parameter that will contain a handle reference to the created reactor item.
 * Ignored when set to NULL.
 * @return The created item or NULL if a memory allocation error occured.
 */
struct reactor_item * dc_reactor_create_socket_item(struct reactor * reactor,
												   SOAP_SOCKET socket,
												   uint8_t status,
												   reactor_cbk callback,
												   void* callback_data,
												   reactor_del_cbk del_callback,
												   int16_t* href);

/**
 * Adds a socket item to the reactor. Socket items can be of four types:
 * 1. Listen socket items are used to create connected sockets when they receive a connection request.
 * The created socket should be used to add a new socket item to the reactor.
 * 2. Read socket items are used to wait until the socket is ready for reading.
 * 3. Write socket items are used to wait until the socket is ready for writing.
 * 4. Exception socket items are used to wait until the socket is ready for exception (whatever it means!).
 * @param reactor pointer to the reactor structure
 * @param socket a platform-dependent socket descriptor. In case of listen socket items, this descriptor
 * must have been properly initialized and bound to the listen address before registration
 * @param status the type of the socket item (must be one of DC_RI_READ, DC_RI_WRITE, DC_RI_EXCEPTION), or-ed
 * with appropriate execution flags.
 * @param callback the callback function to be called when a connection request is received.
 * In case of listen socket items, it is responsible for accepting the connection,
 * setting up and registering the new connected socket with the reactor.
 * @param callback_data data passed to the callback function
 * @param del_callback the callback function to be called when the reactor item is destroyed
 * @param href optional out parameter that will contain a handle reference to the registered reactor item.
 * Ignored when set to NULL.
 * @return DPWS_OK or an error code
 */
int dc_reactor_register_socket(struct reactor * reactor,
							   SOAP_SOCKET socket,
							   uint8_t status,
							   reactor_cbk callback,
							   void* callback_data,
							   reactor_del_cbk del_callback,
							   int16_t* href);

/**
 * Adds a socket item to the reactor. Socket items can be of four types:
 * 1. Listen socket items are used to create connected sockets when they receive a connection request.
 * The created socket should be used to add a new socket item to the reactor.
 * 2. Read socket items are used to wait until the socket is ready for reading.
 * 3. Write socket items are used to wait until the socket is ready for writing.
 * 4. Exception socket items are used to wait until the socket is ready for exception (whatever it means!).
 * @param reactor pointer to the reactor structure
 * @param socket a platform-dependent socket descriptor. In case of listen socket items, this descriptor
 * must have been properly initialized and bound to the listen address before registration
 * @param status the type of the socket item (must be one of DC_RI_READ, DC_RI_WRITE, DC_RI_EXCEPTION), or-ed
 * with appropriate execution flags.
 * @param timeout a duration in milliseconds during which the item will remain in the reactor. A negative value
 * means no timeout. ONLY THIS MODE IS SUPPORTED AT THE MOMENT.
 * @param callback the callback function to be called when a connection request is received.
 * In case of listen socket items, it is responsible for accepting the connection,
 * setting up and registering the new connected socket with the reactor.
 * @param callback_data data passed to the callback function
 * @param del_callback the callback function to be called when the reactor item is destroyed
 * @param href optional out parameter that will contain a handle reference to the registered reactor item.
 * Ignored when set to NULL.
 * @return DPWS_OK or an error code
 */
int dc_reactor_register_socket_with_timeout(struct reactor * reactor,
							   				SOAP_SOCKET socket,
							   				uint8_t status,
							   				int32_t timeout,
							   				reactor_cbk callback,
							   				void* callback_data,
											reactor_del_cbk del_callback,
							   				int16_t* href);

/**
 * Adds a timer item to the reactor.
 * @param reactor pointer to the reactor structure
 * @param internal whether the item should run internally to the reactor. Should normally be DC_FALSE.
 * @param timeout the initial value of the timer in milliseconds, which will be internally converted
 * to the platform absolute time
 * @param period a duration in milliseconds used for reloading a periodic timer
 * @param callback the callback function to be called when the timer expires.
 * @param callback_data data passed to the callback function
 * @param del_callback the callback function to be called when the reactor item is destroyed
 * @param href optional out parameter that will contain a handle reference to the registered reactor item.
 * Ignored when set to NULL.
 * @return DPWS_OK or an error code
 */
int dc_reactor_register_timer(struct reactor * reactor,
		                      DC_BOOL internal,
							  uint32_t timeout,
							  uint32_t period,
							  reactor_cbk callback,
							  void* callback_data,
							  reactor_del_cbk del_callback,
							  int16_t* href);

/**
 * Interrupts the reactor wait loop to allow new items to be processed.
 * @param reactor pointer to the reactor structure
 * @return DPWS_OK or an error code
 */
int dc_reactor_interrupt(struct reactor * reactor);

/**
 * Invalidates the reactor item. This function should be called from within a reactor callback
 * function to indicate to the reactor that the item should be recycled. If the item as a delete
 * callback, this callback will be called. Otherwise, if the item is a socket item, the socket will be closed.
 * @param item the reactor item to be invalidated
 * @return DPWS_OK or an error code
 */
int dc_reactor_item_invalidate(struct reactor_item * item);

/**
 * Sets the socket associated to this reactor item.
 * @param item the reactor item to be invalidated
 * @param socket the socket, which can be SOAP_INVALID_SOCKET if the item
 * socket has been closed externally
 * @return DPWS_OK or an error code
 */
int dc_reactor_item_set_socket(struct reactor_item * item, SOAP_SOCKET socket);

/**
 * Retrieves the socket associated to this reactor item.
 * @param item the reactor item to be invalidated
 * @param socket the retrieved socket, which can be SOAP_INVALID_SOCKET if the item
 * does not contain a valid socket item
 * @return DPWS_OK or an error code
 */
int dc_reactor_item_get_socket(struct reactor_item * item, SOAP_SOCKET* socket);

/**
 * Removes a reactor item from its reactor.
 * @param href the handle reference to the registered reactor item.
 * @return DPWS_OK or an error code
 */
int dc_reactor_unregister_item(int16_t href);

#endif /*DCDPWS_REACTOR_H_*/
