/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|                      ->>  Copyright 2008 Odonata <<-                         |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 1772 $
|                     $Date: 2008-10-09 11:57:03 +0200 (jeu., 09 oct. 2008) $
\*============================================================================*/

/***************************************************************************//**
* \file
* DPWSCore IP sockets portability layer.
*******************************************************************************/

#ifndef DCDCPL_SOCKET_H_
#define DCDCPL_SOCKET_H_

#include "dcpl_target.h"
#include "dcDCPL_Error.h"

#include <string.h>

#define is_ipv6_addr(addr) (strchr((addr), ':'))

/**
 * Structure for TCP sockets options.
 */
typedef struct socket_data {
	uint32_t flags;	/**< Standard BSD bitwise boolean socket options passed directly by the API user. */
	uint32_t buflen;	/**< Send/receive buffer size (set if >0). */
	int tcp_nodelay;	/**< Boolean. If true, the TCP nagle algorithm must be disabled. */
	int keep_alive;	/**< Enable the TCP "keepalive probe" that check after a long inactivity period (several hours) that the peer is alive. */
	uint16_t linger_time;	/**< Number of seconds a close socket can be blocked to transmit pending data (if SO_LINGER is set in \a flags)*/
} socket_data_t;


/**
 * DCPL IP v4 address family (bit flag)
 */
#define DCPL_AF_INET	0x1

/**
 * DCPL IP v6 address family (bit flag)
 */
#define DCPL_AF_INET6	0x2


#ifdef __cplusplus
extern "C" {
#endif

/** Opens an UDP socket for sending datagrams.
 * @param family The address family for the socket (DCPL_AF_INET or
 * DCPL_AF_INET6)
 * @param[out] error A structure to get detailed error information.
 * @return DCPL_OK or any DCPL error code.
 */
DCPL_SOCKET dcpl_udp_open(int family, struct dcpl_error * error);

/** Selects the network interface for sending multicast datagrams on a UDP
 * socket.
 * @param s The UDP socket descriptor.
 * @param family The address family of the socket (DCPL_AF_INET or
 * DCPL_AF_INET6)
 * @param itf_selector The index of the interface to be selected. This index is
 * the interface number for IPv6 (field itf_nb in \a dcpl_ip_addr_t), the local
 * IP address in its numeric form for IPv4 (field ipv4_addr in \a dcpl_ip_addr_t).
 * @param[out] error A structure to get detailed error information.
 * @return DCPL_OK or any DCPL error code.
 */
int dcpl_set_multicast_netif(DCPL_SOCKET s, int family, uint32_t itf_selector, struct dcpl_error * error);

/** Opens an UDP socket for receiving datagrams.
 * @param family The address family for the socket (DCPL_AF_INET or
 * DCPL_AF_INET6)
 * @param itf_addr The local IP address on which listening (selects the
 * interface).
 * @param itf_selector The interface index for the \a itf_addr address. Required
 * for IPv6 link-local listening addresses (else 0). Should be ignored for IPv4.
 * @param port The local UDP port on which listening.
 * @param bind_flags Standard BSD bitwise boolean socket options passed directly
 * by the API user.
 * @param[out] error A structure to get detailed error information.
 * @return DCPL_OK or any DCPL error code.
 */
DCPL_SOCKET dcpl_udp_bind(int family, const char* itf_addr, uint32_t itf_selector, uint16_t port, int bind_flags, struct dcpl_error * error);

/** Opens an UDP socket for receiving multicast datagrams.
 * @param family The address family for the socket (DCPL_AF_INET or
 * DCPL_AF_INET6)
 * @param itf_addr The local IP address on which listening (selects the
 * interface).
 * @param itf_selector The interface index for the \a itf_addr address. Required
 * for IPv6 link-local listening addresses (else 0). Should be ignored for IPv4.
 * @param port The local UDP port on which listening.
 * @param multicast_addr The multicast address of the group on which listening.
 * @param[out] error A structure to get detailed error information.
 * @return DCPL_OK or any DCPL error code.
 */
DCPL_SOCKET dcpl_multicast_bind(int family, const char* itf_addr, uint32_t itf_selector, uint16_t port, const char* multicast_addr, struct dcpl_error * error);

/** Sends a datagram on an opened UDP socket.
 * @param s The UDP socket descriptor.
 * @param buf The character buffer containing the datagram to send.
 * @param n The number of characters of the datagram in \a buf.
 * @param flags Standard "send" flags passed transparently to the BSD
 * implementation.
 * @param host The IP address (IPv4 or v6) of the destination host in its string
 * form.
 * @param itf_selector The local interface index on which to send the datagram
 * required for sending when the destination host uses a link-local IPv6
 * address (else 0). Should be ignored for IPv4.
 * @param port The destination UDP port.
 * @param[out] error A structure to get detailed error information.
 * @return The number of bytes actually sent or any negative DCPL error code.
 */
int dcpl_udp_send(DCPL_SOCKET s, const char *buf, size_t n, int flags, const char *host, uint32_t itf_selector, uint16_t port, struct dcpl_error * error);

/** Sends locally a datagram on an UDP socket using the loopback interface.
 * @param s The UDP socket descriptor.
 * @param buf The character buffer containing the datagram to send.
 * @param n The number of characters of the datagram in \a buf.
 * @param flags Standard "send" flags passed transparently to the BSD
 * implementation.
 * @param af The address family of the socket in order to select the correct
 * loopback address.
 * @param port The UDP port on which to send the datagram.
 * @param[out] error A structure to get detailed error information.
 * @return The number of bytes actually sent or any negative DCPL error code.
 */
int dcpl_udp_send_loopback(DCPL_SOCKET s, const char *buf, size_t n, int flags, int af, unsigned short port, struct dcpl_error * error);

/** Receives a datagram on a bound UDP socket.
 * @param s The UDP socket descriptor.
 * @param buf The character buffer for receiving the datagram.
 * @param n The number of characters available for the datagram in \a buf.
 * @param flags Standard "receive" flags passed transparently to the BSD
 * implementation.
 * @param[out] peer_addr Optional (nullable) buffer for receiving the peer
 * address. This buffer should be long enough to receive an IPv4 or IPv6 address
 * according to the address on which the socket is bound.
 * @param addr_len The length of the provided \a peer_addr buffer.
 * @param[out] peer_port The peer port. May be null if not required.
 * @param[out] itf The reception interface corresponding to the field itf_nb in
 * \a dcpl_ip_addr_t for both IPv4 and IPv6. May be null if not required.
 * @param[out] error A structure to get detailed error information.
 * @return The number of bytes actually received or any negative DCPL error
 * code.
 */
int dcpl_udp_recv(DCPL_SOCKET s, char *buf, size_t n, int flags, char* peer_addr, int addr_len, uint16_t * peer_port, uint32_t * itf, struct dcpl_error * error);

/** Opens a socket for receiving incoming TCP connections.
 * @param family The address family for the socket (DCPL_AF_INET or
 * DCPL_AF_INET6)
 * @param itf_addr The local IP address on which listening.
 * @param itf_selector The interface index for the \a itf_addr address. Required
 * for IPv6 link-local listening addresses (else 0). Should be ignored for IPv4.
 * @param port The local TCP port on which listening.
 * @param backlog The TCP backlog (i.e. The number of pending incoming
 * connections).
 * @param bind_data The TCP options to use for the listening socket.
 * @param[out] error A structure to get detailed error information.
 * @return DCPL_OK or any DCPL error code.
 */
DCPL_SOCKET dcpl_tcp_bind_listener(int family, const char* itf_addr, uint32_t itf_selector, uint16_t port, int backlog, socket_data_t * bind_data, struct dcpl_error * error);

/** Creates a socket for an incoming TCP connection.
 * @param listener The descriptor of the listening socket.
 * @param data The TCP options to set on the new socket.
 * @param[out] peer_addr Optional (nullable) buffer for receiving the peer
 * address. This buffer should be long enough to receive an IPv4 or IPv6 address
 * according to the address on which the listening socket is bound.
 * @param addr_len The length of the provided \a peer_addr buffer.
 * @param[out] peer_port The peer port. May be null if not required.
 * @param[out] itf The reception interface corresponding to the field itf_nb in
 * \a dcpl_ip_addr_t for both IPv4 and IPv6. May be null if not required.
 * @param[out] error A structure to get detailed error information.
 * @return DCPL_OK or any DCPL error code.
 */
DCPL_SOCKET dcpl_tcp_accept(DCPL_SOCKET listener, socket_data_t * data, char* peer_addr, int addr_len, uint16_t * peer_port, uint32_t * itf, struct dcpl_error * error);

/** Creates and connects a TCP socket to a remote host.
 * @param host The IP address (IPv4 or v6) of the destination host in its string
 * form.
 * @param itf_selector The local interface index on which to send the datagram
 * required for sending when the destination host uses a link-local IPv6
 * address (else 0). Should be ignored for IPv4.
 * @param port The destination UDP port.
 * @param timeout The timeout in milliseconds for the connection. Should block
 * if 0.
 * @param data The TCP options to set on the new socket before the connection.
 * @param[out] error A structure to get detailed error information.
 * @return A descriptor for the connected socket or DCPL_INVALID_SOCKET in case
 * of error.
 */
DCPL_SOCKET dcpl_tcp_connect(const char *host, uint32_t itf_selector, uint16_t port, int timeout, socket_data_t * data, struct dcpl_error * error);

/** Retrieves the local address for a TCP connected socket.
 * Can be used after calling dcpl_tcp_connect to establish which route has
 * been used by the IP stack.
 * @param s The connected socket descriptor.
 * @param[out] local_addr_buf Buffer for receiving the local address. This
 * buffer should be long enough to receive an IPv4 or IPv6 address according to
 * the address that was used to connect the socket.
 * @param buf_size The length of the provided \a local_addr_buf buffer.
 * @param[out] error A structure to get detailed error information.
 * @return DCPL_OK or any DCPL error code.
 */
int dcpl_tcp_get_local_address(DCPL_SOCKET s, char * local_addr_buf, int buf_size, struct dcpl_error * error);

/** Stops receive/send operations on a connected TCP socket.
 * Straightforward BSD semantics.
 * @param s The connected socket descriptor.
 * @param how 0 (stops readings), 1 (stops writings) or 2 (stop both).
 * @param[out] error A structure to get detailed error information.
 * @return DCPL_OK or any DCPL error code.
 */
int dcpl_tcp_shutdown(DCPL_SOCKET s, int how, struct dcpl_error * error);


/** Sends data on a connected TCP socket.
 * @param s The TCP socket descriptor.
 * @param buf The character buffer containg the data to send.
 * @param n The number of characters to be sent in \a buf.
 * @param flags Standard "send" flags passed transparently to the BSD
 * implementation.
 * @param timeout The timeout in milliseconds for the sending operation. Should
 * block if 0.
 * @param[out] error A structure to get detailed error information.
 * @return The number of bytes actually sent or any negative DCPL error code.
 */
int dcpl_tcp_send(DCPL_SOCKET s, const char *buf, size_t n, int flags, int timeout, struct dcpl_error * error);

/** Reads data on a connected TCP socket.
 * @param s The connected TCP socket descriptor.
 * @param buf The character buffer for receiving the data.
 * @param n The number of characters available for the data in \a buf.
 * @param flags Standard "receive" flags passed transparently to the BSD
 * implementation.
 * @param timeout The timeout in milliseconds for the reception operation.
 * Should block if 0.
 * @param[out] error A structure to get detailed error information.
 * @return The number of bytes actually received or any negative DCPL error
 * code.
 */
int dcpl_tcp_recv(DCPL_SOCKET s, char *buf, size_t n, int flags, int timeout, struct dcpl_error * error);

/** Peeks data on a connected TCP socket.
 * Similar do \a dcpl_tcp_recv except that the data is not removed from the
 * stream and is still available for reading.
 * @param s The connected TCP socket descriptor.
 * @param buf The character buffer for receiving the data.
 * @param n The number of characters available for the data in \a buf.
 * @param flags Standard "receive" flags passed transparently to the BSD
 * implementation.
 * @param timeout The timeout in milliseconds for the reception operation.
 * Should block if 0.
 * @param[out] error A structure to get detailed error information.
 * @return The number of bytes actually received or any negative DCPL error
 * code.
 */
int dcpl_tcp_peek(DCPL_SOCKET s, char *buf, size_t n, int flags, int timeout, struct dcpl_error * error);

/**
 * Test a connected socket is still alive (not the connection actually).
 * Note that this call can only check that a socket has received a graceful
 * shutdown from the peer or that the "keep-alive" message (that may have been
 * turned on with TCP option) has failed. However this last is genrally sent
 * after several hours of inactivity...
 * @param s The connected TCP socket descriptor.
 * @return 0 if the socket is unconnected. 1 else if nothing was detected.
 */
int dcpl_tcp_socket_valid(DCPL_SOCKET s);

/** Frees all resources associated with any type of socket.
 * May close a TCP connection if running.
 * @param s The socket descriptor to close.
 * @param[out] error A structure to get detailed error information.
 * @return DCPL_OK or any DCPL error code.
 */
int dcpl_closesocket(DCPL_SOCKET s, struct dcpl_error * error);

#ifdef __cplusplus
}
#endif

#endif
