/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2091 $
|                     $Date: 2009-02-17 15:38:44 +0100 (mar, 17 fév 2009) $
\*============================================================================*/

/*
 ** Copyright (c) 1990- 1993, 1996 Open Software Foundation, Inc.
 ** Copyright (c) 1989 by Hewlett-Packard Company, Palo Alto, Ca. &
 ** Digital Equipment Corporation, Maynard, Mass.
 ** To anyone who acknowledges that this file is provided "AS IS"
 ** without any express or implied warranty: permission to use, copy,
 ** modify, and distribute this file for any purpose is hereby
 ** granted without fee, provided that the above copyright notices and
 ** this notice appears in all source code copies, and that none of
 ** the names of Open Software Foundation, Inc., Hewlett-Packard
 ** Company, or Digital Equipment Corporation be used in advertising
 ** or publicity pertaining to distribution of the software without
 ** specific, written prior permission.  Neither Open Software
 ** Foundation, Inc., Hewlett-Packard Company, nor Digital Equipment
 ** Corporation makes any representations about the suitability of
 ** this software for any purpose.
 */

#include "dcCOMN_Uuid.h"
#include <stdio.h>
#include <string.h>

/*----------------------------------------------------------------------------*\
 *                                UUID utilities                              *
\*----------------------------------------------------------------------------*/
#define CLOCK_SEQ_LAST              0x3FFF
#define RAND_MASK                   CLOCK_SEQ_LAST

/** Long integer structure */
typedef struct _unsigned64_t {
	uint32_t          lo;
	uint32_t          hi;
} unsigned64_t;

/*
**  Add two unsigned 64-bit long integers.
*/
#define ADD_64b_2_64b(A, B, sum) \
      { \
          if (!(((A)->lo & 0x80000000UL) ^ ((B)->lo & 0x80000000UL))) { \
              if (((A)->lo&0x80000000UL)) { \
                  (sum)->lo = (A)->lo + (B)->lo; \
                  (sum)->hi = (A)->hi + (B)->hi + 1; \
              } \
              else { \
                  (sum)->lo  = (A)->lo + (B)->lo; \
                  (sum)->hi = (A)->hi + (B)->hi; \
              } \
          } \
          else { \
              (sum)->lo = (A)->lo + (B)->lo; \
              (sum)->hi = (A)->hi + (B)->hi; \
              if (!((sum)->lo&0x80000000UL)) (sum)->hi++; \
          } \
      }

/*
**  Add a 16-bit unsigned integer to a 64-bit unsigned integer.
*/
#define ADD_16b_2_64b(A, B, sum) \
      { \
          (sum)->hi = (B)->hi; \
          if ((B)->lo & 0x80000000UL) { \
              (sum)->lo = (*A) + (B)->lo; \
              if (!((sum)->lo & 0x80000000UL)) (sum)->hi++; \
          } \
          else \
              (sum)->lo = (*A) + (B)->lo; \
      }

/*------------------------- Static Functions prototypes ----------------------*/

static void mult32(uint32_t u, uint32_t v, unsigned64_t *result);
static void get_uuid_system_time(unsigned64_t *uuid_time);
static void true_random_init(void);
static void new_clock_seq();
static char * uuid_generate(char *uuid, unsigned64_t time_now, uint16_t clock_seq, byte * mac_addr);

/*----------------------------------------------------------------------------*\
 *                                   UUID                                     *
\*----------------------------------------------------------------------------*/


/*
**  Global variables.
*/

static unsigned64_t		time_last;	// QUESTION: is it thread safe ?
static uint16_t			clock_seq;
static int				initialized = 0;

static void mult32(uint32_t u, uint32_t v, unsigned64_t *result)
{
    /* Following the notation in Knuth, Vol. 2. */
    uint32_t uuid1, uuid2, v1, v2, temp;

    uuid1 = u >> 16;
    uuid2 = u & 0xFFFF;
    v1 = v >> 16;
    v2 = v & 0xFFFF;
    temp = uuid2 * v2;
    result->lo = temp & 0xFFFF;
    temp = uuid1 * v2 + (temp >> 16);
    result->hi = temp >> 16;
    temp = uuid2 * v1 + (temp & 0xFFFF);
    result->lo += (temp & 0xFFFF) << 16;
    result->hi += uuid1 * v1 + (temp >> 16);
}

static void get_uuid_system_time(unsigned64_t *uuid_time)
{
    unsigned64_t utc, usecs = {0, 0}, os_basetime_diff;

    dcpl_get_time(&uuid_time->hi, &uuid_time->lo);
    mult32(uuid_time->hi, 10000000, &utc);
    usecs.lo = uuid_time->lo/100;
    ADD_64b_2_64b(&usecs, &utc, &utc);	// unit 100ns

    /* Offset between UUID formatted times and Unix formatted times.
    * UUID UTC base time is October 15, 1582.
    * Unix base time is January 1, 1970. */
    os_basetime_diff.lo = 0x13814000;
    os_basetime_diff.hi = 0x01B21DD2;
    ADD_64b_2_64b(&utc, &os_basetime_diff, uuid_time);
}

/*
** See "The Multiple Prime Random Number Generator" by Alexander
** Hass pp. 368-381, ACM Transactions on Mathematical Software,
** 12/87.
*/
static uint32_t rand_m;
static uint32_t rand_ia;
static uint32_t rand_ib;
static uint32_t rand_irand;


static void true_random_init(void)
{
    unsigned64_t t;
    uint16_t seed;

    /* Generating our 'seed' value Start with the current time, but,
    * since the resolution of clocks is system hardware dependent and
    * most likely coarser than our resolution (10 usec) we 'mixup' the
    * bits by xor'ing all the bits together.  This will have the effect
    * of involving all of the bits in the determination of the seed
    * value while remaining system independent.  Then for good measure
    * to ensure a unique seed when there are multiple processes
    * creating UUIDs on a system, we add in the PID.
    */
    rand_m = 971;
    rand_ia = 11113;
    rand_ib = 104322;
    rand_irand = 4181;
    get_uuid_system_time(&t);
    seed  =  (uint16_t)t.lo        & 0xFFFF;
    seed ^= (t.lo >> 16) & 0xFFFF;
    seed ^=  t.hi        & 0xFFFF;
    seed ^= (t.hi >> 16) & 0xFFFF;
    rand_irand += seed + dcpl_process_id();
}

uint16_t true_random(void)
{
    if ((rand_m += 7) >= 9973)
        rand_m -= 9871;
    if ((rand_ia += 1907) >= 99991)
        rand_ia -= 89989;
    if ((rand_ib += 73939) >= 224729)
        rand_ib -= 96233;
    rand_irand = (rand_irand * rand_m) + rand_ia + rand_ib;
    return (uint16_t)((rand_irand >> 16) ^ (rand_irand & RAND_MASK));
}

void uuid_init()
{
    if (!initialized)
    {
        initialized = 1;
        true_random_init();
        get_uuid_system_time(&time_last);
        clock_seq = true_random();
    }
}

static int time_cmp(unsigned64_t *time1, unsigned64_t *time2)
{
    if (time1->hi < time2->hi)
        return -1;
    if (time1->hi > time2->hi)
        return 1;
    if (time1->lo < time2->lo)
        return -1;
    if (time1->lo > time2->lo)
        return 1;
    return 0;
}

static void new_clock_seq()
{
    clock_seq = (clock_seq + 1) % (CLOCK_SEQ_LAST + 1);
    if (clock_seq == 0)
        clock_seq = 1;
}

/* provide a 46 octet char buffer */
static char * uuid_generate(char *uuid, unsigned64_t time_now, uint16_t clock_seq, byte * mac_addr)
{
    /* Construct a uuid with the information we've gathered
    * plus a few constants. */
    sprintf(
        uuid, "%08lx-%04hx-%04hx-%02hhx%02hhx-%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx",
        (uint32_t)time_now.lo,
        (uint16_t)time_now.hi & 0x0000FFFF,
        (uint16_t)((time_now.hi & 0x0FFF0000) >> 16) | (1 << 12),
        (byte)((clock_seq & 0x3F00) >> 8) | 0x80,
        (byte)clock_seq & 0xFF,
        mac_addr[0], mac_addr[1], mac_addr[2],
        mac_addr[3], mac_addr[4], mac_addr[5]
    );
    return uuid;
}

char * uuid_create(char *uuid, byte * mac_addr)	// QUESTION: should protect ?
{
    static unsigned64_t     time_now;
    static uint16_t       time_adjust;
    int                     got_no_time = 0;
    get_uuid_system_time(&time_now);

    do
    {
        switch (time_cmp(&time_now, &time_last))
        {
        case -1: /* Time went backwards. */
            new_clock_seq();
            time_adjust = 0;
            break;
        case 1:
            time_adjust = 0;
            break;
        default:
            if (time_adjust == 0x7FFF)
                /* We're going too fast for our clock; spin. */
                got_no_time = 1;
            else
                time_adjust++;
            break;
        }
    }
    while (got_no_time);

    time_last.lo = time_now.lo;
    time_last.hi = time_now.hi;

    if (time_adjust != 0)
    {
        ADD_16b_2_64b(&time_adjust, &time_now, &time_now);
    }
    return uuid_generate(uuid, time_now, clock_seq, mac_addr);
}

/* provide a SCHEME_UUID_STRING_SIZE octet char buffer */
char * stable_uuid_create(char *uuid, uint32_t instance, byte * mac_addr)
{
    unsigned64_t time = {0, 0};
    time.lo = instance;	// NOTE: hope the time is small enough to be expired and avoid collision
    strcpy(uuid, UUID_SCHEME);
    uuid_generate(uuid + UUID_SCHEME_SIZE, time, 0, mac_addr);
    return uuid;
}

char * format_scheme_uuid(char buffer[SCHEME_UUID_STRING_SIZE], char *uuid)
{
	if (strncmp(UUID_SCHEME, uuid, UUID_SCHEME_SIZE))
	{
	    if (strncmp("uuid:", uuid, 5))
	    {
	        strcpy(buffer, UUID_SCHEME);
	        strncpy(buffer+strlen(UUID_SCHEME), uuid, UUID_STRING_SIZE);
	    }
	    else
	    {
	        strcpy(buffer, "urn:");
	        strncpy(buffer+4, uuid, SCHEME_UUID_STRING_SIZE - 4);
	    }
	}
	else
	    strncpy(buffer, uuid, SCHEME_UUID_STRING_SIZE);

	return buffer[SCHEME_UUID_STRING_SIZE - 1] ? NULL : buffer;
}

