/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2250 $
|                     $Date: 2009-04-07 17:21:13 +0200 (mar, 07 avr 2009) $
\*============================================================================*/

#include "dcCOMN_Handle.h"
#include "dcCOMN_Tools.h"

#include <stdlib.h>
#include <string.h>


/*----------------------------------- Types ----------------------------------*/

struct gc_info {
	short 		href;
	uint16_t	ots;
	struct lru_entry * lruInfo;
};

struct translate_info {
	handle_cbk cbk;
	void * param;
};

/*------------------------- Static Functions prototypes ----------------------*/

static struct lru_entry * findLRU(handlePool_t *pool, short type);
static DC_BOOL find_oldest_cbk(handle_s * h, struct gc_info * gci);
static handle_s * _getHandle(handlePool_t *pool, short handleRef);
static handle_s * _checkoutHandle(handlePool_t *pool, short handleRef);
static int deleteHandle(handlePool_t *pool, short handleRef);
static int gc_lru(handlePool_t *pool, struct lru_entry * lru, uint16_t free_slots_target);

/*----------------------------------------------------------------------------*\
 *                                PRIVATE STUFF                               *
\*----------------------------------------------------------------------------*/

#define TIMESTAMP_HANDLE(h, plru) (h)->ts = (plru)->cit;(plru)->cit = ((plru)->cit == 0xFFFF) ? 0 : (plru)->cit + 1

static struct lru_entry * findLRU(handlePool_t *pool, short type)
{
	struct lru_entry * wLruInfo = NULL;
    for (
    	wLruInfo = pool->lruInfo;
    	wLruInfo && wLruInfo->type != ANY_TYPE && wLruInfo->type != type;
		wLruInfo = wLruInfo->next
    );
    return wLruInfo;
}

static DC_BOOL find_oldest_cbk(handle_s * h, struct gc_info * gci)
{
	if (h->useCount == 1 && ((h->ts < gci->lruInfo->cit && h->ts < gci->ots)
			|| (h->ts > gci->lruInfo->cit && (gci->ots <= gci->lruInfo->cit || h->ts < gci->ots))))
	{
		gci->ots = h->ts;
		gci->href = h->handleRef;
	}
	return DC_FALSE;
}

static handle_s * _getHandle(handlePool_t *pool, short handleRef)
{
    handle_s * wHandle = NULL;

	if (handleRef >= 0) {
	    for (
	        wHandle = pool->buckets[handleRef%pool->bucketNumber];
	        wHandle && wHandle->handleRef != handleRef;
	        wHandle = wHandle->next
	    );
	}
    return wHandle;
}

static handle_s * _checkoutHandle(handlePool_t *pool, short handleRef)
{
    handle_s * wHandle = _getHandle(pool, handleRef);
    if (wHandle) {
    	struct lru_entry * wLruInfo;

        wHandle->useCount++;
    	wLruInfo = findLRU(pool, wHandle->type);
		if (wLruInfo) {
			TIMESTAMP_HANDLE(wHandle, wLruInfo);
			wLruInfo->hCount--;	// handle hold by more than the creator are excluded from the LRU
		}
        DPWSLOG4(DC_HANDLES, "\t<- Checkout handle %d, type %d, use count %d, pool %p\n", wHandle->handleRef, wHandle->type, wHandle->useCount, pool);
    }
    return wHandle;
}

static int deleteHandle(handlePool_t *pool, short handleRef)
{
    int ret = 0, bucketIndex;
	struct lru_entry * wLruInfo;
	handle_s * wHandle, *rHandle;

    bucketIndex = handleRef%pool->bucketNumber;
	wHandle = pool->buckets[bucketIndex];

    if (wHandle->handleRef == handleRef)
    {
        pool->buckets[bucketIndex] = wHandle->next;
        wHandle->next = pool->unusedPool;
        pool->unusedPool = wHandle;
    }
    else
    {
        for (; wHandle;	wHandle = wHandle->next)
        {
            if (wHandle->next && wHandle->next->handleRef == handleRef)
            {
                rHandle = wHandle->next;
                wHandle->next = wHandle->next->next;
                rHandle->next = pool->unusedPool;
                pool->unusedPool = rHandle;
                break;
            }
        }
    }
    // decrease handle count in LRU
	wLruInfo = findLRU(pool, pool->unusedPool->type);
	if (wLruInfo)
		wLruInfo->hCount--;

    if (pool->releaseHook)
        ret = pool->releaseHook(pool->unusedPool);

    return ret;
}

static int gc_lru(handlePool_t *pool, struct lru_entry * lru, uint16_t free_slots_target)
{
    int ret = 0;

	/* release handles until it's under limit */
	while ((lru->hCount  + free_slots_target) > lru->hMax)
	{
		struct gc_info gci;
		gci.href =  -1;
		gci.ots =  lru->cit;
		gci.lruInfo =  lru;

		handleIterate(pool, lru->type, (handle_it_cbk)find_oldest_cbk, &gci);
		if (gci.href  < 0 || deleteHandle(pool, gci.href)) {	// possibly no unused handle to free
			ret = -1;
			goto exit;
		}
	}

exit:
    return ret;
}

/*----------------------------------------------------------------------------*\
 *                          HANDLE API IMPLEMENTATION                         *
\*----------------------------------------------------------------------------*/

int createHandlePool(int mod, handlePool_t *pool, short bucketNumber,
							dcpl_mutex_t * lock, handle_release_cbk releaseHook)
{
    pool->mod = mod;
	pool->bucketNumber = bucketNumber;
    pool->lastHandle = 0;
    pool->lock = lock;
    pool->buckets = (struct handleInfo**)DC_MALLOC(mod, bucketNumber * sizeof(handle_s *));
    if (!pool->buckets)
    	return -1;
    memset(pool->buckets, 0, bucketNumber * sizeof(handle_s *));
    pool->unusedPool = NULL;
    pool->releaseHook = releaseHook;
    pool->lruInfo = NULL;
    return 0;
}

void deleteHandlePool(handlePool_t *pool)
{
    handle_s * wHandle;
    struct lru_entry * wLruInfo;

    // handles should be all in the recycling pool if the handle pool has been cleaned
    for (wHandle = pool->unusedPool; wHandle; wHandle = pool->unusedPool)
    {
        pool->unusedPool = wHandle->next;
        DC_FREE(pool->mod, wHandle);
    }
    DC_FREE(pool->mod, pool->buckets);
    pool->buckets = NULL;
    while (pool->lruInfo) {
    	wLruInfo = pool->lruInfo;
    	pool->lruInfo = pool->lruInfo->next;
    	DC_FREE(pool->mod, wLruInfo);
    }
}

int setHandleLRU(handlePool_t *pool, short type, uint16_t maxHandles)
{
    int ret = 0;
	struct lru_entry * lruInfo;

    if (pool->lock)
    	dcpl_mutex_lock(pool->lock);

    lruInfo = findLRU(pool, type);

	if (lruInfo) {
		lruInfo->hMax = maxHandles;
		ret = gc_lru(pool, lruInfo, 0);
	}
	else {
		lruInfo = (struct lru_entry *)DC_MALLOC(pool->mod, sizeof(struct lru_entry));
	    if (!lruInfo) {
	    	ret = -1;
	    	goto exit;
	    }
		lruInfo->type = type;
		lruInfo->cit = 0;
		lruInfo->hMax = maxHandles;
		lruInfo->hCount = 0;	// Handle count
		lruInfo->next = pool->lruInfo;
		pool->lruInfo = lruInfo;
	}

exit:
    if (pool->lock)
    	dcpl_mutex_unlock(pool->lock);

    return ret;
}

short createHandle(handlePool_t *pool, short type, void * pObject)
{
	handle_s * newHandle = NULL;
    int bucketIndex, ret = 0;
    struct lru_entry * wLruInfo;

    if (pool->lock)
    	dcpl_mutex_lock(pool->lock);

    if (pool->lastHandle == 0x7FFF) {
        ret = -1;
        goto exit;
    }

    /* Check LRU */
	wLruInfo = findLRU(pool, type);

	if (wLruInfo && (ret = gc_lru(pool, wLruInfo, 1)))
		goto exit;

    /* try to recycle */
    if (pool->unusedPool)
    {
        newHandle = pool->unusedPool;
        pool->unusedPool = pool->unusedPool->next;
    }
    else
    {
        newHandle = (struct handleInfo*)DC_MALLOC(pool->mod, sizeof(handle_s));
        if (!newHandle) {
            ret = -1;
            goto exit;
        }
        newHandle->handleRef = pool->lastHandle++;
        newHandle->pObject = NULL;
    }

    /* fill handle */
    newHandle->useCount = 1;
    newHandle->type = type;
    if (wLruInfo)
    {
		TIMESTAMP_HANDLE(newHandle, wLruInfo);
		wLruInfo->hCount++;
    }
    if (pObject)	// to be able to recycle object too
        newHandle->pObject = pObject;
    DPWSLOG4(DC_HANDLES, "\t-> Create handle %d, type %d, use count %d, pool %p\n", newHandle->handleRef, newHandle->type, newHandle->useCount, pool);

    /* insert in bucket */
    bucketIndex = newHandle->handleRef%pool->bucketNumber;
    newHandle->next = pool->buckets[bucketIndex];
    pool->buckets[bucketIndex] = newHandle;

exit:
    if (pool->lock)
        dcpl_mutex_unlock(pool->lock);

	return newHandle ? newHandle->handleRef : -1;
}

handle_s * getHandle(handlePool_t *pool, short handleRef)
{
    handle_s * wHandle;

    if (pool->lock)
    	dcpl_mutex_lock(pool->lock);

    wHandle = _getHandle(pool, handleRef);

    if (pool->lock)
        dcpl_mutex_unlock(pool->lock);

    return wHandle;
}

void * getObject(handlePool_t *pool, short handleRef)
{
    handle_s * wHandle;

    if (pool->lock)
    	dcpl_mutex_lock(pool->lock);

    wHandle = _getHandle(pool, handleRef);

    if (pool->lock)
        dcpl_mutex_unlock(pool->lock);

    return wHandle ? wHandle->pObject : NULL;
}

void * getObjectUnsafe(handlePool_t *pool, short handleRef)
{
    handle_s * wHandle = _getHandle(pool, handleRef);
    return wHandle ? wHandle->pObject : NULL;
}

void * getTypedObject(handlePool_t *pool, short handleRef, short type)
{
    handle_s * wHandle;

    if (pool->lock)
    	dcpl_mutex_lock(pool->lock);

    wHandle = _getHandle(pool, handleRef);

    if (pool->lock)
        dcpl_mutex_unlock(pool->lock);

    return wHandle && wHandle->type == type ? wHandle->pObject : NULL;
}

handle_s * checkoutHandle(handlePool_t *pool, short handleRef)
{
    handle_s * wHandle;

    if (pool->lock)
    	dcpl_mutex_lock(pool->lock);

    wHandle = _checkoutHandle(pool, handleRef);
    if (pool->lock)
        dcpl_mutex_unlock(pool->lock);

    return wHandle;
}

short getHandleUseCount(handlePool_t *pool, short handleRef)
{
    handle_s * wHandle;
    int uc = -1;

    if (pool->lock)
    	dcpl_mutex_lock(pool->lock);
    wHandle =_getHandle(pool, handleRef);
    if (wHandle)
    	uc = wHandle->useCount;
    if (pool->lock)
        dcpl_mutex_unlock(pool->lock);

    return uc;
}

void * checkoutObject(handlePool_t *pool, short handleRef)
{
    handle_s * wHandle;
    void * ret = NULL;

    if (pool->lock)
    	dcpl_mutex_lock(pool->lock);

    wHandle = _checkoutHandle(pool, handleRef);
    if (wHandle)
        ret = wHandle->pObject;

    if (pool->lock)
        dcpl_mutex_unlock(pool->lock);

    return ret;
}

handle_s * handleIterate(handlePool_t *pool, short typeFilter, handle_it_cbk callback, void * param)
{
    handle_s * wHandle = NULL;
    int i;

    if (pool->lock)
    	dcpl_mutex_lock(pool->lock);

    for (i = 0; i < pool->bucketNumber; i++)
    {
        for (wHandle = pool->buckets[i]; wHandle;	wHandle = wHandle->next) {
            if (wHandle && (typeFilter < 0 || wHandle->type == typeFilter)
            			&& callback(wHandle, param))
            	goto exit;
        }
    }
exit:
    if (pool->lock)
        dcpl_mutex_unlock(pool->lock);
    return wHandle;
}

DC_BOOL translate_cbk(handle_s * h, struct translate_info * param)
{
	return param->cbk(h->type, h->pObject, param->param);
}

handle_s * handlePoolIterate(handlePool_t *pool, short typeFilter, handle_cbk callback, void * param)
{
	struct translate_info t_info;
	t_info.cbk = callback;
	t_info.param = param;

	return handleIterate(pool, typeFilter, (handle_it_cbk)translate_cbk, &t_info);
}

void deleteHandles(handlePool_t *pool, short type)
{
    handle_s * wHandle, *rHandle;
    int i;

    if (pool->lock)
    	dcpl_mutex_lock(pool->lock);

    for (i = 0; i < pool->bucketNumber; i++)
    {
        wHandle = pool->buckets[i];

        while (wHandle && (type < 0 || wHandle->type == type))
        {
            pool->buckets[i] = wHandle->next;
            wHandle->next = pool->unusedPool;
            pool->unusedPool = wHandle;
            if (pool->releaseHook)
                pool->releaseHook(pool->unusedPool);
            wHandle = pool->buckets[i];
        }
        for (; wHandle;	wHandle = wHandle->next)
        {
            if (wHandle->next && (type < 0 || wHandle->next->type == type))
            {
                rHandle = wHandle->next;
                wHandle->next = wHandle->next->next;
                rHandle->next = pool->unusedPool;
                pool->unusedPool = rHandle;
                if (pool->releaseHook)
                    pool->releaseHook(pool->unusedPool);
            }
        }
    }

    if (pool->lock)
        dcpl_mutex_unlock(pool->lock);
}

int releaseHandleUnsafe(handlePool_t *pool, short handleRef)
{
    handle_s * handle;
    int ret = 0;
	struct lru_entry * wLruInfo;

    handle = _getHandle(pool, handleRef);
    if (handle)
    {
        handle->useCount--;
        switch (handle->useCount)
        {
        case 0:
			ret = deleteHandle(pool, handleRef);
			break;
        case 1:
        	wLruInfo = findLRU(pool, handle->type);
        	if (wLruInfo)
           		wLruInfo->hCount++;	// get it back in the LRU count
            break;
        }
        DPWSLOG4(DC_HANDLES, "\t-> release handle %d, type %d, use count %d, pool %p\n", handle->handleRef, handle->type, handle->useCount, pool);
    }
    else
        ret = -1;

    return ret;
}

int releaseHandle(handlePool_t *pool, short handleRef)
{
    int ret = 0;

    if (pool->lock)
    	dcpl_mutex_lock(pool->lock);

	ret = releaseHandleUnsafe(pool, handleRef);

    if (pool->lock)
        dcpl_mutex_unlock(pool->lock);

    return ret;
}
