/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2122 $
|                     $Date: 2009-02-25 17:48:23 +0100 (mer, 25 fév 2009) $
\*============================================================================*/

/******************************************************************************\
 *                            Dynamic array utilities                         *
\******************************************************************************/

#include "dcCOMN_DynArray.h"
#include "dcCOMN_Tools.h"

#include <stdlib.h>
#include <string.h>

/*----------------------------------- Types ----------------------------------*/

struct find_info {
	da_cmp_cbk cbk;
	void * key;
};

/*------------------------- Static Functions prototypes ----------------------*/

static void * std_alloc(int mod, void * param, size_t len);
static void std_free(int mod, void * param, void * buf);
static DC_BOOL da_strdup (int mod, struct da_allocator * allocator, char **dest, char **src, void *param);
static DC_BOOL find_bigger(void * p_entry, struct find_info * f_info);

/*----------------------------------------------------------------------------*/

static void * std_alloc(int mod, void * param, size_t len) { return DC_MALLOC(mod, len);}
static void std_free(int mod, void * param, void * buf) { DC_FREE(mod, buf);}

const da_allocator_t default_allocator = {std_alloc, std_free, NULL},  *p_default_allocator = &default_allocator;

int da_new_entry(dyn_array_t * tab)
{
    // test if table size is sufficient, if not allocate and copy
    if (tab->size == 0)
    {
        tab->size = tab->inc + 1;
        tab->tab = tab->allocator->alloc_cbk(tab->mod, tab->allocator->param, tab->size * tab->f_size);
    	if (!tab->tab)
    		return -1;
    }
    else if (tab->nb >= tab->size-1)
    {
        void * t;
        int s = tab->size;
        tab->size = tab->size + tab->inc;
        t = tab->tab;
        tab->tab = tab->allocator->alloc_cbk(tab->mod, tab->allocator->param, tab->size * tab->f_size);
    	if (!tab->tab) {
            tab->allocator->free_cbk(tab->mod, tab->allocator->param, t);
    		return -1;
    	}
        memcpy(tab->tab, t, s * tab->f_size);
        tab->allocator->free_cbk(tab->mod, tab->allocator->param, t);
    }
    // increment count and add null at end of table
    tab->nb++;
    memset(GET_ENTRY(tab, tab->nb), 0, tab->f_size);
    return 0;
}

void da_remove_entry(dyn_array_t * tab, int index, da_free_entry_cbk free_hook)
{
    char * p = (char*)GET_ENTRY(tab, index);
    int tail_size;
    if (free_hook)
        free_hook(tab->mod, tab->allocator, p);
    tail_size = tab->nb - 1 - index;
    if (tail_size > 0)
        memmove(p, p + tab->f_size, tail_size * tab->f_size);
    tab->nb--;
    memset(GET_ENTRY(tab, tab->nb), 0, tab->f_size);
}

int da_insert_entry(dyn_array_t * tab, int index)
{
    int ret = 0;
    if (tab->nb > 0 && index < tab->nb)
    {
        char * p1 = NULL, * p2 = NULL;
        int tail_size;
        p1 = (char*)GET_ENTRY(tab, index);
        tail_size = tab->nb - index;	// do it before !
        ret = da_new_entry(tab);
        if (!ret)
        {
	        p1 = (char*)GET_ENTRY(tab, index);	// do it after !
	        for (p2 = p1 + (tail_size * tab->f_size) - 1; p2 >= p1; p2--)
	            *(p2 + tab->f_size) = *p2;
        }
    }
    else
        ret = da_new_entry(tab);

    return ret;
}

int da_init_copy(dyn_array_t * tab, void* entries, int nentries)
{
    tab->size = nentries + tab->inc;
    tab->tab = tab->allocator->alloc_cbk(tab->mod, tab->allocator->param, tab->size * tab->f_size);
	if (!tab->tab)
		return -1;
    memcpy(tab->tab, entries, nentries * tab->f_size);
    tab->nb = nentries;
    memset(GET_ENTRY(tab, tab->nb), 0, tab->f_size);
    return 0;
}

int da_browse(dyn_array_t * da, da_browse_cbk hook, void * param)
{
    int i;
    for (i = 0; i < da->nb && !hook(GET_ENTRY(da, i), param); i++);
    return i;
}

static DC_BOOL da_strdup (int mod, struct da_allocator * allocator, char **dest, char **src, void *param)
{
	if (*src && (*dest = (char*)allocator->alloc_cbk(mod, allocator->param, (unsigned long)strlen(*src) + 1)))
        strcpy(*dest, *src);
    return DC_TRUE;
}

DA_TYPED(str) * da_string_array_dup(DA_TYPED(str) * dest, DA_TYPED(str) * src)
{
	DA_COPY(dest, src, da_strdup, NULL);
    return dest;
}

static DC_BOOL find_bigger(void * p_entry, struct find_info * f_info)
{
	return f_info->cbk(p_entry, f_info->key) > 0;
}

int da_add_entry_sorted(dyn_array_t * da, da_cmp_cbk cbk, void * key)
{
	struct find_info f_info;
	int pos;

	f_info.cbk = cbk;
	f_info.key = key;

	pos = DA_BROWSE(da, find_bigger, &f_info);
	return da_insert_entry(da, pos) ? -1 : pos;
}

int da_dichotomy_find(dyn_array_t * array, da_cmp_cbk cbk, const void * key)
{
	int lo = 0, hi = array->nb - 1, mid, comp;
	void * p_entry;
	while (hi >= lo)
	{
		mid = (lo + hi) / 2;
		p_entry = GET_ENTRY(array, mid);
		comp = cbk(p_entry, key);
		if (!comp)
			return mid;
		if (comp > 0)
			hi = mid - 1;
		else
			lo = mid + 1;
	}
	return -1;
}

DC_BOOL da_is_included(dyn_array_t *a, dyn_array_t *b, da_cmp_cbk compare)
{
    int i, j;
    if (a == NULL)
        return DC_TRUE;
    if (b == NULL)
        return DC_FALSE;
    for (i = 0; i < a->nb; i++)
    {
        for (
            j = 0;
            j < b->nb && !compare(GET_ENTRY(a, i), GET_ENTRY(b, j));
            j++
        );
        if (j == b->nb)
            return DC_FALSE;
    }
    return DC_TRUE;
}

void da_empty(dyn_array_t * da, da_free_entry_cbk free_hook)
{
    if (free_hook) {
		int i;
		for (i = 0; i < da->nb; i++)
			free_hook(da->mod, da->allocator, GET_ENTRY(da, i));
    }
    da->nb = 0;
}

void da_free(dyn_array_t * da, da_free_entry_cbk free_hook)
{
	da_empty(da, free_hook);
	da->allocator->free_cbk(da->mod, da->allocator->param, da->tab);
    da->tab = NULL;
    da->size = 0;
}

int da_copy(dyn_array_t * dest, dyn_array_t * da, da_deep_copy_cbk dup_hook, void * param)
{
    size_t buf_len = da->nb * da->f_size;
	if (dest->f_size != da->f_size)
		return -1;
    dest->tab = dest->allocator->alloc_cbk(dest->mod, dest->allocator->param, buf_len + da->f_size);
    if (!dest->tab) {
    	dest->nb = dest->size = 0;
    	return -1;
    }
	dest->nb = da->nb;
	dest->size = da->nb + 1;
    memset(GET_ENTRY(dest, dest->nb), 0, dest->f_size);
    if (dup_hook) {
        int i;
	    for (i = 0; i < da->nb; i++) {
	        if (!dup_hook(dest->mod, dest->allocator, GET_ENTRY(dest, i), GET_ENTRY(da, i), param)) {
	        	dest->nb = i;
	        	return -1;
	        }
	    }
    }
    else
        memcpy(dest->tab, da->tab, buf_len);

    return 0;
}
