/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2091 $
|                     $Date: 2009-02-17 15:38:44 +0100 (mar, 17 fév 2009) $
\*============================================================================*/

#ifndef TOOLS_H
# define TOOLS_H

#include "dc/dc_Ctypes.h"
#include "dcCOMN_DynArray.h"
#include "dcCOMN_Mem.h"


/*----------------------------------------------------------------------------*\
 *                Error processing & parameter control macros                 *
\*----------------------------------------------------------------------------*/

#define DC_FUNC_ERROR	error:

#define DC_ERROR_ASSERT(exp, err)	{if (!(exp)) {ret = err;goto error;}}
#define DC_ERROR_ASSERT_ALLOC(assign)	DC_ERROR_ASSERT(assign, DPWS_ERR_EOM)
#define DC_ERROR_ASSERT_NO_RC(exp)	{if (!(exp)) goto error;}
#define DC_ERROR_ASSERT_CALL(func)	{if ((ret = func)) goto error;}
#define DC_ERROR_THROW(err)	{ret = err;goto error;}

#define DC_RETURN_ASSERT(exp, err)	if (!(exp)) return err
#define DC_RETURN_ASSERT_ALLOC(assign)	DC_RETURN_ASSERT(assign, DPWS_ERR_EOM)
#define DC_RETURN_ASSERT_CALL(func)	if ((ret = func)) return ret

#ifdef DC_API_PARAM_CONTROL
# define DC_CHECK_PARAM(exp) DC_RETURN_ASSERT(exp, DPWS_ERR_INVALID_PARAMETER)
# define DC_CHECK_PARAM_NO_RC(var, exp, ret) if (!(exp)) {(var) = DPWS_ERR_INVALID_PARAMETER; return ret;}
# define DC_CHECK_PARAM_RETURN(exp, ret) if (!(exp)) return ret
#else
# define DC_CHECK_PARAM(exp)
# define DC_CHECK_PARAM_NO_RC(var, exp, ret)
# define DC_CHECK_PARAM_RETURN(exp, ret)
#endif

/*----------------------------------------------------------------------------*\
 *                                 XML UTILITIES                              *
\*----------------------------------------------------------------------------*/

/** Converts an XML duration into an integer.
 * @param[in,out] p The XML duration. Note that this string may be modified during the
 * call. Do not call if the string must not be modified.
 * @return The duration in seconds.
 */
int32_t xmlduration2duration(char *p);

/** Converts an integer duration into an XML duration.
 * @param[out] dest the destination buffer. Must be DURATION_MAX_SIZE at least.
 * @param duration The duration in seconds to convert.
 * @return The modified buffer filled with an XML duration string.
 */
char * duration2xmlduration(char * dest, int32_t duration);


/*----------------------------------------------------------------------------*\
 *                                 UTILITY MACROS                             *
\*----------------------------------------------------------------------------*/

/** Test if a character is whitespace.
 * @param c A character to test.
 * @return A boolean value.
 */
#define IS_WS(c) (c == ' ' || c == '\t' || c == 0xA || c == 0xD)

/** Test if a character is a digit.
 * @param c A character to test.
 * @return A boolean value.
 */
#define IS_DIGIT(c) (c >= '0' && c <= '9')

#ifndef MAX
/** Return the maximum value between too numeric values.
 * @param a 1st term
 * @param a 2nd term
 * @return A boolean value.
 */
#define MAX(a,b) ((a)>(b)?(a):(b))
#endif
#ifndef MIN
/** Return the minimum value between too numeric values.
 * @param a 1st term
 * @param a 2nd term
 * @return A boolean value.
 */
#define MIN(a,b) ((a)<(b)?(a):(b))
#endif

/*----------------------------------------------------------------------------*\
 *                                 COMPARISONS                                *
\*----------------------------------------------------------------------------*/

/** URI comparison function based on the RFC2396 and WS-Discovery specification.
 * @param base The reference URI string.
 * @param uri The URI string to test for inclusion.
 * @return DC_TRUE if \a base features an "ancestor" of \a uri (i.e. \a uri is
 * in \a base scope). DC_FALSE else.
 */
DC_BOOL rfc2396_match(char *base, char *uri);

/*----------------------------------------------------------------------------*\
 *                                  NETWORKING                                *
\*----------------------------------------------------------------------------*/

/** Converts a MAC address from its string hexadecimal form to its numeric form.
 * '-', ':' or none are accepted as separator.
 * @param[out] buffer that will receive the result. Must be 6 byte long.
 * @param mac_addr The string containinig the MAC address in its text
 * hexadecimal form.
 * @return A pointer on \a buffer on success. NULL for instance if the provided
 * string is too short.
 */
byte* parse_mac_address(byte * buffer, const char * mac_addr);

/** Tests is the supplied URI is a supported URL.
 * @param addr The URI to test.
 * @return If the URI scheme was identified, return a pointer to the character
 * just after the ':'. NULL else.
 */
const char * is_physical_address(const char * addr);

/*----------------------------------------------------------------------------*\
 *                                    DEBUG                                   *
\*----------------------------------------------------------------------------*/
#ifdef DPWS_DEBUG

#include  "dcDCPL_Os.h"
#include  <stdio.h>

#define DPWS_LOG_INDEX 0
#define DPWS_LOG_INDEX_SENT 1
#define DPWS_LOG_INDEX_RECV 2

extern FILE *fdebug[];

/** Debug categories enumeration that can be used for filtering printed messages.
 * Filtering is done by defining \a DPWS_DEBUG to one of the enumeration token.
 */
typedef enum {
			DC_ALL,	/**< No filtering */
			DC_REGISTRY,	/**< Messages related to the device registry */
			DC_LOCK,	/**< Messages related to mutexes */
			DC_CACHE,		/**< Messages related to the device cache */
			DC_DISCOVERY,	/**< Messages related to WS-Discovery */
			DC_HANDLES,	/**< Messages related to handles */
			DC_TRANSPORT	/**< Messages related to the network transport layer */
		} dpws_debug_cat;

# define CAT_STR_DC_REGISTRY "REGISTRY"	/**< Text for the REGISTRY category */
# define CAT_STR_DC_LOCK "LOCK"	/**< Text for the LOCK category */
# define CAT_STR_DC_HANDLES "HANDLES"	/**< Text for the HANDLES category */
# define CAT_STR_DC_CACHE "CACHE"	/**< Text for the CACHE category */
# define CAT_STR_DC_DISCOVERY "DISCOVERY"	/**< Text for the DISCOVERY category */
# define CAT_STR_DC_TRANSPORT "TRANSPORT"	/**< Text for the TRANSPORT category */

/** Internal message logging macro */
# define DPWSLOG_ANY(CAT, LOG) \
do { if (DPWS_DEBUG == CAT || DPWS_DEBUG == DC_ALL) {\
		uint32_t s, ns;\
		dcpl_get_time(&s, &ns);\
		fprintf(fdebug[DPWS_LOG_INDEX], "\n%s(%d.%03d): ", CAT_STR_##CAT, s, ns/1000000);\
		LOG;\
		fflush(fdebug[DPWS_LOG_INDEX]);\
	}\
} while(0)

/** Message logging macro using the printf format with no parameter.
 * @param CAT The debug category that will be used for filtering.
 * @param MSG Log message.
 */
# define DPWSLOG(CAT, MSG) DPWSLOG_ANY(CAT, fprintf(fdebug[DPWS_LOG_INDEX], MSG);)

/** Message logging macro using the printf format with one single parameter.
 * @param CAT The debug category that will be used for filtering.
 * @param FMT printf format for message.
 * @param P1 A parameter that will be inserted in the format string to produce
 * the log message.
 */
# define DPWSLOG1(CAT, FMT, P1) DPWSLOG_ANY(CAT, fprintf(fdebug[DPWS_LOG_INDEX], FMT, P1))

/** Message logging macro using the printf format with two parameters.
 * @param CAT The debug category that will be used for filtering.
 * @param FMT printf format for message.
 * @param P1 A parameter that will be inserted in the format string to produce
 * the log message.
 * @param P2 A parameter that will be inserted in the format string to produce
 * the log message.
 */
# define DPWSLOG2(CAT, FMT, P1, P2) DPWSLOG_ANY(CAT, fprintf(fdebug[DPWS_LOG_INDEX], FMT, P1, P2))

/** Message logging macro using the printf format with three parameters.
 * @param CAT The debug category that will be used for filtering.
 * @param FMT printf format for message.
 * @param P1 A parameter that will be inserted in the format string to produce
 * the log message.
 * @param P2 A parameter that will be inserted in the format string to produce
 * the log message.
 * @param P3 A parameter that will be inserted in the format string to produce
 * the log message.
 */
# define DPWSLOG3(CAT, FMT, P1, P2, P3) DPWSLOG_ANY(CAT, fprintf(fdebug[DPWS_LOG_INDEX], FMT, P1, P2, P3))

/** Message logging macro using the printf format with four parameters.
 * @param CAT The debug category that will be used for filtering.
 * @param FMT printf format for message.
 * @param P1 A parameter that will be inserted in the format string to produce
 * the log message.
 * @param P2 A parameter that will be inserted in the format string to produce
 * the log message.
 * @param P3 A parameter that will be inserted in the format string to produce
 * the log message.
 * @param P4 A parameter that will be inserted in the format string to produce
 * the log message.
 */
# define DPWSLOG4(CAT, FMT, P1, P2, P3, P4) DPWSLOG_ANY(CAT, fprintf(fdebug[DPWS_LOG_INDEX], FMT, P1, P2, P3, P4))

/** Logs data sent or received on a socket.
 * @param DIR One of 'SENT' or 'RECV.
 * @param A pointer o the data to be logged.
 * @param The length of the data to be logged.
 */
# define DPWSMSG(DIR, MSG, LEN) \
do { if (fdebug[DPWS_LOG_INDEX_##DIR] && (LEN) > 0)\
		{ fwrite((MSG), 1, (LEN), fdebug[DPWS_LOG_INDEX_##DIR]);\
		  fflush(fdebug[DPWS_LOG_INDEX_##DIR]);\
		}\
} while (0)
#else
/** Dummy macro so that to compile even when DWPS_DEBUG is not defined */
# define DPWSLOG(CAT, MSG)
/** Dummy macro so that to compile even when DWPS_DEBUG is not defined */
# define DPWSLOG1(CAT, FMT, P1)
/** Dummy macro so that to compile even when DWPS_DEBUG is not defined */
# define DPWSLOG2(CAT, FMT, P1, P2)
/** Dummy macro so that to compile even when DWPS_DEBUG is not defined */
# define DPWSLOG3(CAT, FMT, P1, P2, P3)
/** Dummy macro so that to compile even when DWPS_DEBUG is not defined */
# define DPWSLOG4(CAT, FMT, P1, P2, P3, P4)
/** Dummy macro so that to compile even when DWPS_DEBUG is not defined */
# define DPWSMSG(DIR, MSG, LEN)
#endif


#endif
