/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2250 $
|                     $Date: 2009-04-07 17:21:13 +0200 (mar, 07 avr 2009) $
\*============================================================================*/
#ifndef HANDLE_H_
#define HANDLE_H_

#include "dc/dc_Types.h"
#include "dcDCPL_Os.h"

/*----------------------------------------------------------------------------*\
 *                                    HANDLES                                 *
\*----------------------------------------------------------------------------*/
#define ANY_TYPE		-1	/**< Handles point on objects that have types. This types means all types. */

/** The handle structure. */
typedef struct handleInfo {
	short		handleRef;	/**< The numeric ID called "handle reference" used as key for the pool hastable (modulo on the href). */
	short		type;	/**< A flag for pointed object distinction. */
	void *		pObject;	/**< The pointer on the managed object. */
	short		useCount;	/**< The current number of checkout performed by API user. When 0, the handle object is free and the handle recycled. */
	uint16_t	ts;		/**< For the case the owning pool uses a LRU, a timestamp set every time the handle is accessed. */
	struct handleInfo * next;	/**< Pointer to the next handle structure in the same bucket. */
} handle_s;

/** A LRU entry for a givent type of object. */
struct lru_entry {
	short		type;	/**< The type of object being limited by an LRU. */
	uint16_t	cit;	/**< Current incremental time (in fact a count not a clock time). Circular. */
	uint16_t	hMax;	/**< The maximum number of handles which is also the maximum handle reference since handles are recycled. */
	uint16_t	hCount;	/**< Current handle count. */
	struct lru_entry * next;	/**< Next LRU node */
};

/** Handle release callback called when the handle is not referenced any more */
typedef int (*handle_release_cbk) (handle_s *);

/** The handle pool structure. */
typedef struct handlePoolInfo {
	int mod; /**< The software module to which belongs the memory allocation for the pool.*/
	short bucketNumber;	/**< The number of buckets in the hashtable. */
	short lastHandle;	/**< Last granted handle reference. */
	handle_s **buckets;	/**< Handle hashtable where the hashcode is href%bucketNumber */
	handle_s *unusedPool;	/**< Chained list of unused handles ready for recycling. */
	dcpl_mutex_t *lock;	/**< An optional mutex for protecting concurrent handle pool access. */
	handle_release_cbk releaseHook;	/**< An optional callback for object content freeing. */
	struct lru_entry * lruInfo;	/**< A potentially empty list of LRU limiting the number of handles in the pool. */
} handlePool_t;

/** Initializes the handle pool structure.
 * @param mod The software module to which belongs the memory allocation for
 * the pool.
 * @param pool A pointer on an uninitialized handle pool structure.
 * @param bucketNumber The number of buckets in the hashtable.
 * @param lock An optional mutex for protecting concurrent handle pool access
 * @param releaseHook An optional callback for object content freeing.
 * @return 0 if the init was OK. -1 if EOM.
 */
int createHandlePool(int mod, handlePool_t *pool, short bucketNumber, dcpl_mutex_t * lock, handle_release_cbk releaseHook);

/** Creates or modifies an LRU for a given object type.
 * It the LRU already exists and the number of handles exceeds the existing
 * number of handles, the oldest are removed to reach the target.
 * @param pool The handle pool.
 * @param type The type of object limited (ANY_TYPE accepted).
 * @param maxHandles The maximum number of admitted handles.
 * @return 0 if OK.
*/
int setHandleLRU(handlePool_t *pool, short type, uint16_t maxHandles);

/** Empties and free the handle pool allocated structures.
 * @param pool The handle pool to free.
 */
void deleteHandlePool(handlePool_t *pool);

/** Creates a new handle for an object.
 * The use count is set to 1.
 * @param pool The handle pool.
 * @param type The type of object to which the handle will point.
 * @param pObject A pointer to the object.
 * @return The granted handle or -1 if EOM.
 */
short createHandle(handlePool_t *pool, short type, void * pObject);

/** Retrieves a handle structure without increasing the use count.
 * @param pool The handle pool.
 * @param handleRef The reference for the handle searched.
 * @return A pointer on the handle structure if found, -1 else.
 */
handle_s * getHandle(handlePool_t *pool, short handleRef);

/** Retrieves a handle structure increasing the use count.
 * @param pool The handle pool.
 * @param handleRef The reference for the handle searched.
 * @return A pointer on the handle structure if found, -1 else.
 */
handle_s * checkoutHandle(handlePool_t *pool, short handleRef);

/** Retrieves the use count for a handle.
 * @param pool The handle pool.
 * @param handleRef The reference of the handle.
 * @return A use count if if handle was found, -1 else.
 */
short getHandleUseCount(handlePool_t *pool, short handleRef);

/** Retrieves a handle object without increasing the use count.
 * @param pool The handle pool.
 * @param handleRef The reference for the handle searched.
 * @return A pointer on the pointed object if found, -1 else.
 */
void * getObject(handlePool_t *pool, short handleRef);

/** Retrieves a handle object increasing the use count.
 * @param pool The handle pool.
 * @param handleRef The reference for the handle searched.
 * @return A pointer on the pointed object if found, -1 else.
 */
void * checkoutObject(handlePool_t *pool, short handleRef);

/** Retrieves a handle object without increasing the use count and without usin
 * the pool lock if any.
 * This function should be used for instance within the context of a browsing
 * already taking the handle pool lock (non-reentrant locks)
 * @param pool The handle pool.
 * @param handleRef The reference for the handle searched.
 * @return A pointer on the pointed object if found, -1 else.
 */
void * getObjectUnsafe(handlePool_t *pool, short handleRef);

/** Retrieves a handle object without increasing the use count but only if of a
 * given type
 * @param pool The handle pool.
 * @param handleRef The reference for the handle searched.
 * @param The type filter
 * @return A pointer on the pointed object if found an d of the right type, -1
 * else.
 */
void * getTypedObject(handlePool_t *pool, short handleRef, short type);

/** Handle pool object browsing callback.
 * @param type The type of the current object.
 * @param p_object A pointer on the current object.
 * @param param Callback user data.
 * @return Should return DC_TRUE when the iterator must stop (match).
 */
typedef DC_BOOL (*handle_cbk) (short type, void * p_object, void * param);

/** Iterate on objects of a pool.
 * @param pool The handle pool.
 * @param typeFilter The type of objects that will be browsed. If ANY_TYPE all
 * handles are considered.
 * @param param Callback user data.
 * @return The handle on which the iteration stopped or NULL if the browsing
 * could be done completely.
 */
handle_s * handlePoolIterate(handlePool_t *pool, short typeFilter, handle_cbk callback, void * param);

/** Handle pool handle browsing callback.
 * @param h A pointer on the current handle structure.
 * @param param Callback user data.
 * @return Should return DC_TRUE when the iterator must stop (match).
 */
typedef DC_BOOL (*handle_it_cbk) (handle_s * h, void *param);

/** Iterate on handles of a pool.
 * @param pool The handle pool.
 * @param typeFilter The type of objects that will be browsed. If ANY_TYPE all
 * handles are considered.
 * @param param Callback user data.
 * @return The handle on which the iteration stopped or NULL if the browsing
 * could be done completely.
 */
handle_s * handleIterate(handlePool_t *pool, short typeFilter, handle_it_cbk callback, void * param);

/** Releases a handle use, decrementing its use count.
 * @param pool The handle pool.
 * @param handleRef The handle reference.
 * @return 0 if the operation could be performed, -1 else.
 */
int releaseHandle(handlePool_t *pool, short handleRef);

/** Releases a handle use, decrementing its use count but without taking the
 * pool lock.
 * This function should be used for instance within the context of a browsing
 * already taking the handle pool lock (non-reentrant locks)
 * @param pool The handle pool.
 * @param handleRef The handle reference.
 * @return 0 if the operation could be performed, -1 else.
 */
int releaseHandleUnsafe(handlePool_t *pool, short handleRef);

/** Recycles all handles of a given type without considering the use count.
 * Used mainly for memory freeing during shutdown but in an order that takes
 * dependencies into account thanks to the type filter.
 * @param pool The handle pool.
 * @param typeFilter The type of objects that will be browsed. If ANY_TYPE all
 * handles are considered.
 */
void deleteHandles(handlePool_t *pool, short typeFilter);

#endif /*HANDLE_H_*/
