/*============================================================================*\
|                                                                              |
|                          SOA4D DPWSCore Samples                              |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|                                                                              |
|        + File info:                                                          |
|                     $Revision: 2197 $
|                     $Date: 2009-03-11 18:56:28 +0100 (mer, 11 mar 2009) $
\*============================================================================*/
#include <stdio.h>

#include "dc/dc_Dpws.h"	// Main DPWSCore API include file.

#include "wshStub.h" // Generated stub & skeleton include file.
#include "WasherClient.h"	// Washing machine client menu.
#include "serverDPWS.h"	// OS-dependent code (especially server loop).

/*
 *	Definitions
 */

#define MAX_SCOPE_LENGTH 255	// Maximum length for computed scope.
#define SAMPLE_SCOPE "http://www.soa4d.org/DPWS/Samples/WashingMachine/Scope"	// The scope for the washing machine device

#define SAMPLE_NS "http://www.soa4d.org/DPWS/Samples/WashingMachine"	// The namespace of the type supported by the searched device (must be the same as in server.c).
#define WASHER_TYPE "WashingMachine"	// The local name of the type supported by the searched device (must be the same as in server.c).


/* Callback function called when new device is added or update in the discovery cache. */
static void device_joining(struct dpws *dpws, short href_device) {
	printf("\n<-- New device on the LAN: %s\n", dpws_cache_get_uuid(dpws, href_device));
}

/* Callback function called when a device is removed from the discovery cache. */
static void device_leaving(struct dpws *dpws, short href_device) {
	printf("\n<-- Device %s leaving the LAN.\n", dpws_cache_get_uuid(dpws, href_device));
}

static void subscription_cancelled(struct event_end_info * eei)
{
	printf("\n<-- Subscription %s\n    from %s cancelled.\n",
			eei->subscription_manager->subscription_id, eei->subscription_manager->address);
	printf(" Status:  %s\n Reason:  %s\n", eei->status, eei->reason);
}

/*
 *	Runtime data
 */

struct dpws	listen_dpws,	// runtime structure used by the event listener.
			dpws;	// runtime structure used by the client.

/*
 *  Utilities
 */

static char diversified_scope[MAX_SCOPE_LENGTH];	// computed scope buffer

/* Function adding the suffix to default scope (if suffix was defined).
 * Returns 0 if ok 1 if error (string too long).
 */
static int compute_diversified_scope(char *scope_suffix)
{
	strcpy(diversified_scope, SAMPLE_SCOPE);
	if (scope_suffix) {
		if (sizeof(SAMPLE_SCOPE) + strlen(scope_suffix) > MAX_SCOPE_LENGTH)
			return 1;
		if (scope_suffix[0] != '/')
			strcat(diversified_scope, "/");
		strcat(diversified_scope, scope_suffix);
	}
	return 0;
}

/* Function printing error message. */
static void print_error(char * msg, struct dpws * dpws) {
	fprintf(stderr, "\nError occured : %s.\n%s\n\n", msg, dpws_get_error_msg(dpws));
}

/* Function displaying usage summary. */
static void simple_usage() {
	fprintf(stderr, "\nUsage: client [-s <scopeDiversifier>] [-p <httpPort>] [-i <ipMode>]\n");
	fprintf(stderr, "Use -?, -h or -help for more information\n\n");
	exit(1);
}

/* Function displaying complete program usage. */
static void usage() {
	fprintf(stderr, "\nUsage: client [-s <scopeDiversifier>] [-p <httpPort>] [-i <ipMode>]\n");
	fprintf(stderr, "\nOptions : \n");
	fprintf(stderr, "\n -s <scopeDiversifier>\n");
	fprintf(stderr, "\n\tThe <scopeDiversifier> is used to avoid conflicts on the local\n");
	fprintf(stderr, "\tnetwork by participating in building the scope of the device.\n");
	fprintf(stderr, "\n -p <httpPort>\n");
	fprintf(stderr, "\n\tThe <httpPort> is the http port used to listen to soap messages.\n");
	fprintf(stderr, "\n -i <ipMode>\n");
	fprintf(stderr, "\n\tThe <ipMode> should be one of '4', '6' or 'dual' according you want\n");
	fprintf(stderr, "\tthe server to listen on all IPv4, IPv6 or both IP addresses. Default is 'dual'.\n");
	fprintf(stderr, "\nNotes : \n");
	fprintf(stderr, "\n\tThe client should be started after the server.\n\n");
	exit(1);
}

/*
 *  Main function performing server initialization.
 *	Usage: client [-s <scope_suffix>] [-p <listenPort>] [-i <ipMode>]
 */
int main(int argc, char* argv[]) {

	short device_proxy = -1, * device_proxies, event_endpoint;
	int nb_devices, port = 8834, i, res = 0, status;
	char *scope_suffix = NULL;
	dc_ip_filter_t ip_selector = {
			NULL,	// consider all interfaces
			DC_FALSE,	// but not loopback
			DC_PROTO_ANY,	// default program setting : dual IPv4/v6
			0,	// no address filtering
			NULL	// no address filtering
		};
	char ** scopes;
	struct device_metadata *metadata;

	// Command line parsing
	if (argc > 1 && (!strcmp(argv[1], "-?") || !strcmp(argv[1], "-h") || !strcmp(argv[1], "-help"))) {
		usage();
	}

	if ((argc < 1) || (argc > 6))
		simple_usage();

	for (i = 1; i < argc; i++) {
		if (!strcmp(argv[i], "-s")) {
			if (++i == argc)
				simple_usage();
			scope_suffix = argv[i];
		} else if (!strcmp(argv[i], "-p")) {
			if (++i == argc)
				simple_usage();
			port = atoi(argv[i]);
			if (port == 0)
				simple_usage();
		} else if (!strcmp(argv[i], "-i")) {
			if (++i == argc)
				simple_usage();
			if (argv[i][0] == '4')
				ip_selector.proto = DC_PROTO_INET;
			else if(argv[i][0] == '6')
				ip_selector.proto = DC_PROTO_INET6;
			// else DUAL
		} else
			simple_usage();
	}

	// Compute diversified scope, the scope will be used to discover devices on the network.
	// If a suffix has been used only the servers started using the same suffix will be
	// detected.
	if (compute_diversified_scope(scope_suffix)) {
		fprintf(stderr, "scope_suffix is too long, should not exceed %d characters\n", MAX_SCOPE_LENGTH - sizeof(SAMPLE_SCOPE));
		simple_usage();
	}

	/* Client program description (client and listener initialization) */
	// 1. Initialize the DPWS stack.
	// 2. Initialize the dpws structure for client operations.
	// 3. Configure the event listener.
	// 4. Initialize the event listener.
	// 5. Start the event listener loop.
	// 6. Find device proxy implementing the service to be invoked.
	// 7. Retrieve and display the device metadata.
	// 8. Start the washer client (event subscription and loop on service invoker).
	// 9. Stop the event listener.

	/* 1. Initialize the DPWS stack */

	// This is the first function to be called when using the DPWS stack.
	// This version allows multiple address selection.
	if ((status = dpws_init_ex(&ip_selector, NULL, DPWS_DEFAULT_VERSION))) {
		fprintf(stderr, "Could not initialize the DPWSCore stack (err %d)\n\n", status);
		exit(1);
	}

	/* 2. Initialize the dpws structure for client operations. */
	if ((status = dpws_client_init(&dpws, NULL) != DPWS_OK)) {
		fprintf(stderr, "Could not initialize client side of the DPWSCore stack (err %d)\n\n", status);
		exit(1);
	}

	/* 3. Configure the event listener. */

	// Configure DPWS server for HELLO and BYE messages reception
	DPWS_SET_PTR_ATT(DC_CACHE_HANDLE, DPWS_PTR_CALLBACK_HELLO, device_joining);
	DPWS_SET_PTR_ATT(DC_CACHE_HANDLE, DPWS_PTR_CALLBACK_BYE, device_leaving);
	DPWS_SET_INT_ATT(DC_REGISTRY_HANDLE, DPWS_INT_HTTP_PORT, port);	// Configure the HTTP listen port.

	event_endpoint = dpws_create_endpoint();	// Create endpoint for event reception.
	if (!event_endpoint) {
		fprintf(stderr, "Could not create endpoint...\n");
		goto error;
	}
	// Configure the function handling incoming events
	DPWS_ADD_PTR_ATT(event_endpoint, DPWS_PTR_HANDLING_FUNCTION, wsh_handle_event);
	DPWS_ADD_PTR_ATT(event_endpoint, DPWS_PTR_CALLBACK_EVENT_END, subscription_cancelled);

	/* 4. Initialize the event listener. */
	if ((status = dpws_server_init(&listen_dpws, NULL)))	// initializes the dpws structure for server loop.
	{
		fprintf(stderr, "Could not initialize DPWS server loop (err %d)\n", status);
		goto error;
	}

	/* 5. Start the event listener loop. */
	// Creates a new thread to serve incoming messages. In the current sample,
	// a thread is created for each received message.
	if (bootServer(&listen_dpws)) {
		fprintf(stderr, "Could not boot server...\n");
		goto error;
	}

	/* 6. Find device proxy implementing the service to be invoked. */

	// The dpws_lookup function retrieves device proxies using the WS-Discovery
	// feature of the DPWSCore stack.
	// It will first look for proxies into the local cache, and then generate a
	// WS-Discovery Probe message if the number of requested proxies is not
	// available in the cache.
	// Note that a type and/or a scope filter can be used but in a very simple
	// form: dpws_lookup_ex function should be used to specify more parameters.

	printf("<-- Looking for device :\n");
	printf(" with type : {%s}%s\n", SAMPLE_NS, WASHER_TYPE);
	printf(" with scope : %s\n", diversified_scope);

	nb_devices = 1;	// The program searches a single device.
	device_proxies = dpws_lookup(&dpws, SAMPLE_NS, WASHER_TYPE, diversified_scope, &nb_devices);

	/* Handling of the dpws_lookup function return values. */
	if (nb_devices != 1) {
		if (nb_devices == 0)
			printf("\nNo Device found, please check probe parameters.\n");
		else
			print_error("Could not execute dpws_lookup", &dpws);
		goto error;
	}

	device_proxy = device_proxies[0];

	// Make sure the device proxy will not be erased from the cache using
	// dpws_pin_proxy because next dpws_end call will release the returned array
	// reference so the potential reception of BYE message (if a discovery
	// listener was running, which is not the case however in this sample) may
	// erase the proxy.
	if (dpws_pin_proxy(device_proxy) != DPWS_OK) {
		print_error("Could not execute dpws_pin_proxy", &dpws);
		goto error;
	}

	// Free dynamically allocated memory used for request processing and releases
	// proxies retrieved using a lookup API. This is good practice to call it
	// after every call to avoid memory peak.
	if (dpws_end(&dpws) != DPWS_OK) {
		print_error("Could not execute dpws_end", &dpws);
		goto error;
	}

	/* 7. Retrieve and display the device metadata. */

	printf("<-- Getting device metadata\n");
	metadata = dpws_get_device_metadata(&dpws, device_proxy);	// The dpws_get_device_metadata uses the WS-Transfer Get request
																// and the WS-MetadataExchange message format with no caching.
	if (metadata) {
		printf("--> Friendly name: %s\n    Model: %s\n", metadata->device_info.friendly_name, metadata->model_info.model_name);
		scopes = dpws_cache_get_scopes(&dpws, device_proxy);	// another API allow cached scope access (part of WS-Discovery data)
		for (; scopes && *scopes; scopes++)
			printf("    Scope: %s\n", *scopes);
	} else {
		print_error("Could not get device metadata", &dpws);
		goto error;
	}

	if (dpws_end(&dpws) != DPWS_OK) {	// Resets structure especially memory allocated for request processing.
		print_error("Could not execute dpws_end", &dpws);
		goto error;
	}

	/* 8. Start the washer client (loop on service invoker). */
	washerClient(&dpws, device_proxy, event_endpoint);

error:
	// Releasing handle on device proxy
	if (device_proxy >= 0) {
		if (dpws_release_proxy(device_proxy) != DPWS_OK)
			print_error("Could not execute dpws_release_proxy", &dpws);
	}

	// Free message processing memory
	if (dpws_end(&dpws) != DPWS_OK)
		print_error("Could not execute dpws_end", &dpws);

	/* 9. Stop the event listener. */

	stopServer();	// 1. Calls the dpws_stop_server function that closes all
	// open listening sockets, schedules the 'Eventing end' notifications
	// and Bye WS-Discovery messages for all the active devices and blocks
	// until all server processing is finished and the main loop blocking
	// on dpws_accept() can exit.
	// 2. Uses platform-specific thread APIs to wait for the main server
	// loop thread dies.

	// Cleans the DPWS stack.
	if ((status = dpws_shutdown()))
		fprintf(stderr, "Could not shut down the DPWSCore stack (err %d)\n\n", status);
}

