/*============================================================================*\
 |                                                                              |
 |                          SOA4D DPWSCore Samples                              |
 |                                                                              |
 |           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
 |                                                                              |
 |                                                                              |
 |        + File info:                                                          |
 |                     $Revision: 2197 $
 |                     $Date: 2009-03-11 18:56:28 +0100 (mer, 11 mar 2009) $
 \*============================================================================*/
#include <stdio.h>

#include "dc/dc_Dpws.h"	// Main DPWSCore API include file.

#include "litStub.h" // Generated stub & skeleton include file.
#include "lit.nsmap" // Generated namespace table include file.


/*
 *	Definitions
 */

#define MAX_SCOPE_LENGTH 255	// Maximum length for computed scope.
#define SAMPLE_SCOPE "http://www.soa4d.org/DPWS/Samples/Lights/Scope"	// The default scope used to search devices (must be the same as in server.c).

/*
IMPORTANT NOTE about types:
	The namespace does not need to be the same for device types and service port
	types as done in this sample. Indeed the qualified names of the device
	supported types are not necessarily related to a wsdl file and can feature
	an abstract device type.
*/
#define SAMPLE_NS "http://www.soa4d.org/DPWS/Samples/Lights"	// The namespace used to qualify device and service port types.
#define SIMPLE_LIGHT_TYPE "SimpleLight"	// The local name of the type supported by the searched device (must be the same as in server.c).
#define SWITCH_POWER_TYPE "SwitchPower"	// The local name of the port type supported by the service (defined in the wsdl file).

enum commands {ON, OFF, STATUS};	// List of possible commands.


/*
 *	Runtime data
 */

struct dpws	dpws;	// runtime structure used by the client.

/*
 *  Utilities
 */

static char diversified_scope[MAX_SCOPE_LENGTH];	// computed scope buffer

/* Function adding the suffix to default scope (if suffix was defined).
 * Returns 0 if ok 1 if error (string too long).
 */
static int compute_diversified_scope(char *scope_suffix)
{
	strcpy(diversified_scope, SAMPLE_SCOPE);
	if (scope_suffix) {
		if (sizeof(SAMPLE_SCOPE) + strlen(scope_suffix) > MAX_SCOPE_LENGTH)
			return 1;
		if (scope_suffix[0] != '/')
			strcat(diversified_scope, "/");
		strcat(diversified_scope, scope_suffix);
	}
	return 0;
}

/* Function printing error message. */
static void print_error(char * msg, struct dpws * dpws) {
	fprintf(stderr, "\nError occurred : %s.\n%s\n\n", msg, dpws_get_error_msg(dpws));
}

/* Function displaying usage summary. */
static void simple_usage() {
	fprintf(stderr, "\nUsage: client [-s <scopeDiversifier>] <ON|OFF|STATUS>\n");
	fprintf(stderr, "Use -?, -h or -help for more information\n\n");
	exit(1);
}

/* Function displaying complete program usage. */
static void usage() {
	fprintf(stderr, "\nUsage: client [-s <scopeDiversifier>] <Command>\n");
	fprintf(stderr, "\nOptions : \n");
	fprintf(stderr, "\n  -s <scopeDiversifier>\n");
	fprintf(stderr, "\n\tThe <scopeDiversifier> is used to avoid conflicts on the local\n");
	fprintf(stderr, "\tnetwork by participating in building the scope of the device.\n");
	fprintf(stderr, "\nCommands : \n");
	fprintf(stderr, "\n\tON\tSwitches the light on.\n");
	fprintf(stderr, "\n\tOFF\tSwitches the light off.\n");
	fprintf(stderr, "\n\tSTATUS\tRetrieves the light status.\n");
	fprintf(stderr, "\nNotes : \n");
	fprintf(stderr, "\n\tThe client should be started after the server.\n");
	fprintf(stderr, "\tIf the server has been started with a scope diversifier the client\n");
	fprintf(stderr, "\tshould use the same.\n\n");
	exit(1);
}


/*
 *  Main function performing server initialization.
 *	Usage: client [-s <scopeDiversifier>] <ON|OFF|STATUS>
 */
int main(int argc, char* argv[]) {

	/* Local variables */
	short device_proxy = -1, * device_proxies, service_proxy = -1, * service_proxies;
	int nb_devices, nb_services, status = DPWS_OK, cmd_index = 1;
	struct wsa_endpoint_ref * invokation_epr;
	struct device_metadata *metadata;
	char *scope_suffix = NULL;
	char ** scopes;
	enum commands command;
	enum lit__PowerState state;

	/* Command line parsing */
	if (argc > 1 && (!strcmp(argv[1], "-?") || !strcmp(argv[1], "-h") || !strcmp(argv[1], "-help"))) {
		usage();
	}

	if ((argc < 2) || (argc > 4)) {
		simple_usage();
	}

	if (!strcmp(argv[1], "-s")) {
		scope_suffix = argv[2];
		cmd_index = 3;
		if (argc != 4)
			simple_usage();
	} else {
		if (argc != 2)
			simple_usage();
	}

	if (!strcmp(argv[cmd_index], "ON"))
		command = ON;
	else if (!strcmp(argv[cmd_index], "OFF"))
		command = OFF;
	else if (!strcmp(argv[cmd_index], "STATUS"))
		command = STATUS;
	else
		simple_usage();

	// Compute diversified scope, the scope will be used to discover devices on the network.
	// If a suffix has been used only the servers started using the same suffix will be detected.
	if (compute_diversified_scope(scope_suffix)) {
		fprintf(stderr, "scope_suffix is too long, should not exceed %d characters\n", MAX_SCOPE_LENGTH - sizeof(SAMPLE_SCOPE));
		simple_usage();
	}

	/* Simple client program description (simple service invokation) */
	// 1. Initialize the dpws stack.
	// 2. Initialize the dpws structure that will be used.
	// 3. Find device proxy implementing the service to be invoked.
	// 4. Retrieve and display the device metadata.
	// 5. Find service proxy of the service to be invoked.
	// 6. Invoke the service.

	/* 1. Initialize the DPWS stack */

	// This is the first function to be called when using the dpws stack. This version
	// selects one single local IPv4 address.
	if ((status = dpws_init())) {
		fprintf(stderr, "Could not initialize the DPWSCore stack (err %d)\n\n", status);
		exit(1);
	}

	/* 2. Initialize the dpws structure that will be used. */
	if ((status = dpws_client_init(&dpws, NULL) != DPWS_OK))	// performs one-time client-side initialization.
	{
		fprintf(stderr, "Could not initialize client side of the DPWSCore stack (err %d)\n\n", status);
		exit(1);
	}

	/* 3. Find device proxy implementing the service to be invoked. */

	// The dpws_lookup function retrieves device proxies using the WS-Discovery
	// feature of the DPWSCore stack.
	// It will first look for proxies into the local cache, and then generate a
	// WS-Discovery Probe message if the number of requested proxies is not
	// available in the cache.
	// Note that a type and/or a scope filter can be used but in a very simple
	// form: dpws_lookup_ex function should be used to specify more parameters.

	printf("<-- Looking for device :\n");
	printf(" with type : {%s}%s\n", SAMPLE_NS, SIMPLE_LIGHT_TYPE);
	printf(" with scope : %s\n", diversified_scope);

	nb_devices = 1;	// The program searches a single device.
	device_proxies = dpws_lookup(&dpws, SAMPLE_NS, SIMPLE_LIGHT_TYPE, diversified_scope, &nb_devices);

	/* Handling of the dpws_lookup function return values. */
	if (nb_devices != 1) {
		if (nb_devices == 0)
			printf("\nNo Device found, please check probe parameters.\n");
		else
			print_error("Could not execute dpws_lookup", &dpws);
		goto error;
	}

	device_proxy = device_proxies[0];

	// Make sure the device proxy will not be erased from the cache using
	// dpws_pin_proxy because next dpws_end call will release the returned array
	// reference so the potential reception of BYE message (if a discovery
	// listener was running, which is not the case however in this sample) may
	// erase the proxy.
	if (dpws_pin_proxy(device_proxy) != DPWS_OK) {
		print_error("Could not execute dpws_pin_proxy", &dpws);
		goto error;
	}
	// Free dynamically allocated memory used for request processing and releases
	// proxies retrieved using a lookup API. This is good practice to call it
	// after every call to avoid memory peak.
	if (dpws_end(&dpws) != DPWS_OK) {
		print_error("Could not execute dpws_end", &dpws);
		goto error;
	}

	/* 4. Retrieve and display the device metadata. */

	printf("<-- Getting device metadata\n");
	metadata = dpws_get_device_metadata(&dpws, device_proxy);	// The dpws_get_device_metadata uses the WS-Transfer Get request
																// and the WS-MetadataExchange message format with no caching.
	if (metadata) {
		printf("--> Friendly name: %s\n    Model: %s\n", metadata->device_info.friendly_name, metadata->model_info.model_name);
		scopes = dpws_cache_get_scopes(&dpws, device_proxy);	// another API allow cached scope access (part of WS-Discovery data)
		for (; scopes && *scopes; scopes++)
			printf("    Scope: %s\n", *scopes);
	} else {
		print_error("Could not get device metadata", &dpws);
		goto error;
	}

	if (dpws_end(&dpws) != DPWS_OK) {	// Good practice to call it after every call to avoid memory peak.
		print_error("Could not execute dpws_end", &dpws);
		goto error;
	}

	/* 5. Find service proxy of the service to be invoked. */

	// The dpws_get_services retrieves one or several service proxies hosted by
	// a remote device using a type filtering (other APIs are available).
	// It will look for proxy available into the local cache or generate a
	// WS_Transfer/WS-MetadataExchange message if necessary.

	printf("Looking for service :\n\tOn device proxy : %d\n", device_proxy);
	printf("\tWith type : {%s}%s\n", SAMPLE_NS, SWITCH_POWER_TYPE);

	nb_services = 1;	// The program searches a single service.
	service_proxies = dpws_get_services(&dpws, device_proxy, SAMPLE_NS, SWITCH_POWER_TYPE, &nb_services);

	if (nb_services != 1)
	{
		if (nb_services == 0)
			printf("No service endpoint found, please check the port type specifications.\n");
		else
			print_error("Could not execute dpws_get_services", &dpws);
		goto error;
	}

	service_proxy = service_proxies[0];

	// Make sure the service proxy will not be erased from the cache for the same
	// reasons than for the hosting device.
	if (dpws_pin_proxy(service_proxy) != DPWS_OK) {
		print_error("Could not execute dpws_pin_proxy", &dpws);
		goto error;
	}
	if (dpws_end(&dpws) != DPWS_OK) {	// Good practice to call it after every call to avoid memory peak.
		print_error("Could not execute dpws_end", &dpws);
		goto error;
	}

	/* 6. Invoke the service. */

	// Retrieves the service proxy default endpoint reference (EPR) used for
	// invocation.
	// Several endpoints may be available and dpws_get_default_endpoint_ref
	// returns the first one. This is why one may have to use dpws_get_endpoint_refs
	// especially if the 1st EPR is not reachable.
	invokation_epr = dpws_get_default_endpoint_ref(&dpws, service_proxy);
	if (!invokation_epr) {
		print_error("Could not find default endpoint reference", &dpws);
		goto error;
	}

	// Depending on the command, make invocation. using functions defined in the litStub.h.
	switch (command)
	{
	case ON:
		printf("\n<-- Switching light ON\n");
		status = dpws_send___lit__Switch(&dpws, invokation_epr, lit__PowerState__ON);	// One way invocation (starting with dpws_send).
		break;
	case OFF:
		printf("\n<-- Switching light OFF\n");
		status = dpws_send___lit__Switch(&dpws, invokation_epr, lit__PowerState__OFF);	// One way invocation (starting with dpws_send).
		break;
	case STATUS:
		printf("\n<-- Getting the light status\n");
		status = dpws_call___lit__GetStatus(&dpws, invokation_epr, NULL, &state);	// Two-way invocation  (starting with dpws_call).
		if (!status)
			printf("\n--> The light is %s\n", state == lit__PowerState__ON ? "ON" : "OFF");
		break;
	}

	if (status)
		fprintf(stderr, dpws_get_error_msg(&dpws));	// prints potential fault message of local fault error code.

error:
	if (dpws_end(&dpws) != DPWS_OK)	// Resets structure especially memory allocated for request processing.
		print_error("Could not execute dpws_end", &dpws);
	if (device_proxy >=0) {	// Releasing handle on proxy kept using dpws_pin_proxy
		if (dpws_release_proxy(device_proxy) != DPWS_OK)
			print_error("Could not execute dpws_release_proxy", &dpws);
	}
	if (service_proxy >= 0) {	// Releasing handle on proxy kept using dpws_pin_proxy
		if (dpws_release_proxy(service_proxy) != DPWS_OK)
			print_error("Could not execute dpws_release_proxy", &dpws);
	}
	if ((status = dpws_shutdown()))	// cleans all stack data and resources.
		fprintf(stderr, "Could not shut down the DPWSCore stack (err %d)\n\n", status);

	return 0;
}

