/*============================================================================*\
|                                                                              |
|                          SOA4D DPWSCore Samples                              |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|                                                                              |
|        + File info:                                                          |
|                     $Revision: 2137 $
|                     $Date: 2009-03-02 19:10:12 +0100 (lun, 02 mar 2009) $
\*============================================================================*/
#include <stdio.h>

#include "dc/dc_Dpws.h"	// Main DPWSCore API include file.
#include "dc/dc_XMLUtils.h"	// included for XML duration conversion utilities

#include "litStub.h" // Generated stub & skeleton include file for light devices.
#include "lit.nsmap" // Generated namespace table include file for light devices.
#include "wshStub.h" // Generated stub & skeleton include file for washing machine devices.
#include "wsh.nsmap" // Generated namespace table include file for washing machine devices.
#include "prjStub.h" // Generated stub & skeleton include file for projector devices.
#include "prj.nsmap" // Generated namespace table include file for projector devices.

#include "fileio.h"	// File utility used for boot sequence & binary data persistence.
#include "serverDPWS.h"	// OS-dependent code (especially server loop).

/*
 * IMPORTANT NOTE: This multi-thread server is supplied as an example for studying purpose
 * and does not meet the usual standards of industrial software. For instance, device data
 * should be protected against concurrent access from server threads.
 */

/*
 *	Device Configuration data
 */

#define MAX_SCOPE_LENGTH 255	// Maximum length for computed scope.
#define SAMPLE_SCOPE "http://www.soa4d.org/DPWS/Samples/Home/Scope"	// The base scope for the sample.

#define BOOT_SEQ_FILE "bootseq.log"	// The name of the file holding the boot sequence number.

// Scope suffixes defining home rooms where devices will be located.
#define KITCHEN_SUFFIX_SCOPE "/Kitchen"
#define BEDROOM_SUFFIX_SCOPE "/Bedroom"
#define LIVING_SUFFIX_SCOPE "/LivingRoom"
#define LAUNDRY_SUFFIX_SCOPE "/LaundryRoom"

#define SAMPLE_NS "http://www.soa4d.org/DPWS/Samples/Home"	// The namespace used to qualify sample device types.

// Namespaces definition for each wsdl file used for service port types.
#define LIGHTING_NS "http://www.soa4d.org/DPWS/Samples/Home/Lighting"
#define LAUNDRY_NS "http://www.soa4d.org/DPWS/Samples/Home/Laundry"
#define HOMECINEMA_NS "http://www.soa4d.org/DPWS/Samples/Home/HomeCinema"

// Qualified names used to specify the device types (could have been prefixed).
struct qname SimpleLightType = { SAMPLE_NS, "SimpleLight" };
struct qname DimmerLightType = { SAMPLE_NS, "DimmerLight" };
struct qname WashingMachineType = { SAMPLE_NS, "WashingMachine" };
struct qname ProjectorType = { SAMPLE_NS, "Projector" };

// Prefixed qualified names used to define the service port types (from WSDL).
// Using a prefix will ensure it will be used in messages.
struct prefixed_qname SwitchPowerPortType = { { LIGHTING_NS, "SwitchPower" }, "lit" };
struct prefixed_qname DimmingPortType = { { LIGHTING_NS, "Dimming" }, "lit" };
struct prefixed_qname WashPortType = { { LAUNDRY_NS, "Wash" }, "wsh" };
struct prefixed_qname DisplayPortType = { { HOMECINEMA_NS, "Display" }, "prj" };

// Structures containing the wsdl namespace and location for the hosted service.
struct wsdl_info LightingWsdl = { LIGHTING_NS, "http://www.soa4d.org/DPWS/Samples/Home/Lighting.wsdl" };
struct wsdl_info WashingMachineWsdl = { LAUNDRY_NS, "http://www.soa4d.org/DPWS/Samples/Home/WashingMachine.wsdl" };
struct wsdl_info ProjectorWsdl = { HOMECINEMA_NS, "http://www.soa4d.org/DPWS/Samples/Home/Projector.wsdl" };

/*
 *	Runtime data
 */

enum lit__PowerState simpleLight1 = lit__PowerState__OFF;// Simple light instance runtime data.
enum lit__PowerState simpleLight2 = lit__PowerState__OFF;// Simple light instance runtime data.

struct dimmerData {	// Device instance runtime data for dimmer
	enum lit__PowerState state; // light state
	short level;	//  instantaneous current dimming level
	short target;	// user set dimming level
	struct dpws dpws;	// runtime structure for event notification & client invocation
	short hEventSource; // for event notification
} dimmer;

struct washerData {	// Device instance runtime data for the washing machine.
	int cycle;	// Will receive the running cycle enumeration value. -1 means no cycle.
	int32_t scheduledEnd;	// time in seconds for the end of the cycle
	struct dpws dpws;	// runtime structure for event notification & client invocation
	short hEventSource;	// event source handle reference used for event notification
} washer;

static struct dpws master_dpws;	// The dpws structure used for message monitoring (main thread of the multi-thread server)


/*
 *  Utilities
 */

static char diversified_scope[MAX_SCOPE_LENGTH];	// computed scope buffer
char * scope_suffix = NULL;	// diversifier passed through the command line

/* Function adding the suffix to default scope (if suffix was defined).
 * Returns 0 if ok 1 if error (string too long).
 */
static int compute_diversified_scope(char *scope_suffix, char* scopeAddOn)
{
	int len = sizeof(SAMPLE_SCOPE) - 1;
	strcpy(diversified_scope, SAMPLE_SCOPE);
	if (scope_suffix) {
		if (len + strlen(scope_suffix) + 1 > MAX_SCOPE_LENGTH)
			return 1;
		if (scope_suffix[0] != '/') {
			strcat(diversified_scope, "/");
			len++;
		}
		strcat(diversified_scope, scope_suffix);
		len += strlen(scope_suffix);
	}
	if (scopeAddOn) {
		if (len + strlen(scopeAddOn) + 1 > MAX_SCOPE_LENGTH)
			return 1;
		if (scopeAddOn[0] != '/')
			strcat(diversified_scope, "/");
		strcat(diversified_scope, scopeAddOn);
	}
	return 0;
}

/* Function printing information related to a device. */
static void print_device_info(short hrefDevice)
{
	int len = 10, sz, i, j;
	short hrefs[10];
	char ** transportAddrs = NULL;

	// One service port (HTTP binding) created by default but potentially multiple
	// local addresses (multiple network interfaces & IP versions)
	transportAddrs = dpws_get_transport_addresses(dpws_get_default_service_port(hrefDevice), &sz);

	printf("\n+ Device [%s] started\n\nAddresses:\n", dpws_get_ptr_att(hrefDevice, DPWS_STR_DEVICE_ID));
	for (i = 0; i < sz; i++) {
		printf("[%d] %s\n", i, transportAddrs[i]);
		dpws_free(transportAddrs[i]);	// because of dpws_get_transport_addresses
	}
	dpws_free(transportAddrs);	// because of dpws_get_transport_addresses

	if (!dpws_get_service_handles(hrefDevice, hrefs, &len)) {
		printf("\n++ Hosted services:\n\n");
		for (i = 0; i < len; i++)
		{
			transportAddrs = dpws_get_transport_addresses(dpws_get_default_service_port(hrefs[i]), &sz);	// same as for device
			printf("Service ID: %s\nAddresses:\n", dpws_get_ptr_att(hrefs[i], DPWS_STR_SERVICE_ID));
			for (j = 0; j < sz; j++) {
				printf("[%d] %s\n", j, transportAddrs[j]);
				dpws_free(transportAddrs[j]);	// because of dpws_get_transport_addresses
			}
			dpws_free(transportAddrs);	// because of dpws_get_transport_addresses
		}
	}
}

/* Function displaying usage summary. */
static void simple_usage() {
	fprintf(stderr, "\nUsage: server [-s <scopeDiversifier>] [-p <httpPort>]\n");
	fprintf(stderr, "Use -?, -h or -help for more information\n\n");
	exit(1);
}

/* Function displaying complete program usage. */
static void usage() {
	fprintf(stderr, "\nUsage: server [-s <scopeDiversifier>] [-p <httpPort>]\n");
	fprintf(stderr, "\nOptions :\n");
	fprintf(stderr, "\n -s <scopeDiversifier>\n");
	fprintf(stderr, "\n\tThe <scopeDiversifier> is used to avoid conflicts on the local\n");
	fprintf(stderr, "\t\tnetwork by participating in building the scope of the device.\n");
	fprintf(stderr, "\n -p <httpPort>\n");
	fprintf(stderr, "\n\tThe <httpPort> is the http port used to listen to soap messages.\n");
	fprintf(stderr, "\nNotes : \n");
	fprintf(stderr, "\n\tThe server should be started before the client.\n\n");
	exit(1);
}

/*
 *  Main function performing server initialization.
 *	Usage: server [-s <scopeDiversifier>] [-p <httpPort>]
 */
int main(int argc, char **argv)
{
	/* Local variables */
	int port = 9876, c, bootSeq, status, i;
	short hDevModel, hServClass, hService, hServPort, hKitchenLight, hBedroomLight, hDimmer, hWasher, hProjector;
	struct localized_string ls = { NULL, NULL };

	/* Command line parsing */
	if (argc > 1 && (!strcmp(argv[1], "-?") || !strcmp(argv[1], "-h") || !strcmp(argv[1], "-help"))) {
		usage();
	}
	if ((argc < 1) || (argc > 5))
		simple_usage();

	for (i = 1; i < argc; i++) {
		if (!strcmp(argv[i], "-s")) {
			if (++i == argc)
				simple_usage();
			scope_suffix = argv[i];
		} else if (!strcmp(argv[i], "-p")) {
			if (++i == argc)
				simple_usage();
			port = atoi(argv[i]);
			if (port == 0)
				simple_usage();
		} else
			simple_usage();
	}

	/* Server program description (multi-device) */
	// 1. Initialize the DPWS stack.
	// 2. Configure server.
	// 	2.1 Set the boot sequence number & others toolkit parameters.
	//  2.2 Create simple light device model.
	//	2.3 Create simple light service class.
	//	2.4 Create kitchen light, configure hosted service & enable device.
	//	2.5 Create bedroom light, configure hosted service & enable device.
	//	2.6 Create dimmer light device model.
	//	2.7 Create dimmer service class.
	//	2.8 Create living room dimmer light, configure hosted service & enable device.
	//	2.9 Create washing machine device model.
	//	2.10 Create washer service class.
	//	2.11 Create washer device, configure hosted service & enable device.
	//	2.12 Create projector device model.
	//	2.13 Create projector service class.
	//	2.14 Create projector device, configure hosted service & enable device.
	// 3. Initialize server.
	// 4. Enable MTOM attachments processing
	// 5. Start server loop.
	// 6. Stop the server.

	printf("Configuring DPWS devices...\n\n");

	/* 1. Initialize the DPWS stack. */

	// This is the first function to be called when using the DPWS stack.
	// This version selects one single local IPv4 address.
	if ((status = dpws_init())) {
		fprintf(stderr, "Could not initialize the DPWSCore stack (err %d)\n\n", status);
		exit(1);
	}

	/* 2. Configure server. */

	// 	2.1 Set the boot sequence number & others toolkit parameters.

	// The boot sequence number is used is WS-Discovery messages and should be stored
	// and incremented each time the server is started.
	// Failing to do so may result in messages ignored by peers.
	bootSeq = readintfromfile(BOOT_SEQ_FILE);	// Reading the boot sequence number from a file
	writeinttofile(bootSeq + 1, BOOT_SEQ_FILE);	// Writing the next boot sequence number to the same file

	/*
	NOTE:
		All configuration APIs may return an error code. It has not been tested
		in this sample for clarity's sake and because the only errors that can
		occur are memory exhaustion (bad omen at initialization time) or due to
		invalid parameters (and we hope this sample has been correctly written :-)).
	*/
	DPWS_SET_INT_ATT(DC_REGISTRY_HANDLE, DPWS_INT_HTTP_PORT, port);	// Configure the HTTP listen port.
	DPWS_SET_INT_ATT(DC_REGISTRY_HANDLE, DPWS_INT_BOOT_SEQ, bootSeq);	// Configure the boot sequence number to use in this session (mandatory).

	printf("Initializing DPWS devices...\n");

	// 2.2 Create simple light device model
	printf("Creating device model...\n");
	hDevModel = dpws_create_device_model();	// Create simple light device model carrying most common device metadata.

	// Configure the mandatory device model attributes.
	ls.s = "BrightBulb";
	DPWS_SET_STR_ATT(hDevModel, DPWS_PTR_MODEL_NAME, &ls);
	ls.s = "Schneider Electric SA";
	DPWS_SET_STR_ATT(hDevModel, DPWS_PTR_MANUFACTURER, &ls);

	// Configure the optional device model attributes.
	DPWS_SET_STR_ATT(hDevModel, DPWS_STR_MODEL_NUMBER, "1.0");
	DPWS_SET_STR_ATT(hDevModel, DPWS_STR_MODEL_URL, "http://www.schneider-electric.com/BrightBulb.html");
	DPWS_SET_STR_ATT(hDevModel, DPWS_STR_PRESENTATION_URL, "index.html"); // Any relative URI will use the default presentation server
	DPWS_SET_STR_ATT(hDevModel, DPWS_STR_MANUFACTURER_URL, "http://www.schneider-electric.com");

	//	2.3 Create simple light service class.
	printf("Creating service class...\n");
	hServClass = dpws_create_service_class();

	// Configure the service class attributes.
	DPWS_ADD_PTR_ATT(hServClass, DPWS_PTR_PREFIXED_TYPE, &SwitchPowerPortType);	// The service implements the 'switch power" port type.
	DPWS_ADD_PTR_ATT(hServClass, DPWS_PTR_WSDL, &LightingWsdl);	// The WSDL information for the previous port type.
	DPWS_ADD_PTR_ATT(hServClass, DPWS_PTR_HANDLING_FUNCTION, &lit_serve_request);	// The generated dispatch function for the port type (from litStub.h).
	DPWS_SET_STR_ATT(hServClass, DPWS_STR_ID, "http://www.schneider-electric.com/DPWS/2006/03/Training/Light");	// A string ID that will used as default for mandatory service ID

	dpws_register_service_class(hDevModel, hServClass);	// Register the service class to the device model so that a hosted service will be created for every registered service class.
	dpws_release_handle(hServClass);	// Hold by model now

	// 2.4 Create kitchen light, configure hosted service & enable device.

	printf("Creating device...\n with type {%s}%s\n", SAMPLE_NS, "SimpleLight");
	// Create a simple light instance from the previously defined model
	hKitchenLight = dpws_create_device(0, hDevModel); // the id (0) must be unique in the current program

	// Configure the mandatory device attributes.
	DPWS_ADD_PTR_ATT(hKitchenLight, DPWS_PTR_TYPE, &SimpleLightType);
	/*
	NOTE:
		Should change every time a change on the device impacts DPWS metadata.
	*/
	DPWS_SET_INT_ATT(hKitchenLight, DPWS_INT_METADATA_VERSION, 1);
	ls.s = "Kitchen light";
	DPWS_SET_STR_ATT(hKitchenLight, DPWS_PTR_FRIENDLY_NAME, &ls);

	// Configure the optional device attributes.
	if (compute_diversified_scope(scope_suffix, KITCHEN_SUFFIX_SCOPE)) {	// Compute diversified scope for kitchen
		fprintf(stderr, "scope_suffix is too long, should not exceed %d characters\n", MAX_SCOPE_LENGTH - sizeof(SAMPLE_SCOPE));
		exit(1);
	}
	printf(" with scope %s\n", diversified_scope);
	DPWS_ADD_STR_ATT(hKitchenLight, DPWS_STR_SCOPE, diversified_scope);

	DPWS_SET_STR_ATT(hKitchenLight, DPWS_STR_FIRMWARE_VERSION, "1.0");
	DPWS_SET_STR_ATT(hKitchenLight, DPWS_STR_SERIAL_NUMBER, "56080001");
	DPWS_SET_PTR_ATT(hKitchenLight, DPWS_PTR_USER_DATA, &simpleLight1); // device instance data

	// Defines the physical address for the silently created hosted service.
	hService = dpws_get_service_by_class(hKitchenLight, hServClass);	// retrieve service created silently using dpws_create_device
	hServPort = dpws_get_default_service_port(hService);	// only one should be created for HTTP
	dpws_release_handle(hService);	// Not needed anymore
	/*
	IMPORTANT NOTE:
		Overriding the default value (a generated UUID) for the service address
		is important since transport addresses are in the the device metadata
		version scope so letting the random default would require an update of
		the metadata version at every server start like for the boot sequence.
	*/
	DPWS_SET_STR_ATT(hServPort, DPWS_STR_ADDRESS, "KitchenLight");	// sets the context path for the service physical address.
	dpws_release_handle(hServPort);	// Not needed anymore

	printf("<-- Enabling device...\n");
	if ((status = dpws_enable_device(hKitchenLight)))	// Enables device (especially schedule hello messages).
	{
		fprintf(stderr, "Could not execute dpws_enable_device (err %d)\n", status);
		dpws_shutdown();
		exit(1);
	}

	// 2.5 Create bedroom light, configure hosted service & enable device.

	printf("Creating device...\n with type {%s}%s\n", SAMPLE_NS, "SimpleLight");
	// Create a simple light instance from the previously defined model
	hBedroomLight = dpws_create_device(1, hDevModel); // the id (1) must be unique in the current program
	dpws_release_handle(hDevModel);	// Simple light model not used anymore.

	// Configure the mandatory device attributes.
	DPWS_ADD_PTR_ATT(hBedroomLight, DPWS_PTR_TYPE, &SimpleLightType);
	/*
	NOTE:
		Should change everytime a change on the device impacts DPWS metadata.
	*/
	DPWS_SET_INT_ATT(hBedroomLight, DPWS_INT_METADATA_VERSION, 1);
	ls.s = "Bedroom light";
	DPWS_SET_STR_ATT(hBedroomLight, DPWS_PTR_FRIENDLY_NAME, &ls);

	// Configure the optional device attributes.
	if (compute_diversified_scope(scope_suffix, BEDROOM_SUFFIX_SCOPE)) {	// Compute diversified scope for bedroom
		fprintf(stderr, "scope_suffix is too long, should not exceed %d characters\n", MAX_SCOPE_LENGTH - sizeof(SAMPLE_SCOPE));
		exit(1);
	}
	printf(" with scope %s\n", diversified_scope);
	DPWS_ADD_STR_ATT(hBedroomLight, DPWS_STR_SCOPE, diversified_scope);

	DPWS_SET_STR_ATT(hBedroomLight, DPWS_STR_FIRMWARE_VERSION, "1.0");
	DPWS_SET_STR_ATT(hBedroomLight, DPWS_STR_SERIAL_NUMBER, "05609540");
	DPWS_SET_PTR_ATT(hBedroomLight, DPWS_PTR_USER_DATA, &simpleLight2); // device instance data

	// Defines the physical address for the silently created hosted service.
	hService = dpws_get_service_by_class(hBedroomLight, hServClass);
	hServPort = dpws_get_default_service_port(hService);	// only one should be created for HTTP
	dpws_release_handle(hService);	// Not needed anymore
	/*
	IMPORTANT NOTE:
		Overriding the default value (a generated UUID) for the service address
		is important since transport addresses are in the the device metadata
		version scope so letting the random default would require an update of
		the metadata version at every server start like for the boot sequence.
	*/
	DPWS_SET_STR_ATT(hServPort, DPWS_STR_ADDRESS, "BedroomLight");	// sets the context path for the service physical address.
	dpws_release_handle(hServPort);	// Not needed anymore

	printf("<-- Enabling device...\n");
	if ((status = dpws_enable_device(hBedroomLight))) {	// Enables device (especially schedule hello messages).
		fprintf(stderr, "Could not execute dpws_enable_device (err %d)\n", status);
		dpws_shutdown();
		exit(1);
	}

	// 2.6 Create dimmer light device model
	printf("Creating device model...\n");
	hDevModel = dpws_create_device_model();	// Create dimmer light device model carrying most common device metadata.

	// Configure the mandatory device model attributes.
	ls.s = "SmoothDimmer";
	DPWS_SET_STR_ATT(hDevModel, DPWS_PTR_MODEL_NAME, &ls);
	ls.s = "Schneider Electric SA";
	DPWS_SET_STR_ATT(hDevModel, DPWS_PTR_MANUFACTURER, &ls);

	// Configure the optional device model attributes.
	DPWS_SET_STR_ATT(hDevModel, DPWS_STR_MODEL_NUMBER, "2.0");
	DPWS_SET_STR_ATT(hDevModel, DPWS_STR_MODEL_URL, "http://www.schneider-electric.com/SmoothDimmer.html");
	DPWS_SET_STR_ATT(hDevModel, DPWS_STR_PRESENTATION_URL, "index.html"); // Any relative URI will use the default presentation server
	DPWS_SET_STR_ATT(hDevModel, DPWS_STR_MANUFACTURER_URL, "http://www.schneider-electric.com");

	// 2.7 Create dimmer service class.
	printf("Creating service class...\n");
	hServClass = dpws_create_service_class();

	// Configure the service class attributes.
	DPWS_ADD_PTR_ATT(hServClass, DPWS_PTR_PREFIXED_TYPE, &SwitchPowerPortType);	// The service implements the 'switch power" port type.
	DPWS_ADD_PTR_ATT(hServClass, DPWS_PTR_PREFIXED_TYPE, &DimmingPortType);	// The service implements the 'dimming" port type.
	DPWS_ADD_PTR_ATT(hServClass, DPWS_PTR_WSDL, &LightingWsdl);	// The WSDL information for the previous port type.
	DPWS_ADD_PTR_ATT(hServClass, DPWS_PTR_HANDLING_FUNCTION, &lit_serve_request);	// The generated dispatch function for the port type (from litStub.h).
	DPWS_SET_STR_ATT(hServClass, DPWS_STR_ID, "http://www.schneider-electric.com/DPWS/2006/03/Training/Dimmer");	// A string ID that will used as default for mandatory service ID

	dpws_register_service_class(hDevModel, hServClass);	// Register the service class to the device model so that a hosted service will be created for every registered service class.
	dpws_release_handle(hServClass);	// Hold by model now

	// 2.8 Create living room dimmer light, configure hosted service & enable device.

	printf("Creating device...\n with type {%s}%s\n", SAMPLE_NS, "DimmerLight");
	// Create a dimmer light instance from the previously defined model
	hDimmer = dpws_create_device(2, hDevModel); // the id (2) must be unique in the current program
	dpws_release_handle(hDevModel);	// Dimmer light model not used anymore.

	// Configure the mandatory device attributes.
	DPWS_ADD_PTR_ATT(hDimmer, DPWS_PTR_TYPE, &DimmerLightType);
	/*
	NOTE:
		Should change everytime a change on the device impacts DPWS metadata.
	*/
	DPWS_SET_INT_ATT(hDimmer, DPWS_INT_METADATA_VERSION, 1);
	ls.s = "Living light";
	DPWS_SET_STR_ATT(hDimmer, DPWS_PTR_FRIENDLY_NAME, &ls);

	// Configure the optional device attributes.
	if (compute_diversified_scope(scope_suffix, LIVING_SUFFIX_SCOPE)) {	// Compute diversified scope for kitchen
		fprintf(stderr, "scope_suffix is too long, should not exceed %d characters\n", MAX_SCOPE_LENGTH - sizeof(SAMPLE_SCOPE));
		exit(1);
	}
	printf(" with scope %s\n", diversified_scope);
	DPWS_ADD_STR_ATT(hDimmer, DPWS_STR_SCOPE, diversified_scope);

	DPWS_SET_STR_ATT(hDimmer, DPWS_STR_FIRMWARE_VERSION, "1.0");
	DPWS_SET_STR_ATT(hDimmer, DPWS_STR_SERIAL_NUMBER, "8609701");

	// Device instance data that will be accessible in the service implementation
	dimmer.state = lit__PowerState__OFF;
	dimmer.level = 100;
	dimmer.target = 100;
	dpws_client_init(&dimmer.dpws, NULL);
	DPWS_SET_PTR_ATT(hDimmer, DPWS_PTR_USER_DATA, &dimmer); // device instance data

	// Defines the physical address for the silently created hosted service.
	hService = dpws_get_service_by_class(hDimmer, hServClass);	// retrieve service created silently using dpws_create_device
	dimmer.hEventSource = hService;	// store event source in the device "user data"
	hServPort = dpws_get_default_service_port(hService);	// only one should be created for HTTP
	dpws_release_handle(hService);	// Not needed anymore
	/*
	IMPORTANT NOTE:
		Overriding the default value (a generated UUID) for the service address
		is important since transport addresses are in the the device metadata
		version scope so letting the random default would require an update of
		the metadata version at every server start like for the boot sequence.
	*/
	DPWS_SET_STR_ATT(hServPort, DPWS_STR_ADDRESS, "LivingLight");	// sets the context path for the service physical address.
	dpws_release_handle(hServPort);	// Not needed anymore

	printf("<-- Enabling device...\n");
	if ((status = dpws_enable_device(hDimmer))) {	// Enables device (especially schedule hello messages).
		fprintf(stderr, "Could not execute dpws_enable_device (err %d)\n", status);
		dpws_shutdown();
		exit(1);
	}

	// 2.9 Create washing machine device model.

	printf("Creating device model...\n");
	hDevModel = dpws_create_device_model();	// Create washer device model carrying most common device metadata.

	// Configure the mandatory device model attributes.
	ls.s = "Salsa";
	DPWS_SET_STR_ATT(hDevModel, DPWS_PTR_MODEL_NAME, &ls);
	ls.s = "Whurlpool";
	DPWS_SET_STR_ATT(hDevModel, DPWS_PTR_MANUFACTURER, &ls);

	// Configure the optional device model attributes.
	DPWS_SET_STR_ATT(hDevModel, DPWS_STR_MODEL_NUMBER, "2.0");
	DPWS_SET_STR_ATT(hDevModel, DPWS_STR_MODEL_URL, "http://www.whurlpool.com/Salsa.html");
	DPWS_SET_STR_ATT(hDevModel, DPWS_STR_PRESENTATION_URL, "index.html"); // Any relative URI will use the default presentation server
	DPWS_SET_STR_ATT(hDevModel, DPWS_STR_MANUFACTURER_URL, "http://www.whurlpool.com");

	// 2.10 Create washer service class.
	printf("Creating service class...\n");
	hServClass = dpws_create_service_class();

	// Configure the service class attributes.
	DPWS_ADD_PTR_ATT(hServClass, DPWS_PTR_PREFIXED_TYPE, &WashPortType);	// The service implements the 'wash" port type.
	DPWS_ADD_PTR_ATT(hServClass, DPWS_PTR_WSDL, &WashingMachineWsdl);	// The WSDL information for the previous port type.
	DPWS_ADD_PTR_ATT(hServClass, DPWS_PTR_HANDLING_FUNCTION, &wsh_serve_request);	// The generated dispatch function for the port type (from wshStub.h).
	DPWS_SET_STR_ATT(hServClass, DPWS_STR_ID, "http://www.schneider-electric.com/DPWS/2006/03/Training/Washer");	// A string ID that will used as default for mandatory service ID

	dpws_register_service_class(hDevModel, hServClass);	// Register the service class to the device model so that a hosted service will be created for every registered service class.
	dpws_release_handle(hServClass);	// Hold by model now

	// 2.11 Create washer device, configure hosted service & enable device.

	printf("Creating device...\n with type {%s}%s\n", SAMPLE_NS, "WashingMachine");
	// Create a washer instance from the previously defined model
	hWasher = dpws_create_device(3, hDevModel); // the id (3) must be unique in the current program
	dpws_release_handle(hDevModel);	// Washing machine model not used anymore.

	// Configure the mandatory device attributes.
	DPWS_ADD_PTR_ATT(hWasher, DPWS_PTR_TYPE, &WashingMachineType);
	/*
	NOTE:
		Should change everytime a change on the device impacts DPWS metadata.
	*/
	DPWS_SET_INT_ATT(hWasher, DPWS_INT_METADATA_VERSION, 1);
	ls.s = "Washing machine";
	DPWS_SET_STR_ATT(hWasher, DPWS_PTR_FRIENDLY_NAME, &ls);

	// Configure the optional device attributes.
	if (compute_diversified_scope(scope_suffix, LAUNDRY_SUFFIX_SCOPE)) {	// Compute diversified scope for laundry
		fprintf(stderr, "scope_suffix is too long, should not exceed %d characters\n", MAX_SCOPE_LENGTH - sizeof(SAMPLE_SCOPE));
		exit(1);
	}
	printf(" with scope %s\n", diversified_scope);
	DPWS_ADD_STR_ATT(hWasher, DPWS_STR_SCOPE, diversified_scope);

	DPWS_SET_STR_ATT(hWasher, DPWS_STR_FIRMWARE_VERSION, "1.0");
	DPWS_SET_STR_ATT(hWasher, DPWS_STR_SERIAL_NUMBER, "85374012");

	// Device instance data that will be accessible in the service implementation
	washer.cycle = -1; // -1 means no cycle
	washer.scheduledEnd = 0;
	dpws_client_init(&washer.dpws, NULL);
	DPWS_SET_PTR_ATT(hWasher, DPWS_PTR_USER_DATA, &washer); // device instance data

	// Defines the physical address for the silently created hosted service.
	hService = dpws_get_service_by_class(hWasher, hServClass);	// retrieve service created silently using dpws_create_device
	washer.hEventSource = hService;	// store event source in the device "user data"
	hServPort = dpws_get_default_service_port(hService);	// only one should be created for HTTP
	dpws_release_handle(hService);	// Not needed anymore
	/*
	IMPORTANT NOTE:
		Overriding the default value (a generated UUID) for the service address
		is important since transport addresses are in the the device metadata
		version scope so letting the random default would require an update of
		the metadata version at every server start like for the boot sequence.
	*/
	DPWS_SET_STR_ATT(hServPort, DPWS_STR_ADDRESS, "WashingMachine");	// sets the context path for the service physical address.
	dpws_release_handle(hServPort);	// Not needed anymore

	printf("<-- Enabling washer device...\n");
	if ((status = dpws_enable_device(hWasher))) {	// Enables device (especially schedule hello messages).
		fprintf(stderr, "Could not execute dpws_enable_device (err %d)\n", status);
		dpws_shutdown();
		exit(1);
	}

	// 2.12 Create projector device model.

	printf("Creating device model...\n");
	hDevModel = dpws_create_device_model();	// Create projector device model carrying most common device metadata.

	// Configure the mandatory device model attributes.
	ls.s = "UltraVision";
	DPWS_SET_STR_ATT(hDevModel, DPWS_PTR_MODEL_NAME, &ls);
	ls.s = "Canyon";
	DPWS_SET_STR_ATT(hDevModel, DPWS_PTR_MANUFACTURER, &ls);

	// Configure the optional device model attributes.
	DPWS_SET_STR_ATT(hDevModel, DPWS_STR_MODEL_NUMBER, "1.1.0");
	DPWS_SET_STR_ATT(hDevModel, DPWS_STR_MODEL_URL, "http://www.canyon.com/UltraVision.html");
	DPWS_SET_STR_ATT(hDevModel, DPWS_STR_PRESENTATION_URL, "index.html"); // Any relative URI will use the default presentation server
	DPWS_SET_STR_ATT(hDevModel, DPWS_STR_MANUFACTURER_URL, "http://www.canyon.com");

	// 2.13 Create projector service class.

	printf("Creating service class...\n");
	hServClass = dpws_create_service_class();

	// Configure the service class attributes.
	DPWS_ADD_PTR_ATT(hServClass, DPWS_PTR_PREFIXED_TYPE, &DisplayPortType);	// The service implements the 'display" port type.
	DPWS_ADD_PTR_ATT(hServClass, DPWS_PTR_WSDL, &ProjectorWsdl);	// The WSDL information for the previous port type.
	DPWS_ADD_PTR_ATT(hServClass, DPWS_PTR_HANDLING_FUNCTION, &prj_serve_request);	// The generated dispatch function for the port type (from prjStub.h).
	DPWS_SET_STR_ATT(hServClass, DPWS_STR_ID, "http://www.schneider-electric.com/DPWS/2006/03/Training/Projector");	// A string ID that will used as default for mandatory service ID

	dpws_register_service_class(hDevModel, hServClass);	// Register the service class to the device model so that a hosted service will be created for every registered service class.

	// 2.14 Create projector device, configure hosted service & enable device.

	printf("Creating device...\n with type {%s}%s\n", SAMPLE_NS, "Projector");
	// Create a projector instance from the previously defined model.
	hProjector = dpws_create_device(4, hDevModel); // the id (4) must be unique in the current program
	dpws_release_handle(hDevModel);	// Projector model not used anymore.

	// Configure the mandatory device attributes.
	DPWS_ADD_PTR_ATT(hProjector, DPWS_PTR_TYPE, &ProjectorType);
	/*
	NOTE:
		Should change everytime a change on the device impacts DPWS metadata.
	*/
	DPWS_SET_INT_ATT(hProjector, DPWS_INT_METADATA_VERSION, 1);
	ls.s = "Home Cinema Projector";
	DPWS_SET_STR_ATT(hProjector, DPWS_PTR_FRIENDLY_NAME, &ls);

	// Configure the optional device attributes.
	if (compute_diversified_scope(scope_suffix, LIVING_SUFFIX_SCOPE)) {	// Compute diversified scope for living
		fprintf(stderr, "scope_suffix is too long, should not exceed %d characters\n", MAX_SCOPE_LENGTH - sizeof(SAMPLE_SCOPE));
		exit(1);
	}
	printf(" with scope %s\n", diversified_scope);
	DPWS_ADD_STR_ATT(hProjector, DPWS_STR_SCOPE, diversified_scope);

	DPWS_SET_STR_ATT(hProjector, DPWS_STR_FIRMWARE_VERSION, "2.1");
	DPWS_SET_STR_ATT(hProjector, DPWS_STR_SERIAL_NUMBER, "54272855997");

	// Defines the physical address for the silently created hosted service.
	hService = dpws_get_service_by_class(hProjector, hServClass);	// retrieve service created silently using dpws_create_device
	hServPort = dpws_get_default_service_port(hService);	// only one should be created for HTTP
	dpws_release_handle(hService);	// Not needed anymore
	/*
	IMPORTANT NOTE:
		Overriding the default value (a generated UUID) for the service address
		is important since transport addresses are in the the device metadata
		version scope so letting the random default would require an update of
		the metadata version at every server start like for the boot sequence.
	*/
	DPWS_SET_STR_ATT(hServPort, DPWS_STR_ADDRESS, "Projector");	// sets the context path for the service physical address.
	dpws_release_handle(hServPort);	// Not needed anymore

	printf("<-- Enabling projector device...\n");
	if ((status = dpws_enable_device(hProjector))) {	// Enables device (especially schedule hello messages).
		fprintf(stderr, "Could not execute dpws_enable_device (err %d)\n", status);
		dpws_shutdown();
		exit(1);
	}

	/* 3. Initialize server. */
	if ((status = dpws_server_init(&master_dpws, NULL)))	// initializes the dpws structure for server-side operations.
	{
		fprintf(stderr, "Could not initialize DPWS server (err %d)\n", status);
		dpws_shutdown();
		exit(1);
	}

	// Print devices and services information
	print_device_info(hBedroomLight);
	print_device_info(hKitchenLight);
	print_device_info(hDimmer);
	print_device_info(hWasher);
	print_device_info(hProjector);

	// Not needed anymore and hold by the registry since device activation
	dpws_release_handle(hBedroomLight);
	dpws_release_handle(hKitchenLight);
	dpws_release_handle(hDimmer);
	dpws_release_handle(hWasher);
	dpws_release_handle(hProjector);

	/* 4. Enable MTOM attachments processing. */
	soap_set_mode(&master_dpws.soap, SOAP_ENC_MTOM);

	/* 5. Start server loop. */
	// Creates a new thread to serve incoming messages. In the current sample,
	// a thread is created for each received message.
	if (bootServer(&master_dpws)) {
		fprintf(stderr, "Could not boot server...\n");
		dpws_shutdown();
		exit(1);
	}

	/* 6. Stop the server */

	printf("\n\nDPWS server ready. Type 'q' or 'Q' to stop it smoothly.\n\n");
	do {
		c = getchar();
	}
	while (c !='q' && c != 'Q');

	stopServer();	// 1. Calls the dpws_stop_server function that closes all
	// open listening sockets, schedules the 'Eventing end' notifications
	// and Bye WS-Discovery messages for all the active devices and blocks
	// until all server processing is finished and the main loop blocking
	// on dpws_accept() can exit.
	// 2. Uses platform-specific thread APIs to wait for the main server
	// loop thread dies.

	return 0;
}

/*
 *                   Service operations
 */

/* Change the light ON/OFF state */
int __lit__Switch(struct dpws* dpws, enum lit__PowerState lit__Power)
{
	enum lit__PowerState * pState;

	// The hosting device or service user data can be retrieved in the context
	// using a set of APIs among which dpws_get_device_user_data.
	pState = (enum lit__PowerState *)dpws_get_device_user_data(dpws);
	/*
	IMPORTANT NOTE about user data:
		The protection of user data is not performed by the DPWS stack so user
		should normally provide it. Since the 'light' sample is mono-thread, no
		concurrency issue can happen.
	*/
	*pState = lit__Power;
	printf("--> Setting light %s\n", lit__Power == lit__PowerState__ON ? "ON" : "OFF");
	return DPWS_OK;
}

/* Retrieves the light ON/OFF status */
int __lit__GetStatus(struct dpws* dpws, enum lit__PowerState *lit__Power)
{
	printf("--> Requesting light status\n");
	*lit__Power = *(enum lit__PowerState *)dpws_get_device_user_data(dpws);
	/*
	IMPORTANT NOTE about user data:
		The protection of user data is not performed by the DPWS stack so user
		should normally provide it. Since the 'light' sample is mono-thread, no
		concurrency issue can happen.
	*/
	printf("<-- Light status is %s\n", *lit__Power == lit__PowerState__ON ? "ON" : "OFF");
	return DPWS_OK;
}

/* Sets the level for dimmer light. */
int __lit__SetLevel(struct dpws* dpws, struct _lit__LightLevelTarget *lit__LightLevelTarget)
{
	short inc;
	struct dimmerData * pState;
	int status = DPWS_OK;

	// The hosting device or service user data can be retrieved in the context
	// using a set of APIs among which dpws_get_device_user_data.
	pState = (struct dimmerData *)dpws_get_device_user_data(dpws);

	/*
	IMPORTANT NOTE about user data:
		The protection of user data is not performed by the DPWS stack so user
		should normally provide it. This sample is especially concerned since
		multi-thread.
	*/

	/* Fault sending in case of incorrect parameters */
	if (lit__LightLevelTarget->__item < 0 || lit__LightLevelTarget->__item> 100)
		return dpws_fault(dpws, SENDER_FAULT_VALUE, "Level out of bounds.", NULL, NULL);

	pState->target = lit__LightLevelTarget->__item;
	if (lit__LightLevelTarget->TransitionDuration)
	{
		printf("--> Setting light level to %d%% in %d seconds\n", pState->target, *lit__LightLevelTarget->TransitionDuration);
		inc = (pState->target - pState->level)/(*lit__LightLevelTarget->TransitionDuration);

		/* Shifting implementation. */
		printf("Light level becomes ");
		if (inc> 0) {	/* Up shifting */
			for (; (pState->level + inc) <= pState->target; pState->level += inc) {
				portableSleep(1);
				printf(" %d%%", pState->level);
			}
		}
		else {	/* Down shifting */
			for (; (pState->level + inc) >= pState->target; pState->level += inc) {
				portableSleep(1);
				printf(" %d%%", pState->level);
			}
		}
	}
	else
		printf("--> Setting light level to %d%%\n", pState->target);

	pState->level = lit__LightLevelTarget->__item;
	printf("Light level reached %d%% .\n", pState->level);

	/* Target reached WS-Eventing notification */
	printf("<-- Notifying Light level target reached\n");
	if ((status = dpws_notify___lit__TargetReached(&pState->dpws, pState->hEventSource, pState->level)))
		fprintf(stderr, "Could not send 'target reached' event (err %d)\n", status);
	dpws_end(&pState->dpws);

	return status;
}

/* Retrieves the current dimmer target level. */
int __lit__GetLevelTarget(struct dpws* dpws, short *lit__LightLevel)
{
	struct dimmerData * pState = (struct dimmerData *)dpws_get_device_user_data(dpws);
	printf("--> Requesting light level target\n");
	/*
	IMPORTANT NOTE about user data:
		The protection of user data is not performed by the DPWS stack so user
		should normally provide it. Since the 'light' sample is mono-thread, no
		concurrency issue can happen.
	*/
	*lit__LightLevel = pState->target;
	printf("<-- Light level target is %d%%\n", pState->target);
	return DPWS_OK;
}

/* Retrieves the current dimmer instant level. */
int __lit__GetLevel(struct dpws* dpws, short *lit__LightLevel)
{
	struct dimmerData * pState = (struct dimmerData *)dpws_get_device_user_data(dpws);
	printf("--> Requesting light level\n");
	/*
	IMPORTANT NOTE about user data:
		The protection of user data is not performed by the DPWS stack so user
		should normally provide it. Since the 'light' sample is mono-thread, no
		concurrency issue can happen.
	*/
	*lit__LightLevel = pState->level;
	printf("<-- Light level is %d%%\n", pState->level);
	return DPWS_OK;
}

/* Starts a washing cycle. */
int __wsh__LaunchCycle(struct dpws* dpws, struct _wsh__LaunchCycle *wsh__LaunchCycle)
{
	struct washerData * pState;
	struct _wsh__CycleEnd cycleEnd;
	int32_t cycleDuration = 0;
	int nbEndpt = 1, status, nbProxies;
	short * device_proxies;

	// The hosting device or service user data can be retrieved in the context
	// using a set of APIs among which dpws_get_device_user_data.
	pState = (struct washerData *)dpws_get_device_user_data(dpws);

	/*
	IMPORTANT NOTE about user data:
		The protection of user data is not performed by the DPWS stack so user
		should normally provide it. This sample is especially concerned since
		multi-thread.
	*/

	printf("--> Requesting cycle launch\n");

	if (pState->cycle != -1) {
		printf("<-- Error : Cycle is already running!\n");
		return dpws_fault(dpws, RECEIVER_FAULT_VALUE, "Cycle running", NULL, NULL);	// triggers a fault
	}
	printf("Starting cycle...") ;
	switch (wsh__LaunchCycle->Name)	// cycle type switch
	{
		case wsh__Cycle__Gentle:
			cycleDuration = 10;
			printf("GENTLE (duration 10s)\n");
			break;
		case wsh__Cycle__Regular:
			cycleDuration = 20;
			printf("REGULAR (duration 20s)\n");
			break;
		case wsh__Cycle__Heavy:
			cycleDuration = 30;
			printf("HEAVY (duration 30s)\n");
			break;
	}
	pState->scheduledEnd = portableTime() + cycleDuration;	// set the time of the cycle end.
	pState->cycle = wsh__LaunchCycle->Name;	// keep trace of the type of cycle running

	printf("Starting cycle...\n");
	portableSleep(cycleDuration);	// sleep for the cycle duration
	pState->cycle = -1;	// cycle ended

	/* Cycle end WS-Eventing notification */
	printf("<-- Notifying end of cycle\n") ;
	cycleEnd.CycleName = wsh__LaunchCycle->Name;
	if ((status = dpws_notify___wsh__CycleEnded(&pState->dpws, pState->hEventSource, &cycleEnd)))	// send an event to subscribers using a specific dpws structure
		fprintf(stderr, "Could not send 'cycle end' event (err %d)\n", status);
	dpws_end(&pState->dpws);	// free memory allocated for event message

	/* Light on kitchen light */
	if (compute_diversified_scope(scope_suffix, KITCHEN_SUFFIX_SCOPE))
	{
		fprintf(stderr, "scope_suffix is too long, should not exceed %d characters\n", MAX_SCOPE_LENGTH - strlen(SAMPLE_SCOPE) -1);
		return DPWS_OK;
	}

	// Use WS-Discovery to find kitchen light device
	device_proxies = dpws_lookup(&pState->dpws, SimpleLightType.ns, SimpleLightType.lname, diversified_scope, &nbProxies);

	if (nbProxies == 1)
	{ // No more : one asked
		short * service_proxies;
		nbProxies = 1;
		service_proxies = dpws_get_services(&pState->dpws, device_proxies[0], SwitchPowerPortType.qname.ns, SwitchPowerPortType.qname.lname, &nbProxies);
		if (nbProxies > 0)
		{	// one expected
			printf("<-- Switching kitchen light ON\n") ;
			if ((status = dpws_send___lit__Switch(&pState->dpws, dpws_get_default_endpoint_reference(&pState->dpws, service_proxies[0]), lit__PowerState__ON)))
				fprintf(stderr, "Could not light on kitchen light (err %d)\n", status);
		}
	}
	dpws_end(&pState->dpws);	// free memory allocated for invocation

	return DPWS_OK;
}

/* Retrieves the current washing machine status. */
int __wsh__GetCycleStatus(struct dpws* dpws, struct _wsh__CycleStatus *wsh__CycleStatus)
{
	struct washerData * pState;

	// The hosting device or service user data can be retrieved in the context
	// using a set of APIs among which dpws_get_device_user_data.
	pState = (struct washerData *)dpws_get_device_user_data(dpws);
	/*
	IMPORTANT NOTE about user data:
		The protection of user data is not performed by the DPWS stack so user
		should normally provide it. This sample is especially concerned since
		multi-thread.
	*/

	printf("--> Requesting cycle status\n");
	if (pState->cycle == -1) {
		wsh__CycleStatus->TimeLeft = NULL;
		printf("<-- No cycle running\n");
	} else {
		/*
		NOTE about result allocation:
			Even if the top level structure is allocated by the skeleton,
			contents are under service implementor's responsibility.
		*/
		char * duration = dpws_malloc(dpws, DURATION_MAX_SIZE + 1);	// Constant defined in the DPWS API
		wsh__CycleStatus->TimeLeft = dpws_malloc(dpws, sizeof(struct struct_1));	// message transient allocation
		wsh__CycleStatus->TimeLeft->__item = dcxml_duration2xmlduration(duration, pState->scheduledEnd - portableTime());
		wsh__CycleStatus->TimeLeft->CycleName = pState->cycle;
		switch (wsh__CycleStatus->TimeLeft->CycleName) {
		case wsh__Cycle__Gentle:
			printf("<-- Cycle name GENTLE , time left is %s\n", wsh__CycleStatus->TimeLeft->__item);
			break;
		case wsh__Cycle__Regular:
			printf("<-- Cycle name REGULAR , time left is %s\n", wsh__CycleStatus->TimeLeft->__item);
			break;
		case wsh__Cycle__Heavy:
			printf("<-- Cycle name HEAVY , time left is %s\n", wsh__CycleStatus->TimeLeft->__item);
			break;
		}
	}
	return DPWS_OK;
}

/* Receives and saves received file (supposed to be an image). */
int __prj__DisplayImage(struct dpws* dpws, struct prj__AttachmentType *prj__Image)
{
	printf("--> Requesting image display\n");
	if (writefile("MyImage.jpg", prj__Image->Param.__ptr, prj__Image->Param.__size) < 0) {
		printf("Cannot save attachment\n");
		return SOAP_OK;
	}
	printf("Attachment received and saved under MyImage.jpg (size: %d, type: %s)\n", prj__Image->Param.__size, prj__Image->Param.type);
	return DPWS_OK;
}
