/*============================================================================*\
|                                                                              |
|                          SOA4D DPWSCore Samples                              |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|                                                                              |
|        + File info:                                                          |
|                     $Revision: 2219 $
|                     $Date: 2009-03-25 10:27:00 +0100 (mer, 25 mar 2009) $
\*============================================================================*/
#include <stdio.h>

/*
 * IMPORTANT NOTE: This multi-thread server is supplied as an example for studying purpose
 * and does not meet the usual standards of industrial software. For instance, device data
 * should be protected against concurrent access from server threads.
 */

#include "dc/dc_XMLConfiguration.h"	// "Local" core dynamic deployment feature. Includes basic stack API (dc/dc_Dpws.h)
#include "dc/dc_Sun.h"	// Streaming API for XML configuration file persistence
#include "dc/dc_Dyndepl.h"	// Holds dynamic deployment server features enabling remote operations.

#include "serverDPWS.h"	// OS-dependent code (especially server loop).

/*
 * NOTE about XML configuration:
 * The following service class implementations are called 'static' because they
 * use the dynamic deployment feature but are not dynamically deployed since
 * their code is provided with this sample. The 'hack' consists, instead of
 * having one service class loader for instance for DLLs, in having one service
 * class loader for each 'static' service class implementation hence 'SSCL'
 * (Static Service Class Loader).
 * One can consider this as a hack bringing a little more complexity but also
 * forces to consider service implementations as separate modules, which
 * improves programs design.
 */
#include "sscl_lit.h"	// Lighting service class implementation
#include "sscl_dim.h"	// Dimmer service class implementation
#include "sscl_wsh.h"	// Washer service class implementation
#include "sscl_prj.h"	// Projector service class implementation

/*
 *	Runtime data
 */

static struct dpws	master_dpws,	// The dpws structure used for message monitoring (main thread of the multi-thread server)
					conf_dpws;	// Used for file XML access.

// Qualified names identifying service class implementations.
// They are defined in their respective sscl_###.c/.h files.
extern struct qname	lit_iqn, dim_iqn, wsh_iqn, prj_iqn;

/*
 *  Utilities
 */

/* Function printing information related to a service class. */
static void print_service_class_info(short href_sclass)
{
	printf("\n+ Service Class\n\n  ID  : %s\n", (char *)dpws_get_ptr_att(href_sclass, DPWS_STR_ID));
}

/* Function printing information related to a device. */
static void print_device_info(short hrefDevice)
{
	int len = 10, sz, i, j;
	short hrefs[10];
	char ** transportAddrs = NULL;

	// One service port (HTTP binding) created by default but potentially multiple
	// local addresses (multiple network interfaces & IP versions)
	transportAddrs = dpws_get_transport_addresses(dpws_get_default_service_port(hrefDevice), &sz);

	printf("\n+ Device [%s] started\n\nAddresses:\n", dpws_get_ptr_att(hrefDevice, DPWS_STR_DEVICE_ID));
	for (i = 0; i < sz; i++) {
		printf("[%d] %s\n", i, transportAddrs[i]);
		dpws_free(transportAddrs[i]);	// because of dpws_get_transport_addresses
	}
	dpws_free(transportAddrs);	// because of dpws_get_transport_addresses

	if (!dpws_get_service_handles(hrefDevice, hrefs, &len)) {
		printf("\n++ Hosted services:\n\n");
		for (i = 0; i < len; i++)
		{
			transportAddrs = dpws_get_transport_addresses(dpws_get_default_service_port(hrefs[i]), &sz);	// same as for device
			printf("Service ID: %s\nAddresses:\n", dpws_get_ptr_att(hrefs[i], DPWS_STR_SERVICE_ID));
			for (j = 0; j < sz; j++) {
				printf("[%d] %s\n", j, transportAddrs[j]);
				dpws_free(transportAddrs[j]);	// because of dpws_get_transport_addresses
			}
			dpws_free(transportAddrs);	// because of dpws_get_transport_addresses
		}
	}
}

/* This callback will be called by the dynamic deployment feature every time
 * the DPWS stack configuration is modified so that is can saved to a file.
 */
void save_cbk()
{
	int status;
	void * sun_stream;

	/* Open an output SUN stream from a file. */
	sun_stream = sun_write_init(NULL, DPWS_CONFIG_DEFAULT_ID, NULL);
	if (!sun_stream) {
		fprintf(stderr, "Could not open configuration file for saving...\n");
		exit(1);
	}

	/* Save configuration to the stream. Note that there is no need to increment
	 * the boot sequence and that it will close automatically the stream.
	 */
	status = dpws_sun_save_config(&conf_dpws, sun_stream, DC_FALSE);
	if (status) {
		fprintf(stderr, "Could not save DPWS registry (err:%d)...\n", status);
		exit(1);
	}
}

/* Makes a file path from a directory path and file name. */
static char * build_path(char * dest, char * dir, char * file)
{
	int i;
	for (i = 0; dir[i]; i++)
		dest[i] = dir[i];
	if (dest[i - 1] != '/')
		dest[i++] = '/';
	dir = dest + i;
	for (i = 0; file[i]; i++)
		dir[i] = file[i];
	dir[i] = '\0';
	return dest;
}

/* Called when the listener exits. */
void shutdownProgram(struct dpws * dpws)
{
	dpws_config_shutdown();	// Cleans XML Configuration/dynamic deployment resources
	dpws_server_shutdown(dpws);	// Cleans server-side resources (registry). dpws is nullable (kept for backwards compatibility)
}

/* Function printing error message. */
static void print_error(char * msg, int status, struct dpws * dpws)
{
	fprintf(stderr, "\nError occured : %s.\n", msg);
	if (status)
		fprintf(stderr, "Error number is %d\n\n", status);
	else if (dpws)
		fprintf(stderr, "%s\n\n", dpws_get_error_msg(dpws));

	shutdownProgram(NULL);
	exit(1);
}

/* Function displaying complete program usage. */
static void usage()
{
	fprintf(stderr, "\nUsage: server <dataPath>\n\nParameters :\n\n <dataPath>");
	fprintf(stderr, " Folder where the dpwscore.xml configuration file should be read.\n");
	exit(1);
}

/*
 *  Main function performing server initialization.
 *	Usage: server <dataPath>
 */
int main (int argc, char **argv)
{
	/* Local variables */
	int status, sz, i, c;
	short handles[10];
	char data_path[64];
	void * sun_stream;

	/* Command line parsing */
	if (argc == 1)
		usage();

	/* Server program description (multi-device) */
	// 1. Initialize the DPWS stack.
	// 2. Initialize the XML configuration mechanism.
	// 3. Configure server.
	//	3.1 Register service class loaders
	//	3.2 Load configuration.
	// 4. Save configuration.
	// 5. Initialize server.
	// 6. Enable MTOM attachments processing
	// 7. Print configuration information.
	// 7. Start server loop.
	// 8. Stop the server

	/* 1. Initialize the DPWS stack. */

	// This is the first function to be called when using the DPWS stack.
	// This version selects one single local IPv4 address.
	if ((status = dpws_init()))
		print_error("Could not initialize the DPWSCore stack", status, NULL);

	/* 2. Initialize the XML configuration mechanism. */
	if ((status = dpws_config_init()))
		print_error("Could not initialize the XML configuration feature", status, NULL);

	/* 3. Configure server. */

	/*
	IMPORTANT NOTE:
		Even if the dynamic deployment feature uses only public APIs of the basic
		stack, it is not designed to be used along with 'manual' configuration
		since it builds extra configuration data.
	*/

	// 3.1 Register service class loaders
	dpws_register_dyndepl_loader();	// Registers the 'dynamic deployment server' service class
	dpws_register_loader(&lit_iqn, lit_load_cbk);	// Registers the 'static' service class loader for simple lights
	dpws_register_loader(&dim_iqn, dim_load_cbk);	// Registers the 'static' service class loader for dimmer lights
	dpws_register_loader(&wsh_iqn, wsh_load_cbk);	// Registers the 'static' service class loader for washer
	dpws_register_loader(&prj_iqn, prj_load_cbk);	// Registers the 'static' service class loader for projector

	// 3.2 Load configuration.
	printf("Initializing DPWS devices...\n");
	sun_stream  = sun_read_init(NULL, build_path(data_path, argv[1], DPWS_CONFIG_DEFAULT_ID));	// open SUN input stream on a file
	if (!sun_stream)
		print_error("Could not open configuration file", 0, NULL);
	status = dpws_sun_load_config(&conf_dpws, sun_stream);	// Loads stack configuration from the SUN stream.
	if (status)
		print_error("Could not load DPWS configuration", status, NULL);

	/* 4. Save configuration. */

	// Save immediately configuration for incremented boot sequence persistence.
	// NOTE that the backup is done in the working directory.
	sun_stream  = sun_write_init(NULL, DPWS_CONFIG_DEFAULT_ID, NULL);
	if (!sun_stream)
		print_error("Could not open configuration file for saving", 0, NULL);
	status = dpws_sun_save_config(&conf_dpws, sun_stream, DC_TRUE);
	if (status)
		print_error("Could not save DPWS registry", status, NULL);

	dpws_register_config_cbk(save_cbk);	// allow to trigger an action when the configuration is changed by the dynamic deployment service.

	/* 5. Initialize server. */
	// Calls the dpws_server_init function.
	// This function initializes the dpws structure and performs server-side initialization.
	if ((status = dpws_server_init(&master_dpws, NULL)))	// initializes the dpws structure for server-side operations.
		print_error("Could not initialize DPWS server", status, NULL);

	/* 6. Enable MTOM attachments processing */
	soap_set_mode(&master_dpws.soap, SOAP_ENC_MTOM);

	/* 7. Print configuration information. */

	/* Print information on configured service classes. */
	sz = 10;	// Get service class handles (at most 10).
	status = dpws_get_service_class_handles(handles, &sz);
	if (status) {
		if (status == DPWS_ERR_MORE_RESULTS)
			fprintf(stderr, "Only 10 service classes listed...\n");
		else
			print_error("Could not print DPWS service classes", status, &master_dpws);
	}
	for (i = 0; i < sz; i++)
		print_service_class_info(handles[i]);

	dpws_release_handles(handles, sz);	// Release handles.


	/* Print information on configured devices. */
	sz = 10;	// Get device handles (at most 10).
	status = dpws_get_device_handles(handles, &sz);
	if (status) {
		if (status == DPWS_ERR_MORE_RESULTS)
			fprintf(stderr, "Only 10 device listed...\n");
		else
			print_error("Could not print DPWS devices", status, &master_dpws);
	}
	for (i = 0; i < sz; i++)
		print_device_info(handles[i]);

	dpws_release_handles(handles, sz);	// Release handles.

	/* 8. Start server loop */
	// Creates a new thread to serve incoming messages. In the current sample,
	// a thread is created for each received message.
	if (bootServer(&master_dpws))
		print_error("Could not boot server", 0, &master_dpws);

	/* 6. Stop the server */

	printf("\n\nDPWS server ready. Type 'q' or 'Q' to stop it smoothly.\n\n");
	do {
		c = getchar();
	}
	while (c !='q' && c != 'Q');

	stopServer();	// 1. Calls the dpws_stop_server function that closes all
	// open listening sockets, schedules the 'Eventing end' notifications
	// and Bye WS-Discovery messages for all the active devices and blocks
	// until all server processing is finished and the main loop blocking
	// on dpws_accept() can exit.
	// 2. Uses platform-specific thread APIs to wait for the main server
	// loop thread dies.

	return 0;
}
