/*============================================================================*\
|                                                                              |
|                          SOA4D Abstraction Layer                             |
|                                                                              |
|               ->>  Copyright 2008 Schneider Electric SA <<-                  |
|                                                                              |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 1.4 $
|                     $Date: 2008/02/11 07:57:02 $
\*============================================================================*/
/*******************************************************************************
*                        Win32 socket common features                          *
*******************************************************************************/
#include "al_ip.h"

/*----------------------------------------------------------------------------*\
 *                           IP network error management                      *
\*----------------------------------------------------------------------------*/
typedef struct
{
	int win_err;
	int al_err;
} err_trans_t;

static const err_trans_t err_trans_table[] =
{
	{WSA_NOT_ENOUGH_MEMORY, AL_ENOMEM},
	{WSA_INVALID_PARAMETER, AL_EINVAL},
	{WSA_OPERATION_ABORTED, AL_EINTR},
	{WSA_IO_INCOMPLETE, 	AL_EIO},
	{WSA_IO_PENDING, 		AL_EIO},
	{WSAEINTR, 				AL_EINTR},
	{WSAEBADF, 				AL_EBADF},
	{WSAEACCES, 			AL_EACCES},
	{WSAEFAULT, 			AL_EFAULT},
	{WSAEINVAL, 			AL_EINVAL},
	{WSAEMFILE, 			AL_EMFILE},
	{WSAEWOULDBLOCK, 		AL_EAGAIN},
	{WSAEINPROGRESS,		AL_EINPROGRESS},
	{WSAEALREADY, 			AL_EALREADY},
	{WSAENOTSOCK, 			AL_ENOTSOCK},
	{WSAEDESTADDRREQ, 		AL_EDESTADDRREQ},
	{WSAEMSGSIZE, 			AL_EMSGSIZE},
	{WSAEPROTOTYPE, 		AL_EPROTOTYPE},
	{WSAENOPROTOOPT,		AL_ENOPROTOOPT},
	{WSAEPROTONOSUPPORT, 	AL_EPROTONOSUPPORT},
	{WSAESOCKTNOSUPPORT, 	AL_ESOCKTNOSUPPORT},
	{WSAEOPNOTSUPP, 		AL_EOPNOTSUPP},
	{WSAEPFNOSUPPORT, 		AL_EPFNOSUPPORT},
	{WSAEAFNOSUPPORT, 		AL_EAFNOSUPPORT},
	{WSAEADDRINUSE, 		AL_EADDRINUSE},
	{WSAEADDRNOTAVAIL, 		AL_EADDRNOTAVAIL},
	{WSAENETDOWN, 			AL_ENETDOWN},
	{WSAENETUNREACH, 		AL_ENETUNREACH},
	{WSAENETRESET, 			AL_ENETRESET},
	{WSAECONNABORTED, 		AL_ECONNABORTED},
	{WSAECONNRESET, 		AL_ECONNRESET},
	{WSAENOBUFS, 			AL_ENOBUFS},
	{WSAEISCONN, 			AL_EISCONN},
	{WSAENOTCONN, 			AL_ENOTCONN},
	{WSAESHUTDOWN, 			AL_ESHUTDOWN},
	{WSAETOOMANYREFS, 		AL_ETOOMANYREFS},
	{WSAETIMEDOUT, 			AL_ETIMEDOUT},
	{WSAECONNREFUSED, 		AL_ECONNREFUSED},
	{WSAELOOP, 				AL_ELOOP},
	{WSAENAMETOOLONG, 		AL_ENAMETOOLONG},
	{WSAEHOSTDOWN, 			AL_EHOSTDOWN},
	{WSAEHOSTUNREACH,		AL_EHOSTUNREACH},
	{WSAENOTEMPTY, 			AL_ENOTEMPTY},
	{WSASYSNOTREADY, 		AL_ENXIO},
	{WSANOTINITIALISED,		AL_ENXIO},
	{WSAEDISCON, 			AL_ECONNRESET},
	{WSAECANCELLED, 		AL_EINTR},
	{WSAEINVALIDPROVIDER,  	AL_EHOSTDOWN},
	{WSAEPROVIDERFAILEDINIT,AL_EHOSTDOWN},
	{WSASERVICE_NOT_FOUND, 	AL_EHOSTDOWN},
	{WSA_E_CANCELLED, 		AL_EINTR},
	{WSAHOST_NOT_FOUND, 	AL_EHOSTDOWN}
};

static int native2al_error(int win_err)
{
	int i, max = sizeof(err_trans_table)/sizeof(err_trans_t);

	// Search for the Windows error in the translation table
	for(i = 0; err_trans_table[i].win_err != win_err && i < max; i++);

	// If no error found, return  AL_ERROR
    return (i == max) ? AL_ERROR : err_trans_table[i].al_err;
}

int al_ip_error (void)
{
#ifdef AL_STACK_WINSOCK_CE
	return native2al_error(GetLastError());
#else
	return native2al_error(WSAGetLastError());
#endif
}

/*----------------------------------------------------------------------------*\
 *                               IP stack init                                *
\*----------------------------------------------------------------------------*/
int al_ip_init()
{
	WSADATA w;
	return WSAStartup(MAKEWORD(1, 1), &w) ? AL_ERROR : AL_SUCCESS;
}

int al_ip_shutdown() { return WSACleanup() ? AL_ERROR : AL_SUCCESS;}

/*----------------------------------------------------------------------------*\
 *                  Sockets creation and management functions                 *
\*----------------------------------------------------------------------------*/

int al_socket(int domain, int type, int protocol)
{
	SOCKET s;
	return (s = socket(domain, type, protocol)) == INVALID_SOCKET ? al_ip_error() : (int)s;
}

int al_shutdown (int s, int how)
{
	return shutdown((SOCKET)s, how) ? AL_ERROR : AL_SUCCESS;
}

int al_sock_close(int s)
{
	return closesocket((SOCKET)s) ? AL_ERROR : AL_SUCCESS;
}

int al_setsockopt(int s, int level, int optname, const void *optval, int optlen)
{
	return setsockopt((SOCKET)s, level, optname, (const char *)optval, optlen) ? AL_ERROR : AL_SUCCESS;
}

int al_getsockopt(int s, int level, int optname, void *optval, int *optlen)
{
	return getsockopt((SOCKET)s, level, optname, (char *)optval, optlen) ? AL_ERROR : AL_SUCCESS;
}

int al_getsockname(int s, al_sockaddr_t *addr, int *addrlen)
{
#ifdef AL_SOCKADDR_FULL_OPAQUE
	return (int)getsockname((SOCKET)s, (struct sockaddr*)addr, addrlen);
#else
	int ret;
	struct sockaddr_storage saddr;
	int saddr_len = sizeof(struct sockaddr_storage);
	ret = getsockname((SOCKET)s, (struct sockaddr*) &saddr, &saddr_len); // INVALID_SOCKET = AL_ERROR = -1
	if (ret >= 0) {
		if (convert2aladdr(addr, &saddr, addrlen))
			ret = AL_ERROR;
	}
	return ret;
#endif
}

int al_ioctl(int s, int cmd, int * arg)
{
	if (cmd == AL_FIONBIO)
		return ioctlsocket((SOCKET)s, FIONBIO, (u_long *)arg) ? AL_ERROR : AL_SUCCESS;
	else
		return AL_EOPTION;
}

#ifndef AL_SOCKADDR_FULL_OPAQUE
static struct sockaddr * convert2naddr(struct sockaddr_storage * dest, const al_sockaddr_t * src)	// TODO: check if cast is possible
{
	switch (((const al_sockaddr_in_t *)src)->sin_family)
	{
	case AF_INET:
		((struct sockaddr_in*)dest)->sin_family = AF_INET; // Note assumption
		((struct sockaddr_in*)dest)->sin_addr.s_addr = ((al_sockaddr_in_t*)src)->sin_addr.addr;
		((struct sockaddr_in*)dest)->sin_port = ((al_sockaddr_in_t*)src)->sin_port;
		break;

#ifdef AL_HAVE_IPV6
	case AF_INET6:
		((struct sockaddr_in6*)dest)->sin6_family = AF_INET6;
		((struct sockaddr_in6*)dest)->sin6_port = ((al_sockaddr_in6_t*)src)->sin6_port;
		((struct sockaddr_in6*)dest)->sin6_flowinfo = ((al_sockaddr_in6_t*)src)->sin6_flowinfo;
		((struct sockaddr_in6*)dest)->sin6_scope_id = ((al_sockaddr_in6_t*)src)->sin6_scope_id;
		memcpy(&((struct sockaddr_in6 *)dest)->sin6_addr, &((al_sockaddr_in6_t*)src)->sin6_addr, sizeof(al_in6_addr_t));
		break;
#endif

	default:
		return NULL;
	}
	return (struct sockaddr *)dest;
}

static int convert2aladdr(al_sockaddr_t * dest, const struct sockaddr_storage * src, int *destlen)	// TODO: check if cast is possible
{
	switch (((struct sockaddr_in *)src)->sin_family)
	{
	case AF_INET:
    if (*destlen < sizeof(al_sockaddr_in_t))
      return AL_ERROR;
    *destlen = sizeof(al_sockaddr_in_t);
		((al_sockaddr_in_t *)dest)->sin_family = AF_INET; // Note assumption
		((al_sockaddr_in_t *)dest)->sin_addr.addr = ((struct sockaddr_in *)src)->sin_addr.s_addr;
		((al_sockaddr_in_t *)dest)->sin_port = ((struct sockaddr_in *)src)->sin_port;
		break;
#ifdef AL_HAVE_IPV6
	case AF_INET6:
    if (*destlen < sizeof(al_sockaddr_in6_t))
      return AL_ERROR;
    *destlen = sizeof(al_sockaddr_in6_t);
		((al_sockaddr_in6_t *)dest)->sin6_family = AF_INET;
		((al_sockaddr_in6_t *)dest)->sin6_port = ((struct sockaddr_in6 *)src)->sin6_port;
		((al_sockaddr_in6_t *)dest)->sin6_flowinfo = ((struct sockaddr_in6 *)src)->sin6_flowinfo;
		((al_sockaddr_in6_t *)dest)->sin6_scope_id = ((struct sockaddr_in6 *)src)->sin6_scope_id;
		memcpy(&((al_sockaddr_in6_t *)dest)->sin6_addr, &((struct sockaddr_in6 *)src)->sin6_addr, sizeof(struct in6_addr));
		break;
#endif
	default:
		return AL_ERROR;
	}
	return 0;
}
#endif /* AL_SOCKADDR_FULL_OPAQUE */

int al_accept(int s, al_sockaddr_t *addr, int *addrlen)
{
#ifdef AL_SOCKADDR_FULL_OPAQUE
  return (int)accept((SOCKET)s, (struct sockaddr*)addr, addrlen);
#else
	int ret;
	struct sockaddr_storage saddr;
	int saddr_len = sizeof(struct sockaddr_storage);
	ret = (int)accept((SOCKET)s, (struct sockaddr*)&saddr, &saddr_len);	// INVALID_SOCKET = AL_ERROR = -1
  if (ret >= 0)
  {
    if (convert2aladdr(addr, &saddr, addrlen))
      ret = AL_ERROR;
  }
	return ret;
#endif
}

int al_bind(int s, const al_sockaddr_t *addr, int addrlen)
{
#ifdef AL_SOCKADDR_FULL_OPAQUE
  return bind((SOCKET)s, (struct sockaddr*)addr, addrlen);
#else
  int ret;
	struct sockaddr_storage addr_trans;
  ret = bind((SOCKET)s, convert2naddr(&addr_trans, addr), sizeof(struct sockaddr_storage));
	return ret ? AL_ERROR : AL_SUCCESS;
#endif
}

int al_connect(int s, const al_sockaddr_t *addr, int  addrlen)
{
  int ret;
#ifdef AL_SOCKADDR_FULL_OPAQUE
	ret = connect((SOCKET)s, (struct sockaddr*)addr, addrlen);
#else
	struct sockaddr_storage addr_trans;
	convert2naddr(&addr_trans, addr);
	ret = connect((SOCKET)s, (struct sockaddr*)&addr_trans, sizeof(struct sockaddr_storage));
#endif
	return ret ? AL_ERROR : AL_SUCCESS;
}

int al_listen(int s, int backlog)
{
	return listen((SOCKET)s, backlog) ? AL_ERROR : AL_SUCCESS;
}

// al_fd_set defined in implementation
int al_select(
		int nfds, al_fd_set *readfds, al_fd_set *writefds,
		al_fd_set *exceptfds, al_timeval_t *timeout
		)
{
	return select(nfds, readfds, writefds, exceptfds, (struct timeval *)timeout); // timeval cast is possible and SOCKET_ERROR is -1
}

/*----------------------------------------------------------------------------*\
 *                Address management and translation functions                *
\*----------------------------------------------------------------------------*/
#ifndef AL_F0LLOW_RFC3493 // WILL BE OBSOLETED
int al_inet_ntoa(uint32_t addr, char *buffer)
{
	struct in_addr inaddr;
	char * s;
	inaddr.s_addr = addr;
	s = inet_ntoa(inaddr);	// TODO: not thread-safe
	if (s)
	{
		strcpy(buffer, s);
		return AL_SUCCESS;
	}
	else
		return AL_ERROR;
}

int al_inet_aton(const char *str, uint32_t *addr)
{
	*addr = inet_addr(str);
	if (*addr == INADDR_NONE)
		return AL_EINVAL;
	else
		return AL_SUCCESS;
}
#endif

int al_inet_pton(int af, const char *host, void* addr)
{
  int rc;
  if (af == AF_INET6)
  {
    int lg = sizeof(struct sockaddr_in6);
    struct sockaddr_in6   local;
    rc = WSAStringToAddressA((char*)host, af, 0, (struct sockaddr*)&local, &lg);
	if (rc)
		return 0;
    memcpy(addr, &local.sin6_addr, sizeof(local.sin6_addr));
  }
  else if (af == AF_INET)
  {
    int in_addr = inet_addr(host);
	if (in_addr == -1)
		return 0;
    memcpy(addr, &in_addr, sizeof(in_addr));
  }
  else
	  return AL_EAFNOSUPPORT;

  return 1;
}

const char* al_inet_ntop(int af, const void* src, char *dst, int len)
{
	int rc, lg;
	if (af == AF_INET6) {
		struct sockaddr_in6 local;
	    lg = sizeof(struct sockaddr_in6);
		local.sin6_family = AF_INET6;
		local.sin6_port = 0;
		local.sin6_flowinfo = 0;
		local.sin6_scope_id = 0;
		memcpy(&local.sin6_addr, src, sizeof(local.sin6_addr));
		memset(dst, 0, len);
		rc = WSAAddressToStringA((struct sockaddr *)&local, lg, NULL, dst, &len);
		if (rc)
			return NULL;
	}
	else if (af == AF_INET) {
		struct sockaddr_in local;
	    lg = sizeof(struct sockaddr_in);
		local.sin_family = AF_INET;
		local.sin_port = 0;
		memcpy(&local.sin_addr, src, sizeof(local.sin_addr));
		memset(dst, 0, len);
		rc = WSAAddressToStringA((struct sockaddr *)&local, lg, NULL, dst, &len);
		if (rc)
			return NULL;
	}
	else
	{
		// TODO: al_errno = AL_AFNOSUPPORT;
		return NULL;
	}
	return dst;
}


/*----------------------------------------------------------------------------*\
 *                               DNS functions                                *
\*----------------------------------------------------------------------------*/

int al_gethostbyname(const char *name, al_in_addr_t * inaddr)
{
	struct hostent * he;
	// TODO: not thread-safe
	he = gethostbyname(name);
	if (he)
	{
		if (he->h_addrtype != AF_INET || he->h_length != sizeof(al_in_addr_t))
			return AL_EAFNOSUPPORT;
		he->h_addr_list[0];
		memcpy(inaddr, he->h_addr_list[0], he->h_length);
		return AL_SUCCESS;
	}
	else
	{
		switch (h_errno)
		{
		case WSANOTINITIALISED: case WSAENETDOWN:
			return AL_ENETDOWN;
		case WSAHOST_NOT_FOUND: case WSAEFAULT:
			return AL_EHOSTNOTFOUND;
		case WSATRY_AGAIN: case WSAEINPROGRESS:
			return AL_EAGAIN;
		case WSANO_RECOVERY:
			return AL_EINTERNAL;
		case WSANO_DATA:
			return AL_ENODATA;
		case WSAEINTR:
			return AL_EINTR;
		default:
			return AL_ERROR;	// TODO: should be a translated code.
		}
	}
}


/*----------------------------------------------------------------------------*\
 *                           Socket read functions                            *
\*----------------------------------------------------------------------------*/

int al_read (int d, void *buf, size_t len)
{
	return _read(d, buf, (unsigned int)len);
}

int al_recv (int s, void *buf, int len, unsigned int flags)
{
	return recv((SOCKET)s, (char *)buf, len, (int)flags);	// INVALID_SOCKET = AL_ERROR = -1
}

int al_recvfrom	(int			s,
				 void			*buf,
				 int 	len,
				 unsigned int 	flags,
				 al_sockaddr_t	*from,
				 int 	*fromlen)
{
#ifdef AL_SOCKADDR_FULL_OPAQUE
	return recvfrom((SOCKET)s, (char *)buf, len, (int)flags, (struct sockaddr*)from, fromlen);
#else
	int ret;
	struct sockaddr_storage saddr;
	int saddr_len = sizeof(struct sockaddr_storage);
	ret = recvfrom((SOCKET) s, (char *) buf, len, (int) flags,
			(struct sockaddr*) &saddr, &saddr_len);
	if (ret > 0 && from && convert2aladdr(from, &saddr, fromlen))
		ret = AL_ERROR;
	return ret;
#endif
}

#ifdef AL_HAVE_RECVMSG	// MinGW has direct access to the WSARecvMsg API but no CMSG_ macros

#include <mswsock.h>

#define MAX(a,b) (a < b ? b : a)

int al_recvon	(int			s,
				 void *			buf,
				 int 			len,
				 unsigned int 	flags,
				 al_sockaddr_t*	from,
				 int *			fromlen,
				 uint32_t * 	itf)
{
	if (itf)
	{
		int ret;
		GUID WSARecvMsg_GUID = WSAID_WSARECVMSG;
		LPFN_WSARECVMSG WSARecvMsg;
		DWORD nb;
		WSAMSG mh;
		char cbuf[WSA_CMSG_SPACE(MAX(sizeof(struct in_pktinfo), sizeof(struct in6_pktinfo)))];
		WSABUF iov = {len, buf};

		if (WSAIoctl((SOCKET)s, SIO_GET_EXTENSION_FUNCTION_POINTER, &WSARecvMsg_GUID,
			sizeof(WSARecvMsg_GUID),&WSARecvMsg, sizeof(WSARecvMsg), &nb, NULL, NULL) == SOCKET_ERROR)
			return AL_ERROR;

		if (from)
		{
#ifdef AL_SOCKADDR_FULL_OPAQUE
			mh.name = (LPSOCKADDR)from;
			mh.namelen = *fromlen;
#else
			struct sockaddr_storage saddr;
			mh.name = (struct sockaddr*)&saddr;
			mh.namelen = sizeof(struct sockaddr_storage);
#endif
		}
		else
		{
			mh.name = NULL;
			mh.namelen = 0;
		}

		mh.lpBuffers = &iov;
		mh.dwBufferCount = 1;
		mh.Control.buf = cbuf;
		mh.Control.len = sizeof(cbuf);
		mh.dwFlags = flags;

		ret = WSARecvMsg((SOCKET)s, &mh, &nb, NULL, NULL);

		/* Receive auxiliary data in msgh */
		if (ret == SOCKET_ERROR)
			return AL_ERROR;
		else
		{
			WSACMSGHDR *cmptr;

			for (cmptr = WSA_CMSG_FIRSTHDR(&mh); cmptr; cmptr = WSA_CMSG_NXTHDR(&mh, cmptr))
			{
				if (cmptr->cmsg_level == IPPROTO_IP && cmptr->cmsg_type == IP_PKTINFO)
				{
					*itf = ((struct in_pktinfo *) WSA_CMSG_DATA(cmptr))->ipi_ifindex;
					break;
				}
				if (cmptr->cmsg_level == IPPROTO_IPV6 && cmptr->cmsg_type == IPV6_PKTINFO)
				{
					*itf = (uint32_t)((struct in6_pktinfo *) WSA_CMSG_DATA(cmptr))->ipi6_ifindex;
					break;
				}
			}
			if (cmptr == NULL)
				ret = AL_ERROR;
		}

#ifndef AL_SOCKADDR_FULL_OPAQUE
		if (ret > 0 && from && convert2aladdr(from, &saddr, fromlen))
			return AL_ERROR;
#endif
		return nb;
	}
	else
		return al_recvfrom(s, buf, len, flags, from, fromlen);
}
#endif

/*----------------------------------------------------------------------------*\
 *                           Socket write functions                           *
\*----------------------------------------------------------------------------*/

int al_write (int d, const void *buf, size_t len)
{
	return _write(d, buf, (unsigned int)len);
}

int al_send (int s, const void *buf, int len, unsigned int flags)
{
	return send((SOCKET)s, (char *)buf, len, (int)flags);	// INVALID_SOCKET = AL_ERROR = -1
}

int al_sendto (int					s,
			  const void 			*buf,
			  int 					len,
			  unsigned int 			flags,
			  const al_sockaddr_t	*to,
			  int 					tolen)
{
#ifdef AL_SOCKADDR_FULL_OPAQUE
  return sendto((SOCKET)s, (char *)buf, len, (int)flags, (const struct sockaddr*)to, tolen);	// INVALID_SOCKET = AL_ERROR = -1
#else
	struct sockaddr_storage n_to;
  convert2naddr(&n_to, to);
	return sendto((SOCKET)s, (char *)buf, len, (int)flags, (const struct sockaddr*)&n_to , sizeof(struct sockaddr_storage));	// INVALID_SOCKET = AL_ERROR = -1
#endif
}
