/*============================================================================*\
 |                                                                              |
 |                          SOA4D Abstraction Layer                             |
 |                                                                              |
 |               ->>  Copyright 2008 Schneider Electric SA <<-                  |
 |                                                                              |
 |                                                                              |
 |       + File info:                                                           |
 |                     $Revision: 1.5 $
 |                     $Date: 2008/02/11 07:57:02 $
 \*============================================================================*/
/*******************************************************************************
 *                      Socket Address management functions                     *
 *******************************************************************************/
#include "al_ip.h"
#include "al_mem.h"

#include <stdlib.h>
#include <stdio.h>
#include <memory.h>

int al_sockaddr_checkFamily(const al_sockaddr_t* sockAddr)
{
	struct sockaddr_in* addr = (struct sockaddr_in *) sockAddr;
	if (addr->sin_family == PF_INET)
		return 0;

#ifdef AL_HAVE_IPV6
	if (addr->sin_family == PF_INET6)
	return 1;
#endif
	return AL_ESOCKTNOSUPPORT;
}

int al_sockaddr_init(int domain, al_sockaddr_t* sockAddrStorage, int sockAddrLen)
{
	int sockaddr_len;

#ifdef AL_HAVE_IPV6
	if ((domain == ALPF_INET_IPDFT) || (domain == PF_INET6) || (domain == AF_INET6))
	{
		struct sockaddr_in6* sockaddr_in6 = (struct sockaddr_in6 *) sockAddrStorage;
		sockaddr_len = sizeof(struct sockaddr_in6);
		if (sockAddrLen < sockaddr_len)
		return AL_EINVAL;

		sockaddr_in6->sin6_family = AF_INET6;
		sockaddr_in6->sin6_port = 0;
		sockaddr_in6->sin6_flowinfo = 0;
		sockaddr_in6->sin6_scope_id = 0;
		sockaddr_in6->sin6_addr = in6addr_any;

		return sockaddr_len;
	}
#endif // AL_HAVE_IPV6
#ifdef AL_HAVE_IPV4
	if ((domain == ALPF_INET_IPDFT) || (domain == PF_INET) || (domain == AF_INET))
	{
		struct sockaddr_in* sockaddr_in = (struct sockaddr_in *) sockAddrStorage;
		sockaddr_len = sizeof(struct sockaddr_in);
		if (sockAddrLen < sockaddr_len)
		return AL_EINVAL;

		sockaddr_in->sin_family = PF_INET; //AF_INET;
		sockaddr_in->sin_port = 0;
		sockaddr_in->sin_addr.s_addr = htonl(INADDR_ANY);
		return sockaddr_len;
	}
#endif
	return AL_ESOCKTNOSUPPORT;
}

int al_sockaddr_getDomain(const al_sockaddr_t* sockAddr)
{
	struct sockaddr_in* addr = (struct sockaddr_in *) sockAddr;

	if (addr->sin_family == PF_INET)
		return PF_INET;
#ifdef AL_HAVE_IPV6
	else if (addr->sin_family == PF_INET6)
	return PF_INET6;
#endif

	return AL_ESOCKTNOSUPPORT;
}

int al_sockaddr_getLength(const al_sockaddr_t* sockAddr)
{
	struct sockaddr_in* addr = (struct sockaddr_in *) sockAddr;

	if (addr->sin_family == PF_INET)
		return sizeof(struct sockaddr_in);
#ifdef AL_HAVE_IPV6
	else if (addr->sin_family == PF_INET6)
	return sizeof(struct sockaddr_in6);
#endif

	return AL_ESOCKTNOSUPPORT;
}

int al_sockaddr_getPort(const al_sockaddr_t* sockAddrStorage, uint16_t* port)
{
	int rc;

	rc = al_sockaddr_checkFamily(sockAddrStorage);
	if (rc < 0)
		return rc;
	if (rc)
	{
#ifdef AL_HAVE_IPV6
		*port = ntohs(((struct sockaddr_in6 *)sockAddrStorage)->sin6_port);
#else
		return AL_ESOCKTNOSUPPORT;
#endif
	}
	else
	{
#ifdef AL_HAVE_IPV4
		*port = ntohs(((struct sockaddr_in *) sockAddrStorage)->sin_port);
#else
		return AL_ESOCKTNOSUPPORT;
#endif
	}
	return AL_SUCCESS;
}

int al_sockaddr_getScope(const al_sockaddr_t* sockAddrStorage, uint32_t* scope)
{
	int rc;

	rc = al_sockaddr_checkFamily(sockAddrStorage);
	if (rc < 0)
		return rc;
	if (rc)
	{
#ifdef AL_HAVE_IPV6
		*scope = ((struct sockaddr_in6 *)sockAddrStorage)->sin6_scope_id;
#else
		return AL_ESOCKTNOSUPPORT;
#endif
	}
	return AL_SUCCESS;
}

int al_sockaddr_getNetAddr(const al_sockaddr_t* sockAddrStorage,
		al_ipNetAddr_t* ipNetAddr)
{
	int rc;
	rc = al_sockaddr_checkFamily(sockAddrStorage);
	if (rc < 0)
		return rc;

	if (rc)
	{
#ifdef AL_HAVE_IPV6
		struct sockaddr_in6 * sockaddr_in6 = (struct sockaddr_in6 *) sockAddrStorage;
		ipNetAddr->netAddrLen = 16;
		memcpy(&ipNetAddr->addr.netAddr6, &sockaddr_in6->sin6_addr,sizeof(struct in6_addr));
#else
		return AL_ESOCKTNOSUPPORT;
#endif
	}
	else
	{
#ifdef AL_HAVE_IPV4
		struct sockaddr_in * sockaddr_in = (struct sockaddr_in *) sockAddrStorage;
#ifdef AL_HAVE_IPV6
		ipNetAddr->netAddrLen = 4;
		memcpy(&ipNetAddr->addr.netAddr4, &sockaddr_in->sin_addr, sizeof(struct in_addr));
#else
		memcpy(ipNetAddr, &sockaddr_in->sin_addr, sizeof(struct in_addr));
#endif
#else
		return AL_ESOCKTNOSUPPORT;
#endif
	}
	return AL_SUCCESS;
}

int al_sockaddr_getNetAddrToAddrStr(const al_sockaddr_t* sockAddrStorage,
		char *addrStr, int len, int scope)
{
	const char* result;
	int rc;

	rc = al_sockaddr_checkFamily(sockAddrStorage);
	if (rc < 0)
		return rc;

	if (rc > 0)
	{
#ifdef AL_HAVE_IPV6
		const struct sockaddr_in6* sockaddr_in6 = (const struct sockaddr_in6*) sockAddrStorage;
		result = al_inet_ntop(sockaddr_in6->sin6_family, (const char*)&sockaddr_in6->sin6_addr, addrStr, len);
		if (scope && result && IN6_IS_ADDR_LINKLOCAL(&sockaddr_in6->sin6_addr))
		{
			for (; *addrStr; addrStr++);
			sprintf(addrStr, "%%%lu", sockaddr_in6->sin6_scope_id);
		}
#else
		return AL_ESOCKTNOSUPPORT;
#endif
	}
	else
	{
#ifdef AL_HAVE_IPV4
		const struct sockaddr_in* sockaddr_in = (const struct sockaddr_in*) sockAddrStorage;
		result = al_inet_ntop(sockaddr_in->sin_family, (const char*)&sockaddr_in->sin_addr, addrStr, len);
#else
		return AL_ESOCKTNOSUPPORT;
#endif
	}

	if (result == NULL)
		return AL_ERROR;

	return AL_SUCCESS;
}

int al_sockaddr_setPort(al_sockaddr_t* sockAddrStorage, uint16_t port)
{
	int rc;

	rc = al_sockaddr_checkFamily(sockAddrStorage);
	if (rc < 0)
		return rc;

	if (rc)
	{
#ifdef AL_HAVE_IPV6
		((struct sockaddr_in6 *)sockAddrStorage)->sin6_port = htons(port);
#else
		return AL_ESOCKTNOSUPPORT;
#endif
	}
	else
	{
#ifdef AL_HAVE_IPV4
		((struct sockaddr_in *)sockAddrStorage)->sin_port = htons(port);
#else
		return AL_ESOCKTNOSUPPORT;
#endif
	}
	return AL_SUCCESS;
}

int al_sockaddr_setScope(al_sockaddr_t* sockAddrStorage, uint32_t scope)
{
	int rc;

	rc = al_sockaddr_checkFamily(sockAddrStorage);
	if (rc < 0)
		return rc;

	if (rc)
	{
#ifdef AL_HAVE_IPV6
		((struct sockaddr_in6 *) sockAddrStorage)->sin6_scope_id = scope;
#else
		return AL_ESOCKTNOSUPPORT;
#endif
	}
	// else do nothing
	return AL_SUCCESS;
}

int al_sockaddr_setNetAddrFromAddrStr(al_sockaddr_t* sockAddrStorage,
		const char *addrStr)
{
	int result;
	int rc;

	rc = al_sockaddr_checkFamily(sockAddrStorage);
	if (rc < 0)
		return rc;

	if (rc > 0)
	{
#ifdef AL_HAVE_IPV6
		struct sockaddr_in6* sockaddr_in6 = (struct sockaddr_in6*) sockAddrStorage;
		int i;
		char tmpbuf[AL_INET6_ADDRSTRLEN];
		// copy until potential %
		for (i = 0; i < AL_INET6_ADDRSTRLEN && addrStr[i]; i++)
		{
			if (addrStr[i] == '%')
			{
				sockaddr_in6->sin6_scope_id = atoi((const char *)(addrStr + i + 1));
				break;
			}
			tmpbuf[i] = addrStr[i];
		}
		tmpbuf[i] = 0;

		result = al_inet_pton(sockaddr_in6->sin6_family, tmpbuf, (char*)&sockaddr_in6->sin6_addr);
#else
		return AL_ESOCKTNOSUPPORT;
#endif
	}
	else
	{
#ifdef AL_HAVE_IPV4
		struct sockaddr_in* sockaddr_in = (struct sockaddr_in*) sockAddrStorage;
		result = al_inet_pton(sockaddr_in->sin_family, addrStr, (char*)&sockaddr_in->sin_addr);
#else
		return AL_ESOCKTNOSUPPORT;
#endif
	}
	return result <= 0 ? AL_ERROR : AL_SUCCESS;
}

int al_sockaddr_setInAddrLoopBack(al_sockaddr_t* sockAddrStorage)
{
	int rc;

	rc = al_sockaddr_checkFamily(sockAddrStorage);
	if (rc < 0)
		return rc;

	if (rc)
	{
#ifdef AL_HAVE_IPV6
		struct sockaddr_in6 * sockaddr_in6 = (struct sockaddr_in6 *) sockAddrStorage;
		sockaddr_in6->sin6_addr = in6addr_loopback;
#else
		return AL_ESOCKTNOSUPPORT;
#endif
	}
	else
	{
#ifdef AL_HAVE_IPV4
		struct sockaddr_in * sockaddr_in = (struct sockaddr_in *) sockAddrStorage;
		sockaddr_in->sin_addr.s_addr = htonl(INADDR_LOOPBACK);
#else
		return AL_ESOCKTNOSUPPORT;
#endif
	}
	return AL_SUCCESS;
}

int al_sockaddr_isLoopback(al_ipNetAddr_t* ipNetAddr)
{
#ifdef AL_HAVE_IPV6
	if (ipNetAddr->netAddrLen == 4)
		return ipNetAddr->addr.netAddr4.addr == htonl(INADDR_LOOPBACK);
	else if (ipNetAddr->netAddrLen == 16)
	{
		int i;
		for (i = 0; i < 16; i++)
		{
			if (ipNetAddr->addr.netAddr6.sAddr[i] != in6addr_loopback.s6_addr[i])
			return 0;
		}
	}
	return 1;
#else
	return *ipNetAddr == htonl(INADDR_LOOPBACK);
#endif
}

#ifdef AL_HAVE_IPV6
int al_getaddrinfo(int af, const char *host, uint16_t port, al_sockaddr_t ** addrs, int * nbAddrs)
{
	al_sockaddr_t sa;
	int ret;

	ret = al_sockaddr_init(af, &sa, sizeof(al_sockaddr_t));
	if (ret < 0)
		return ret;

	if (al_sockaddr_setNetAddrFromAddrStr(&sa, host))	// This filters multicast addresses since getaddrinfo that return a strange SOCK_RAW on Linux
	{	// DNS resolution
		int i = 0;
		struct addrinfo * resList, hints, *wResult;

		memset(&hints, 0, sizeof(struct addrinfo));
		hints.ai_family = af;
		hints.ai_socktype = SOCK_DGRAM;	// To retrieve only one which is correct since we don't consider the 'service' param
		hints.ai_protocol = af;
		if (getaddrinfo(host, NULL, &hints, &resList))
			return AL_ERROR;

		/* Count results */
		*nbAddrs = 0;
		for (wResult = resList; wResult; wResult = wResult->ai_next)
			(*nbAddrs)++;

		*addrs = AL_MALLOC(*nbAddrs * sizeof(al_sockaddr_t));

		for (wResult = resList; wResult; wResult = wResult->ai_next)
		{
#ifdef AL_SOCKADDR_FULL_OPAQUE
			memcpy((*addrs)+i, wResult->ai_addr, wResult->ai_addrlen);
#else
			int socklen;
			socklen = al_sockaddr_init(af, addrs[i], sizeof(al_sockaddr_t));
			if (socklen < 0) {
				AL_FREE(*addrs);
				return AL_ERROR;
			}
			convert2aladdr(addrs[i], resList->ai_addr, &socklen);
#endif
			al_sockaddr_setPort((*addrs)+i, port);	// Could be done using the "service" param is the string form was available
			i++;
		}
		freeaddrinfo(resList);
	}
	else	// numeric address
	{
		*addrs = AL_MALLOC(sizeof(al_sockaddr_t));
		*nbAddrs = 1;
		memcpy(*addrs, &sa, sizeof(al_sockaddr_t));
		al_sockaddr_setPort(*addrs, port);
	}
	return AL_SUCCESS;
}
#else
int al_getaddrinfo(int af, const char *host, uint16_t port, al_sockaddr_t ** addrs, int * nbAddrs)
{
	int ret;

	*addrs = AL_MALLOC(sizeof(al_sockaddr_t));
	*nbAddrs = 1;

	ret = al_sockaddr_init(af, *addrs, sizeof(al_sockaddr_t));
	if (ret < 0)
		return ret;

	if (al_sockaddr_setNetAddrFromAddrStr(*addrs, host))
	{
		al_in_addr_t in_addr;
		ret = al_gethostbyname(host, &in_addr);
		if (ret < 0) {
			AL_FREE(*addrs);
			return ret;
		}
	}
	al_sockaddr_setPort(*addrs, port);
	return AL_SUCCESS;
}
#endif

int al_freeaddrinfo(al_sockaddr_t * addrs)
{
	AL_FREE(addrs);
	return AL_SUCCESS;
}

