/*============================================================================*\
|                                                                              |
|                          SOA4D Abstraction Layer                             |
|                                                                              |
|               ->>  Copyright 2008 Schneider Electric SA <<-                  |
|                                                                              |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 1.5 $
|                     $Date: 2008/02/11 07:57:02 $
\*============================================================================*/
/*******************************************************************************
*                            Standard BSD IP features                          *
*******************************************************************************/
#include "al_ip.h"
#include <netdb.h>
#include <arpa/inet.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>

#include <sys/socket.h>
#include <netinet/in.h>
/*----------------------------------------------------------------------------*\
 *                           IP network error management                      *
\*----------------------------------------------------------------------------*/

typedef struct
{
	int bsd_err;
	int al_err;
} err_trans_t;

static const err_trans_t err_trans_table[] =
{
	{ENOMEM, 			AL_ENOMEM},
	{EINVAL, 			AL_EINVAL},
	{EINTR, 			AL_EINTR},
	{EIO, 				AL_EIO},
	{EBADF, 			AL_EBADF},
	{EACCES, 			AL_EACCES},
	{EFAULT, 			AL_EFAULT},
	{EMFILE, 			AL_EMFILE},
	{EAGAIN, 			AL_EAGAIN},
	{EINPROGRESS,		AL_EINPROGRESS},
	{EALREADY, 			AL_EALREADY},
	{ENOTSOCK, 			AL_ENOTSOCK},
	{EDESTADDRREQ, 		AL_EDESTADDRREQ},
	{EMSGSIZE, 			AL_EMSGSIZE},
	{EPROTOTYPE, 		AL_EPROTOTYPE},
	{ENOPROTOOPT,		AL_ENOPROTOOPT},
	{EPROTONOSUPPORT, 	AL_EPROTONOSUPPORT},
	{ESOCKTNOSUPPORT, 	AL_ESOCKTNOSUPPORT},
	{EOPNOTSUPP, 		AL_EOPNOTSUPP},
	{EPFNOSUPPORT, 		AL_EPFNOSUPPORT},
	{EAFNOSUPPORT, 		AL_EAFNOSUPPORT},
	{EADDRINUSE, 		AL_EADDRINUSE},
	{EADDRNOTAVAIL, 	AL_EADDRNOTAVAIL},
	{ENETDOWN, 			AL_ENETDOWN},
	{ENETUNREACH, 		AL_ENETUNREACH},
	{ENETRESET, 		AL_ENETRESET},
	{ECONNABORTED, 		AL_ECONNABORTED},
	{ECONNRESET, 		AL_ECONNRESET},
	{ENOBUFS, 			AL_ENOBUFS},
	{EISCONN, 			AL_EISCONN},
	{ENOTCONN, 			AL_ENOTCONN},
	{ESHUTDOWN, 		AL_ESHUTDOWN},
	{ETOOMANYREFS, 		AL_ETOOMANYREFS},
	{ETIMEDOUT, 		AL_ETIMEDOUT},
	{ECONNREFUSED, 		AL_ECONNREFUSED},
	{ELOOP, 			AL_ELOOP},
	{ENAMETOOLONG, 		AL_ENAMETOOLONG},
	{EHOSTDOWN, 		AL_EHOSTDOWN},
	{EHOSTUNREACH,		AL_EHOSTUNREACH},
	{ENOTEMPTY, 		AL_ENOTEMPTY},
	{ENXIO, 			AL_ENXIO},
	{ENOSPC, 			AL_ENOSPC}
};

static int native2al_error(int bsd_err)
{
	int i, max = sizeof(err_trans_table)/sizeof(err_trans_t);

	// Search for the Windows error in the translation table
	for(i = 0; err_trans_table[i].bsd_err != bsd_err && i < max; i++);

	// If no error found, return  AL_ERROR
    return (i == max) ? AL_ERROR : err_trans_table[i].al_err;
}

int al_ip_error (void) { return native2al_error(errno);}

/*----------------------------------------------------------------------------*\
 *                               IP stack init                                *
\*----------------------------------------------------------------------------*/
int al_ip_init() { return AL_SUCCESS;}

int al_ip_shutdown() { return AL_SUCCESS;}

/*----------------------------------------------------------------------------*\
 *                  Sockets creation and management functions                 *
\*----------------------------------------------------------------------------*/

int al_socket(int domain, int type, int protocol)
{
	int s;
	s = socket(domain, type, protocol);
	if (s < 0)
		return al_ip_error();
#ifdef AL_HAVE_IPV6
#ifdef IPPROTO_IPV6
	if (domain == AF_INET6)	// By default under Linux IPV6 sockets receive also IPv4. Disable for uniform behavior.
	{
		int on = 1;
		if (setsockopt(s, IPPROTO_IPV6, IPV6_V6ONLY, &on, sizeof(on)) == -1)
		{
			close(s);
			return al_ip_error();
		}
	}
#endif
#endif
	return s;
}

int al_shutdown(int s, int how) { return shutdown(s, how);}

int al_sock_close(int s) { return close(s);}

int al_setsockopt(int s, int level, int optname, const void *optval, int optlen)
{
#ifdef AL_HAVE_IPV6
	if (level == IPPROTO_IPV6 && optname == IPV6_PKTINFO)	// Linux kernel 2.6.24 requires to set IPV6_RECVPKTINFO to receive IPV6_PKTINFO...
		optname = IPV6_RECVPKTINFO;
#endif
	return setsockopt(s, level, optname, optval, optlen);
}

int al_getsockopt(int s, int level, int optname, void *optval, int *optlen)
{
	return getsockopt(s, level, optname, optval, optlen);
}

int al_getsockname(int s, al_sockaddr_t *addr, int *addrlen)
{
#ifdef AL_SOCKADDR_FULL_OPAQUE
	return getsockname(s, (struct sockaddr*)addr, addrlen);
#else
	int ret;
	struct sockaddr_storage saddr;
	int saddr_len = sizeof(struct sockaddr_storage);
	ret = getsockname((SOCKET)s, (struct sockaddr*) &saddr, &saddr_len); // INVALID_SOCKET = AL_ERROR = -1
	if (ret >= 0) {
		if (convert2aladdr(addr, &saddr, addrlen))
			ret = AL_ERROR;
	}
	return ret;
#endif
}

int al_ioctl(int s, int cmd, int * arg)
{
	if (cmd == AL_FIONBIO)
	{
		if (*arg)
			return fcntl(s, F_SETFL, fcntl(s, F_GETFL)|O_NONBLOCK);
		else
			return fcntl(s, F_SETFL, fcntl(s, F_GETFL)&~O_NONBLOCK);
	}
	else
		return AL_EOPTION;
}

#ifndef AL_SOCKADDR_FULL_OPAQUE
static struct sockaddr * convert2naddr(struct sockaddr_storage * dest, const al_sockaddr_t * src)	// TODO: check if cast is possible
{
	switch (((const al_sockaddr_in_t *)src)->sin_family)
	{
	case AF_INET:
		((struct sockaddr_in*)dest)->sin_family = AF_INET; // Note assumption
		((struct sockaddr_in*)dest)->sin_addr.s_addr = ((al_sockaddr_in_t*)src)->sin_addr.addr;
		((struct sockaddr_in*)dest)->sin_port = ((al_sockaddr_in_t*)src)->sin_port;
		break;

#ifdef AL_HAVE_IPV6
	case AF_INET6:
		((struct sockaddr_in6*)dest)->sin6_family = AF_INET6;
		((struct sockaddr_in6*)dest)->sin6_port = ((al_sockaddr_in6_t*)src)->sin6_port;
		((struct sockaddr_in6*)dest)->sin6_flowinfo = ((al_sockaddr_in6_t*)src)->sin6_flowinfo;
		((struct sockaddr_in6*)dest)->sin6_scope_id = ((al_sockaddr_in6_t*)src)->sin6_scope_id;
		memcpy(&((struct sockaddr_in6 *)dest)->sin6_addr, &((al_sockaddr_in6_t*)src)->sin6_addr, sizeof(al_in6_addr_t));
		break;
#endif

	default:
		return NULL;
	}
	return (struct sockaddr *)dest;
}

static int convert2aladdr(al_sockaddr_t * dest, const struct sockaddr_storage * src, int *destlen)	// TODO: check if cast is possible
{
	switch (((struct sockaddr_in *)src)->sin_family)
	{
	case AF_INET:
    if (*destlen < sizeof(al_sockaddr_in_t))
      return AL_ERROR;
    *destlen = sizeof(al_sockaddr_in_t);
		((al_sockaddr_in_t *)dest)->sin_family = AF_INET; // Note assumption
		((al_sockaddr_in_t *)dest)->sin_addr.addr = ((struct sockaddr_in *)src)->sin_addr.s_addr;
		((al_sockaddr_in_t *)dest)->sin_port = ((struct sockaddr_in *)src)->sin_port;
		break;
#ifdef AL_HAVE_IPV6
	case AF_INET6:
    if (*destlen < sizeof(al_sockaddr_in6_t))
      return AL_ERROR;
    *destlen = sizeof(al_sockaddr_in6_t);
		((al_sockaddr_in6_t *)dest)->sin6_family = AF_INET;
		((al_sockaddr_in6_t *)dest)->sin6_port = ((struct sockaddr_in6 *)src)->sin6_port;
		((al_sockaddr_in6_t *)dest)->sin6_flowinfo = ((struct sockaddr_in6 *)src)->sin6_flowinfo;
		((al_sockaddr_in6_t *)dest)->sin6_scope_id = ((struct sockaddr_in6 *)src)->sin6_scope_id;
		memcpy(&((al_sockaddr_in6_t *)dest)->sin6_addr, &((struct sockaddr_in6 *)src)->sin6_addr, sizeof(struct in6_addr));
		break;
#endif
	default:
		return AL_ERROR;
	}
	return 0;
}
#endif /* AL_SOCKADDR_FULL_OPAQUE */

int al_accept(int s, al_sockaddr_t *addr, int *addrlen)
{
#ifdef AL_SOCKADDR_FULL_OPAQUE
	return accept(s, (struct sockaddr*)addr, addrlen);
#else
	int ret;
	struct sockaddr_storage saddr;
	int saddr_len = sizeof(struct sockaddr_storage);
	ret = accept(s, (struct sockaddr*)&saddr, &saddr_len);	// INVALID_SOCKET = AL_ERROR = -1
	if (ret >= 0)
	{
		if (convert2aladdr(addr, &saddr, addrlen))
			ret = AL_ERROR;
	}
	return ret;
#endif
}

int al_bind(int s, const al_sockaddr_t *addr, int addrlen)
{
#ifdef AL_SOCKADDR_FULL_OPAQUE
	return bind(s, (struct sockaddr*)addr, addrlen);
#else
	struct sockaddr_storage addr_trans;
	return bind(s, convert2naddr(&addr_trans, addr), sizeof(struct sockaddr_storage));
#endif
}

int al_connect(int s, const al_sockaddr_t *addr, int  addrlen)
{
#ifdef AL_SOCKADDR_FULL_OPAQUE
	return connect(s, (struct sockaddr*)addr, addrlen);
#else
	struct sockaddr_storage addr_trans;
	convert2naddr(&addr_trans, addr);
	return connect(s, (struct sockaddr*)&addr_trans, sizeof(struct sockaddr_storage));
#endif
}

int al_listen(int s, int backlog) {	return listen(s, backlog);}

// al_fd_set defined in implementation
int al_select(
		int nfds, al_fd_set *readfds, al_fd_set *writefds,
		al_fd_set *exceptfds, al_timeval_t *timeout
		)
{
	struct timeval tv, *ptv = NULL;
	if (timeout) {
		tv.tv_sec = timeout->sec;
		tv.tv_usec = timeout->usec;
		ptv = &tv;
	}
	return select(nfds, readfds, writefds, exceptfds, ptv);
}

/*----------------------------------------------------------------------------*\
 *                Address management and translation functions                *
\*----------------------------------------------------------------------------*/
#ifndef AL_F0LLOW_RFC3493 // WILL BE OBSOLETED
int al_inet_ntoa(uint32_t addr, char *buffer)
{
	struct in_addr inaddr;
	char * s;
	inaddr.s_addr = addr;
	s = inet_ntoa(inaddr);	// TODO: not thread-safe
	if (s)
	{
		strcpy(buffer, s);
		return AL_SUCCESS;
	}
	else
		return AL_ERROR;
}

int al_inet_aton(const char *str, uint32_t *addr)
{
	*addr = inet_addr(str);
	if (*addr == INADDR_NONE)
		return AL_EINVAL;
	else
		return AL_SUCCESS;
}
#endif

int al_inet_pton(int af, const char *host, void* addr)
{
	int ret;

	ret = inet_pton(af, host, addr);
	if (ret < 0 && errno == EAFNOSUPPORT)
		ret = AL_EAFNOSUPPORT;

	return ret;
}

const char* al_inet_ntop(int af, const void* src, char *dst, int len)
{
	return inet_ntop(af, src, dst, len);
}


/*----------------------------------------------------------------------------*\
 *                               DNS functions                                *
\*----------------------------------------------------------------------------*/

int al_gethostbyname(const char *name, al_in_addr_t * inaddr)
{
	struct hostent * he;
	// TODO: not thread-safe
	he = gethostbyname(name);
	if (he)
	{
		if (he->h_addrtype != AF_INET || he->h_length != sizeof(al_in_addr_t))
			return AL_EAFNOSUPPORT;
		memcpy(inaddr, he->h_addr_list[0], he->h_length);
		return AL_SUCCESS;
	}
	else
	{
		switch (h_errno)
		{
		case NETDB_INTERNAL:
			return AL_ENETDOWN;
		case HOST_NOT_FOUND:
			return AL_EHOSTNOTFOUND;
		case TRY_AGAIN:
			return AL_EAGAIN;
		case NO_RECOVERY:
			return AL_EINTERNAL;
		case NO_DATA:
			return AL_ENODATA;
		default:
			return AL_ERROR;	// TODO: should be a translated code.
		}
	}
}

/*----------------------------------------------------------------------------*\
 *                           Socket read functions                            *
\*----------------------------------------------------------------------------*/

int al_read (int d, void *buf, size_t len)
{
	return read(d, buf, len);
}

int al_recv (int s, void *buf, int len, unsigned int flags)
{
	return recv(s, buf, len, flags);
}

int al_recvfrom	(int			s,
				 void			*buf,
				 int 	len,
				 unsigned int 	flags,
				 al_sockaddr_t	*from,
				 int 	*fromlen)
{
#ifdef AL_SOCKADDR_FULL_OPAQUE
	return recvfrom(s, buf, len, flags, (struct sockaddr*)from, fromlen);
#else
	int ret;
	struct sockaddr_storage saddr;
	int saddr_len = sizeof(struct sockaddr_storage);
	ret = recvfrom(s, buf, len, flags, (struct sockaddr*)&saddr, &saddr_len);
	if (ret >= 0 && from && convert2aladdr(from, &saddr, fromlen))
		ret = AL_ERROR;
  return ret;
#endif
}

#ifdef AL_HAVE_RECVMSG	// Cygwin does not seem to have in_pktinfo (no IPV6 support)

#define MAX(a,b) (a < b ? b : a)

int al_recvon	(int			s,
				 void *			buf,
				 int 			len,
				 unsigned int 	flags,
				 al_sockaddr_t*	from,
				 int *			fromlen,
				 uint32_t * 	itf)
{
	if (itf)
	{
		int ret;
		struct msghdr mh;
		struct iovec iov;

		char cbuf[CMSG_SPACE(MAX(sizeof(struct in_pktinfo), sizeof(struct in6_pktinfo)))];
		struct cmsghdr *cmptr;

		if (from)
		{
#ifdef AL_SOCKADDR_FULL_OPAQUE
		mh.msg_name = from;
		mh.msg_namelen = *fromlen;
#else
		struct sockaddr_storage saddr;
		mh.msg_name = (struct sockaddr*)&saddr;
		mh.msg_namelen = sizeof(struct sockaddr_storage);
#endif
		}
		else
		{
			mh.msg_name = NULL;
			mh.msg_namelen = 0;
		}

		iov.iov_base = buf;
		iov.iov_len = len;
		mh.msg_iov = &iov;
		mh.msg_iovlen = 1;
		mh.msg_control = cbuf;
		mh.msg_controllen = sizeof(cbuf);
		mh.msg_flags = 0;

		ret = recvmsg(s, &mh, flags);

		/* Receive auxiliary data in msgh */
		if (ret >= 0)
		{
			for (cmptr = CMSG_FIRSTHDR(&mh); cmptr; cmptr = CMSG_NXTHDR(&mh, cmptr))
			{
				if (cmptr->cmsg_level == IPPROTO_IP && cmptr->cmsg_type == IP_PKTINFO)
				{
					*itf = ((struct in_pktinfo *) CMSG_DATA(cmptr))->ipi_ifindex;
					break;
				}
				if (cmptr->cmsg_level == IPPROTO_IPV6 && cmptr->cmsg_type == IPV6_PKTINFO)
				{
					*itf = (uint32_t)((struct in6_pktinfo *) CMSG_DATA(cmptr))->ipi6_ifindex;
					break;
				}
			}
			if (cmptr == NULL)
				ret = AL_ERROR;
		}
		else
			ret = AL_ERROR;

#ifndef AL_SOCKADDR_FULL_OPAQUE
		if (ret > 0 && from && convert2aladdr(from, &saddr, fromlen))
			ret = AL_ERROR;
#endif
		return ret;
	}
	else
		return al_recvfrom(s, buf, len, flags, from, fromlen);
}
#endif

/*----------------------------------------------------------------------------*\
 *                           Socket write functions                           *
\*----------------------------------------------------------------------------*/

int al_write (int d, const void *buf, size_t len)
{
	return write(d, buf, len);
}

int al_send (int s, const void *buf, int len, unsigned int flags)
{
	return send(s, buf, len, flags);
}

int al_sendto (int					s,
			  const void 			*buf,
			  int 					len,
			  unsigned int 			flags,
			  const al_sockaddr_t	*to,
			  int 					tolen)
{
#ifdef AL_SOCKADDR_FULL_OPAQUE
	return sendto(s, buf, len, flags, (const struct sockaddr*)to, tolen);	// INVALID_SOCKET = AL_ERROR = -1
#else
	struct sockaddr_storage n_to;
	convert2naddr(&n_to, to);
	return sendto(s, buf, len, flags, (const struct sockaddr*)&n_to , sizeof(struct sockaddr_storage));	// INVALID_SOCKET = AL_ERROR = -1
#endif
}
