/*============================================================================*\
|                                                                              |
|                          SOA4D Abstraction Layer                             |
|                                                                              |
|               ->>  Copyright 2008 Schneider Electric SA <<-                  |
|                                                                              |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 1.9 $
|                     $Date: 2008/02/11 07:57:02 $
\*============================================================================*/
/*******************************************************************************
*                                 IP features                                  *
*******************************************************************************/
#ifndef AL_IP_H_
#define AL_IP_H_

/** \file
    Abstract Layer (AL): BSD sockets portable API.
    <em>Note</em>: This API is mainly an indirection allowing the user to solve
    portability issues about:
    <ul>
    	<li>The socket address structure incompatibilities.</li>
    	<li>Error code translation.</li>
    	<li>The type of parameter for some of the functions.</li>
    </ul>
    Won't be addressed:
    <ul>
    	<li>The parameter semantics and types for socket, setsockopt and iocl APIs.</li>
    </ul>
    This API is made to be implemented on top of an existing BSD API.
*/

#include "al_common.h"

#if defined AL_STACK_WINSOCK || AL_STACK_WINSOCK_CE
# include "alIP_Winsock.h"
#elif defined AL_STACK_LWIP
# include "al_ip_lwip.h"
#elif defined AL_STACK_BSD
# include "alIP_Bsd.h"
#elif defined AL_STACK_ANDERSEN
# include "alIP_Andersen.h"
#elif defined AL_STACK_NEXGEN
#include "alIP_Nexgen.h"
#else
# error "No stack defined ..."
#endif


#define AL_SOCKADDR_VERS  1


/*
 * This following flag can be defined,
 * when all DPWS software does not call deprecated functions
 * as inet_aton(), inet_ntoa,, ...
 * Today, it is undefined to keep a IP backward compatibily.
 */
/* #define AL_F0LLOW_RFC3493 */

/*
 * Define the default IP version to be used in the function al_sockaddr_init()
 */
//#define ALPF_INET_IPDFT    AF_INET /* Force to use IPv4-Only */
#define ALPF_INET_IPDFT    0       /* Use IPv6 if it is supported by the target */

/*----------------------------------------------------------------------------*\
 *                               IP stack init                                *
\*----------------------------------------------------------------------------*/

/** \name IP stack general API.
 */
/*@{*/

/**
 * Initialize the IP stack.
 * Must be called before any other call to AL IP APIs. Was introduced mainly for
 * WSAStartup support.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_ip_init(void);	// mainly for WSAStartup. Per process init

/**
 * Shutdowns the IP stack.
 * Must be called when no other call to AL IP APIs is required. Was introduced
 * mainly for WSACleanup support.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_ip_shutdown(void);

/*----------------------------------------------------------------------------*\
 *                           IP network error management                      *
\*----------------------------------------------------------------------------*/

/**
 * Returns the last AL IP error status.
 * Error codes returned are AL translation of native codes. Note that separating
 * error code retrieval from the API responsible for the error is not
 * thread-safe which may not be a problem in a RTOS context (depending on the
 * preemption scheme).
 * @return An AL IP error code.
 */
int al_ip_error(void);

/*@}*/

/*----------------------------------------------------------------------------*\
 *                               Types definitions                            *
\*----------------------------------------------------------------------------*/

/** \name IP socket API.
 */
/*@{*/

/**
 * Portable duration structure.
 */
typedef struct al_timeval
{
    long    sec;	/**< Number of seconds */
    long    usec;	/**< Number of microseconds */
} al_timeval_t;


/*----------------------------------------------------------------------------*\
 *                      Internet domain socket address                        *
\*----------------------------------------------------------------------------*/

/**
 * IPv4 address structure.
 */
typedef struct al_in_addr
{
	uint32_t    addr;     /**< 32 bits IP address, network byte order. s_addr renamed addr to avoid macro conflict on windows. */
} al_in_addr_t;

/**
 * IPv6 address structure.
 */
typedef struct al_in6_addr {
     uint8_t    sAddr[16];    /* 128 bits IP address, network byte order */
} al_in6_addr_t;


#ifndef AL_SOCKADDR_FULL_OPAQUE
/**
 * Socket address usable for the IPv4 protocol.
 */
typedef struct al_sockaddr_in
{
#ifdef AL_HAVE_SOCKADDR_LEN
    uint8_t         sin_len;
    uint8_t         sin_family;     /**< AF_INET */
#else
    int16_t         sin_family;     /**< AF_INET */
#endif
    uint16_t        sin_port;       /**< 16 bits port number, net byte order */
    al_in_addr_t    sin_addr;       /**< 32 bits IP address :(IPv4) */
    unsigned char   sin_zero[8];    /**< Unused. Could be removed if some AL implementation did not perform direct copy of native structure */
} al_sockaddr_in_t;

/**
 * Socket address usable for the IPv6 protocol.
 */
typedef struct al_sockaddr_in6
{
#ifdef AL_HAVE_SOCKADDR_LEN
    uint8_t         sin6_len;
    uint8_t         sin6_family;     /**< AF_INET */
#else
    int16_t         sin6_family;     /**< AF_INET */
#endif
    uint16_t        sin6_port;      /**< 16 bits port number, net byte order */
    uint32_t        sin6_flowinfo;  /**< 32 bits flow label, net byte order */
    al_in6_addr_t   sin6_addr;      /**< 128 bits IP address (IPv6) */
    uint32_t        sin6_scope_id;  /**< 32 bits interface identifier */
} al_sockaddr_in6_t;

#endif /* AL_SOCKADDR_FULL_OPAQUE */

/**
 * Socket address usable for any protocol.
 */
typedef struct al_sockaddr
{
#ifdef AL_SOCKADDR_FULL_OPAQUE
    uint8_t         sa_data[AL_SOCKADDR_STORAGE_SIZE];
#else
#ifdef AL_HAVE_SOCKADDR_LEN
    uint8_t         sa_len;
    uint8_t         sa_family;     /**< AF_INET */
#else
    int16_t         sa_family;     /**< AF_INET */
#endif
    uint8_t         sa_data[AL_SOCKADDR_DATA_SIZE];
#endif
} al_sockaddr_t;



/*----------------------------------------------------------------------------*\
 *                           Address conversion macros                        *
\*----------------------------------------------------------------------------*/

#if defined (AL_BIG_ENDIAN)         /* big endian processor definition */
# define al_htons(host)       (unsigned short)(host)	/**< Converts a short from the host to the network order. */
# define al_ntohs(net)        al_htons(net)	/**< Converts a short from the network to the host order. */
# define al_htonl(host)       (unsigned long)(host)	/**< Converts a long from the host to the network order. */
# define al_ntohl(net)        al_htonl(net)	/**< Converts a long from the network to the host order. */
#elif defined (AL_LITTLE_ENDIAN)    /* little endian processor definition */
# define al_htons(host)       ((((unsigned short)(host) >> 8) & 0x00FF) | (((unsigned short)(host) << 8) & 0xFF00))	/**< Converts a short from the host to the network order. */
# define al_ntohs(net)        al_htons(net)	/**< Converts a short from the network to the host order. */
# define al_htonl(host)       ((((unsigned long)(host) >> 24) & 0x000000FF) | (((unsigned long)(host) >> 8)  & 0x0000FF00) | \
                              (((unsigned long)(host) << 8)  & 0x00FF0000) | (((unsigned long)(host) << 24) & 0xFF000000))	/**< Converts a long from the host to the network order. */
# define al_ntohl(net)        al_htonl(net)	/**< Converts a long from the network to the host order. */
#else
# error "AL_XXX_ENDIAN macro not defined"
#endif


/*----------------------------------------------------------------------------*\
 *                  Sockets creation and management functions                 *
\*----------------------------------------------------------------------------*/

/**
 * Create a new socket.
 * This function creates a new socket in the specified protocol domain and
 * returns a descriptor for it that can be used by other socket routines.
 * @param domain BSD domains macros should be used. Only AF_INET (IPv4) is
 * required to be supported by the implementation.
 * @param type BSD domains macros should be used. SOCK_STREAM for TCP,
 * SOCK_DGRAM for UDP. Other types may be supported since parameters should be
 * passed directly to the underlying BSD implementation.
 * @param protocol Protocol number. Generally ignored for the AF_INET family.
 * @return A socket descriptor or an AL error code.
 */
int al_socket(int domain, int type, int protocol);


#define AL_SHUTDOWN_RCV 0	/**< Receiving disallowed */
#define AL_SHUTDOWN_SND 1	/**< Sending disallowed */
#define AL_SHUTDOWN_ALL 2	/**< Receiving and sending disallowed */

/**
 * Shut down the whole or part of a socket connection.
 * @param s Socket descriptor.
 * @param how. One of \a AL_SHUTDOWN_RCV, \a AL_SHUTDOWN_SND, \a AL_SHUTDOWN_ALL.
 * @return AL_SUCCESS if no error occurred. If AL_ERROR is returned, use
 * al_ip_error() to retrieve the AL IP error code.
 */
int al_shutdown(int s, int how);

/**
 * Delete a socket descriptor.
 * @param s Socket descriptor.
 * @return AL_SUCCESS if no error occurred. If AL_ERROR is returned, use
 * al_ip_error() to retrieve the AL IP error code.
 */
int al_sock_close(int s);

/**
 * Retrieves a socket address.
 * @param s Socket descriptor.
 * @param[out] addr The structure that will receive the socket address.
 * @param[out] addrlen The length of the actual returned \a addr parameter.
 * @return AL_SUCCESS if no error occurred. If AL_ERROR is returned, use
 * al_ip_error() to retrieve the AL IP error code.
 */
int al_getsockname(int s, al_sockaddr_t *addr, int *addrlen);

/**
 * This function lets the application set options on a socket.
 * <em>Note<em>: The option selection parameters use directly underlying BSD
 * implementation macros. A potential problem is possible since the \a optval
 * parameter type may differ from one implementation to another...
 * @param s Socket descriptor.
 * @param level Typical BSD values are SOL_SOCKET, IPPROTO_IP, IPPROTO_TCP,
 * IPPROTO_UDP.
 * @param optname The option identifier relative to the \a level selected.
 * @param optval. Several types of parameters can be used for the option value:
 * <ul>
 * 	<li>Boolean options generally use an \a int value.</li>
 * 	<li>Other options generally use specific structures.</li>
 * 	<li>Some int values may use different integer types. This API will define
 *  a portable type for options for which such incompatibilities have been
 *  detected:
 * 		<ul>
 * 			<li>For IP_MULTICAST_TTL an unsigned char will be used.</li>
 * 		</ul>
 *  </li>
 * </ul>
 * @param optlen The length of the \a optval parameter.
 * @return AL_SUCCESS if no error occurred. If AL_ERROR is returned, use
 * al_ip_error() to retrieve the AL IP error code.
 */
int al_setsockopt(int s, int level, int optname, const void *optval, int optlen);

/**
 * This function lets the application get options on a socket.
 * @see al_setsockopt().
 * @param s Socket descriptor.
 * @param level As for al_setsockopt().
 * @param optname As for al_setsockopt().
 * @param[out] optval.  As for al_setsockopt() but as an output. The buffer
 * must be supplied by the caller.
 * @param[out] optlen The length of the actual returned \a optval parameter.
 * @return AL_SUCCESS if no error occurred. If AL_ERROR is returned, use
 * al_ip_error() to retrieve the AL IP error code.
 */
int al_getsockopt(int s, int level, int optname, void *optval, int *optlen);

/**
 * Portable command flag for specifying non-blocking IOs. A boolean value is
 * expected.
 */
#define AL_FIONBIO	1

/**
 * Perform a command on the socket descriptor.
 * Some implementations may implement this function using fcntl, ioctl or
 * ioctlsocket according to the portable command implemented.
 * @param s Socket descriptor.
 * @param The command to perform. Only portable commands defined in AL IP are
 * supported.
 * @param arg The command argument.
 * @return AL_SUCCESS if no error occurred. AL_EOPTION if the \a cmd is not
 * supported. An AL error code else.
 */
int al_ioctl(int s, int cmd, int * arg);

/**
 * Accept a connect on a socket.
 * This function is called on a listening connection-oriented socket in order
 * create a new socket for an incoming connection.
 * @param s Socket descriptor.
 * @param[out] addr A socket address that will be filled with the connecting
 * entity address. The socket address structure buffer is provided by the caller.
 * @param[in,out] addrlen Should be initialized with the provided \a addr
 * buffer and will be filled with the actual size of the returned structure.
 * @return The socket descriptor to use for connected communications. If an
 * error AL_ERROR is returned: use al_ip_error() to retrieve the AL IP error
 * code.
 */
int al_accept(int s, al_sockaddr_t 		*addr, int *addrlen);

/**
 * Assign a local protocol address to a socket.
 * @param s Socket descriptor.
 * @param addr The local address to which the socket is to be bound.
 * @param addrlen The size of the \a addr parameter.
 * @return AL_SUCCESS if no error occurred. If AL_ERROR is returned, use
 * al_ip_error() to retrieve the AL IP error code.
 */
int al_bind(int s, const al_sockaddr_t *addr, int  addrlen);

/**
 * Initiate a connection on a socket.
 * @param s Socket descriptor.
 * @param addr The peer address for the connection.
 * @param addrlen The size of the \a addr parameter.
 * @return AL_SUCCESS if no error occurred. If AL_ERROR is returned, use
 * al_ip_error() to retrieve the AL IP error code.
 */
int al_connect(int s, const al_sockaddr_t *addr, int  addrlen);

/**
 * Listen for connections on a socket.
 * @param s Socket descriptor.
 * @param backlog The maximum length the queue of pending connections may grow
 * to.
 * @return AL_SUCCESS if no error occurred. If AL_ERROR is returned, use
 * al_ip_error() to retrieve the AL IP error code.
 */
int al_listen(int s, int backlog);

/**
 * Monitors a set of sockets for read or write.
 * This function examines a set of socket descriptors to check if some data can
 * be read or written to a specified socket. Note that the <tt>al_fd_set</tt>
 * type is defined in a platform dependent way (generally preprocessor macro).
 * The socket to be checked must be added to the file descriptors array before
 * this function is called. After the call the array only contains descriptors
 * ready for the specified action. This function must be used in polling mode
 * when the application has to wait for multiple events. The sets of file
 * descriptors should be accessed with the following macros:
 * <ul>
 * 	<li>FD_ZERO( &fdset ) Resets the fdset set.</li>
 *	<li>FD_SET( fd , &fdset ) Adds the fd file descriptor to the fdset set.</li>
 *	<li>FD_CLR( fd , &fdset ) Removes the fd file descriptor from the fdset
 * 	set.</li>
 *	<li>FD_ISSET( fd , &fdset ) Tests if fd is a member of the fdset set.</li>
 * </ul>
 * @param nfds Maximum file descriptor number plus one.
 * @param readfds Set of file descriptors to test for reading, may be NULL.
 * @param writefds Set of file descriptor to test for writing, may be NULL.
 * @param exceptfds Set of file descriptor to test for exceptional condition,
 * may be NULL.
 * @param timeout Maximum delay to wait for, may be NULL.
 * @return The number of ready descriptors that are contained in the descriptor
 * sets, or AL_ERROR if an error occurred. If the time limit expires, select()
 * returns 0 (AL_SUCCESS). If select() returns with an error, including one due
 * to an interrupted call, the descriptor sets will be unmodified.
 */
int al_select(	int 			nfds,
				al_fd_set 		*readfds,
				al_fd_set 		*writefds,
				al_fd_set 		*exceptfds,
				al_timeval_t 	*timeout);


/*----------------------------------------------------------------------------*\
 *                Address management and translation functions                *
\*----------------------------------------------------------------------------*/

#ifndef AL_F0LLOW_RFC3493 // NEED TO OBSOLETE THESE TWO FUNCTIONS
/**
 * This function converts a 32-bits internet address into a string in standard
 * dot notation.
 * @param addr An IP host address in the network byte order.
 * @param[out] buffer A buffer for receiving the string address. Should be 16
 * byte long at least.
 * @return AL_SUCCESS if no error occured. If AL_ERROR is returned, use
 * al_ip_error() to retrieve the AL IP error code.
 */
int al_inet_ntoa(uint32_t addr, char *buffer);

/**
 * This function converts an internet address in standard dot notation into
 * a 32-bits internet address.
 * @param str Internet address in the numbers-and-dots notation.
 * @param[out] addr A buffer that will receive the IP host address in the
 * network byte order.
 * @return AL_SUCCESS if no error occured. If AL_ERROR is returned, use
 * al_ip_error() to retrieve the AL IP error code.
 */
int al_inet_aton(const char *str, uint32_t *addr);

#endif /* AL_F0LLOW_RFC3493 */

/*
 * See RFC3493
 */
#define AL_INET4_ADDRSTRLEN        16
#define AL_INET6_ADDRSTRLEN        46

/**
 * Convert a Network address to Printable characters string according to the
 * required domain specified by af.
 * @param af The address family.
 * @param src A pointer to a network address structure.
 * @param dst A buffer to receive the printable address.
 * @param len The size of the dst buffer.
 * @return The converted string or NULL if an error occured. The error can be
 * retrieved using al_ip_error() especially AL_EAFNOSUPPORT or AL_ENOSPC if
 * the supplied buffer is not big enough.
 */
const char* al_inet_ntop(int af, const void* src, char *dst, int len);

/**
 * Convert a printable characters string to a network address
 * according to the required domain specified by af.
 * @param af The address family.
 * @param src The address in the string format.
 * @param[in,out] dst A pointer to a structure that will receive the binary
 * network address.
 * @return AL_EAFNOSUPPORT if the address family is not supported, 0 if
 * the input address is invalid or a positive value if the conversion was
 * successful.
 */
int al_inet_pton(int af, const char *src, void *dst);


/*----------------------------------------------------------------------------*\
 *                           Socket read functions                            *
\*----------------------------------------------------------------------------*/

/**
 * Read data from the socket.
 * Note that the flags used for this API are the native BSD ones and are not
 * redefined by the AL API.
 * @param s Socket descriptor.
 * @param buf[out] Pointer to a buffer for receiving data.
 * @param len Maximum number of bytes to read.
 * @param flags Typical BSD flags are MSG_PEEK, MSG_DONTWAIT, MSG_OOB...
 * @return The number of received bytes. If AL_ERROR is returned, use
 * al_ip_error() to retrieve the AL IP error code.
 */
int al_recv(int s, void *buf, int len, unsigned int flags);

/**
 * Reads data from a file descriptor.
 * This API is somehow redundant with al_recv() but works on both files and
 * sockets which may be useful for some applications.
 * @param d A descriptor.
 * @param buf[out] Pointer to a buffer for receiving data.
 * @param len Maximum number of bytes to read.
 * @return The number of received bytes. 0 if end of file was encountered. If
 * AL_ERROR is returned, use al_ip_error() to retrieve the AL IP error code.
 */
int al_read(int d, void *buf, size_t len);

/**
 * Read data from the socket.
 * This function can be used both for datagram or connection-oriented but
 * is mainly used for datagram sockets in order to retrieve the sender
 * address.
 * @param s Socket descriptor.
 * @param buf[out] Pointer to a buffer for receiving data.
 * @param len Maximum number of bytes to read.
 * @param flags Typical BSD flags are MSG_PEEK, MSG_DONTWAIT, MSG_OOB...
 * @param[out] from A socket address that will be filled with the source
 * entity address. The socket address structure buffer is provided by the caller.
 * @param[in,out] fromlen Should be initialized with the provided \a from
 * buffer and will be filled with the actual size of the returned structure.
 * @return The number of received bytes. If AL_ERROR is returned, use
 * al_ip_error() to retrieve the AL IP error code.
 */
int al_recvfrom	(int			s,
				 void			*buf,
				 int			len,
				 unsigned int 	flags,
				 al_sockaddr_t	*from,
				 int 			*fromlen);

/**
 * Read data from the socket.
 * This function can be used for datagram sockets in order to retrieve the sender
 * address and the reception interface if the socket was bound to INADDR_ANY.
 * @param s Socket descriptor.
 * @param buf[out] Pointer to a buffer for receiving data.
 * @param len Maximum number of bytes to read.
 * @param flags Typical BSD flags are MSG_PEEK, MSG_DONTWAIT, MSG_OOB...
 * @param[out] from A socket address that will be filled with the source
 * entity address. The socket address structure buffer is provided by the caller.
 * @param[in,out] fromlen Should be initialized with the provided \a from
 * buffer and will be filled with the actual size of the returned structure.
 * @param[out] itf If not NULL, the reception interface is retrieved provided
 * the IP(V6)_PKTINFO has been turned on on the socket.
 * @return The number of received bytes. If AL_ERROR is returned, use
 * al_ip_error() to retrieve the AL IP error code.
 */
int al_recvon	(int			s,
				 void			*buf,
				 int 			len,
				 unsigned int 	flags,
				 al_sockaddr_t	*from,
				 int			*fromlen,
				 uint32_t		*itf);

/*----------------------------------------------------------------------------*\
 *                           Socket write functions                           *
\*----------------------------------------------------------------------------*/

/**
 * Send a message from a socket.
 * Note that the flags used for this API are the native BSD ones and are not
 * redefined by the AL API.
 * @param s Socket descriptor.
 * @param buf Pointer to a buffer containing data to be sent.
 * @param len Number of bytes to send.
 * @param flags Typical BSD flags are MSG_DONTROUTE, MSG_OOB...
 * @return The number of sent bytes. If AL_ERROR is returned, use al_ip_error()
 * to retrieve the AL IP error code.
 */
int al_send (int s, const void *buf, int len, unsigned int flags);

/**
 * Write to a file descriptor.
 * This API is somehow redundant with al_send() but works on both files and
 * sockets which may be useful for somme applications.
 * @param d A descriptor.
 * @param buf Pointer to a buffer containing data to be sent.
 * @param len Number of bytes to send.
 * @return The number of sent bytes. If AL_ERROR is returned, use al_ip_error()
 * to retrieve the AL IP error code.
 */
int al_write (int d, const void *buf, size_t len);

/**
 * Send a message from a socket.
 * This function can be used both for datagram or connection-oriented but is
 * mainly used for datagram sockets in order to specify the destination address.
 * Note that the flags used for this API are the native BSD ones and are not
 * redefined by the AL API.
 * @param s Socket descriptor.
 * @param buf Pointer to a buffer containing data to be sent.
 * @param len Number of bytes to send.
 * @param flags Typical BSD flags are MSG_DONTROUTE, MSG_OOB...
 * @param to A socket address that will be filled with the source entity
 * address. The socket address for the target of the message.
 * @param tolen Should contain the size of the provided \a to buffer.
 * @return The number of sent bytes. If AL_ERROR is returned, use al_ip_error()
 * to retrieve the AL IP error code.
 */
int al_sendto (int					s,
			  const void 			*buf,
			  int 					len,
			  unsigned int 			flags,
			  const al_sockaddr_t	*to,
			  int 					tolen);

/*@}*/

/*----------------------------------------------------------------------------*\
 *                Socket Address management functions                         *
\*----------------------------------------------------------------------------*/

/** \name SockAddr API.
 */
/*@{*/

#ifdef AL_HAVE_IPV6
typedef struct al_ipNetAddr
{
	uint16_t netAddrLen; /** 4 for IPv4 , 16 pour IPv6, */
    union {
      al_in_addr_t   netAddr4;
      al_in6_addr_t  netAddr6;
    } addr;
} al_ipNetAddr_t;
#else
typedef uint32_t  al_ipNetAddr_t;
#endif

/**
 * Return an indicator specifying if the Socket IP Address variable is IPv4
 * or IPv6 address.
 * @param sockAddr Pointer to a Socket IP Address variable.
 * @return On success, the function returns:
 * <ul>
 *   <li>0 when the domain is IPv4</li>
 *   <li>1 when the domain is IPv6</li>
 * </ul>
 * Otherwise, the function returns a negative value specifying the error cause.
 */
int al_sockaddr_checkFamily(const al_sockaddr_t* sockAddr);

/**
 * Initialize the sockAddr variable with the default values according to the
 * requested IP version and the configuration of the implemented IP stack
 * (IPv4-Only, IPv6-Only or Dual IPv4/IPv6). The IP address will set to the ANY
 * IP Address value.
 * @param domain Specifies the INET domain requested by the user application.
 * When it is set to ALPF_INET_IPDFT, the abstract layer selects the implemented
 * IP version. For a dual IP stack, it will be IPv6.
 * @param sockAddr Pointer to a Socket IP Address variable in the user space.
 * @return On success, the function returns the effective byte length used for
 * the opaque sockAddr structure. Otherwise, the function returns a negative
 * value specifying the error cause (TR_EINVAL, TR_ESOCKTNOSUPPORT).
 */
int al_sockaddr_init(int domain, al_sockaddr_t* sockAddr, int addrlen);

/**
 * Returns the domain specified into an Socket IP Address variable.
 * @param sockAddr Pointer to a Socket IP Address variable.
 * @return On success, the function returns the protocol family for the address.
 * Otherwise, the function returns a negative value specifying the error cause.
 */
int al_sockaddr_getDomain(const al_sockaddr_t* sockAddr);

/**
 * Retrieves the length of the Socket IP Address variable  according to the used
 * domain (IPv4 or IPv6).
 * @param sockAddr Pointer to a Socket IP Address variable.
 * @return On success, the function returns the effective byte length used
 * for the opaque sockAddr structure. Otherwise, the function returns a
 * negative value specifying the error cause.
 */
int al_sockaddr_getLength(const al_sockaddr_t* sockAddr);

/**
 * Retrieves the port number from the Socket IP Address variable.
 * @param sockAddr Pointer to a Socket IP Address variable.
 * @param port Pointer to a variable that will be used to return the port number.
 * @return On success AL_SUCCESS(0). Otherwise, the function returns a negative
 * value specifying the error cause	AL_ESOCKTNOSUPPORT.
 */
int al_sockaddr_getPort(const al_sockaddr_t* sockAddr, uint16_t* port);

/**
 * Retrieves the scope identifier from the Socket IP Address variable.
 * @param sockAddr Pointer to a Socket IP Address variable.
 * @param scope Pointer to a variable that will be used to return the scope
 * identifier.
 * @return AL_SUCCESS or an AL error code.
 */
int al_sockaddr_getScope(const al_sockaddr_t* sockAddr, uint32_t* scope);

/**
 * Retrieves the IP Address (in network format) from the Socket IP Address
 * variable.
 * @param sockAddr Pointer to a Socket IP Address variable.
 * @param[out] ipNetAddr Pointer to a variable that will be used to return the IP
 * address in network format.
 * @return AL_SUCCESS or an AL error code.
 */
int al_sockaddr_getNetAddr(const al_sockaddr_t* sockAddr, al_ipNetAddr_t* ipNetAddr);

/**
 * Retrieves the IP Address from the Socket IP Address variable in a string with
 * the standard text presentation format.
 * @param sockAddr Pointer to a Socket IP Address variable.
 * @param addrStr Pointer to a buffer that will be used to return the IP
 * address in dot format. The size should be AL_INET6_ADDRSTRLEN at least long.
 * @param len The length of the \a addrStr buffer.
 * @param scope This boolean specifies if the scope identifier must be appended to
 * the string address (for link-local addresses only).
 * @return AL_SUCCESS or an AL error code.
 */
int al_sockaddr_getNetAddrToAddrStr(const al_sockaddr_t* sockAddr, char *addrStr, int len, int scope);

/**
 * Test if an IP address is the loopback.
 * @param sockAddr Pointer to a Socket IP Address variable.
 * @return A boolean value.
 */
int al_sockaddr_isLoopback(al_ipNetAddr_t* ipNetAddr);

/**
 * Sets the port number in the Socket IP Address variable.
 * Before calling this function, the variable must be initialized by
 * al_sockaddr_init() function.
 * @param sockAddr Pointer to a Socket IP Address variable that has been already
 * initialized.
 * @return AL_SUCCESS or an AL error code.
 */
int al_sockaddr_setPort(al_sockaddr_t* sockAddr, uint16_t port);

/**
 * Sets the scope identifier in the Socket IP Address variable.
 * Before calling this function, the variable must be initialized by
 * al_sockaddr_init() function.
 * @param sockAddr Pointer to a Socket IP Address variable that has been already
 * initialized.
 * @return AL_SUCCESS or an AL error code.
 */
int al_sockaddr_setScope(al_sockaddr_t* sockAddr, uint32_t scope);

/**
 * Sets the IP Address in the Socket IP Address variable from the address name
 * in the standard text presentation format.
 * @param sockAddr Pointer to a Socket IP Address variable that has been already
 * initialized.
 * @param Pointer to a character string specifying the IP address. It can contain
 * the scope identifier (%n) that is parsed contrary to al_inet_ntop().
 * @return AL_SUCCESS or an AL error code.
 */
int al_sockaddr_setNetAddrFromAddrStr(al_sockaddr_t* sockAddr, const char *addrStr);

/**
 * Set the IP Loopback address in the Socket IP Address variable according to
 * the IP domain specified by this variable (IPv4: 127.0.0.1, IPv6: ::0).
 * @param sockAddr Pointer to a Socket IP Address variable that has been already
 * initialized.
 * @return AL_SUCCESS or an AL error code.
 */
int al_sockaddr_setInAddrLoopBack( al_sockaddr_t* sockAddr);
/*@}*/


/*----------------------------------------------------------------------------*\
 *                               DNS functions                                *
\*----------------------------------------------------------------------------*/

/** \name DNS API.
 */
/*@{*/

/**
 * Get IP address for a network host.
 * This is a very simplified version of gethostbyname since the original has
 * some defects:
 * <ul>
 * 	<li>not reentrant (the returned result is static),</li>
 *  <li>allocation of results by the callee with variable data-length and
 *  polyporphism</li>
 * </ul>
 * @param name The host name for which IP address must be retrieved.
 * @param[out] addr A structure that will contain the host address.
 * @return AL_SUCCESS if no error occured. If AL_ERROR is returned, use
 * al_ip_error() to retrieve the AL IP error code.
 */
int al_gethostbyname(const char *name, al_in_addr_t * addr);

/**
 * Simplified version of getaddrinfo that allows al_sockaddr_t structure
 * construction whatever is the format of the supplied address (name or IP
 * address, V4 or V6).
 * It replaces al_gethostbyname() which is obsolete for IPV6.
 * @param af The address family (AF_INET, AF_INET6).
 * @param host IP address or host name for which socket addresses must be
 * retrieved. If host is an IPv6 address, it may contain a scope suffix for
 * interface specification using "%<netif index>".
 * @param port The port that should be set in socket addresses.
 * @param[out] addrs A array of returned socket address structures allocated
 * by the API. It should be freed by the caller using \a al_freeaddrinfo.
 * @param[out] nbAddr The number of entries of the return array.
 * @return AL_SUCCESS or an AL IP error code.
 */
int al_getaddrinfo(int af, const char *host, uint16_t port, al_sockaddr_t ** addrs, int * nbAddrs);

/**
 * Releases \a al_getaddrinfo result memory.
 * @param[in] addrs The array of socket address structures returned by
 * \a al_getaddrinfo.
 * @return AL_SUCCESS or an AL IP error code.
 */
int al_freeaddrinfo(al_sockaddr_t * addrs);

/*@}*/

#endif /*AL_IP_H_*/
