/*============================================================================*\
|                                                                              |
|                          SOA4D DPWSCore Samples                              |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|                                                                              |
|        + File info:                                                          |
|                     $Revision: 2197 $
|                     $Date: 2009-03-11 18:56:28 +0100 (mer, 11 mar 2009) $
\*============================================================================*/
#include <stdio.h>

#include "dc/dc_Dpws.h"	// Main DPWSCore API include file.

/* Generated stub & skeleton include file. */
#include "litStub.h"
#include "wshStub.h"
#include "prjStub.h"

#include "LightingClient.h"	// Lights client menu.
#include "WasherClient.h"	// Washing machine client menu.
#include "ProjectorClient.h"	// Projector client menu.

#include "serverDPWS.h"	// OS-dependent code (especially server loop).

/*
 *	Definitions
 */

#define MAX_SCOPE_LENGTH 255	// Maximum length for computed scope.
#define SAMPLE_SCOPE "http://www.soa4d.org/DPWS/Samples/Home/Scope"	// The base scope for the sample.

#define SAMPLE_NS "http://www.soa4d.org/DPWS/Samples/Home"	// The namespace of the types supported by the searched devices (must be the same as in server.c).
#define SIMPLE_LIGHT_TYPE "SimpleLight"	// The local name of the type supported by the searched lights (must be the same as in server.c).
#define DIMMER_LIGHT_TYPE "DimmerLight"	// The local name of the type supported by the searched dimmer lights (must be the same as in server.c).
#define WASHER_TYPE "WashingMachine"	// The local name of the type supported by the searched washing machines (must be the same as in server.c).
#define PROJECTOR_TYPE "Projector"	// The local name of the type supported by the searched projectors (must be the same as in server.c).

/* Callback function called when new device is added or update in the discovery cache. */
static void device_joining(struct dpws *dpws, short href_device) {
	printf("\n<-- New device on the LAN: %s\n", dpws_cache_get_uuid(dpws, href_device));
}

/* Callback function called when a device is removed from the discovery cache. */
static void device_leaving(struct dpws *dpws, short href_device) {
	printf("\n<-- Device %s leaving the LAN.\n", dpws_cache_get_uuid(dpws, href_device));
}

static void subscription_cancelled(struct event_end_info * eei)
{
	printf("\n<-- Subscription %s\n    from %s cancelled.\n",
			eei->subscription_manager->subscription_id, eei->subscription_manager->address);
	printf(" Status:  %s\n Reason:  %s\n", eei->status, eei->reason);
}

/*
 *	Runtime data
 */

struct dpws	listen_dpws,	// runtime structure used by the event listener.
			dpws;	// runtime structure used by the client.


/*
 *  Utilities
 */

static char diversified_scope[MAX_SCOPE_LENGTH];	// computed scope buffer

/* Function adding the suffix to default scope (if suffix was defined).
 * Returns 0 if ok 1 if error (string too long).
 */
static int compute_diversified_scope(char *scope_suffix)
{
	strcpy(diversified_scope, SAMPLE_SCOPE);
	if (scope_suffix) {
		if (sizeof(SAMPLE_SCOPE) + strlen(scope_suffix) > MAX_SCOPE_LENGTH)
			return 1;
		if (scope_suffix[0] != '/')
			strcat(diversified_scope, "/");
		strcat(diversified_scope, scope_suffix);
	}
	return 0;
}

/* Function printing error message. */
static void print_error(char * msg, struct dpws * dpws) {
	fprintf(stderr, "\nError occured : %s.\n%s\n\n", msg, dpws_get_error_msg(dpws));
}

/* Function displaying usage summary. */
static void simple_usage() {
	fprintf(stderr, "\nUsage: client [-s <scopeDiversifier>] [-p <httpPort>]\n");
	fprintf(stderr, "Use -?, -h or -help for more information\n\n");
	exit(1);
}

/* Function displaying complete program usage. */
static void usage() {
	fprintf(stderr, "\nUsage: client [-s <scopeDiversifier>] [-p <httpPort>]\n");
	fprintf(stderr, "\nOptions : \n");
	fprintf(stderr, "\n -s <scopeDiversifier>\n");
	fprintf(stderr, "\n\tThe <scopeDiversifier> is used to avoid conflicts on the local\n");
	fprintf(stderr, "\tnetwork by participating in building the scope of the device.\n");
	fprintf(stderr, "\n -p <httpPort>\n");
	fprintf(stderr, "\n\tThe <httpPort> is the http port used to listen to soap messages.\n");
	fprintf(stderr, "\nNotes : \n");
	fprintf(stderr, "\n\tThe client should be started after the server.\n\n");
	exit(1);
}

/*
 *  Main function performing server intialization.
 *	Usage: Usage: client [-s <scope_suffix>] [-p <listenPort>]
 */
int main(int argc, char* argv[]) {

	short device_proxy = -1, * device_proxies, event_endpoint;
	int nbDevEndPt = 10, indexDev = 0, port = 8889, ret = DPWS_OK, i, status;
	char * scope_suffix = NULL;
	char ** scopes;
	struct device_metadata *metadata;
	qname_t type = {SAMPLE_NS, NULL};
	discovery_filter_t filter = {&type, 1, NULL, 0, WSD_MATCH_BY_UNSPECIFIED};

	// Command line parsing
	if (argc > 1 && (!strcmp(argv[1], "-?") || !strcmp(argv[1], "-h") || !strcmp(argv[1], "-help"))) {
		usage();
	}

	if ((argc < 1) || (argc > 5))
		simple_usage();

	for (i = 1; i < argc; i++) {
		if (!strcmp(argv[i], "-s")) {
			if (++i == argc)
				simple_usage();
			scope_suffix = argv[i];
		} else if (!strcmp(argv[i], "-p")) {
			if (++i == argc)
				simple_usage();
			port = atoi(argv[i]);
			if (port == 0)
				simple_usage();
		} else
			simple_usage();
	}

	// Compute diversified scope, the scope will be used to discover devices on the network.
	// If a suffix has been used only the servers started using the same suffix will be
	// detected.
	if (compute_diversified_scope(scope_suffix)) {
		fprintf(stderr, "scope_suffix is too long, should not exceed %d characters\n", MAX_SCOPE_LENGTH - sizeof(SAMPLE_SCOPE));
		simple_usage();
	}

	/* Client program description (client and listener initialization) */
	// 1. Initialize the DPWS stack.
	// 2. Initialize the dpws structure for client operations.
	// 3. Configure the event listener.
	// 4. Initialize the event listener.
	// 5. Start the event listener loop.
	// 6. Loop on menu display depending on device supported types
	//	6.1 Find available devices.
	//	6.2 Retrieve and display device info.
	//	6.3 Call menu depending on device types.
	// 7. Stop the event listener.

	/* 1. Initialize the DPWS stack */

	// This is the first function to be called when using the DPWS stack.
	// This version selects one single local IPv4 address.
	if ((status = dpws_init())) {
		fprintf(stderr, "Could not initialize the DPWSCore stack (err %d)\n\n", status);
		exit(1);
	}

	/* 2. Initialize the dpws structure for client operations. */
	// Initializes the dpws structure and performs one-time client-side initialisation.
	if ((status = dpws_client_init(&dpws, NULL) != DPWS_OK)) {
		fprintf(stderr, "Could not initialize client side of the DPWSCore stack (err %d)\n\n", status);
		exit(1);
	}

	/* 3. Configure the event listener. */

	// Configure DPWS server for HELLO and BYE messages reception
	DPWS_SET_PTR_ATT(DC_CACHE_HANDLE, DPWS_PTR_CALLBACK_HELLO, device_joining);
	DPWS_SET_PTR_ATT(DC_CACHE_HANDLE, DPWS_PTR_CALLBACK_BYE, device_leaving);
	DPWS_SET_INT_ATT(DC_REGISTRY_HANDLE, DPWS_INT_HTTP_PORT, port);	// Configure the HTTP listen port.

	event_endpoint = dpws_create_endpoint();	// Create endpoint for event reception.
	if (!event_endpoint) {
		fprintf(stderr, "Could not create endpoint...\n");
		goto error;
	}
	// Configure the functions handling incoming events
	DPWS_ADD_PTR_ATT(event_endpoint, DPWS_PTR_HANDLING_FUNCTION, lit_handle_event);
	DPWS_ADD_PTR_ATT(event_endpoint, DPWS_PTR_HANDLING_FUNCTION, wsh_handle_event);
	DPWS_ADD_PTR_ATT(event_endpoint, DPWS_PTR_CALLBACK_EVENT_END, subscription_cancelled);

	/* 4. Initialize the event listener. */
	if ((status = dpws_server_init(&listen_dpws, NULL)))	// initializes the dpws structure for server loop.
	{
		fprintf(stderr, "Could not initialize DPWS server loop (err %d)\n", status);
		goto error;
	}

	/* 5. Start the event listener loop. */
	// Creates a new thread to serve incoming messages. In the current sample,
	// a thread is created for each received message.
	if (bootServer(&listen_dpws)) {
		fprintf(stderr, "Could not boot server...\n");
		goto error;
	}

	/* 6. Loop on menu display depending on device supported types. */
	for (;;)
	{
		int nb_devices = 10;

		// 6.1 Find available devices.
		printf("<-- Looking for devices :\n with scope : %s\n", diversified_scope);
		device_proxies = dpws_lookup(&dpws, NULL, NULL, diversified_scope, &nb_devices);
		if (nb_devices == -1) {
			print_error("Could not execute dpws_lookup", &dpws);
			goto error;
		}
		else if (nb_devices == 0) {
			printf("\nNo Device found.\n");
			goto error;
		}

		// 6.2 Retrieve and display device info.
		printf("\n### Main Menu ###\n\n%d devices found.\n\n", nb_devices);
		for (i = 0; i < nb_devices; i++)
		{
			metadata = dpws_get_device_metadata(&dpws, device_proxies[i]);	// The dpws_get_device_metadata uses the WS-Transfer Get request
																			// and the WS-MetadataExchange message format with no caching.
			if (metadata) {
				printf("\n-> Index %d - %s\n   Model: %s\n", i,
						metadata->device_info.friendly_name,
						metadata->model_info.model_name);
				scopes = dpws_cache_get_scopes(&dpws, device_proxies[i]);
				for(;scopes && *scopes; scopes++)
					printf("   Scope: %s\n", *scopes);
			}
			else {
				printf("-> Device %s not reachable.\n", dpws_cache_get_uuid(&dpws, device_proxies[i]));
				fprintf(stderr, dpws_get_error_msg(&dpws));
			}
		}

		/* Select device */
		for (;;)
		{
			printf("\nEnter device index (or Q to exit): ");
			if (scanf("%d", &indexDev) == 0)
				goto error;
			printf("\n");
			if (indexDev >= 0 || indexDev < nbDevEndPt)
				break;
			printf("Incorrect device number\n");
		}

		/* Once the device chosen, release the device proxies except the chosen one. */
		device_proxy = device_proxies[indexDev];
		// Make sure the device proxy will not be erased from the cache using
		// dpws_pin_proxy because next dpws_end call will release the returned array
		// reference so the potential reception of BYE message (if a discovery
		// listener was running, which is not the case however in this sample) may
		// erase the proxy.
		if (dpws_pin_proxy(device_proxy) != DPWS_OK) {
			print_error("Could not execute dpws_pin_proxy", &dpws);
			goto error;
		}

		// Free dynamically allocated memory used for request processing and releases
		// proxies retrieved using a lookup API. This is good practice to call it
		// after every call to avoid memory peak.
		if (dpws_end(&dpws) != DPWS_OK) {
			print_error("Could not execute dpws_end", &dpws);
			goto error;
		}

		// 6.3 Call menu depending on device types.
		type.lname = SIMPLE_LIGHT_TYPE;
		if (dpws_device_proxy_match(device_proxy, &filter))
			simpleLightClient(&dpws, device_proxy);
		else
		{
			type.lname = DIMMER_LIGHT_TYPE;
			if (dpws_device_proxy_match(device_proxy, &filter))
				dimmerLightClient(&dpws, device_proxy, event_endpoint);
			else
			{
				type.lname = WASHER_TYPE;
				if (dpws_device_proxy_match(device_proxy, &filter))
					washerClient(&dpws, device_proxy, event_endpoint);
				else
				{
					type.lname = PROJECTOR_TYPE;
					if (dpws_device_proxy_match(device_proxy, &filter))
						projectorClient(&dpws, device_proxy);
					else
						printf("Unknown device type.");
				}
			}
		}
	}

error:
	// Releasing handle on device proxy.
	if (device_proxy >= 0) {
		if (dpws_release_proxy(device_proxy) != DPWS_OK) {
			print_error("Could not execute dpws_release_proxy", &dpws);
		}
	}
	if (dpws_end(&dpws) != DPWS_OK) {	// exited from the loop without dpws_end
		print_error("Could not execute dpws_end", &dpws);
		goto error;
	}

	/* 7. Stop the event listener. */

	stopServer();	// 1. Calls the dpws_stop_server function that closes all
	// open listening sockets, schedules the 'Eventing end' notifications
	// and Bye WS-Discovery messages for all the active devices and blocks
	// until all server processing is finished and the main loop blocking
	// on dpws_accept() can exit.
	// 2. Uses platform-specific thread APIs to wait for the main server
	// loop thread dies.

	// Cleans the DPWS stack.
	if ((status = dpws_shutdown()))
		fprintf(stderr, "Could not shut down the DPWSCore stack (err %d)\n\n", status);
}
