/*============================================================================*\
|                                                                              |
|                          SOA4D DPWSCore Samples                              |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|                                                                              |
|        + File info:                                                          |
|                     $Revision: 2153 $
|                     $Date: 2009-03-05 17:36:55 +0100 (jeu, 05 mar 2009) $
\*============================================================================*/

#include "dc/dc_Dpws.h"	// Main DPWSCore API include file.

#include "litStub.h"	// Generated stub & skeleton include file.
#include "lit.nsmap"	// Generated namespace table include file.

#define LIGHTING_NS "http://www.soa4d.org/DPWS/Samples/Home/Lighting"	// The namespace used to qualify port types (copy from wsdl file).
#define SWITCH_POWER_TYPE "SwitchPower"	// The local name of the "switch power" port type supported by the service (defined in the wsdl file).
#define DIMMING_TYPE "Dimming"	// The local name of the "dimmer" port type supported by the service (defined in the wsdl file).


/*
 *  Utilities
 */

// Variables for menu processing.
static char command[32], param1[32];
static int status, nbParams;

/* Function printing error message. */
static void print_error(char * msg, struct dpws * dpws) {
	fprintf(stderr, "\nError occured : %s.\n%s\n\n", msg, dpws_get_error_msg(dpws));
}

/* Function to get the washing machine service proxy for a given device proxy. */
static short getServiceProxy(struct dpws *dpws, short device_proxy, char* type)
{
	int nb_services = 1;	// One service required.
	short hService = -1;

	// The dpws_get_services retrieves one or several service proxies hosted by
	// a remote device using a type filtering (other APIs are available).
	// It will look for proxy available into the local cache or generate a
	// WS_Transfer/WS-MetadataExchange message if necessary.
	short * service_proxies = dpws_get_services(dpws, device_proxy, LIGHTING_NS, type, &nb_services);

	/* The program searches a single service. */
	if (nb_services != 1) {
		if (nb_services == 0)
			printf("No service endpoint found, please check the port type specifications.\n");
		else
			print_error("Could not execute dpws_get_services", dpws);
		goto exit;
	}

	// Make sure the service proxy will not be erased from the cache for instance
	// because of a bye message.
	if (dpws_pin_proxy(service_proxies[0]) != DPWS_OK) {
		print_error("Could not execute dpws_pin_proxy", dpws);
		goto exit;
	}
	hService = service_proxies[0];

exit:
	// Clears structures and free dynamically allocated memory used for request processing.
	if (dpws_end(dpws) != DPWS_OK)
		print_error("Could not execute dpws_end", dpws);

	return hService;
}

void simpleLightClient(struct dpws *dpws, short device_proxy)
{
	short service_proxy;
	struct wsa_endpoint_ref * servEndPt;

	/* Simple Light client program description (service invocation) */
	// 1. Find service proxy of the service to be invoked.
	// 2. Invocation loop
	//  2.1 Get the default invocation endpoint reference.
	//	2.2 Invoke the service.
	// 3. Release service proxy.

	/* 1. Find service proxy of the service to be invoked. */
	service_proxy = getServiceProxy(dpws, device_proxy, SWITCH_POWER_TYPE);
	if (service_proxy < 0)
		return;

	/* 2. Invocation loop */
	printf("\nSimpleLight device type commands (Q to quit this menu):\n\n");
	printf("\n Switch (ON|OFF)>\n");
	printf("\n\tSets the new light status.\n");
	printf("\tON  -> set the light status to ON.\n");
	printf("\tOFF -> set the light status to OFF.\n");
	printf("\n GetStatus\n\n\tRetrieves the current status of the light.\n");

	for (;;)
	{
		printf ("\nEnter command:\n");
		nbParams = scanf("%32s", command);
		if (nbParams == 0) {
			printf("Type mismatch !\n");
			continue;
		}

		/* 2.1 Get the default invocation endpoint reference. */

		// Retrieves the service proxy default endpoint reference (EPR) used for
		// invocation.
		// Several endpoints may be available and dpws_get_default_endpoint_ref
		// returns the first one. This is why one may have to use dpws_get_endpoint_refs
		// especially if the 1st EPR is not reachable.
		servEndPt = dpws_get_default_endpoint_reference(dpws, service_proxy);	// This is done every time because dpws_end has been called.
		if (!servEndPt) {
			print_error("Could not find default endpoint reference", dpws);
			return;
		}

		/* 2.2 Invoke the service. */

		// Command parsing & switch
		if (!strcasecmp(command, "Switch"))	// Switch light
		{
			nbParams = scanf("%32s", param1);
			if (nbParams != 1)
				printf("Bad parameter count !\n");
			else {
				enum lit__PowerState state = (strcasecmp(param1, "ON") ? lit__PowerState__OFF : lit__PowerState__ON);
				printf("<-- Switch light %s\n", param1);
				status = dpws_send___lit__Switch(dpws, servEndPt, state);
				if (status) {
					printf("Could not switch light : %s.\n",dpws_get_error_msg(dpws));
				}
			}
		}
		else if (!strcasecmp(command,"GetStatus"))	// get light state
		{
			enum lit__PowerState state;
			printf("<-- Getting light status ...\n");
			status = dpws_call___lit__GetStatus(dpws, servEndPt, NULL, &state);
			if (!status) {
				printf("--> Light status is %s\n", state == lit__PowerState__ON ? "ON" : "OFF");
			} else {
				printf("Could not retrieve light status : %s.\n",dpws_get_error_msg(dpws));
			}
		}
		else if (!strcasecmp(command, "Q"))	// Exit menu
		{
			printf("Exiting simple light menu...\n");
			break;
		}
		else
			printf("Unknown command.\n");

		if (status)
			fprintf(stderr, dpws_get_error_msg(dpws));

		// Clears structures and free dynamically allocated memory used for request processing.
		if (dpws_end(dpws) != DPWS_OK)
			print_error("Could not execute dpws_end", dpws);
	}

	/* 3. Release service proxy. */
	if (dpws_release_proxy(service_proxy) != DPWS_OK)
		print_error("Could not execute dpws_release_proxy", dpws);
}


void dimmerLightClient(struct dpws *dpws, short device_proxy, short event_endpoint)
{
	short service_proxy, event_sink;
	struct wsa_endpoint_ref * subscManager = NULL, * servEndPt;	// event subscription variables
	char *duration = "PT1H";	// 1 hour in 'XML'

	/* Dimmer light client program description (service invocation && event subscription) */
	// 1. Find service proxy of the event source service.
	// 2. Get the default invocation endpoint reference.
	// 3. Subscribe to the end of cycle event.
	// 4. Invocation loop
	//  4.1 Find service proxy of the service to be invoked.
	//  4.2 Get the default invocation endpoint reference.
	//	4.3 Invoke the service.
	//  4.4 Release service proxy.
	//  4.5 Free message processing memory.
	// 5. Unsubscribe to the end of cycle event.

	/* 1. Find service proxy of the event source service. */
	service_proxy = getServiceProxy(dpws, device_proxy, DIMMING_TYPE);
	if (service_proxy < 0)
		return;

	/* 2. Get the default invocation endpoint reference. */

	// Retrieves the service proxy default endpoint reference (EPR) used for
	// invocation.
	// Several endpoints may be available and dpws_get_default_endpoint_ref
	// returns the first one. This is why one may have to use dpws_get_endpoint_refs
	// especially if the 1st EPR is not reachable.
	servEndPt = dpws_get_default_endpoint_reference(dpws, service_proxy);
	if (!servEndPt) {
		print_error("Could not find default endpoint reference", dpws);
		return;
	}

	/* 3. Subscribe to the end of cycle event. */
	printf("<-- Subscribe to Dimming events for %s\n", duration);
 	event_sink = dpws_get_default_service_port(event_endpoint);

	// Subscribes to a WS-Eventing event source.
	// notify_to and end_to are the same local service port (i.e. network
 	// endpoints). Note that no action filter is used here.
 	subscManager = dpws_event_subscribe_ex(dpws, servEndPt, event_sink, event_sink, NULL, &duration);
	if (!subscManager) {
		print_error("Could not subscribe to Wash events", dpws);
		return;
	}

	// The endpoint reference (EPR) of the subscription manager is copied for
	// later reuse (to avoid release by next dpws_end). The subscription
	// manager EPR can be considered as a subscription ticket.
	subscManager = dpws_endpoint_ref_dup(subscManager);
	if (!subscManager) {
		print_error("Could not execute dpws_endpoint_ref_dup", dpws);
		return;
	}
	printf("--> Successful subscription to Dimming events for %s\n", duration);

	if (dpws_end(dpws) != DPWS_OK) {	// Good practice to call it after every call to avoid memory peak.
		print_error("Could not execute dpws_end", dpws);
		return;
	}

	/* 4. Invocation loop */

	printf("\nDimmerLight device type commands (Q to quit this menu):\n\n");
	printf("\n Switch (ON|OFF)>\n");
	printf("\n\tSets the light to a given status.\n");
	printf("\tON  -> Sets the lights status to ON.\n");
	printf("\tOFF  -> Sets the lights status to OFF.\n");
	printf("\n GetStatus\n");
	printf("\n\tRetrieves the current status of the light.\n");
	printf("\n SetLevel <%% of power> <duration (s)\n");
	printf("\n\tSets the new level of the dimming light and the duration it should take to reach it.\n");
	printf("\n GetLevel\n");
	printf("\n\tRetrieves the current level of the light.\n");
	printf("\n GetTarget\n");
	printf("\n\tRetrieves the current target level of the light.\n");

	for (;;)
	{
		printf ("\nEnter command:\n");
		nbParams = scanf("%32s", command);
		if (nbParams == 0) {
			printf("Type mismatch !\n");
			continue;
		}

		// Command parsing & switch
		if (!strcasecmp(command, "Switch"))	// Switch light
		{
			nbParams = scanf("%32s", param1);
			if (nbParams != 1)
				printf("Bad parameter count !\n");
			else
			{
				enum lit__PowerState state = (strcasecmp(param1, "ON") ? lit__PowerState__OFF : lit__PowerState__ON);

				//  4.1 Find service proxy of the service to be invoked.
				service_proxy = getServiceProxy(dpws, device_proxy, SWITCH_POWER_TYPE);
				if (service_proxy < 0)
					return;

				//  4.2 Get the default invocation endpoint reference.
				servEndPt = dpws_get_default_endpoint_reference(dpws, service_proxy);	// Last was freed by dpws_end
				if (!servEndPt) {
					print_error("Could not find default endpoint reference", dpws);
					return;
				}
				//	4.3 Invoke the service.
				printf("<-- Switch light %s\n", param1);
				status = dpws_send___lit__Switch(dpws, servEndPt, state);
				if (status) {
					printf("Could not switch light : %s.\n",dpws_get_error_msg(dpws));
				}
			}
		}
		else if (!strcasecmp(command, "GetStatus"))	// get light state
		{
			enum lit__PowerState state;

			//  4.1 Find service proxy of the service to be invoked.
			service_proxy = getServiceProxy(dpws, device_proxy, SWITCH_POWER_TYPE);
			if (service_proxy < 0)
				return;

			//  4.2 Get the default invocation endpoint reference.
			servEndPt = dpws_get_default_endpoint_reference(dpws, service_proxy);	// Last was freed by dpws_end
			if (!servEndPt) {
				print_error("Could not find default endpoint reference", dpws);
				return;
			}
			//	4.3 Invoke the service.
			printf("<-- Getting light status ...\n");
			status = dpws_call___lit__GetStatus(dpws, servEndPt, NULL, &state);
			if (!status)
				printf("--> Light status is %s\n", state == lit__PowerState__ON ? "ON" : "OFF");
			else
				printf("Could not retrieve light status : %s.\n",dpws_get_error_msg(dpws));
		}
		else if (!strcasecmp(command, "SetLevel"))	// set dimmer level
		{
			int level, duration;
			nbParams = scanf("%d %d", &level, &duration);
			if (nbParams < 1 || nbParams > 2)
				printf("Bad parameter count !\n");
			else
			{
				struct _lit__LightLevelTarget levelTarget = {level, NULL};
				if (duration != 0)
					levelTarget.TransitionDuration = (short *)&duration;

				//  4.1 Find service proxy of the service to be invoked.
				service_proxy = getServiceProxy(dpws, device_proxy, DIMMING_TYPE);
				if (service_proxy < 0)
					return;

				//  4.2 Get the default invocation endpoint reference.
				servEndPt = dpws_get_default_endpoint_reference(dpws, service_proxy);	// Last was freed by dpws_end
				if (!servEndPt) {
					print_error("Could not find default endpoint reference", dpws);
					return;
				}
				//	4.3 Invoke the service.
				printf("<-- Set the light level to %d%% in %d seconds\n", level, duration);
				status = dpws_send___lit__SetLevel(dpws, servEndPt, &levelTarget);
				if (status)
					printf("Could not set light level : %s.\n",dpws_get_error_msg(dpws));
			}
		}
		else if (!strcasecmp(command,"GetLevel"))	// retrieve dimmer level
		{
			short level;

			//  4.1 Find service proxy of the service to be invoked.
			service_proxy = getServiceProxy(dpws, device_proxy, DIMMING_TYPE);
			if (service_proxy < 0)
				return;

			//  4.2 Get the default invocation endpoint reference.
			servEndPt = dpws_get_default_endpoint_reference(dpws, service_proxy);	// Last was freed by dpws_end
			if (!servEndPt) {
				print_error("Could not find default endpoint reference", dpws);
				return;
			}
			//	4.3 Invoke the service.
			printf("<-- Getting light level ...\n");
			status = dpws_call___lit__GetLevel(dpws, servEndPt, NULL, &level);
			if (!status)
				printf("--> The light level is %d%%\n", level);
			else
				printf("Could not retrieve light level : %s.\n",dpws_get_error_msg(dpws));
		}
		else if (!strcasecmp(command,"GetTarget"))	// retrieve dimmer target level
		{
			short target;

			//  4.1 Find service proxy of the service to be invoked.
			service_proxy = getServiceProxy(dpws, device_proxy, DIMMING_TYPE);
			if (service_proxy < 0)
				return;

			//  4.2 Get the default invocation endpoint reference.
			servEndPt = dpws_get_default_endpoint_reference(dpws, service_proxy);	// Last was freed by dpws_end
			if (!servEndPt) {
				print_error("Could not find default endpoint reference", dpws);
				return;
			}
			//	4.3 Invoke the service.
			printf("<-- Getting light target level ...\n");
			status = dpws_call___lit__GetLevelTarget(dpws, servEndPt, NULL, &target);
			if (!status)
				printf("--> The light target level is %d%%\n", target);
			else
				printf("Could not retrieve light target level : %s.\n",dpws_get_error_msg(dpws));
		}
		else if (!strcasecmp(command, "Q"))	// Exit menu
		{
			printf("Exiting dimmer menu...\n");
			break;
		}
		else
			printf("Unknown command.\n");

		if (status)
			fprintf(stderr, dpws_get_error_msg(dpws));

		/*  4.5 Release service proxy. */
		if (dpws_release_proxy(service_proxy) != DPWS_OK)
			print_error("Could not execute dpws_release_proxy", dpws);

		/*  4.6 Free message processing memory. */
		if (dpws_end(dpws) != DPWS_OK)
			print_error("Could not execute dpws_end", dpws);
	}

	/* 5. Unsubscribe to the end of cycle event. */
	printf("<-- Unsubscribe to Dimmer events.\n");

	// The dpws_event_unsubscribe function cancels a running subscription to a
	// WS-Eventing event source.
	if (dpws_event_unsubscribe(dpws, subscManager) == DPWS_OK)
		printf("--> Successful unsubscription to Dimmer events.\n");
	else
		print_error("Could not execute dpws_event_unsubscribe (may be normal if server has been stopped first)", dpws);

	dpws_endpoint_ref_free(subscManager);	// Free subscription manager EPR.
}

/*
 *                    WS-Eventing handlers implementation
 */

/* WS-Eventing handler */
int __lit__TargetReached(struct dpws* dpws, short lit__LightLevel)
{
	printf("--> Event 'TargetReached' received ! Level = %d\n", lit__LightLevel);
	return SOAP_OK;
}

/*
IMPORTANT NOTE:
	The following handler is not a WS-Eventing handler but a handler for asynchronous
	responses to request-reply invocations. It must however  be implemented even if
	not used since referenced by the same dispatch function.
 */
int __lit__GetStatus_handler(struct dpws* dpws, enum lit__PowerState *a)
{
	return SOAP_OK;
}

int __lit__GetLevelTarget_handler(struct dpws* dpws, short *a)
{
	return SOAP_OK;
}

int __lit__GetLevel_handler(struct dpws* dpws, short *a)
{
	return SOAP_OK;
}

