/*
 *  CUnit - A Unit testing framework library for C.
 *  Copyright (C) 2001  Anil Kumar
 *  Copyright (C) 2004  Anil Kumar, Jerry St.Clair
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 *  Contains the Console Test Interface  implementation.
 *
 *  Created By     : Anil Kumar on ...(in month of Aug 2001)
 *  Last Modified  : 19/Aug/2001
 *  Comment        : Added initial console interface functions without
 *                   any run functionality.
 *  Email          : aksaharan@yahoo.com
 *
 *  Modified       : 24/Aug/2001 by Anil Kumar
 *  Comment        : Added compare_strings, show_errors, list_suites,
 *                   list_tests function declarations.
 *  Email          : aksaharan@yahoo.com
 *
 *  Modified       : 17-Jul-2004 (JDS)
 *  Comment        : New interface, doxygen comments, reformat console output
 *  Email          : jds2@users.sourceforge.net
 */

/** @file
 * Console test interface with interactive output (implementation).
 */
/** @addtogroup Console
 @{
*/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <assert.h>
#include <string.h>
#include <stdarg.h>

#include "cunit_.h"

void CU_trim_left(char* szString);
void CU_trim_right(char* szString);

/** Console interface status flag. */
typedef enum
{
  CONTINUE = 1,   /**< Continue processing commands in current menu. */
  MOVE_UP,        /**< Move up to the previous menu. */
  STOP            /**< Stop processing (user selected 'Quit'). */
} STATUS;

/** currently running suite. */
static CU_pSuite f_pRunningSuite = NULL;

/* Forward declaration of module functions */
static void console_registry_level_run(CU_pTestRegistry pRegistry);
static STATUS console_suite_level_run(CU_pSuite pSuite);

static CU_ErrorCode console_run_all_tests(CU_pTestRegistry pRegistry);
static CU_ErrorCode console_run_suite(CU_pSuite pSuite);
static CU_ErrorCode console_run_single_test(CU_pSuite pSuite, CU_pTest pTest);

static void console_test_start_message_handler(const CU_pTest pTest, const CU_pSuite pSuite);
static void console_test_complete_message_handler(const CU_pTest pTest, const CU_pSuite pSuite, const CU_pFailureRecord pFailure);
static void console_all_tests_complete_message_handler(const CU_pFailureRecord pFailure);
static void console_suite_init_failure_message_handler(const CU_pSuite pSuite);

static CU_ErrorCode select_test(CU_pSuite pSuite, CU_pTest* pTest);
static CU_ErrorCode select_suite(CU_pTestRegistry pRegistry, CU_pSuite* pSuite);

static void list_suites(CU_pTestRegistry pRegistry);
static void list_tests(CU_pSuite pSuite);
static void show_failures(void);

/*------------------------------------------------------------------------*/
/** Run registered CUnit tests using the console interface. */
void CU_console_run_tests(void)
{
  CU_pTestRegistry pRegistry = CU_get_registry();

  /*
   *   To avoid user from cribbing about the output not coming onto
   *   screen at the moment of SIGSEGV.
   */
  setvbuf(stdout, NULL, _IONBF, 0);
  setvbuf(stderr, NULL, _IONBF, 0);

  fprintf(stdout, "\n\n     CUnit : A Unit testing framework for C."
                  "\n     http://cunit.sourceforge.net/\n\n");

  if (NULL == pRegistry) {
    fprintf(stderr, "\n\nFATAL ERROR - Test registry is not initialized.\n");
    CU_set_error(CUE_NOREGISTRY);
  }
  else {
    CU_set_test_start_handler(console_test_start_message_handler);
    CU_set_test_complete_handler(console_test_complete_message_handler);
    CU_set_all_test_complete_handler(console_all_tests_complete_message_handler);
    CU_set_suite_init_failure_handler(console_suite_init_failure_message_handler);

    console_registry_level_run(pRegistry);
  }
}

/*------------------------------------------------------------------------*/
/** Main loop for console interface.
 * Displays actions and responds based on user imput.
 * @param pRegistry The CU_pTestRegistry to use for testing (non-NULL).
 */
static void console_registry_level_run(CU_pTestRegistry pRegistry)
{
  int chChoice;
  char szTemp[256];
  CU_pSuite pSuite = NULL;
  STATUS eStatus = CONTINUE;

  assert(pRegistry);

  while (CONTINUE == eStatus)
  {
    fprintf(stdout, "\n*************** CUNIT CONSOLE - MAIN MENU ***********************"
                    "\n(R)un all, (S)elect suite, (L)ist suites, Show (F)ailures, (Q)uit"
                    "\nEnter Command : ");
    chChoice = getchar();
    fgets(szTemp, sizeof(szTemp), stdin);

    switch (tolower(chChoice)) {
      case 'r':
        console_run_all_tests(pRegistry);
        break;

      case 's':
        if (CUE_SUCCESS == select_suite(pRegistry, &pSuite)) {
          if (STOP == console_suite_level_run(pSuite))
            eStatus = STOP;
        }
        else {
          fprintf(stdout, "\nSuite not found.\n");
        }
        break;

      case 'l':
        list_suites(pRegistry);
        break;

      case 'f':
        show_failures();
        break;

      case 'q':
        eStatus = STOP;
        break;

      /* To stop gcc from cribbing */
      default:
        break;
    }
  }
}

/*------------------------------------------------------------------------*/
/** Run a selected suite within the console interface.
 * Displays actions and responds based on user imput.
 * @param pSuite The suite to use for testing (non-NULL).
 */
static STATUS console_suite_level_run(CU_pSuite pSuite)
{
  int chChoice;
  char szTemp[256];
  CU_pTest pTest = NULL;
  STATUS eStatus = CONTINUE;

  assert(pSuite);

  while (CONTINUE == eStatus) {

    fprintf(stdout, "\n*************** CUNIT CONSOLE - SUITE MENU *******************************"
                    "\n(R)un All, (S)elect test, (L)ist tests, Show (F)ailures, (M)ove up, (Q)uit"
                    "\nEnter Command : ");
    chChoice = getchar();
    fgets(szTemp, sizeof(szTemp), stdin);

    switch (tolower(chChoice)) {
      case 'r':
        console_run_suite(pSuite);
        break;

      case 's':
        if (CUE_SUCCESS == select_test(pSuite, &pTest)) {
          console_run_single_test(pSuite, pTest);
        }
        break;

      case 'l':
        list_tests(pSuite);
        break;

      case 'f':
        show_failures();
        break;

      case 'm':
        eStatus = MOVE_UP;
        break;

      case 'q':
        eStatus = STOP;
        break;

      /* To stop gcc from cribbing */
      default:
        break;
    }
  }
  return eStatus;
}

/*------------------------------------------------------------------------*/
/** Run all tests within the console interface.
 * The test registry is changed to the specified registry
 * before running the tests, and reset to the original
 * registry when done.
 * @param pRegistry The CU_pTestRegistry containing the tests
 *                  to be run (non-NULL).
 * @return An error code indicating the error status
 *         during the test run.
 */
static CU_ErrorCode console_run_all_tests(CU_pTestRegistry pRegistry)
{
  CU_pTestRegistry pOldRegistry = NULL;
  CU_ErrorCode result;

  assert(pRegistry);

  f_pRunningSuite = NULL;

  pOldRegistry = CU_set_registry(pRegistry);
  result = CU_run_all_tests();
  CU_set_registry(pOldRegistry);
  return result;
}

/*------------------------------------------------------------------------*/
/** Run a specified suite within the console interface.
 * @param pSuite The suite to be run (non-NULL).
 * @return An error code indicating the error status
 *         during the test run.
 */
static CU_ErrorCode console_run_suite(CU_pSuite pSuite)
{
  f_pRunningSuite = NULL;
  return CU_run_suite(pSuite);
}

/*------------------------------------------------------------------------*/
/** Run a specific test for the specified suite within 
 * the console interface.
 * @param pSuite The suite containing the test to be run (non-NULL).
 * @param pTest  The test to be run (non-NULL).
 * @return An error code indicating the error status
 *         during the test run.
 */
static CU_ErrorCode console_run_single_test(CU_pSuite pSuite, CU_pTest pTest)
{
  f_pRunningSuite = NULL;
  return CU_run_test(pSuite, pTest);
}

/*------------------------------------------------------------------------*/
/** Read the name of a test from standard input and
 * locate the test having the specified name.
 * A pointer to the located test is stored in pTest
 * upon return.
 * @param pSuite The suite to be queried.
 * @param ppTest Pointer to location to store the selected test.
 * @return CUE_SUCCESS if a test was successfully selected,
 *         CUE_NOTEST otherwise.  On return, ppTest points
 *         to the test selected.
 */
static CU_ErrorCode select_test(CU_pSuite pSuite, CU_pTest* ppTest)
{
  char szTestName[MAX_TEST_NAME_LENGTH];

  fprintf(stdout,"\nEnter Test Name : ");
  fgets(szTestName, MAX_TEST_NAME_LENGTH, stdin);
  sscanf(szTestName, "%[^\n]s", szTestName);

  *ppTest = CU_get_test_by_name(szTestName, pSuite);

  return (*ppTest) ? CUE_SUCCESS : CUE_NOTEST;
}

/*------------------------------------------------------------------------*/
/** Read the name of a suite from standard input and
 * locate the suite having the specified name.
 * The input string is used to locate the suite having the
 * indicated name, which is returned in pSuite.  ppSuite will
 * be NULL if there is no suite registered in pRegistry having
 * the input name.  Returns NULL if the suite is successfully
 * located, non-NULL otherwise.
 * @param pRegistry The CU_pTestRegistry to query (non-NULL).
 * @param ppSuite Pointer to location to store the selected suite.
 * @return CUE_SUCCESS if a suite was successfully selected,
 *         CUE_NOSUITE otherwise.  On return, ppSuite points
 *         to the suite selected.
 */
static CU_ErrorCode select_suite(CU_pTestRegistry pRegistry, CU_pSuite* ppSuite)
{
  char szSuiteName[MAX_SUITE_NAME_LENGTH];

  assert(pRegistry);

  fprintf(stdout,"\n\nEnter Suite Name : ");
  fgets(szSuiteName, MAX_SUITE_NAME_LENGTH, stdin);
  sscanf(szSuiteName, "%[^\n]s", szSuiteName);

  *ppSuite = CU_get_suite_by_name(szSuiteName, pRegistry);

  return (*ppSuite) ? CUE_SUCCESS : CUE_NOSUITE;
}

/*------------------------------------------------------------------------*/
/** List the suites in a registry to standard output.
 * @param pRegistry The CU_pTestRegistry to query (non-NULL).
 */
static void list_suites(CU_pTestRegistry pRegistry)
{
  CU_pSuite pCurSuite = NULL;
  int i;

  assert(pRegistry);
  if (0 == pRegistry->uiNumberOfSuites) {
    fprintf(stdout, "\nNo suites registered.\n");
    return;
  }

  assert(pRegistry->pSuite);

  fprintf(stdout, "\n--------------------- Registered Suites --------------------------");
  fprintf(stdout, "\n     Suite Name                          Init?  Cleanup?  # Tests\n");

  for (i = 1, pCurSuite = pRegistry->pSuite; pCurSuite; pCurSuite = pCurSuite->pNext, ++i) {
    fprintf(stdout, "\n%3d. %-34.33s   %3s     %3s       %3d",
            i,
            pCurSuite->pName,
            pCurSuite->pInitializeFunc ? "YES" : "NO",
            pCurSuite->pCleanupFunc ? "YES" : "NO",
            pCurSuite->uiNumberOfTests);
  }
  fprintf(stdout, "\n------------------------------------------------------------------"
                  "\nTotal Number of Suites : %-d\n", pRegistry->uiNumberOfSuites);
}

/*------------------------------------------------------------------------*/
/** List the tests in a suite to standard output.
 * @param pSuite The suite to query (non-NULL).
 */
static void list_tests(CU_pSuite pSuite)
{
  CU_pTest pCurTest = NULL; 
  unsigned int uiCount;

  assert(pSuite);
  if (0 == pSuite->uiNumberOfTests) {
    fprintf(stdout, "\nSuite %s contains no tests.\n", pSuite->pName);
    return;
  }

  assert(pSuite->pTest);

  fprintf(stdout, "\n--------------- Test List ---------------------------------");
  fprintf(stdout, "\n      Test Names (Suite: %s)\n", pSuite->pName);

  for (uiCount = 1, pCurTest = pSuite->pTest; pCurTest; uiCount++, pCurTest = pCurTest->pNext) {
    fprintf(stdout, "\n%3d.  %s", uiCount, pCurTest->pName);
  }
  fprintf(stdout, "\n-----------------------------------------------------------"
                  "\nTotal Number of Tests : %-d\n", pSuite->uiNumberOfTests);
}

/*------------------------------------------------------------------------*/
/** Display the record of test failures on standard output. */
static void show_failures(void)
{
  int i;
  CU_pFailureRecord pFailure = CU_get_failure_list();

  if (NULL == pFailure) {
    fprintf(stdout, "\nNo failures.\n");
  }
  else {

    fprintf(stdout, "\n--------------- Test Run Failures -------------------------");
    fprintf(stdout, "\n   src_file:line# : (suite:test) : failure_condition\n");

    for (i = 1 ; pFailure ; pFailure = pFailure->pNext, i++) {
      fprintf(stdout, "\n%d. %s:%d : (%s : %s) : %s", i,
          pFailure->strFileName ? pFailure->strFileName : "",
          pFailure->uiLineNumber,
          pFailure->pSuite ? pFailure->pSuite->pName : "",
          pFailure->pTest ? pFailure->pTest->pName : "",
          pFailure->strCondition ? pFailure->strCondition : "");
    }
    fprintf(stdout, "\n-----------------------------------------------------------"
                    "\nTotal Number of Failures : %-d\n", i - 1);
  }
}

/*------------------------------------------------------------------------*/
/** Handler function called at start of each test.
 * @param pTest  The test being run.
 * @param pSuite The suite containing the test.
 */
static void console_test_start_message_handler(const CU_pTest pTest, const CU_pSuite pSuite)
{
  assert(pTest);
  assert(pSuite);
  
  /* Comparing the Addresses rather than the Group Names. */
  if (!f_pRunningSuite || f_pRunningSuite != pSuite) {
    fprintf(stdout, "\nRunning Suite : %s", pSuite->pName);
    fprintf(stdout, "\n\tRunning test : %s", pTest->pName);
    f_pRunningSuite = pSuite;
  }
  else {
    fprintf(stdout, "\n\tRunning test : %s", pTest->pName);
  }
}

/*------------------------------------------------------------------------*/
/** Handler function called at completion of each test.
 * @param pTest   The test being run.
 * @param pSuite  The suite containing the test.
 * @param pFailure Pointer to the 1st failure record for this test.
 */
static void console_test_complete_message_handler(const CU_pTest pTest, const CU_pSuite pSuite, const CU_pFailureRecord pFailure)
{
	/*
	*   For console interface do nothing. This is useful only for the test
	*   interface where UI is involved.  Just silence compiler warnings.
	*/
	(void)pTest;
	(void)pSuite;
	(void)pFailure;


	if( pFailure != NULL ){
		fprintf( stdout, " *" );
	}
}

/*------------------------------------------------------------------------*/
/** Handler function called at completion of all tests in a suite.
 * @param pFailure Pointer to the test failure record list.
 */
static void console_all_tests_complete_message_handler(const CU_pFailureRecord pFailure)
{
	CU_pRunSummary pRunSummary = CU_get_run_summary();
	CU_pTestRegistry pRegistry = CU_get_registry();

	(void)pFailure; /* not used in console interface - silence warning */

	assert(pRunSummary);
	assert(pRegistry);

	fprintf(stdout,"\n\n--Run Summary: Type      Total     Ran  Passed  Failed"
				   "\n               suites %8u%8u     n/a%8u"
				   "\n               tests  %8u%8u%8u%8u"
				   "\n               asserts%8u%8u%8u%8u\n",
          pRegistry->uiNumberOfSuites,
          pRunSummary->nSuitesRun,
          pRunSummary->nSuitesFailed,
          pRegistry->uiNumberOfTests,
          pRunSummary->nTestsRun,
          pRunSummary->nTestsRun - pRunSummary->nTestsFailed,
          pRunSummary->nTestsFailed,
          pRunSummary->nAsserts,
          pRunSummary->nAsserts,
          pRunSummary->nAsserts - pRunSummary->nAssertsFailed,
          pRunSummary->nAssertsFailed);

	if( pRunSummary->nSuitesFailed  == 0 &&
		pRunSummary->nTestsFailed   == 0 &&
		pRunSummary->nAssertsFailed == 0 ){
		fprintf(stdout,
			"\n OOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOO"
			"\n O                                  O"
			"\n O  GGGG   OOOO   OOOO  DDDD    !!  O"
			"\n O G      O    O O    O D   DD  !!  O"
			"\n O G  GGG O    O O    O D    D  !!  O"
			"\n O G    G O    O O    O D   D       O"
			"\n O  GGGG   OOOO   OOOO  DDDD    !!  O"
			"\n O                                  O"
			"\n OOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOO"
			"\n" );
	} else {
		fprintf(stdout,
			"\n XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX"
			"\n X                                  X"
			"\n X BBBBB   AAAA  DDDD    XX      XX X"
			"\n X B    B A    A D   DD    XX  XX   X"
			"\n X BBBBB  AAAAAA D    D      XX     X"
			"\n X B    B A    A D   DD    XX  XX   X"
			"\n X BBBBBB A    A DDDD    XX      XX X"
			"\n X                                  X"
			"\n XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX"
			"\n" );
	}
}
 
/*------------------------------------------------------------------------*/
/** Handler function called when suite initialization fails.
 * @param pSuite The suite for which initialization failed.
 */
static void console_suite_init_failure_message_handler(const CU_pSuite pSuite)
{
  assert(pSuite);

  fprintf(stdout, "\nWARNING - Suite initialization failed for %s.",
          pSuite->pName);
}

/** @} */



/** Structure containing mappings of special characters to
 * xml entity codes.
 */
static const struct {
	char special_char;
	char* replacement;
} bindings [] = {
    {'&', "&amp;"},
    {'>', "&gt;"},
    {'<', "&lt;"}
};

/*------------------------------------------------------------------------*/
/** Check whether a character is a special xml character.
 * This function performs a lookup of the specified character in
 * the bindings structure.  If it is a special character, its
 * index into the bindings array is returned.  If not, -1 is returned.
 * @param ch The character to check
 * @return Index into bindings if a special character, -1 otherwise.
 */
static int get_index(char ch)
{
	int length = sizeof(bindings)/sizeof(bindings[0]);
	int counter;

	for (counter = 0; counter < length && bindings[counter].special_char != ch; ++counter)
		;

	return (counter < length ? counter : -1);
}

/*------------------------------------------------------------------------*/
/** Convert special characters in the specified string to
 * xml entity codes.  The character-entity mappings are
 * contained in the struct bindings.  Note that conversion
 * to entities increases the length of the converted string.
 * The worst case conversion would be a string consisting
 * entirely of entity characters of length CUNIT_MAX_ENTITY_LEN.
 * If szDest does not have enough room to convert an entity,
 * it will not be converted.  It is the caller's responsibility
 * to make sure there is sufficient room in szDest to hold the
 * converted string.
 * @param szSrc  Source string to convert (non-NULL).
 * @param szDest Location to hold the converted string (non-NULL).
 * @param maxlen Maximum number of characters szDest can hold.
 * @return  The number of special characters converted.
 * @todo Consider a function calculating and returning the
 *       converted length of a given string.
 */
int CU_translate_special_characters(const char* szSrc, char* szDest, int maxlen)
{
	int count = 0;
	int src = 0;
	int dest = 0;
	int length = (int)strlen(szSrc);
	int conv_index;

  assert(szSrc);
  assert (szDest);

	memset(szDest, 0, maxlen);
	while ((dest < maxlen) && (src < length)) {

		if ((-1 != (conv_index = get_index(szSrc[src]))) &&
        ((dest + (int)strlen(bindings[conv_index].replacement)) <= maxlen)) {
			strcat(szDest, bindings[conv_index].replacement);
			dest += (int)strlen(bindings[conv_index].replacement);
			++count;
		} else {
			szDest[dest++] = szSrc[src];
		}

		++src;
	}

	return count;
}

/*------------------------------------------------------------------------*/
/** Case-insensitive string comparison.
 * @param szSrc  1st string to compare.
 * @param szDest 2nd string to compare.
 * @return  NULL if the strings are equal, non-NULL otherwise.
 */
int CU_compare_strings(const char* szSrc, const char* szDest)
{
	while (*szSrc && *szDest && toupper(*szSrc) == toupper(*szDest)) {
		szSrc++;
		szDest++;
	}

	return *szSrc - *szDest;
}

/*------------------------------------------------------------------------*/
/** Trim leading and trailing whitespace from the specified string.
 * @param szString  The string to trim.
 */
void CU_trim(char* szString)
{
	CU_trim_left(szString);
	CU_trim_right(szString);
}

/*------------------------------------------------------------------------*/
/** Trim leading whitespace from the specified string.
 * @param szString  The string to trim.
 */
void CU_trim_left(char* szString)
{
	int nOffset = 0;
	char* szSrc = szString;
	char* szDest = szString;

	assert(szString);

	/* Scan for the spaces in the starting of string. */
	for (; *szSrc; szSrc++, nOffset++) {
		if (!isspace(*szSrc))
			break;
	}

	for(; nOffset && (0 != (*szDest = *szSrc)); szSrc++, szDest++)
		;
}

/*------------------------------------------------------------------------*/
/** Trim trailing whitespace from the specified string.
 * @param szString  The string to trim.
 */
void CU_trim_right(char* szString)
{
	size_t nLength;
	char* szSrc = szString;

	assert(szString);
	nLength = strlen(szString);
	/*
	 * Scan for specs in the end of string.
	 */
	for (; nLength && isspace(*(szSrc + nLength - 1)); nLength--)
		;

	*(szSrc + nLength) = '\0';
}

/** @} */

#ifdef CUNIT_BUILD_TESTS
#include "test_cunit.h"

#define MAX_LEN 100

static void test_CU_translate_special_characters(void)
{
  int nchars;
  char dest[MAX_LEN];

  /* empty src */
  strcpy(dest, "random initialized string");
  nchars = CU_translate_special_characters("", dest, MAX_LEN);
  TEST(0 == nchars);
  TEST(!strcmp(dest, ""));

  /* 1 char src */
	memset(dest, 0, MAX_LEN);
  nchars = CU_translate_special_characters("#", dest, 1);
  TEST(0 == nchars);
  TEST(!strcmp(dest, "#"));

	memset(dest, 0, MAX_LEN);
  nchars = CU_translate_special_characters("&", dest, 1);
  TEST(0 == nchars);
  TEST(!strcmp(dest, "&"));

	memset(dest, 0, MAX_LEN);
  nchars = CU_translate_special_characters("&", dest, 4);
  TEST(0 == nchars);
  TEST(!strcmp(dest, "&"));

	memset(dest, 0, MAX_LEN);
  nchars = CU_translate_special_characters("&", dest, 5);
  TEST(1 == nchars);
  TEST(!strcmp(dest, "&amp;"));

  /* maxlen=0 */
  strcpy(dest, "random initialized string");
  nchars = CU_translate_special_characters("some <<string & another>", dest, 0);
  TEST(0 == nchars);
  TEST(!strcmp(dest, "random initialized string"));

  /* maxlen < len(converted szSrc) */
	memset(dest, 0, MAX_LEN);
  nchars = CU_translate_special_characters("some <<string & another>", dest, 1);
  TEST(0 == nchars);
  TEST(!strcmp(dest, "s"));

	memset(dest, 0, MAX_LEN);
  nchars = CU_translate_special_characters("some <<string & another>", dest, 6);
  TEST(0 == nchars);
  TEST(!strcmp(dest, "some <"));

	memset(dest, 0, MAX_LEN);
  nchars = CU_translate_special_characters("some <<string & another>", dest, 9);
  TEST(1 == nchars);
  TEST(!strcmp(dest, "some &lt;"));

	memset(dest, 0, MAX_LEN);
  nchars = CU_translate_special_characters("some <<string & another>", dest, 10);
  TEST(1 == nchars);
  TEST(!strcmp(dest, "some &lt;<"));

	memset(dest, 0, MAX_LEN);
  nchars = CU_translate_special_characters("some <<string & another>", dest, 13);
  TEST(2 == nchars);
  TEST(!strcmp(dest, "some &lt;&lt;"));

	memset(dest, 0, MAX_LEN);
  nchars = CU_translate_special_characters("some <<string & another>", dest, 14);
  TEST(2 == nchars);
  TEST(!strcmp(dest, "some &lt;&lt;s"));

	memset(dest, 0, MAX_LEN);
  nchars = CU_translate_special_characters("some <<string & another>", dest, 21);
  TEST(2 == nchars);
  TEST(!strcmp(dest, "some &lt;&lt;string &"));

	memset(dest, 0, MAX_LEN);
  nchars = CU_translate_special_characters("some <<string & another>", dest, 22);
  TEST(2 == nchars);
  TEST(!strcmp(dest, "some &lt;&lt;string & "));

	memset(dest, 0, MAX_LEN);
  nchars = CU_translate_special_characters("some <<string & another>", dest, 23);
  TEST(2 == nchars);
  TEST(!strcmp(dest, "some &lt;&lt;string & a"));

	memset(dest, 0, MAX_LEN);
  nchars = CU_translate_special_characters("some <<string & another>", dest, 24);
  TEST(2 == nchars);
  TEST(!strcmp(dest, "some &lt;&lt;string & an"));

	memset(dest, 0, MAX_LEN);
  nchars = CU_translate_special_characters("some <<string & another>", dest, 25);
  TEST(3 == nchars);
  TEST(!strcmp(dest, "some &lt;&lt;string &amp;"));

	memset(dest, 0, MAX_LEN);
  nchars = CU_translate_special_characters("some <<string & another>", dest, 26);
  TEST(3 == nchars);
  TEST(!strcmp(dest, "some &lt;&lt;string &amp; "));

  /* maxlen == len(converted szSrc) */
	memset(dest, 0, MAX_LEN);
  nchars = CU_translate_special_characters("some <<string & another>", dest, 37);
  TEST(4 == nchars);
  TEST(!strcmp(dest, "some &lt;&lt;string &amp; another&gt;"));

  /* maxlen > len(converted szSrc) */
	memset(dest, 0, MAX_LEN);
  nchars = CU_translate_special_characters("some <<string & another>", dest, MAX_LEN);
  TEST(4 == nchars);
  TEST(!strcmp(dest, "some &lt;&lt;string &amp; another&gt;"));

  /* no special characters */
	memset(dest, 0, MAX_LEN);
  nchars = CU_translate_special_characters("some string or another", dest, MAX_LEN);
  TEST(0 == nchars);
  TEST(!strcmp(dest, "some string or another"));

  /* only special characters */
	memset(dest, 0, MAX_LEN);
  nchars = CU_translate_special_characters("<><><<>>&&&", dest, MAX_LEN);
  TEST(11 == nchars);
  TEST(!strcmp(dest, "&lt;&gt;&lt;&gt;&lt;&lt;&gt;&gt;&amp;&amp;&amp;"));
}

static void test_CU_compare_strings(void)
{
  TEST(0 == CU_compare_strings("",""));
  TEST(0 == CU_compare_strings("@","@"));
  TEST(0 == CU_compare_strings("D","d"));
  TEST(0 == CU_compare_strings("s1","s1"));
  TEST(0 == CU_compare_strings("s1","S1"));
  TEST(0 != CU_compare_strings("s1","s12"));
  TEST(0 == CU_compare_strings("this is string 1","tHIS iS sTRING 1"));
  TEST(0 == CU_compare_strings("i have \t a tab!","I have \t a tab!"));
  TEST(0 != CU_compare_strings("not the same"," not the same"));
}

static void test_CU_trim(void)
{
  char string[MAX_LEN];

  strcpy(string, "");
  CU_trim(string);
  TEST(!strcmp("", string));

  strcpy(string, " ");
  CU_trim(string);
  TEST(!strcmp("", string));

  strcpy(string, "    ");
  CU_trim(string);
  TEST(!strcmp("", string));

  strcpy(string, " b");
  CU_trim(string);
  TEST(!strcmp("b", string));

  strcpy(string, "  B");
  CU_trim(string);
  TEST(!strcmp("B", string));

  strcpy(string, "s ");
  CU_trim(string);
  TEST(!strcmp("s", string));

  strcpy(string, "S  ");
  CU_trim(string);
  TEST(!strcmp("S", string));

  strcpy(string, "  5   ");
  CU_trim(string);
  TEST(!strcmp("5", string));

  strcpy(string, "~ & ^ ( ^  ");
  CU_trim(string);
  TEST(!strcmp("~ & ^ ( ^", string));

  strcpy(string, "  ~ & ^ ( ^");
  CU_trim(string);
  TEST(!strcmp("~ & ^ ( ^", string));

  strcpy(string, "  ~ & ^ ( ^  ");
  CU_trim(string);
  TEST(!strcmp("~ & ^ ( ^", string));
}

static void test_CU_trim_left(void)
{
  char string[MAX_LEN];

  strcpy(string, "");
  CU_trim_left(string);
  TEST(!strcmp("", string));

  strcpy(string, " ");
  CU_trim_left(string);
  TEST(!strcmp("", string));

  strcpy(string, "    ");
  CU_trim_left(string);
  TEST(!strcmp("", string));

  strcpy(string, " b");
  CU_trim_left(string);
  TEST(!strcmp("b", string));

  strcpy(string, "  B");
  CU_trim_left(string);
  TEST(!strcmp("B", string));

  strcpy(string, "s ");
  CU_trim_left(string);
  TEST(!strcmp("s ", string));

  strcpy(string, "S  ");
  CU_trim_left(string);
  TEST(!strcmp("S  ", string));

  strcpy(string, "  5   ");
  CU_trim_left(string);
  TEST(!strcmp("5   ", string));

  strcpy(string, "~ & ^ ( ^  ");
  CU_trim_left(string);
  TEST(!strcmp("~ & ^ ( ^  ", string));

  strcpy(string, "  ~ & ^ ( ^");
  CU_trim_left(string);
  TEST(!strcmp("~ & ^ ( ^", string));

  strcpy(string, "  ~ & ^ ( ^  ");
  CU_trim_left(string);
  TEST(!strcmp("~ & ^ ( ^  ", string));
}

static void test_CU_trim_right(void)
{
  char string[MAX_LEN];

  strcpy(string, "");
  CU_trim_right(string);
  TEST(!strcmp("", string));

  strcpy(string, " ");
  CU_trim_right(string);
  TEST(!strcmp("", string));

  strcpy(string, "    ");
  CU_trim_right(string);
  TEST(!strcmp("", string));

  strcpy(string, " b");
  CU_trim_right(string);
  TEST(!strcmp(" b", string));

  strcpy(string, "  B");
  CU_trim_right(string);
  TEST(!strcmp("  B", string));

  strcpy(string, "s ");
  CU_trim_right(string);
  TEST(!strcmp("s", string));

  strcpy(string, "S  ");
  CU_trim_right(string);
  TEST(!strcmp("S", string));

  strcpy(string, "  5   ");
  CU_trim_right(string);
  TEST(!strcmp("  5", string));

  strcpy(string, "~ & ^ ( ^  ");
  CU_trim_right(string);
  TEST(!strcmp("~ & ^ ( ^", string));

  strcpy(string, "  ~ & ^ ( ^");
  CU_trim_right(string);
  TEST(!strcmp("  ~ & ^ ( ^", string));

  strcpy(string, "  ~ & ^ ( ^  ");
  CU_trim_right(string);
  TEST(!strcmp("  ~ & ^ ( ^", string));
}

void test_cunit_Util(void)
{

  test_cunit_start_tests("Util.c");

  test_CU_translate_special_characters();
  test_CU_compare_strings();
  test_CU_trim();
  test_CU_trim_left();
  test_CU_trim_right();

  test_cunit_end_tests();
}

#endif    /* CUNIT_BUILD_TESTS */




/*
 *  Global/Static Definitions
 */
static CU_pTestRegistry  f_pTestRegistry = NULL; /**< The active internal Test Registry. */

/*
 * Private function forward declarations
 */
static void      cleanup_test_registry(CU_pTestRegistry pRegistry);
static CU_pSuite create_suite(const char* strName, CU_InitializeFunc pInit, CU_CleanupFunc pClean);
static void      cleanup_suite(CU_pSuite pSuite);
static void      insert_suite(CU_pTestRegistry pRegistry, CU_pSuite pSuite);
static CU_pTest  create_test(const char* strName, CU_TestFunc pTestFunc);
static void      cleanup_test(CU_pTest pTest);
static void      insert_test(CU_pSuite pSuite, CU_pTest pTest);

static BOOL suite_exists(CU_pTestRegistry pRegistry, const char* szSuiteName);
static BOOL test_exists(CU_pSuite pSuite, const char* szTestName);

/*
 *  Public Interface functions
 */
/*------------------------------------------------------------------------*/
/** Initialize the test registry.
 * Any existing registry is freed, including all stored suites and
 * associated tests.  The most recent stored test results are also cleared.
 * <P><B>This function must not be called during a test run (checked
 * by assertion)</B></P>.
 * @return  CUE_NOMEMORY if memory for the new registry cannot be allocated,
 *          CUE_SUCCESS otherwise.
 * @see CU_cleanup_registry
 * @see CU_get_registry
 * @see CU_set_registry
 */
CU_ErrorCode CU_initialize_registry(void)
{
  CU_ErrorCode result;

  assert(!CU_is_test_running());

  CU_set_error(result = CUE_SUCCESS);

  if (NULL != f_pTestRegistry) {
    CU_cleanup_registry();
  }

  f_pTestRegistry = CU_create_new_registry();
  if (NULL == f_pTestRegistry) {
    CU_set_error(result = CUE_NOMEMORY);
  }

  return result;
}

/*------------------------------------------------------------------------*/
/** Clear the test registry.
 * The active test registry is freed, including all stored suites and
 * associated tests.  The most recent stored test results are also cleared.
 * After calling this function, CUnit suites cannot be added until
 * CU_initialize_registry() or CU_set_registry() is called.  Further, any
 * pointers to suites or test cases held by the user will be invalidated
 * by calling this function.
 * <P>This function may be called multiple times without generating an
 * error condition.  However, <B>this function must not be called during
 * a test run (checked by assertion)</B></P>.
 * @see CU_initialize_registry
 * @see CU_get_registry
 * @see CU_set_registry
 */
void CU_cleanup_registry(void)
{
  assert(!CU_is_test_running());

  CU_set_error(CUE_SUCCESS);
  CU_destroy_existing_registry(&f_pTestRegistry);  /* supposed to handle NULL ok */
  CU_clear_previous_results();
  CU_DUMP_MEMORY_USAGE(NULL);
}

/*------------------------------------------------------------------------*/
/** Retrieve a pointer to the current test registry.
 * Returns NULL if the registry has not been initialized using
 * CU_initialize_registry.  Directly accessing the registry
 * should not be necessary for most users.  This function is
 * provided primarily for internal and testing purposes.
 * @return A pointer to the current registry (NULL if uninitialized).
 * @see CU_initialize_registry
 * @see CU_set_registry
 */
CU_pTestRegistry CU_get_registry(void)
{
  return f_pTestRegistry;
}

/*------------------------------------------------------------------------*/
/** Set the registry to an existing CU_pTestRegistry instance.
 * A pointer to the original registry is returned.  Note that the
 * original registry is not freed, and it becomes the caller's
 * responsibility to do so.  Directly accessing the registry
 * should not be necessary for most users.  This function is
 * provided primarily for internal and testing purposes.
 * <P><B>This function must not be called during a test run (checked
 * by assertion)</B></P>.
 * @return A pointer to the original registry that was replaced.
 * @see CU_initialize_registry
 * @see CU_cleanup_registry
 * @see CU_get_registry
 */
CU_pTestRegistry CU_set_registry(CU_pTestRegistry pRegistry)
{
  CU_pTestRegistry pOldRegistry = f_pTestRegistry;

  assert(!CU_is_test_running());

  CU_set_error(CUE_SUCCESS);

  if (pRegistry != f_pTestRegistry) {
    f_pTestRegistry = pRegistry;
  }

  return pOldRegistry;
}

/*------------------------------------------------------------------------*/
/** Create a new test suite and add it to the test registry.
 * This function creates a new test suite having the specified
 * name and initialization/cleanup functions and adds it to the
 * test registry.  It returns a pointer to the newly-created suite,
 * which will be NULL if there was a problem with the suite creation
 * or addition.<br /><br />
 * CU_add_suite() sets the following error codes:
 * -CUE_NOREGISTRY if the registry hasn't been initialized.
 * -CUE_NO_SUITENAME if strName is NULL.
 * -CUE_DUP_SUITE if a suite having strName is already registered.
 * -CUE_NOMEMORY if a memory allocation failed.<BR /><BR />
 * NOTE - the CU_pSuite pointer returned should NOT BE FREED BY
 * THE USER.  The suite is freed by the CUnit system when
 * CU_cleanup_registry() is called.
 * <P><B>This function must not be called during a test run (checked
 * by assertion)</B></P>.
 * @param strName Name for the new test suite (unique, non-NULL).
 * @param pInit   Initialization function to call before running suite.
 * @param pClean  Cleanup function to call after running suite.
 * @return A pointer to the newly-created suite (NULL if creation failed)
 */
CU_pSuite CU_add_suite(const char* strName, CU_InitializeFunc pInit, CU_CleanupFunc pClean)
{
  CU_pSuite pRetValue = NULL;
  CU_ErrorCode error = CUE_SUCCESS;

  assert(!CU_is_test_running());

  if (!f_pTestRegistry) {
    error = CUE_NOREGISTRY;
  }
  else if (!strName) {
    error = CUE_NO_SUITENAME;
  }
  else if (suite_exists(f_pTestRegistry, strName)) {
    error = CUE_DUP_SUITE;
  }
  else {
    pRetValue = create_suite(strName, pInit, pClean);
    if (NULL == pRetValue)
      error = CUE_NOMEMORY;
    else
      insert_suite(f_pTestRegistry, pRetValue);
  }

  CU_set_error(error);
  return pRetValue;
}

/*------------------------------------------------------------------------*/
/** Create a new test case and add it to a test suite.
 * This function creates a new test having the specified name and
 * function, and adds it to the specified suite.  At present, there is
 * no mechanism for creating a test case independent of a suite, although
 * this function does return a pointer to the newly-created test.<br /><br />
 * CU_add_test() sets the following error codes:
 * -CUE_NOSUITE if pSuite is NULL.
 * -CUE_NO_TESTNAME if strName is NULL.
 * -CUE_DUP_TEST if a test having strName is already registered to pSuite.
 * -CUE_NOMEMORY if a memory allocation failed.<BR /><BR />
 * NOTE - the CU_pTest pointer returned should NOT BE FREED BY
 * THE USER.  The test is freed by the CUnit system when
 * CU_cleanup_registry() is called.
 * <P><B>This function must not be called during a test run (checked
 * by assertion)</B></P>.
 * @param pSuite  Test suite to which to add new test.
 * @param strName Name for the new test case (unique to pSuite, non-NULL).
 * @param pTest   Function to call when running the test.
 * @return A pointer to the newly-created test (NULL if creation failed)
 */
CU_pTest CU_add_test(CU_pSuite pSuite, const char* strName, CU_TestFunc pTestFunc)
{
  CU_pTest pRetValue = NULL;
  CU_ErrorCode error = CUE_SUCCESS;

  assert(!CU_is_test_running());

  if (!pSuite) {
    error = CUE_NOSUITE;
  }
  else if (!strName) {
    error = CUE_NO_TESTNAME;
  }
   else if(!pTestFunc) {
    error = CUE_NOTEST;
  }
  else if (test_exists(pSuite, strName)) {
    error = CUE_DUP_TEST;
  }
  else {
    pRetValue = create_test(strName, pTestFunc);
    if (NULL == pRetValue) {
      error = CUE_NOMEMORY;
    }
    else {
      f_pTestRegistry->uiNumberOfTests++;
      insert_test(pSuite, pRetValue);
    }
  }

  CU_set_error(error);
  return pRetValue;
}

/*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*/
/*  This section is based conceptually on code
 *  Copyright (C) 2004  Aurema Pty Ltd.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  Derived from code contributed by K. Cheung and Aurema Pty Ltd. (thanks!)
 *    int test_group_register(test_group_t *tg)
 *    int test_suite_register(test_suite_t *ts)
 */
/*------------------------------------------------------------------------*/
/* Registers multiple suite arrays in CU_SuiteInfo format.
 * The function accepts a variable number of suite arrays to
 * be registered.  The number of arrays is indicated by
 * the value of the 1st argument, suite_count.  Each suite
 * in each array is registered with the CUnit test registry,
 * along with all of the associated tests.
 * @param	suite_count The number of CU_SuiteInfo* arguments to follow.
 * @param ... suite_count number of CU_SuiteInfo* arguments.  NULLs are ignored.
 * @return A CU_ErrorCode indicating the error status.
 * @see CU_register_suites()
 */
CU_ErrorCode CU_register_nsuites(int suite_count, ...)
{
	CU_SuiteInfo *pSuiteItem = NULL;
  CU_TestInfo  *pTestItem = NULL;
  CU_pSuite     pSuite = NULL;

  va_list argptr;
  int i;

  va_start(argptr, suite_count);

  for (i=0 ; i<suite_count ; ++i) {
    pSuiteItem = va_arg(argptr, CU_pSuiteInfo);
    if (NULL != pSuiteItem) {
      for ( ; pSuiteItem->pName; pSuiteItem++) {
        if (NULL != (pSuite = CU_add_suite(pSuiteItem->pName, pSuiteItem->pInitFunc, pSuiteItem->pCleanupFunc))) {
          for (pTestItem = pSuiteItem->pTests; pTestItem->pName; pTestItem++)
            if (!CU_add_test(pSuite, pTestItem->pName, pTestItem->pTestFunc))
              return CU_get_error();
        }
        else
          return CU_get_error();
      }
    }
  }
	return CU_get_error();
}

/*------------------------------------------------------------------------*/
/** Registers the suites in a single CU_SuiteInfo array..
 * Multiple arrays can be registered using CU_register_nsuites().
 * @param	suite_info NULL-terminated array of CU_SuiteInfo
 *                   items to register.
 * @return A CU_ErrorCode indicating the error status.
 * @see CU_register_suites()
 */
CU_ErrorCode CU_register_suites(CU_SuiteInfo suite_info[])
{
  return CU_register_nsuites(1, suite_info);
}
/*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*/

/*
 *  Private static function definitions
 */
/*------------------------------------------------------------------------*/
/** Internal function to clean up the specified test registry.
 * cleanup_suite() will be called for each registered suite to perform
 * cleanup of the associated test cases.  Then, the suite's memory will
 * be freed.  Note that any pointers to tests or suites in pRegistry
 * held by the user will be invalidated by this function.  Severe problems
 * can occur if this function is called during a test run involving pRegistry.
 * Note that memory held for data members in the registry (e.g. pName) and
 * the registry itself are not freed by this function.
 * @see cleanup_suite()
 * @see cleanup_test()
 * @param pRegistry CU_pTestRegistry to clean up (non-NULL).
 */
static void cleanup_test_registry(CU_pTestRegistry pRegistry)
{
  CU_pSuite pCurSuite = NULL;
  CU_pSuite pNextSuite = NULL;

  assert(pRegistry);

  pCurSuite = pRegistry->pSuite;
  while (pCurSuite) {
    pNextSuite = pCurSuite->pNext;
    cleanup_suite(pCurSuite);

    CU_FREE(pCurSuite);
    pCurSuite = pNextSuite;
  }
  pRegistry->pSuite = NULL;
  pRegistry->uiNumberOfSuites = 0;
  pRegistry->uiNumberOfTests = 0;
}

/*------------------------------------------------------------------------*/
/** Internal function to create a new test suite having the specified parameters.
 * This function creates a new test suite having the specified
 * name and initialization/cleanup functions.  The strName cannot
 * be NULL (checked by assertion), but either or both function
 * pointers can be.  A pointer to the newly-created suite is returned,
 * or NULL if there was an error allocating memory for the new suite.
 * It is the responsibility of the caller to destroy the returned
 * suite (use cleanup_suite() before freeing the returned pointer).
 * @param strName Name for the new test suite (non-NULL).
 * @param pInit   Initialization function to call before running suite.
 * @param pClean  Cleanup function to call after running suite.
 * @return A pointer to the newly-created suite (NULL if creation failed)
 */
static CU_pSuite create_suite(const char* strName, CU_InitializeFunc pInit, CU_CleanupFunc pClean)
{
  CU_pSuite pRetValue = (CU_pSuite)CU_MALLOC(sizeof(CU_Suite));

  assert(strName);

  if (NULL != pRetValue) {
    pRetValue->pName = (char *)CU_MALLOC(strlen(strName)+1);
    if (NULL == pRetValue->pName) {
      CU_FREE(pRetValue);
      pRetValue = NULL;
    }
    else {
      strcpy(pRetValue->pName, strName);
      pRetValue->pInitializeFunc = pInit;
      pRetValue->pCleanupFunc = pClean;
      pRetValue->pTest = NULL;
      pRetValue->pNext = NULL;
      pRetValue->pPrev = NULL;
      pRetValue->uiNumberOfTests = 0;
    }
  }

  return pRetValue;
}

/*------------------------------------------------------------------------*/
/** Internal function to clean up the specified test suite.
 * Each test case registered with pSuite will be freed.
 * Severe problems can occur if this function is called
 * during a test run involving pSuite.
 * @see cleanup_test_registry()
 * @see cleanup_test()
 * @param pSuite CU_pSuite to clean up (non-NULL).
 */
static void cleanup_suite(CU_pSuite pSuite)
{
  CU_pTest pCurTest = NULL;
  CU_pTest pNextTest = NULL;

  assert(pSuite);

  pCurTest = pSuite->pTest;
  while (pCurTest) {
    pNextTest = pCurTest->pNext;

    cleanup_test(pCurTest);

    CU_FREE(pCurTest);
    pCurTest = pNextTest;
  }
  if (pSuite->pName)
    CU_FREE(pSuite->pName);

  pSuite->pName = NULL;
  pSuite->pTest = NULL;
  pSuite->uiNumberOfTests = 0;
}

/*------------------------------------------------------------------------*/
/** Internal function to insert a suite into a registry.
 * The suite name is assumed to be unique.  Internally, the list
 * of suites is a double-linked list, which this function manages.
 * Insertion of duplicate (or NULL) pSuites is not allowed.
 * Severe problems can occur if this function is called during a
 * test run involving pRegistry.
 * @param pRegistry CU_pTestRegistry to insert into (non-NULL).
 * @param pSuite    CU_pSuite to insert (non-NULL).
 * @see insert_test()
 */
static void insert_suite(CU_pTestRegistry pRegistry, CU_pSuite pSuite)
{
  CU_pSuite pCurSuite = NULL;

  assert(pRegistry);
  assert(pSuite);

  pCurSuite = pRegistry->pSuite;

  assert(pCurSuite != pSuite);

  pSuite->pNext = NULL;
  pRegistry->uiNumberOfSuites++;

  /* if this is the 1st suite to be added... */
  if (NULL == pCurSuite) {
    pRegistry->pSuite = pSuite;
    pSuite->pPrev = NULL;
  }
  /* otherwise, add it to the end of the linked list... */
  else {
    while (NULL != pCurSuite->pNext) {
      pCurSuite = pCurSuite->pNext;
      assert(pCurSuite != pSuite);
    }

    pCurSuite->pNext = pSuite;
    pSuite->pPrev = pCurSuite;
  }
}

/*------------------------------------------------------------------------*/
/** Internal function to create a new test case having the specified parameters.
 * This function creates a new test having the specified name and
 * test function.  The strName cannot be NULL (checked by assertion),
 * but the function pointer may be.  A pointer to the newly-created 
 * test is returned, or NULL if there was an error allocating memory for
 * the new test.  It is the responsibility of the caller to destroy the
 * returned test (use cleanup_test() before freeing the returned pointer).
 * @param strName   Name for the new test.
 * @param pTestFunc Test function to call when running this test.
 * @return A pointer to the newly-created test (NULL if creation failed)
 */
static CU_pTest create_test(const char* strName, CU_TestFunc pTestFunc)
{
  CU_pTest pRetValue = (CU_pTest)CU_MALLOC(sizeof(CU_Test));

  assert(strName);

  if (NULL != pRetValue) {
    pRetValue->pName = (char *)CU_MALLOC(strlen(strName)+1);
    if (NULL == pRetValue->pName) {
      CU_FREE(pRetValue);
      pRetValue = NULL;
    }
    else {
      strcpy(pRetValue->pName, strName);
      pRetValue->pTestFunc = pTestFunc;
      pRetValue->pJumpBuf = NULL;
      pRetValue->pNext = NULL;
      pRetValue->pPrev = NULL;
    }
  }

  return pRetValue;
}

/*------------------------------------------------------------------------*/
/** Internal function to clean up the specified test.
 * All memory associated with the test will be freed.
 * Severe problems can occur if this function is called
 * during a test run involving pTest.
 * @see cleanup_test_registry()
 * @see cleanup_suite()
 * @param pTest CU_pTest to clean up (non-NULL).
 */
static void cleanup_test(CU_pTest pTest)
{
  assert(pTest);

  if (pTest->pName)
    CU_FREE(pTest->pName);

  pTest->pName = NULL;
}

/*------------------------------------------------------------------------*/
/** Internal function to insert a test into a suite.
 * The test name is assumed to be unique.  Internally, the list
 * of tests in a suite is a double-linked list, which this
 * function manages.   Insertion of duplicate tests (or NULL
 * pTest) is not allowed.  Severe problems can occur if this function
 * is called during a test run involving pSuite.
 * @param pSuite CU_pSuite to insert into (non-NULL).
 * @param pTest  CU_pTest to insert (non-NULL).
 * @see insert_suite()
 */
static void insert_test(CU_pSuite pSuite, CU_pTest pTest)
{
  CU_pTest pCurTest = NULL;

  assert(pSuite);
  assert(pTest);

  pCurTest = pSuite->pTest;

  assert(pCurTest != pTest);

  pTest->pNext = NULL;
  pSuite->uiNumberOfTests++;
  /* if this is the 1st suite to be added... */
  if (NULL == pCurTest) {
    pSuite->pTest = pTest;
    pTest->pPrev = NULL;
  }
  else {
    while (NULL != pCurTest->pNext) {
      pCurTest = pCurTest->pNext;
      assert(pCurTest != pTest);
    }

    pCurTest->pNext = pTest;
    pTest->pPrev = pCurTest;
  }
}

/*------------------------------------------------------------------------*/
/** Internal function to check whether a suite having a specified
 * name already exists.
 * @param pRegistry   CU_pTestRegistry to check (non-NULL).
 * @param szSuiteName Suite name to check.
 * @return TRUE if suite exists in the registry, FALSE otherwise.
 */
static BOOL suite_exists(CU_pTestRegistry pRegistry, const char* szSuiteName)
{
  CU_pSuite pSuite = NULL;

  assert(pRegistry);

  pSuite = pRegistry->pSuite;
  while (pSuite) {
    if (!CU_compare_strings(szSuiteName, pSuite->pName))
      return TRUE;
    pSuite = pSuite->pNext;
  }

  return FALSE;
}

/*------------------------------------------------------------------------*/
/** Internal function to check whether a test having a specified
 * name is already registered in a given suite.
 * @param pSuite     CU_pSuite to check.
 * @param szTestName Test case name to check.
 * @return TRUE if test exists in the suite, FALSE otherwise.
 */
static BOOL test_exists(CU_pSuite pSuite, const char* szTestName)
{
  CU_pTest pTest = NULL;

  assert(pSuite);

  pTest = pSuite->pTest;
  while (pTest) {
    if (!CU_compare_strings(szTestName, pTest->pName))
      return TRUE;
    pTest = pTest->pNext;
  }

  return FALSE;
}

/*------------------------------------------------------------------------*/
/** Create and initialize a new test registry.
 * Returns a pointer to a new, initialized registry (NULL if
 * memory could not be allocated).  It is the caller's
 * responsibility to destroy and free the new registry
 * (unless it is made the active test registry using
 * CU_set_registry().
 */
CU_pTestRegistry CU_create_new_registry(void)
{
  CU_pTestRegistry pRegistry = (CU_pTestRegistry)CU_MALLOC(sizeof(CU_TestRegistry));
  if (NULL != pRegistry) {
    pRegistry->pSuite = NULL;
    pRegistry->uiNumberOfSuites = 0;
    pRegistry->uiNumberOfTests = 0;
  }

  return pRegistry;
}

/*------------------------------------------------------------------------*/
/** Destroy and free all memory for an existing test registry.
 * The active test registry is destroyed by the CUnit system in
 * CU_cleanup_registry(), so only call this function on registries
 * created independently of the internal CUnit system.  Once a
 * registry is made the active test registry using CU_set_registry(),
 * its destruction will be handled by the framework.  Passing a
 * NULL *ppRegistry will have no effect.
 * @param ppRegistry Address of a pointer to the registry to destroy.
 */
void CU_destroy_existing_registry(CU_pTestRegistry* ppRegistry)
{
  /* Note - CU_cleanup_registry counts on being able to pass NULL */
  if (NULL != *ppRegistry) {
    cleanup_test_registry(*ppRegistry);
    CU_FREE(*ppRegistry);
    *ppRegistry = NULL;
  }
}

/*------------------------------------------------------------------------*/
/** Retrieve a pointer to the suite having the specified name.
 * Scans the pRegistry and returns a pointer to the first
 * suite located having the specified name.
 * @param szSuiteName The name of the suite to locate.
 * @param pRegistry   The registry to scan.
 * @return Pointer to the first suite having the specified name,
 *         NULL if not found.
 */
CU_pSuite CU_get_suite_by_name(const char* szSuiteName, CU_pTestRegistry pRegistry)
{
	CU_pSuite pSuite = NULL;
	CU_pSuite pCur = NULL;

  assert(pRegistry);

	pCur = pRegistry->pSuite;
	while (pCur) {
		if (!CU_compare_strings(pCur->pName, szSuiteName)) {
			pSuite = pCur;
			break;
		}
		pCur = pCur->pNext;
	}

	return pSuite;
}

/*------------------------------------------------------------------------*/
/** Retrieve a pointer to the test case having the specified name.
 * Scans the pSuite and returns a pointer to the first
 * test case located having the specified name.
 * @param szTestName The name of the test case to locate.
 * @param pSuite     The suite to scan.
 * @return Pointer to the first test case having the specified name,
 *         NULL if not found.
 */
CU_pTest CU_get_test_by_name(const char* szTestName, CU_pSuite pSuite)
{
	CU_pTest pTest = NULL;
	CU_pTest pCur = NULL;

  assert(pSuite);

	pCur = pSuite->pTest;
	while (pCur) {
		if (!CU_compare_strings(pCur->pName, szTestName)) {
			pTest = pCur;
			break;
		}
		pCur = pCur->pNext;
	}

	return pTest;
}

/** @} */

/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/
#ifdef CUNIT_BUILD_TESTS
#include "test_cunit.h"

static int sfunc1(void)
{
  return 0;
}

static void test1(void)
{
}

/*--------------------------------------------------*/
static void test_CU_initialize_registry(void)
{
  CU_pTestRegistry pReg = NULL;
  unsigned int ndeallocs_before;

  /* initial state */
  TEST(NULL == CU_get_registry());

  /* after normal initialization */
  TEST(CUE_SUCCESS == CU_initialize_registry());
  pReg = CU_get_registry();
  TEST_FATAL(NULL != pReg);
  TEST(0 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);
  TEST(NULL == pReg->pSuite);

  /* after reinitialization */
  TEST(0 < test_cunit_get_n_memevents(pReg));
  ndeallocs_before = test_cunit_get_n_deallocations(pReg);
  TEST(CUE_SUCCESS == CU_initialize_registry());
  TEST((ndeallocs_before + 1) == test_cunit_get_n_deallocations(pReg));
  pReg = CU_get_registry();
  TEST_FATAL(NULL != pReg);
  TEST(0 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);
  TEST(NULL == pReg->pSuite);

  /* after cleanup */
  CU_cleanup_registry();
  TEST(NULL == CU_get_registry());

  /* if malloc fails */
  test_cunit_deactivate_malloc();
  TEST(CUE_NOMEMORY == CU_initialize_registry());
  TEST(NULL == CU_get_registry());
  test_cunit_activate_malloc();
}

/*--------------------------------------------------*/
static void test_CU_cleanup_registry(void)
{
  /* make sure calling with uninitialized registry does not crash */
  CU_cleanup_registry();
  CU_cleanup_registry();
  CU_cleanup_registry();
  CU_cleanup_registry();
  CU_cleanup_registry();

  /* nothing more to do over test_CU_initialize_registry() */
}

/*--------------------------------------------------*/
static void test_CU_add_suite(void)
{
  CU_pSuite pSuite = NULL;
  CU_pSuite pSuite2 = NULL;
  CU_pSuite pSuite3 = NULL;
  CU_pSuite pSuite4 = NULL;
  CU_pTestRegistry pReg = NULL;

  CU_cleanup_registry();  /* make sure registry not initialized */

  /* error condition - registry not initialized */
  pSuite = CU_add_suite("suite1", NULL, NULL);
  TEST(CUE_NOREGISTRY == CU_get_error());
  TEST(NULL == pSuite);

  /* error condition - no name */
  CU_initialize_registry();
  pReg = CU_get_registry();

  pSuite = CU_add_suite(NULL, NULL, NULL);
  TEST(CUE_NO_SUITENAME == CU_get_error());
  TEST(NULL == pSuite);
  TEST(0 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);

  /* error condition - duplicate name */
  CU_initialize_registry();
  pReg = CU_get_registry();

  pSuite = CU_add_suite("suite1", NULL, NULL);
  TEST(CUE_SUCCESS == CU_get_error());
  TEST(NULL != pSuite);
  TEST(1 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);

  pSuite = CU_add_suite("suite1", NULL, NULL);
  TEST(CUE_DUP_SUITE == CU_get_error());
  TEST(NULL == pSuite);
  TEST(1 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);

  /* error condition - memory allocation failure */
  CU_initialize_registry();
  pReg = CU_get_registry();

  test_cunit_deactivate_malloc();
  pSuite = CU_add_suite("suite1", NULL, NULL);
  TEST(CUE_NOMEMORY == CU_get_error());
  TEST(NULL == pSuite);
  TEST(0 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);
  test_cunit_activate_malloc();

  /* normal creation & cleanup */
  CU_initialize_registry();
  pReg = CU_get_registry();

  pSuite = CU_add_suite("suite1", NULL, NULL);
  TEST(CUE_SUCCESS == CU_get_error());
  TEST(NULL != pSuite);
  TEST(1 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);
  TEST(CU_get_suite_by_name("suite1", pReg) == pSuite);
  TEST(pReg->pSuite == pSuite);

  TEST(!strcmp("suite1", pSuite->pName));
  TEST(pSuite->pTest == NULL);            /* no tests added yet */
  TEST(pSuite->uiNumberOfTests == 0);     /* no tests added yet */
  TEST(pSuite->pInitializeFunc == NULL);  /* no init function */
  TEST(pSuite->pCleanupFunc == NULL);     /* no cleanup function */
  TEST(pSuite->pNext == NULL);            /* no more suites added yet */

  pSuite2 = CU_add_suite("suite2", sfunc1, NULL);
  TEST(CUE_SUCCESS == CU_get_error());
  TEST(NULL != pSuite2);
  TEST(2 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);
  TEST(CU_get_suite_by_name("suite2", pReg) == pSuite2);

  pSuite3 = CU_add_suite("suite3", NULL, sfunc1);
  TEST(CUE_SUCCESS == CU_get_error());
  TEST(NULL != pSuite3);
  TEST(3 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);
  TEST(CU_get_suite_by_name("suite3", pReg) == pSuite3);

  pSuite4 = CU_add_suite("suite4", sfunc1, sfunc1);
  TEST(CUE_SUCCESS == CU_get_error());
  TEST(NULL != pSuite4);
  TEST(4 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);
  TEST(CU_get_suite_by_name("suite4", pReg) == pSuite4);

  TEST(pReg->pSuite == pSuite);

  TEST(!strcmp("suite1", pSuite->pName));
  TEST(pSuite->pTest == NULL);              /* no tests added yet */
  TEST(pSuite->uiNumberOfTests == 0);       /* no tests added yet */
  TEST(pSuite->pInitializeFunc == NULL);    /* no init function */
  TEST(pSuite->pCleanupFunc == NULL);       /* no cleanup function */
  TEST(pSuite->pNext == pSuite2);           /* now have another suite */

  TEST(!strcmp("suite2", pSuite2->pName));
  TEST(pSuite2->pTest == NULL);             /* no tests added yet */
  TEST(pSuite2->uiNumberOfTests == 0);      /* no tests added yet */
  TEST(pSuite2->pInitializeFunc == sfunc1); /* no init function */
  TEST(pSuite2->pCleanupFunc == NULL);      /* no cleanup function */
  TEST(pSuite2->pNext == pSuite3);          /* next suite in list */

  TEST(!strcmp("suite3", pSuite3->pName));
  TEST(pSuite3->pTest == NULL);             /* no tests added yet */
  TEST(pSuite3->uiNumberOfTests == 0);      /* no tests added yet */
  TEST(pSuite3->pInitializeFunc == NULL);   /* no init function */
  TEST(pSuite3->pCleanupFunc == sfunc1);    /* no cleanup function */
  TEST(pSuite3->pNext == pSuite4);          /* next suite in list */

  TEST(!strcmp("suite4", pSuite4->pName));
  TEST(pSuite4->pTest == NULL);             /* no tests added yet */
  TEST(pSuite4->uiNumberOfTests == 0);      /* no tests added yet */
  TEST(pSuite4->pInitializeFunc == sfunc1); /* no init function */
  TEST(pSuite4->pCleanupFunc == sfunc1);    /* no cleanup function */
  TEST(pSuite4->pNext == NULL);             /* end of suite list */

  TEST(0 != test_cunit_get_n_memevents(pSuite));
  TEST(0 != test_cunit_get_n_memevents(pSuite2));
  TEST(0 != test_cunit_get_n_memevents(pSuite3));
  TEST(0 != test_cunit_get_n_memevents(pSuite4));

  TEST(test_cunit_get_n_allocations(pSuite) != test_cunit_get_n_deallocations(pSuite));
  TEST(test_cunit_get_n_allocations(pSuite2) != test_cunit_get_n_deallocations(pSuite2));
  TEST(test_cunit_get_n_allocations(pSuite3) != test_cunit_get_n_deallocations(pSuite3));
  TEST(test_cunit_get_n_allocations(pSuite4) != test_cunit_get_n_deallocations(pSuite4));

  CU_cleanup_registry();

  TEST(test_cunit_get_n_allocations(pSuite) == test_cunit_get_n_deallocations(pSuite));
  TEST(test_cunit_get_n_allocations(pSuite2) == test_cunit_get_n_deallocations(pSuite2));
  TEST(test_cunit_get_n_allocations(pSuite3) == test_cunit_get_n_deallocations(pSuite3));
  TEST(test_cunit_get_n_allocations(pSuite4) == test_cunit_get_n_deallocations(pSuite4));
}

/*--------------------------------------------------*/
static void test_CU_add_test(void)
{
  CU_pSuite pSuite1 = NULL;
  CU_pSuite pSuite2 = NULL;
  CU_pTest pTest1 = NULL;
  CU_pTest pTest2 = NULL;
  CU_pTest pTest3 = NULL;
  CU_pTest pTest4 = NULL;
  CU_pTestRegistry pReg = NULL;

  CU_initialize_registry();
  pReg = CU_get_registry();

  /* error condition - no suite */
  pTest1 = CU_add_test(pSuite1, "test1", test1);
  TEST(CUE_NOSUITE == CU_get_error());
  TEST(NULL == pTest1);

  /* error condition - no name */
  pSuite1 = CU_add_suite("suite1", NULL, NULL);
  TEST(CUE_SUCCESS == CU_get_error());
  pTest1 = CU_add_test(pSuite1, NULL, test1);
  TEST(CUE_NO_TESTNAME == CU_get_error());
  TEST(NULL == pTest1);
  TEST(1 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);
  TEST(0 == pSuite1->uiNumberOfTests);

  /* error condition - no test function */
  pTest1 = CU_add_test(pSuite1, "test1", NULL);
  TEST(CUE_NOTEST == CU_get_error());
  TEST(NULL == pTest1);
  TEST(1 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);
  TEST(0 == pSuite1->uiNumberOfTests);

  /* error condition - duplicate name */
  CU_initialize_registry();

  pSuite1 = CU_add_suite("suite1", NULL, NULL);
  TEST(CUE_SUCCESS == CU_get_error());
  pTest1 = CU_add_test(pSuite1, "test1", test1);
  TEST(CUE_SUCCESS == CU_get_error());
  TEST(NULL != pTest1);
  TEST(1 == pReg->uiNumberOfSuites);
  TEST(1 == pReg->uiNumberOfTests);
  TEST(1 == pSuite1->uiNumberOfTests);

  pTest2 = CU_add_test(pSuite1, "test1", test1);
  TEST(CUE_DUP_TEST == CU_get_error());
  TEST(NULL == pTest2);
  TEST(1 == pReg->uiNumberOfSuites);
  TEST(1 == pReg->uiNumberOfTests);
  TEST(1 == pSuite1->uiNumberOfTests);

  /* error condition - memory allocation failure */
  CU_initialize_registry();

  pSuite1 = CU_add_suite("suite1", NULL, NULL);
  TEST(CUE_SUCCESS == CU_get_error());
  test_cunit_deactivate_malloc();
  pTest1 = CU_add_test(pSuite1, "test1", test1);
  test_cunit_activate_malloc();
  TEST(CUE_NOMEMORY == CU_get_error());
  TEST(NULL == pTest1);
  TEST(1 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);
  TEST(0 == pSuite1->uiNumberOfTests);

  /* normal creation & cleanup */
  CU_initialize_registry();

  pSuite1 = CU_add_suite("suite1", NULL, NULL);
  TEST(CUE_SUCCESS == CU_get_error());
  pSuite2 = CU_add_suite("suite2", sfunc1, sfunc1);
  TEST(CUE_SUCCESS == CU_get_error());

  pTest1 = CU_add_test(pSuite1, "test1", test1);
  TEST(CUE_SUCCESS == CU_get_error());
  TEST(NULL != pTest1);
  TEST(2 == pReg->uiNumberOfSuites);
  TEST(1 == pReg->uiNumberOfTests);
  TEST(1 == pSuite1->uiNumberOfTests);
  TEST(0 == pSuite2->uiNumberOfTests);
  TEST(pSuite1->pTest == pTest1);
  TEST(pSuite2->pTest == NULL);

  pTest2 = CU_add_test(pSuite2, "test2", test1);
  TEST(CUE_SUCCESS == CU_get_error());
  TEST(NULL != pTest2);
  TEST(2 == pReg->uiNumberOfSuites);
  TEST(2 == pReg->uiNumberOfTests);
  TEST(1 == pSuite1->uiNumberOfTests);
  TEST(1 == pSuite2->uiNumberOfTests);
  TEST(pSuite1->pTest == pTest1);
  TEST(pSuite2->pTest == pTest2);

  pTest3 = CU_add_test(pSuite1, "test3", test1);
  TEST(CUE_SUCCESS == CU_get_error());
  TEST(NULL != pTest3);
  TEST(2 == pReg->uiNumberOfSuites);
  TEST(3 == pReg->uiNumberOfTests);
  TEST(2 == pSuite1->uiNumberOfTests);
  TEST(1 == pSuite2->uiNumberOfTests);
  TEST(pSuite1->pTest == pTest1);
  TEST(pSuite2->pTest == pTest2);

  pTest4 = CU_add_test(pSuite1, "test4", test1);
  TEST(CUE_SUCCESS == CU_get_error());
  TEST(NULL != pTest4);
  TEST(2 == pReg->uiNumberOfSuites);
  TEST(4 == pReg->uiNumberOfTests);
  TEST(3 == pSuite1->uiNumberOfTests);
  TEST(1 == pSuite2->uiNumberOfTests);
  TEST(pSuite1->pTest == pTest1);
  TEST(pSuite2->pTest == pTest2);

  TEST(!strcmp("test1", pTest1->pName));
  TEST(pTest1->pNext == pTest3);
  TEST(pTest1->pJumpBuf == NULL);
  TEST(pTest1->pTestFunc == test1);
  TEST(CU_get_test_by_name("test1", pSuite1) == pTest1);
  TEST(CU_get_test_by_name("test1", pSuite2) == NULL);

  TEST(!strcmp("test2", pTest2->pName));
  TEST(pTest2->pNext == NULL);
  TEST(pTest2->pJumpBuf == NULL);
  TEST(pTest2->pTestFunc == test1);
  TEST(CU_get_test_by_name("test2", pSuite1) == NULL);
  TEST(CU_get_test_by_name("test2", pSuite2) == pTest2);

  TEST(!strcmp("test3", pTest3->pName));
  TEST(pTest3->pNext == pTest4);
  TEST(pTest3->pJumpBuf == NULL);
  TEST(pTest3->pTestFunc == test1);
  TEST(CU_get_test_by_name("test3", pSuite1) == pTest3);
  TEST(CU_get_test_by_name("test3", pSuite2) == NULL);

  TEST(!strcmp("test4", pTest4->pName));
  TEST(pTest4->pNext == NULL);
  TEST(pTest4->pJumpBuf == NULL);
  TEST(pTest4->pTestFunc == test1);
  TEST(CU_get_test_by_name("test4", pSuite1) == pTest4);
  TEST(CU_get_test_by_name("test4", pSuite2) == NULL);

  TEST(0 != test_cunit_get_n_memevents(pSuite1));
  TEST(0 != test_cunit_get_n_memevents(pSuite2));
  TEST(0 != test_cunit_get_n_memevents(pTest1));
  TEST(0 != test_cunit_get_n_memevents(pTest2));
  TEST(0 != test_cunit_get_n_memevents(pTest3));
  TEST(0 != test_cunit_get_n_memevents(pTest4));

  TEST(test_cunit_get_n_allocations(pSuite1) != test_cunit_get_n_deallocations(pSuite1));
  TEST(test_cunit_get_n_allocations(pSuite2) != test_cunit_get_n_deallocations(pSuite2));
  TEST(test_cunit_get_n_allocations(pTest1) != test_cunit_get_n_deallocations(pTest1));
  TEST(test_cunit_get_n_allocations(pTest2) != test_cunit_get_n_deallocations(pTest2));
  TEST(test_cunit_get_n_allocations(pTest3) != test_cunit_get_n_deallocations(pTest3));
  TEST(test_cunit_get_n_allocations(pTest4) != test_cunit_get_n_deallocations(pTest4));

  CU_cleanup_registry();

  TEST(test_cunit_get_n_allocations(pSuite1) == test_cunit_get_n_deallocations(pSuite1));
  TEST(test_cunit_get_n_allocations(pSuite2) == test_cunit_get_n_deallocations(pSuite2));
  TEST(test_cunit_get_n_allocations(pTest1) == test_cunit_get_n_deallocations(pTest1));
  TEST(test_cunit_get_n_allocations(pTest2) == test_cunit_get_n_deallocations(pTest2));
  TEST(test_cunit_get_n_allocations(pTest3) == test_cunit_get_n_deallocations(pTest3));
  TEST(test_cunit_get_n_allocations(pTest4) == test_cunit_get_n_deallocations(pTest4));
}

/*--------------------------------------------------*/
static void test_CU_get_registry(void)
{
  CU_cleanup_registry();
  TEST(NULL == CU_get_registry());

  CU_initialize_registry();
  TEST(NULL != CU_get_registry());
  TEST(f_pTestRegistry == CU_get_registry());

  CU_cleanup_registry();
}

/*--------------------------------------------------*/
static void test_CU_set_registry(void)
{
  CU_pTestRegistry pReg1 = NULL;
  CU_pTestRegistry pReg2 = NULL;
  CU_pSuite pSuite1 = NULL;
  CU_pSuite pSuite2 = NULL;

  CU_initialize_registry();
  pSuite1 = CU_add_suite("suite1", NULL, NULL);
  pSuite2 = CU_add_suite("suite2", NULL, NULL);

  CU_add_test(pSuite1, "test1", test1);
  CU_add_test(pSuite1, "test2", test1);
  CU_add_test(pSuite2, "test1", test1);
  CU_add_test(pSuite2, "test2", test1);

  pReg1 = CU_get_registry();

  TEST(pReg1->pSuite == pSuite1);
  TEST(pReg1->uiNumberOfSuites == 2);
  TEST(pReg1->uiNumberOfTests == 4);
  TEST(0 < test_cunit_get_n_memevents(pReg1));
  TEST(test_cunit_get_n_allocations(pReg1) != test_cunit_get_n_deallocations(pReg1));

  CU_set_registry(NULL);

  TEST(test_cunit_get_n_allocations(pReg1) != test_cunit_get_n_deallocations(pReg1));

  CU_cleanup_registry();

  TEST(test_cunit_get_n_allocations(pReg1) != test_cunit_get_n_deallocations(pReg1));

  pReg2 = CU_create_new_registry();
  CU_set_registry(pReg2);

  TEST(pReg1->pSuite == pSuite1);
  TEST(pReg1->uiNumberOfSuites == 2);
  TEST(pReg1->uiNumberOfTests == 4);
  TEST(test_cunit_get_n_allocations(pReg1) != test_cunit_get_n_deallocations(pReg1));

  TEST(CU_get_registry()->pSuite == NULL);
  TEST(CU_get_registry()->uiNumberOfSuites == 0);
  TEST(CU_get_registry()->uiNumberOfTests == 0);
  TEST(0 < test_cunit_get_n_memevents(pReg2));
  TEST(test_cunit_get_n_allocations(pReg2) != test_cunit_get_n_deallocations(pReg2));

  CU_cleanup_registry();

  TEST(pReg1->pSuite == pSuite1);
  TEST(pReg1->uiNumberOfSuites == 2);
  TEST(pReg1->uiNumberOfTests == 4);
  TEST(test_cunit_get_n_allocations(pReg1) != test_cunit_get_n_deallocations(pReg1));
  TEST(test_cunit_get_n_allocations(pReg2) == test_cunit_get_n_deallocations(pReg2));

  CU_set_registry(pReg1);
  CU_cleanup_registry();
  TEST(test_cunit_get_n_allocations(pReg1) == test_cunit_get_n_deallocations(pReg1));
}

/*--------------------------------------------------*/
static void test_CU_create_new_registry(void)
{
  CU_pTestRegistry pReg = NULL;

  CU_cleanup_registry();
  pReg = CU_create_new_registry();

  TEST(0 < test_cunit_get_n_memevents(pReg));
  TEST(test_cunit_get_n_allocations(pReg) != test_cunit_get_n_deallocations(pReg));

  TEST(pReg->pSuite == NULL);
  TEST(pReg->uiNumberOfSuites == 0);
  TEST(pReg->uiNumberOfTests == 0);

  CU_cleanup_registry();
  TEST(test_cunit_get_n_allocations(pReg) != test_cunit_get_n_deallocations(pReg));

  CU_destroy_existing_registry(&pReg);
  TEST(test_cunit_get_n_allocations(pReg) == test_cunit_get_n_deallocations(pReg));
}

/*--------------------------------------------------*/
static void test_CU_destroy_existing_registry(void)
{
  /* covered by test_CU_create_new_registry() */
}

/*--------------------------------------------------*/
static void test_CU_get_suite_by_name(void)
{
  /* covered by test_CU_add_suite() */
}

/*--------------------------------------------------*/
static void test_CU_get_test_by_name(void)
{
  /* covered by test_CU_add_test() */
}

/*--------------------------------------------------*/
static void test_cleanup_test_registry(void)
{
  CU_pSuite pSuite1 = NULL;
  CU_pSuite pSuite2 = NULL;
  CU_pTest pTest1 = NULL;
  CU_pTest pTest2 = NULL;
  CU_pTest pTest3 = NULL;
  CU_pTest pTest4 = NULL;
  CU_pTestRegistry pReg = CU_create_new_registry();

  TEST_FATAL(NULL != pReg);
  TEST(0 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);

  /* create tests to register */
  pTest1 = create_test("test1", test1);
  pTest2 = create_test("test2", NULL);
  pTest3 = create_test("test3", test1);
  pTest4 = create_test("", NULL);

  /* create suites to hold tests */
  pSuite1 = create_suite("suite1", NULL, NULL);
  pSuite2 = create_suite("suite2", sfunc1, sfunc1);
  insert_suite(pReg, pSuite1);
  insert_suite(pReg, pSuite2);

  insert_test(pSuite1, pTest1);
  insert_test(pSuite1, pTest2);
  insert_test(pSuite1, pTest3);
  insert_test(pSuite2, pTest4);

  TEST(2 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);   /* not managed in primitive functions */
  TEST(3 == pSuite1->uiNumberOfTests);
  TEST(1 == pSuite2->uiNumberOfTests);
  TEST(pSuite1->pTest == pTest1);
  TEST(pSuite2->pTest == pTest4);
  TEST(pTest1->pNext == pTest2);
  TEST(pTest1->pPrev == NULL);
  TEST(pTest2->pNext == pTest3);
  TEST(pTest2->pPrev == pTest1);
  TEST(pTest3->pNext == NULL);
  TEST(pTest3->pPrev == pTest2);
  TEST(pTest4->pNext == NULL);
  TEST(pTest4->pPrev == NULL);

  TEST(0 != test_cunit_get_n_memevents(pReg));
  TEST(0 != test_cunit_get_n_memevents(pSuite1));
  TEST(0 != test_cunit_get_n_memevents(pSuite2));
  TEST(0 != test_cunit_get_n_memevents(pTest1));
  TEST(0 != test_cunit_get_n_memevents(pTest2));
  TEST(0 != test_cunit_get_n_memevents(pTest3));
  TEST(0 != test_cunit_get_n_memevents(pTest4));

  TEST(test_cunit_get_n_allocations(pReg) != test_cunit_get_n_deallocations(pReg));
  TEST(test_cunit_get_n_allocations(pSuite1) != test_cunit_get_n_deallocations(pSuite1));
  TEST(test_cunit_get_n_allocations(pSuite2) != test_cunit_get_n_deallocations(pSuite2));
  TEST(test_cunit_get_n_allocations(pTest1) != test_cunit_get_n_deallocations(pTest1));
  TEST(test_cunit_get_n_allocations(pTest2) != test_cunit_get_n_deallocations(pTest2));
  TEST(test_cunit_get_n_allocations(pTest3) != test_cunit_get_n_deallocations(pTest3));
  TEST(test_cunit_get_n_allocations(pTest4) != test_cunit_get_n_deallocations(pTest4));

  cleanup_test_registry(pReg);
  CU_FREE(pReg);

  TEST(test_cunit_get_n_allocations(pReg) == test_cunit_get_n_deallocations(pReg));
  TEST(test_cunit_get_n_allocations(pSuite1) == test_cunit_get_n_deallocations(pSuite1));
  TEST(test_cunit_get_n_allocations(pSuite2) == test_cunit_get_n_deallocations(pSuite2));
  TEST(test_cunit_get_n_allocations(pTest1) == test_cunit_get_n_deallocations(pTest1));
  TEST(test_cunit_get_n_allocations(pTest2) == test_cunit_get_n_deallocations(pTest2));
  TEST(test_cunit_get_n_allocations(pTest3) == test_cunit_get_n_deallocations(pTest3));
  TEST(test_cunit_get_n_allocations(pTest4) == test_cunit_get_n_deallocations(pTest4));
}

/*--------------------------------------------------*/
static void test_create_suite(void)
{
  CU_pSuite pSuite1 = NULL;
  CU_pSuite pSuite2 = NULL;
  CU_pSuite pSuite3 = NULL;
  CU_pSuite pSuite4 = NULL;

  /* error condition - memory allocation failure */
  test_cunit_deactivate_malloc();
  pSuite1 = create_suite("suite1", NULL, NULL);
  TEST(NULL == pSuite1);
  test_cunit_activate_malloc();

  /* normal creation & cleanup */
  pSuite1 = create_suite("suite1", NULL, NULL);
  TEST(NULL != pSuite1);
  TEST(!strcmp("suite1", pSuite1->pName));
  TEST(pSuite1->pTest == NULL);            /* no tests added yet */
  TEST(pSuite1->uiNumberOfTests == 0);     /* no tests added yet */
  TEST(pSuite1->pInitializeFunc == NULL);  /* no init function */
  TEST(pSuite1->pCleanupFunc == NULL);     /* no cleanup function */
  TEST(pSuite1->pNext == NULL);            /* no more suites added yet */

  pSuite2 = create_suite("suite2", sfunc1, NULL);
  TEST(NULL != pSuite2);
  TEST(!strcmp("suite2", pSuite2->pName));
  TEST(pSuite2->pTest == NULL);             /* no tests added yet */
  TEST(pSuite2->uiNumberOfTests == 0);      /* no tests added yet */
  TEST(pSuite2->pInitializeFunc == sfunc1); /* init function */
  TEST(pSuite2->pCleanupFunc == NULL);      /* no cleanup function */
  TEST(pSuite2->pNext == NULL);             /* no more suites added yet */

  pSuite3 = create_suite("suite3", NULL, sfunc1);
  TEST(NULL != pSuite3);
  TEST(!strcmp("suite3", pSuite3->pName));
  TEST(pSuite3->pTest == NULL);            /* no tests added yet */
  TEST(pSuite3->uiNumberOfTests == 0);     /* no tests added yet */
  TEST(pSuite3->pInitializeFunc == NULL);  /* no init function */
  TEST(pSuite3->pCleanupFunc == sfunc1);   /* cleanup function */
  TEST(pSuite3->pNext == NULL);            /* no more suites added yet */

  pSuite4 = create_suite("suite4", sfunc1, sfunc1);
  TEST(NULL != pSuite4);
  TEST(!strcmp("suite4", pSuite4->pName));
  TEST(pSuite4->pTest == NULL);             /* no tests added yet */
  TEST(pSuite4->uiNumberOfTests == 0);      /* no tests added yet */
  TEST(pSuite4->pInitializeFunc == sfunc1); /* no init function */
  TEST(pSuite4->pCleanupFunc == sfunc1);    /* cleanup function */
  TEST(pSuite4->pNext == NULL);             /* no more suites added yet */

  TEST(0 != test_cunit_get_n_memevents(pSuite1));
  TEST(test_cunit_get_n_allocations(pSuite1) != test_cunit_get_n_deallocations(pSuite1));
  cleanup_suite(pSuite1);
  CU_FREE(pSuite1);
  TEST(test_cunit_get_n_allocations(pSuite1) == test_cunit_get_n_deallocations(pSuite1));

  TEST(0 != test_cunit_get_n_memevents(pSuite2));
  TEST(test_cunit_get_n_allocations(pSuite2) != test_cunit_get_n_deallocations(pSuite2));
  cleanup_suite(pSuite2);
  CU_FREE(pSuite2);
  TEST(test_cunit_get_n_allocations(pSuite2) == test_cunit_get_n_deallocations(pSuite2));

  TEST(0 != test_cunit_get_n_memevents(pSuite3));
  TEST(test_cunit_get_n_allocations(pSuite3) != test_cunit_get_n_deallocations(pSuite3));
  cleanup_suite(pSuite3);
  CU_FREE(pSuite3);
  TEST(test_cunit_get_n_allocations(pSuite3) == test_cunit_get_n_deallocations(pSuite3));

  TEST(0 != test_cunit_get_n_memevents(pSuite4));
  TEST(test_cunit_get_n_allocations(pSuite4) != test_cunit_get_n_deallocations(pSuite4));
  cleanup_suite(pSuite4);
  CU_FREE(pSuite4);
  TEST(test_cunit_get_n_allocations(pSuite4) == test_cunit_get_n_deallocations(pSuite4));
}

/*--------------------------------------------------*/
static void test_cleanup_suite(void)
{
  /* covered by test_create_suite() and test_create_test() */
}

/*--------------------------------------------------*/
static void test_insert_suite(void)
{
  CU_pSuite pSuite1 = NULL;
  CU_pSuite pSuite2 = NULL;
  CU_pSuite pSuite3 = NULL;
  CU_pSuite pSuite4 = NULL;
  CU_pTestRegistry pReg = CU_create_new_registry();

  TEST_FATAL(NULL != pReg);
  TEST(0 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);
  TEST(NULL == pReg->pSuite);
  TEST(FALSE == suite_exists(pReg, "suite1"));
  TEST(FALSE == suite_exists(pReg, "suite2"));
  TEST(FALSE == suite_exists(pReg, "suite3"));
  TEST(FALSE == suite_exists(pReg, "suite4"));
  TEST(FALSE == suite_exists(pReg, "suite5"));
  TEST(FALSE == suite_exists(pReg, ""));

  /* normal creation & cleanup */
  pSuite1 = create_suite("suite1", NULL, NULL);
  insert_suite(pReg, pSuite1);
  TEST(1 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);
  TEST(pReg->pSuite == pSuite1);
  TEST(pSuite1->pNext == NULL);
  TEST(TRUE == suite_exists(pReg, "suite1"));
  TEST(FALSE == suite_exists(pReg, "suite2"));
  TEST(FALSE == suite_exists(pReg, "suite3"));
  TEST(FALSE == suite_exists(pReg, "suite4"));
  TEST(FALSE == suite_exists(pReg, "suite5"));
  TEST(FALSE == suite_exists(pReg, ""));

  pSuite2 = create_suite("suite2", sfunc1, NULL);
  insert_suite(pReg, pSuite2);
  TEST(2 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);
  TEST(pReg->pSuite == pSuite1);
  TEST(pSuite1->pNext == pSuite2);
  TEST(pSuite2->pNext == NULL);
  TEST(TRUE == suite_exists(pReg, "suite1"));
  TEST(TRUE == suite_exists(pReg, "suite2"));
  TEST(FALSE == suite_exists(pReg, "suite3"));
  TEST(FALSE == suite_exists(pReg, "suite4"));
  TEST(FALSE == suite_exists(pReg, "suite5"));
  TEST(FALSE == suite_exists(pReg, ""));

  pSuite3 = create_suite("suite3", NULL, sfunc1);
  insert_suite(pReg, pSuite3);
  TEST(3 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);
  TEST(pReg->pSuite == pSuite1);
  TEST(pSuite1->pNext == pSuite2);
  TEST(pSuite2->pNext == pSuite3);
  TEST(pSuite3->pNext == NULL);
  TEST(TRUE == suite_exists(pReg, "suite1"));
  TEST(TRUE == suite_exists(pReg, "suite2"));
  TEST(TRUE == suite_exists(pReg, "suite3"));
  TEST(FALSE == suite_exists(pReg, "suite4"));
  TEST(FALSE == suite_exists(pReg, "suite5"));
  TEST(FALSE == suite_exists(pReg, ""));

  pSuite4 = create_suite("suite4", sfunc1, sfunc1);
  insert_suite(pReg, pSuite4);
  TEST(4 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);
  TEST(pReg->pSuite == pSuite1);
  TEST(pSuite1->pNext == pSuite2);
  TEST(pSuite2->pNext == pSuite3);
  TEST(pSuite3->pNext == pSuite4);
  TEST(pSuite4->pNext == NULL);
  TEST(TRUE == suite_exists(pReg, "suite1"));
  TEST(TRUE == suite_exists(pReg, "suite2"));
  TEST(TRUE == suite_exists(pReg, "suite3"));
  TEST(TRUE == suite_exists(pReg, "suite4"));
  TEST(FALSE == suite_exists(pReg, "suite5"));
  TEST(FALSE == suite_exists(pReg, ""));

  TEST(0 != test_cunit_get_n_memevents(pReg));
  TEST(0 != test_cunit_get_n_memevents(pSuite1));
  TEST(0 != test_cunit_get_n_memevents(pSuite2));
  TEST(0 != test_cunit_get_n_memevents(pSuite3));
  TEST(0 != test_cunit_get_n_memevents(pSuite4));

  TEST(test_cunit_get_n_allocations(pReg) != test_cunit_get_n_deallocations(pReg));
  TEST(test_cunit_get_n_allocations(pSuite1) != test_cunit_get_n_deallocations(pSuite1));
  TEST(test_cunit_get_n_allocations(pSuite2) != test_cunit_get_n_deallocations(pSuite2));
  TEST(test_cunit_get_n_allocations(pSuite3) != test_cunit_get_n_deallocations(pSuite3));
  TEST(test_cunit_get_n_allocations(pSuite4) != test_cunit_get_n_deallocations(pSuite4));

  cleanup_test_registry(pReg);
  TEST(FALSE == suite_exists(pReg, "suite1"));
  TEST(FALSE == suite_exists(pReg, "suite2"));
  TEST(FALSE == suite_exists(pReg, "suite3"));
  TEST(FALSE == suite_exists(pReg, "suite4"));
  TEST(FALSE == suite_exists(pReg, "suite5"));
  TEST(FALSE == suite_exists(pReg, ""));
  CU_FREE(pReg);

  TEST(test_cunit_get_n_allocations(pReg) == test_cunit_get_n_deallocations(pReg));
  TEST(test_cunit_get_n_allocations(pSuite1) == test_cunit_get_n_deallocations(pSuite1));
  TEST(test_cunit_get_n_allocations(pSuite2) == test_cunit_get_n_deallocations(pSuite2));
  TEST(test_cunit_get_n_allocations(pSuite3) == test_cunit_get_n_deallocations(pSuite3));
  TEST(test_cunit_get_n_allocations(pSuite4) == test_cunit_get_n_deallocations(pSuite4));
}

/*--------------------------------------------------*/
static void test_create_test(void)
{
  CU_pTest pTest1 = NULL;
  CU_pTest pTest2 = NULL;

  /* error condition - memory allocation failure */
  test_cunit_deactivate_malloc();
  pTest1 = create_test("test1", test1);
  test_cunit_activate_malloc();
  TEST(NULL == pTest1);

  /* normal creation & cleanup */
  pTest1 = create_test("test1", test1);
  TEST(NULL != pTest1);                                                    
  TEST(pTest1->pTestFunc == test1);
  TEST(!strcmp("test1", pTest1->pName));
  TEST(pTest1->pNext == NULL);
  TEST(pTest1->pPrev == NULL);
  TEST(pTest1->pJumpBuf == NULL);

  pTest2= create_test("test2", NULL);
  TEST(NULL != pTest2);
  TEST(pTest2->pTestFunc == NULL);
  TEST(!strcmp("test2", pTest2->pName));
  TEST(pTest2->pNext == NULL);
  TEST(pTest2->pPrev == NULL);
  TEST(pTest2->pJumpBuf == NULL);

  TEST(0 != test_cunit_get_n_memevents(pTest1));
  TEST(0 != test_cunit_get_n_memevents(pTest2));

  TEST(test_cunit_get_n_allocations(pTest1) != test_cunit_get_n_deallocations(pTest1));
  TEST(test_cunit_get_n_allocations(pTest2) != test_cunit_get_n_deallocations(pTest2));

  cleanup_test(pTest1);
  CU_FREE(pTest1);
  cleanup_test(pTest2);
  CU_FREE(pTest2);

  TEST(test_cunit_get_n_allocations(pTest1) == test_cunit_get_n_deallocations(pTest1));
  TEST(test_cunit_get_n_allocations(pTest2) == test_cunit_get_n_deallocations(pTest2));
}

/*--------------------------------------------------*/
static void test_insert_test(void)
{
  CU_pSuite pSuite1 = NULL;
  CU_pSuite pSuite2 = NULL;
  CU_pTest pTest1 = NULL;
  CU_pTest pTest2 = NULL;
  CU_pTest pTest3 = NULL;
  CU_pTest pTest4 = NULL;

  /* create tests to register */
  pTest1 = create_test("test1", test1);
  pTest2 = create_test("test2", NULL);
  pTest3 = create_test("test3", test1);
  pTest4 = create_test("", NULL);

  /* create suites to hold tests */
  pSuite1 = create_suite("suite1", NULL, NULL);
  pSuite2 = create_suite("suite2", sfunc1, sfunc1);

  TEST(FALSE == test_exists(pSuite1, "test1"));
  TEST(FALSE == test_exists(pSuite1, "test2"));
  TEST(FALSE == test_exists(pSuite1, "test3"));
  TEST(FALSE == test_exists(pSuite1, "test4"));
  TEST(FALSE == test_exists(pSuite1, ""));
  TEST(FALSE == test_exists(pSuite2, "test1"));
  TEST(FALSE == test_exists(pSuite2, "test2"));
  TEST(FALSE == test_exists(pSuite2, "test3"));
  TEST(FALSE == test_exists(pSuite2, "test4"));
  TEST(FALSE == test_exists(pSuite2, ""));

  insert_test(pSuite1, pTest1);
  insert_test(pSuite1, pTest2);
  insert_test(pSuite1, pTest3);
  insert_test(pSuite2, pTest4);

  TEST(TRUE == test_exists(pSuite1, "test1"));
  TEST(TRUE == test_exists(pSuite1, "test2"));
  TEST(TRUE == test_exists(pSuite1, "test3"));
  TEST(FALSE == test_exists(pSuite1, "test4"));
  TEST(FALSE == test_exists(pSuite1, ""));
  TEST(FALSE == test_exists(pSuite2, "test1"));
  TEST(FALSE == test_exists(pSuite2, "test2"));
  TEST(FALSE == test_exists(pSuite2, "test3"));
  TEST(FALSE == test_exists(pSuite2, "test4"));
  TEST(TRUE == test_exists(pSuite2, ""));

  TEST(3 == pSuite1->uiNumberOfTests);
  TEST(1 == pSuite2->uiNumberOfTests);
  TEST(pSuite1->pTest == pTest1);
  TEST(pSuite2->pTest == pTest4);
  TEST(pTest1->pNext == pTest2);
  TEST(pTest1->pPrev == NULL);
  TEST(pTest2->pNext == pTest3);
  TEST(pTest2->pPrev == pTest1);
  TEST(pTest3->pNext == NULL);
  TEST(pTest3->pPrev == pTest2);        
  TEST(pTest4->pNext == NULL);
  TEST(pTest4->pPrev == NULL);

  TEST(0 != test_cunit_get_n_memevents(pSuite1));
  TEST(0 != test_cunit_get_n_memevents(pSuite2));
  TEST(0 != test_cunit_get_n_memevents(pTest1));
  TEST(0 != test_cunit_get_n_memevents(pTest2));
  TEST(0 != test_cunit_get_n_memevents(pTest3));
  TEST(0 != test_cunit_get_n_memevents(pTest4));

  TEST(test_cunit_get_n_allocations(pSuite1) != test_cunit_get_n_deallocations(pSuite1));
  TEST(test_cunit_get_n_allocations(pSuite2) != test_cunit_get_n_deallocations(pSuite2));
  TEST(test_cunit_get_n_allocations(pTest1) != test_cunit_get_n_deallocations(pTest1));
  TEST(test_cunit_get_n_allocations(pTest2) != test_cunit_get_n_deallocations(pTest2));
  TEST(test_cunit_get_n_allocations(pTest3) != test_cunit_get_n_deallocations(pTest3));
  TEST(test_cunit_get_n_allocations(pTest4) != test_cunit_get_n_deallocations(pTest4));

  cleanup_suite(pSuite1);

  TEST(FALSE == test_exists(pSuite1, "test1"));
  TEST(FALSE == test_exists(pSuite1, "test2"));
  TEST(FALSE == test_exists(pSuite1, "test3"));
  TEST(FALSE == test_exists(pSuite1, "test4"));
  TEST(FALSE == test_exists(pSuite1, ""));
  TEST(FALSE == test_exists(pSuite2, "test1"));
  TEST(FALSE == test_exists(pSuite2, "test2"));
  TEST(FALSE == test_exists(pSuite2, "test3"));
  TEST(FALSE == test_exists(pSuite2, "test4"));
  TEST(TRUE == test_exists(pSuite2, ""));

  cleanup_suite(pSuite2);

  TEST(FALSE == test_exists(pSuite1, "test1"));
  TEST(FALSE == test_exists(pSuite1, "test2"));
  TEST(FALSE == test_exists(pSuite1, "test3"));
  TEST(FALSE == test_exists(pSuite1, "test4"));
  TEST(FALSE == test_exists(pSuite1, ""));
  TEST(FALSE == test_exists(pSuite2, "test1"));
  TEST(FALSE == test_exists(pSuite2, "test2"));
  TEST(FALSE == test_exists(pSuite2, "test3"));
  TEST(FALSE == test_exists(pSuite2, "test4"));
  TEST(FALSE == test_exists(pSuite2, ""));

  CU_FREE(pSuite1);
  CU_FREE(pSuite2);

  TEST(test_cunit_get_n_allocations(pSuite1) == test_cunit_get_n_deallocations(pSuite1));
  TEST(test_cunit_get_n_allocations(pSuite2) == test_cunit_get_n_deallocations(pSuite2));
  TEST(test_cunit_get_n_allocations(pTest1) == test_cunit_get_n_deallocations(pTest1));
  TEST(test_cunit_get_n_allocations(pTest2) == test_cunit_get_n_deallocations(pTest2));
  TEST(test_cunit_get_n_allocations(pTest3) == test_cunit_get_n_deallocations(pTest3));
  TEST(test_cunit_get_n_allocations(pTest4) == test_cunit_get_n_deallocations(pTest4));
}

/*--------------------------------------------------*/
static void test_cleanup_test(void)
{
  char** ppName;
  CU_pTest pTest1 = create_test("test1", NULL);

  TEST_FATAL(NULL != pTest1);

  ppName = &pTest1->pName;

  TEST(0 != test_cunit_get_n_memevents(pTest1));
  TEST(0 != test_cunit_get_n_memevents(*ppName));

  TEST(test_cunit_get_n_allocations(pTest1) != test_cunit_get_n_deallocations(pTest1));
  TEST(test_cunit_get_n_allocations(*ppName) != test_cunit_get_n_deallocations(*ppName));

  cleanup_test(pTest1);
  CU_FREE(pTest1);

  TEST(test_cunit_get_n_allocations(pTest1) == test_cunit_get_n_deallocations(pTest1));
  TEST(test_cunit_get_n_allocations(*ppName) == test_cunit_get_n_deallocations(*ppName));
}

/*--------------------------------------------------*/
static void test_suite_exists(void)
{
  /* covered by test_insert_suite() */
}

/*--------------------------------------------------*/
static void test_test_exists(void)
{
  /* covered by test_insert_test() */
}

/*--------------------------------------------------*/
static void group_A_case_1(void)
{
	CU_ASSERT_TRUE(1);
}

static void group_A_case_2(void)
{
	CU_ASSERT_TRUE(2);
}

static void group_B_case_1(void)
{
	CU_ASSERT_FALSE(1);
}

static void group_B_case_2(void)
{
	CU_ASSERT_FALSE(2);
}

static CU_TestInfo group_A_test_cases[] = {
	{ "1", group_A_case_1 },
	{ "2", group_A_case_2 },
	CU_TEST_INFO_NULL,
};

static CU_TestInfo group_B_test_cases[] = {
	{ "1", group_B_case_1 },
	{ "2", group_B_case_2 },
	CU_TEST_INFO_NULL,
};

static CU_SuiteInfo suites0[] = {
	CU_SUITE_INFO_NULL,
};

static CU_SuiteInfo suites1[] = {
	{ "A1", NULL, NULL, group_A_test_cases },
	{ "B1", NULL, NULL, group_B_test_cases },
	CU_SUITE_INFO_NULL,
};

static CU_SuiteInfo suites2[] = {
	{ "A2", NULL, NULL, group_A_test_cases },
	{ "B2", NULL, NULL, group_B_test_cases },
	CU_SUITE_INFO_NULL,
};

static void test_register_suite(void)
{
  CU_pTestRegistry pReg = NULL;
  CU_ErrorCode status;

  if (CU_initialize_registry()) {
    fprintf(stderr, "\nError initializing registry in test_register_suite().");
    return;
  }

  pReg = CU_get_registry();

  /* test initial condition */
  TEST_FATAL(NULL != pReg);
  TEST(0 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);

  /* test CU_register_suites() with NULL */
  status = CU_register_suites(NULL);
  TEST(CUE_SUCCESS == status);
  TEST(0 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);

  /* test CU_register_suites() with empty array */
  status = CU_register_suites(suites0);
  TEST(CUE_SUCCESS == status);
  TEST(0 == pReg->uiNumberOfSuites);
  TEST(0 == pReg->uiNumberOfTests);

  /* test CU_register_suites() with ok array */
  status = CU_register_suites(suites1);
  TEST(CUE_SUCCESS == status);
  TEST(2 == pReg->uiNumberOfSuites);
  TEST(4 == pReg->uiNumberOfTests);

  /* test CU_register_suites() with duplicate suite name */
  status = CU_register_suites(suites1);
  TEST(CUE_DUP_SUITE == status);
  TEST(2 == pReg->uiNumberOfSuites);
  TEST(4 == pReg->uiNumberOfTests);

  CU_cleanup_registry();

  if (CU_initialize_registry()) {
    fprintf(stderr, "\nError initializing registry in test_register_suite().");
    return;
  }

  pReg = CU_get_registry();

  /* test CU_register_nsuites() with ok arrays */
  status = CU_register_nsuites(2, suites1, suites2);
  TEST(CUE_SUCCESS == status);
  TEST(4 == pReg->uiNumberOfSuites);
  TEST(8 == pReg->uiNumberOfTests);
}

/*--------------------------------------------------*/
void test_cunit_TestDB(void)
{
  test_cunit_start_tests("TestDB.c");

  test_CU_initialize_registry();
  test_CU_cleanup_registry();
  test_CU_add_suite();
  test_CU_add_test();
  test_CU_get_registry();
  test_CU_set_registry();
  test_CU_create_new_registry();
  test_CU_destroy_existing_registry();
  test_CU_get_suite_by_name();
  test_CU_get_test_by_name();
  test_cleanup_test_registry();
  test_create_suite();
  test_cleanup_suite();
  test_insert_suite();
  test_create_test();
  test_cleanup_test();
  test_insert_test();
  test_suite_exists();
  test_test_exists();
  test_register_suite();

  test_cunit_end_tests();
}

#endif    /* CUNIT_BUILD_TESTS */



static BOOL       f_bTestIsRunning = FALSE; /**< Flag for whether a test run is in progress */
static CU_pSuite  f_pCurSuite = NULL;       /**< Pointer to the suite currently being run. */
static CU_pTest   f_pCurTest  = NULL;       /**< Pointer to the test currently being run. */

/** CU_RunSummary to hold results of each test run. */
static CU_RunSummary f_run_summary = {0, 0, 0, 0, 0, 0, 0};
/** CU_pFailureRecord to hold head of failure record list of each test run. */
static CU_pFailureRecord f_failure_list = NULL;
/** Keeps track of end of f_run_summary list for message handlers. */
static CU_pFailureRecord f_last_failure = NULL;

/* Forward declarations of static functions. */
static void         clear_previous_results(CU_pRunSummary pRunSummary, CU_pFailureRecord* ppFailure);
static void         cleanup_failure_list(CU_pFailureRecord* ppFailure);
static CU_ErrorCode run_single_suite(CU_pSuite pSuite, CU_pRunSummary pRunSummary);
static CU_ErrorCode run_single_test(CU_pTest pTest, CU_pRunSummary pRunSummary);
static void         add_failure(CU_pFailureRecord* ppFailure, CU_pRunSummary pRunSummary,
                                unsigned int uiLineNumber, char szCondition[],
                                char szFileName[], CU_pSuite pSuite, CU_pTest pTest);

/** Pointer to the function to be called before running a test. */
static CU_TestStartMessageHandler        f_pTestStartMessageHandler = NULL;
/** Pointer to the function to be called after running a test. */
static CU_TestCompleteMessageHandler     f_pTestCompleteMessageHandler = NULL;
/** Pointer to the function to be called when all tests have been run. */
static CU_AllTestsCompleteMessageHandler f_pAllTestsCompleteMessageHandler = NULL;
/** Pointer to the function to be called if a suite initialization function returns an error. */
static CU_SuiteInitFailureMessageHandler f_pSuiteInitFailureMessageHandler = NULL;

/*------------------------------------------------------------------------*/
/** Assertion implementation function.
 * All CUnit assertions reduce to a call to this function.
 * It should only be called during an active test run (checked
 * by assertion).  This means that CUnit assertions should only
 * be used in registered test functions during a test run.
 * @param bValue        Value of the assertion (TRUE or FALSE).
 * @param uiLine        Line number of failed test statement.
 * @param strCondition  String containing logical test that failed.
 * @param strFile       Source file where test statement failed.
 * @param strFunction   Function where test statement failed.
 * @param bFatal        TRUE to abort test (via longjmp()), FALSE to continue test.
 * @return As a convenience, returns the value of the assertion.
 */
BOOL CU_assertImplementation(BOOL bValue, unsigned int uiLine,
                             char strCondition[], char strFile[],
                             char strFunction[], BOOL bFatal)
{
  /* not used in current implementation - stop compiler warning */
  (void)strFunction;

  /* these should always be non-NULL (i.e. a test run is in progress) */
  assert(f_pCurSuite);
  assert(f_pCurTest);

  ++f_run_summary.nAsserts;
  if (!bValue) {
    ++f_run_summary.nAssertsFailed;
    add_failure(&f_failure_list, &f_run_summary,
                uiLine, strCondition, strFile, f_pCurSuite, f_pCurTest);

    //if (bFatal && f_pCurTest->pJumpBuf)
	if ( f_pCurTest->pJumpBuf)
      longjmp(*(f_pCurTest->pJumpBuf), 1);
  }

  return bValue;
}

/*
 * Get/Set functions for Message Handlers.
 */
/*------------------------------------------------------------------------*/
/** Set the message handler to call before each test is run. */
void CU_set_test_start_handler(CU_TestStartMessageHandler pTestStartHandler)
{
  f_pTestStartMessageHandler = pTestStartHandler;
}

/*------------------------------------------------------------------------*/
/** Set the message handler to call after each test is run. */
void CU_set_test_complete_handler(CU_TestCompleteMessageHandler pTestCompleteHandler)
{
  f_pTestCompleteMessageHandler = pTestCompleteHandler;
}

/*------------------------------------------------------------------------*/
/** Set the message handler to call after all tests have been run. */
void CU_set_all_test_complete_handler(CU_AllTestsCompleteMessageHandler pAllTestsCompleteHandler)
{
  f_pAllTestsCompleteMessageHandler = pAllTestsCompleteHandler;
}

/*------------------------------------------------------------------------*/
/** Set the message handler to call when a suite
 * initialization function returns an error.
 */
void CU_set_suite_init_failure_handler(CU_SuiteInitFailureMessageHandler pSuiteInitFailureHandler)
{
  f_pSuiteInitFailureMessageHandler = pSuiteInitFailureHandler;
}

/*------------------------------------------------------------------------*/
/** Retrieve the message handler called before each test is run. */
CU_TestStartMessageHandler CU_get_test_start_handler(void)
{
  return f_pTestStartMessageHandler;
}

/*------------------------------------------------------------------------*/
/** Retrieve the message handler called after each test is run. */
CU_TestCompleteMessageHandler CU_get_test_complete_handler(void)
{
  return f_pTestCompleteMessageHandler;
}

/*------------------------------------------------------------------------*/
/** Retrieve the message handler called after all tests are run. */
CU_AllTestsCompleteMessageHandler CU_get_all_test_complete_handler(void)
{
  return f_pAllTestsCompleteMessageHandler;
}

/*------------------------------------------------------------------------*/
/** Retrieve the message handler called when a suite
 * initialization error occurs.
 */
CU_SuiteInitFailureMessageHandler CU_get_suite_init_failure_handler(void)
{
  return f_pSuiteInitFailureMessageHandler;
}

/*
 * Functions to get the Run statistics for the Test Run.
 */
/*------------------------------------------------------------------------*/
/** Retrieve the number of suites completed during the previous run.
 * The count is reset each time the client initiates a run.
 * @see CU_get_number_of_tests_run()
 */
unsigned int CU_get_number_of_suites_run(void)
{
  return f_run_summary.nSuitesRun;
}

/*------------------------------------------------------------------------*/
/** Retrieve the number of suites which failed to initialize
 * during the previous run.
 * The count is reset each time the client initiates a run.
 * @see CU_get_number_of_tests_run()
 */
unsigned int CU_get_number_of_suites_failed(void)
{
  return f_run_summary.nSuitesFailed;
}

/*------------------------------------------------------------------------*/
/** Retrieve the number of tests completed during the previous run.
 * The count is reset each time the client initiates a run.
 * @see CU_get_number_of_suites_run()
 */
unsigned int CU_get_number_of_tests_run(void)
{
  return f_run_summary.nTestsRun;
}

/*------------------------------------------------------------------------*/
/** Retrieve the number of tests which contained failed
 * assertions during the previous run.
 * The count is reset each time the client initiates a run.
 * @see CU_get_number_of_suites_run()
 */
unsigned int CU_get_number_of_tests_failed(void)
{
  return f_run_summary.nTestsFailed;
}

/*------------------------------------------------------------------------*/
/** Retrieve the number of assertions processed during the last run.
 * The count is reset each time the client initiates a run.
 * @see CU_get_number_of_successes()
 * @see CU_get_number_of_failures()
 */
unsigned int CU_get_number_of_asserts(void)
{
  return f_run_summary.nAsserts;
}

/*------------------------------------------------------------------------*/
/** Retrieve the number of successful assertions during the last run.
 * The count is reset each time the client initiates a run.
 * @see CU_get_number_of_failures()
 */
unsigned int CU_get_number_of_successes(void)
{
  return (f_run_summary.nAsserts - f_run_summary.nAssertsFailed);
}

/*------------------------------------------------------------------------*/
/** Retrieve the number of failed assertions during the last run.
 * The count is reset each time the client initiates a run.
 * @see CU_get_number_of_successes()
 */
unsigned int CU_get_number_of_failures(void)
{
  return f_run_summary.nAssertsFailed;
}

/*------------------------------------------------------------------------*/
/** Retrieve the number failure records created during 
 * the previous run.  Note that this may be more than the
 * number of failed assertions, since failure records may also
 * be created for failed suite initialization and cleanup.
 * The count is reset each time the client initiates a run.
 */
unsigned int CU_get_number_of_failure_records(void)
{
  return f_run_summary.nFailureRecords;
}

/*------------------------------------------------------------------------*/
/** Retrieve the list of failures which occurred during
 * the last test run.  Note that the pointer returned
 * is invalidated when the client initiates a run using
 * CU_run_all_tests(), CU_run_suite(), or CU_run_test().
 * @see CU_get_number_of_successes()
 */
const CU_pFailureRecord CU_get_failure_list(void)
{
  return f_failure_list;
}

/*------------------------------------------------------------------------*/
/** Retrieve the entire run summary for the last test run.
 * Note that the pFailure pointer in the run summary is
 * invalidated when the client initiates a run using
 * CU_run_all_tests(), CU_run_suite(), or CU_run_test().
 * @see CU_get_number_of_successes()
 */
const CU_pRunSummary CU_get_run_summary(void)
{
  return &f_run_summary;
}

/*
 * Functions for running suites and tests.
 */
/*------------------------------------------------------------------------*/
/** Run all tests in all suites registered in the test registry.
 * The suites are run in the order registered in the test registry.
 * For each registered suite, any initialization function is first
 * called, the suite is run using run_single_suite(), and finally
 * any  suite cleanup function is called.  If an error condition
 * (other than CUE_NOREGISTRY) occurs during the run, the action
 * depends on the current error action (see CU_set_error_action()).
 * @return A CU_ErrorCode indicating the first error condition
 *         encountered while running the tests.
 * @see CU_run_suite() to run the tests in a specific suite.
 * @see CU_run_test() for run a specific test only.
 */
CU_ErrorCode CU_run_all_tests(void)
{
  CU_pTestRegistry pRegistry = CU_get_registry();
  CU_pSuite pSuite = NULL;
  CU_ErrorCode result;
  CU_ErrorCode result2;

  CU_set_error(result = CUE_SUCCESS);
  if (!pRegistry) {
    CU_set_error(result = CUE_NOREGISTRY);
  }
  else {
    /* test run is starting - set flag */
    f_bTestIsRunning = TRUE;

    /* Clear results from the previous run */
    clear_previous_results(&f_run_summary, &f_failure_list);

    pSuite = pRegistry->pSuite;
    while (pSuite && (!result || (CU_get_error_action() == CUEA_IGNORE))) {
      /* if the suite has tests, run it */
      if (pSuite->uiNumberOfTests) {
        result2 = run_single_suite(pSuite, &f_run_summary);
        result = (CUE_SUCCESS == result) ? result2 : result;  /* result = 1st error encountered */
      }
      pSuite = pSuite->pNext;
    }

    /* test run is complete - clear flag */
    f_bTestIsRunning = FALSE;

    if (f_pAllTestsCompleteMessageHandler)
     (*f_pAllTestsCompleteMessageHandler)(f_failure_list);
  }

  return result;
}

/*------------------------------------------------------------------------*/
/** Run all tests in a specified suite.
 * The suite need not be registered in the test registry to be run.
 * Any initialization function for the suite is first called,
 * then the suite is run using run_single_suite(), and any
 * suite cleanup function is called.  Note that the
 * run statistics (counts of tests, successes, failures)
 * are initialized each time this function is called.
 * If an error condition occurs during the run, the action
 * depends on the current error action (see CU_set_error_action()).
 * @param pSuite The suite containing the test (non-NULL)
 * @return A CU_ErrorCode indicating the first error condition
 *         encountered while running the suite.  CU_run_suite()
 *         sets and returns CUE_NOSUITE if pSuite is NULL.  Other
 *         error codes can be set during suite initialization or
 *         cleanup or during test runs.
 * @see CU_run_all_tests() to run all suites.
 * @see CU_run_test() to run a single test in a specific suite.
 */
CU_ErrorCode CU_run_suite(CU_pSuite pSuite)
{
  CU_ErrorCode result;

  CU_set_error(result = CUE_SUCCESS);
  if (!pSuite) {
    CU_set_error(result = CUE_NOSUITE);
  }
  else {
    /* test run is starting - set flag */
    f_bTestIsRunning = TRUE;

    /* Clear results from the previous run */
    clear_previous_results(&f_run_summary, &f_failure_list);

    if (pSuite->uiNumberOfTests)
      result = run_single_suite(pSuite, &f_run_summary);

    /* test run is complete - clear flag */
    f_bTestIsRunning = FALSE;

    if (f_pAllTestsCompleteMessageHandler)
      (*f_pAllTestsCompleteMessageHandler)(f_failure_list);
  }

  return result;
}

/*------------------------------------------------------------------------*/
/** Run a specific test in a specified suite.
 * The suite need not be registered in the test registry to be run,
 * although the test must be registered in the specified suite.
 * Any initialization function for the suite is first
 * called, then the test is run using run_single_test(), and
 * any suite cleanup function is called.  Note that the
 * run statistics (counts of tests, successes, failures)
 * are initialized each time this function is called.
 * @param pSuite The suite containing the test (non-NULL)
 * @param pTest  The test to run (non-NULL)
 * @return A CU_ErrorCode indicating the first error condition
 *         encountered while running the suite.  CU_run_test()
 *         sets and returns CUE_NOSUITE if pSuite is NULL,
 *         CUE_NOTEST if pTest is NULL, and CUE_TEST_NOT_IN_SUITE
 *         if pTest is not registered in pSuite.  Other
 *         error codes can be set during suite initialization or
 *         cleanup or during the test run.
 * @see CU_run_all_tests() to run all tests/suites.
 * @see CU_run_suite() to run all tests in a specific suite.
 */
CU_ErrorCode CU_run_test(CU_pSuite pSuite, CU_pTest pTest)
{
  CU_ErrorCode result;
  CU_ErrorCode result2;

  CU_set_error(result = CUE_SUCCESS);
  if (!pSuite) {
    CU_set_error(result = CUE_NOSUITE);
  }
  else if (!pTest) {
    CU_set_error(result = CUE_NOTEST);
  }
  else if (NULL == CU_get_test_by_name(pTest->pName, pSuite)) {
    CU_set_error(result = CUE_TEST_NOT_IN_SUITE);
  }
  else {
    /* test run is starting - set flag */
    f_bTestIsRunning = TRUE;

    /* Clear results from the previous run */
    clear_previous_results(&f_run_summary, &f_failure_list);

    f_pCurTest = NULL;
    f_pCurSuite = pSuite;

    if ((pSuite->pInitializeFunc) && (*pSuite->pInitializeFunc)()) {
      if (f_pSuiteInitFailureMessageHandler) {
        (*f_pSuiteInitFailureMessageHandler)(pSuite);
      }
      f_run_summary.nSuitesFailed++;
      add_failure(&f_failure_list, &f_run_summary,
                  0, "Suite Initialization failed - Test Skipped", "CUnit System", pSuite, pTest);
      CU_set_error(result = CUE_SINIT_FAILED);
      /* test run is complete - clear flag */
      f_bTestIsRunning = FALSE;
    }
    /* reach here if no suite initialization, or if it succeeded */
    else {
      result2 = run_single_test(pTest, &f_run_summary);
      result = (CUE_SUCCESS == result) ? result2 : result;

      if ((pSuite->pCleanupFunc) && (*pSuite->pCleanupFunc)()) {
        f_run_summary.nSuitesFailed++;
        add_failure(&f_failure_list, &f_run_summary,
                    0, "Suite cleanup failed.", "CUnit System", pSuite, pTest);
        result = (CUE_SUCCESS == result) ? CUE_SCLEAN_FAILED : result;
        CU_set_error(CUE_SCLEAN_FAILED);
      }

      /* test run is complete - clear flag */
      f_bTestIsRunning = FALSE;

      if (f_pAllTestsCompleteMessageHandler)
        (*f_pAllTestsCompleteMessageHandler)(f_failure_list);

      f_pCurSuite = NULL;
    }
  }

  return result;
}

/*------------------------------------------------------------------------*/
/** Initialize the run summary information stored from
 * the previous test run.  Resets the run counts to zero,
 * and frees any memory associated with failure records.
 * Calling this function multiple times, while inefficient,
 * will not cause an error condition.
 * @see clear_previous_results()
 */
void CU_clear_previous_results(void)
{
  clear_previous_results(&f_run_summary, &f_failure_list);
}

/*------------------------------------------------------------------------*/
/** Retrieve a pointer to the currently-running suite (NULL if none).
 */
CU_pSuite CU_get_current_suite(void)
{
  return f_pCurSuite;
}

/*------------------------------------------------------------------------*/
/** Retrieve a pointer to the currently-running test (NULL if none).
 */
CU_pTest CU_get_current_test(void)
{
  return f_pCurTest;
}

/*------------------------------------------------------------------------*/
/** Returns <CODE>TRUE</CODE> if a test run is in progress,
 * <CODE>TRUE</CODE> otherwise.
 */
BOOL CU_is_test_running(void)
{
  return f_bTestIsRunning;
}

/*------------------------------------------------------------------------*/
/** Record a failed test.
 * This function is called whenever a test fails to record the
 * details of the failure.  This includes user assertion failures
 * and system errors such as failure to initialize a suite.
 * @param ppFailure    Pointer to head of linked list of failure
 *                     records to append with new failure record.
 *                     If NULL, it will be set to point to the new
 *                     failure record.
 * @param pRunSummary  Pointer to CU_RunSummary keeping track of failure records
 *                     (ignored if NULL).
 * @param uiLineNumber Line number of the failure, if applicable.
 * @param szCondition  Description of failure condition
 * @param szFileName   Name of file, if applicable
 * @param pSuite       The suite being run at time of failure
 * @param pTest        The test being run at time of failure
 */
void add_failure(CU_pFailureRecord* ppFailure, CU_pRunSummary pRunSummary,
                 unsigned int uiLineNumber, char szCondition[],
                 char szFileName[], CU_pSuite pSuite, CU_pTest pTest)
{
  CU_pFailureRecord pFailureNew = NULL;
  CU_pFailureRecord pTemp = NULL;

  pFailureNew = (CU_pFailureRecord)CU_MALLOC(sizeof(CU_FailureRecord));

  if (!pFailureNew)
    return;

  pFailureNew->strFileName = NULL;
  pFailureNew->strCondition = NULL;
  if (szFileName) {
    pFailureNew->strFileName = (char*)CU_MALLOC(strlen(szFileName) + 1);
    if(!pFailureNew->strFileName) {
      CU_FREE(pFailureNew);
      return;
    }
    strcpy(pFailureNew->strFileName, szFileName);
  }

  if (szCondition) {
    pFailureNew->strCondition = (char*)CU_MALLOC(strlen(szCondition) + 1);
    if (!pFailureNew->strCondition) {
      if(pFailureNew->strFileName) {
        CU_FREE(pFailureNew->strFileName);
      }
      CU_FREE(pFailureNew);
      return;
    }
    strcpy(pFailureNew->strCondition, szCondition);
  }

  pFailureNew->uiLineNumber = uiLineNumber;
  pFailureNew->pTest = pTest;
  pFailureNew->pSuite = pSuite;
  pFailureNew->pNext = NULL;
  pFailureNew->pPrev = NULL;

  pTemp = *ppFailure;
  if (pTemp) {
    while (pTemp->pNext) {
      pTemp = pTemp->pNext;
    }
    pTemp->pNext = pFailureNew;
    pFailureNew->pPrev = pTemp;
  }
  else {
    *ppFailure = pFailureNew;
  }

  if (NULL != pRunSummary)
    ++(pRunSummary->nFailureRecords);
  f_last_failure = pFailureNew;
}

/*
 *  Local function for result set initialization/cleanup.
 */
/*------------------------------------------------------------------------*/
/** Initialize the run summary information in the
 * specified structure.  Resets the run counts to zero,
 * and calls cleanup_failure_list() if failures
 * were recorded by the last test run.
 * Calling this function multiple times, while inefficient,
 * will not cause an error condition.
 * @param pRunSummary CU_RunSummary to initialize.
 * @see CU_clear_previous_results()
 */
static void clear_previous_results(CU_pRunSummary pRunSummary, CU_pFailureRecord* ppFailure)
{
  pRunSummary->nSuitesRun = 0;
  pRunSummary->nSuitesFailed = 0;
  pRunSummary->nTestsRun = 0;
  pRunSummary->nTestsFailed = 0;
  pRunSummary->nAsserts = 0;
  pRunSummary->nAssertsFailed = 0;
  pRunSummary->nFailureRecords = 0;

  if (NULL != *ppFailure)
    cleanup_failure_list(ppFailure);

  f_last_failure = NULL;
}

/*------------------------------------------------------------------------*/
/** Free all memory allocated for the linked list of
 * test failure records.  pFailure is reset to NULL
 * after its list is cleaned up.
 * @param ppFailure Pointer to head of linked list of
 *                  CU_pFailureRecords to clean.
 * @see CU_clear_previous_results()
 */
static void cleanup_failure_list(CU_pFailureRecord* ppFailure)
{
  CU_pFailureRecord pCurFailure = NULL;
  CU_pFailureRecord pNextFailure = NULL;

  pCurFailure = *ppFailure;

  while (pCurFailure) {

    if (pCurFailure->strCondition)
      CU_FREE(pCurFailure->strCondition);

    if (pCurFailure->strFileName)
      CU_FREE(pCurFailure->strFileName);

    pNextFailure = pCurFailure->pNext;
    CU_FREE(pCurFailure);
    pCurFailure = pNextFailure;
  }

  *ppFailure = NULL;
}

/*------------------------------------------------------------------------*/
/** Run all tests in a specified suite.
 * Internal function to run all tests in a suite.  The suite
 * need not be registered in the test registry to be run.
 * If the CUnit system is in an error condition after running
 * a test, no additional tests are run.
 * @param pSuite The suite containing the test (non-NULL).
 * @param pRunSummary The CU_RunSummary to receive the results (non-NULL).
 * @return A CU_ErrorCode indicating the status of the run.
 * @see CU_run_suite() for public interface function.
 * @see CU_run_all_tests() for running all suites.
 */
static CU_ErrorCode run_single_suite(CU_pSuite pSuite, CU_pRunSummary pRunSummary)
{
  CU_pTest pTest = NULL;
  CU_ErrorCode result;
  CU_ErrorCode result2;

  assert(pSuite);
  assert(pRunSummary);

  f_pCurTest = NULL;
  f_pCurSuite = pSuite;

  CU_set_error(result = CUE_SUCCESS);
  /* call the suite initialization function, if any */
  if ((pSuite->pInitializeFunc) && (*pSuite->pInitializeFunc)()) {
    if (f_pSuiteInitFailureMessageHandler) {
      (*f_pSuiteInitFailureMessageHandler)(pSuite);
    }
    pRunSummary->nSuitesFailed++;
    add_failure(&f_failure_list, &f_run_summary,
                0, "Suite Initialization failed - Suite Skipped", "CUnit System", pSuite, NULL);
    CU_set_error(result = CUE_SINIT_FAILED);
  }
  /* reach here if no suite initialization, or if it succeeded */
  else {
    pTest = pSuite->pTest;
    while (pTest && (!result || (CU_get_error_action() == CUEA_IGNORE))) {
      result2 = run_single_test(pTest, pRunSummary);
      result = (CUE_SUCCESS == result) ? result2 : result;
      pTest = pTest->pNext;
    }
    pRunSummary->nSuitesRun++;

    /* call the suite cleanup function, if any */
    if ((pSuite->pCleanupFunc) && (*pSuite->pCleanupFunc)()) {
      pRunSummary->nSuitesFailed++;
      add_failure(&f_failure_list, &f_run_summary,
                  0, "Suite cleanup failed.", "CUnit System", pSuite, pTest);
      CU_set_error(CUE_SCLEAN_FAILED);
      result = (CUE_SUCCESS == result) ? CUE_SCLEAN_FAILED : result;
    }
  }

  f_pCurSuite = NULL;
  return result;
}

/*------------------------------------------------------------------------*/
/** Run a specific test.
 * Internal function to run a test case.  This includes
 * calling any handler to be run before executing the test,
 * running the test's function (if any), and calling any
 * handler to be run after executing a test.
 * @param pTest The test to be run (non-NULL).
 * @param pRunSummary The CU_RunSummary to receive the results (non-NULL).
 * @return A CU_ErrorCode indicating the status of the run.
 * @see CU_run_test() for public interface function.
 * @see CU_run_all_tests() for running all suites.
 */
CU_ErrorCode run_single_test(CU_pTest pTest, CU_pRunSummary pRunSummary)
{
  unsigned int nStartFailures;
  /* keep track of the last failure BEFORE running the test */
  CU_pFailureRecord pLastFailure = f_last_failure;
  jmp_buf buf;

  assert(pTest);
  assert(pRunSummary);

  nStartFailures = pRunSummary->nAssertsFailed;

  CU_set_error(CUE_SUCCESS);
  f_pCurTest = pTest;

  if (f_pTestStartMessageHandler)
    (*f_pTestStartMessageHandler)(f_pCurTest, f_pCurSuite);

  /* set jmp_buf and run test */
  pTest->pJumpBuf = &buf;
  if (!setjmp(buf))
    if (pTest->pTestFunc)
      (*pTest->pTestFunc)();

  pRunSummary->nTestsRun++;

  /* if additional assertions have failed... */
  if (pRunSummary->nAssertsFailed > nStartFailures) {
    pRunSummary->nTestsFailed++;
    if (pLastFailure)
      pLastFailure = pLastFailure->pNext;  /* was a failure before - go to next one */
    else
      pLastFailure = f_failure_list;       /* no previous failure - go to 1st one */
  }
  else
    pLastFailure = NULL;                   /* no additional failure - set to NULL */

  if (f_pTestCompleteMessageHandler)
    (*f_pTestCompleteMessageHandler)(f_pCurTest, f_pCurSuite, pLastFailure);

  pTest->pJumpBuf = NULL;
  f_pCurTest = NULL;

  return CU_get_error();
}

/** @} */

#ifdef CUNIT_BUILD_TESTS
#include "test_cunit.h"

typedef enum TET {
  TEST_START = 1,
  TEST_COMPLETE,
  ALL_TESTS_COMPLETE,
  SUITE_INIT_FAILED,
} TestEventType;

typedef struct TE {
  TestEventType     type;
  CU_pSuite         pSuite;
  CU_pTest          pTest;
  CU_pFailureRecord pFailure;
  struct TE *       pNext;
} TestEvent, * pTestEvent;

static int f_nTestEvents = 0;
static pTestEvent f_pFirstEvent = NULL;

static void add_test_event(TestEventType type, CU_pSuite psuite,
                           CU_pTest ptest, CU_pFailureRecord pfailure)
{
  pTestEvent pNewEvent = (pTestEvent)malloc(sizeof(TestEvent));
  pTestEvent pNextEvent = NULL;

  pNewEvent->type = type;
  pNewEvent->pSuite = psuite;
  pNewEvent->pTest = ptest;
  pNewEvent->pFailure = pfailure;
  pNewEvent->pNext = NULL;

  pNextEvent = f_pFirstEvent;
  if (pNextEvent) {
    while (pNextEvent->pNext) {
      pNextEvent = pNextEvent->pNext;
    }
    pNextEvent->pNext = pNewEvent;
  }
  else {
    f_pFirstEvent = pNewEvent;
  }
  ++f_nTestEvents;
}

static void clear_test_events(void)
{
  pTestEvent pCurrentEvent = f_pFirstEvent;
  pTestEvent pNextEvent = NULL;

  while (pCurrentEvent) {
    pNextEvent = pCurrentEvent->pNext;
    free(pCurrentEvent);
    pCurrentEvent = pNextEvent;
  }

  f_pFirstEvent = NULL;
  f_nTestEvents = 0;
}

static void test_start_handler(const CU_pTest pTest, const CU_pSuite pSuite)
{
  TEST(CU_is_test_running());
  TEST(pSuite == CU_get_current_suite());
  TEST(pTest == CU_get_current_test());

  add_test_event(TEST_START, pSuite, pTest, NULL);
}

static void test_complete_handler(const CU_pTest pTest, const CU_pSuite pSuite,
                                  const CU_pFailureRecord pFailure)
{
  TEST(CU_is_test_running());
  TEST(pSuite == CU_get_current_suite());
  TEST(pTest == CU_get_current_test());

  add_test_event(TEST_COMPLETE, pSuite, pTest, pFailure);
}

static void test_all_complete_handler(const CU_pFailureRecord pFailure)
{
  TEST(!CU_is_test_running());

  add_test_event(ALL_TESTS_COMPLETE, NULL, NULL, pFailure);
}

static void suite_init_failure_handler(const CU_pSuite pSuite)
{
  TEST(CU_is_test_running());
  TEST(pSuite == CU_get_current_suite());

  add_test_event(SUITE_INIT_FAILED, pSuite, NULL, NULL);
}

void test_succeed(void) { CU_TEST(TRUE); }
void test_fail(void) { CU_TEST(FALSE); }
int suite_fail(void) { return 1; }

/*-------------------------------------------------*/
/* tests:
 *    CU_set_test_start_handler()
 *    CU_set_test_complete_handler()
 *    CU_set_all_test_complete_handler()
 *    CU_set_suite_init_failure_handler()
 *    CU_get_test_start_handler()
 *    CU_get_test_complete_handler()
 *    CU_get_all_test_complete_handler()
 *    CU_get_suite_init_failure_handler()
 */
static void test_message_handlers(void)
{
  CU_pSuite pSuite1 = NULL;
  CU_pSuite pSuite2 = NULL;
  CU_pTest  pTest1 = NULL;
  CU_pTest  pTest2 = NULL;
  CU_pTest  pTest3 = NULL;
  pTestEvent pEvent = NULL;
  CU_pRunSummary pRunSummary = NULL;

  TEST(!CU_is_test_running());

  /* handlers should be NULL on startup */
  TEST(NULL == CU_get_test_start_handler());
  TEST(NULL == CU_get_test_complete_handler());
  TEST(NULL == CU_get_all_test_complete_handler());
  TEST(NULL == CU_get_suite_init_failure_handler());

  /* register some suites and tests */
  CU_initialize_registry();
  pSuite1 = CU_add_suite("suite1", NULL, NULL);
  pTest1 = CU_add_test(pSuite1, "test1", test_succeed);
  pTest2 = CU_add_test(pSuite1, "test2", test_fail);
  pTest3 = CU_add_test(pSuite1, "test3", test_succeed);
  pSuite2 = CU_add_suite("suite2", suite_fail, NULL);
  CU_add_test(pSuite2, "test4", test_succeed);

  TEST_FATAL(CUE_SUCCESS == CU_get_error());

  /* first run tests without handlers set */
  clear_test_events();
  CU_run_all_tests();

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(1 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(3 == CU_get_number_of_tests_run());
  TEST(1 == CU_get_number_of_tests_failed());
  TEST(3 == CU_get_number_of_asserts());
  TEST(2 == CU_get_number_of_successes());
  TEST(1 == CU_get_number_of_failures());
  TEST(2 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  /* set handlers to local functions */
  CU_set_test_start_handler(test_start_handler);
  CU_set_test_complete_handler(test_complete_handler);
  CU_set_all_test_complete_handler(test_all_complete_handler);
  CU_set_suite_init_failure_handler(suite_init_failure_handler);

  /* confirm handlers set properly */
  TEST(test_start_handler == CU_get_test_start_handler());
  TEST(test_complete_handler == CU_get_test_complete_handler());
  TEST(test_all_complete_handler == CU_get_all_test_complete_handler());
  TEST(suite_init_failure_handler == CU_get_suite_init_failure_handler());

  /* run tests again with handlers set */
  clear_test_events();
  CU_run_all_tests();

  TEST(8 == f_nTestEvents);
  if (8 == f_nTestEvents) {
    pEvent = f_pFirstEvent;
    TEST(TEST_START == pEvent->type);
    TEST(pSuite1 == pEvent->pSuite);
    TEST(pTest1 == pEvent->pTest);
    TEST(NULL == pEvent->pFailure);

    pEvent = pEvent->pNext;
    TEST(TEST_COMPLETE == pEvent->type);
    TEST(pSuite1 == pEvent->pSuite);
    TEST(pTest1 == pEvent->pTest);
    TEST(NULL == pEvent->pFailure);

    pEvent = pEvent->pNext;
    TEST(TEST_START == pEvent->type);
    TEST(pSuite1 == pEvent->pSuite);
    TEST(pTest2 == pEvent->pTest);
    TEST(NULL == pEvent->pFailure);

    pEvent = pEvent->pNext;
    TEST(TEST_COMPLETE == pEvent->type);
    TEST(pSuite1 == pEvent->pSuite);
    TEST(pTest2 == pEvent->pTest);
    TEST(NULL != pEvent->pFailure);

    pEvent = pEvent->pNext;
    TEST(TEST_START == pEvent->type);
    TEST(pSuite1 == pEvent->pSuite);
    TEST(pTest3 == pEvent->pTest);
    TEST(NULL == pEvent->pFailure);

    pEvent = pEvent->pNext;
    TEST(TEST_COMPLETE == pEvent->type);
    TEST(pSuite1 == pEvent->pSuite);
    TEST(pTest3 == pEvent->pTest);
    TEST(NULL == pEvent->pFailure);

    pEvent = pEvent->pNext;
    TEST(SUITE_INIT_FAILED == pEvent->type);
    TEST(pSuite2 == pEvent->pSuite);
    TEST(NULL == pEvent->pTest);
    TEST(NULL == pEvent->pFailure);

    pEvent = pEvent->pNext;
    TEST(ALL_TESTS_COMPLETE == pEvent->type);
    TEST(NULL == pEvent->pSuite);
    TEST(NULL == pEvent->pTest);
    TEST(NULL != pEvent->pFailure);
    TEST(NULL != pEvent->pFailure->pNext);
    TEST(NULL == pEvent->pFailure->pNext->pNext);
    TEST(pEvent->pFailure == CU_get_failure_list());
  }

  TEST(1 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(3 == CU_get_number_of_tests_run());
  TEST(1 == CU_get_number_of_tests_failed());
  TEST(3 == CU_get_number_of_asserts());
  TEST(2 == CU_get_number_of_successes());
  TEST(1 == CU_get_number_of_failures());
  TEST(2 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  /* clear handlers and run again */
  CU_set_test_start_handler(NULL);
  CU_set_test_complete_handler(NULL);
  CU_set_all_test_complete_handler(NULL);
  CU_set_suite_init_failure_handler(NULL);

  TEST(NULL == CU_get_test_start_handler());
  TEST(NULL == CU_get_test_complete_handler());
  TEST(NULL == CU_get_all_test_complete_handler());
  TEST(NULL == CU_get_suite_init_failure_handler());

  clear_test_events();
  CU_run_all_tests();

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(1 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(3 == CU_get_number_of_tests_run());
  TEST(1 == CU_get_number_of_tests_failed());
  TEST(3 == CU_get_number_of_asserts());
  TEST(2 == CU_get_number_of_successes());
  TEST(1 == CU_get_number_of_failures());
  TEST(2 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  CU_cleanup_registry();
  clear_test_events();
}

static BOOL f_exit_called = FALSE;

/* intercept exit for testing of CUEA_ABORT action */
void test_exit(int status)
{
  (void) status;  /* not used */
  f_exit_called = TRUE;
}

/*-------------------------------------------------*/
static void test_CU_run_all_tests(void)
{
  CU_pSuite pSuite1 = NULL;
  CU_pSuite pSuite2 = NULL;
  CU_pSuite pSuite3 = NULL;
  CU_pRunSummary pRunSummary = NULL;

  /* error - uninitialized registry  (CUEA_IGNORE) */
  CU_cleanup_registry();
  CU_set_error_action(CUEA_IGNORE);

  TEST(CUE_NOREGISTRY == CU_run_all_tests());
  TEST(CUE_NOREGISTRY == CU_get_error());

  /* error - uninitialized registry  (CUEA_FAIL) */
  CU_cleanup_registry();
  CU_set_error_action(CUEA_FAIL);

  TEST(CUE_NOREGISTRY == CU_run_all_tests());
  TEST(CUE_NOREGISTRY == CU_get_error());

  /* error - uninitialized registry  (CUEA_ABORT) */
  CU_cleanup_registry();
  CU_set_error_action(CUEA_ABORT);

  f_exit_called = FALSE;
  CU_run_all_tests();
  TEST(TRUE == f_exit_called);
  f_exit_called = FALSE;

  /* run with no tests registered */
  CU_set_error_action(CUEA_IGNORE);
  CU_initialize_registry();
  clear_test_events();
  TEST(CUE_SUCCESS == CU_run_all_tests());

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(0 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(0 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(0 == CU_get_number_of_failure_records());
  TEST(NULL == CU_get_failure_list());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  /* register some suites and tests */
  CU_initialize_registry();
  pSuite1 = CU_add_suite("suite1", NULL, NULL);
  CU_add_test(pSuite1, "test1", test_succeed);
  CU_add_test(pSuite1, "test2", test_fail);
  CU_add_test(pSuite1, "test3", test_succeed);
  CU_add_test(pSuite1, "test4", test_fail);
  CU_add_test(pSuite1, "test5", test_succeed);
  pSuite2 = CU_add_suite("suite2", suite_fail, NULL);
  CU_add_test(pSuite2, "test6", test_succeed);
  CU_add_test(pSuite2, "test7", test_succeed);
  pSuite3 = CU_add_suite("suite3", NULL, NULL);
  CU_add_test(pSuite3, "test8", test_fail);
  CU_add_test(pSuite3, "test9", test_succeed);

  TEST_FATAL(CUE_SUCCESS == CU_get_error());

  /* run all tests (CUEA_IGNORE) */
  clear_test_events();
  CU_set_error_action(CUEA_IGNORE);
  TEST(CUE_SINIT_FAILED == CU_run_all_tests());

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(2 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(7 == CU_get_number_of_tests_run());
  TEST(3 == CU_get_number_of_tests_failed());
  TEST(7 == CU_get_number_of_asserts());
  TEST(4 == CU_get_number_of_successes());
  TEST(3 == CU_get_number_of_failures());
  TEST(4 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  /* run all tests (CUEA_FAIL) */
  clear_test_events();
  CU_set_error_action(CUEA_FAIL);
  TEST(CUE_SINIT_FAILED == CU_run_all_tests());

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(1 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(5 == CU_get_number_of_tests_run());
  TEST(2 == CU_get_number_of_tests_failed());
  TEST(5 == CU_get_number_of_asserts());
  TEST(3 == CU_get_number_of_successes());
  TEST(2 == CU_get_number_of_failures());
  TEST(3 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  /* run all tests (CUEA_ABORT) */
  clear_test_events();
  CU_set_error_action(CUEA_ABORT);

  f_exit_called = FALSE;
  TEST(CUE_SINIT_FAILED == CU_run_all_tests());
  TEST(TRUE == f_exit_called);
  f_exit_called = FALSE;

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(1 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(5 == CU_get_number_of_tests_run());
  TEST(2 == CU_get_number_of_tests_failed());
  TEST(5 == CU_get_number_of_asserts());
  TEST(3 == CU_get_number_of_successes());
  TEST(2 == CU_get_number_of_failures());
  TEST(3 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  /* clean up after testing */
  CU_set_error_action(CUEA_IGNORE);
  CU_cleanup_registry();
  clear_test_events();
}

/*-------------------------------------------------*/
static void test_CU_run_suite(void)
{
  CU_pSuite pSuite1 = NULL;
  CU_pSuite pSuite2 = NULL;
  CU_pSuite pSuite3 = NULL;
  CU_pSuite pSuite4 = NULL;
  CU_pRunSummary pRunSummary = NULL;

  /* error - NULL suite (CUEA_IGNORE) */
  CU_set_error_action(CUEA_IGNORE);

  TEST(CUE_NOSUITE == CU_run_suite(NULL));
  TEST(CUE_NOSUITE == CU_get_error());

  /* error - NULL suite (CUEA_FAIL) */
  CU_set_error_action(CUEA_FAIL);

  TEST(CUE_NOSUITE == CU_run_suite(NULL));
  TEST(CUE_NOSUITE == CU_get_error());

  /* error - NULL suite (CUEA_ABORT) */
  CU_set_error_action(CUEA_ABORT);

  f_exit_called = FALSE;
  CU_run_suite(NULL);
  TEST(TRUE == f_exit_called);
  f_exit_called = FALSE;

  /* register some suites and tests */
  CU_initialize_registry();
  pSuite1 = CU_add_suite("suite1", NULL, NULL);
  CU_add_test(pSuite1, "test1", test_succeed);
  CU_add_test(pSuite1, "test2", test_fail);
  CU_add_test(pSuite1, "test3", test_succeed);
  CU_add_test(pSuite1, "test4", test_fail);
  CU_add_test(pSuite1, "test5", test_succeed);
  pSuite2 = CU_add_suite("suite2", suite_fail, NULL);
  CU_add_test(pSuite2, "test6", test_succeed);
  CU_add_test(pSuite2, "test7", test_succeed);
  pSuite3 = CU_add_suite("suite3", NULL, suite_fail);
  CU_add_test(pSuite3, "test8", test_fail);
  CU_add_test(pSuite3, "test9", test_succeed);
  pSuite4 = CU_add_suite("suite4", NULL, NULL);

  TEST_FATAL(CUE_SUCCESS == CU_get_error());

  /* run each suite (CUEA_IGNORE) */
  clear_test_events();
  CU_set_error_action(CUEA_IGNORE);

  TEST(CUE_SUCCESS == CU_run_suite(pSuite1));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(1 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(5 == CU_get_number_of_tests_run());
  TEST(2 == CU_get_number_of_tests_failed());
  TEST(5 == CU_get_number_of_asserts());
  TEST(3 == CU_get_number_of_successes());
  TEST(2 == CU_get_number_of_failures());
  TEST(2 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SINIT_FAILED == CU_run_suite(pSuite2));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(0 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(0 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(1 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SCLEAN_FAILED == CU_run_suite(pSuite3));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(1 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(2 == CU_get_number_of_tests_run());
  TEST(1 == CU_get_number_of_tests_failed());
  TEST(2 == CU_get_number_of_asserts());
  TEST(1 == CU_get_number_of_successes());
  TEST(1 == CU_get_number_of_failures());
  TEST(2 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SUCCESS == CU_run_suite(pSuite4));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(0 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(0 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(0 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  /* run each suite (CUEA_FAIL) */
  clear_test_events();
  CU_set_error_action(CUEA_FAIL);

  TEST(CUE_SUCCESS == CU_run_suite(pSuite1));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(1 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(5 == CU_get_number_of_tests_run());
  TEST(2 == CU_get_number_of_tests_failed());
  TEST(5 == CU_get_number_of_asserts());
  TEST(3 == CU_get_number_of_successes());
  TEST(2 == CU_get_number_of_failures());
  TEST(2 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SINIT_FAILED == CU_run_suite(pSuite2));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(0 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(0 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(1 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SCLEAN_FAILED == CU_run_suite(pSuite3));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(1 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(2 == CU_get_number_of_tests_run());
  TEST(1 == CU_get_number_of_tests_failed());
  TEST(2 == CU_get_number_of_asserts());
  TEST(1 == CU_get_number_of_successes());
  TEST(1 == CU_get_number_of_failures());
  TEST(2 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SUCCESS == CU_run_suite(pSuite4));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(0 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(0 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(0 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  /* run each suite (CUEA_ABORT) */
  clear_test_events();
  CU_set_error_action(CUEA_ABORT);

  f_exit_called = FALSE;
  TEST(CUE_SUCCESS == CU_run_suite(pSuite1));
  TEST(FALSE == f_exit_called);
  f_exit_called = FALSE;

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(1 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(5 == CU_get_number_of_tests_run());
  TEST(2 == CU_get_number_of_tests_failed());
  TEST(5 == CU_get_number_of_asserts());
  TEST(3 == CU_get_number_of_successes());
  TEST(2 == CU_get_number_of_failures());
  TEST(2 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SINIT_FAILED == CU_run_suite(pSuite2));
  TEST(TRUE == f_exit_called);
  f_exit_called = FALSE;

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(0 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(0 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(1 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SCLEAN_FAILED == CU_run_suite(pSuite3));
  TEST(TRUE == f_exit_called);
  f_exit_called = FALSE;

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(1 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(2 == CU_get_number_of_tests_run());
  TEST(1 == CU_get_number_of_tests_failed());
  TEST(2 == CU_get_number_of_asserts());
  TEST(1 == CU_get_number_of_successes());
  TEST(1 == CU_get_number_of_failures());
  TEST(2 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SUCCESS == CU_run_suite(pSuite4));
  TEST(FALSE == f_exit_called);
  f_exit_called = FALSE;

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(0 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(0 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(0 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  /* clean up after testing */
  CU_set_error_action(CUEA_IGNORE);
  CU_cleanup_registry();
  clear_test_events();
}

/*-------------------------------------------------*/
static void test_CU_run_test(void)
{
  CU_pSuite pSuite1 = NULL;
  CU_pSuite pSuite2 = NULL;
  CU_pSuite pSuite3 = NULL;
  CU_pTest pTest1 = NULL;
  CU_pTest pTest2 = NULL;
  CU_pTest pTest3 = NULL;
  CU_pTest pTest4 = NULL;
  CU_pTest pTest5 = NULL;
  CU_pTest pTest6 = NULL;
  CU_pTest pTest7 = NULL;
  CU_pTest pTest8 = NULL;
  CU_pTest pTest9 = NULL;
  CU_pRunSummary pRunSummary = NULL;

  /* register some suites and tests */
  CU_initialize_registry();
  pSuite1 = CU_add_suite("suite1", NULL, NULL);
  pTest1 = CU_add_test(pSuite1, "test1", test_succeed);
  pTest2 = CU_add_test(pSuite1, "test2", test_fail);
  pTest3 = CU_add_test(pSuite1, "test3", test_succeed);
  pTest4 = CU_add_test(pSuite1, "test4", test_fail);
  pTest5 = CU_add_test(pSuite1, "test5", test_succeed);
  pSuite2 = CU_add_suite("suite2", suite_fail, NULL);
  pTest6 = CU_add_test(pSuite2, "test6", test_succeed);
  pTest7 = CU_add_test(pSuite2, "test7", test_succeed);
  pSuite3 = CU_add_suite("suite3", NULL, suite_fail);
  pTest8 = CU_add_test(pSuite3, "test8", test_fail);
  pTest9 = CU_add_test(pSuite3, "test9", test_succeed);

  TEST_FATAL(CUE_SUCCESS == CU_get_error());

  /* error - NULL suite (CUEA_IGNORE) */
  CU_set_error_action(CUEA_IGNORE);

  TEST(CUE_NOSUITE == CU_run_test(NULL, pTest1));
  TEST(CUE_NOSUITE == CU_get_error());

  /* error - NULL suite (CUEA_FAIL) */
  CU_set_error_action(CUEA_FAIL);

  TEST(CUE_NOSUITE == CU_run_test(NULL, pTest1));
  TEST(CUE_NOSUITE == CU_get_error());

  /* error - NULL test (CUEA_ABORT) */
  CU_set_error_action(CUEA_ABORT);

  f_exit_called = FALSE;
  CU_run_test(NULL, pTest1);
  TEST(TRUE == f_exit_called);
  f_exit_called = FALSE;

  /* error - NULL test (CUEA_IGNORE) */
  CU_set_error_action(CUEA_IGNORE);

  TEST(CUE_NOTEST == CU_run_test(pSuite1, NULL));
  TEST(CUE_NOTEST == CU_get_error());

  /* error - NULL test (CUEA_FAIL) */
  CU_set_error_action(CUEA_FAIL);

  TEST(CUE_NOTEST == CU_run_test(pSuite1, NULL));
  TEST(CUE_NOTEST == CU_get_error());

  /* error - NULL test (CUEA_ABORT) */
  CU_set_error_action(CUEA_ABORT);

  f_exit_called = FALSE;
  CU_run_test(pSuite1, NULL);
  TEST(TRUE == f_exit_called);
  f_exit_called = FALSE;

  /* error - test not in suite (CUEA_IGNORE) */
  CU_set_error_action(CUEA_IGNORE);

  TEST(CUE_TEST_NOT_IN_SUITE == CU_run_test(pSuite3, pTest1));
  TEST(CUE_TEST_NOT_IN_SUITE == CU_get_error());

  /* error - NULL test (CUEA_FAIL) */
  CU_set_error_action(CUEA_FAIL);

  TEST(CUE_TEST_NOT_IN_SUITE == CU_run_test(pSuite3, pTest1));
  TEST(CUE_TEST_NOT_IN_SUITE == CU_get_error());

  /* error - NULL test (CUEA_ABORT) */
  CU_set_error_action(CUEA_ABORT);

  f_exit_called = FALSE;
  CU_run_test(pSuite3, pTest1);
  TEST(TRUE == f_exit_called);
  f_exit_called = FALSE;

  /* run each test (CUEA_IGNORE) */
  clear_test_events();
  CU_set_error_action(CUEA_IGNORE);

  TEST(CUE_SUCCESS == CU_run_test(pSuite1, pTest1));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(1 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(1 == CU_get_number_of_asserts());
  TEST(1 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(0 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SUCCESS == CU_run_test(pSuite1, pTest2));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(1 == CU_get_number_of_tests_run());
  TEST(1 == CU_get_number_of_tests_failed());
  TEST(1 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(1 == CU_get_number_of_failures());
  TEST(1 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SUCCESS == CU_run_test(pSuite1, pTest3));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(1 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(1 == CU_get_number_of_asserts());
  TEST(1 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(0 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SUCCESS == CU_run_test(pSuite1, pTest4));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(1 == CU_get_number_of_tests_run());
  TEST(1 == CU_get_number_of_tests_failed());
  TEST(1 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(1 == CU_get_number_of_failures());
  TEST(1 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SUCCESS == CU_run_test(pSuite1, pTest5));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(1 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(1 == CU_get_number_of_asserts());
  TEST(1 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(0 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SINIT_FAILED == CU_run_test(pSuite2, pTest6));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(0 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(0 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(1 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SINIT_FAILED == CU_run_test(pSuite2, pTest7));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(0 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(0 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(1 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SCLEAN_FAILED == CU_run_test(pSuite3, pTest8));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(1 == CU_get_number_of_tests_run());
  TEST(1 == CU_get_number_of_tests_failed());
  TEST(1 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(1 == CU_get_number_of_failures());
  TEST(2 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SCLEAN_FAILED == CU_run_test(pSuite3, pTest9));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(1 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(1 == CU_get_number_of_asserts());
  TEST(1 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(1 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  /* run each test (CUEA_FAIL) */
  clear_test_events();
  CU_set_error_action(CUEA_FAIL);

  TEST(CUE_SUCCESS == CU_run_test(pSuite1, pTest1));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(1 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(1 == CU_get_number_of_asserts());
  TEST(1 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(0 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SUCCESS == CU_run_test(pSuite1, pTest2));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(1 == CU_get_number_of_tests_run());
  TEST(1 == CU_get_number_of_tests_failed());
  TEST(1 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(1 == CU_get_number_of_failures());
  TEST(1 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SUCCESS == CU_run_test(pSuite1, pTest3));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(1 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(1 == CU_get_number_of_asserts());
  TEST(1 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(0 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SUCCESS == CU_run_test(pSuite1, pTest4));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(1 == CU_get_number_of_tests_run());
  TEST(1 == CU_get_number_of_tests_failed());
  TEST(1 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(1 == CU_get_number_of_failures());
  TEST(1 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SUCCESS == CU_run_test(pSuite1, pTest5));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(1 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(1 == CU_get_number_of_asserts());
  TEST(1 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(0 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SINIT_FAILED == CU_run_test(pSuite2, pTest6));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(0 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(0 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(1 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SINIT_FAILED == CU_run_test(pSuite2, pTest7));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(0 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(0 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(1 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SCLEAN_FAILED == CU_run_test(pSuite3, pTest8));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(1 == CU_get_number_of_tests_run());
  TEST(1 == CU_get_number_of_tests_failed());
  TEST(1 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(1 == CU_get_number_of_failures());
  TEST(2 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SCLEAN_FAILED == CU_run_test(pSuite3, pTest9));

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(1 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(1 == CU_get_number_of_asserts());
  TEST(1 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(1 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  /* run each test (CUEA_ABORT) */
  clear_test_events();
  CU_set_error_action(CUEA_ABORT);
  f_exit_called = FALSE;

  TEST(CUE_SUCCESS == CU_run_test(pSuite1, pTest1));
  TEST(FALSE == f_exit_called);
  f_exit_called = FALSE;

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(1 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(1 == CU_get_number_of_asserts());
  TEST(1 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(0 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SUCCESS == CU_run_test(pSuite1, pTest2));
  TEST(FALSE == f_exit_called);
  f_exit_called = FALSE;

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(1 == CU_get_number_of_tests_run());
  TEST(1 == CU_get_number_of_tests_failed());
  TEST(1 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(1 == CU_get_number_of_failures());
  TEST(1 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SUCCESS == CU_run_test(pSuite1, pTest3));
  TEST(FALSE == f_exit_called);
  f_exit_called = FALSE;

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(1 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(1 == CU_get_number_of_asserts());
  TEST(1 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(0 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SUCCESS == CU_run_test(pSuite1, pTest4));
  TEST(FALSE == f_exit_called);
  f_exit_called = FALSE;

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(1 == CU_get_number_of_tests_run());
  TEST(1 == CU_get_number_of_tests_failed());
  TEST(1 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(1 == CU_get_number_of_failures());
  TEST(1 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SUCCESS == CU_run_test(pSuite1, pTest5));
  TEST(FALSE == f_exit_called);
  f_exit_called = FALSE;

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(0 == CU_get_number_of_suites_failed());
  TEST(1 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(1 == CU_get_number_of_asserts());
  TEST(1 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(0 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SINIT_FAILED == CU_run_test(pSuite2, pTest6));
  TEST(TRUE == f_exit_called);
  f_exit_called = FALSE;

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(0 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(0 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(1 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SINIT_FAILED == CU_run_test(pSuite2, pTest7));
  TEST(TRUE == f_exit_called);
  f_exit_called = FALSE;

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(0 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(0 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(1 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SCLEAN_FAILED == CU_run_test(pSuite3, pTest8));
  TEST(TRUE == f_exit_called);
  f_exit_called = FALSE;

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(1 == CU_get_number_of_tests_run());
  TEST(1 == CU_get_number_of_tests_failed());
  TEST(1 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(1 == CU_get_number_of_failures());
  TEST(2 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  TEST(CUE_SCLEAN_FAILED == CU_run_test(pSuite3, pTest9));
  TEST(TRUE == f_exit_called);
  f_exit_called = FALSE;

  TEST(0 == f_nTestEvents);
  TEST(NULL == f_pFirstEvent);
  TEST(0 == CU_get_number_of_suites_run());
  TEST(1 == CU_get_number_of_suites_failed());
  TEST(1 == CU_get_number_of_tests_run());
  TEST(0 == CU_get_number_of_tests_failed());
  TEST(1 == CU_get_number_of_asserts());
  TEST(1 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(1 == CU_get_number_of_failure_records());
  pRunSummary = CU_get_run_summary();
  TEST(pRunSummary->nSuitesRun      == CU_get_number_of_suites_run());
  TEST(pRunSummary->nSuitesFailed   == CU_get_number_of_suites_failed());
  TEST(pRunSummary->nTestsRun       == CU_get_number_of_tests_run());
  TEST(pRunSummary->nTestsFailed    == CU_get_number_of_tests_failed());
  TEST(pRunSummary->nAsserts        == CU_get_number_of_asserts());
  TEST(pRunSummary->nAssertsFailed  == CU_get_number_of_failures());
  TEST(pRunSummary->nFailureRecords == CU_get_number_of_failure_records());

  /* clean up after testing */
  CU_set_error_action(CUEA_IGNORE);
  CU_cleanup_registry();
  clear_test_events();
}

/*-------------------------------------------------*/
static void test_CU_get_number_of_suites_run(void)
{
  /* tested adequately in other tests */
}

/*-------------------------------------------------*/
static void test_CU_get_number_of_suites_failed(void)
{
  /* tested adequately in other tests */
}

/*-------------------------------------------------*/
static void test_CU_get_number_of_tests_run(void)
{
  /* tested adequately in other tests */
}

/*-------------------------------------------------*/
static void test_CU_get_number_of_tests_failed(void)
{
  /* tested adequately in other tests */
}

/*-------------------------------------------------*/
static void test_CU_get_number_of_asserts(void)
{
  /* tested adequately in other tests */
}

/*-------------------------------------------------*/
static void test_CU_get_number_of_successes(void)
{
  /* tested adequately in other tests */
}

/*-------------------------------------------------*/
static void test_CU_get_number_of_failures(void)
{
  /* tested adequately in other tests */
}

/*-------------------------------------------------*/
static void test_CU_get_failure_list(void)
{
  /* tested adequately in other tests */
}

/*-------------------------------------------------*/
static void test_CU_get_run_summary(void)
{
  /* tested adequately in other tests */
}

/*-------------------------------------------------*/
static void test_CU_get_current_suite(void)
{
  /* tested adequately in other tests */
}

/*-------------------------------------------------*/
static void test_CU_get_current_test(void)
{
  /* tested adequately in other tests */
}

/*-------------------------------------------------*/
static void test_CU_is_test_running(void)
{
  /* tested adequately in other tests */
}

/*-------------------------------------------------*/
static void test_CU_assertImplementation(void)
{
  CU_Test dummy_test;
  CU_Suite dummy_suite;
  CU_pFailureRecord pFailure1 = NULL;
  CU_pFailureRecord pFailure2 = NULL;
  CU_pFailureRecord pFailure3 = NULL;
  CU_pFailureRecord pFailure4 = NULL;
  CU_pFailureRecord pFailure5 = NULL;
  CU_pFailureRecord pFailure6 = NULL;

  CU_clear_previous_results();

  TEST(NULL == CU_get_failure_list());
  TEST(0 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_failures());
  TEST(0 == CU_get_number_of_failure_records());

  /* fool CU_assertImplementation into thinking test run is in progress */
  f_pCurTest = &dummy_test;
  f_pCurSuite = &dummy_suite;

  /* asserted value is TRUE*/
  TEST(TRUE == CU_assertImplementation(TRUE, 100, "Nothing happened 0.", "dummy0.c", "dummy_func0", FALSE));

  TEST(NULL == CU_get_failure_list());
  TEST(1 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_failures());
  TEST(0 == CU_get_number_of_failure_records());

  TEST(TRUE == CU_assertImplementation(TRUE, 101, "Nothing happened 1.", "dummy1.c", "dummy_func1", FALSE));

  TEST(NULL == CU_get_failure_list());
  TEST(2 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_failures());
  TEST(0 == CU_get_number_of_failure_records());

  /* asserted value is FALSE */
  TEST(FALSE == CU_assertImplementation(FALSE, 102, "Something happened 2.", "dummy2.c", "dummy_func2", FALSE));

  TEST(NULL != CU_get_failure_list());
  TEST(3 == CU_get_number_of_asserts());
  TEST(1 == CU_get_number_of_failures());
  TEST(1 == CU_get_number_of_failure_records());

  TEST(FALSE == CU_assertImplementation(FALSE, 103, "Something happened 3.", "dummy3.c", "dummy_func3", FALSE));

  TEST(NULL != CU_get_failure_list());
  TEST(4 == CU_get_number_of_asserts());
  TEST(2 == CU_get_number_of_failures());
  TEST(2 == CU_get_number_of_failure_records());

  TEST(FALSE == CU_assertImplementation(FALSE, 104, "Something happened 4.", "dummy4.c", "dummy_func4", FALSE));

  TEST(NULL != CU_get_failure_list());
  TEST(5 == CU_get_number_of_asserts());
  TEST(3 == CU_get_number_of_failures());
  TEST(3 == CU_get_number_of_failure_records());

  if (3 == CU_get_number_of_failure_records()) {
    pFailure1 = CU_get_failure_list();
    TEST(102 == pFailure1->uiLineNumber);
    TEST(!strcmp("dummy2.c", pFailure1->strFileName));
    TEST(!strcmp("Something happened 2.", pFailure1->strCondition));
    TEST(&dummy_test == pFailure1->pTest);
    TEST(&dummy_suite == pFailure1->pSuite);
    TEST(NULL != pFailure1->pNext);
    TEST(NULL == pFailure1->pPrev);

    pFailure2 = pFailure1->pNext;
    TEST(103 == pFailure2->uiLineNumber);
    TEST(!strcmp("dummy3.c", pFailure2->strFileName));
    TEST(!strcmp("Something happened 3.", pFailure2->strCondition));
    TEST(&dummy_test == pFailure2->pTest);
    TEST(&dummy_suite == pFailure2->pSuite);
    TEST(NULL != pFailure2->pNext);
    TEST(pFailure1 == pFailure2->pPrev);

    pFailure3 = pFailure2->pNext;
    TEST(104 == pFailure3->uiLineNumber);
    TEST(!strcmp("dummy4.c", pFailure3->strFileName));
    TEST(!strcmp("Something happened 4.", pFailure3->strCondition));
    TEST(&dummy_test == pFailure3->pTest);
    TEST(&dummy_suite == pFailure3->pSuite);
    TEST(NULL == pFailure3->pNext);
    TEST(pFailure2 == pFailure3->pPrev);
  }
  else
    FAIL("Unexpected number of failure records.");

  /* confirm destruction of failure records */
  pFailure4 = pFailure1;
  pFailure5 = pFailure2;
  pFailure6 = pFailure3;
  TEST(0 != test_cunit_get_n_memevents(pFailure4));
  TEST(test_cunit_get_n_allocations(pFailure4) != test_cunit_get_n_deallocations(pFailure4));
  TEST(0 != test_cunit_get_n_memevents(pFailure5));
  TEST(test_cunit_get_n_allocations(pFailure5) != test_cunit_get_n_deallocations(pFailure5));
  TEST(0 != test_cunit_get_n_memevents(pFailure6));
  TEST(test_cunit_get_n_allocations(pFailure6) != test_cunit_get_n_deallocations(pFailure6));

  CU_clear_previous_results();
  TEST(0 != test_cunit_get_n_memevents(pFailure4));
  TEST(test_cunit_get_n_allocations(pFailure4) == test_cunit_get_n_deallocations(pFailure4));
  TEST(0 != test_cunit_get_n_memevents(pFailure5));
  TEST(test_cunit_get_n_allocations(pFailure5) == test_cunit_get_n_deallocations(pFailure5));
  TEST(0 != test_cunit_get_n_memevents(pFailure6));
  TEST(test_cunit_get_n_allocations(pFailure6) == test_cunit_get_n_deallocations(pFailure6));
  TEST(0 == CU_get_number_of_asserts());
  TEST(0 == CU_get_number_of_successes());
  TEST(0 == CU_get_number_of_failures());
  TEST(0 == CU_get_number_of_failure_records());

  f_pCurTest = NULL;
  f_pCurSuite = NULL;
}

/*-------------------------------------------------*/
static void test_CU_clear_previous_results(void)
{
  /* covered by test_CU_assertImplementation() */
}

/*-------------------------------------------------*/
static void test_clear_previous_results(void)
{
  /* covered by test_CU_clear_previous_result() */
}

/*-------------------------------------------------*/
static void test_cleanup_failure_list(void)
{
  /* covered by test_clear_previous_result() */
}

/*-------------------------------------------------*/
static void test_run_single_suite(void)
{
  /* covered by test_CU_run_suite() */
}

/*-------------------------------------------------*/
static void test_run_single_test(void)
{
  /* covered by test_CU_run_test() */
}

/*-------------------------------------------------*/
static void test_add_failure(void)
{
  CU_Test test1;
  CU_Suite suite1;
  CU_pFailureRecord pFailure1 = NULL;
  CU_pFailureRecord pFailure2 = NULL;
  CU_pFailureRecord pFailure3 = NULL;
  CU_pFailureRecord pFailure4 = NULL;
  CU_RunSummary run_summary = {0, 0, 0, 0, 0, 0, 0};

  /* test under memory exhaustion */
  test_cunit_deactivate_malloc();
  add_failure(&pFailure1, &run_summary, 100, "condition 0", "file0.c", &suite1, &test1);
  TEST(NULL == pFailure1);
  TEST(0 == run_summary.nFailureRecords);
  test_cunit_activate_malloc();

  /* normal operation */
  add_failure(&pFailure1, &run_summary, 101, "condition 1", "file1.c", &suite1, &test1);
  TEST(1 == run_summary.nFailureRecords);
  if (TEST(NULL != pFailure1)) {
    TEST(101 == pFailure1->uiLineNumber);
    TEST(!strcmp("condition 1", pFailure1->strCondition));
    TEST(!strcmp("file1.c", pFailure1->strFileName));
    TEST(&test1 == pFailure1->pTest);
    TEST(&suite1 == pFailure1->pSuite);
    TEST(NULL == pFailure1->pNext);
    TEST(NULL == pFailure1->pPrev);
    TEST(pFailure1 == f_last_failure);
    TEST(0 != test_cunit_get_n_memevents(pFailure1));
    TEST(test_cunit_get_n_allocations(pFailure1) != test_cunit_get_n_deallocations(pFailure1));
  }

  add_failure(&pFailure1, &run_summary, 102, "condition 2", "file2.c", NULL, &test1);
  TEST(2 == run_summary.nFailureRecords);
  if (TEST(NULL != pFailure1)) {
    TEST(101 == pFailure1->uiLineNumber);
    TEST(!strcmp("condition 1", pFailure1->strCondition));
    TEST(!strcmp("file1.c", pFailure1->strFileName));
    TEST(&test1 == pFailure1->pTest);
    TEST(&suite1 == pFailure1->pSuite);
    TEST(NULL != pFailure1->pNext);
    TEST(NULL == pFailure1->pPrev);
    TEST(pFailure1 != f_last_failure);
    TEST(0 != test_cunit_get_n_memevents(pFailure1));
    TEST(test_cunit_get_n_allocations(pFailure1) != test_cunit_get_n_deallocations(pFailure1));

    if (TEST(NULL != (pFailure2 = pFailure1->pNext))) {
      TEST(102 == pFailure2->uiLineNumber);
      TEST(!strcmp("condition 2", pFailure2->strCondition));
      TEST(!strcmp("file2.c", pFailure2->strFileName));
      TEST(&test1 == pFailure2->pTest);
      TEST(NULL == pFailure2->pSuite);
      TEST(NULL == pFailure2->pNext);                          
      TEST(pFailure1 == pFailure2->pPrev);
      TEST(pFailure2 == f_last_failure);
      TEST(0 != test_cunit_get_n_memevents(pFailure2));
      TEST(test_cunit_get_n_allocations(pFailure2) != test_cunit_get_n_deallocations(pFailure2));
    }
  }

  pFailure3 = pFailure1;
  pFailure4 = pFailure2;
  clear_previous_results(&run_summary, &pFailure1);

  TEST(0 == run_summary.nFailureRecords);
  TEST(0 != test_cunit_get_n_memevents(pFailure3));
  TEST(test_cunit_get_n_allocations(pFailure3) == test_cunit_get_n_deallocations(pFailure3));
  TEST(0 != test_cunit_get_n_memevents(pFailure4));
  TEST(test_cunit_get_n_allocations(pFailure4) == test_cunit_get_n_deallocations(pFailure4));
}

/*-------------------------------------------------*/
void test_cunit_TestRun(void)
{
  test_cunit_start_tests("TestRun.c");

  test_message_handlers();

  test_CU_run_all_tests();
  test_CU_run_suite();
  test_CU_run_test();
  test_CU_get_number_of_suites_run();
  test_CU_get_number_of_suites_failed();
  test_CU_get_number_of_tests_run();
  test_CU_get_number_of_tests_failed();
  test_CU_get_number_of_asserts();
  test_CU_get_number_of_successes();
  test_CU_get_number_of_failures();
  test_CU_get_failure_list();
  test_CU_get_run_summary();
  test_CU_get_current_suite();
  test_CU_get_current_test();
  test_CU_is_test_running();
  test_CU_clear_previous_results();
  test_CU_assertImplementation();
  test_clear_previous_results();
  test_cleanup_failure_list();
  test_run_single_suite();
  test_run_single_test();
  test_add_failure();

  test_cunit_end_tests();
}

#endif    /* CUNIT_BUILD_TESTS */

/*
 *	Global/Static Definitions
 */
/** Local variable holding the current error code. */
static CU_ErrorCode g_error_number = CUE_SUCCESS;
/** Local variable holding the current error action code. */
static CU_ErrorAction g_error_action = CUEA_IGNORE;

/* Private function forward declarations */
static const char* get_error_desc(CU_ErrorCode error);

#ifdef CUNIT_DO_NOT_DEFINE_UNLESS_BUILDING_TESTS
void test_exit(int status);
#endif

/*------------------------------------------------------------------------*/
/** Set the error code.
 * This function is used internally by CUnit implementation functions
 * when an error condition occurs within the framework.  It should
 * not generally be called by user code.  NOTE that if the current
 * error action is CUEA_ABORT, then calling this function will
 * result in exit() being called for the current application.
 * @param error CU_ErrorCode indicating the current error condition.
 * @see CU_get_error()
 * @see CU_get_error_msg()
 * @see CU_ErrorCode
 */
void CU_set_error(CU_ErrorCode error)
{
  if ((error != CUE_SUCCESS) && (g_error_action == CUEA_ABORT)) {
#ifndef CUNIT_DO_NOT_DEFINE_UNLESS_BUILDING_TESTS
    fprintf(stderr, "\nAborting due to error #%d: %s\n",
            error,
            get_error_desc(error));
    exit(error);
#else
    test_exit(error);
#endif
  }

  g_error_number = error;
}

/*------------------------------------------------------------------------*/
/** Get the error code.
 * CUnit implementation functions set the error code to indicate the
 * status of the most recent operation.  In general, the CUnit functions
 * will clear the code to CUE_SUCCESS, then reset it to a specific error
 * code if an exception condition is encountered.  Some functions
 * return the code, others leave it to the user to inspect if desired.
 * @return The current error condition code.
 * @see CU_get_error_msg()
 * @see CU_ErrorCode
 */
CU_ErrorCode CU_get_error(void)
{
	return g_error_number;
}

/*------------------------------------------------------------------------*/
/** Get the message corresponding to the error code.
 * CUnit implementation functions set the error code to indicate the
 * of the most recent operation.  In general, the CUnit functions will
 * clear the code to CUE_SUCCESS, then reset it to a specific error
 * code if an exception condition is encountered.  This function allows
 * the user to retrieve a descriptive error message corresponding to the
 * error code set by the last operation.
 * @return A message corresponding to the current error condition.
 * @see CU_get_error()
 * @see CU_ErrorCode
 */
const char* CU_get_error_msg(void)
{
	return get_error_desc(g_error_number);
}

/*------------------------------------------------------------------------*/
/** Set the action to take when an error condition occurs.
 * This function should be used to specify the action to take
 * when an error condition is encountered.  The default action is
 * CUEA_IGNORE, which results in errors being ignored and test runs
 * being continued (if possible).  A value of CUEA_FAIL causes test
 * runs to stop as soon as an error condition occurs, while
 * CU_ABORT causes the application to exit on any error.
 * @param action CU_ErrorAction indicating the new error action.
 * @see CU_get_error_action()
 * @see CU_set_error()
 * @see CU_ErrorAction
 */
void CU_set_error_action(CU_ErrorAction action)
{
  g_error_action = action;
}

/*------------------------------------------------------------------------*/
/** Get the current error action code.
 * @return The current error action code.
 * @see CU_set_error_action()
 * @see CU_set_error()
 * @see CU_ErrorAction
 */
CU_ErrorAction CU_get_error_action(void)
{
  return g_error_action;
}

/*
 *	Private static function definitions
 */
/*------------------------------------------------------------------------*/
/** Internal function to look up the error message for a specified
 * error code.  An empty string is returned if iError is not a member
 * of CU_ErrorCode.
 * @param iError  CU_ErrorCode to look up.
 * @return Pointer to a string containing the error message.
 * @see CU_get_error_msg()
 */
static const char* get_error_desc(CU_ErrorCode iError)
{
	int iMaxIndex;

	static const char* ErrorDescription[] = {
		"No Error",                             /* CUE_SUCCESS - 0 */
		"Memory allocation failed.",            /* CUE_NOMEMORY - 1 */
		"",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
		"Test registry does not exist.",          /* CUE_NOREGISTRY - 10 */
		"Registry already exists.",               /* CUE_REGISTRY_EXISTS - 11 */
		"",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
		"NULL suite not allowed.",                /* CUE_NOSUITE - 20 */
		"Suite name cannot be NULL.",             /* CUE_NO_SUITENAME - 21 */
		"Suite initialization function failed.",  /* CUE_SINIT_FAILED - 22 */
		"Suite cleanup function failed.",         /* CUE_SCLEAN_FAILED - 23 */
		"Suite having name already registered.",  /* CUE_DUP_SUITE - 24 */
		"",
    "",
    "",
    "",
    "",
		"NULL test not allowed.",                 /* CUE_NOTEST - 30 */
		"Test name cannot be NULL.",              /* CUE_NO_TESTNAME - 31 */
		"Test having this name already in suite.",/* CUE_DUP_TEST - 32 */
		"Test not registered in specified suite.",/* CUE_TEST_NOT_IN_SUITE - 33 */
    "",
    "",
    "",
    "",
    "",
    "",
    "Error opening file.",                    /* CUE_FOPEN_FAILED - 40 */
    "Error closing file.",                    /* CUE_FCLOSE_FAILED - 41 */
    "Bad file name.",                         /* CUE_BAD_FILENAME - 42 */
    "Error during write to file.",            /* CUE_WRITE_ERROR - 43 */
    "Undefined Error"
		};

	iMaxIndex = sizeof(ErrorDescription)/sizeof(char *) - 1;
	if ((int)iError > iMaxIndex)
		return ErrorDescription[iMaxIndex];
	else
		return ErrorDescription[iError];
}

/** @} */

#ifdef CUNIT_BUILD_TESTS
#include "test_cunit.h"

void test_cunit_CUError(void)
{
  CU_ErrorCode old_err = CU_get_error();
  CU_ErrorAction old_action = CU_get_error_action();

  test_cunit_start_tests("CUError.c");

  /* CU_set_error() & CU_get_error() */
  CU_set_error(CUE_NOMEMORY);
  TEST(CU_get_error() != CUE_SUCCESS);
  TEST(CU_get_error() == CUE_NOMEMORY);

  CU_set_error(CUE_NOREGISTRY);
  TEST(CU_get_error() != CUE_SUCCESS);
  TEST(CU_get_error() == CUE_NOREGISTRY);

  /* CU_get_error_msg() */
  CU_set_error(CUE_SUCCESS);
  TEST(!strcmp(CU_get_error_msg(), get_error_desc(CUE_SUCCESS)));

  CU_set_error(CUE_NOTEST);
  TEST(!strcmp(CU_get_error_msg(), get_error_desc(CUE_NOTEST)));

  CU_set_error(CUE_NOMEMORY);
  TEST(!strcmp(CU_get_error_msg(), get_error_desc(CUE_NOMEMORY)));
  TEST(strcmp(CU_get_error_msg(), get_error_desc(CUE_SCLEAN_FAILED)));

  TEST(!strcmp(get_error_desc(100), "Undefined Error"));

  /* CU_set_error_action() & CU_get_error_action() */
  CU_set_error_action(CUEA_FAIL);
  TEST(CU_get_error_action() != CUEA_IGNORE);
  TEST(CU_get_error_action() == CUEA_FAIL);
  TEST(CU_get_error_action() != CUEA_ABORT);

  CU_set_error_action(CUEA_ABORT);
  TEST(CU_get_error_action() != CUEA_IGNORE);
  TEST(CU_get_error_action() != CUEA_FAIL);
  TEST(CU_get_error_action() == CUEA_ABORT);

  /* reset  values */
  CU_set_error(old_err);
  CU_set_error_action(old_action);

  test_cunit_end_tests();
}

#endif    /* CUNIT_BUILD_TESTS */



#ifdef MEMTRACE

#define MAX_FILE_NAME_LENGTH  256

/** Name for memory dump file. */
static const char* f_szDefaultDumpFileName = "CUnit-Memory-Dump.xml";

#ifdef CUNIT_BUILD_TESTS
/** For testing use (only) to simulate memory exhaustion -
 * if FALSE, allocation requests will always fail and return NULL.
 */
static BOOL f_bTestCunitMallocActive = TRUE;
#endif

/** Structure holding the details of a memory allocation/deallocation event. */
typedef struct mem_event {
  unsigned int      Size;
  unsigned int      AllocLine;
  char              AllocFilename[MAX_FILE_NAME_LENGTH];
  unsigned int      DeallocLine;
  char              DeallocFilename[MAX_FILE_NAME_LENGTH];
  struct mem_event* pNext;
} MEMORY_EVENT;
typedef MEMORY_EVENT* PMEMORY_EVENT;

#define NOT_ALLOCATED 0
#define NOT_DELETED 0

/** Structure holding the details of a memory node having allocation/deallocation events. */
typedef struct mem_node {
  void*             pLocation;
  unsigned int      EventCount;
  PMEMORY_EVENT     pFirstEvent;
  struct mem_node*  pNext;
} MEMORY_NODE;
typedef MEMORY_NODE* PMEMORY_NODE;

static PMEMORY_NODE f_pMemoryTrackerHead = NULL;  /**< Head of double-linked list of memory nodes. */
static unsigned int f_nMemoryNodes = 0;           /**< Counter for memory nodes created. */
/*------------------------------------------------------------------------*/
/** Locate the memory node for the specified memory location (returns NULL if none). */
static PMEMORY_NODE find_memory_node(void* pLocation)
{
  PMEMORY_NODE pMemoryNode = f_pMemoryTrackerHead;
  while (pMemoryNode) {
    if (pLocation == pMemoryNode->pLocation)
      break;
    pMemoryNode = pMemoryNode->pNext;
  }

  return pMemoryNode;
}
/*------------------------------------------------------------------------*/
/** Create a new memory node for the specified memory location. */
static PMEMORY_NODE create_memory_node(void* pLocation)
{
  PMEMORY_NODE pMemoryNode = NULL;
  PMEMORY_NODE pTempNode = NULL;

  /* a memory node for pLocation should not exist yet */
  if (NULL == (pMemoryNode = find_memory_node(pLocation))) {

    pMemoryNode = malloc(sizeof(MEMORY_NODE));
    assert(pMemoryNode && "Memory Allocation for new memory node failed in create_memory_node().");

    pMemoryNode->pLocation = pLocation;
    pMemoryNode->EventCount = 0;
    pMemoryNode->pFirstEvent = NULL;
    pMemoryNode->pNext = NULL;

    /* add new node to linked list */
    pTempNode = f_pMemoryTrackerHead;
    if (NULL == pTempNode) {
      f_pMemoryTrackerHead = pMemoryNode;
    }
    else {
      while (NULL != pTempNode->pNext) {
        pTempNode = pTempNode->pNext;
      }
      pTempNode->pNext = pMemoryNode;
    }

    ++f_nMemoryNodes;
  }
  return pMemoryNode;
}
/*------------------------------------------------------------------------*/
/** Add a new memory event having the specified parameters. */
static PMEMORY_EVENT add_memory_event(PMEMORY_NODE pMemoryNode,
                                      int size,
                                      unsigned int alloc_line,
                                      const char* alloc_filename)
{
  PMEMORY_EVENT pMemoryEvent = NULL;
  PMEMORY_EVENT pTempEvent = NULL;

  assert (pMemoryNode);

  pMemoryEvent = malloc(sizeof(MEMORY_EVENT));
  assert(pMemoryEvent && "Memory Allocation for new memory event failed in create_memory_event().");

  pMemoryEvent->Size = size;
  pMemoryEvent->AllocLine = alloc_line;
  strncpy(pMemoryEvent->AllocFilename, alloc_filename, MAX_FILE_NAME_LENGTH-1);
  pMemoryEvent->AllocFilename[MAX_FILE_NAME_LENGTH-1] = 0;
  pMemoryEvent->DeallocLine = NOT_DELETED;
  pMemoryEvent->DeallocFilename[0] = 0;
  pMemoryEvent->pNext = NULL;

  /* add new event to linked list */
  pTempEvent = pMemoryNode->pFirstEvent;
  if (NULL == pTempEvent) {
    pMemoryNode->pFirstEvent = pMemoryEvent;
  }
  else {
    while (NULL != pTempEvent->pNext) {
      pTempEvent = pTempEvent->pNext;
    }
    pTempEvent->pNext = pMemoryEvent;
  }

  ++pMemoryNode->EventCount;

  return pMemoryEvent;
}
/*------------------------------------------------------------------------*/
/** Record memory allocation event. */
static PMEMORY_NODE allocate_memory(int nSize,
                                    void* pLocation,
                                    unsigned int uiAllocationLine,
                                    const char* szAllocationFile)
{
  PMEMORY_NODE  pMemoryNode = NULL;

  /* attempt to locate an existing record for this pLocation */
  pMemoryNode = find_memory_node(pLocation);

  /* pLocation not found - create a new event record */
  if (NULL == pMemoryNode)
    pMemoryNode = create_memory_node(pLocation);

  /* add the new event record */
  add_memory_event(pMemoryNode, nSize, uiAllocationLine, szAllocationFile);

  return pMemoryNode;
}

/*------------------------------------------------------------------------*/
/** Record memory deallocation event. */
static void deallocate_memory(void* pLocation, unsigned int uiDeletionLine, const char* szDeletionFileName)
{
  PMEMORY_NODE  pMemoryNode = NULL;
  PMEMORY_EVENT pTempEvent = NULL;

  assert(uiDeletionLine);
  assert(szDeletionFileName);

  /* attempt to locate an existing record for this pLocation */
  pMemoryNode = find_memory_node(pLocation);

  /* if no entry, then an unallocated pointer was freed */
  if (NULL == pMemoryNode) {
    pMemoryNode = create_memory_node(pLocation);
    pTempEvent = add_memory_event(pMemoryNode, 0, NOT_ALLOCATED, "");
  }
  else {
    /* there should always be at least 1 event for an existing memory node */
    assert(pMemoryNode->pFirstEvent);

    /* locate last memory event for this pLocation */
    pTempEvent = pMemoryNode->pFirstEvent;
    while (NULL != pTempEvent->pNext)
      pTempEvent = pTempEvent->pNext;

    /* if pointer has already been freed, create a new event for double deletion */
    if (NOT_DELETED != pTempEvent->DeallocLine)
      pTempEvent = add_memory_event(pMemoryNode, pTempEvent->Size, NOT_ALLOCATED, "");
  }

  pTempEvent->DeallocLine = uiDeletionLine;
  strncpy(pTempEvent->DeallocFilename, szDeletionFileName, MAX_FILE_NAME_LENGTH-1);
  pTempEvent->DeallocFilename[MAX_FILE_NAME_LENGTH-1] = 0;
}

/*------------------------------------------------------------------------*/
/** Custom calloc function with memory event recording. */
void* CU_calloc(size_t nmemb, size_t size, unsigned int uiLine, const char* szFileName)
{
  void* pVoid = NULL;

#ifdef CUNIT_BUILD_TESTS
  if (!f_bTestCunitMallocActive)
    return NULL;
#endif

  pVoid = calloc(nmemb, size);
  if (pVoid)
    allocate_memory(nmemb * size, pVoid, uiLine, szFileName);

  return pVoid;
}

/*------------------------------------------------------------------------*/
/** Custom malloc function with memory event recording. */
void* CU_malloc(size_t size, unsigned int uiLine, const char* szFileName)
{
  void* pVoid = NULL;

#ifdef CUNIT_BUILD_TESTS
  if (!f_bTestCunitMallocActive)
    return NULL;
#endif

  pVoid = malloc(size);
  if (pVoid)
    allocate_memory(size, pVoid, uiLine, szFileName);

  return pVoid;
}

/*------------------------------------------------------------------------*/
/** Custom free function with memory event recording. */
void CU_free(void *ptr, unsigned int uiLine, const char* szFileName)
{
  deallocate_memory(ptr, uiLine, szFileName);
  free(ptr);
}

/*------------------------------------------------------------------------*/
/** Custom realloc function with memory event recording. */
void* CU_realloc(void *ptr, size_t size, unsigned int uiLine, const char* szFileName)
{
  void* pVoid = NULL;

  deallocate_memory(ptr, uiLine, szFileName);

#ifdef CUNIT_BUILD_TESTS
  if (!f_bTestCunitMallocActive) {
    free(ptr);
    return NULL;
  }
#endif

  pVoid = realloc(ptr, size);

  if (pVoid)
    allocate_memory(size, pVoid, uiLine, szFileName);

  return pVoid;
}

/*------------------------------------------------------------------------*/
/** Print a report of memory events to file. */
void CU_dump_memory_usage(const char* szFilename)
{
  char* szDumpFileName = (char*)f_szDefaultDumpFileName;
  unsigned int nValid;
  unsigned int nInvalid;
  PMEMORY_NODE pTempNode = NULL;
  PMEMORY_EVENT pTempEvent = NULL;
  FILE* pFile = NULL;
  time_t tTime = 0;

  /* use the specified file name, if supplied) */
  if ((NULL != szFilename) && strlen(szFilename) > 0)
    szDumpFileName = (char*)szFilename;

  if (NULL == (pFile = fopen(szDumpFileName, "w"))) {
    fprintf(stderr, "Failed to open file \"%s\" : %s", szDumpFileName, strerror(errno));
    return;
  }

  setvbuf(pFile, NULL, _IONBF, 0);

  fprintf(pFile, "<\?xml version=\"1.0\" \?>");
  fprintf(pFile, "\n<\?xml-stylesheet type=\"text/xsl\" href=\"Memory-Dump.xsl\" \?>");
  fprintf(pFile, "\n<!DOCTYPE MEMORY_DUMP_REPORT SYSTEM \"Memory-Dump.dtd\">");
  fprintf(pFile, "\n<MEMORY_DUMP_REPORT>");
  fprintf(pFile, "\n  <MD_HEADER/>");
  fprintf(pFile, "\n  <MD_RUN_LISTING>");

  nValid = 0;
  nInvalid = 0;
  pTempNode = f_pMemoryTrackerHead;
  while (pTempNode) {
    fprintf(pFile, "\n    <MD_RUN_RECORD>");
    fprintf(pFile, "\n      <MD_POINTER> %p </MD_POINTER>", pTempNode->pLocation);
    fprintf(pFile, "\n      <MD_EVENT_COUNT> %d </MD_EVENT_COUNT>", pTempNode->EventCount);

    pTempEvent = pTempNode->pFirstEvent;
    while (pTempEvent) {
      fprintf(pFile, "\n      <MD_EVENT_RECORD>");
      fprintf(pFile, "\n        <MD_SIZE> %d </MD_SIZE>", pTempEvent->Size);
      fprintf(pFile, "\n        <MD_ALLOC_FILE> %s </MD_ALLOC_FILE>", pTempEvent->AllocFilename);
      fprintf(pFile, "\n        <MD_ALLOC_LINE> %d </MD_ALLOC_LINE>", pTempEvent->AllocLine);
      fprintf(pFile, "\n        <MD_DEALLOC_FILE> %s </MD_DEALLOC_FILE>", pTempEvent->DeallocFilename);
      fprintf(pFile, "\n        <MD_DEALLOC_LINE> %d </MD_DEALLOC_LINE>", pTempEvent->DeallocLine);
      fprintf(pFile, "\n      </MD_EVENT_RECORD>");

      if ((0 != pTempEvent->AllocLine) && (0 != pTempEvent->DeallocLine))
        ++nValid;
      else
        ++nInvalid;

      pTempEvent = pTempEvent->pNext;
    }

    fprintf(pFile, "\n    </MD_RUN_RECORD>");
    pTempNode = pTempNode->pNext;
  }

  fprintf(pFile, "\n  </MD_RUN_LISTING>");

  fprintf(pFile, "\n  <MD_SUMMARY>");
  fprintf(pFile, "\n    <MD_SUMMARY_VALID_RECORDS> %d </MD_SUMMARY_VALID_RECORDS>", nValid);
  fprintf(pFile, "\n    <MD_SUMMARY_INVALID_RECORDS> %d </MD_SUMMARY_INVALID_RECORDS>", nInvalid);
  fprintf(pFile, "\n    <MD_SUMMARY_TOTAL_RECORDS> %d </MD_SUMMARY_TOTAL_RECORDS>", nValid + nInvalid);
  fprintf(pFile, "\n  </MD_SUMMARY>");

  time(&tTime);
  fprintf(pFile, "\n  <MD_FOOTER> Memory Trace for CUnit Run at %s </MD_FOOTER>", ctime(&tTime));
  fprintf(pFile, "</MEMORY_DUMP_REPORT>");

  fclose(pFile);
}

#endif  // MEMTRACE

/** @} */

#ifdef CUNIT_BUILD_TESTS
#include "test_cunit.h"

/** Deactivate CUnit memory allocation
 * After calling this function, all Cunit memory
 * allocation routines will fail and return NULL.
 */
void test_cunit_deactivate_malloc(void)
{
  f_bTestCunitMallocActive = FALSE;
}

/** Activate CUnit memory allocation
 * After calling this function, all Cunit memory
 * allocation routines will behave normally (allocating
 * memory if it is available).
 */
void test_cunit_activate_malloc(void)
{
  f_bTestCunitMallocActive = TRUE;
}

/** Retrieve the number of memory events recorded for a given pointer. */
unsigned int test_cunit_get_n_memevents(void* pLocation)
{
  PMEMORY_NODE pNode = find_memory_node(pLocation);
  return (pNode) ? pNode->EventCount : 0;
}

/** Retrieve the number of memory allocations recorded for a given pointer. */
unsigned int test_cunit_get_n_allocations(void* pLocation)
{
  PMEMORY_NODE pNode = find_memory_node(pLocation);
  PMEMORY_EVENT pEvent = NULL;
  int result = 0;

  if (pNode) {
    pEvent = pNode->pFirstEvent;
    while (pEvent) {
      if (pEvent->AllocLine != NOT_ALLOCATED)
        ++result;
      pEvent = pEvent->pNext;
    }
  }

  return result;
}

/** Retrieve the number of memory deallocations recorded for a given pointer. */
unsigned int test_cunit_get_n_deallocations(void* pLocation)
{
  PMEMORY_NODE pNode = find_memory_node(pLocation);
  PMEMORY_EVENT pEvent = NULL;
  int result = 0;

  if (pNode) {
    pEvent = pNode->pFirstEvent;
    while (pEvent) {
      if (pEvent->DeallocLine != NOT_DELETED)
        ++result;
      pEvent = pEvent->pNext;
    }
  }

  return result;
}

void test_CU_calloc(void)
{
  void* ptr1 = NULL;
  void* ptr2 = calloc(2, sizeof(int));
  unsigned int n2 = test_cunit_get_n_memevents(ptr2);

  /* test allocation failure */
  test_cunit_deactivate_malloc();
  ptr1 = CU_CALLOC(2, sizeof(int));
  TEST(NULL == ptr1);
  TEST(test_cunit_get_n_allocations(ptr1) == test_cunit_get_n_deallocations(ptr1));
  TEST(test_cunit_get_n_allocations(ptr2) == test_cunit_get_n_deallocations(ptr2));
  test_cunit_activate_malloc();

  /* normal allocation */
  ptr1 = CU_CALLOC(2, sizeof(int));
  TEST_FATAL(NULL != ptr1);
  TEST(test_cunit_get_n_allocations(ptr1) != test_cunit_get_n_deallocations(ptr1));
  TEST(test_cunit_get_n_allocations(ptr2) == test_cunit_get_n_deallocations(ptr2));

  CU_FREE(ptr1);
  TEST(test_cunit_get_n_allocations(ptr1) == test_cunit_get_n_deallocations(ptr1));
  TEST(test_cunit_get_n_allocations(ptr2) == test_cunit_get_n_deallocations(ptr2));
  TEST(n2 == test_cunit_get_n_memevents(ptr2));

  free(ptr2);
}

void test_CU_malloc(void)
{
  void* ptr1 = NULL;
  void* ptr2 = malloc(sizeof(int));
  unsigned int n2 = test_cunit_get_n_memevents(ptr2);

  /* test allocation failure */
  test_cunit_deactivate_malloc();
  ptr1 = CU_MALLOC(sizeof(int));
  TEST(NULL == ptr1);
  TEST(test_cunit_get_n_allocations(ptr1) == test_cunit_get_n_deallocations(ptr1));
  TEST(test_cunit_get_n_allocations(ptr2) == test_cunit_get_n_deallocations(ptr2));
  test_cunit_activate_malloc();

  /* normal allocation */
  ptr1 = CU_MALLOC(sizeof(int));
  TEST_FATAL(NULL != ptr1);
  TEST(test_cunit_get_n_allocations(ptr1) != test_cunit_get_n_deallocations(ptr1));
  TEST(test_cunit_get_n_allocations(ptr2) == test_cunit_get_n_deallocations(ptr2));

  CU_FREE(ptr1);
  TEST(test_cunit_get_n_allocations(ptr1) == test_cunit_get_n_deallocations(ptr1));
  TEST(test_cunit_get_n_allocations(ptr2) == test_cunit_get_n_deallocations(ptr2));
  TEST(n2 == test_cunit_get_n_memevents(ptr2));

  free(ptr2);
}

void test_CU_free(void)
{
  /* covered by other test functions */
}

void test_CU_realloc(void)
{
  void* ptr1 = CU_MALLOC(sizeof(int));
  void* ptr2 = malloc(sizeof(int));
  void* ptr3;
  void* ptr4;
  unsigned int n2 = test_cunit_get_n_memevents(ptr2);

  /* test allocation failure */
  test_cunit_deactivate_malloc();
  ptr1 = CU_REALLOC(ptr1, sizeof(long int));
  TEST(NULL == ptr1);
  TEST(test_cunit_get_n_allocations(ptr1) == test_cunit_get_n_deallocations(ptr1));
  TEST(test_cunit_get_n_allocations(ptr2) == test_cunit_get_n_deallocations(ptr2));
  test_cunit_activate_malloc();

  /* normal allocation */
  ptr3 = CU_MALLOC(sizeof(int));
  TEST_FATAL(NULL != ptr3);
  TEST(test_cunit_get_n_allocations(ptr1) == test_cunit_get_n_deallocations(ptr1));
  TEST(test_cunit_get_n_allocations(ptr2) == test_cunit_get_n_deallocations(ptr2));
  TEST(test_cunit_get_n_allocations(ptr3) != test_cunit_get_n_deallocations(ptr3));

  ptr4 = CU_REALLOC(ptr3, sizeof(long int));
  TEST_FATAL(NULL != ptr4);
  TEST(test_cunit_get_n_allocations(ptr1) == test_cunit_get_n_deallocations(ptr1));
  TEST(test_cunit_get_n_allocations(ptr2) == test_cunit_get_n_deallocations(ptr2));
  if (ptr3 != ptr4)
    TEST(test_cunit_get_n_allocations(ptr3) == test_cunit_get_n_deallocations(ptr3));
  TEST(test_cunit_get_n_allocations(ptr4) != test_cunit_get_n_deallocations(ptr4));

  CU_FREE(ptr4);
  TEST(test_cunit_get_n_allocations(ptr1) == test_cunit_get_n_deallocations(ptr1));
  TEST(test_cunit_get_n_allocations(ptr2) == test_cunit_get_n_deallocations(ptr2));
  TEST(test_cunit_get_n_allocations(ptr3) == test_cunit_get_n_deallocations(ptr3));
  TEST(test_cunit_get_n_allocations(ptr4) == test_cunit_get_n_deallocations(ptr4));
  TEST(n2 == test_cunit_get_n_memevents(ptr2));

  free(ptr2);
}

/** The main internal testing function for MyMem.c. */
void test_cunit_MyMem(void)
{
  test_cunit_start_tests("MyMem.c");

  test_CU_calloc();
  test_CU_malloc();
  test_CU_free();
  test_CU_realloc();

  test_cunit_end_tests();
}

#endif    /* CUNIT_BUILD_TESTS */


/*

  
static void testAssertTrue(void)
{
  AssertTrue(TRUE);
  AssertTrue(!FALSE);

  AssertTrue(TRUE);
  AssertTrue(!FALSE);
}

static void testAssertFalse(void)
{
  AssertFalse(FALSE);
  AssertFalse(!TRUE);

  AssertFalse(FALSE);
  AssertFalse(FALSE);
}

static void testAssertEqual(void)
{
  AssertEqual(10, 10);
  AssertEqual(0, 0);
  AssertEqual(0, -0);
  AssertEqual(-12, -12);

  AssertEqual(10, 10);
  AssertEqual(0, 0);
  AssertEqual(0, -0);
  AssertEqual(-12, -12);
}

static void testAssertNotEqual(void)
{
  AssertNotEqual(10, 11);
  AssertNotEqual(0, -1);
  AssertNotEqual(-12, -11);

  AssertNotEqual(10, 11);
  AssertNotEqual(0, -2);
  AssertNotEqual(0, 1);
  AssertNotEqual(-12, -13);
}

static void testAssertPtrEqual(void)
{
  AssertPtrEqual((void*)0x100, (void*)0x100);

  AssertPtrEqual((void*)0x101, (void*)0x101);
}

static void testAssertPtrNotEqual(void)
{
  AssertPtrNotEqual((void*)0x100, (void*)0x101);

  AssertPtrNotEqual((void*)0x100, (void*)0x102);
}

static void testAssertPtrNull(void)
{
  AssertPtrNull((void*)(NULL));
  AssertPtrNull((void*)(0x0));

  AssertPtrNull((void*)0x0);
}

static void testAssertPtrNotNull(void)
{
  AssertPtrNotNull((void*)0x100);

  AssertPtrNull(NULL);
  AssertPtrNotNull((void*)0x01);
}

static void testAssertStringEqual(void)
{
  char str1[] = "test";
  char str2[] = "test";
  char str3[] = "test";

  AssertStringEqual(str1, str2);

  AssertStringEqual(str1, str3);
  AssertStringEqual(str3, str2);
}

static void testAssertStringNotEqual(void)
{
  char str1[] = "test";
  char str2[] = "test";
  char str3[] = "suite";

  AssertStringNotEqual(str1, str3);
  AssertStringNotEqual(str3, str2);

  AssertStringNotEqual(str1, str3);
}

static void testAssertNStringEqual(void)
{
  char str1[] = "test";
  char str2[] = "testgfsg";
  char str3[] = "tesgfsg";

  AssertNstringEqual(str1, str2, strlen(str1));
  AssertNstringEqual(str1, str1, strlen(str1));
  AssertNstringEqual(str1, str1, strlen(str1) + 1);

  AssertNstringEqual(str2, str3, 3);
  AssertNstringEqual(str1, str3, 3);
}

static void testAssertNStringNotEqual(void)
{
  char str1[] = "test";
  char str2[] = "tevt";
  char str3[] = "testgfsg";

  AssertNstringNotEqual(str1, str2, 3);
  AssertNstringNotEqual(str1, str3, strlen(str1) + 1);

  AssertNstringNotEqual(str1, str2, 3);
  AssertNstringNotEqual(str2, str3, 3);
}

static void testAssertDoubleEqual(void)
{
  AssertDoubleEqual(10, 10.0001, 0.0001);
  AssertDoubleEqual(10, 10.0001, -0.0001);
  AssertDoubleEqual(-10, -10.0001, 0.0001);
  AssertDoubleEqual(-10, -10.0001, -0.0001);

  AssertDoubleEqual(10, 10.0001, 0.0001);
  AssertDoubleEqual(10, 10.0001, -0.0001);
  AssertDoubleEqual(-10, -10.0001, 0.0001);
  AssertDoubleEqual(-10, -10.0001, -0.0001);
}

static void testAssertDoubleNotEqual(void)
{
  AssertDoubleNotEqual(10, 10.001, 0.0001);
  AssertDoubleNotEqual(10, 10.001, -0.0001);
  AssertDoubleNotEqual(-10, -10.001, 0.0001);
  AssertDoubleNotEqual(-10, -10.001, -0.0001);

  AssertDoubleNotEqual(10, 10.02, 0.01);
  AssertDoubleNotEqual(10, 10.02, -0.01);
  AssertDoubleNotEqual(-10, -10.02, 0.01);
  AssertDoubleNotEqual(-10, -10.02, -0.01);
}

static void testAbort(void)
{
	//CU_TEST_FATAL(TRUE);
	//CU_TEST_FATAL(TRUE);
	//fprintf(stderr, "\nFatal assertion failed to abort test in testAbortIndirect1\n");
	//exit(1);
}

static void testAbortIndirect(void)
{
  //testAbort();
  //fprintf(stderr, "\nFatal assertion failed to abort test in testAbortIndirect2\n");
  //exit(1);
}

static void testFatal(void)
{
  //CU_TEST_FATAL(TRUE);
  //testAbortIndirect();
  //fprintf(stderr, "\nFatal assertion failed to abort test in testFatal\n");
  //exit(1);
}

void AddTests(void)
{
	CU_pSuite pSuite;

	pSuite = CU_add_suite("TestBooleanAssert", NULL, NULL);
	CU_add_test(pSuite, "testAssertTrue", testAssertTrue);
	CU_add_test(pSuite, "testAssertFalse", testAssertFalse);

	pSuite = CU_add_suite("TestEqualityAssert", NULL, NULL);
	CU_add_test(pSuite, "testAssertEqual", testAssertEqual);
	CU_add_test(pSuite, "testAssertNotEqual", testAssertNotEqual);

	pSuite = CU_add_suite("TestPointerAssert", NULL, NULL);
	CU_add_test(pSuite, "testAssertPtrEqual", testAssertPtrEqual);
	CU_add_test(pSuite, "testAssertPtrNotEqual", testAssertPtrNotEqual);

	pSuite = CU_add_suite("TestNullnessAssert", NULL, NULL);
	CU_add_test(pSuite, "testAssertPtrNull", testAssertPtrNull);
	CU_add_test(pSuite, "testAssertPtrNotNull", testAssertPtrNotNull);

	pSuite = CU_add_suite("TestStringAssert", NULL, NULL);
	CU_add_test(pSuite, "testAssertStringEqual", testAssertStringEqual);
	CU_add_test(pSuite, "testAssertStringNotEqual", testAssertStringNotEqual);

	pSuite = CU_add_suite("TestNStringAssert", NULL, NULL);
	CU_add_test(pSuite, "testAssertNStringEqual", testAssertNStringEqual);
	CU_add_test(pSuite, "testAssertNStringNotEqual", testAssertNStringNotEqual);

	pSuite = CU_add_suite("TestDoubleAssert", NULL, NULL);
	CU_add_test(pSuite, "testAssertDoubleEqual", testAssertDoubleEqual);
	CU_add_test(pSuite, "testAssertDoubleNotEqual", testAssertDoubleNotEqual);

	pSuite = CU_add_suite("TestFatal", NULL, NULL);
	CU_add_test(pSuite, "testFatal", testFatal);
}

int main(int argc, char* argv[])
{
	if (CU_initialize_registry()) {
		printf("\nInitialization of Test Registry failed.");
	} else {
		AddTests();
		CU_console_run_tests();
		CU_cleanup_registry();
	}

	return 0;
}

*/
