unit ecma_sockobject;

//socketgobject
//2001/04/27 ~
//by Wolfy

{$IFDEF VER130}
{$ELSE}
  {$WARN SYMBOL_PLATFORM OFF}
  {$WARN UNIT_PLATFORM OFF}
{$ENDIF}


interface

uses
  windows,classes,sysutils,dialogs,syncobjs,forms,
  ecma_type,ecma_expr,hashtable,regexpr,jconvert,
  ecma_misc,ecma_object,myclasses,FileCtrl,ecma_extobject,
  gsocket,gsocketmisc,gsockethttp,gsockethttps,
  gsocketftp,gsocketpop3,cookielib,winsock,
  gsocketsmtp,gsockethtml;

type
  //URL͕
  TJURLInfoObject = class(TJObject)
  private
    FInfo: TUrlInfo;
    function GetDir: String;
    function GetFilename: String;
    function GetHost: String;
    function GetPass: String;
    function GetPath: String;
    function GetPort: String;
    function GetProtocol: String;
    function GetQuery: String;
    function GetUrl: String;
    function GetUser: String;
    procedure SetDir(const Value: String);
    procedure SetFilename(const Value: String);
    procedure SetHost(const Value: String);
    procedure SetPass(const Value: String);
    procedure SetPath(const Value: String);
    procedure SetPort(const Value: String);
    procedure SetProtocol(const Value: String);
    procedure SetQuery(const Value: String);
    procedure SetUrl(const Value: String);
    procedure SetUser(const Value: String);
    function GetHostname: String;
    procedure SetHostname(const Value: String);
  public
    constructor Create(AFactory: TJObjectFactory; Param: TJValueList); override;
  published
    property url: String read GetUrl write SetUrl;
    property uri: String read GetUrl write SetUrl;
    property href: String read GetUrl write SetUrl;
    property protocol: String read GetProtocol write SetProtocol;
    property scheme: String read GetProtocol write SetProtocol;
    property username: String read GetUser write SetUser;
    property user: String read GetUser write SetUser;
    property userid: String read GetUser write SetUser;
    property password: String read GetPass write SetPass;
    property pass: String read GetPass write SetPass;
    property host: String read GetHost write SetHost;
    property hostname: String read GetHostname write SetHostname;
    property port: String read GetPort write SetPort;
    property path: String read GetPath write SetPath;
    property directory: String read GetDir write SetDir;
    property dir: String read GetDir write SetDir;
    property filename: String read GetFilename write SetFilename;
    property query: String read GetQuery write SetQuery;
    property search: String read GetQuery write SetQuery;
  end;
  //http
  TJCookieObject = class(TJObject)
  private
    FCookie: TCookie;
    FExpires: TJDateObject;
    function GetDomain: String;
    function GetExpires: TJDateObject;
    function GetPath: String;
    procedure SetDomain(const Value: String);
    procedure SetPath(const Value: String);
  public
    constructor Create(AFactory: TJObjectFactory; Param: TJValueList); override;
    destructor Destroy; override;
    property Cookie: TCookie read FCookie;
  published
    property domain: String read GetDomain write SetDomain;
    property path: String read GetPath write SetPath;
    property expires: TJDateObject read GetExpires;
  end;

  TJResponseObject = class(TJObject)
  private
    FCookie: TJCookieObject;
    FCode: Integer;
    FText: String;
    function GetCookie: TJCookieObject;
  protected
    function GetPropertyList: String; override;
  public
    constructor Create(AFactory: TJObjectFactory; Param: TJValueList); override;
    destructor Destroy; override;
    procedure SetStatus(ACode: Integer; AText: String);
    procedure Clear;
  published
    property cookie: TJCookieObject read GetCookie;
    property code: Integer read FCode;
    property text: String read FText;
  end;

  TJBaseSocketObject = class(TJObject)
  private
    FDebug: Boolean;
    FOnPrint: TOnPrineEvent;
  protected
    //event
    procedure SockOnStatus(Sender: TObject; const Status: String);
    function GetTimeout: Integer; virtual; abstract;
    procedure SetTimeout(const Value: Integer); virtual; abstract;
  public
    property OnPrint: TOnPrineEvent read FOnPrint write FOnPrint;
  published
    property timeout: Integer read GetTimeout write SetTimeout;
    property debug: Boolean read FDebug write FDebug;
  end;


  TJHTTPObject = class(TJBaseSocketObject)
  private
    FHTTP: TgHTTP;
    FRequestHeader: TJObject;
    FResponseHeader: TJResponseObject;

    function DoGet(Param: TJValueList): TJValue;
    function DoGetFile(Param: TJValueList): TJValue;
    function DoPost(Param: TJValueList): TJValue;
    function DoHead(Param: TJValueList): TJValue;
    function DoRequest(Param: TJValueList): TJValue;
    function DoResponse(Param: TJValueList): TJValue;
    function DoCapture(Param: TJValueList): TJValue;
    function DoReadln(Param: TJValueList): TJValue;
    function DoRead(Param: TJValueList): TJValue;
    function DoWriteln(Param: TJValueList): TJValue;
    function DoWrite(Param: TJValueList): TJValue;
    function DoConnect(Param: TJValueList): TJValue;
    function DoDisconnect(Param: TJValueList): TJValue;

    function GetAutoRedirect: Boolean;
    function GetLength: Integer;
    function GetVersion: String;
    procedure SetAutoRedirect(const Value: Boolean);
    procedure SetVersion(const Value: String);
    function GetRequestHeader: TJObject;
    function GetResponseHeader: TJResponseObject;
    function GetProxy: String;
    procedure SetProxy(const Value: String);
  protected
    function GetTimeout: Integer; override;
    procedure SetTimeout(const Value: Integer); override;
  public
    constructor Create(AFactory: TJObjectFactory; Param: TJValueList); override;
    destructor Destroy; override;
  published
    property requestHeader: TJObject read GetRequestHeader;
    property responseHeader: TJResponseObject read GetResponseHeader;
    property version: String read GetVersion write SetVersion;
    property length: Integer read GetLength;
    property autoRedirect: Boolean read GetAutoRedirect write SetAutoRedirect;
    property proxy: String read GetProxy write SetProxy;
  end;

  TJHTTPSObject = class(TJBaseSocketObject)
  private
    FHTTPS: TgHTTPS;
    FRequestHeader: TJObject;
    FResponseHeader: TJResponseObject;

    function DoGet(Param: TJValueList): TJValue;
    function DoGetFile(Param: TJValueList): TJValue;
    function DoPost(Param: TJValueList): TJValue;
    function DoHead(Param: TJValueList): TJValue;
    function DoRequest(Param: TJValueList): TJValue;
    function DoResponse(Param: TJValueList): TJValue;
    function DoCapture(Param: TJValueList): TJValue;
    function DoReadln(Param: TJValueList): TJValue;
    function DoRead(Param: TJValueList): TJValue;
    function DoConnect(Param: TJValueList): TJValue;
    function DoDisconnect(Param: TJValueList): TJValue;
    
    function GetAutoRedirect: Boolean;
    function GetLength: Integer;
    function GetVersion: String;
    procedure SetAutoRedirect(const Value: Boolean);
    procedure SetVersion(const Value: String);
    function GetRequestHeader: TJObject;
    function GetResponseHeader: TJResponseObject;
    function GetProxy: String;
    procedure SetProxy(const Value: String);
  protected
    function GetTimeout: Integer; override;
    procedure SetTimeout(const Value: Integer); override;
  public
    constructor Create(AFactory: TJObjectFactory; Param: TJValueList); override;
    destructor Destroy; override;
  published
    property requestHeader: TJObject read GetRequestHeader;
    property responseHeader: TJResponseObject read GetResponseHeader;
    property version: String read GetVersion write SetVersion;
    property length: Integer read GetLength;
    property autoRedirect: Boolean read GetAutoRedirect write SetAutoRedirect;
    property proxy: String read GetProxy write SetProxy;
  end;
  //tcp socket
  TJTCPSocketObject = class(TJBaseSocketObject)
  private
    FSock: TgSocket;

    function DoConnect(Param: TJValueList): TJValue;
    function DoDisconnect(Param: TJValueList): TJValue;
    function DoAbort(Param: TJValueList): TJValue;
    function DoRead(Param: TJValueList): TJValue;
    function DoReadln(Param: TJValueList): TJValue;
    function DoWrite(Param: TJValueList): TJValue;
    function DoWriteln(Param: TJValueList): TJValue;
    function DoCapture(Param: TJValueList): TJValue;
    function DoSendFile(Param: TJValueList): TJValue;
    function DoBind(Param: TJValueList): TJValue;
    function DoAccept(Param: TJValueList): TJValue;
    function DoIsConnected(Param: TJValueList): TJValue;

    function GetHost: String;
    function GetPort: Integer;
    procedure SetHost(const Value: String);
    procedure SetPort(const Value: Integer);
  protected
     function GetTimeout: Integer; override;
    procedure SetTimeout(const Value: Integer); override;
  public
    constructor Create(AFactory: TJObjectFactory; Param: TJValueList); override;
    destructor Destroy; override;
  published
    property host: String read GetHost write SetHost;
    property port: Integer read GetPort write SetPort;
  end;

  TJMailObject = class(TJObject)
  private
    FMailMessage: TMailMessage;
    FAttach: TJStringsObject;

    function DoClear(Param: TJValueList): TJValue;
    function GetAttachments: TJObject;
    function GetBody: String;
    function GetHeader: String;
    function GetLength: Integer;
    function GetMessage: String;
    function GetNumber: Integer;
    procedure SetMessage(const Value: String);
  protected
    function GetPropertyList: String; override;
  public
    constructor Create(AFactory: TJObjectFactory; Param: TJValueList); override;
    destructor Destroy; override;
    procedure Clear;
    function GetValue(S: String; ArrayStyle: Boolean): TJValue; override;
    procedure SetValue(S: String; Value: TJValue; ArrayStyle: Boolean); override;
  published
    property number: Integer read GetNumber;
    property length: Integer read GetLength;
    property attachments: TJObject read GetAttachments;
    property header: String read GetHeader;
    property body: String read GetBody;
    property message: String read GetMessage write SetMessage;
  end;

  TJPOP3Object = class(TJBaseSocketObject)
  private
    FPOP: TgPOP3;
    FMail: TJMailObject;

    function DoConnect(Param: TJValueList): TJValue;
    function DoDisconnect(Param: TJValueList): TJValue;
    function DoGetMail(Param: TJValueList): TJValue;
    function DoGetSummary(Param: TJValueList): TJValue;
    function DoDelete(Param: TJValueList): TJValue;
    function GetAttachFilePath: String;
    function GetDeleteOnRead: Boolean;
    function GetMail: TJMailObject;
    function GetMailCount: Integer;
    function GetPassword: String;
    function GetUserId: String;
    procedure SetAttachFilePath(const Value: String);
    procedure SetDeleteOnRead(const Value: Boolean);
    procedure SetPassword(const Value: String);
    procedure SetUserId(const Value: String);
    function GetHost: String;
    function GetPort: Integer;
    procedure SetHost(const Value: String);
    procedure SetPort(const Value: Integer);
  protected
    function GetTimeout: Integer; override;
    procedure SetTimeout(const Value: Integer); override;
  public
    constructor Create(AFactory: TJObjectFactory; Param: TJValueList); override;
    destructor Destroy; override;
  published
    property attachPath: String read GetAttachFilePath write SetAttachFilePath ;
    property deleteOnRead: Boolean read GetDeleteOnRead write SetDeleteOnRead ;
    property mailCount: Integer read GetMailCount;
    property length: Integer read GetMailCount;
    property password: String read GetPassword write SetPassword ;
    property userid: String read GetUserId write SetUserId ;
    property mail: TJMailObject read GetMail;
    property host: String read GetHost write SetHost;
    property port: Integer read GetPort write SetPort;
  end;

  TJSMTPObject = class(TJBaseSocketObject)
  private
    FSMTP: TgSMTP;
    FMail: TJMailObject;

    function DoConnect(Param: TJValueList): TJValue;
    function DoDisconnect(Param: TJValueList): TJValue;
    function DoSendMail(Param: TJValueList): TJValue;
    function GetHost: String;
    function GetMail: TJMailObject;
    function GetPort: Integer;
    procedure SetHost(const Value: String);
    procedure SetPort(const Value: Integer);
  protected
    function GetTimeout: Integer; override;
    procedure SetTimeout(const Value: Integer); override;
  public
    constructor Create(AFactory: TJObjectFactory; Param: TJValueList); override;
    destructor Destroy; override;
  published
    property mail: TJMailObject read GetMail;
    property host: String read GetHost write SetHost;
    property port: Integer read GetPort write SetPort;
  end;

  TJHtmlTagObject = class(TJObject)
  private
    FName: String;
  protected
    function GetPropertyList: String; override;
  public
    constructor Create(AFactory: TJObjectFactory; Param: TJValueList); override;
    function ToString(Value: PJValue = nil): String; override;
    procedure SetValue(S: String; Value: TJValue; ArrayStyle: Boolean); override;

    procedure Assign(Source: THtmlTag);
  published
    property name: String read FName;
  end;

  TJHtmlParserObject = class(TJObject)
  private
    FHtml: String;
    FParser: THtmlParser;

    function DoParse(Param: TJValueList): TJValue;
    function DoParseFile(Param: TJValueList): TJValue;
    function DoClear(Param: TJValueList): TJValue;
    function GetCount: Integer;
    function GetHtml: String;
    function GetText: String;
  protected
    function GetPropertyList: String; override;
  public
    constructor Create(AFactory: TJObjectFactory; Param: TJValueList); override;
    destructor Destroy; override;
    function GetValue(S: String; ArrayStyle: Boolean): TJValue; override;
    function ToString(Value: PJValue = nil): String; override;
  published
    property text: String read GetText;
    property html: String read GetHtml;
    property count: Integer read GetCount;
    property length: Integer read GetCount;
  end;

  
    





implementation



{ TJURLInfoObject }

constructor TJURLInfoObject.Create(AFactory: TJObjectFactory;
  Param: TJValueList);
var
  v: TJValue;
begin
  inherited;
  RegistName('URL');

  if IsParam1(Param) then
  begin
    v := Param[0];
    FInfo := ParseUrl(AsString(@v));
  end;
end;

function TJURLInfoObject.GetDir: String;
begin
  Result := FInfo.Dir;
end;

function TJURLInfoObject.GetFilename: String;
begin
  Result := FInfo.FileName;
end;

function TJURLInfoObject.GetHost: String;
begin
  if FInfo.Port <> '' then
    Result := FInfo.host + ':' + FInfo.Port
  else
    Result := FInfo.Host;
end;

function TJURLInfoObject.GetHostname: String;
begin
  Result := FInfo.Host;
end;

function TJURLInfoObject.GetPass: String;
begin
  Result := FInfo.Password;
end;

function TJURLInfoObject.GetPath: String;
begin
  Result := FInfo.Path;
end;

function TJURLInfoObject.GetPort: String;
begin
  Result := FInfo.Port;
end;

function TJURLInfoObject.GetProtocol: String;
begin
  Result := FInfo.Protocol;
end;

function TJURLInfoObject.GetQuery: String;
begin
  Result := FInfo.Query;
end;

function TJURLInfoObject.GetUrl: String;
begin
  Result := BuildUrl(FInfo);
end;

function TJURLInfoObject.GetUser: String;
begin
  Result := FInfo.Userid;
end;

procedure TJURLInfoObject.SetDir(const Value: String);
begin
  FInfo.Dir := Value;
end;

procedure TJURLInfoObject.SetFilename(const Value: String);
begin
  FInfo.FileName := Value;
end;

procedure TJURLInfoObject.SetHost(const Value: String);
var
  index: Integer;
begin
  index := Pos(':',Value);
  if index > 0 then
  begin
    FInfo.Host := Copy(Value,1,index - 1);
    FInfo.Port := Copy(Value,index + 1,MaxInt);
  end
  else
    FInfo.Host := Value;
end;

procedure TJURLInfoObject.SetHostname(const Value: String);
begin
  FInfo.Host := Value;
end;

procedure TJURLInfoObject.SetPass(const Value: String);
begin
  FInfo.Password := Value;
end;

procedure TJURLInfoObject.SetPath(const Value: String);
begin
  FInfo.Path := Value;
end;

procedure TJURLInfoObject.SetPort(const Value: String);
begin
  FInfo.Port := Value;
end;

procedure TJURLInfoObject.SetProtocol(const Value: String);
begin
  FInfo.Protocol := Value;
end;

procedure TJURLInfoObject.SetQuery(const Value: String);
begin
  FInfo.Query := Value;
end;

procedure TJURLInfoObject.SetUrl(const Value: String);
begin
  FInfo := ParseUrl(Value);
end;

procedure TJURLInfoObject.SetUser(const Value: String);
begin
  FInfo.UserId := Value;
end;


{ TJCookieObject }

constructor TJCookieObject.Create(AFactory: TJObjectFactory;
  Param: TJValueList);
begin
  inherited;
  RegistName('Cookie');

  FCookie := TCookie.Create;
  FExpires := TJDateObject.Create(FFactory,nil);
  FExpires.IncRefCount;
end;

destructor TJCookieObject.Destroy;
begin
  FExpires.DecRefCount;
  FCookie.Free;
  inherited;
end;

function TJCookieObject.GetDomain: String;
begin
  Result := FCookie.Domain;
end;

function TJCookieObject.GetExpires: TJDateObject;
begin
  FExpires.UTC := FCookie.Expire;
  Result := FExpires;                 
end;

function TJCookieObject.GetPath: String;
begin
  Result := FCookie.Path;
end;

procedure TJCookieObject.SetDomain(const Value: String);
begin
  FCookie.Domain := Value;
end;

procedure TJCookieObject.SetPath(const Value: String);
begin
  FCookie.Path := Value;
end;

{ TJResponseObject }

procedure TJResponseObject.Clear;
begin
  ClearProperties;
  FCode := 999;
  FText := '';
end;

constructor TJResponseObject.Create(AFactory: TJObjectFactory;
  Param: TJValueList);
begin
  inherited;
  RegistName('Response');

  FCookie := TJCookieObject.Create(AFactory,nil);
  FCookie.IncRefCount;
end;

destructor TJResponseObject.Destroy;
begin
  FCookie.DecRefCount;
  inherited;
end;

function TJResponseObject.GetCookie: TJCookieObject;
begin
  Result := FCookie;
end;

function TJResponseObject.GetPropertyList: String;
//SẴo𓾂
var
  sl: TStringList;
  i: Integer;
  v: TJValue;
begin
  sl := TStringList.Create;
  try
    sl.Text := Members.Keys;
    for i := sl.Count - 1 downto 0 do
    begin
      //ȊO͍폜
      v := Members[sl[i]];
      if not IsString(@v) then
        sl.Delete(i);
    end;

    Result := Trim(sl.Text);
  finally
    sl.Free;
  end;
end;

procedure TJResponseObject.SetStatus(ACode: Integer; AText: String);
begin
  FCode := ACode;
  FText := AText;
end;

{ TJHTTPObject }

constructor TJHTTPObject.Create(AFactory: TJObjectFactory;
  Param: TJValueList);
begin
  inherited;
  RegistName('HTTP');

  FHTTP := TgHTTP.Create;
  FHTTP.OnStatus := SockOnStatus;
  FRequestHeader := TJObject.Create(FFactory,nil);
  FRequestHeader.IncRefCount;
  FResponseHeader := TJResponseObject.Create(AFactory,nil);
  FResponseHeader.IncRefCount;

  RegistMethod('get',DoGet);
  RegistMethod('getFile',DoGetFile);
  RegistMethod('post',DoPost);
  RegistMethod('head',DoHead);
  RegistMethod('request',DoRequest);
  RegistMethod('response',DoResponse);
  RegistMethod('capture',DoCapture);
  RegistMethod('readln',DoReadln);
  RegistMethod('read',DoRead);
  RegistMethod('writeln',DoWriteln);
  RegistMethod('write',DoWrite);
  RegistMethod('connect',DoConnect);
  RegistMethod('disconnect',DoDisconnect);
  RegistMethod('open',DoConnect);
  RegistMethod('close',DoDisconnect);
end;

destructor TJHTTPObject.Destroy;
begin
  FRequestHeader.DecRefCount;
  FResponseHeader.DecRefCount;

  FHTTP.OnStatus := nil;
  FHTTP.Free;
  inherited;           
end;

function TJHTTPObject.DoCapture(Param: TJValueList): TJValue;
var
  v: TJValue;
begin
  EmptyValue(Result);
  if IsParam1(Param) then
  begin
    v := Param[0];
    try
      FHTTP.CaptureFile(AsString(@v));
    except
      raise EJThrow.Create(E_SOCKET,'HTTP.capture error');
    end;
  end;
end;

function TJHTTPObject.DoConnect(Param: TJValueList): TJValue;
begin
  EmptyValue(Result);
  try
    FHTTP.Connect;
  except
    raise EJThrow.Create(E_SOCKET,'HTTP.connect error');
  end;
end;

function TJHTTPObject.DoDisconnect(Param: TJValueList): TJValue;
begin
   EmptyValue(Result);
  try
    FHTTP.Disconnect;
  except
    raise EJThrow.Create(E_SOCKET,'HTTP.disconnect error');
  end;
end;

function TJHTTPObject.DoGet(Param: TJValueList): TJValue;
var
  v: TJValue;
  sl: TStringList;
  i: Integer;
begin
  EmptyValue(Result);
  if IsParam1(Param) then
  begin      
    try
      sl := TStringList.Create;
      try
        //request
        FHTTP.ReqHeader.Clear;
        sl.Text := FRequestHeader.Members.Keys;
        for i := 0 to sl.Count - 1 do
        begin
          v := FRequestHeader.Members[sl[i]];
          if IsString(@v) then
            FHTTP.ReqHeader[sl[i]] := AsString(@v);
        end;

        v := Param[0];
        Result := BuildString(FHTTP.Get(AsString(@v)));
        //response
        FResponseHeader.Clear;
        FResponseHeader.SetStatus(FHTTP.StatusNo,FHTTP.Status);
        FResponseHeader.FCookie.FCookie.Assign(FHTTP.ResHeader.Cookie);
        sl.Text := FHTTP.ResHeader.Hash.Keys;
        for i := 0 to sl.Count - 1 do
          FResponseHeader.Members[sl[i]] := BuildString(FHTTP.ResHeader.Hash[sl[i]]);

      finally
        sl.Free;
      end;
    except
      raise EJThrow.Create(E_SOCKET,'HTTP.get error');
    end;
  end;
end;

function TJHTTPObject.DoGetFile(Param: TJValueList): TJValue;
var
  v,f: TJValue;
  sl: TStringList;
  i: Integer;
begin
  EmptyValue(Result);
  if IsParam2(Param) then
  begin      
    try
      sl := TStringList.Create;
      try
        //request
        FHTTP.ReqHeader.Clear;
        sl.Text := FRequestHeader.Members.Keys;
        for i := 0 to sl.Count - 1 do
        begin
          v := FRequestHeader.Members[sl[i]];
          if IsString(@v) then
            FHTTP.ReqHeader[sl[i]] := AsString(@v);
        end;

        v := Param[0];
        f := Param[1];
        FHTTP.GetFile(AsString(@v),AsString(@f));
        //response
        FResponseHeader.Clear;
        FResponseHeader.SetStatus(FHTTP.StatusNo,FHTTP.Status);
        FResponseHeader.FCookie.FCookie.Assign(FHTTP.ResHeader.Cookie);
        sl.Text := FHTTP.ResHeader.Hash.Keys;
        for i := 0 to sl.Count - 1 do
          FResponseHeader.Members[sl[i]] := BuildString(FHTTP.ResHeader.Hash[sl[i]]);

      finally
        sl.Free;
      end;
    except
      raise EJThrow.Create(E_SOCKET,'HTTP.get error');
    end;
  end;

end;

function TJHTTPObject.DoHead(Param: TJValueList): TJValue;
var
  v: TJValue;
  sl: TStringList;
  i: Integer;
begin
  EmptyValue(Result);
  if IsParam1(Param) then
  begin      
    try
      sl := TStringList.Create;
      try
        //request
        FHTTP.ReqHeader.Clear;
        sl.Text := FRequestHeader.Members.Keys;
        for i := 0 to sl.Count - 1 do
        begin
          v := FRequestHeader.Members[sl[i]];
          if IsString(@v) then
            FHTTP.ReqHeader[sl[i]] := AsString(@v);
        end;

        v := Param[0];
        FHTTP.Head(AsString(@v));
        //response
        FResponseHeader.Clear;
        FResponseHeader.SetStatus(FHTTP.StatusNo,FHTTP.Status);
        FResponseHeader.FCookie.FCookie.Assign(FHTTP.ResHeader.Cookie);
        sl.Text := FHTTP.ResHeader.Hash.Keys;
        for i := 0 to sl.Count - 1 do
          FResponseHeader.Members[sl[i]] := BuildString(FHTTP.ResHeader.Hash[sl[i]]);

      finally
        sl.Free;
      end;
    except
      raise EJThrow.Create(E_SOCKET,'HTTP.head error');
    end;
  end;
end;

function TJHTTPObject.DoPost(Param: TJValueList): TJValue;
var
  v,data: TJValue;
  sl: TStringList;
  i: Integer;
begin
  EmptyValue(Result);
  if IsParam2(Param) then
  begin
    try
      sl := TStringList.Create;
      try
        //request
        FHTTP.ReqHeader.Clear;
        sl.Text := FRequestHeader.Members.Keys;
        for i := 0 to sl.Count - 1 do
        begin
          v := FRequestHeader.Members[sl[i]];
          if IsString(@v) then
            FHTTP.ReqHeader[sl[i]] := AsString(@v);
        end;

        v := Param[0];
        data := Param[1];
        Result := BuildString(FHTTP.Post(AsString(@v),AsString(@data)));
        //response
        FResponseHeader.Clear;
        FResponseHeader.SetStatus(FHTTP.StatusNo,FHTTP.Status);
        FResponseHeader.FCookie.FCookie.Assign(FHTTP.ResHeader.Cookie);
        sl.Text := FHTTP.ResHeader.Hash.Keys;
        for i := 0 to sl.Count - 1 do
          FResponseHeader.Members[sl[i]] := BuildString(FHTTP.ResHeader.Hash[sl[i]]);

      finally
        sl.Free;
      end;
    except
      raise EJThrow.Create(E_SOCKET,'HTTP.post error');
    end;
  end;

end;

function TJHTTPObject.DoRead(Param: TJValueList): TJValue;
var
  v: TJValue;
  len: Integer;
begin
  if IsParam1(Param) then
  begin
    v := Param[0];
    len := AsInteger(@v);
  end
  else
    len := -1;

  try
    Result := BuildString(FHTTP.Read(len))
  except
    raise EJThrow.Create(E_SOCKET,'HTTP.read error');
  end;
end;

function TJHTTPObject.DoReadln(Param: TJValueList): TJValue;
begin
  try
    Result := BuildString(FHTTP.Readln)
  except
    raise EJThrow.Create(E_SOCKET,'HTTP.readln error');
  end;
end;

function TJHTTPObject.DoRequest(Param: TJValueList): TJValue;
var
  v,meth,ur,data: TJValue;
  sl: TStringList;
  i: Integer;
begin
  EmptyValue(Result);
  if IsParam2(Param) then
  begin
    try
      sl := TStringList.Create;
      try
        //request
        FHTTP.ReqHeader.Clear;
        sl.Text := FRequestHeader.Members.Keys;
        for i := 0 to sl.Count - 1 do
        begin
          v := FRequestHeader.Members[sl[i]];
          if IsString(@v) then
            FHTTP.ReqHeader[sl[i]] := AsString(@v);
        end;

        meth := Param[0];
        ur := Param[1];
        if IsParam3(Param) then
          data := Param[2]
        else
          data := BuildString('');

        FHTTP.Request(AsString(@meth),AsString(@ur),AsString(@data));
      finally
        sl.Free;
      end;
    except
      raise EJThrow.Create(E_SOCKET,'HTTP.request error');
    end;
  end;

end;

function TJHTTPObject.DoResponse(Param: TJValueList): TJValue;
var
  sl: TStringList;
  i: Integer;
begin
  EmptyValue(Result);
  try
    FHTTP.Response;
    //response
    FResponseHeader.Clear;
    FResponseHeader.SetStatus(FHTTP.StatusNo,FHTTP.Status);
    FResponseHeader.FCookie.FCookie.Assign(FHTTP.ResHeader.Cookie);
    sl := TStringList.Create;
    try
      sl.Text := FHTTP.ResHeader.Hash.Keys;
      for i := 0 to sl.Count - 1 do
        FResponseHeader.Members[sl[i]] := BuildString(FHTTP.ResHeader.Hash[sl[i]]);

    finally
      sl.Free;
    end;
  except
    raise EJThrow.Create(E_SOCKET,'HTTP.response error');
  end;

end;

function TJHTTPObject.DoWrite(Param: TJValueList): TJValue;
var
  v: TJValue;
begin
  EmptyValue(Result);
  if IsParam1(Param) then
  begin
    v := Param[0];
    try
      FHTTP.Write(AsString(@v));
    except
      raise EJThrow.Create(E_SOCKET,'HTTP.write error');
    end;
  end;
end;

function TJHTTPObject.DoWriteln(Param: TJValueList): TJValue;
var
  v: TJValue;
  s: String;
begin
  EmptyValue(Result);
  if IsParam1(Param) then
  begin
    v := Param[0];
    s := AsString(@v);
  end
  else
    s := '';

  try
    FHTTP.Writeln(s);
  except
    raise EJThrow.Create(E_SOCKET,'HTTP.writeln error');
  end;
end;

function TJHTTPObject.GetAutoRedirect: Boolean;
begin
  Result := FHTTP.AutoRedirect;
end;

function TJHTTPObject.GetLength: Integer;
begin
  Result := FHTTP.BodyLength;
end;

function TJHTTPObject.GetProxy: String;
begin
  Result := FHTTP.Proxy;
end;

function TJHTTPObject.GetRequestHeader: TJObject;
begin
  Result := FRequestHeader;
end;

function TJHTTPObject.GetResponseHeader: TJResponseObject;
begin
  Result := FResponseHeader;
end;

function TJHTTPObject.GetTimeout: Integer;
begin
  Result := FHTTP.Timeout;
end;

function TJHTTPObject.GetVersion: String;
begin
  Result :=  FHTTP.Version;
end;

procedure TJHTTPObject.SetAutoRedirect(const Value: Boolean);
begin
  FHTTP.AutoRedirect := Value;
end;

procedure TJHTTPObject.SetProxy(const Value: String);
begin
  FHTTP.Proxy := Value;
end;

procedure TJHTTPObject.SetTimeout(const Value: Integer);
begin
  FHTTP.Timeout := Value;
end;

procedure TJHTTPObject.SetVersion(const Value: String);
begin
  FHTTP.Version := Value;
end;


{ TJHTTPSObject }

constructor TJHTTPSObject.Create(AFactory: TJObjectFactory;
  Param: TJValueList);
begin
  inherited;
  RegistName('HTTPS');

  FHTTPS := TgHTTPS.Create;
  FHTTPS.OnStatus := SockOnStatus;
  FRequestHeader := TJObject.Create(AFactory,nil);
  FRequestHeader.IncRefCount;
  FResponseHeader := TJResponseObject.Create(AFactory,nil);
  FResponseHeader.IncRefCount;

  RegistMethod('get',DoGet);
  RegistMethod('getFile',DoGetFile);
  RegistMethod('post',DoPost);
  RegistMethod('head',DoHead);
  RegistMethod('request',DoRequest);
  RegistMethod('response',DoResponse);
  RegistMethod('capture',DoCapture);
  RegistMethod('readln',DoReadln);
  RegistMethod('read',DoRead);
  RegistMethod('connect',DoConnect);
  RegistMethod('disconnect',DoDisconnect);
  RegistMethod('open',DoConnect);
  RegistMethod('close',DoDisconnect);
end;

destructor TJHTTPSObject.Destroy;
begin
  FRequestHeader.DecRefCount;
  FResponseHeader.DecRefCount;
  FHTTPS.OnStatus := nil;
  FHTTPS.Free;
  inherited;
end;

function TJHTTPSObject.DoCapture(Param: TJValueList): TJValue;
var
  v: TJValue;
begin
  EmptyValue(Result);
  if IsParam1(Param) then
  begin
    v := Param[0];
    try
      FHTTPS.CaptureFile(AsString(@v));
    except
      raise EJThrow.Create(E_SOCKET,'HTTPS.capture error');
    end;
  end;
end;

function TJHTTPSObject.DoConnect(Param: TJValueList): TJValue;
begin
  EmptyValue(Result);
  try
    FHTTPS.Connect;
  except
    raise EJThrow.Create(E_SOCKET,'HTTPS.connect error');
  end;
end;

function TJHTTPSObject.DoDisconnect(Param: TJValueList): TJValue;
begin
  EmptyValue(Result);
  try
    FHTTPS.Disconnect;
  except
    raise EJThrow.Create(E_SOCKET,'HTTPS.disconnect error');
  end;
end;

function TJHTTPSObject.DoGet(Param: TJValueList): TJValue;
var
  v: TJValue;
  sl: TStringList;
  i: Integer;
begin
  EmptyValue(Result);
  if IsParam1(Param) then
  begin      
    try
      sl := TStringList.Create;
      try
        //request
        FHTTPS.ReqHeader.Clear;
        sl.Text := FRequestHeader.Members.Keys;
        for i := 0 to sl.Count - 1 do
        begin
          v := FRequestHeader.Members[sl[i]];
          if IsString(@v) then
            FHTTPS.ReqHeader[sl[i]] := AsString(@v);
        end;

        v := Param[0];
        Result := BuildString(FHTTPS.Get(AsString(@v)));
        //response
        FResponseHeader.Clear;
        FResponseHeader.SetStatus(FHTTPS.StatusNo,FHTTPS.Status);
        FResponseHeader.FCookie.FCookie.Assign(FHTTPS.ResHeader.Cookie);
        sl.Text := FHTTPS.ResHeader.Hash.Keys;
        for i := 0 to sl.Count - 1 do
          FResponseHeader.Members[sl[i]] := BuildString(FHTTPS.ResHeader.Hash[sl[i]]);

      finally
        sl.Free;
      end;
    except
      raise EJThrow.Create(E_SOCKET,'HTTPS.get error');
    end;
  end;
end;

function TJHTTPSObject.DoGetFile(Param: TJValueList): TJValue;
var
  v,f: TJValue;
  sl: TStringList;
  i: Integer;
begin
  EmptyValue(Result);
  if IsParam2(Param) then
  begin      
    try
      sl := TStringList.Create;
      try
        //request
        FHTTPS.ReqHeader.Clear;
        sl.Text := FRequestHeader.Members.Keys;
        for i := 0 to sl.Count - 1 do
        begin
          v := FRequestHeader.Members[sl[i]];
          if IsString(@v) then
            FHTTPS.ReqHeader[sl[i]] := AsString(@v);
        end;

        v := Param[0];
        f := Param[1];
        FHTTPS.GetFile(AsString(@v),AsString(@f));
        //response
        FResponseHeader.Clear;
        FResponseHeader.SetStatus(FHTTPS.StatusNo,FHTTPS.Status);
        FResponseHeader.FCookie.FCookie.Assign(FHTTPS.ResHeader.Cookie);
        sl.Text := FHTTPS.ResHeader.Hash.Keys;
        for i := 0 to sl.Count - 1 do
          FResponseHeader.Members[sl[i]] := BuildString(FHTTPS.ResHeader.Hash[sl[i]]);

      finally
        sl.Free;
      end;
    except
      raise EJThrow.Create(E_SOCKET,'HTTPS.get error');
    end;
  end;

end;

function TJHTTPSObject.DoHead(Param: TJValueList): TJValue;
var
  v: TJValue;
  sl: TStringList;
  i: Integer;
begin
  EmptyValue(Result);
  if IsParam1(Param) then
  begin      
    try
      sl := TStringList.Create;
      try
        //request
        FHTTPS.ReqHeader.Clear;
        sl.Text := FRequestHeader.Members.Keys;
        for i := 0 to sl.Count - 1 do
        begin
          v := FRequestHeader.Members[sl[i]];
          if IsString(@v) then
            FHTTPS.ReqHeader[sl[i]] := AsString(@v);
        end;

        v := Param[0];
        FHTTPS.Head(AsString(@v));
        //response
        FResponseHeader.Clear;
        FResponseHeader.SetStatus(FHTTPS.StatusNo,FHTTPS.Status);
        FResponseHeader.FCookie.FCookie.Assign(FHTTPS.ResHeader.Cookie);
        sl.Text := FHTTPS.ResHeader.Hash.Keys;
        for i := 0 to sl.Count - 1 do
          FResponseHeader.Members[sl[i]] := BuildString(FHTTPS.ResHeader.Hash[sl[i]]);

      finally
        sl.Free;
      end;
    except
      raise EJThrow.Create(E_SOCKET,'HTTPS.head error');
    end;
  end;
end;

function TJHTTPSObject.DoPost(Param: TJValueList): TJValue;
var
  v,data: TJValue;
  sl: TStringList;
  i: Integer;
begin
  EmptyValue(Result);
  if IsParam2(Param) then
  begin
    try
      sl := TStringList.Create;
      try
        //request
        FHTTPS.ReqHeader.Clear;
        sl.Text := FRequestHeader.Members.Keys;
        for i := 0 to sl.Count - 1 do
        begin
          v := FRequestHeader.Members[sl[i]];
          if IsString(@v) then
            FHTTPS.ReqHeader[sl[i]] := AsString(@v);
        end;

        v := Param[0];
        data := Param[1];
        Result := BuildString(FHTTPS.Post(AsString(@v),AsString(@data)));
        //response
        FResponseHeader.Clear;
        FResponseHeader.SetStatus(FHTTPS.StatusNo,FHTTPS.Status);
        FResponseHeader.FCookie.FCookie.Assign(FHTTPS.ResHeader.Cookie);
        sl.Text := FHTTPS.ResHeader.Hash.Keys;
        for i := 0 to sl.Count - 1 do
          FResponseHeader.Members[sl[i]] := BuildString(FHTTPS.ResHeader.Hash[sl[i]]);

      finally
        sl.Free;
      end;
    except
      raise EJThrow.Create(E_SOCKET,'HTTPS.post error');
    end;
  end;

end;

function TJHTTPSObject.DoRead(Param: TJValueList): TJValue;
var
  v: TJValue;
  len: Integer;
begin
  if IsParam1(Param) then
  begin
    v := Param[0];
    len := AsInteger(@v);
  end
  else
    len := -1;

  try
    Result := BuildString(FHTTPS.Read(len))
  except
    raise EJThrow.Create(E_SOCKET,'HTTPS.read error');
  end;
end;

function TJHTTPSObject.DoReadln(Param: TJValueList): TJValue;
begin
  try
    Result := BuildString(FHTTPS.Readln)
  except
    raise EJThrow.Create(E_SOCKET,'HTTPS.readln error');
  end;
end;

function TJHTTPSObject.DoRequest(Param: TJValueList): TJValue;
var
  v,meth,ur,data: TJValue;
  sl: TStringList;
  i: Integer;
begin
  EmptyValue(Result);
  if IsParam2(Param) then
  begin
    try
      sl := TStringList.Create;
      try
        //request
        FHTTPS.ReqHeader.Clear;
        sl.Text := FRequestHeader.Members.Keys;
        for i := 0 to sl.Count - 1 do
        begin
          v := FRequestHeader.Members[sl[i]];
          if IsString(@v) then
            FHTTPS.ReqHeader[sl[i]] := AsString(@v);
        end;

        meth := Param[0];
        ur := Param[1];
        if IsParam3(Param) then
          data := Param[2]
        else
          data := BuildString('');

        FHTTPS.Request(AsString(@meth),AsString(@ur),AsString(@data));
      finally
        sl.Free;
      end;
    except
      raise EJThrow.Create(E_SOCKET,'HTTPS.request error');
    end;
  end;

end;

function TJHTTPSObject.DoResponse(Param: TJValueList): TJValue;
var
  sl: TStringList;
  i: Integer;
begin
  EmptyValue(Result);
  try
    FHTTPS.Response;
    //response
    FResponseHeader.Clear;
    FResponseHeader.SetStatus(FHTTPS.StatusNo,FHTTPS.Status);
    FResponseHeader.FCookie.FCookie.Assign(FHTTPS.ResHeader.Cookie);
    sl := TStringList.Create;
    try
      sl.Text := FHTTPS.ResHeader.Hash.Keys;
      for i := 0 to sl.Count - 1 do
        FResponseHeader.Members[sl[i]] := BuildString(FHTTPS.ResHeader.Hash[sl[i]]);

    finally
      sl.Free;
    end;
  except
    raise EJThrow.Create(E_SOCKET,'HTTPS.response error');
  end;

end;

function TJHTTPSObject.GetAutoRedirect: Boolean;
begin
  Result := FHTTPS.AutoRedirect;
end;

function TJHTTPSObject.GetLength: Integer;
begin
  Result := FHTTPS.BodyLength;
end;

function TJHTTPSObject.GetProxy: String;
begin
  Result := FHTTPS.Proxy;
end;

function TJHTTPSObject.GetRequestHeader: TJObject;
begin
  Result := FRequestHeader;
end;

function TJHTTPSObject.GetResponseHeader: TJResponseObject;
begin
  Result := FResponseHeader;
end;

function TJHTTPSObject.GetTimeout: Integer;
begin
  Result := FHTTPS.Timeout;
end;

function TJHTTPSObject.GetVersion: String;
begin
  Result :=  FHTTPS.Version;
end;

procedure TJHTTPSObject.SetAutoRedirect(const Value: Boolean);
begin
  FHTTPS.AutoRedirect := Value;
end;

procedure TJHTTPSObject.SetProxy(const Value: String);
begin
  FHTTPS.Proxy := Value;
end;

procedure TJHTTPSObject.SetTimeout(const Value: Integer);
begin
  FHTTPS.Timeout := Value;
end;

procedure TJHTTPSObject.SetVersion(const Value: String);
begin
  FHTTPS.Version := Value;
end;


{ TJTCPSocketObject }

constructor TJTCPSocketObject.Create(AFactory: TJObjectFactory;
  Param: TJValueList);
var
  v: TJValue;
begin
  inherited;
  RegistName('TCPSocket');

  if IsParam1(Param) then
  begin
    v := Param[0];
    try
      FSock.CreateFromServer(BUFFER_SIZE,AsInteger(@v));
    except
      raise EJThrow.Create(E_SOCKET,'TCPSocket.create error');
    end;
  end
  else
    FSock := TgSocket.Create(BUFFER_SIZE);

  FSock.OnStatus := SockOnStatus;

  RegistMethod('connect',DoConnect);
  RegistMethod('disconnect',DoDisconnect);
  RegistMethod('open',DoConnect);
  RegistMethod('close',DoDisconnect);
  RegistMethod('abort',DoAbort);
  RegistMethod('read',DoRead);
  RegistMethod('readln',DoReadln);
  RegistMethod('write',DoWrite);
  RegistMethod('writeln',DoWriteln);
  RegistMethod('capture',DoCapture);
  RegistMethod('sendFile',DoSendFile);
  RegistMethod('bind',DoBind);
  RegistMethod('accept',DoAccept);
  RegistMethod('isConnected',DoIsConnected);
end;

destructor TJTCPSocketObject.Destroy;
begin
  FSock.OnStatus := nil;
  FSock.Free;
  inherited;
end;

function TJTCPSocketObject.DoAbort(Param: TJValueList): TJValue;
begin
  EmptyValue(Result);
  try
    FSock.Abort;
  except
    raise EJThrow.Create(E_SOCKET,'TCPSocket.abort error');
  end;                                                     
end;

function TJTCPSocketObject.DoAccept(Param: TJValueList): TJValue;
var
  s: TSocket;
  obj: TJTCPSocketObject;
  list: TJValueList;
begin
  try
    s := FSock.Accept;
    if s = INVALID_SOCKET then
      raise EJThrow.Create(E_SOCKET,'TCPSocket.accept error')
    else begin
      list := TJValueList.Create;
      try
        list.Add(BuildInteger(S));
        //socket𑗂
        obj := TJTCPSocketObject.Create(FFactory,list);
        Result := BuildObject(obj);
      finally
        list.Free;
      end;
    end;
  except
    raise EJThrow.Create(E_SOCKET,'TCPSocket.accept error');
  end; 
end;

function TJTCPSocketObject.DoBind(Param: TJValueList): TJValue;
begin
  EmptyValue(Result);
  try
    FSock.Bind;
  except
    raise EJThrow.Create(E_SOCKET,'TCPSocket.bind error');
  end; 
end;

function TJTCPSocketObject.DoCapture(Param: TJValueList): TJValue;
var
  v: TJValue;
begin
  EmptyValue(Result);
  if IsParam1(Param) then
  begin
    v := Param[0];
    try
      FSock.CaptureFile(AsString(@v));
    except
      raise EJThrow.Create(E_SOCKET,'TCPSocket.capture error');
    end;
  end;
end;

function TJTCPSocketObject.DoConnect(Param: TJValueList): TJValue;
begin
  EmptyValue(Result);
  try
    FSock.Connect;
  except
    raise EJThrow.Create(E_SOCKET,'TCPSocket.connect error');
  end;
end;

function TJTCPSocketObject.DoDisconnect(Param: TJValueList): TJValue;
begin
  EmptyValue(Result);
  try
    FSock.Disconnect;
  except
    raise EJThrow.Create(E_SOCKET,'TCPSocket.disconnect error');
  end;
end;

function TJTCPSocketObject.DoIsConnected(Param: TJValueList): TJValue;
begin
  Result := BuildBool(FSock.Connected);
end;

function TJTCPSocketObject.DoRead(Param: TJValueList): TJValue;
var
  v: TJValue;
  len: Integer;
begin
  EmptyValue(Result);
  if IsParam1(Param) then
  begin
    v := Param[0];
    len := AsInteger(@v);
  end
  else
    len := -1;

  try
    Result := BuildString(FSock.Read(len));
  except
    raise EJThrow.Create(E_SOCKET,'TCPSocket.read error');
  end;
end;

function TJTCPSocketObject.DoReadln(Param: TJValueList): TJValue;
begin
  EmptyValue(Result);
  try
    Result := BuildString(FSock.Readln);
  except
    raise EJThrow.Create(E_SOCKET,'TCPSocket.readln error');
  end; 
end;

function TJTCPSocketObject.DoSendFile(Param: TJValueList): TJValue;
var
  v: TJValue;
begin
  EmptyValue(Result);
  if IsParam1(Param) then
  begin
    v := Param[0];
    try
      FSock.SendFile(AsString(@v));
    except
      raise EJThrow.Create(E_SOCKET,'TCPSocket.sendFile error');
    end;
  end;
end;

function TJTCPSocketObject.DoWrite(Param: TJValueList): TJValue;
var
  v: TJValue;
begin
  EmptyValue(Result);
  if IsParam1(Param) then
  begin
    v := Param[0];
    try
      FSock.Write(AsString(@v));
    except
      raise EJThrow.Create(E_SOCKET,'TCPSocket.write error');
    end;
  end;

end;

function TJTCPSocketObject.DoWriteln(Param: TJValueList): TJValue;
var
  v: TJValue;
  s: String;
begin
  EmptyValue(Result);
  if IsParam1(Param) then
  begin
    v := Param[0];
    s := AsString(@v);
  end
  else
    s := '';

  try
    FSock.Writeln(s);
  except
    raise EJThrow.Create(E_SOCKET,'TCPSocket.writeln error');
  end;
end;

function TJTCPSocketObject.GetHost: String;
begin
  Result := FSock.Host;
end;

function TJTCPSocketObject.GetPort: Integer;
begin
  Result := FSock.Port;
end;

function TJTCPSocketObject.GetTimeout: Integer;
begin
  Result := FSock.Timeout;
end;

procedure TJTCPSocketObject.SetHost(const Value: String);
begin
  FSock.Host := Value;
end;

procedure TJTCPSocketObject.SetPort(const Value: Integer);
begin
  FSock.Port := Value;
end;

procedure TJTCPSocketObject.SetTimeout(const Value: Integer);
begin
  FSock.Timeout := Value;
end;

{ TJBaseSocketObject }

procedure TJBaseSocketObject.SockOnStatus(Sender: TObject;
  const Status: String);
begin
  if FDebug and Assigned(FOnPrint) then
    FOnPrint(Status);
end;

{ TJMailObject }

procedure TJMailObject.Clear;
begin
  FMailMessage.Clear;
end;

constructor TJMailObject.Create(AFactory: TJObjectFactory;
  Param: TJValueList);
begin
  inherited;
  RegistName('Mail');

  FMailMessage := TMailMessage.Create;
  FAttach := TJStringsObject.Create(AFactory,nil);
  FAttach.IncRefCount;

  RegistMethod('clear',DoClear);
end;

destructor TJMailObject.Destroy;
begin
  FAttach.DecRefCount;
  FMailMessage.Free;
  inherited;
end;

function TJMailObject.DoClear(Param: TJValueList): TJValue;
begin
  EmptyValue(Result);
  Clear;
end;

function TJMailObject.GetAttachments: TJObject;
begin
  FAttach.Strings.Assign(FMailMessage.Attachments);
  Result := FAttach;
end;

function TJMailObject.GetBody: String;
begin
  Result := FMailMessage.Body.Text;
end;

function TJMailObject.GetHeader: String;
begin
  Result := FMailMessage.Head.Text;
end;

function TJMailObject.GetLength: Integer;
begin
  Result := FMailMessage.Size;
end;

function TJMailObject.GetPropertyList: String;
begin
  Result := FMailMessage.Hash.Keys;
end;

function TJMailObject.GetMessage: String;
begin
  Result := FMailMessage.Message;
end;

function TJMailObject.GetNumber: Integer;
begin
  Result := FMailMessage.Number;
end;

function TJMailObject.GetValue(S: String; ArrayStyle: Boolean): TJValue;
begin
  EmptyValue(Result);
  if GetDefaultProperty(S,Result) then
    Exit
  else begin
    if ArrayStyle then
    begin
      //z
      if FMailMessage.Hash.HasKey(S) then
        Result := BuildString(FMailMessage.Value[S])
      else
        raise EJThrow.Create(E_KEY,S);
    end
    else begin
      //o
      if FMailMessage.Hash.HasKey(S) then
        Result := BuildString(FMailMessage.Value[S])
      else
        raise EJThrow.Create(E_NAME,S);
    end;
  end;   
end;

procedure TJMailObject.SetMessage(const Value: String);
begin
  FMailMessage.Message := Value;
end;

procedure TJMailObject.SetValue(S: String; Value: TJValue;
  ArrayStyle: Boolean);
begin
  if SetDefaultProperty(S,Value) then
    Exit
  else begin
    FMailMessage.Hash[S] := AsString(@Value)
  end;

end;

{ TJPOP3Object }

constructor TJPOP3Object.Create(AFactory: TJObjectFactory;
  Param: TJValueList);
begin
  inherited;
  RegistName('POP3');

  FPOP := TgPOP3.Create(BUFFER_SIZE);
  FPOP.OnStatus := SockOnStatus;
  FMail := TJMailObject.Create(AFactory,nil);
  FMail.IncRefCount;

  RegistMethod('connect', DoConnect);
  RegistMethod('disconnect', DoDisconnect);
  RegistMethod('open', DoConnect);
  RegistMethod('close', DoDisconnect);
  RegistMethod('getMail', DoGetMail);
  RegistMethod('getSummary', DoGetSummary);
  RegistMethod('delete', DoDelete);
end;

destructor TJPOP3Object.Destroy;
begin
  FMail.DecRefCount;
  FPOP.OnStatus := nil;
  FPOP.Free;
  inherited;
end;

function TJPOP3Object.DoConnect(Param: TJValueList): TJValue;
begin
  EmptyValue(Result);
  try
    FPOP.Connect;
  except
    raise EJThrow.Create(E_SOCKET,'POP3.connect');
  end;
end;

function TJPOP3Object.DoDelete(Param: TJValueList): TJValue;
var
  v: TJValue;
begin
  EmptyValue(Result);
  if IsParam1(Param) then
  begin
    v := Param[0];
    try
      FPOP.DeleteMailMessage(AsInteger(@v));
    except
      raise EJThrow.Create(E_SOCKET,'POP3.delete');
    end;
  end;
end;

function TJPOP3Object.DoDisconnect(Param: TJValueList): TJValue;
begin
  EmptyValue(Result);
  try
    FPOP.Disconnect;
  except
    raise EJThrow.Create(E_SOCKET,'POP3.disconnect');
  end;
end;

function TJPOP3Object.DoGetMail(Param: TJValueList): TJValue;
var
  v: TJValue;
begin
  EmptyValue(Result);
  if IsParam1(Param) then
  begin
    v := Param[0];
    try
      FPOP.GetMailMessage(AsInteger(@v));
      FMail.FMailMessage.Assign(FPOP.MailMessage);
      Result := BuildObject(FMail);
    except
      raise EJThrow.Create(E_SOCKET,'POP3.getMail');
    end;
  end;
end;

function TJPOP3Object.DoGetSummary(Param: TJValueList): TJValue;
var
  v: TJValue;
begin
  EmptyValue(Result);
  if IsParam1(Param) then
  begin
    v := Param[0];
    try
      FPOP.GetSummary(AsInteger(@v));
      FMail.FMailMessage.Assign(FPOP.MailMessage);
      Result := BuildObject(FMail);
    except
      raise EJThrow.Create(E_SOCKET,'POP3.getSummary');
    end;
  end;
end;

function TJPOP3Object.GetAttachFilePath: String;
begin
  Result := FPOP.AttachFilePath;
end;

function TJPOP3Object.GetDeleteOnRead: Boolean;
begin
  Result := FPOP.DeleteOnRead;
end;

function TJPOP3Object.GetHost: String;
begin
  Result := FPOP.Host;
end;

function TJPOP3Object.GetMail: TJMailObject;
begin
  Result := FMail;
end;

function TJPOP3Object.GetMailCount: Integer;
begin
  Result := FPOP.MailCount;
end;

function TJPOP3Object.GetPassword: String;
begin
  Result := FPOP.Password;
end;

function TJPOP3Object.GetPort: Integer;
begin
  Result := FPOP.Port;
end;

function TJPOP3Object.GetTimeout: Integer;
begin
  Result := FPOP.Timeout;
end;

function TJPOP3Object.GetUserId: String;
begin
  Result := FPOP.UserId;
end;

procedure TJPOP3Object.SetAttachFilePath(const Value: String);
begin
  FPOP.AttachFilePath := Value;
end;

procedure TJPOP3Object.SetDeleteOnRead(const Value: Boolean);
begin
  FPOP.DeleteOnRead := Value;
end;

procedure TJPOP3Object.SetHost(const Value: String);
begin
  FPOP.Host := Value;
end;

procedure TJPOP3Object.SetPassword(const Value: String);
begin
  FPOP.Password := Value;
end;

procedure TJPOP3Object.SetPort(const Value: Integer);
begin
  FPOP.Port := Value;
end;

procedure TJPOP3Object.SetTimeout(const Value: Integer);
begin
  FPOP.Timeout := Value;
end;

procedure TJPOP3Object.SetUserId(const Value: String);
begin
  FPOP.UserId := Value;
end;

{ TJSMTPObject }

constructor TJSMTPObject.Create(AFactory: TJObjectFactory;
  Param: TJValueList);
begin
  inherited;
  RegistName('SMTP');
  FSMTP := TgSMTP.Create(BUFFER_SIZE);
  FSMTP.OnStatus := SockOnStatus;
  FMail := TJMailObject.Create(AFactory,nil);
  FMail.IncRefCount;

  RegistMethod('connect', DoConnect);
  RegistMethod('disconnect', DoDisconnect);
  RegistMethod('open', DoConnect);
  RegistMethod('close', DoDisconnect);
  RegistMethod('sendMail', DoSendMail);
end;

destructor TJSMTPObject.Destroy;
begin
  FMail.DecRefCount;
  FSMTP.OnStatus := nil;
  FSMTP.Free;
  inherited;
end;

function TJSMTPObject.DoConnect(Param: TJValueList): TJValue;
begin
  EmptyValue(Result);
  try
    FSMTP.Connect;
  except
    raise EJThrow.Create(E_SOCKET,'SMTP.connect');
  end;
end;

function TJSMTPObject.DoDisconnect(Param: TJValueList): TJValue;
begin
  EmptyValue(Result);
  try
    FSMTP.Disconnect;
  except
    raise EJThrow.Create(E_SOCKET,'SMTP.disconnect');
  end;
end;

function TJSMTPObject.DoSendMail(Param: TJValueList): TJValue;
begin
  EmptyValue(Result);
  try
    FSMTP.MailMessage.Assign(FMail.FMailMessage);
    FSMTP.SendMail;
  except
    raise EJThrow.Create(E_SOCKET,'SMTP.sendMail');
  end;
end;

function TJSMTPObject.GetHost: String;
begin
  Result := FSMTP.Host;
end;

function TJSMTPObject.GetMail: TJMailObject;
begin
  Result := FMail;
end;

function TJSMTPObject.GetPort: Integer;
begin
  Result := FSMTP.Port;
end;

function TJSMTPObject.GetTimeout: Integer;
begin
  Result := FSMTP.Timeout;
end;

procedure TJSMTPObject.SetHost(const Value: String);
begin
  FSMTP.Host := Value;
end;

procedure TJSMTPObject.SetPort(const Value: Integer);
begin
  FSMTP.Port := Value;
end;

procedure TJSMTPObject.SetTimeout(const Value: Integer);
begin
  FSMTP.Timeout := Value;
end;



{ TJHtmlTagObject }

procedure TJHtmlTagObject.Assign(Source: THtmlTag);
var
  sl: TStringList;
  i: Integer;
begin
  //clear
  ClearMembers;
  RegistMethods;

  FName := Source.Name;
  sl := TStringList.Create;
  try
    sl.Text := Source.Keys;
    for i := 0 to sl.Count - 1 do
      SetValue(sl[i],BuildString(Source[sl[i]]),True);
  finally
    sl.Free;
  end;
end;

constructor TJHtmlTagObject.Create(AFactory: TJObjectFactory;
  Param: TJValueList);
begin
  inherited;
  RegistName('HtmlTag');
end;

function TJHtmlTagObject.GetPropertyList: String;
begin
  Result := inherited GetPropertyList;//Members.Keys;
end;

procedure TJHtmlTagObject.SetValue(S: String; Value: TJValue;
  ArrayStyle: Boolean);
begin
  if ArrayStyle then
    Members[S] := Value;
end;

function TJHtmlTagObject.ToString(Value: PJValue): String;
var
  keys: TSTringList;
  i: Integer;
  v: TJValue;
  s: String;
begin
  Result := '<' + FName + ' ';
  keys := Members.KeyList;
  s := '';
  for i := 0 to keys.Count - 1 do
  begin
    v := GetValue(keys[i],True);
    if AsString(@v) <> '' then
      s := s + keys[i] + '="' + AsSTring(@v) + '" '
    else
      s := s + keys[i] + ' ';
  end;

  Result := Result + TrimRight(s) + '>';
end;

{ TJHtmlParserObject }

constructor TJHtmlParserObject.Create(AFactory: TJObjectFactory;
  Param: TJValueList);
begin
  inherited;
  RegistName('HtmlParser');
  RegistMethod('parse',DoParse);
  RegistMethod('clear',DoClear);
  RegistMethod('parseFile',DoParseFile);

  FParser := THtmlParser.Create('');
  DoParse(Param);
end;

destructor TJHtmlParserObject.Destroy;
begin
  FreeAndNil(FParser);
  inherited;
end;

function TJHtmlParserObject.DoClear(Param: TJValueList): TJValue;
begin
  Result := BuildNull;
  FParser.Clear;
end;

function TJHtmlParserObject.DoParse(Param: TJValueList): TJValue;
var
  v: TJValue;
begin
  Result := BuildNull;
  if IsParam1(Param) then
  begin
    v := Param[0];
    FHtml := AsString(@v);
    FParser.Parse(FHtml);
  end;
end;

function TJHtmlParserObject.DoParseFile(Param: TJValueList): TJValue;
var
  sl: TStringList;
  f: String;
  v: TJValue;
begin
  Result := BuildNull;
  if IsParam1(Param) then
  begin
    v := Param[0];
    f := AsString(@v);
    if FileExists(f) then
    begin
      sl := TStringList.Create;
      try
        sl.LoadFromFile(f);
        FHtml := sl.Text;
        FParser.Parse(FHtml);
      finally
        sl.Free;
      end;
    end;
  end;
end;

function TJHtmlParserObject.GetCount: Integer;
begin
  Result := FParser.Count;
end;

function TJHtmlParserObject.GetHtml: String;
begin
  Result := FHtml;
end;

function TJHtmlParserObject.GetPropertyList: String;
var
  i: Integer;
  sl: TStringList;
begin
  sl := TStringList.Create;
  try
    for i := 0 to FParser.Count - 1 do
      sl.Add(IntToStr(i));

    Result := Trim(sl.Text);
  finally
    sl.Free;
  end;
end;

function TJHtmlParserObject.GetText: String;
begin
  Result := FParser.Text;
end;

function TJHtmlParserObject.GetValue(S: String;
  ArrayStyle: Boolean): TJValue;

  function TryNumber(S: String; var Num: Integer): Boolean;
  begin
    Num := 0;
    Result := False;
    try
      Num := StrToInt(S);
      Result := True;
    except
    end;
  end;
  
var
  tag: THtmlTag;
  index: Integer;
  obj: TJHtmlTagObject;
begin
  if ArrayStyle and TryNumber(S,index) then
  begin
    tag := FParser[index];
    if not Assigned(tag) then
      Result := BuildNull
    else begin
      obj := TJHtmlTagObject.Create(FFactory,nil);
      obj.Assign(tag);
      Result := BuildObject(obj);
    end;
  end
  else
    Result := inherited GetValue(S,ArrayStyle);
end;

function TJHtmlParserObject.ToString(Value: PJValue): String;
begin
  Result := FHtml;
end;

end.
