/***************************************************************************
 *            cxp-profile.c
 *
 *  Mon Nov 29 23:50:12 2004
 *  Copyright  2004  Yasumichi Akahoshi
 *  yasumichi@users.sourceforge.jp
 ****************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "cxp-profile.h"

enum
{
	CXP_PROFILE_APPNAME = 1,
	CXP_PROFILE_SECTION,
};

/**
 * @brief private member of CxpProfile
 */
typedef struct
{
	gchar *appname;			/**< application name */
	gchar *section;			/**< section name */
	GHashTable *config;		/**< hash table of configuration */
	gboolean dispose_has_run;	/**< Is dispose funciton executed */
} CxpProfilePrivate;

#define CXP_PROFILE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), CXP_TYPE_PROFILE, CxpProfilePrivate))

static GObjectClass *parent_class = NULL;

/*
 * privete methods
 */
static void cxp_profile_init (GTypeInstance * instance, gpointer g_class);
static void cxp_profile_class_init (gpointer g_class, gpointer g_class_data);
static void cxp_profile_dispose (GObject * obj);
static void cxp_profile_finalize (GObject * obj);
static void cxp_profile_get_property (GObject *object, guint property_id, GValue *value, GParamSpec *pspec);
static void cxp_profile_set_property (GObject *object, guint property_id, const GValue *value, GParamSpec *pspec);
static void cxp_profile_save_file (gpointer key, gpointer value,
				   gpointer channel);
static void cxp_profile_load_file (CxpProfile * profile);

/**
 * regist CxpProfile to GTypeTable.
 * @return GType
 */
GType cxp_profile_get_type (void)
{
	static GType type = 0;

	if (type == 0)
	{
		static const GTypeInfo info = {
			sizeof (CxpProfileClass),
			NULL,	/* base_init */
			NULL,	/* base_finalize */
			cxp_profile_class_init,	/* class_init */
			NULL,	/* class_finalize */
			NULL,	/* class_data */
			sizeof (CxpProfile),
			0,	/* n_preallocs */
			cxp_profile_init	/* instance_init */
		};
		type = g_type_register_static (G_TYPE_OBJECT, "CxpProfileType",
					       &info, 0);
	}

	return type;
}

/**
 * Init instance
 */
static void cxp_profile_init (GTypeInstance * instance, gpointer g_class)
{
	CxpProfilePrivate *priv = CXP_PROFILE_GET_PRIVATE(instance);

	priv->appname = NULL;
	priv->section = NULL;
	priv->config = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
	priv->dispose_has_run = FALSE;
}

/**
 * class init CxpProfileClass
 *
 */
static void cxp_profile_class_init (gpointer g_class, gpointer g_class_data)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (g_class);
	GParamSpec *pspec;

	g_type_class_add_private (g_class, sizeof (CxpProfilePrivate));

	gobject_class->dispose = cxp_profile_dispose;
	gobject_class->finalize = cxp_profile_finalize;
	gobject_class->set_property = cxp_profile_set_property;
	gobject_class->get_property = cxp_profile_get_property;

	/* Install properties. */
	pspec = g_param_spec_string ("appname",
			"Application's name",
			"Set application's name",
			NULL,
			G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
			CXP_PROFILE_APPNAME,
			pspec);

	pspec = g_param_spec_string ("section",
			"Section's name",
			"Set section's name",
			NULL,
			G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
			CXP_PROFILE_SECTION,
			pspec);

	parent_class = g_type_class_peek_parent (g_class);
}

static void cxp_profile_dispose (GObject * obj)
{
	CxpProfile *self = CXP_PROFILE (obj);
	CxpProfilePrivate *priv = CXP_PROFILE_GET_PRIVATE(obj);
	GIOChannel *channel;
	gchar *filename;

	if (priv->dispose_has_run)
	{
		return;
	}

	priv->dispose_has_run = TRUE;
	filename = g_build_filename (g_get_home_dir (),
				     ".cxp",
				     priv->appname,
				     priv->section, NULL);
	if ((channel = g_io_channel_new_file (filename, "w", NULL)) != NULL)
	{
		g_hash_table_foreach (priv->config, cxp_profile_save_file,
				      channel);
		g_io_channel_flush (channel, NULL);

		g_io_channel_unref (channel);
	}
	g_free (filename);

	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

static void cxp_profile_finalize (GObject * obj)
{
	CxpProfilePrivate *priv = CXP_PROFILE_GET_PRIVATE(obj);

	g_free (priv->appname);
	g_free (priv->section);
	g_hash_table_destroy (priv->config);

	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

static void cxp_profile_set_property (GObject *object, guint property_id, const GValue *value, GParamSpec *pspec)
{
	CxpProfilePrivate *private = CXP_PROFILE_GET_PRIVATE(object);

	switch (property_id) {
		case CXP_PROFILE_APPNAME:
			g_free (private->appname);
			private->appname = g_value_dup_string (value);
			break;
		case CXP_PROFILE_SECTION:
			g_free (private->section);
			private->section = g_value_dup_string (value);
			break;
		default:
			/* We don't have any other property... */
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
			break;
	}
}

static void cxp_profile_get_property (GObject *object, guint property_id, GValue *value, GParamSpec *pspec)
{
	CxpProfilePrivate *private = CXP_PROFILE_GET_PRIVATE(object);

	switch (property_id) {
		case CXP_PROFILE_APPNAME:
			g_value_set_string (value, private->appname);
			break;
		case CXP_PROFILE_SECTION:
			g_value_set_string (value, private->section);
			break;
		default:
			/* We don't have any other property... */
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
			break;
	}
}

CxpProfile *cxp_profile_new (const gchar * appname, const gchar * section)
{
	CxpProfile *profile;
	gchar *cxpdir;
	gchar *appdir;
	CxpProfilePrivate *priv;

	profile = CXP_PROFILE (g_object_new (CXP_TYPE_PROFILE, "appname", appname, "section", section, NULL));
	cxpdir = g_build_filename (g_get_home_dir (), ".cxp", NULL);
	if (!g_file_test (cxpdir, G_FILE_TEST_EXISTS))
	{
		if (mkdir (cxpdir, S_IFDIR | 0777) != 0)
		{
			g_error ("%s couldn't be created.\n", cxpdir);
		}
	}
	appdir = g_build_filename (cxpdir, appname, NULL);
	if (!g_file_test (appdir, G_FILE_TEST_EXISTS))
	{
		if (mkdir (appdir, S_IFDIR | 0777) != 0)
		{
			g_error ("%s couldn't be created.\n", appdir);
		}
	}
	cxp_profile_load_file (profile);

	return profile;
}

gint cxp_profile_get_integer (CxpProfile * profile, const gchar * name)
{
	gint retval;
	gchar *value;
	CxpProfilePrivate *priv = CXP_PROFILE_GET_PRIVATE(profile);

	value = (gchar *) g_hash_table_lookup (priv->config, name);
	if (value == NULL)
	{
		retval = 0;
		g_hash_table_replace (priv->config, g_strdup (name),
				      g_strdup ("0"));
	}
	else
	{
		retval = atoi (value);
	}

	return retval;
}

gchar *cxp_profile_get_string (CxpProfile * profile, const gchar * name)
{
	gchar *value;
	CxpProfilePrivate *priv = CXP_PROFILE_GET_PRIVATE(profile);

	value = (gchar *) g_hash_table_lookup (priv->config, name);

	return g_strdup (value);
}

void cxp_profile_set_integer (CxpProfile * profile, const gchar * key,
			      gint value)
{
	CxpProfilePrivate *priv = CXP_PROFILE_GET_PRIVATE(profile);

	g_hash_table_replace (priv->config, g_strdup (key),
			      g_strdup_printf ("%d", value));
}

void cxp_profile_set_string (CxpProfile * profile, const gchar * key,
			     const gchar * value)
{
	CxpProfilePrivate *priv = CXP_PROFILE_GET_PRIVATE(profile);

	g_hash_table_replace (priv->config, g_strdup (key),
			      g_strdup (value));
}


static void cxp_profile_load_file (CxpProfile * profile)
{
	gchar *filename;
	CxpProfilePrivate *priv = CXP_PROFILE_GET_PRIVATE(profile);
	GHashTable *config = priv->config;
	gchar *line;
	gchar **array;
	gchar *name;
	gchar *value;
	GIOChannel *channel;

	filename = g_build_filename (g_get_home_dir (),
				     ".cxp",
				     priv->appname,
				     priv->section, NULL);
	if ((channel = g_io_channel_new_file (filename, "r", NULL)) == NULL)
	{
		g_free (filename);
		filename =
			g_build_filename (PACKAGE_CONF_DIR,
					  priv->appname,
					  priv->section, NULL);
		channel = g_io_channel_new_file (filename, "r", NULL);
	}

	if (channel != NULL)
	{
		while (g_io_channel_read_line
		       (channel, &line, NULL, NULL, NULL) == G_IO_STATUS_NORMAL)
		{
			g_strstrip (line);
			if (strlen (line) > 0)
			{
				array = g_strsplit (line, "=", 2);
				name = g_strdup (array[0]);
				value = g_strdup (array[1]);
				g_hash_table_replace (config, name, value);
			}
			g_free (line);
		}
		g_io_channel_unref (channel);
	}
	g_free (filename);
}

static void cxp_profile_save_file (gpointer key, gpointer value,
				   gpointer channel)
{
	GIOChannel *chan = (GIOChannel *) channel;
	gchar *line;

	line = g_strdup_printf ("%s=%s\n", (gchar *) key, (gchar *) value);
	g_io_channel_write_chars (chan, line, -1, NULL, NULL);
	g_free (line);
}
