/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * http://aipostyle.com/
 * 
 * Copyright(C) 2010 avanza Co.,Ltd. All rights reserved.
 * http://www.avnz.co.jp/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.aimluck.eip.extcabinet;

import java.io.File;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.cayenne.DataObjectUtils;
import org.apache.cayenne.DataRow;
import org.apache.cayenne.access.DataContext;
import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.cayenne.query.SelectQuery;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALNumberField;
import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.extcabinet.ExtCabinetFileFormData;
import com.aimluck.eip.extcabinet.ExtFolderInfo;
import com.aimluck.eip.extcabinet.util.ExtCabinetUtils;
import com.aimluck.eip.cayenne.om.portlet.AvzTFileAcl;
import com.aimluck.eip.cayenne.om.portlet.AvzTCabinetFile;
import com.aimluck.eip.cayenne.om.portlet.AvzTCabinetFolder;
import com.aimluck.eip.cayenne.om.portlet.auto._AvzTFileAcl;
import com.aimluck.eip.cayenne.om.portlet.auto._AvzTCabinetFile;
import com.aimluck.eip.cayenne.om.portlet.auto._AvzTCabinetFolder;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALEipManager;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.fileupload.beans.FileuploadLiteBean;
import com.aimluck.eip.fileupload.util.FileuploadUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.DatabaseOrmService;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.services.eventlog.ALEventlogConstants;
import com.aimluck.eip.services.eventlog.ALEventlogFactoryService;
import com.aimluck.eip.util.ALEipUtils;

/**
 * 文書管理のファイルフォームデータを管理するクラス <BR>
 *
 */
public class ExtCabinetFileFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
      .getLogger(ExtCabinetFileFormData.class.getName());

  /** フォルダ ID */
  private ALNumberField folder_id;
  
  /** 元の親フォルダ ID */
  private ALNumberField org_parent_id;

  /** ファイルタイトル */
  private ALStringField file_title;

  /** ファイル名 */
  private ALStringField file_name;

  /** ファイルサイズ */
  private ALNumberField file_size;

  /** メモ */
  private ALStringField note;

  /** フォルダ情報一覧 */
  private List<ExtFolderInfo> folder_hierarchy_list;

  /** 選択されたフォルダ情報 */
  private ExtFolderInfo selected_folderinfo = null;

  /** 添付フォルダ名 */
  private String folderName = null;

  private String fileid = null;

  private String org_id = null;

  private DataContext dataContext;

  /** 添付ファイルリスト */
  private List<FileuploadLiteBean> fileuploadList = null;

  /** Validate用(添付ファイルID) */
  private String[] fileids;

  private RunData rundata;
  
  /** アクセス権限リスト by motegi 　*/
  private List<Map<String, String>> aclRequestList;
  
  /** アクセス権限リスト（更新時表示用） by motegi */
  private List<ExtCabinetAclResultData> aclList;
  
  private boolean is_owner;
  
  /** ログインユーザーID */
  private long login_user_id;
  

  /**
   *
   * @param action
   * @param rundata
   * @param context
   * @see com.aimluck.eip.common.ALAbstractFormData#init(com.aimluck.eip.modules.actions.common.ALAction,
   *      org.apache.turbine.util.RunData, org.apache.velocity.context.Context)
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);

    // 自ポートレットからのリクエストであれば、パラメータを展開しセッションに保存する。
    if (ALEipUtils.isMatch(rundata, context)) {
      // ENTITY ID
      if (rundata.getParameters().containsKey(ExtCabinetUtils.REQUEST_KEY_FOLDER_ID)) {
        ALEipUtils.setTemp(rundata, context, ExtCabinetUtils.SESSION_KEY_FOLDER_ID,
            rundata.getParameters().getString(ExtCabinetUtils.REQUEST_KEY_FOLDER_ID));
      }
    }

    org_id = DatabaseOrmService.getInstance().getOrgId(rundata);
    dataContext = DatabaseOrmService.getInstance().getDataContext();
    this.rundata = rundata;

    String tmpfid = ALEipUtils.getTemp(rundata, context,
        ExtCabinetUtils.SESSION_KEY_FOLDER_ID);
    int fid = ExtCabinetUtils.ROOT_FODLER_ID;
    if (tmpfid != null && !"".equals(tmpfid)) {
      try {
        fid = Integer.parseInt(tmpfid);
      } catch (Exception e) {
        fid = ExtCabinetUtils.ROOT_FODLER_ID;
      }
    }

    folder_hierarchy_list = ExtCabinetUtils.getFolderList(ExtCabinetUtils.ROOT_PARENT_FODLER_ID);
    ExtCabinetUtils.setFolderVisibleForForm(folder_hierarchy_list, rundata);

    int size = folder_hierarchy_list.size();
    for (int i = 0; i < size; i++) {
      ExtFolderInfo info = (ExtFolderInfo) folder_hierarchy_list.get(i);
      if (info.getFolderId() == fid) {
        selected_folderinfo = info;
      }
    }

    folderName = rundata.getParameters().getString("folderName");

    fileids = rundata.getParameters().getStrings("attachments");
    
    login_user_id = ALEipUtils.getALEipUser(rundata).getUserId().getValue();
  }

  /**
   * 各フィールドを初期化します。 <BR>
   *
   * @see com.aimluck.eip.common.ALData#initField()
   */
  public void initField() {
    // フォルダ ID
    folder_id = new ALNumberField();
    folder_id.setFieldName("フォルダ");
    folder_id.setNotNull(true);
    folder_id.setValue(0);
    // ファイルタイトル
    file_title = new ALStringField();
    file_title.setFieldName("タイトル");
    file_title.setTrim(true);
    // フォルダ名
    file_name = new ALStringField();
    file_name.setFieldName("ファイル名");
    file_name.setTrim(true);
    // ファイルサイズ
    file_size = new ALNumberField();
    file_size.setFieldName("ファイルサイズ");
    // メモ
    note = new ALStringField();
    note.setFieldName("メモ");
    note.setTrim(true);
    // 編集前の元の親フォルダ
    org_parent_id = new ALNumberField();
    org_parent_id.setFieldName("編集前の元の親フォルダ");
    org_parent_id.setValue(0);

    fileuploadList = new ArrayList<FileuploadLiteBean>();
  }

  /*
   * @see com.aimluck.eip.common.ALAbstractFormData#setFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context,
      ArrayList msgList) throws ALPageNotFoundException, ALDBErrorException {

    boolean res = super.setFormData(rundata, context, msgList);

    if (res) {
      if (!rundata.getParameters().containsKey(ExtCabinetUtils.REQUEST_KEY_FOLDER_ID)) {
        try {
          folder_id.setValue(selected_folderinfo.getFolderId());
        } catch (Exception e) {
          folder_id.setValue(ExtCabinetUtils.ROOT_FODLER_ID);
        }
      }

      if (ALEipConstants.MODE_UPDATE.equals(getMode())) {
        fileid = ALEipUtils.getTemp(rundata, context, ALEipConstants.ENTITY_ID);
      }
      
      if (ALEipConstants.MODE_NEW_FORM.equals(getMode()) ) {
      	aclList = ExtCabinetUtils.loadFolderAclList(selected_folderinfo.getFolderId());
      	
      	//親フォルダから引き継いだアクセス権限にログインユーザーのレコードが
      	//あれば除外します。これはルートフォルダ直下にフォルダやファイルを作成
      	//する時が対象の処理です。
      	aclList = ExtCabinetUtils.excludeLoginUser(aclList, login_user_id);
      }
      
      //motegi start アクセス権限情報をリクエストから取得します。
      if (ALEipConstants.MODE_INSERT.equals(getMode())
          || ALEipConstants.MODE_UPDATE.equals(getMode())) {
        aclRequestList = new ArrayList<Map<String, String>>();
        String[] target_ids = rundata.getParameters().getStrings("acl[target_id][]");
        String[] target_types = rundata.getParameters().getStrings("acl[target_type][]");
        String[] r = rundata.getParameters().getStrings("acl[readable][]");
        String[] w = rundata.getParameters().getStrings("acl[writable][]");
        String[] d = rundata.getParameters().getStrings("acl[deletable][]");
        if( target_ids!=null && target_ids.length>0 ) {
          int acl_num = target_ids.length;
          for( int i=0; i<acl_num; i++ ) {
            Map<String, String> map = new HashMap<String, String>();
            map.put(ExtCabinetUtils.AUTHORITY_TARGET_ID, target_ids[i]);
            map.put(ExtCabinetUtils.AUTHORITY_TARGET_TYPE, target_types[i]);
            map.put(ExtCabinetUtils.AUTHORITY_R, r[i]);
            map.put(ExtCabinetUtils.AUTHORITY_U, w[i]);
            map.put(ExtCabinetUtils.AUTHORITY_D, d[i]);
            aclRequestList.add(map);
          }
          System.out.println(aclRequestList.toString());
        }
      }
      //motegi end
    }

    try {
      fileuploadList = FileuploadUtils.getFileuploadList(rundata);
      FileuploadLiteBean filebean = null;
      if (fileuploadList != null) {
        // 新規にアップロードしたデータをつめる
        filebean = (FileuploadLiteBean) fileuploadList.get(0);
        file_name.setValue(filebean.getFileName());
      }
    } catch (Exception ex) {
      logger.error("Exception", ex);
    }

    return res;
  }

  /**
   * ファイルの各フィールドに対する制約条件を設定します。 <BR>
   *
   * @see com.aimluck.eip.common.ALAbstractFormData#setValidator()
   */
  @Override
  protected void setValidator() {
    // ファイルタイトル必須項目
    file_title.setNotNull(true);
    // ファイルタイトルの文字数制限
    file_title.limitMaxLength(128);
    // ファイル名必須項目
    file_name.setNotNull(true);
    // ファイル名の文字数制限
    file_name.limitMaxLength(128);
    // メモの文字数制限
    note.limitMaxLength(1000);
  }

  /**
   * ファイルのフォームに入力されたデータの妥当性検証を行います。 <BR>
   *
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#validate(java.util.ArrayList)
   */
  @Override
  protected boolean validate(ArrayList msgList) {
    // ファイルタイトル
    file_title.validate(msgList);
    // ファイル名
    if (ALEipConstants.MODE_INSERT.equals(getMode())) {
      file_name.validate(msgList);
    } else {
      if (fileuploadList != null) {
        if (fileuploadList.size() > 0) {
          file_name.validate(msgList);
        }
      }
    }

    if (fileids != null) {
      if (fileids[0].equals("")) {
        msgList.add("『 <span class='em'>ファイル名</span> 』を入力してください。");
      }
    }

    // メモ
    note.validate(msgList);
    if (existsFileName()) {
      msgList.add("このファイル名と同じファイルがすでに存在するため、登録できません。ファイル名を変更してください。");
    }

    /** 編集アクセス制限 */
    if (ALEipConstants.MODE_INSERT.equals(getMode()) ) {
      if (!ExtCabinetUtils.isFileCreatableFolder((int) folder_id.getValue(), rundata))
        msgList.add("フォルダにファイルの追加権限がありません。");
    }
    
    if (ALEipConstants.MODE_UPDATE.equals(getMode())) {
      //フォルダ追加の場合には、親フォルダに編集権限があるかチェック
      if (folder_id.getValue() != org_parent_id.getValue()) {
        //更新時に元親フォルダと新親フォルダが異なるということは、フォルダの移動が
        //指定されたということ。
        //移動する権限（元親フォルダに削除権限、新親フォルダに追加権限）があるかチェック
        if( !ExtCabinetUtils.isFileCreatableFolder((int)folder_id.getValue(), rundata) ) {
          msgList.add("移動先のフォルダにファイルの追加権限がありません。");
        }
      }
    }
    
    return (msgList.size() == 0);
  }

  private boolean existsFileName() {
    String fname = file_name.getValue();
    if (fname == null || "".equals(fname))
      return false;

    try {
      SelectQuery query = new SelectQuery(AvzTCabinetFile.class);
      query.addCustomDbAttribute(AvzTCabinetFile.FILE_NAME_COLUMN);
      if (ALEipConstants.MODE_INSERT.equals(getMode())) {
        Expression exp = ExpressionFactory.matchDbExp(
            _AvzTCabinetFolder.FOLDER_ID_PK_COLUMN, Integer
                .valueOf((int) folder_id.getValue()));
        query.setQualifier(exp);
      } else if (ALEipConstants.MODE_UPDATE.equals(getMode())) {
        Expression exp1 = ExpressionFactory.matchDbExp(
            _AvzTCabinetFolder.FOLDER_ID_PK_COLUMN, Integer
                .valueOf((int) folder_id.getValue()));
        query.setQualifier(exp1);
        Expression exp2 = ExpressionFactory.noMatchDbExp(
            _AvzTCabinetFile.FILE_ID_PK_COLUMN, Integer.valueOf(fileid));
        query.andQualifier(exp2);
      }

      List list = dataContext.performQuery(query);
      if (list != null && list.size() > 0) {
        DataRow dataRow = null;
        int size = list.size();
        for (int i = 0; i < size; i++) {
          dataRow = (DataRow) list.get(i);
          if (fname.equals(ALEipUtils.getObjFromDataRow(dataRow,
              AvzTCabinetFile.FILE_NAME_COLUMN))) {
            return true;
          }
        }
      }
    } catch (Exception e) {
      return true;
    }

    return false;
  }

  /**
   * ファイルをデータベースから読み出します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#loadFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      ArrayList msgList) {
    try {
      // オブジェクトモデルを取得
      AvzTCabinetFile file = ExtCabinetUtils.getAvzTCabinetFile(rundata, context);
      if (file == null)
        return false;

      // 親フォルダ
      folder_id.setValue(file.getFolderId().intValue());
      // ファイルタイトル
      file_title.setValue(file.getFileTitle());
      // ファイル名
      file_name.setValue(file.getFileName());
      // ファイルサイズ
      file_size.setValue(file.getFileSize().longValue());
      // メモ
      note.setValue(file.getNote());
      // フォルダ名
      folderName = Integer.toString(file.getFolderId());

      aclList = ExtCabinetUtils.loadFileAclList(file.getFileId());
      
      is_owner = (file.getCreateUserId().intValue()==(int)login_user_id);

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * ファイルをデータベースとファイルシステムから削除します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#deleteFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      ArrayList msgList) {
    try {
      // オブジェクトモデルを取得
      AvzTCabinetFile file = ExtCabinetUtils.getAvzTCabinetFile(rundata, context);
      if (file == null)
        return false;

      // ファイルのパス
      String local_file_pass = file.getFilePath();

      // entityIdの取得
      int entityId = file.getFileId();
      // todo名の取得
      String fileName = file.getFileTitle();

      // ファイルを削除
      dataContext.deleteObject(file);
      dataContext.commitChanges();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(entityId,
          ALEventlogConstants.PORTLET_TYPE_CABINET_FILE, fileName);

      if (local_file_pass != null && !"".equals(local_file_pass)) {
        // ローカルファイルに保存されているファイルを削除する．
        File local_file = null;
        local_file = new File(ExtCabinetUtils.getSaveDirPath(org_id)
            + local_file_pass);
        if (local_file.exists()) {
          local_file.delete();
        }
      }

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * ファイルをデータベースとファイルシステムに格納します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#insertFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      ArrayList msgList) {
    try {
      int uid = ALEipUtils.getUserId(rundata);
      FileuploadLiteBean filebean = (FileuploadLiteBean) fileuploadList.get(0);
      
      // ファイルの移動先
      String filename = FileuploadUtils.getNewFileName(ExtCabinetUtils
          .getSaveDirPath(org_id));

      // ファイルの移動先
      File destFile = new File(ExtCabinetUtils.getAbsolutePath(org_id, filename));

      // 一時保存の添付ファイル（ブラウザからアップロードされたファイルは自動的に一時保存されている）
      File srcFile = FileuploadUtils.getAbsolutePath(org_id, uid, folderName,
          filebean.getFileId());

      // ファイルの移動（一時保存 to C:\aipo\dpl003\tomcat\webapps\aipo\WEB-INF\files\org001\extcabinet）
      FileuploadUtils.copyFile(srcFile, destFile);

      double fileSize = Math.ceil(destFile.length() / 1024.0);

      AvzTCabinetFolder folder = (AvzTCabinetFolder) DataObjectUtils
          .objectForPK(dataContext, AvzTCabinetFolder.class, Integer
              .valueOf((int) folder_id.getValue()));

      // 新規オブジェクトモデル
      AvzTCabinetFile file = (AvzTCabinetFile) dataContext
          .createAndRegisterNewObject(AvzTCabinetFile.class);
      
      // 親フォルダ
      file.setToAvzTCabinetFolder(folder);
      // ファイルタイトル
      file.setFileTitle(file_title.getValue());
      // ファイル名
      file.setFileName(file_name.getValue());
      // ファイルサイズ
      file.setFileSize(Long.valueOf((long) fileSize));
      // ファイルパス
      file.setFilePath(ExtCabinetUtils.getRelativePath(filename));
      // ファイル
      // file.setFile(fileData);
      // メモ
      file.setNote(note.getValue());
      // 作成者
      file.setCreateUserId(Integer.valueOf(uid));
      // 更新者
      file.setUpdateUserId(Integer.valueOf(uid));
      // 作成日
      file.setCreateDate(Calendar.getInstance().getTime());
      // 更新日
      file.setUpdateDate(Calendar.getInstance().getTime());
      //アクセス権限の設定
      ExtCabinetUtils.insertFileAcl(file, aclRequestList);
      // フォルダを登録
      dataContext.commitChanges();

      File folder_tmp = FileuploadUtils.getFolder(org_id, uid, folderName);
      // 添付ファイル保存先のフォルダを削除
      FileuploadUtils.deleteFolder(folder_tmp);

    } catch (Exception ex) {
      logger.error("Exception", ex);
      dataContext.rollbackChanges();
      return false;
    }
    return true;
  }

  /**
   * データベースとファイルシステムに格納されているファイルを更新します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#updateFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      ArrayList msgList) {
    try {
      int uid = ALEipUtils.getUserId(rundata);

      // オブジェクトモデルを取得
      AvzTCabinetFile file = ExtCabinetUtils.getAvzTCabinetFile(rundata, context);
      if (file == null)
        return false;

      // アップロードしたファイルは存在するか
      boolean is_upload = true;
      if (fileuploadList == null) {
        is_upload = false;
      }
      if (is_upload) {
        // アップロードが確認できた場合
        FileuploadLiteBean filebean = (FileuploadLiteBean) fileuploadList
            .get(0);
        String filename = FileuploadUtils.getNewFileName(ExtCabinetUtils
            .getSaveDirPath(org_id));
        // ファイルの移動先
        File destFile = new File(ExtCabinetUtils.getAbsolutePath(org_id, filename));

        // 一時保存の添付ファイル
        File srcFile = FileuploadUtils.getAbsolutePath(org_id, uid, folderName,
            filebean.getFileId());

        // ファイルの移動
        FileuploadUtils.copyFile(srcFile, destFile);

        // ファイル名
        file.setFileName(file_name.getValue());

        // ファイルサイズ
        double fileSize = Math.ceil(destFile.length() / 1024.0);
        file.setFileSize(Long.valueOf((long) fileSize));

        // ファイルパス
        file.setFilePath(ExtCabinetUtils.getRelativePath(filename));
        // 作成者
        file.setCreateUserId(Integer.valueOf(uid));
        // 作成日
        file.setCreateDate(Calendar.getInstance().getTime());
      }

      // 親フォルダ
      AvzTCabinetFolder folder = (AvzTCabinetFolder) DataObjectUtils
          .objectForPK(dataContext, AvzTCabinetFolder.class, Integer
              .valueOf((int) folder_id.getValue()));
      file.setToAvzTCabinetFolder(folder);

      // ファイルタイトル
      file.setFileTitle(file_title.getValue());
      // メモ
      file.setNote(note.getValue());
      // 更新者
      file.setUpdateUserId(Integer.valueOf(uid));
      // 更新日
      file.setUpdateDate(Calendar.getInstance().getTime());

      // オーナーの場合はアクセス制御 入れ替え
      is_owner = (file.getCreateUserId().intValue()==(int)login_user_id);
//      if( is_owner ) {
        SelectQuery query = new SelectQuery(AvzTFileAcl.class);
        Expression exp = ExpressionFactory.matchExp(
            _AvzTFileAcl.FILE_ID_PROPERTY, file.getFileId());
        query.setQualifier(exp);
        List acls = dataContext.performQuery(query);
        dataContext.deleteObjects(acls);
        
        ExtCabinetUtils.insertFileAcl(file, aclRequestList);
//      }
      
      // フォルダを更新
      dataContext.commitChanges();

    } catch (Exception ex) {
      logger.error("Exception", ex);
      dataContext.rollbackChanges();
      return false;
    }
    return true;
  }

  public String getFolderName() {
    return folderName.toString();
  }

  /**
   *
   *
   * @return
   */
  public ALNumberField getFolderId() {
    return folder_id;
  }

  /**
   * ファイルタイトルを取得する． <BR>
   *
   * @return
   */
  public ALStringField getFileTitle() {
    return file_title;
  }

  /**
   * ファイル名を取得する． <BR>
   *
   * @return
   */
  public ALStringField getFileName() {
    return file_name;
  }

  /**
   * ファイルサイズを取得する． <BR>
   *
   * @return
   */
  public ALNumberField getFileSize() {
    return file_size;
  }

  /**
   * メモを取得する． <BR>
   *
   * @return
   */
  public ALStringField getNote() {
    return note;
  }

  public List<ExtFolderInfo> getFolderHierarchyList() {
    return folder_hierarchy_list;
  }

  public ExtFolderInfo getSelectedFolderInfo() {
    return selected_folderinfo;
  }

  /**
   *
   * @param id
   * @return
   */
  public boolean isMatch(int id1, long id2) {
    return id1 == (int) id2;
  }

  public List<FileuploadLiteBean> getAttachmentFileNameList() {
    return fileuploadList;
  }
  
  public List<ExtCabinetAclResultData> getAclList() {
    return aclList;
  }

  /**
   * アクセス権限チェック用メソッド。<br />
   * アクセス権限の機能名を返します。
   *
   * @return
   */
  @Override
  public String getAclPortletFeature() {
    return ALAccessControlConstants.POERTLET_FEATURE_CABINET_FILE;
  }

  public Map getPostMap() {
    return ALEipManager.getInstance().getPostMap();
  }
  
  public boolean isOwner() {
    return is_owner;
  }
}
