/*
 * $Header: /home/cvspublic/jakarta-jetspeed/scratchpad/jetspeed-cms/src/java/org/apache/jetspeed/services/cms/repository/slide/CmsContentStore.java,v 1.1 2003/03/04 11:28:16 raphael Exp $
 * $Revision: 1.1 $
 * $Date: 2003/03/04 11:28:16 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.jetspeed.services.cms.repository.slide;

import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Date;
import java.io.*;
import org.apache.slide.common.*;
import org.apache.slide.store.*;
import org.apache.slide.structure.*;
import org.apache.slide.content.*;
import org.apache.slide.util.logger.Logger;

import org.apache.jetspeed.services.resources.JetspeedResources;

/**
 * Filesystem implementation of ContentStore .
 *
 * @author <a href="mailto:remm@apache.org">Remy Maucherat</a>
 * @author <a href="mailto:christophe.lombart@skynet.be">Christophe Lombart</a>

 */
public class CmsContentStore extends AbstractService
    implements ContentStore {
    
    
    // -------------------------------------------------------------- Constants
    
    
    public static final int BUFFER_SIZE = 2048;
    public static final String CHARACTER_ENCODING = "8859_1";
    
    
    // ----------------------------------------------------- Instance Variables
    
    
    /**
     * Path from which files are stored.
     */
    private String rootpath;
    
    
    /**
     * True if versioning.
     */
    private boolean version = true;
    
    /**
     * True if content written from a prior server run should be removed.
     * WARNING: setting this option to true will remove all files, located
     * under the RootPath location!
     */
    private boolean removePriorContent = false;
    
    /**
     * Default rootpath is the current directory
     */
	private final String DEFAULT_ROOTPATH=".";
    
    // ---------------------------------------------------- ServiceImpl Methods
    
    
    /**
     * Read parameters.
     *
     * @param parameters Hashtable containing the parameters' name
     * and associated value
     */
    public synchronized void setParameters(Hashtable parameters)
        throws ServiceParameterErrorException,
        ServiceParameterMissingException {
        // A parameter will tell were serialization files are.
        rootpath = JetspeedResources.getString("services.CmsService.slide.content.rootpath");
        if (rootpath == null) {
            // Default is that files are stored starting in the
            // current directory
            rootpath = DEFAULT_ROOTPATH;
        }
        String versionValue = (String) parameters.get("services.CmsService.slide.content.version");
        if (versionValue != null) {
            version = Boolean.valueOf(versionValue).booleanValue();
        }
        String removePriorContentValue = (String) parameters.get("services.CmsService.slide.content.resetBeforeStarting");
        if (removePriorContentValue != null) {
            removePriorContent = removePriorContentValue.equalsIgnoreCase("true");
        }
            
    }
    
    
    /**
     * Connects to store.
     *
     * @exception VersionException
     */
    public synchronized void connect()
        throws ServiceConnectionFailedException {
    }
    
    
    /**
     * Disconnects from content store.
     *
     * @exception ServiceDisconnectionFailedException
     */
    public synchronized void disconnect()
        throws ServiceDisconnectionFailedException {
    }
    
    
    /**
     * Initializes content store.
     *
     * @exception ServiceInitializationFailedException Throws an exception
     * if the store has already been initialized before
     */
    public synchronized void initialize(NamespaceAccessToken token)
        throws ServiceInitializationFailedException {
        try {
            // remove prior content specified at rootPath
            if ((removePriorContent) && (!DEFAULT_ROOTPATH.equals(rootpath))) 
                reset();
            File baseDir = new File(rootpath);
			token.getLogger().log("FileContentStore rootpath: " + baseDir.getAbsolutePath() ,Logger.INFO);
            baseDir.mkdirs();
        } catch (Exception e) {
            e.printStackTrace();
            throw new ServiceInitializationFailedException(this, e);
        }
    }
    
    
    /**
     * Deletes content store.
     *
     * @exception ServiceResetFailedException
     */
    public void reset()
        throws ServiceResetFailedException {
        String[] filter = new String[1];
        filter[0] = "*.*";
        new slidestore.reference.fileDeleter.DeleterWalker(rootpath, filter );
    }
    
    
    /**
     * This function tells whether or not the service is connected.
     *
     * @return boolean true if we are connected
     * @exception ServiceAccessException Service access error
     */
    public boolean isConnected()
        throws ServiceAccessException {
        return true;
    }
    
    
    /**
     * Does nothing here.
     *
     * @exception ServiceAccessException Unspecified service access error
     */
    public synchronized void commit()
        throws ServiceAccessException {
    }
    
    
    // --------------------------------------------------- ContentStore Methods
    
    
    /**
     * Retrive revision content.
     *
     * @param uri Uri
     * @param revisionNumber Node revision number
     */
    public NodeRevisionContent retrieveRevisionContent
        (Uri uri, NodeRevisionDescriptor revisionDescriptor)
        throws ServiceAccessException, RevisionNotFoundException {
        NodeRevisionContent result = null;
        String revisionUri = null;
        if (version)
            revisionUri = uri.toString() + "_"
                + revisionDescriptor.getRevisionNumber();
        else
            revisionUri = uri.toString();
        try {
            File file = new File(rootpath + revisionUri);
            FileInputStream is = new FileInputStream(file);
            InputStreamReader reader = new InputStreamReader
                (is, CHARACTER_ENCODING);
            result = new NodeRevisionContent();
            result.setContent(reader);
            result.setContent(is);
        } catch (FileNotFoundException e) {
            throw new RevisionNotFoundException
                (uri.toString(),
                 revisionDescriptor.getRevisionNumber());
        } catch (IOException e) {
            e.printStackTrace();
            throw new ServiceAccessException(this, e.getMessage());
        }
        return result;

    }
    
    
    /**
     * Create a new revision
     *
     * @param uri Uri
     * @param revisionDescriptor Node revision descriptor
     * @param revisionContent Node revision content
     */
    public void createRevisionContent
        (Uri uri, NodeRevisionDescriptor revisionDescriptor,
         NodeRevisionContent revisionContent)
        throws ServiceAccessException, RevisionAlreadyExistException {
        String revisionUri = null;
        if (version)
            revisionUri = uri.toString() + "_"
                + revisionDescriptor.getRevisionNumber();
        else
            revisionUri = uri.toString();
        
        try {
            File file = new File(rootpath + revisionUri);
            File parentFile = new File(file.getParent());
            if ((parentFile != null) && (!parentFile.exists())) {
                parentFile.mkdirs();
            }
            
            boolean created = !file.exists();
            if (!created) {
                throw new RevisionAlreadyExistException
                    (uri.toString(), revisionDescriptor.getRevisionNumber());
            }
            
            InputStream is = revisionContent.streamContent();
            
            if (is != null) {
                OutputStream os = new FileOutputStream(file);
                // We copy 8 ko with each read
                byte[] buffer = new byte[BUFFER_SIZE];
                long position = 0;
                long contentLength = revisionDescriptor.getContentLength();
                
                while (true) {
                    int nChar = is.read(buffer);
                    if (nChar == -1) {
                        break;
                    }
                    os.write(buffer, 0, nChar);
                    position = position + nChar;
                }
                os.close();
                is.close();
                
                if (contentLength != -1) {
                    if (position != contentLength) {
			// set content length so that repository is consistent
			revisionDescriptor.setContentLength(position);
                        if (position < contentLength) {
                            // Not enough bytes read !!!
                            throw new IOException("Not enough bytes read");
                        }
                        if (position > contentLength) {
                            // Not enough bytes read !!!
                            throw new IOException("Too many bytes read");
                        }
                        // FIXME : Delete the file
                    }
                } else {
                    revisionDescriptor.setContentLength(position);
                }
                
            } else {
            }
            
        } catch (IOException e) {
            throw new ServiceAccessException(this, e.getMessage());
        } catch(RevisionAlreadyExistException e) {
            throw e; // we do NOT want this caught by next clause.
        } catch (Exception e) {
            e.printStackTrace();
            throw new ServiceAccessException(this, e.getMessage());
        }
    }
    
    
    /**
     * Modify the latest revision of an object.
     *
     * @param uri Uri
     * @param revisionDescriptor Node revision descriptor
     * @param revisionContent Node revision content
     */
    public void storeRevisionContent
        (Uri uri, NodeRevisionDescriptor revisionDescriptor,
         NodeRevisionContent revisionContent)
        throws ServiceAccessException, RevisionNotFoundException {
        String revisionUri = null;
        if (version)
            revisionUri = uri.toString() + "_"
                + revisionDescriptor.getRevisionNumber();
        else
            revisionUri = uri.toString();
        try {
            File file = new File(rootpath + revisionUri);
            
            InputStream is = revisionContent.streamContent();
            
            if (is != null) {
                OutputStream os = null;
                try {
                    os = new FileOutputStream(file);
                } catch (FileNotFoundException ex) {
                    // Try to create the parent directory and try again
                    File parentFile = new File(file.getParent());
                    if ((parentFile != null) && (!parentFile.exists())) {
                        parentFile.mkdirs();
                    }
                    os = new FileOutputStream(file);
                }
                // We copy 8 ko with each read
                byte[] buffer = new byte[BUFFER_SIZE];
                long position = 0;
                long contentLength = revisionDescriptor.getContentLength();
                
                while (true) {
                    int nChar = is.read(buffer);
                    if (nChar == -1) {
                        break;
                    }
                    os.write(buffer, 0, nChar);
                    position = position + nChar;
                }
                os.close();
                is.close();
                
                if (contentLength != -1) {
                    if (position != contentLength) {
                        if (position < contentLength) {
                            // Not enough bytes read !!!
                            throw new IOException("Not enough bytes read");
                        }
                        if (position > contentLength) {
                            // Not enough bytes read !!!
                            throw new IOException("Too many bytes read");
                        }
                        // FIXME : Delete the file
                    }
                } else {
                    revisionDescriptor.setContentLength(position);
                }
                
            } else {
            }
            
        } catch (FileNotFoundException e) {
            throw new RevisionNotFoundException
                (uri.toString(), revisionDescriptor.getRevisionNumber());
        } catch (IOException e) {
            e.printStackTrace();
            throw new ServiceAccessException(this, e.getMessage());
        }
    }
    
    
    /**
     * Remove revision.
     *
     * @param uri Uri
     * @param revisionNumber Node revision number
     */
    public void removeRevisionContent(Uri uri,
                                      NodeRevisionDescriptor revisionDescriptor)
        throws ServiceAccessException {
        String revisionUri = null;
        if (version)
            revisionUri = uri.toString() + "_" + revisionDescriptor.getRevisionNumber();
        else
            revisionUri = uri.toString();
        
        try {
            File file = new File(rootpath + revisionUri);
            boolean deleted = file.delete();
            File parentFile = new File(file.getParent());
            if (parentFile != null) {
                parentFile.delete();
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new ServiceAccessException(this, e.getMessage());
        }
    }
    
}
