/*
 * Copyright(C) 2009 avanza Co.,Ltd. All rights reserved.
 * http://www.avnz.co.jp/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.aimluck.eip.searchmap;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;

import org.apache.cayenne.access.DataContext;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.AvzTSearchMap;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.DatabaseOrmService;
import com.aimluck.eip.searchmap.util.SearchMapUtils;
import com.aimluck.eip.services.eventlog.ALEventlogConstants;
import com.aimluck.eip.services.eventlog.ALEventlogFactoryService;
import com.aimluck.eip.util.ALEipUtils;

/**
 * SearchMapのフォームデータを管理するクラスです。 <BR>
 * 
 */
public class SearchMapFormData extends ALAbstractFormData {
  
  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
      .getLogger(SearchMapFormData.class.getName());

  /** 検索キーワード(住所) */
  private ALStringField keyword;

  private DataContext dataContext;

  /**
   * init
   * 
   * @param action
   * @param rundata
   * @param context
   * @see com.aimluck.eip.common.ALAbstractFormData#init(com.aimluck.eip.modules.actions.common.ALAction,
   *      org.apache.turbine.util.RunData, org.apache.velocity.context.Context)
   */
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
  
    super.init(action, rundata, context);

    dataContext = DatabaseOrmService.getInstance().getDataContext();
  }

  /**
   * 各フィールドを初期化します。 <BR>
   * 
   * @see com.aimluck.eip.common.ALData#initField()
   */
  public void initField() {

    // 検索キーワード(住所)
    keyword = new ALStringField();
    keyword.setFieldName("住所");
    keyword.setTrim(true);
    
  }

  /**
   * SearchMapの各フィールドに対する制約条件を設定します。 <BR>
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#setValidator()
   */
  protected void setValidator() {
    // 検索キーワード(住所)の空欄チェック
    keyword.setNotNull(true);
    // 検索キーワード(住所)の文字数制限
    keyword.limitMaxLength(64);
  }

  /**
   * SearchMapのフォームに入力されたデータの妥当性検証を行います。 <BR>
   * 
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#validate(java.util.ArrayList)
   */
  protected boolean validate(ArrayList msgList) {

    try {
      // 検索キーワード(住所)
      keyword.validate(msgList);
    } catch (Exception e) {
      logger.error("Exception", e);
    }
    return (msgList.size() == 0);
  }

  /**
   * SearchMapをデータベースから読み出します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#loadFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  protected boolean loadFormData(RunData rundata, Context context,
      ArrayList msgList) {

    return true;
  }

  /**
   * SearchMapをデータベースから削除します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#deleteFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  protected boolean deleteFormData(RunData rundata, Context context,
      ArrayList msgList) {
    try {
      // オブジェクトモデルを取得
      AvzTSearchMap searchmap = SearchMapUtils.getAvzTSearchMap(rundata, context);
      if (searchmap == null)
        return false;

      // SearchMapを削除
      dataContext.deleteObject(searchmap);
      dataContext.commitChanges();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(searchmap.getId(),
          ALEventlogConstants.PORTLET_TYPE_NONE, searchmap.getKeyword());

    } catch (Exception ex) {
      dataContext.rollbackChanges();
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * SearchMapをデータベースに格納します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#insertFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  protected boolean insertFormData(RunData rundata, Context context,
      ArrayList msgList) {
    try {
      // 新規オブジェクトモデル
      AvzTSearchMap searchMap = (AvzTSearchMap) dataContext
          .createAndRegisterNewObject(AvzTSearchMap.class);
      //現在時刻を取得
      Date now = Calendar.getInstance().getTime();
      // ユーザーID
      searchMap.setUserId((Integer) ALEipUtils.getUserId(rundata));      
      // 検索キーワード(住所)
      searchMap.setKeyword(keyword.getValue());
      // 作成日
      searchMap.setCreateDate(now);
      // 更新日
      searchMap.setUpdateDate(now);
      // SearchMapを登録
      dataContext.commitChanges();
      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
          searchMap.getId(), ALEventlogConstants.PORTLET_TYPE_NONE,
          keyword.getValue());
      
    } catch (Exception ex) {
      dataContext.rollbackChanges();
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * データベースに格納されているSearchMapを更新します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#updateFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  protected boolean updateFormData(RunData rundata, Context context,
      ArrayList msgList) {

    return true;
  }
  
  /**
   * Keywordを取得します。 <BR>
   * 
   * @return
   */
  public ALStringField getKeyword(){
    return keyword;
  }

}
