/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.portlet;

import org.apache.jetspeed.portlet.event.MessageEvent;
import org.apache.jetspeed.portlet.event.MessageListener;

import java.io.InputStream;
import java.io.IOException;

import java.util.Locale;
import java.util.Enumeration;

/**
 ** The <CODE>PortletContext</CODE> interface defines a portlet's view
 ** of the portlet container within which each portlet is running.
 ** The <CODE>PortletContext</CODE> also allows a portlet to access
 ** resources available to it. Using the context, a portlet can access
 ** the portlet log, and obtain URL references to resources.
 **
 ** @author   <A HREF="mailto:tboehme@us.ibm.com">Thomas F. Boehme</A>
 **/

public interface PortletContext
{
    /**
     ** Returns the value of the context-wide initialization parameter with
     ** the given name, or <CODE>null</CODE> if the parameter does not exist.
     **
     ** <P>
     ** This method can make available configuration information useful to
     ** all portlets.
     **
     ** @param   name
     **          the parameter name
     **
     ** @return   the parameter value, <BR>
     **           or <CODE>null</CODE> if it does not exist
     **/

    public String getInitParameter (String name);

    /**
     ** Returns the names of this contexts initialization parameters as
     ** an enumeration, or an empty enumeration if this context has no
     ** initialization parameters.
     **
     ** @return   an enumeration of parameter names
     **/

    public Enumeration getInitParameterNames ();

    /**
     ** Associates an attribute with the given name and value
     ** with this context. If a portlet needs to communicate
     ** information to embedded servlets or JSP, this methods
     ** can used carry the information along.
     **
     ** @param   name
     **          the attribute name
     ** @param   value
     **          the attribute value
     **/

    public void setAttribute (String name, Object value);

    /**
     ** Returns the attribute value with the given name, or <CODE>null</CODE>
     ** if no such attribute exists.
     **
     ** <P>
     ** The context attributes can be used to share information between
     ** the portlets of one portlet application.
     **
     ** @param   name
     **          the attribute name
     **
     ** @return   the attribute value
     **/

    public Object getAttribute (String name);

    /**
     ** Returns an enumeration of the attribute names that this portlet
     ** context is aware of.
     **
     ** @return    an enumeration of attribute names
     **/

    public Enumeration getAttributeNames ();

    /**
     ** Removes the attribute with the given name.
     **
     ** @param   name
     **          the name of attribute to be removed
     **/

    public void removeAttribute (String name);

    /**
     * <P>
     * Allows the portlet to delegate the rendering to another resource
     * as specified by the given path. The path has to be relative
     * and will be resolved by this method, so that the portlet's
     * resources are accessed.
     * </P>
     * <P>
     * To access protected resources the path has to be prefixed with
     * /portlet-inf/ (e.g. /portlet-inf/myportlet/myportlet.jsp).
     * Otherwise, the direct path is used. (e.g. /myportlet/myportlet.jsp).
     * </P>
     * <P>
     * This method is enabled for multi-language and multi-device support.
     * For example, a jsp file "/myportlet/mytemplate.jsp" will be searched for
     * in the following order, when accessing via HTML-Browser:
     * 1. /html/en/US/IE/mytemplate.jsp
     * 2. /html/en/US/mytemplate.jsp
     * 3. /html/en/mytemplate.jsp
     * 4. /html/mytemplate.jsp
     * 5. /en/US/IE/mytemplate.jsp
     * 6. /en/US/mytemplate.jsp
     * 7. /en/mytemplate.jsp
     * 8. /mytemplate.jsp
     * </P>
     * 
     * @param path
     *                 the path of the delegate resource
     * @param request
     *                 the portlet request
     * @param response
     *                 the portlet response
     * @exception PortletException
     *                   if the delegated resource has trouble fulfilling the
     *                   rendering request
     * @exception IOException
     *                   if the streaming causes an I/O problem
     */
    public void include (String path,
                         PortletRequest request,
                         PortletResponse response) throws PortletException,
                                                          IOException;

     /**
      * Returns the resource located at the given path as an
      * <CODE>InputStream</CODE> object.
      * 
      * <P>
      * The data in the <CODE>InputStream</CODE> can be of any type
      * or length. The method returns <CODE>null</CODE> if no resource
      * exists at the given path.
      * </P>
      * <P>
      * To access protected resources the path has to be prefixed with
      * /portlet-inf/ (e.g. /portlet-inf/myportlet/myportlet.jsp).
      * Otherwise, the direct path is used. (e.g. /myportlet/myportlet.jsp).
      * </P>
      * 
      * @param path
      *               the path to the resource
      * @return the input stream
      */
     public InputStream getResourceAsStream (String path);

     /**
      ** Returns the localized text resource with the given key and
      ** using the given locale.
      **
      ** <P>
      ** To use this feature, the <CODE>CLASSPATH</CODE> has to contain a
      ** resource bundle with the same name (including the package) as the
      ** portlet.
      **
      ** @param   key
      **          the text key
      ** @param   bundleName
      **          the name of the resource bundle
      ** @param   locale
      **          the locale to observe
      **
      ** @return   the localized text resource
      **/

     public String getText (String key, String bundleName, Locale locale);

     /**
      * Sends the given message to all portlets on the same
      * page that have the given name. If the portlet name is
      * <CODE>null</CODE> the message is broadcast to all portlets.
      * If more than one instance of the portlet with the given
      * name exists on the current page, the message is
      * sent to every single instance of that portlet. If the
      * source portlet has the same name as the target portlet(s),
      * the message will not be sent to avoid possible cyclic calls.
      * 
      * <P>
      * The portlet(s) with the given name will only receive the
      * message event if it has/they have registered the appropriate
      * listener. This is done by registering the listener in the
      * portlet descriptor.
      * </P>
      * 
      * @param aPortletName
      *                 the name of the portlet(s) that this
      * @param aMessage
      *                 the message to be sent
      */
     public void send (String aPortletName, PortletMessage aMessage);

     /**
      ** Returns the major version of the Portlet API that this portlet
      ** container supports.
      **
      ** @return   the major version
      **
      ** @see   #getMinorVersion()
      **/

     public int getMajorVersion ();

     /**
      ** Returns the minor version of the Portlet API that this portlet
      ** container supports.
      **
      ** @return   the minor version
      **
      ** @see   #getMajorVersion()
      **/

     public int getMinorVersion ();

     /**
      ** Returns the name and version of the portlet container which the
      ** portlet is running in.
      **
      ** <P>
      ** The form of the returned string is <IT>servername/versionnumber</IT>.
      ** For the Jetspeed Framework this method may return the string
      ** <CODE>Apache Jetspeed/1.3a1</CODE>.
      **
      ** @return   the string containing at least name and version number
      **/

      public String getContainerInfo ();

      /**
       ** Returns the portlet log which allows the portlet to write
       ** informational, warning, or error messages to a log.
       **
       ** @return   the portlet log
       **/

      public PortletLog getLog ();
}
