require 'umigame/xunit'
require 'umigame/creator'
require 'umigame/adder'
require 'umigame/sourcecode'
require 'umigame/method'

module Umigame

  class CppUnit < XUnit
    def CppUnit.version
      "CppUnit 1.9.8"
    end
    
    def CppUnit.usage
      print <<"EOB"

#{CppUnit.version}(--lang cppx)
  support option: -a, -i, -m, -N, -P, -r, -s
EOB
    end
    
    def CppUnit.reference
      print <<"EOB"
Assertion Quick Reference
  for #{CppUnit.version}

CPPUNIT_ASSERT(condition) 
CPPUNIT_ASSERT_MESSAGE(message, condition) 
CPPUNIT_FAIL(message) 
CPPUNIT_ASSERT_EQUAL(expected, actual) 
CPPUNIT_ASSERT_EQUAL_MESSAGE(message, expected, actual) 
CPPUNIT_ASSERT_DOUBLES_EQUAL(expected, actual, delta) 
EOB
    end
  end
  
  class CppUnitMethod < Method
    def initialize(class_name, method_name, statement)
      super(class_name, method_name, "//", statement)
    end
    
    def decl
     ret = []
      ret.push "void #{name}() {"
      ret.push @statement
      ret.push "}"
      return prefix + ret.join("\n" + prefix) + "\n"
    end
    
    def suite
      return "#{prefix}CPPUNIT_TEST(#{name});\n"
    end
  end
  
  class CppUnitSourceCode < SourceCode
    def add_method(methodname, prefix=false, statement=nil)
      if statement
        statement =~ /^(.*)\}/m
        statement = $1
      else
        statement = "  CPPUNIT_FAIL(\"no implementation\");"
      end
      method = CppUnitMethod.new(@class_name, methodname, statement)
      super(method, prefix)
    end
    
    def print
      include = ""
      decl = ""
      suite = <<"EOB"
  CPPUNIT_TEST_SUITE(#{@class_name});
//CUPPA:suite=+
EOB
      @include.each do |file|
        include +="\ninclude \"#{file}\""
      end
      @method.each do |method|
        decl += "\n" + method.decl
        suite += method.suite
      end
      suite += "//CUPPA:suite=-\n  CPPUNIT_TEST_SUITE_END();"
      class_end = @class_end.join
      return <<"EOB"
#include <cppunit/extensions/HelperMacros.h>
#include <cppunit/TestAssert.h>
//CUPPA:include=+#{include}
//CUPPA:include=-
#{@class_begin.join}//CUPPA:decl=+#{decl}
//CUPPA:decl=-

#{suite}
#{class_end}
EOB
    end
  end
  
  class CppUnitAdder < Adder
    def initialize
      super("//", CppUnitSourceCode.new)
    end
    
    protected
    
    def parse_include
      do_parse_include(/#include "(.+)"/)
    end
    
    def parse_class_begin
      do_parse_class_begin(/class (.+) : public /)
    end
    
    def parse_method
      do_parse_method(/^(..)void (.+)\(\) \{/)
    end
    
    def parse_suite
      do_parse_suite(/CPPUNIT_TEST_SUITE\(#{@code.class_name}\);/, 
        /(\/\/|\s\s)\s*CPPUNIT_TEST\((.*)\);/, 0, 1)
    end
  end
  
  class CppUnitCreator < Creator
    
    attr_reader :code
    
    def filename(name)
      name + ".cpp"
    end
    
    def eval_argv
      # nothing to do
    end
    
    def write_skeleton
      code = CppUnitSourceCode.new
      code.class_name = @skeleton
      @include.each do |includefile|
        code.add_include includefile
      end
      code.class_begin.push <<"EOB"

class #{@skeleton} : public CppUnit::TestFixture {
  private:
  // your stuff...
  
  public:
  
  virtual void setUp() {
    // initialize
  }
  
  virtual void tearDown() {
    // terminate
  }

EOB
      code.has_suite = !@prefix

      @testmethod.each do | testcase |
        code.add_new_method testcase
      end
      code.class_end.push "};\n"
      unless @noregist
        code.class_end.push "CPPUNIT_TEST_SUITE_REGISTRATION(#{@skeleton});\n"
      end
      return code.print
    end
    
    def write_main
     return <<"EOF"
#include <iostream>
#include <string>
#include <cppunit/ui/text/TestRunner.h>
#include <cppunit/XmlOutputter.h>
#include <cppunit/TextOutputter.h>
#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>

int main(int argc, char* argv[]) {
  int format = 0;
  int target = 0;
  std::string xsl;
  std::string ns;
  for ( int i = 1; i < argc; ++i ) {
    std::string arg(argv[i]);
    if ( arg == "--text"      ) format = 0;
    if ( arg == "--xml"       ) format = 1;
    if ( arg == "--compiler"  ) format = 2;
    if ( arg == "--cerr"      ) target = 1;
    if ( arg == "--xsl"       ) xsl = argv[++i];
    if ( arg == "--namespace" ) ns  = argv[++i];
  }
  CppUnit::TextUi::TestRunner runner;
  if ( ns.empty() )
    runner.addTest(CppUnit::TestFactoryRegistry::getRegistry().makeTest());
  else
    runner.addTest(CppUnit::TestFactoryRegistry::getRegistry(ns).makeTest());
  CppUnit::Outputter* outputter = 0;
  std::ostream* stream = target ? &std::cerr : &std::cout;
  switch ( format ) {
  case 0 :
    outputter = new CppUnit::TextOutputter(&runner.result(),*stream);
    break;
  case 1 :
    outputter = new CppUnit::XmlOutputter(&runner.result(),*stream, "shift_jis");
    static_cast<CppUnit::XmlOutputter*>(outputter)->setStyleSheet(xsl);
    break;
  case 2 :
    outputter = new CppUnit::CompilerOutputter(&runner.result(),*stream);
    break;
  }
  runner.setOutputter(outputter);
  return runner.run() ? 0 : 1;
}
EOF
    end
  end
end
