#include "curie.h"
#include <cuppa/tokenizer.h>

using namespace curie;


#if defined(__unix__) || defined(__linux__)
#  include <unistd.h>
#  include <stdio.h>
#endif

#ifdef __BORLANDC__
#  include <dir.h>
#endif

#ifdef _MSC_VER
#  include <direct.h>
#  include <stdio.h>
namespace {
  inline char *getcwd(char *cwd, size_t size){
    return ::_getcwd(cwd, size);
  }
}

#endif


std::string Curie::getcwd(){
  char cwd[FILENAME_MAX];
  char *res = ::getcwd(cwd, sizeof(cwd));
  if (res == NULL)
    throw error("Failed to get the current working directory.");

  for (char *ch = cwd; ((int)*ch) != 0; ++ch)
    *ch = (*ch == '\\') ? '/' : *ch;

  return std::string(cwd);
}



bool ClassDef::hasPureVirtual() const {
  for (Methods::const_iterator it = methods_.begin();
       it != methods_.end(); ++it)
    if (it->isPureVirtual_)
      return true;
  return false;
}



std::string Curie::createRelativePath(const std::string &path) {
  std::string relpath;


  std::string cwd = getcwd();
  std::string str= path;

  if (cwd.at(0) == '/') {// UNIX
    cwd = cwd.substr(1, cwd.length());
    str = str.substr(1, str.length());
  }
  cuppa::tokenizer<> cwdtok(cwd);
  cuppa::tokenizer<> pathtok(str);

  bool matched = true;
  int numTenTens = 0; // TenTen means '..'

  while ( cwdtok.has_more_token("/") && pathtok.has_more_token("/") && matched){
    std::string cwdtoken = cwdtok.next_token("/");
    std::string pathtoken = pathtok.next_token("/");
    matched = (cwdtoken == pathtoken);
    if (!matched)
      ++numTenTens;
  }

  while (cwdtok.has_more_token("/")) {
    ++numTenTens;
    cwdtok.next_token("/");
  }

  for (; numTenTens != 0; --numTenTens)
    relpath += "../";

  if (pathtok.has_more_token("/"))
    relpath += pathtok.next_token("/");

  while (pathtok.has_more_token("/"))
    relpath += '/' + pathtok.next_token("/");

  return relpath;
}

void Curie::setStripPath(const std::string &str){
  stripPath_ = str;
  if (stripPath_.at(stripPath_.length() - 1) != '/')
    stripPath_ += "/";
}

void Curie::checkClassDefs() {
  // remove dtor.
  for (ClassDefs::iterator defit = classDefs_.begin(); defit != classDefs_.end(); ++defit){
    for (Methods::iterator metit = defit->methods_.begin(); 
	 metit != defit->methods_.end(); ++metit){
      if (metit->name_.at(0) == '~'){
	defit->methods_.erase(metit);
	break;
      }
    }
  }
}



void Curie::addFile(const std::string &fileName) {
  files_.push_back(fileName);
}


int Curie::createTestDefs(){

  testDefs_.clear();
  for (Files::iterator it = files_.begin(); it != files_.end(); ++it) {
    int res = parseFile(*it, classDefs_);
    if (res)
      return res;
  }

  checkClassDefs();
  createTestNames(classDefs_, stripPath_, testDefs_);

  if (relative_)
    for (TestDefs::iterator tit = testDefs_.begin();
	 tit != testDefs_.end(); ++tit)
      tit->includeName_ = createRelativePath(tit->includeName_);
  return 0;
}


/** output arguments for 'Makefile' */
void Curie::outputMakeArgs(std::ostream &ostm) const {

  std::string sources;
  for (TestDefs::const_iterator it = testDefs_.begin(); it != testDefs_.end(); ++it){
    if (!it->hasPureVirtual_)
      sources += it->testName_ + ".cpp ";
  }
  if (mainName_.length() != 0)
    sources += mainName_ + ".cpp";

  ostm << "SRCS=\"" << sources << "\"" << std::endl;

  if (mainName_.length() != 0)
    ostm << "MAIN=\"" << mainName_ << "\"" << std::endl;
}

/** --skeleton */
void Curie::outputSkeletons(std::ostream &ostm) const{

 for (TestDefs::const_iterator defit = testDefs_.begin(); defit != testDefs_.end(); ++defit) {
   if (forceEnabled_)
     ostm << "--force ";

   ostm << "--skeleton=" << defit->testName_ << " ";

   if (defit->hasPureVirtual_)
     ostm << "--noregist ";
   if (defit->includeName_.length() != 0)
     ostm << "--include=" << defit->includeName_;

   for (TestMethods::const_iterator memit = defit->testMethods_.begin();
	memit != defit->testMethods_.end();
	++memit) {
     ostm << " ";
     if (memit->pureVirtual_)
       ostm << ".";
     ostm <<  memit->name_;
   }

   ostm << std::endl;

 }
}


/** --main */
void  Curie::outputMain(std::ostream &ostm) const {
  if (mainName_.length() != 0) {
    if (forceEnabled_)
      ostm << "--force ";
    ostm << "--main=" << mainName_ << std::endl;
  }
}
