/*
 * 'cocuppa' : COmpact CppUnit Pre Process Aid
 *   $Revision: 1.3 $ $Date: 2002/08/27 08:38:41 $
 *
 *    by episteme
 *
 *  creates test-skeletons for CppUnit 1.9.2 or later
 */

#include <iostream>  // cerr/cout/endl
#include <fstream>   // ofstream
#include <algorithm> // copy

#include "cuppa/cocuppa.h"   // cocuppa
#include "cuppa/tokenizer.h" // tokenizer

// system does NOT have getopt
#define NO_GETOPT

/*
 * getopt
 */
#ifdef NO_GETOPT
#ifndef __STDC__
#  define __STDC__ 1
#  include "getopt.h" // getopt_long
#  undef __STDC__
#else
#  include "getopt.h" // getopt_long
#endif
#else
#include <getopt.h>
#endif

using namespace std;

bool exist(const string& path) {
  ifstream stream(path.c_str());
  return stream.is_open();
}

ofstream* open_file(const string& path, bool force_open =false) {
  cerr << path << " ...";
  ofstream* strm = 0;
  if ( !force_open && exist(path) ) {
    cerr << "already exists. (skip)\n";
  } else {
    strm = new ofstream(path.c_str());
    if ( strm->is_open() ) {
      cerr << "create.\n";
    } else {
      cerr << "creation fail.\n";
      delete strm;
      strm = 0;
    }
  }
  return strm;
}

void make_suite(cuppa::cocuppa& co, bool force) {
  ofstream* strm;
  if ( co.get_standalone() ) {
    strm = open_file(co.get_suite() + IMPL_SUFFIX, force);
    if ( strm ) {
      co.make_suite_decl(*strm);
      co.make_suite_impl(*strm);
      delete strm;
    }
  } else {
    strm = open_file(co.get_suite() + DECL_SUFFIX, force);
    if ( strm ) {
      co.make_suite_decl(*strm);
      delete strm;
      strm = open_file(co.get_suite() + IMPL_SUFFIX, force);
      if ( strm ) {
        co.make_suite_impl(*strm);
        delete strm;
      }
    }
  }
}

void version() {
  cout << "cocuppa $Date: 2002/08/27 08:38:41 $" << endl;
}

void usage() {
  cout <<
"--abstract, -A         : abstract test\n"
"--cin                  : input from standard-input\n"
"--force, -f            : force creation\n"
"--help, -h             : display help\n"
"--include,   -i <file> : insert #include \"file\"\n"
"--Include,   -I <file> : insert #include <file>\n"
"--main,      -m <Name> : create Name" IMPL_SUFFIX "for main()\n"
"--namespace, -n <name> : wrapped around namespace name { ... }\n"
"--noregist,  -N        : disable registration\n"
"--prefix,    -P        : prepend 'test' to methods\n"
"--separate,  -p        : separate decl-part & impl-part\n"
"--skeleton,  -s <Name> : create skeleton Name" IMPL_SUFFIX "\n"
"--super,     -S <Base> : Base's derivative\n"
"--version,   -v        : print version\n"
;
}

/*
 * cocuppa atarts here!!
 */
int cocuppa_main(int argc, char* argv[]) {

  cuppa::cocuppa co;

  string gen_main;
  bool   force = false;

  while ( true ) {
    int option_index = 0;
    static struct option long_options[] = {
      { "abstract",   0, 0, 'A'},
      { "force",      0, 0, 'f'},
      { "help",       0, 0, 'h'},
      { "include",    1, 0, 'i'},
      { "Include",    1, 0, 'I'},
      { "main",       1, 0, 'm'},
      { "namespace",  1, 0, 'n'},
      { "noregist",   0, 0, 'N'},
      { "prefix",     0, 0, 'P'},
      { "separate",   0, 0, 'p'},
      { "skeleton",   1, 0, 's'},
      { "super",      1, 0, 'S'},
      { "version",    0, 0, 'v'},
      { 0, 0, 0, 0 }
    };

    int c = getopt_long(argc, argv, "ANPS:fhi:I:m:n:o:ps:v",
                        long_options, &option_index);
    if ( c == -1 ) break;

    switch ( c ) {
    case 'A' : co.set_standalone(false);    // fall through
    case 'N' : co.set_noregist(true);       break;
    case 'P' : co.set_prefix(true);         break;
    case 'S' : co.set_super(optarg);        break;
    case 'f' : force = true;                break;
    case 'h' : usage();                     return 0;
    case 'i' : co.add_include(optarg);      break;
    case 'I' : co.add_include(optarg,true); break;
    case 'm' : gen_main = optarg;           break;
    case 'n' : co.set_namespace(optarg);    break;
    case 'p' : co.set_standalone(false);    break;
    case 's' : co.set_suite(optarg);        break;
    case 'v' : version();                   return 0;
    }
  }

  // suite
  if ( !co.get_suite().empty() ) {
    while ( optind < argc ) {
      string method = argv[optind++];
      int methodlen = method.size() - 1;
      if ( method[methodlen] != '.' ) {
        co.add_case(method, false);
      } else {
        co.add_case(method.substr(0,methodlen), true);
      }
    }
    make_suite(co, force);
  }

  // main
  if ( !gen_main.empty() ) {
    ofstream* strm = open_file(gen_main + IMPL_SUFFIX, force);
    if ( strm ) {
      cuppa::cocuppa::make_main(*strm);
      delete strm;
    }
  }

  return 0;
}

/*
 * entry point
 */
int main(int argc, char* argv[]) {

  if ( argc < 2 ) {
    cerr << "'cocuppa --help' for help.\n";
    return 0;
  }

  string cin_found = "--cin";
  for ( int i = 1; i < argc; ++i ) {
    if ( argv[i] == cin_found ) {
      cin_found = "";
      break;
    }
  }

  if ( cin_found.empty() ) {
    string line;
    while ( getline(cin, line) ) {
      // output to cout if not started with '-'
      if ( line.empty() || line[0] != '-' ) {
        cout << line << endl;
        continue;
      }
      vector<char*> cocuppa_argv;
      cocuppa_argv.push_back("cocuppa");
      string token;
      cuppa::tokenizer<> tokenizer(line);
      while ( tokenizer.next_token(" \t\n",token) ) {
        char* arg = new char[token.size()+1];
        *copy(token.begin(), token.end(), arg) = '\0';
        cocuppa_argv.push_back(arg);
      }
      optind = 0;
      int result = cocuppa_main(cocuppa_argv.size(), &cocuppa_argv[0]);
      for ( vector<char*>::size_type i = 1; i < cocuppa_argv.size(); ++i ) {
        delete [] cocuppa_argv[i];
      }
      if ( result ) return result; // zڧҎÎz
    }
    return 0;
  }

  return cocuppa_main(argc, argv);
}
